package BalanceQueue;

use strict;
use warnings;
use utf8;

use feature 'state';

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::ListUtils;

use Settings;

=head2 Именованные константы с приоритетами для ленивой очереди balance_info_queue: PRIORITY_NNNNNN

    Нужны чтобы не использовать магические числа и чтобы все приоритеты были рядом,
    можно было легко переприотизировать и оценить что важнее + понять легко понимать
    кто насыпал объекты в переотправку.
    
    Чем больше приоритет, тем раньше переотправится объект.
    Допустимые значения: [-128, 127]
    
    Если захочется использовать повторяющиеся значения, нужно будет отключить проверку
        в юнит-тесте BalanceQueue/priorities.t (сделать истинной константу SKIP_DUPLICATE_PRIORITITES)

=item PRIORITY_BS_EXPORT_NEW_CAMPAIGN
=item PRIORITY_CAMP_WITH_BLOCKED_MONEY
=item PRIORITY_ONE_SHOT_CHANGE_AGENCY_TO_AGENCY
=item PRIORITY_ONE_SHOT_CHANGE_MANAGER_TO_AGENCY
=item PRIORITY_CAMPS_ON_MASS_AGENCY_MANAGER_CHANGED
=item PRIORITY_CAMPS_ON_AGENCY_MANAGER_CHANGED
=item PRIORITY_CAMPS_ON_AGENCY_MASS_MANAGER_CHANGE
=item PRIORITY_CAMP_ON_SERVICED
=item PRIORITY_CAMP_ON_MANAGER_CHANGED
=item PRIORITY_CAMP_ON_NEW_BILLING_AGGREGATE
=item PRIORITY_CAMP_ON_ENABLE_WALLET_API
=item PRIORITY_CAMP_ON_ENABLE_WALLET
=item PRIORITY_CAMPS_ON_ENABLE_WALLET
=item PRIORITY_USER_ON_SWITCHED_AGENCY_CHIEF
=item PRIORITY_USER_ON_SWITCHED_CLIENT_CHIEF
=item PRIORITY_USER_ON_MODIFY_USER
=item PRIORITY_USER_ON_SAVING_EASY_BANNER
=item PRIORITY_USER_ON_SAVING_EASY_CAMP
=item PRIORITY_USER_ON_SAVING_NEW_CAMP
=item PRIORITY_CHANGED_CLIENT_CITY
=item PRIORITY_USER_ON_SAVING_CAMP
=item PRIORITY_CAMP_ON_CHANGED_NAME
=item PRIORITY_CHANGED_USER_EMAILS
=item PRIORITY_CHANGED_USER_SETTINGS
=item PRIORITY_DEFAULT

=cut
use constant PRIORITY_BS_EXPORT_NEW_CAMPAIGN                =>  120;
use constant PRIORITY_CAMP_WITH_BLOCKED_MONEY               =>  115;
use constant PRIORITY_ONE_SHOT_CHANGE_AGENCY_TO_AGENCY      =>   91;
use constant PRIORITY_ONE_SHOT_CHANGE_MANAGER_TO_AGENCY     =>   90;
use constant PRIORITY_CAMP_ON_ENABLE_WALLET                 =>   77;
use constant PRIORITY_CAMPS_ON_ENABLE_WALLET                =>   76;
use constant PRIORITY_CAMPS_ON_MASS_AGENCY_MANAGER_CHANGED  =>   75;
use constant PRIORITY_CAMPS_ON_AGENCY_MANAGER_CHANGED       =>   74;
use constant PRIORITY_CAMPS_ON_AGENCY_MASS_MANAGER_CHANGE   =>   73;
use constant PRIORITY_CAMP_ON_SERVICED                      =>   72;
use constant PRIORITY_CAMP_ON_MANAGER_CHANGED               =>   71;
use constant PRIORITY_CAMP_ON_NEW_BILLING_AGGREGATE         =>  113;
use constant PRIORITY_CAMP_ON_ENABLE_WALLET_API             =>   65;
use constant PRIORITY_USER_ON_SWITCHED_AGENCY_CHIEF         =>   55;
use constant PRIORITY_USER_ON_SWITCHED_CLIENT_CHIEF         =>   54;
use constant PRIORITY_USER_ON_MODIFY_USER                   =>   45;
use constant PRIORITY_USER_ON_SAVING_EASY_BANNER            =>   42;
use constant PRIORITY_USER_ON_SAVING_EASY_CAMP              =>   41;
use constant PRIORITY_USER_ON_SAVING_NEW_CAMP               =>   32;
use constant PRIORITY_CHANGED_CLIENT_CITY                   =>   30;
use constant PRIORITY_USER_ON_SAVING_CAMP                   =>   21;
use constant PRIORITY_CAMP_ON_CHANGED_NAME                  =>   20;
use constant PRIORITY_CHANGED_USER_EMAILS                   =>   12;
use constant PRIORITY_CHANGED_USER_SETTINGS                 =>   11;
use constant PRIORITY_DEFAULT                               =>    1;


=head2 add_to_balance_info_queue

    Добавляет один или несколько объектов (cid, uid) в очередь ленивой переотправки в баланс.

    add_to_balance_info_queue($operator_uid, $obj_type, $cid_or_uid);
    add_to_balance_info_queue($UID, 'cid' => 1234);
    add_to_balance_info_queue($UID, 'uid' => 10999940);
    add_to_balance_info_queue($UID, 'cid' => [123,456,789]);
    add_to_balance_info_queue($UID, 'cid' => [123,456,789], BalanceQueue::PRIORITY_DEFAULT);

=cut

sub add_to_balance_info_queue {
    my ($operator_uid, $obj_type, $cid_or_uid, $priority) = @_;
    $priority //= PRIORITY_DEFAULT;

    die "obj_type at add_to_balance_info_queue have to be uid or cid" unless ($obj_type =~ m/^uid|cid$/);
    $cid_or_uid = [$cid_or_uid] unless ref($cid_or_uid) eq 'ARRAY';
    return undef unless $cid_or_uid && @$cid_or_uid;

    # TODO: после отказа от хранения обработанных и ошибочных (от send_status) завести уникальный ключ obj_type+cid_or_uid
    # и вставлять одним запросом с ON DUPLICATE KEY UPDATE, исключая текущие два запроса и гонки
    # если объект уже стоит в очереди с большим или таким же приоритетом, заново не ставим. с меньшим -- ставим второй раз.
    my $exist_values = get_one_column_sql(PPC($obj_type => $cid_or_uid), [
        'SELECT DISTINCT cid_or_uid FROM balance_info_queue',
        WHERE => {send_status => 'Wait', obj_type => $obj_type, cid_or_uid => SHARD_IDS, priority__ge => $priority},
    ]);
    my $values_left = xminus($cid_or_uid, $exist_values);
    my $total_inserted = 0;
    if ($values_left && @$values_left) {
        foreach_shard $obj_type => $values_left, chunk_size => 1_000, sub {
            my ($shard, $chunk) = @_;
            my @data_for_insert = map { [$operator_uid // 0, $obj_type, $_, $priority] } @$chunk;
            $total_inserted += do_mass_insert_sql(PPC(shard => $shard),
                'INSERT INTO balance_info_queue (operator_uid, obj_type, cid_or_uid, priority) VALUES %s',
                \@data_for_insert);
        };
    }
    return $total_inserted;
}


=head3 get_priorities_hash

    Возвращает ссылку на хеш с именованными константами приоритетов

=cut
sub get_priorities_hash {
    no strict 'refs';
    state $hash //= { map {$_ => &{$_} } grep {$_ =~ m/PRIORITY_/} keys %{__PACKAGE__.'::'} };
    return $hash;
}

1;
