package BannerFlags;

use Direct::Modern;

use Yandex::I18n;
use List::MoreUtils qw/any none uniq/;
use Settings;
use Readonly;

=head1 BannerFlags

Модуль для работы с флагами на баннере.
Предоставляет интерфейс для  сериализации и десериализации соотвествующего поля в таблице banners
и для десериализации флагов которые приходят из модерации.

=cut

=head1

  Cписок флагов, для которых необходимо предоставление лиценизии. DIRECT-70199, DIRECT-72623
  Переехал из ModLicense.pm

=cut

Readonly our @GEO_LEGAL_FLAGS => qw/
    acids
    alcohol
    banks
    binary_options
    credit_consumer_cooperative
    detective
    dietarysuppl
    explosions
    forex
    forex_broker
    gamble
    insurance
    loan
    maternity_capital
    med_equipment
    med_services
    mfi
    not_medicine
    optics
    pawnshop
    payment
    pharmacy
    popular_medicine
    pseudoweapon
    psychology
    sports_nutrition
    tech_inspection
    tobacco
    transport
    veterinary
    abortion
    car_numbers
    illegal_diet
    drugs
    bookmaker
    detective_equipment
    jewelry
    mediation
    notary
    realtor
    weapons
    magic
    tattoo
    charity
    lottery
    telemed
    electronic_osago
    work_abroad
    foreign_trade
    ad_on_transport
    cryptocurrency
    illegal
    collector
    milk_substitute
    certification
    diploma
    legal_vape
    education
    people
    project_declaration
    social_advertising
    rkn/;

=head1 %AD_WARNINGS
    хеш с вариантами флагов, которые могут быть доступны пользователям

    age_limits -
        признак того, что флаг является возростным ограничением
        а API нужен для выставления поля "AgeLabel"
        только значение этих флагов можно менять через Директ
    no_show -
        если 1 значит флаг не нужно показывать клиентам он только для БК
    is_common_warn -
        поля не помеченые этим флагом являются взаимоисключаюшими
    postfix -
        прибавляется к значению перед выводом,
        например если флаг "age", значение "6", а постфикс "+"
        пользователь увидит возрастное ограничение "6+"
    include_warnings - массив []
        список флагов, которые включены в текущий флаг (текуший флаг является супертипом для флагов из include_warnings)

=cut

our %AD_WARNINGS = (
    'abortion'  =>  {long_text => iget_noop("Есть противопоказания. Посоветуйтесь с врачом. Возможен вред здоровью."),
                    short_text => iget_noop("аборты")},
    'medicine'  =>  {
        long_text => iget_noop("Есть противопоказания. Посоветуйтесь с врачом."),
        short_text => iget_noop("медицина"),
    },
    'alcohol'   =>  {long_text => iget_noop("Чрезмерное потребление вредно."),
                    short_text => iget_noop("алкоголь")},
    'tobacco'   =>  {long_text => iget_noop("Курение вредит вашему здоровью"),
                    short_text => iget_noop("табак"),
                    no_show_in_template => 1,
    },
    'age'       =>  {variants => [qw/18 16 12 6 0/],
                     postfix => '+',
                     default => 18,
                     age_limit => 1,
                     is_common_warn => 1,
                     no_show => 1},
    'plus18'    =>  {no_show => 1, related_flags => [qw/age/],is_common_warn => 1,},
    'baby_food' =>  {long_text => iget_noop("Проконсультируйтесь со специалистом. Для питания детей с %d месяцев"), 
                     short_text => iget_noop("детское питание"), 
                     variants => [0 .. 12],
                     postfix => 'months',
                     default => '11',
                     age_limit => 1
                    },
    'dietarysuppl' => { 
        long_text => iget_noop("Не является лекарством"),
        short_text => iget_noop("БАД"),
    },
    'project_declaration' => { 
        long_text => iget_noop("Проектная декларация на рекламируемом сайте"),
        short_text => iget_noop("Проектная декларация"),
    },
    'tragic' =>  {
        is_common_warn => 1,
        no_show => 1,
    },
    'asocial' => {
        is_common_warn => 1,
        no_show => 1,
    },
    'pseudoweapon' => {
        long_text => iget_noop('Конструктивно сходные с оружием изделия'),
        short_text => iget_noop('не оружие'),
    },
    'forex' =>  {
        no_show => 1,
    },
);
for my $flag (keys %AD_WARNINGS) {
    my $parent = $AD_WARNINGS{$flag}->{parent};
    next if !$parent;
    croak "Unknown parent $parent"  if !$AD_WARNINGS{$parent};
    push @{$AD_WARNINGS{$parent}->{children}}, $flag;
}


our @MONITOR_FLAGS = qw//;

my $RESET_VALUE = '99+';

=head2 get_banner_flags_as_hash

    Функция десериализует поле флагов баннера в хеш.
    Заодно подставляются зависимые флаги.
    
    $flags = "flag1,flag2:value,..."

    Именованые параметры:
    all_flags => 1 -- вернуть все флаги, даже не описаные в %AD_WARNINGS и помеченный как подтип другого флага (см. include_warnings)
    with_postfix => 1 -- возврящать постфикс вместе с возрастной меткой
    no_parent
    no_children

    Возвращает хеш:
    $result = {
        flag1 => 1,
        flag2 => 'value',
    }

=cut

sub get_banner_flags_as_hash
{
    my ($flags, %opt) = @_;
    my %result;

    my %flags = map {   
        my ($warn_name, $value) = map { lc($_) } split /:/, $_, 2;
        ($warn_name => $value)
    } split( /,/, $flags || '');
    my %orig_flag = %flags;

    foreach my $warn_name (keys %orig_flag) {
        if (!$opt{no_children}) {
            # если не установлен ни один из уточняющих флагов - ставим все (для интерфейса)
            my $children = $AD_WARNINGS{$warn_name}->{children};
            if ($children && none {exists $orig_flag{$_}} @$children) {
                $flags{$_} = $orig_flag{$warn_name}  for @$children;
            }
        }

        if (!$opt{no_parent}) {
            # если установлен только уточняющий - ставим общий (для БК)
            my $parent = $AD_WARNINGS{$warn_name}->{parent};
            if ($parent && !exists $orig_flag{$parent} ) {
                $flags{$parent} = $orig_flag{$warn_name};
            }
        }

        # related_flags вычисляем для первого уровня,
        # т.е. не вычисляем related_flags(related_flags)
        my @related_flags = @{_get_related_flags($warn_name)};
        $flags{$_} //= undef for @related_flags;
        delete $flags{$warn_name} if @related_flags;
    }
    my @delete_flags = map { !exists $AD_WARNINGS{$_} ? $_ : @{$AD_WARNINGS{$_}->{include_warnings} || []} } keys %flags; 
    delete @flags{@delete_flags} if !$opt{all_flags};

    foreach my $warn_name (keys %flags) {    
        my $value = $flags{$warn_name};
        if (exists $AD_WARNINGS{$warn_name}) {
            if (!defined $value) {
                $value = $AD_WARNINGS{$warn_name}{default};
            }
            if (defined $AD_WARNINGS{$warn_name}{postfix}
                && $opt{with_postfix}) {
                $value .= $AD_WARNINGS{$warn_name}{postfix};
            }
        }
        _add_flag_in_output(\%result, $warn_name, $value);
    }
    return \%result;
}

=head2 _add_flag_in_output(result_hash_ref, flag_name, flag_value)

    Добавляет флаг в выдачу, если его там еще нет

=cut

sub _add_flag_in_output($$$){
    my ($result, $warn_name, $value) = @_;
    if (!exists $result->{$warn_name}){
        $result->{$warn_name} = defined $value ? $value : 1;
    }
}

=head2 filter_warnings(@flags_names)

    Оставить из списка флагов только те,
    которые, можно показывать пользователю,
    сейчас это все флаги кроме 'age'

=cut

sub filter_warnings(@){
    my @flags_names = @_;
    my @warnings = grep { exists $AD_WARNINGS{$_} && !$AD_WARNINGS{$_}{no_show}} @flags_names;
    return \@warnings;
}

=head2 get_all_warnings

    Получить все предупреждения которые можно коказывать пользователю
    Возвращает хеш аналогичный %AD_WARNINGS только отфильтрованный

=cut

sub get_all_warnings {
    my %opts = @_;
    my $warnings = filter_warnings(keys %AD_WARNINGS);
    if ($opts{for_template}) {
        $warnings = [grep { !$AD_WARNINGS{$_}{no_show_in_template} } @$warnings];
    }
    my %flags = map {$_ => $AD_WARNINGS{$_}} @$warnings;
    return \%flags;
}

=head2 serialize_banner_flags_hash

    Сериализует хеш с флагами для сохранения в базе Директа.
    Обратная функция к get_banner_flags_as_hash

=cut

sub serialize_banner_flags_hash
{
    my ($flags) = @_;
    my @accumulator;

    foreach my $flag (grep {is_valid_banner_flag_name($_)} keys %$flags) {
        if (exists $AD_WARNINGS{$flag} && defined $AD_WARNINGS{$flag}{variants}) {
            push @accumulator, $flag.":".$flags->{$flag};
        }
        else {
            push @accumulator, $flag;
        }
    }

    return @accumulator ? join(",", sort &uniq(@accumulator)) : undef;
}


=head2 is_valid_banner_flag_name

    Проверяет имя флага на допустимый набор символов

=cut
sub is_valid_banner_flag_name
{
    my ($flag) = @_;
    return $flag =~ /^[a-zA-Z0-9_\-:]+$/ ? 1 : 0;
}


=head2 get_warnings_text_for_flags($flags)

    $flags = "flag1,flag2:value,..."
    Возвращает список текстовых предупреждений для  строки флагов из баннера.

=cut

sub get_warnings_text_for_flags
{
    my ($flags) = @_;
    my @result;

    my @kv = split(/,/, $flags || '');
    
    foreach my $part (@kv) {
        my ($warn_name, $value) = map { lc($_) } split(/:/, $part, 2);
        if (exists $AD_WARNINGS{$warn_name}
            && defined $AD_WARNINGS{$warn_name}{long_text}
            && $AD_WARNINGS{$warn_name}{long_text} ne ''
        ) {
            push @result, iget($AD_WARNINGS{$warn_name}{long_text}, $value);
        }
    }
    
    return [uniq @result];
}

=head2 get_retargeting_warnings_flags($group)

    Получить предупреждения и флаги для ретаргетинга
    Параметры позиционные:
        $group - хеш группы, должен содержать как минимум ключ banners
    Параметры именованные:
        detailed => 1 - отдавать помимо текстовых предупреждений также хеши вида:
                        {text => $text, link => $support_href}

=cut
sub get_retargeting_warnings_flags
{
    my ($group, %params) = @_;

    my @warnings;

    if (any {$_ && /\b(?:unfamily|tragic)\b/} map {$_->{flags}} @{$group->{banners}}) {
        my $text = iget('Объявление взрослой/трагической тематики не показывается в сетях');
        if ($params{detailed}) {
            push @warnings, {text => $text};
        } else {
            push @warnings, $text;
        }
    } elsif (any {$_ && /\b(?:asocial)\b/} map {$_->{flags}} @{$group->{banners}}) {
        my $text = iget('Не показывается по интересам пользователя и условиям подбора аудитории');
        if ($params{detailed}) {
            push @warnings, {text => $text, link => 'direct-tooltips/ads-not-shown.xml'};
        } else {
            push @warnings, $text;
        }
    }

    return if !@warnings;

    foreach my $banner (@{$group->{banners}}) {
        $banner->{warnings} ||= [];
        push @{$banner->{warnings}}, @warnings;
    }

    return;
}

=head2 convert_flags_to_moderation_format

Конвертирует хеш с флагами в формате Директа в хеш понятный модерации.

=cut

sub convert_flags_to_moderation_format
{
    my ($flags) = @_;

    my %result;

    foreach my $flag_name (sort keys %$flags) {
        if (exists $AD_WARNINGS{$flag_name} && defined $AD_WARNINGS{$flag_name}{age_limit}) {
            my $value = $flags->{$flag_name};
            $result{$flag_name} = $flag_name.$flags->{$flag_name};
        }
        else {
            $result{$flag_name} =  $flags->{$flag_name};
        }
    }
    
    return \%result;
}

=head2 _get_age_limit_flag(@flags_names)

    Получает из списка флагов тот, что является возростным ограничением
    
=cut

sub _get_age_limit_flag(@)
{
    my @flags_names = @_;
       
    my @age_limit_flags = (grep {exists $AD_WARNINGS{$_} && $AD_WARNINGS{$_}{age_limit}} @flags_names);
    my $age_limit_flag = shift @age_limit_flags;
    return $age_limit_flag;
}

=head2 get_age_limits($flags, $additional_flags)

    Соотвествующие флаги установленные на баннер и ассет преобразуются в возрастное ограничение в формате который понимает БК.
    Принимает:
        $flags = "flag1,flag2:value,..."
        $additional_flags = "flag2,flag3:value,..."
            
=cut

sub get_age_limits
{
    my ($flags, $additional_flags) = @_;
    my $age = get_banner_flags_as_hash($flags || "", all_flags => 1)->{'age'};
    my $age_2 = get_banner_flags_as_hash($additional_flags || "", all_flags => 1)->{'age'};
    if (!defined $age
        || defined $age_2 && $age_2 > $age) {
        $age = $age_2;
    }
    if (defined $age) {
        $age .= $AD_WARNINGS{'age'}{postfix};
    }
    return $age;
}

=head2 _find_flag($value)

    По значению определяет название флага, если известен постфикс
    Нужно для того, чтобы суперпользователь мог поставить флаг,
    который не был выставлен на модерации
=cut

sub _find_flag($){
    my $value = shift;
    if ($value =~ /^\d+(.+)/) {
        my $current_postfix = $1;
        for my $flag_name (keys %AD_WARNINGS){
            if (exists $AD_WARNINGS{$flag_name}->{postfix}
                && $AD_WARNINGS{$flag_name}->{postfix} eq $current_postfix
            ) {
                my $valid_variants = _get_valid_variants($flag_name, 1);
                if(exists $valid_variants->{$value}){
                    return $flag_name;
                }
            }
        }
    }
    return;
}

=head2 validate_ad_age_label(%)

    role => роль оператора
    age_label => новое значение метки
    old_flags => старое значение метки
    camp_type => тип кампании (text|mobile_content)
    not_empty => не позволять значению новой возрастной метки быть пустым. Считать пустое значение невалидным.

=cut

sub validate_ad_age_label
{
    my %O = @_;
    return 0 if $O{not_empty} && (!defined $O{age_label} || $O{age_label} =~ /^\s*$/ || $O{age_label} =~ /^\Q$RESET_VALUE\E$/ );
    my $old_flags = get_banner_flags_as_hash($O{old_flags}, all_flags => 0);
    my $internal_user = $O{role} && $O{role} =~ /^(support|placer|super)$/ ? 1 : 0;
    $O{camp_type} = 'text' unless defined $O{camp_type};

    my @flags_names = keys %$old_flags;
    my $age_limit_flag = _get_age_limit_flag(@flags_names);
    if (!$age_limit_flag) {
        $age_limit_flag = _find_flag($O{age_label});
    }    
    return 0 unless $age_limit_flag;
    if (validate_flag_value($age_limit_flag, $O{age_label}, $internal_user)) {
        if ($internal_user || $O{camp_type} eq 'mobile_content') {
            return 1;
        } else {

            my $related_flags = _get_related_flags($age_limit_flag);
            
            my $is_exists_related_flags = any { defined $old_flags->{$_} } @$related_flags;
            
            if (defined $old_flags->{$age_limit_flag} || $is_exists_related_flags) {
                return 1;
            } else {
                # обычным клиентам - нельзя устанавливать метку, если она не была установлена ранее
                return 0;
            }
        }

    }
    return 0;
}

=head2 validate_flag_value

    Проверяет имеет ли значение данного флага валидное значения.
    На входе:
        flag - имя флага
        value - значение флага
        internal_user - 1 - запрашивается для внутреннего пользователя; 0 - не для внутреннего пользователя.
                        для некоторых ролей список доступных значений может различаться. По умолчанию 1.

=cut
sub validate_flag_value {
    my ($flag, $value, $internal_user) = @_;
    $internal_user = 1 unless defined ($internal_user);
    my %VALID_AGE_LABELS = %{_get_valid_variants($flag, $internal_user)};
    return ($VALID_AGE_LABELS{normalize_flag_value($flag, $value)}) ? 1 : 0;

}
=head2 _get_valid_variants($flag_name, $internal_user)

    Генерит хеш с вариантами валидных значений для флага

=cut

sub _get_valid_variants($$){
    
    my ($flag_name, $internal_user) = @_;
    my $flag_settings = $AD_WARNINGS{$flag_name};
    my %valid_variants = map {$_ . $flag_settings->{postfix} => 1} @{ $flag_settings->{variants} };
    
    if ($internal_user) {
        # для внутренних пользователей, для соблюдения правила обратной совместимости
        # означает "снять возрастную метку"
        $valid_variants{$RESET_VALUE} = 1;
    }
    
    return \%valid_variants;
}

=head2 _get_related_flags($flag_name)

    Получает связанные флаги.
    Возвращает ссылку на массив

=cut

sub _get_related_flags($){
    
    my $flag_name = shift;
    return $AD_WARNINGS{$flag_name} ? $AD_WARNINGS{$flag_name}{related_flags} || [] : []; 
}


=head2 set_flag($flags, $new_flag, $value)

    Установить/добавить новое значение флага для баннера
    Изменять набор флагов можно только в модерации, через этот метод можно менять только значение возрастных меток у флагов

        $flags - текущее значение флагов(хеш)
        $new_flag - имя флага
        $value - значение флага
        
    Возвращает
        ($new_flags, $is_changed)
    
=cut

sub set_flag {
    my ($flags, $new_flag, $value, $force) = @_;

    # $flags is flat hash
    my %new_flags = %$flags;
    my $new_flag_info = $AD_WARNINGS{$new_flag};
    
    if (!$force && (!exists $flags->{$new_flag} || !$new_flag_info->{age_limit} || $value == -1)) {
        return (\%new_flags, '');
    }

    if ($new_flag_info && !$new_flag_info->{is_common_warn}) {
        delete $new_flags{$_} for grep {exists $AD_WARNINGS{$_} && !defined $AD_WARNINGS{$_}{is_common_warn}} keys %AD_WARNINGS;
    }

    if (defined $value && $value == -1) {
        my $related_flags = $new_flag_info->{related_flags} || [];
        push @$related_flags, $new_flag;
        delete @new_flags{@$related_flags};
    } else {
        $new_flags{$new_flag} = $value // 1;
    }

    return \%new_flags, (serialize_banner_flags_hash($flags) // '') ne (serialize_banner_flags_hash(\%new_flags) // '');
}

=head2 normalize_flag_value

    приводит флаг в "нормальный" вид, то есть из всех допустимых написаний приводит к каноническому.
    Например, age => "16" приводит к "16+".

=cut
sub normalize_flag_value {
    my ($flag, $value) = @_;
    $value = $RESET_VALUE unless defined $value && $value !~ /^\s*$/;
    return $value unless defined $flag && exists $AD_WARNINGS{$flag}->{variants};
    my ($val, $postfix) = $value =~ /^(\d+)(.*)/;
    return $value unless defined $val;
    $postfix = $AD_WARNINGS{$flag}->{postfix} unless $postfix && $AD_WARNINGS{$flag}->{postfix} eq $postfix;
    return sprintf("%d%s", $val, $postfix);
}

1;


