package BannerImages;

=head1 DESCRIPTION

    Модуль для работы с картинкой в баннере

=cut

=head1 INFO

    Жизненный путь картинки от момента загрузки пользователем до отправки в БК выглядит так: 
    * Пользователь загружает произвольное изображение 
    * Оно сохраняется в MDS (сам файл) и ppc.banner_images_uploads (хеш)
    * Изображение уменьшается под размер редактора, сохраняется (тоже в MDS), и отдается в
      интерфейс на редактирование 
        ** вместе с изображением отдаются размеры исходной картинки 
    * По окончании редактирования из интерфейса приходят
      координаты прямоугольника (относительно исходной картинки, именно для этого
      мы отдавали исходные размеры).  
    * Вырезаная картинка сохраняется в Аватарницу
    * Исходная картинка и картинка для редактора удаляются (скриптом, спустя некоторое время)

    Для идентификации картинки служит ее md5_b64, по которому можно показать картинку в интерфейсе (http://.../images/$md5)

    Неочевидная особенность ImageMagick: Если попробовать изменить размер картинки 1х1 вот так:
        $im->Resize(geometry => '100x68'), то картинка получится 68х68 -- IM сохраняет пропорции
    Чтобы быть более убедительным по отношению к IM, надо на него кричать:
        $im->Resize(geometry => '100x68!'); # note the ! at end

    Сообщения "Произошла ошибка %d" - это внутренние ошибки, которые теоретически никогда не должны случиться.

=cut

use Direct::Modern;

use Settings;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::I18n;
use Yandex::HashUtils qw/hash_merge hash_cut hash_copy/;
use Yandex::ListUtils qw(nsort);
use Yandex::SendMail;
use Direct::Storage;
use Direct::Avatars;

use ShardingTools;
use Tools;
use MTools;
use List::MoreUtils qw/uniq any/;
use List::Util qw/max min first/;
use List::UtilsBy qw/partition_by/;
use PrimitivesIds;
use BannerImages::Pool;
use Direct::Validation::BannersMobileContent;
use Tools qw/log_cmd/;
use LogTools qw//;
use Property;
use JavaIntapi::GenerateObjectIds;

use Image::Magick qw//;
use Image::ExifTool qw/ImageInfo/;
use JSON;
use Try::Tiny;
use POSIX qw/ceil/;

use base qw/Exporter/;
our @EXPORT = qw/
    banner_check_image
    crop_resize_image
    resize_image_for_edit
    get_campaign_images
    get_banner_image
    get_banners_images
    banner_assign_image
    mass_banner_assign_image
    mass_banner_remove_image
    banner_image_check_size
    banner_image_check_min_size
    banner_wide_image_check_min_size
    banner_save_image
    banner_prepare_save_image
    banner_image_get_original
    get_image_hash_by_url
/;

our $MAX_IMAGE_FILE_SIZE_IN_MEGABYTES = 10;
our $MAX_IMAGE_FILE_SIZE = ceil($MAX_IMAGE_FILE_SIZE_IN_MEGABYTES * 1024 * 1024 / 3) * 4; # максимальный размер картинки - 10mb base64

our $MAX_IMAGEAD_FILE_SIZE_IN_KILOBYTES = 120;
our $MAX_IMAGEAD_FILE_SIZE = $MAX_IMAGEAD_FILE_SIZE_IN_KILOBYTES * 1024;

our $MAX_IMAGE_SIZE = 5_000; # pixels
our $BANNER_IMAGE_LIMIT = 3_000; # пережимаем картинки

our $EDITOR_SIZE_WIDTH = 400;
our $EDITOR_SIZE_HEIGHT = 300;

our $EDITOR_SIZE = $EDITOR_SIZE_WIDTH.'x'.$EDITOR_SIZE_HEIGHT;

# допустимые размеры и соотношения сторон
our $BANNER_IMAGE_MIN_SIZE = 450;
our $MIN_LOGO_SIZE = 80;
our $BANNER_IMAGE_RATIO = 4/3;
our $BANNER_IMAGE_WIDE_RATIO = 16/9;
our $BANNER_IMAGE_WIDE_MIN_WIDTH_SIZE = 1080;
our $BANNER_IMAGE_WIDE_MIN_HEIGHT_SIZE = int($BANNER_IMAGE_WIDE_MIN_WIDTH_SIZE/$BANNER_IMAGE_WIDE_RATIO);

# названия форматов в аватарнице
# в процедуре, которая сохраняет данные в banner_images_formats, есть фильтр, который
# не даёт записывать в поле ограниченного размера форматы, о которых мы (Директ) не знаем.
# прежде чем добавлять в этот массив новый формат, убедись, что сериализованные данные
# умещаются в размер, заданный для ppc.banner_images_formats.formats
# (а влезает туда ~1024/40 = 25 записей)
our %AVATARS_FORMAT = (
    small =>    [qw/ x90 y90 x80 y80 y65 y110 y129 y150 x150 /],
    regular =>  [qw/ x90 y90 x80 y80 y65 y110 y129 y150 x150 y180 y160 x180 x160 y300 x300 x450 y450 /],
    wide =>     [qw/ wx1080 wy300 wy150 wx150 wx600 wx300 /],
    logo =>     [qw/orig/],
);

our %RESTRICT_IMAGE_TYPES = (
    mobile_content => \@Direct::Validation::BannersMobileContent::ALLOWED_IMAGE_TYPES,
    (map {($_ => [])} qw/mcb geo wallet performance/),
);


our $MDS_IMAGE_NAMESPACE ||= 'direct';



sub _is_avatar_format_supported {
    my ($type, $format) = @_;
    state $type_cache = {};

    croak "Unknown type $type"  if !$AVATARS_FORMAT{$type};
    my $format_hash = $type_cache->{$type} //= { map {($_ => 1)} @{$AVATARS_FORMAT{$type}} };

    return exists $format_hash->{$format};
}


# обертка над MTools::check_image
# прикладной код про картинки + вычисление имени картинки (хеша)
sub banner_check_image
{
    my ($idata, $im) = @_;
    my $iinfo = check_image($idata);
    if (defined $im) {
        # check_image не всегда возвращает правильные размеры, почему-то иногда они равны размерам до ресайза :(
        $iinfo->{$_} = $im->Get($_) for qw/width height/;
    }
    unless ($iinfo->{contentType} and $iinfo->{contentType} =~ m!image/(jpeg|png|gif)! ) {
        $iinfo->{error} = iget("Недопустимый тип файла изображения, используйте графические форматы GIF, JPEG, PNG");
    }
    return $iinfo;
}

=head2 banner_save_image

Сохранить баннерную картинку
Параметр - объект от banner_check_image

прикладной код

=cut

sub banner_save_image
{
    my ($img, $uid) = @_;
    my $storage = Direct::Storage->new();
    $storage->save('banner_images_uploads', \$img->{img}, ClientID => get_clientid(uid => $uid));
}

=head2 banner_prepare_save_image

Сохранить баннерную картинку, предварительно преобразовав к нужному формату
Параметр - бинарное представление картинки
Опция — keep_jpeg=1|0 — оставлять формат JPEG у JPEG-картинок
Возвращает хеш с описанием картинки или ошибку { error => '...' }

Прикладной код

=cut

sub banner_prepare_save_image
{
    my ($idata, $shard_param, %O) = @_;

    my $check = banner_check_image($idata);
    $check->{error} = iget("Размер файла больше допустимого (%d)", $MAX_IMAGE_FILE_SIZE) if (!$check->{error} && $check->{size} > $BannerImages::MAX_IMAGE_FILE_SIZE);
    return $check if $check->{error};

    my $image_type = banner_image_check_size($check->{width}, $check->{height});
    $check->{error} = iget('Размер изображения некорректен') if (!$check->{error} && (!$image_type || $image_type eq 'small'));
    return $check if ($check->{error});

    my $im = _im_open_data($idata);

    return { error => iget('Произошла ошибка %d', 6) } unless $im;

    if ($im->[1]) {
        # анимированная картинка - берем первый кадр
        $im = $im->[0];
    }

    my $data = _im_get_data($im, keep_jpeg => $O{keep_jpeg});

    $check = banner_check_image($data, $im);
    my $save_result = eval { save_image_avatars($check, $shard_param) }
        || { error => iget('Ошибка при сохранении файла') };

    return $save_result  if $save_result->{error};
    return hash_merge $check, $save_result->{result};
}


=head2 resize_image_for_edit

    Функция изменяет размер изображения под размер редактора, 
    и сохраняет его для последующего использования
    Параметры:
        $img - объект от banner_check_image с описанием картинки, которую будем редактировать (исходной)
        $cid - id кампании
        $filename - имя файла или URL
        $uid - id пользователя
=cut

sub resize_image_for_edit
{
    my ($img, $cid, $filename, $uid) = @_;
    $filename ||= 'unknown';
    my $upload_id = do_insert_into_table(PPC(cid => $cid), 'banner_images_uploads', { 
        hash_orig => $img->{md5}, 
        cid => $cid, 
        name => $filename, 
        date_added__dont_quote => 'now()' 
    }, on_duplicate_key_update => 1, key => 'id');
    my $im = _im_open_data($img->{img});
    return { error => iget("Произошла ошибка %d", 6) } unless $im;
    if ($im->[1]) {
        $im = $im->[0];
    }
    # перегоняем картинку в PNG, сохраняем первый кадр анимации
    my $data = _im_get_data($im);
    my $iinfo = banner_check_image($data, $im);
    return $iinfo if $iinfo->{error};
    if ($iinfo->{width} > $EDITOR_SIZE_WIDTH or $iinfo->{height} > $EDITOR_SIZE_HEIGHT) {
        # картинка не вписывается в редактор? подгоняем размеры
        $im->Scale(geometry => $EDITOR_SIZE);
        $data = _im_get_data($im);
        $iinfo = banner_check_image($data, $im);
        if ($iinfo->{width} > $EDITOR_SIZE_WIDTH or $iinfo->{height} > $EDITOR_SIZE_HEIGHT) {
            die "failed to resize image (source image - '$img->{md5}'";
        }
        $iinfo->{upload_id} = $upload_id;
        banner_save_image($iinfo, $uid);
        return $iinfo if $iinfo->{error};
        do_update_table(PPC(cid => $cid), 'banner_images_uploads', { hash_edit => $iinfo->{md5} }, where => { id => $upload_id });
    }
    else {
        banner_save_image($iinfo, $uid);
        do_update_table(PPC(cid => $cid), 'banner_images_uploads', { hash_edit => $iinfo->{md5} }, where => { id => $upload_id });
        $iinfo->{upload_id} = $upload_id;
    }
    return $iinfo;
}

=head2 banner_image_check_size

Проверить размеры картинки. Параметры - ширина и высота

Возвращает код типа картинки, или undef для неподходящих размеров

=cut

sub banner_image_check_size
{
    my ($width, $height) = @_;

    return if max($width, $height) > $MAX_IMAGE_SIZE;
    return 'wide' if _is_valid_wide_image_size($width, $height);
    return 'small' if _is_valid_small_image_size($width, $height);
    return 'regular' if _is_valid_regular_image_size($width, $height);
    return 'logo'  if _is_valid_logo_size($width, $height);
    return;
}


=head2 are_logos_enabled

Проверить, что доступ из апи к лого включен

=cut

sub are_logos_enabled {
    state $logo_in_api_is_enabled = Property->new('logo_in_api_is_enabled');
    return $logo_in_api_is_enabled->get(60);
}

sub _is_valid_logo_size {
    my ($width, $height) = @_;
    return are_logos_enabled() && $width == $height && $width == $MIN_LOGO_SIZE;
}

sub _is_valid_regular_image_size {
    my ($width, $height) = @_;

    return if !banner_image_check_min_size($width, $height);
    return if max($width/$height,$height/$width) > $BANNER_IMAGE_RATIO;
    return 1;
}


sub _is_valid_small_image_size {
    my ($width, $height) = @_;

    return if !(min($width, $height) >= 150 && min($width, $height) < 450);
    return if max($width/$height,$height/$width) > $BANNER_IMAGE_RATIO;
    return 1;
}


sub _is_valid_wide_image_size {
    my ($width, $height) = @_;

    return if $width < $BANNER_IMAGE_WIDE_MIN_WIDTH_SIZE;
    return if abs($width / $BANNER_IMAGE_WIDE_RATIO - $height) > 0.51; # сравниваем с точностью до пол-пиксела
    return 1;
}


=head2 banner_image_check_min_size

Проверить что картинка имеет минимально допустимые размеры
Параметры - ширина и высота

=cut

sub banner_image_check_min_size {
    my ($width, $height) = @_;

    return min($width, $height) >= $BANNER_IMAGE_MIN_SIZE;
}


=head2 banner_wide_image_check_min_size

Проверить что картинка имеет минимально допустимые размеры для мобильных объявлений
Параметры - ширина и высота

=cut

sub banner_wide_image_check_min_size {
    my ($width, $height) = @_;

    return $width >= $BANNER_IMAGE_WIDE_MIN_WIDTH_SIZE && $height >= $BANNER_IMAGE_WIDE_MIN_HEIGHT_SIZE;
}


=head2 is_image_type_allowed_for_campaign

Проверить, что для кампании допустим тип картинок

=cut

sub is_image_type_allowed_for_campaign {
    my ($camp_type, $image_type) = @_;

    my $restrict = $RESTRICT_IMAGE_TYPES{$camp_type};
    return 1  if !$restrict;

    return any {$_ eq $image_type} @$restrict;
}




=head2 crop_resize_image

    Функция вырезает указанный прямоугольник из *исходного* изображения (загруженного пользователем)
    и приводит этот прямоугольник к нужному формату
    Параметры:
        $cid - id кампании
        $md5 - хеш от картинки, которую будем редактировать
        $x  --
        $y  --
        $x2 --
        $y2 -- координаты прямоугольника, который будет вырезан
        $upload_id - id из banner_images_uploads

=cut

sub crop_resize_image
{
    my ($cid, $uid, $md5, $x, $y, $x2, $y2, $upload_id) = @_;
    my ($width, $height) = ($x2-$x, $y2-$y);

    my $image_type = banner_image_check_size($width, $height);
    return { error => iget("Размер изображения некорректен") }  if !$image_type or $image_type eq 'small' or $image_type eq 'logo';

    my $im = _im_open_data(_im_data_by_hash(get_clientid(uid => $uid), $md5));
    return { error => iget("Произошла ошибка %d", 2) } unless $im;
    my $orig_width = $im->Get('width');
    my $orig_height = $im->Get('height');
    if ($orig_width < $x2 or $orig_height < $y2) {
        return { error => iget("Произошла ошибка %d", 3) };
    }
    if ($im->[1]) {
        # анимированная картинка - берем первый кадр
        $im = $im->[0];
    }
    # пользователь в редакторе работает с уменьшенной копией, и координаты пересчитываются самим редактором
    $im->Set(page => '0x0+0+0');
    $im->Crop(geometry => sprintf "%dx%d+%d+%d", ($width), ($height), $x, $y); # размер прямоугольника+верхне-левая точка
    $im->Set(page => '0x0+0+0');

    # ресайзим до лимита
    my $scale = min($BANNER_IMAGE_LIMIT/$height, $BANNER_IMAGE_LIMIT/$width);
    if ($scale < 1) {
        my $scale_width = int($width*$scale + 0.5);
        my $scale_height = int($height*$scale + 0.5);
        my $banner_size = sprintf "%dx%d!", $scale_width, $scale_height;
        $im->Scale(geometry => $banner_size);
    }

    my $data = _im_get_data($im);
    my $iinfo = banner_check_image($data, $im);

    my $save_result = save_image_avatars($iinfo, { cid => $cid });
    return $save_result if $save_result->{error};

    do_update_table(PPC(cid => $cid), 'banner_images_uploads', {
            hash_final => $iinfo->{md5},
        }, 
        where => { cid => $cid, hash_orig => $md5, id => $upload_id } 
    );

    my $res = {
        image => $iinfo->{md5},
        image_type => $image_type,
        image_width => $iinfo->{width},
        image_height => $iinfo->{height},
        mds_group_id => $save_result->{result}->{mds_group_id},
        namespace => $save_result->{result}->{namespace},
    };
    return $res;
}

=head2 save_image_avatars

Сохранить картинку в аватарнице и записать размеры картинки в базу

Что-то типа Direct::Storage

=cut

sub save_image_avatars
{
    my ($iinfo, $shard_param) = @_;
    my $image_hash = $iinfo->{md5};

    my $image_type = banner_image_check_size($iinfo->{width}, $iinfo->{height});
    if (!$image_type){
        send_alert(Carp::longmess(sprintf( 'wrong image size: %sx%s', $iinfo->{width}, $iinfo->{height} )), 'wrong image size at save_image_avatars');
        return { 
            error => iget('Ошибка при обработке файла'),
            error_state => 'wrong_image',
        };
    }

    # проверяем, есть ли уже картинка в новой аватарнице
    my $old_record = get_one_line_sql(PPC(%$shard_param), [
            "select image_hash, image_type, namespace, mds_group_id from banner_images_formats",
            where => {image_hash => $image_hash},
        ]);
    return {result => $old_record}  if $old_record;

    my $avatars = Direct::Avatars->new(namespace => $MDS_IMAGE_NAMESPACE);
    my $res = $avatars->put($image_hash => $iinfo->{img});

    my $meta = $res->{meta};
    if (defined $meta) {
        delete $meta->{NNetFeatures};
        delete $meta->{NeuralNetClasses};
    }

    # сохраняем ответ аватарницы (json), чтобы потом записать его в базу banner_images_formats.mds_meta
    my $mds_meta = to_json($res, {canonical => 1});

    my $size_hash = $res->{sizes};
    my $orig = $size_hash->{orig};
    croak "No 'orig' format for $image_hash"  if !$orig;

    if ($orig->{width} != $iinfo->{width} || $orig->{height} != $iinfo->{height}) {
        die "mds returned invalid size $orig->{width}x$orig->{height}, expected $iinfo->{width}x$iinfo->{height}";
    }

    my %sizes =
        map {($_ => hash_cut $size_hash->{$_}, qw/width height smart-center/)}
        grep { _is_avatar_format_supported($image_type => $_) }
        keys %$size_hash;

    validate_avatars_sizes($image_type => \%sizes);

    die "Can not save image (hash $image_hash) without formats" if !%sizes;

    my $json = to_json(\%sizes, {canonical => 1});

    my $db_record = {
            image_hash => $image_hash,
            mds_group_id => $res->{'group-id'},
            namespace => $MDS_IMAGE_NAMESPACE,
            image_type => $image_type,
            width => $orig->{width},
            height => $orig->{height},
            formats => $json,
            avatars_host => $res->{avatars_host},
            mds_meta => $mds_meta,
        };
    do_insert_into_table(PPC(%$shard_param), 'banner_images_formats', $db_record,
        on_duplicate_key_update => 1,
    );

    return {result => hash_cut $db_record, qw/image_hash image_type namespace mds_group_id/};
}

=head2 validate_avatars_sizes

    Проверить, что все размеры картинок совпадают с ожидаемыми.
    Ловим ситуацию, когда картинка y150 имеет высоту 149 px

    В случае несовпадения умирает

=cut

sub validate_avatars_sizes
{
    my ($type, $sizes) = @_;
    for my $format_id (keys %$sizes) {
        my $size = $sizes->{$format_id};
        next if !_is_avatar_format_supported($type => $format_id);
        if (!is_valid_format($format_id => $size)) {
            die "invalid format for $format_id: $size->{width} x $size->{height}";
        }
    }
    return;
}


=head2 is_valid_format($id, $img_info)
    
    Проверить, что параметры картинки удовлетворяют формату
    $id - описание формата, например 'y150'
    $img_info - { width => $w, height => $h }

=cut

sub is_valid_format
{
    my ($id, $img_info) = @_;

    if ($id eq 'orig') {
        return 1;
    }

    my ($axis, $px) = ($id =~ /(\w)(\d+)/);

    unless ($axis && $px) {
        die "invalid image format '$id'";
    }
    my %axis2side = (
        x => 'width',
        y => 'height',
    );
    unless ($img_info->{ $axis2side{$axis} }) {
        return 0;
    }
    return $img_info->{ $axis2side{$axis} } == $px ? 1 : 0;
}

=head2 banner_image_get_original

Получить исходное изображение (md5) и имя файла/url по upload_id

=cut

sub banner_image_get_original
{
    my ($cid, $upload_id) = @_;
    return get_one_line_sql(PPC(cid => $cid), "select hash_orig, name from banner_images_uploads where cid = ? and id = ?", $cid, $upload_id);
}

=head2 get_campaign_images

    Получить все картинки кампании
    Возвращает массив из md5

=cut

sub get_campaign_images
{
    my ($cid) = @_;
    return get_all_sql(PPC(cid => $cid), [
            "select bim.image_hash as image, bimp.name, bimf.image_type, bimf.mds_group_id, bimf.avatars_host
            from banner_images bim 
            join banners b on bim.bid = b.bid
            join campaigns c on c.cid = b.cid
            join users u on u.uid = c.uid
            join banner_images_pool bimp on bimp.image_hash = bim.image_hash and bimp.ClientID = u.ClientID
            join banner_images_formats bimf on bimf.image_hash = bim.image_hash
            ", 
            where => { 'c.cid' => SHARD_IDS, 'bim.statusShow' => 'Yes' }, 'GROUP BY bim.image_hash order by max(date_added) desc']);
}


=head2 get_campaign_available_images_mass

Получить все картинки, доступные для кампаний

=cut

sub get_campaign_available_images_mass
{
    my ($cids) = @_;

    return {}  if !@$cids;

    state $type_by_type_sql = +{
        map {($_ => sql_condition({image_type => $RESTRICT_IMAGE_TYPES{$_}}))}
        keys %RESTRICT_IMAGE_TYPES
    };
    state $type_by_type_case_sql = sql_case(
        'c.type' => $type_by_type_sql,
        default__dont_quote => sql_condition({image_type => [qw/regular wide/]}),
        dont_quote_value => 1,
    );

    my $images = get_all_sql(PPC(cid => $cids), [
            "select cid, image_hash as image, bimp.name, bimf.image_type, 
                    bimf.width AS image_width, bimf.height AS image_height,
                    bimf.mds_group_id
            FROM campaigns c
            JOIN users u USING(uid)
            JOIN banner_images_pool bimp on bimp.ClientID = u.ClientID
            JOIN banner_images_formats bimf USING(image_hash)", 
            where => {
                cid => $cids,
                _TEXT => $type_by_type_case_sql,
            },
            'order by create_time desc',
        ]);

    my %images_by_cid = partition_by {$_->{cid}} @$images;
    delete $_->{cid} for map {@$_} values %images_by_cid;
    return \%images_by_cid;
}


=head2 get_banner_image

    Получить картинку (md5) для данного баннера
    Параметры: $bid

=cut

sub get_banner_image
{
    my ($bid) = @_;
    my ($selected_banners_num, $data) = get_banners_images(bid => $bid);
    return $data->[0]{image_hash};
}

=head2 get_banners_images

    Возвращает массив структур вида {bid => 123, image_hash => '12af3', statusModerate => 'Yes', cid => 345}
    Параметры: 
        bid - массив идентификаторов объявлений, или один идентификатор
        cid - массив идентификаторов кампаний, или один идентификатор
        uid - массив идентификаторов главных представителей клиентов, или один идентификатор
        image_hash - массив хэшей картинок, или один хэш
        statusModerate
        limit
        offset

=cut

sub get_banners_images
{
    my (%OPT) = @_;

    my $where = {};

    my $sql = "SELECT SQL_CALC_FOUND_ROWS 
               bi.bid, 
               bi.image_hash, 
               bi.statusModerate, 
               b.cid
               FROM banner_images bi
               JOIN banners b USING (bid)
               JOIN campaigns c USING (cid)";

    $where->{'bi.bid'} = $OPT{bid} if exists $OPT{bid};
    $where->{'bi.image_hash'} = $OPT{image_hash} if exists $OPT{image_hash};
    $where->{'b.cid'} = $OPT{cid} if defined $OPT{cid};
    $where->{'c.uid'} = $OPT{uid} if defined $OPT{uid};
    my %shard = choose_shard_param($where, [qw/uid cid bid/]);

    if (scalar keys %$where == 0) {
        die "get_banners_images: called without condition";
    }

    $where->{'bi.statusShow'} = 'Yes';
    $where->{'c.statusEmpty'} = "No";
    $where->{'bi.statusModerate'} = $OPT{statusModerate} if defined $OPT{statusModerate};

    my $limitoffsetsql = '';
    if ($OPT{limit}) {
        $limitoffsetsql = "limit $OPT{limit} offset ".($OPT{offset}||0);
    }

    my $data = get_all_sql(PPC(%shard), [$sql, where => $where, "order by bi.image_id $limitoffsetsql"]);

    my $selected_banners_num = select_found_rows(PPC(%shard));

    return ($selected_banners_num, $data);
}

=head2 banner_assign_image

   Назначить баннеру картинку из ранее загруженных
   Параметры:
    cid - номер кампании
    bid - номер баннера
    md5 - хеш картинки
   Опционально:
    status - статус модерации
    opt - { name - имя файла по-умолчанию (если ничего не найдется для заданного хеша в БД),
            ClientID - ID клиента, баннеру которого привязывается картинка,
            skip_pool - не добавлять картинку в пул клиента (либо это не требуется, либо она уже была добавлена ранее) }

=cut

sub banner_assign_image
{
    my ($cid, $bid, $md5, $status, $opt) = @_;
    $opt ||= {};
    
    mass_banner_assign_image({
        bid            => $bid, 
        cid            => $cid,
        image_hash     => $md5, 
        statusModerate => $status,
        %{hash_cut $opt, qw/name ClientID skip_pool/} 
    });
}

=head2 mass_banner_assign_image

    Назначение текстовому баннеру картинки, с загрузкой это картинки в пул клиента.
    На вход подается массив хэшей вида:
    {
        bid =>
        cid => 
        ClientID => 
        image_hash =>
        statusModerate => Ready|New New - сохранить картиночный баннер как черновик, Ready - отправить на модерацию 
        
        name => имя картинки 
        skip_pool => 1 - не добавлять картинку в пул клиента
    }
    
    Обязательные поля bid, image_hash, statusModerate
    
    Поля cid, ClientID могут быть вычислены внутри функции (т.е. можно их не передавать)
    Если name не заданно, оно будет искаться среди загруженных картинок по cid или из пула картинок клиента (ClientID)
    
=cut

# заполнение недостающих полей { cid => , ClientID => }
sub _images_fill_cid_clientid {
    my $images = shift;

    my (@clientid_by_cid, @cid_by_bid);
    foreach my $img (@$images) {
        push @cid_by_bid, $img unless $img->{cid}; 
        # cid в $img появится позже если его нет
        push @clientid_by_cid, $img unless $img->{ClientID};
    }

    if (@cid_by_bid) {
        my $bid2cid = get_bid2cid(bid => [map {$_->{bid}} @cid_by_bid]);
        $_->{cid} = $bid2cid->{$_->{bid}} foreach @cid_by_bid;    
    }

    if (@clientid_by_cid) {
        my $cid2clientid = get_cid2clientid(cid => [map {$_->{cid}} @clientid_by_cid]);
        $_->{ClientID} = $cid2clientid->{$_->{cid}} foreach @clientid_by_cid;
    }

    return;
}

# добавление картинок в пул клиента
sub _add_images_to_pool {
    
    my ($shard, $images) = @_;
    
    my @without_name_images = grep {!$_->{name}} @$images;
    if (@without_name_images) {
        
        my (@cids, @hashes);
        foreach (@without_name_images) {
            push @cids, $_->{cid};
            push @hashes, $_->{image_hash};
        }
        my $names = get_hash_sql(PPC(shard => $shard), 
            ["SELECT CONCAT(cid, ';', hash_final), name FROM banner_images_uploads",
            WHERE => {cid => [uniq @cids], hash_final => \@hashes}]);
            
        my @find_in_pool;
        foreach my $img (@without_name_images) {
            my $key = $img->{cid} . ';' . $img->{image_hash}; 
            if (exists $names->{$key}) {
                $img->{name} = $names->{$key}; 
            } else {
                push @find_in_pool, $img;
            }
        }
            
        if (@find_in_pool) {

            my (@client_ids, @hashes);
            foreach (@find_in_pool) {
                push @client_ids, $_->{ClientID};
                push @hashes, $_->{image_hash};
            }
            (undef, my $added_imgs) = BannerImages::Pool::get_items(\@client_ids, \@hashes);
            my %pool_images;
            foreach (@$added_imgs) {
                $pool_images{$_->{ClientID} . ';' . $_->{image_hash}} = $_;
            }
            
            foreach my $img (@find_in_pool) {
                my $key = $img->{ClientID} . ';' . $img->{image_hash};
                if (exists $pool_images{$key}) {
                    $img->{name} = $pool_images{$key}->{name};
                    $img->{skip_pool} = 1;
                } else {
                    $img->{name} = $img->{image_hash};
                }
            }
        }
    }
    
    my @to_pool;
    foreach my $img (@$images) {
        next if $img->{skip_pool};
        push @to_pool, hash_cut $img, qw/ClientID image_hash name/;
    }
    BannerImages::Pool::add_items(\@to_pool) if @to_pool;
}

sub mass_banner_assign_image {
    my @images_to_save = @_;

    save_banner_images(\@images_to_save);

    my $bids = [map {$_->{bid}} @images_to_save];

    do_update_table(PPC(bid => $bids), 'banners',
        { statusBsSynced => 'No', LastChange__dont_quote => 'LastChange' },
        where => { bid => SHARD_IDS });

    my $adgroup_ids = get_pids(bid => $bids);
    do_update_table(PPC(pid => $adgroup_ids), 'phrases',
        { statusBsSynced => 'No', LastChange__dont_quote => 'LastChange' },
        where => { pid => SHARD_IDS});

    return;
}

=head2 save_banner_images(\@images)

    Сохраняет изображения и их привязки к объявлениям

=cut

sub save_banner_images {
    my $images_to_save = shift;

    _images_fill_cid_clientid($images_to_save);

    foreach_shard ClientID => $images_to_save, sub {
        my ($shard, $images) = @_;
        _save_banner_images_in_shard($shard, $images);
    };

    return;
}

sub _save_banner_images_in_shard {
    my ($shard, $images) = @_;

    _add_images_to_pool($shard, $images);
    my $bids = [map {$_->{bid}} @$images];
    my $bid2imageid = get_hash_sql(PPC(shard => $shard), 
        ["select bid, image_id from banner_images bim", where => { bid => $bids }]);

    my @values;
    my @fields = qw/image_id bid image_hash statusModerate statusShow/;
    for my $image (@$images) {
        my $image_prepared = hash_copy {}, $image, keys %$image;
        if (exists $bid2imageid->{$image->{bid}}) {
            $image_prepared->{image_id} = $bid2imageid->{$image->{bid}};
        } else {
            $image_prepared->{image_id} = get_new_id("bid", bid => $image->{bid});
        }
        $image_prepared->{statusShow} = "Yes";
        $image_prepared->{statusModerate} ||= "New";

        my @data_row;
        for (@fields) {
            push @data_row, $image_prepared->{$_};
        }
        push @values, \@data_row;
    }

    my $insert_sql = 'INSERT INTO banner_images ('.join(',', @fields).')
                      VALUES %s
                      ON DUPLICATE KEY UPDATE 
                            image_hash = values(image_hash),
                            statusModerate = values(statusModerate),
                            statusShow = values(statusShow)';

    do_mass_insert_sql(PPC(shard => $shard), $insert_sql, \@values);

    return;
}

=head2 mass_banner_remove_image

    Принимает список номеров баннеров для удаления у них картинки

    mass_banner_remove_image(\@bids);

=cut

sub mass_banner_remove_image
{
    my ($bids) = @_;

    my $images = get_all_sql(PPC(bid => $bids), ["
        SELECT bi.bid, b.cid, bi.image_id
        FROM banner_images bi
        INNER JOIN banners b ON bi.bid = b.bid
     ", WHERE => {
            'bi.bid' => SHARD_IDS,
            'bi.statusShow' => 'Yes',
        },
    ]);
    my @bids_filtered = map { $_->{bid} } @$images;

    # запись из таблицы не удаляем, если картинка крутилась в БК (для статистики)
    # они будут почищены скриптом protected/ppcRemoveBannerImages.pl
    do_update_table(PPC(bid => \@bids_filtered), 'banner_images', { statusShow => 'No' }, where => { bid => SHARD_IDS });
    do_update_table(PPC(bid => \@bids_filtered), 'banners', 
        { statusBsSynced => 'No', LastChange__dont_quote => 'LastChange' }, 
        where => { bid => SHARD_IDS });

    # с модерации удаляем на всякий случай все
    my $cid2bids;
    for my $item (@$images) {
        push @{$cid2bids->{$item->{cid}}}, $item->{bid};
    }

    my $cid2uid = get_cid2uid(cid => [keys %$cid2bids]);
    log_cmd({
        cmd => '_delete_banner_image',
        UID => $LogTools::context{UID}, uid => $cid2uid->{$_->{cid}},
        bid => $_->{bid}, cid => $_->{cid}, image_id => $_->{image_id} 
    }) for (@$images);

    # удаляем причину отклонения картинки модератором - нет картинки, нет причины.
    do_delete_from_table(PPC(bid => \@bids_filtered), 'mod_reasons', where => { id => SHARD_IDS, type => 'image' });

    # удалить версии объектов для модерации
    do_delete_from_table(PPC(bid => \@bids_filtered), 'mod_object_version', where => { obj_id => SHARD_IDS, obj_type => 'image'});
}

=head2 mass_copy_banner_image

   Копирование изображений, для copy_camp 
   %O - flags из copy_camp

=cut

sub mass_copy_banner_image
{
    my ($from_cid, $to_cid, $bid_from2to, %O) = @_;

    my @bids_from = nsort keys %$bid_from2to;
    my $bids_cnt = scalar(@bids_from);
    my $new_image_ids = JavaIntapi::GenerateObjectIds->new(object_type => 'banner',
            count => $bids_cnt, client_id => $O{client_id})->call();
    my %old_bid2new_image_id = map { $bids_from[$_] => $new_image_ids->[$_] } 0 .. $bids_cnt - 1;
    my $old_image_ad_ids = $O{image_ad_ids} // [];

    do_insert_select_sql( PPC(cid => $from_cid),
        'INSERT IGNORE INTO banner_images_formats(image_hash, image_type, width, height, formats, namespace, mds_group_id, avatars_host, mds_meta) VALUES %s',
        [  'SELECT DISTINCT i.image_hash, image_type, width, height, formats, namespace, mds_group_id, avatars_host, mds_meta
            FROM banner_images i JOIN banner_images_formats f
            ON i.image_hash = f.image_hash',
            WHERE => {bid => \@bids_from},
        ],
        dbw => PPC(cid => $to_cid)
    );

    my @banner_images_fields_to_copy = qw(
        image_hash
        name
        statusShow
    );
    my %banner_images_override = (
        image_id => \%old_bid2new_image_id,
        bid => $bid_from2to,
    );

    if ($O{copy_moderate_status}) {
        $banner_images_override{statusModerate__sql} = "IF(statusModerate IN ('Sent', 'Sending'), 'Ready', statusModerate)";
    }
    else {
        $banner_images_override{statusModerate} = 'New';
    }

    my ($banner_images_fields_str, $banner_images_values_str) = make_copy_sql_strings(\@banner_images_fields_to_copy, \%banner_images_override, by => 'bid');
    do_insert_select_sql(PPC(cid => $from_cid), "INSERT INTO banner_images ($banner_images_fields_str) VALUES %s", 
                                               ["SELECT $banner_images_values_str FROM banner_images", WHERE => { bid => \@bids_from }],
                                               dbw => PPC(cid => $to_cid));
    if ($from_cid != $to_cid) {
        # если копируем в рамках разных кампаний - есть шанс что они принадлежат разным клиентам, потому надо добавлять в пул
        my $from_client_id = $O{old_client_id};
        die "no old_client_id given" unless $from_client_id;
        my $to_client_id = $O{client_id};
        die "no client_id given" unless $to_client_id;
        if ($from_client_id != $to_client_id) {
            my $data = get_all_sql(PPC(cid => $from_cid), ["
                                        SELECT bimp.name, bimp.image_hash 
                                          FROM banner_images bim 
                                               JOIN banner_images_pool bimp on bimp.ClientID = ? and bimp.image_hash = bim.image_hash
                                    ", WHERE => { 'bim.bid' => \@bids_from } ], $from_client_id);
            if (@$old_image_ad_ids) {
                push @$data, @{get_all_sql(PPC(cid => $from_cid), [
                    "SELECT imp.name, imp.image_hash
                    FROM banner_images_pool imp
                    JOIN images im using(image_hash)",
                    WHERE => { image_id => [ map { $_->{image_id} } @$old_image_ad_ids ], ClientID => $from_client_id },
                ])};
            }

            # картинки мультибаннера
            push @$data, @{get_all_sql(PPC(cid => $from_cid), [
                "SELECT imp.name, imp.image_hash
                FROM banner_images_pool imp
                JOIN banner_multicards bm USING (image_hash)",
                WHERE => { 'bm.bid' => \@bids_from, 'imp.ClientID' => $from_client_id },
            ])};

            # при копировании между шардами нужно протащить форматы
            if (get_shard(ClientID => $from_client_id) != get_shard(ClientID => $to_client_id)) {
                my @hashes = uniq map {$_->{image_hash}} @$data;
                my $fields_str = "image_hash, image_type, width, height, formats, mds_group_id, namespace, avatars_host, mds_meta";
                do_insert_select_sql(PPC(cid => $from_cid),
                    "INSERT IGNORE INTO banner_images_formats ($fields_str) VALUES %s",
                    ["SELECT $fields_str FROM banner_images_formats", WHERE => {image_hash => \@hashes}],
                    dbw => PPC(cid => $to_cid),
                );
            }

            my $ids = JavaIntapi::GenerateObjectIds->new(object_type => 'images_pool',
                    count => scalar(@$data), client_id => $to_client_id)->call();
            my @rows = map { [shift(@$ids), $to_client_id, $_->{name}, $_->{image_hash} ] } @$data;
            do_mass_insert_sql(PPC(cid => $to_cid), 
                               "INSERT IGNORE INTO banner_images_pool (imp_id, ClientID, name, image_hash) VALUES %s",
                               \@rows
                );
        }
    }
}

#########################
# вспомогательные функции

=head2 _im_data_by_hash

=cut

sub _im_data_by_hash
{
    my ($client_id, $md5) = @_;
    my $storage = Direct::Storage->new();
    my $file = $storage->get_file('banner_images_uploads', ClientID => $client_id, filename => $md5);
    return $file->content;
}

=head2 _im_open_data

    получить объект ImageMagick из сырых данных

=cut

sub _im_open_data
{
    my ($data) = @_;
    return undef unless $data;
    my $im = new Image::Magick;
    my $error = $im->BlobToImage($data);
    if ($error) {
        warn "Failed to get image from data: $error\n";
        return undef;
    }
    return $im;
}

=head2 _im_get_data

    Из объекта ImageMagick сделать PNG data или оставить JPEG data

=cut

sub _im_get_data
{
    my ($im, %O) = @_;
    if (!$O{keep_jpeg} || uc($im->Get("magick")) ne 'JPEG') {
        $im->Set(magick => 'png');
        $im->Set(quality => 9); # max compression level
    }
    $im->Set('date:modify' => '');
    $im->Set('date:create' => '');
    my $data = $im->ImageToBlob();
    return $data;
}

=head2 get_image_relative_url

=cut

sub get_image_relative_url {
    my ($pool_item) = @_;

    my $image_hash = $pool_item->{image_hash} || $pool_item->{hash};
    return '' unless $image_hash;

    my $mds_group_id = $pool_item->{mds_group_id};
    croak "No image_type for $image_hash" if ! defined $pool_item->{image_type} || $pool_item->{image_type} eq '';
    croak "No mds_group_id for image_ad $image_hash"  if $pool_item->{image_type} eq 'image_ad' && !$mds_group_id;

    if ($mds_group_id) {
        my $namespace = $pool_item->{namespace}  or croak "No namespace for $image_hash";
        return "/images/$namespace/$mds_group_id/$image_hash";
    } else {
        return "/images/$image_hash";
    }
}


=head2 get_image_url

=cut

sub get_image_url {
    my ($pool_item, $opt) = @_;

    my $host = $opt->{host} || 'direct.yandex.ru';

    my $relative_url = get_image_relative_url($pool_item);
    return '' unless $relative_url;
    return "https://$host$relative_url";
}


=head2 get_hash_by_url

получить image_hash картинки по url (если он соответствует нашему внутреннему url картинок)

=cut

sub get_image_hash_by_url
{
    my $url = shift;

    state $namespace_re = join q{|} => qw/ direct direct-picture /;
    state $image_url_re = qr!
        ^https?://
        [^/]+\.yandex\.(?:ru|ua|kz|by|com|com\.tr)
        /images
        (?:/($namespace_re)/(\d+))?
        /([^/]{22})
        (?:/\w*|$)
    !xms;
    my ($namespace, $mds_group_id, $hash) = $url =~ $image_url_re;
    return if !$hash;

    return ($namespace, $mds_group_id, $hash);
}

1;
