package BannerImages::Pool;

=head1 DESCRIPTION
    
    Модуль для работы с пулом картинок клиента
    Работа с таблицей banner_images_pool
    
=cut 

use strict;
use warnings;
use utf8;

use List::MoreUtils qw(any none uniq);
use List::Util qw/max min sum/;
use List::UtilsBy qw/partition_by/;

use Yandex::DBTools;
use Yandex::DBShards;
use BannerImages::Queue;

use Settings;

use ShardingTools;
use Primitives;
use LogTools;
use JavaIntapi::GenerateObjectIds;

use JSON qw/from_json/;

# минимальное ограничение на пул (формула MINIMUM_IMAGES_IN_POOL + 2 * active_banners_num)
our $MINIMUM_IMAGES_IN_POOL = 1000;

=head2 add_items($items)

    метод для добавления картинок в пул клиента
    Параметры:
        $items - массив картинок [], каждая картинка - это
                { ClientID => ID клиента,
                  name => имя картинки/файла,
                  image_hash => md5-хеш картинки }
        
    Возвращает:
        Массив [] исходных картинок с добавленным полем imp_id - ID картинки в пуле 
    
=cut

sub add_items {

    my ($items) = @_;

    my %clientid2items = partition_by {$_->{ClientID}} @$items;
    foreach my $client_id (keys %clientid2items) {

        my $images = $clientid2items{$client_id};  
        my $new_ids = JavaIntapi::GenerateObjectIds->new(object_type => 'images_pool',
                count => scalar(@$images), client_id => $client_id)->call();
        foreach (@$images) {
            $_->{imp_id} = shift @$new_ids;
        }
        do_mass_insert_sql(PPC(ClientID => $client_id),
            'INSERT INTO banner_images_pool (imp_id, ClientID, name, image_hash)
            VALUES %s ON DUPLICATE KEY UPDATE name = VALUES(name)',
            [map { [ $_->{imp_id}, $client_id, $_->{name}, $_->{image_hash} ] } @$images]);

        foreach my $img (@$images) {
            LogTools::log_messages('BannerImagesPool', "add/renew item: $img->{imp_id}\t$img->{ClientID}\t$img->{image_hash}");
        }
    }

    return $items;
}

=head2 delete_items

    Удаление элементов из пула, по идентификаторам, на вход - ссылка на массив идентификаторов.

=cut

sub delete_items {
    my ($imp_ids) = @_;
    my $cnt = do_delete_from_table(PPC(banner_images_pool_id => $imp_ids), 'banner_images_pool', where => { imp_id => SHARD_IDS });
    delete_shard('banner_images_pool_id' => $imp_ids);
    return $cnt;
}

=pod get_items

    Получить картинки из пула клиента, опции:
        with_assigned_info => получить информацию, привязана ли картинка хотя бы к одному баннеру клиента
        with_size => получить информацию о типе и размерах картинки
        clientid2chiefuid => нужно для получения информации по привязке
        assigned => 1 | 0 привязаны или нет, если не передано, то все
        source => direct получить только картинки, загруженные через директ
        limit
        offset

=cut

sub get_items {
    my ($clientids, $image_hashes, %O) = @_;
    die("Incorrect input parameters")
        unless (ref $clientids eq 'ARRAY'
                && (! $image_hashes || ref $image_hashes eq 'ARRAY')
        );
    my @result = ();

    my $where = {};
    $where->{image_hash} = $image_hashes if $image_hashes;

    $where->{image_type__not_in} = $O{exclude_types} if $O{exclude_types};

    $where->{source} = $O{source} if $O{source};

    my @fields = qw/bip.imp_id bip.ClientID bip.image_hash bip.name bif.mds_group_id bif.image_type bif.namespace/;
    push @fields, 'bif.formats'  if $O{with_size};

    # тут inner join, чтобы не отдавались картинки с потерянными форматами
    # (вообще такого быть не должно, но пока встречается)
    my $pool = get_all_sql(PPC(ClientID => $clientids), [
        SELECT => join(', ' => @fields),
        "FROM banner_images_pool bip
        JOIN banner_images_formats bif USING (image_hash)",
        where => {ClientID => SHARD_IDS, %$where}, "ORDER BY bip.imp_id"
    ]);

    if ($O{with_size}) {
        foreach my $item (@$pool) {
            if ($item->{formats}) {
                $item->{formats} = from_json($item->{formats});
            }
        }
    }

    if ($O{with_assigned_info}) {
        my $chiefuids;
        if (scalar keys %{$O{clientid2chiefuid}}) {
            $chiefuids = [uniq values %{$O{clientid2chiefuid}}];
        } else {
            die("Need chief uid");
        }
        my $assigned_where = {};
        $assigned_where->{"c.uid"} = $chiefuids;
        $assigned_where->{image_hash} = $image_hashes if $image_hashes;
        $assigned_where->{"c.statusEmpty"} = "No";
        # $assigned_where->{"bi.statusShow"} = "Yes";

        my @shard = choose_shard_param($assigned_where, ['uid'], set_shard_ids => 1);

        my $assigned_hash;
        my $assigned = get_all_sql(PPC(@shard), [
                        "SELECT c.uid, bid, image_hash 
                        FROM banner_images bi 
                        LEFT JOIN banners b USING (bid)
                        LEFT JOIN campaigns c using (cid)",
                        where => { %$assigned_where, 'bi.statusShow' => 'Yes' },
                        'UNION ALL',
                        "SELECT c.uid, bid, image_hash
                        FROM images bi
                        LEFT JOIN banners b USING (bid)
                        LEFT JOIN campaigns c ON b.cid = c.cid",
                        where => $assigned_where,
                        'UNION ALL',
                        "SELECT c.uid, bid, image_hash
                        FROM banner_logos bl
                        LEFT JOIN banners b USING (bid)
                        LEFT JOIN campaigns c ON b.cid = c.cid",
                        where => $assigned_where,
                        ]);

        for my $item (@$assigned) {
            $assigned_hash->{$item->{uid}}{$item->{image_hash}} = 1;
        }

        my @new_pool;

        for my $item (@$pool) {
            $item->{assigned} = $assigned_hash->{$O{clientid2chiefuid}->{$item->{ClientID}}}{$item->{image_hash}} ? 1 : 0;
            if (!exists $O{assigned} || $item->{assigned} == $O{assigned}) {
                push @new_pool, $item;
            }
        }
        @result = @new_pool;
    } else {
        @result = @$pool;
    }

    if ($O{limit}) {
        my $start = $O{offset};
        my $stop = min($start + $O{limit}, scalar(@result)) - 1;
        return (scalar(@result), [@result[$start..$stop]]);
    } else {
        return (scalar(@result), \@result);
    }
}

=head2 get_users_pool_limits

    Возвращает информацию по лимитам на пул для списка клиентов,
    на вход ссылка на хэш clientid => chiefuid (uid нужен, чтобы получить активные баннеры)

=cut

sub get_users_pool_limits {
    my ($clientid2chiefuid) = @_;
    my @result;

    my $clientids = [uniq keys %$clientid2chiefuid];
    my $uids = [uniq values %$clientid2chiefuid];

    my $images_in_pool = get_hashes_hash_sql(PPC(ClientID => $clientids), ["SELECT ClientID, count(*) as cnt from banner_images_pool", 
                                            where => {ClientID => SHARD_IDS}, 
                                            "GROUP BY ClientID 
                                            ORDER BY ClientID"
                                          ]);

    # получаем количество потенциально активных баннеров: принятных на модерации, не архивных
    my $user_active_banners = get_hashes_hash_sql(PPC(uid => $uids), [
                             "SELECT c.uid, count(*) as cnt 
                              FROM campaigns c
                                   JOIN banners b using (cid)",
                              where => {
                                "c.uid" => SHARD_IDS,
                                "c.statusEmpty" => "No",
                              }, 
                              "GROUP BY c.uid"]);

    my $tasks_in_queue = BannerImages::Queue::get_clients_items_in_queue($clientids);

    my $clients_min_limits = mass_query_special_user_option($clientids, 'image_pool_limit');

    for my $clientid (@$clientids) {
        my $uid = $clientid2chiefuid->{$clientid};
        my $min_limit = defined $clients_min_limits->{$clientid} ? $clients_min_limits->{$clientid} : $MINIMUM_IMAGES_IN_POOL;
        my $total = $min_limit + 2 * ($user_active_banners->{$uid}{cnt} || 0);
        my $cnt = ($images_in_pool->{$clientid}{cnt} || 0) + ($tasks_in_queue->{$clientid}{count} || 0);
        push @result, {ClientID => $clientid, cnt => $cnt, total => $total};
    }
    return \@result;
}

1;

