package BannerImages::Queue;

=head1 DESCRIPTION
    
    $Id$
    
    Модуль для работы с очередью картинок на обработку/скачивание
    Работа с таблицей banner_images_process_queue_bid
    
=cut 

use strict;
use warnings;
use utf8;

use Yandex::DateTime;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::DBQueue;
use Yandex::Log;
use Yandex::I18n;
use Yandex::HashUtils qw/hash_copy/;

use Settings;

use Tools;

our $log ||= Yandex::Log->new(log_file_name => 'ppcProcessImageQueue.log', date_suf => "%Y%m%d");

=head2 _add_item

    метод для добавления картинки на скачивание и привязки к баннеру
    Параметры:
        $operator_uid - uid пользователя, добавляющего задание
        $item - { ClientID - ID клиента, для которого предназначено изображение
                  url => URL картинки для скачивания,
                  name => имя картинки/файла (необязательный),
                  bid => ID баннера, или список ID баннеров [1,2,3] (необязательнный) }
    Возвращает:
        $job - Объект Yandex::DBQueue::Job
=cut

sub _add_item {
    my $operator_uid = shift;
    my $item = shift;

    die "Incorrect input parameters" unless (ref $item eq 'HASH' && $item->{url} && $item->{ClientID} && $operator_uid);

    my $ad_type = $item->{ad_type} // 'text';

    my $job = Yandex::DBQueue->new(PPC(ClientID => $item->{ClientID}), 'banner_images')->insert_job({
        job_id => get_new_id('job_id'),
        uid => $operator_uid,
        ClientID => $item->{ClientID},
        args => {
            url => $item->{url},
            name => $item->{name} // parse_image_name_from_url($item->{url}),
            xls_id => $item->{xls_id},
            ( $ad_type eq 'image_ad'
                ? (
                    pid => $item->{pid},
                    ad_type => 'image_ad',
                    href => $item->{href},
                ) 
                : ()
            )
        },
    });

    return $job;
}

=head2 add_items

add_items
    $items -- array of hashes
        ClientID => $client_id,
        url => $image_url,
        bid => banners.bid

    %opt:
        UID -- operator uid

returns array of $job_id

=cut

sub add_items
{
    my ($items, %opt) = @_;
    my @job_ids;
    foreach_shard ClientID => $items, sub {
        my ($shard, $shard_items) = @_;
        my %seen;
        my @insert;
        for my $item (@$shard_items) {
            my $job = _add_item($opt{UID}, $item);
            my $job_id = $job->job_id;

            push @job_ids, $job_id;
            if ($item->{bid}) {
                push @insert, [ $item->{bid}, $job_id ];
            }
        }
        if (@insert) {
            do_mass_insert_sql(PPC(shard => $shard), 'insert ignore into banner_images_process_queue_bid (bid, job_id) values %s',
                \@insert,
            );
        }
    };

    return \@job_ids;
}


=head2 update_xls_id
    меняет привязку втягиваемых изображений к втянутому xls-файлу

    Параметры:
        $xls_id - ID втянутого в систему xls файла
        $jobs - массив job_id (Yandex::DBQueue::Job)

=cut

sub update_xls_id {
    my ($shard, $xls_id, $jobs) = @_;

    die "Incorrect input parameters" unless defined $xls_id && ref $jobs;

    do_update_table(PPC(shard => $shard),
        'banner_images_process_queue_bid',
        { xls_id => $xls_id },
        where => {
            job_id => $jobs,
        },
    );
    return 1;
}

=head2 get_items_status
    возвращаем состояние заданий на скачивание, по их ID

    Параметры:
        $operator_uid - id оператора
        $client_ids - [123, 345, 456] - id клиентов, для которых было инициировано задание
        $job_ids - [1,2,3] - список ID заданий в очереди, по которым необходимо получить информацию
    Возвращает список из двух элементов:
        - количество заданий в очереди
        - список заданий с информацией об их состоянии
        [
         { job_id - ID в очереди,
           image_hash - /опционально/ md5 скачанной картинки,
           status - статус задания New|Process|Success|Failed
           error - /опционально/ текст ошибки },
         {...},
         ...
        ]

=cut

sub get_items_status {
    my ($operator_uid, $client_ids, $job_ids, %OPT) = @_;

    die("Incorrect input parameters")
        unless (($OPT{role} =~ m/(super|support)/ || $operator_uid)
                && ref $client_ids eq 'ARRAY'
                && scalar @$client_ids
        );

    my $queue = Yandex::DBQueue->new(PPC(ClientID => $client_ids), 'banner_images');
    
    my $where;
    $where->{uid} = $operator_uid if !$OPT{role} || $OPT{role} !~ m/(super|support)/;
    $where->{ClientID} = $client_ids;
    $where->{job_id} = $job_ids;

    my $jobs = $queue->find_jobs(%$where, limit => $OPT{limit}, offset => $OPT{offset});
    
    my $selected_tasks_num = scalar @$jobs;
    
    my @tasks;
    for my $job (@$jobs) {
        my $task = {
            job_id => $job->job_id,
            ClientID => $job->ClientID,
            name => $job->args->{name},
            status => $job->status,
            source_url => $job->args->{url},
        };
        if ($job->has_result) {
            if ($job->result->{image_hash}) {
                hash_copy $task, $job->result, qw/image_hash image_type namespace mds_group_id/;
            }
            if ($job->status eq 'Failed') {
                $task->{error} = $job->result->{error};
            }
        }
        push @tasks, $task;
    }
    
    return ($selected_tasks_num, \@tasks);
}

=head2 get_clients_items_in_queue

  Получить количество необработанных заданий на скачивание картинок для списка клиентов,
  выбираются задания со статусом Grabbed и New
  На вход - ссылка на массив ClientID,
  Выходные данные - хэш вида ClientID => {ClientID => 123, count => 12}

=cut

sub get_clients_items_in_queue {
    my ($client_ids) = @_;
    my $queue = Yandex::DBQueue->new(PPC(ClientID => $client_ids), 'banner_images');
    my $stat = $queue->get_client_statistics($client_ids, status => [ 'New', 'Grabbed' ]);
    return $stat;
}

1;

