package BannerStorage;

use Direct::Modern;

use Encode;
use Yandex::Log;
use JSON;
use Direct::Model::Creative;
use Direct::Model::TurboLanding;
use Carp;
use Yandex::HTTP qw/http_parallel_request/;
use Yandex::Trace;
use Yandex::LiveFile;
use Yandex::ListUtils qw/chunks/;
use List::Util qw/any sum0/;
use Settings;
use Yandex::DBTools qw/get_hash_sql/;
use TextTools qw/smartstrip2/;
use Tools qw//;
use Property;
use Math::Round qw/round/;
use SolomonTools;


# ID объкта причины отклонения ({"objects":[{"id":16}]}), относящейся только к performance шаблонам
# необходим для того, чтобы запрашивать переводы только нужных причин отклонения
our $BS_REST_PERFORMANCE_REASON_OBJECT_ID //= 16;

#Максимальное число performance-креативов, которые можно получить одним запросом
our $BS_REST_MAX_CREATIVES_PER_REQUEST //= 20;

our $BS_REST_API_URL //= 'https://rest-api.bannerstorage.yandex.net/rest/v0.1';

# продакшн-токен, все данные в BS привязаны к нему
our $BS_REST_API_TOKEN;

# файл с oauth token
our $BS_REST_TOKEN_FILE //= '/etc/direct-tokens/banner_storage_auth.txt';

#Максимальное число canvas-креативов, которые можно получить одним запросом
our $BS_CANVAS_MAX_CREATIVES_PER_REQUEST //= 50;

#URL API для canvas-креативов
our $BS_CANVAS_API_URL //= 'http://canvas-back.direct.yandex.net/direct';

#URL API для video_addition-креативов
our $BS_VIDEO_ADDITION_API_URL //= 'http://canvas-back.direct.yandex.net/video/direct';

# продакшн-токен API для canvas-креативов, все данные в BS привязаны к нему
our $BS_CANVAS_API_TOKEN;

# файл с oauth token
our $BS_CANVAS_TOKEN_FILE //= '/etc/direct-tokens/banner_storage_canvas_auth.txt';

# url для получения турболендингов
my $UNIFIED_TURBOLANDINGS_PROP = Property->new("UNIFIED_TURBOLANDINGS");
our $BS_TURBOLANDINGS_API_URL;
our $UNIFIED_TURBOLANDINGS_API_URL;
our $BS_TURBOLANDINGS_TOKEN //= $BS_CANVAS_API_TOKEN;
our $BS_TURBOLANDINGS_TOKEN_FILE = $BS_CANVAS_TOKEN_FILE;
our $BS_TURBOLANDINGS_MAX_PER_REQUEST //= 150;

# шаблоны, для которых допускаем пустой layouts
our %TEMPLATE_WITHOUT_LAYOUTS = map {$_ => 1} (
    # по данным BS: https://st.yandex-team.ru/DIRECT-76524#1519208487000
    320,330,331,332,333,334,335,336,337,
    619,620,684,692,694,
    703,704,706,707,708,713,748,755,756,757,758,759,760,771,772,
    811,812,813,814,815,816,817,818,819,1014,
    # и ещё по факту
    547,769,770,
);

my $TEMPLATE_INFO;

=head2 receive_creatives($creative_ids)

Получить актуальное состояние креативов в BannerStorage.

Параметры:
    $creative_ids - [] массив id креативов

Результат:
    [$creative1, $creative2] - $creative is Direct::Model::Creative

=cut

# perl -I./protected/ -I./perl/settings/ -MBannerStorage -E 'use Settings; BannerStorage::receive_creatives([986097]);'

sub receive_creatives {
    my ($creative_ids) = @_;

    return [] unless @$creative_ids;

    unless (defined $TEMPLATE_INFO) {
       $TEMPLATE_INFO = get_hash_sql(PPCDICT, [
            "SELECT id as template_id, json_content FROM banner_storage_dict",
            WHERE => { type => 'template' },
       ]);
    }    
    
    my $bs_creatives = [];
    foreach my $chunk (chunks($creative_ids, $BS_REST_MAX_CREATIVES_PER_REQUEST)){ 
        my $bs_creatives_res = banner_storage_call( GET => 'creatives', {
                ids => (join ',', @$chunk),
                include => 'rejectReasons,group,businessType,layoutCode,previewUrl,isPredeployed',
            });
        if ($bs_creatives_res->{error}) {
            die $bs_creatives_res->{error};
        }
        push @$bs_creatives, @{$bs_creatives_res->{content}->{items} // []}
    }
    my @creatives = map { _creative_from_bs_response($_) } @{$bs_creatives};
    return \@creatives;
}

=head2 _get_duration

Функция получает объект креатива, полученный от BS.

Возвращает округленное целое значение продолжительности видео.
Если не находит, то возвращает undef.

=cut

sub _get_duration
{
    my $bs_creative = shift;

    my @file_ids;
    for my $param (@{$bs_creative->{parameters}}) {
        next unless $param->{paramType};
        my $type_name = lc $param->{paramType};

        if ($type_name eq 'file' && exists $param->{values} && ref $param->{values} eq 'ARRAY' &&
                scalar @{$param->{values}} > 0 && $param->{values}->[0]) {
            push @file_ids, $param->{values}->[0];
        }
    }

    for my $file_id (@file_ids) {
        my $file_info = banner_storage_call(GET => "files/$file_id/extended", {});

        if ($file_info->{error}) {
            die $file_info->{error};
        }
        if ($file_info->{content}->{fileTypeInfo}->{mimeType} =~ /video.*/i) {
            return round($file_info->{content}->{duration});
        }
    }

    return;
}

=head2 _creative_from_bs_response

Функция превращает объект креатива, полученный от BS, в Direct::Creative

=cut

sub _creative_from_bs_response
{
    my $bs_creative = shift;
    
    state $status_moderate = [undef, 'New', 'Sent', 'No', 'Yes'];
    
    for my $param (@{$bs_creative->{parameters}}) {
        next unless $param->{paramType};
        my $type_name = lc $param->{paramType};
        if ($type_name eq 'alt') {
            $bs_creative->{alt_text} = $param->{values}->[0]; 
        } elsif ($type_name eq 'httpurl' || $type_name eq 'click') {
            $bs_creative->{href} = $param->{values}->[0];
        }
    }

    my $create_time = $bs_creative->{group}->{dateCreate};
    $create_time =~ s!(\d{4}-\d{2}-\d{2})T(\d{2}:\d{2}:\d{2}).*!$1 $2! if $create_time;

    if (exists $bs_creative->{tag} && defined $bs_creative->{tag} && length($bs_creative->{tag}) > 0) {
        # признаков старых креативов два: есть tag или нет groupId. любого из них достаточно, чтобы считать
        # креатив старым. у себя мы считаем старыми те, у кого пустой groupId
        delete $bs_creative->{groupId};
    }

    my $template_id = $bs_creative->{templateId};
    my $template_info = decode_json(encode('UTF-8',$TEMPLATE_INFO->{$template_id}));
    my $creative_type = ($template_info->{templateType} == 9) ? "performance" : "bannerstorage";
    $bs_creative->{duration} = _get_duration($bs_creative) if $creative_type eq "bannerstorage";

    my %creative_data = (
        (map { $_ => $bs_creative->{$_} } qw/id name width height alt_text href live_preview_url duration is_adaptive/),
        creative_group_id => $bs_creative->{groupId},
        group_name => $bs_creative->{group}->{name},
        group_create_time => $create_time,
        preview_url => $bs_creative->{screenshotUrl} || $bs_creative->{thumbnailUrl},
        live_preview_url => $bs_creative->{preview}->{url},
        has_screenshot => ( $bs_creative->{screenshotUrl} ? 1 : 0 ),
        has_packshot => 0,
        status_moderate => $status_moderate->[$bs_creative->{status}],
        template_id => $template_id,
        creative_type => $creative_type,
        version => $bs_creative->{version},
        business_type => $bs_creative->{businessType}->{directId} // 'retail',
        layout_id => $bs_creative->{layoutCode}->{layoutId},
        theme_id =>  $bs_creative->{layoutCode}->{themeId},
        stock_creative_id => $bs_creative->{id},
        moderation_comment => $bs_creative->{moderationComment},
        moderate_info_json => $creative_type eq "performance" ? undef : encode_json(
            {
                html => {
                    url => $bs_creative->{preview}->{url}
                },
            }
        ),
        is_bannerstorage_predeployed => ((defined $bs_creative->{predeployed} && $bs_creative->{predeployed} eq 'true') ? 1 : 0),
    );
    
    my $creative = Direct::Model::Creative->new(%creative_data);

    $creative->rejection_reason_ids([grep { defined $_ && $_ > 0 } map { $_->{id} } @{$bs_creative->{rejectReasons}}]);

    return $creative;
}

=head2 receive_canvas_creatives($creative_ids)

По переданному массиву $creative_ids получить canvas-креативы из BannerStorage.

=cut

sub receive_canvas_creatives {
    my ($client_id, $creative_ids) = @_;
    
    my @result;
    foreach my $chunk (chunks($creative_ids, $BS_CANVAS_MAX_CREATIVES_PER_REQUEST)){
        my $bs_creatives_res = banner_storage_canvas_call(
            GET => 'creatives',
            {client_id => $client_id, ids => (join ',', @$chunk) },
        );
        push @result, @{$bs_creatives_res->{content}} if $bs_creatives_res->{content};
    }
    
    return \@result;
}

=head2 receive_video_addition_creatives($creative_ids)

По переданному массиву $creative_ids получить video_addition-креативы из BannerStorage.

=cut

sub receive_video_addition_creatives {
    my ($client_id, $creative_ids) = @_;

    my @result;
    foreach my $chunk (chunks($creative_ids, $BS_CANVAS_MAX_CREATIVES_PER_REQUEST)){
        my $bs_creatives_res = banner_storage_video_addition_call(
            GET => 'creatives',
            {client_id => $client_id, ids => (join ',', @$chunk) },
        );
        push @result, @{$bs_creatives_res->{content}} if $bs_creatives_res->{content};
    }

    return \@result;
}

=head2 generate_creatives($conditions)

По переданному массиву $conditions сгенерировать креативы в BannerStorage.
Условие состоит из:
    category_ids - категории каталогии которые есть у баннера
    count - сколько выдать креативов для этих категорий
    creative_type - тип креатва. Поддерживает video_addition
    locale - локаль. на каком языке будет генерироваться креатив (Пример локали ru_RU)

В ответе приходит на каждое условие список creative_id

=cut

sub generate_creatives {
    my ($client_id, $conditions) = @_;
    Tools::force_number_recursive($conditions);

    my $obj_num = sum0 map { $_->{count} } @$conditions;
    my $profile = Yandex::Trace::new_profile('banner_storage:generate_creatives', obj_num => $obj_num);
    my $request_json = encode_json({client_id => int($client_id), conditions => $conditions });
    my $bs_creatives_res = banner_storage_video_addition_call(
        POST => 'generate-creatives',
        $request_json,
        headers => { 'Content-Type' => 'application/json' },
        timeout => 60,
    );

    die "generate creatives error. message: @{[$bs_creatives_res->{error}//'(undef)']}" if !$bs_creatives_res->{content};


    return $bs_creatives_res->{content};
}

=head2 banner_storage_video_addition_call($creative_ids)

Выпоняет запрос к API video_addition-креативов

=cut

sub banner_storage_video_addition_call {
    my ($http_method, $method, $request, %opt) = @_;

    $opt{url} //= $BS_VIDEO_ADDITION_API_URL;
    $opt{auth_token} //= _define_auth_token($BS_CANVAS_API_TOKEN, $BS_CANVAS_TOKEN_FILE);

    return banner_storage_call($http_method, $method, $request, %opt);
}

=head2 banner_storage_canvas_call($http_method, $method, $request, %opt)

Выпоняет запрос к API canvas-креативов

=cut

sub banner_storage_canvas_call {
    my ($http_method, $method, $request, %opt) = @_;
    
    $opt{url} //= $BS_CANVAS_API_URL;
    $opt{auth_token} //= _define_auth_token($BS_CANVAS_API_TOKEN, $BS_CANVAS_TOKEN_FILE);
    
    return banner_storage_call($http_method, $method, $request, %opt);
}

=head2 banner_storage_turbolandings_call($http_method, $method, $request, %opt)

Выпоняет запрос к API canvas-креативов

=cut

sub banner_storage_turbolandings_call {
    my ($http_method, $method, $request, %opt) = @_;
    
    $opt{url} //= turbolandings_api_url();
    $opt{auth_token} //= _define_auth_token($BS_TURBOLANDINGS_TOKEN, $BS_TURBOLANDINGS_TOKEN_FILE);
    
    return banner_storage_call($http_method, $method, $request, %opt);
}

=head2 get_rejection_reasons

Получить все причины отклонения креативов на модерации
У причин отклонения отрезаются все whitespace символы в начале и в конце

Результат:
    {reason_id => description}
    description - человекочитаемое название на русском

=cut

sub get_rejection_reasons {
    my $result = banner_storage_call(GET => 'dictionaries/rejectreasons');
    if ($result->{error}) {
        die $result->{error};
    }
    my @result;
    for my $item (@{$result->{content}->{items}}) {
        if (exists $item->{objects} && defined $item->{objects}
            && any { $_->{id} eq $BS_REST_PERFORMANCE_REASON_OBJECT_ID } @{ $item->{objects} }) {
                push @result, {
                    id => $item->{id},
                    text => smartstrip2($item->{text}),
                    name => smartstrip2($item->{name}),
                };
        }
    }
    return \@result;
}


# --------------------------------------------------------------------

=head2 update_geo_on_creatives

Обновляем страны на креативе

Параметр:
    {
        creative_id1 => 'geo,geo2,geo3',
        creative_id2 => 'geo1,geo2',
        ...
    }

Возвращаем ошибки:
{
    creative_id1 => undef,
    creative_id2 => {faultcode => 1, faultstring => 'error',...},
    ...
}

https://doc.yandex-team.ru/rtb/doc/banner-store/CreativeUpdateGeo.xml

Тестирование:
    perl -ME -MBannerStorage -E 'my $res = BannerStorage::update_geo_on_creatives({272963691 => "187"}); p($res)'

=cut

sub update_geo_on_creatives($) {
    my ($creatives_list) = @_;
    return {} unless %$creatives_list;
    my $all_results = {};

    for my $creative_id (keys %$creatives_list) {
        my $geo_str = $creatives_list->{$creative_id};
        my $request = { items => [ map { { id => int($_), exclude => 0 } } split /,/, $geo_str ] };
        my $request_json = encode_json($request);
        my $result = banner_storage_call(PUT => "creatives/$creative_id/geolocations", $request_json,
            headers => { 'Content-Type' => 'application/json' });
        if ($result->{error}) {
            $all_results->{$creative_id} = { faultcode => 1, faultstring => $result->{error} };
        } else {
            $all_results->{$creative_id} = undef;
        }
    }

    return $all_results;
}

=head2 get_bs_turbolandings_by_client_id

Получить из BS список турболендингов клиента

Выводится не более 150 последних турболендингов.

Параметры:
    client_id - идентификатор клиента,
    tl_ids - список идентификаторов турболендингов

Возвращаем  hashref на структуру с турболендингами:
    {
        turbolanding_id1 => {
            tl_id       => turbolanding_id1,
            client_id    => client_id,
            name        => имя турболендинга, может меняться через конструктор
            href        => ссылка на турбостраницу,
            metrika_counters_json => счетчики и цели, сериализованные в JSON
        },
        turbolanding_id2 => { ... },
        ...
    }


Тестирование:
    perl -ME -MBannerStorage -E 'my $res = BannerStorage::get_bs_turbolandings_by_client_id(39285146); p($res)'

=cut
sub get_bs_turbolandings_by_client_id {
    my ($client_id, $tl_ids) = @_;

    croak 'client_id required' unless $client_id;
    
    my %filter = $tl_ids ? (ids => join(',', @$tl_ids)) : ();
    my $bs_turbolandings_res = banner_storage_turbolandings_call(
                GET => 'landings',
                {
                    client_id => $client_id,
                    %filter,
                    limit => $BS_TURBOLANDINGS_MAX_PER_REQUEST,
                },
                url => turbolandings_api_url(),
    );
    die $bs_turbolandings_res->{error} if ($bs_turbolandings_res->{error});

    my %bs_turbolandings = map { Direct::Model::TurboLanding->from_bs_response_to_hash($client_id => $_)}
            @{$bs_turbolandings_res->{content}->{items} // []};
    
    return \%bs_turbolandings;
}

# --------------------------------------------------------------------

=head2 get_all_geo_ids

Получаем все geo-id

https://doc.yandex-team.ru/rtb/doc/banner-store/GeoLocationGetList.xml

Тестирование:
    perl -ME -MBannerStorage -E 'my $res = BannerStorage::get_all_geo_ids()'

=cut

sub get_all_geo_ids() {
    my $result = banner_storage_call(GET => 'dictionaries/geolocations');
    if ($result->{error}) {
        die $result->{error};
    }
    return $result->{content}->{items};
}


=head2 banner_storage_call($http_method, $method, $request)

Вызвать метод BannerStore c логированием

my $response = banner_storage_call(GET => 'creatives', ... );

=cut

sub banner_storage_call {
    
    my ($http_method, $method, $request, %opt) = @_;

    my $method_name = $method =~ s!/\d+/!/!gr; # creatives/$id/geolocations => creatives/geolocations
    $method_name =~ s!/!:!g;

    my $base_url = $opt{url} // $BS_REST_API_URL;
    my $auth_token = $opt{auth_token} // sprintf( 'OAuth: %s', _define_auth_token($BS_REST_API_TOKEN, $BS_REST_TOKEN_FILE));
    unless ($auth_token) {
        die "auth token missed";
    }

    my $profile = Yandex::Trace::new_profile("banner_storage:$method_name");
    
    state ($log, $json);
    $log //= Yandex::Log->new(
        no_log => 1,
        use_syslog => 1,
        syslog_prefix => 'INTAPI',
        log_file_name => "BannerStorage_REST.log",
    );
    $json //= JSON->new->allow_unknown(1)->allow_blessed(1)->convert_blessed(1);

    my $url = Yandex::HTTP::make_url($base_url . "/$method", ( $http_method eq 'GET' ? ($request) : () ));

    $log->out(['REQUEST:', $url, $method, $auth_token, $request]);
    my $response = http_parallel_request($http_method => { 1 => {
        url => $url,
        ($http_method ne 'GET' ? (body => $request) : () ),
        timeout => $opt{timeout} // 60,
        ($http_method eq 'GET' ? (soft_timeout => 1, num_attempts => 2) : ()),
        log => $log,
        headers => {
            %{$opt{headers}//{}},
            Authorization => $auth_token,
        },
    }})->{1};

    my $result = {};
    my $stat = {};
    if ($response->{is_success}) {
        utf8::decode($response->{content});
        $result->{content} = eval { $json->decode($response->{content}) } if $response->{content};
        if ($@) {
            $stat->{unparsable}++;
            $result->{error} = $@;
        } else {
            $stat->{success}++;
        }
    }
    else {
        my $status = SolomonTools::guess_status_by_code($response->{headers}->{Status});
        $stat->{$status}++;
        $result->{error} = $response->{content} || $response->{headers}->{Reason};
    }

    # через этот метод проходят запросы на разные урлы, запишем информацию о конкретном урле
    my $solomon_sub_system = $base_url =~ s!(^https?://)!!r;
    my $solomon_method_name = $method =~ s!/\d+/!/NNN/!gr; # creatives/234234/geolocations => creatives/NNN/geolocations
    _send_metrics_to_solomon($solomon_sub_system, $solomon_method_name, $stat);

    $log->out(['RESPONSE:', $result, ( $result->{error} ? ( $response->{headers}->{Reason}) : () ) ]);

    return $result;
}

sub _define_auth_token {
    my ($auth_token, $auth_token_file) = @_;
 
    state %token_live_file;
    
    unless (defined $auth_token) {
        $token_live_file{$auth_token_file} //= Yandex::LiveFile->new(filename => $auth_token_file);
        $auth_token = $token_live_file{$auth_token_file}->data;
        chomp $auth_token;
    }
    
    return $auth_token;
}

sub _send_metrics_to_solomon {
    my ($sub_system, $method, $stat) = @_;
    # у значений меток есть недопустимые символы, уберём их
    $sub_system =~ s![*?"'\`]+!_!g;
    $method =~ s![*?"'\`]+!_!g;
    SolomonTools::send_requests_stat({external_system => "banner_storage", "sub_system" => $sub_system, method => $method}, $stat);
}


=head2 is_unified_turbolandings_enabled

    включен ли новый("единый") конструктор турболэндингов

=cut
sub is_unified_turbolandings_enabled {
    return $UNIFIED_TURBOLANDINGS_PROP->get(60) ? 1 : 0;
}


=head2 turbolandings_api_url

    Вернуть адрес актуального api турболэндингов

=cut
sub turbolandings_api_url {
    return is_unified_turbolandings_enabled() ? $UNIFIED_TURBOLANDINGS_API_URL : $BS_TURBOLANDINGS_API_URL;
}

1;
