=head1 NAME

    BannerTemplates

=head1 DESCRIPTION

    Модуль, в котором собраны методы для работы с шаблонными объявлениями.
    Вынесен из BannersCommon.

=cut

package BannerTemplates;

use strict;
use warnings;

use utf8;

use parent qw/Exporter/;

our @EXPORT = qw(
    $TEMPLATE_METKA
    $TEMPLATE_METKA_OLD

    is_template_banner
    add_banner_template_fields
    get_banners_template_fields
    get_template_phrase
    process_banner_template
    replace_template
    clear_template_phrase
    validate_banner_template
);

use Settings;
use URI::Escape qw/uri_escape_utf8/;
use Yandex::I18n;

our $TEMPLATE_METKA = qr{\#($Settings::ALLOW_BANNER_LETTER_RE*?)\#}i;
our $TEMPLATE_METKA_OLD = qr{\*(шаблон)\*}i;

# копия из Direct::Validation::Banners
# убрать после переезда URLDomain::validate_banner_href в Direct::Validation
our $MAX_DISPLAY_HREF_LENGTH = 20;


=head2 int is_template_banner( HASHREF )

Проверяет является ли баннер шаблонным

=cut

sub is_template_banner
{
    my $banner = shift;
    my $flag = shift;

    if( ($banner->{title} && $banner->{title} =~ /$TEMPLATE_METKA/si)
        || ($banner->{title_extension} && $banner->{title_extension} =~ /$TEMPLATE_METKA/si)
        || ($banner->{body} && $banner->{body} =~ /$TEMPLATE_METKA/si)
        || ($banner->{href} && $banner->{href} =~ /$TEMPLATE_METKA/si)
        || ($banner->{display_href} && $banner->{display_href} =~ /$TEMPLATE_METKA/si)
        || (!$flag && $banner->{title} && $banner->{title} =~ /$TEMPLATE_METKA_OLD/si)
        || (!$flag && $banner->{title_extension} && $banner->{title_extension} =~ /$TEMPLATE_METKA_OLD/si)
        || (!$flag && $banner->{body} && $banner->{body} =~ /$TEMPLATE_METKA_OLD/si)
    ) {
        return 1;
    }
    return 0;
}

=head2 add_banner_template_fields



=cut

sub add_banner_template_fields {
    my $banner = shift || return;
    my $phrases = shift || $banner->{phrases};
    my $result = get_banners_template_fields([$banner], $phrases);
    return $result->{$banner->{bid}};
}

=head2 get_banners_template_fields
    
    Сгенерировать поля, которые будут использоваться при отрисовки объявления.

=cut
sub get_banners_template_fields {
    my $banners = shift;
    my $phrases = shift;

    my $possible_phrases = [map { $_->{phrase} } grep {
        !$_->{is_suspended} && ($_->{statusModerate} // '') ne 'No' &&
        (exists $_->{context_stop_flag} ? !$_->{context_stop_flag} : 1)
    } @{$phrases}];

    my $results = {};

    foreach my $banner (@$banners) {
        my $rand_phrase = get_template_phrase($possible_phrases);
        my $result = $results->{$banner->{bid}} = {};
        $result->{rand_phrase} = $rand_phrase;

        $result->{is_template_banner} = is_template_banner($banner);
        $result->{is_template_banner_href} = $banner->{href} && $banner->{href} =~ /$TEMPLATE_METKA/si ? 1: 0;

        if ($result->{is_template_banner}) {
            $result->{templ_title} = process_banner_template($banner->{title}, $rand_phrase, $MAX_TITLE_LENGTH);
            $result->{templ_title_extension} = process_banner_template($banner->{title_extension}, $rand_phrase, $MAX_TITLE_EXTENSION_LENGTH);
            $result->{templ_body} = process_banner_template($banner->{body}, $rand_phrase, $MAX_BODY_LENGTH);
            $result->{templ_href} = replace_template($banner->{href}, {phrase => $rand_phrase, escape => 1, max_length => $MAX_URL_LENGTH});
            $result->{templ_display_href} = replace_template($banner->{display_href}, {
                    phrase => $rand_phrase,
                    max_length => $MAX_DISPLAY_HREF_LENGTH,
                });
        } else {
            $result->{rand_phrase} = '';
            $result->{templ_title} = $banner->{title};
            $result->{templ_title_extension} = $banner->{title_extension};
            $result->{templ_body} = $banner->{body};
            $result->{templ_href} = $banner->{href};
            $result->{templ_display_href} = $banner->{display_href};
        }
    }
    return $results;
}


=head2 get_template_phrase( arrayref )

Выбираем случайную фразу, обрезаем в ней минус слова и лишние символы.

=cut

sub get_template_phrase
{
    my @phrases = @{$_[0]};

    my $rand_phrase = $phrases[ int( rand( scalar @phrases ) ) ] || '';
    
    # обрезаем лишние симолы и минус слова
    return clear_template_phrase( $rand_phrase );
}

=head2 process_banner_template

=cut
sub process_banner_template
{
    my $templ = shift || return;
    my $phrase = shift;
    my $max_length = shift;

    my $text_for_check = $templ;
    #в валидации длины не учитываются только те узкие символы, которые были до подстановки фразы в шаблон
    $text_for_check =~ s/$NARROW_SYMBOLS_RE//g;

    if ($phrase) {
        $text_for_check =~ s/$TEMPLATE_METKA/<em>$phrase<\/em>/gi;
        $text_for_check =~ s/$TEMPLATE_METKA_OLD/<em>$phrase<\/em>/gi;
    }
    $text_for_check =~ s!</?em>!!g;
    $text_for_check =~ s!&quot;!"!g;
    my $length_clear_text = length($text_for_check);

    # если лимит на длину со случайной фразой превышен, то подставляем фразу по умолчанию (или если не задана случайная фраза)
    if (! $phrase || ($max_length && $length_clear_text > $max_length)) {
        $templ =~ s/$TEMPLATE_METKA/length($1) ? "<em>$1<\/em>" : ""/gie;
        $templ =~ s/$TEMPLATE_METKA_OLD/length($1) ? "<em>$1<\/em>" : ""/gie;
    } else {
        $templ =~ s/$TEMPLATE_METKA/<em>$phrase<\/em>/gi;
        $templ =~ s/$TEMPLATE_METKA_OLD/<em>$phrase<\/em>/gi;
    }

    return $templ;
}

=head2 replace_template(text, phrase)

    Replace template label in text by phrase(if set) or default phrase

=cut

sub replace_template
{
    my ($text, $opt) = @_;

    return undef if !defined $text;

    my $text_for_check = $text;
    if ($opt->{phrase}) {
        my $phrase = clear_template_phrase($opt->{phrase});
        if ($opt->{escape}) {
            $phrase = uri_escape_utf8($phrase);
        }
        $text_for_check =~ s/$TEMPLATE_METKA/$phrase/gsi;
    }

    # если лимит на длину со случайной фразой превышен, то подставляем фразу по умолчанию (или если не задана случайная фраза)
    if (! $opt->{phrase} || ($opt->{max_length} && length($text_for_check) > $opt->{max_length})) {
        if ($opt->{escape}) {
            $text =~ s/$TEMPLATE_METKA/uri_escape_utf8($1)/gsie;
        } else {
            $text =~ s/$TEMPLATE_METKA/$1/gsi;
        }
    } else {
        $text = $text_for_check;
    }

    return $text;
}

=head2 clear_template_phrase( string )

Обрезаем минус фразы и лишние нетекстовые допустимые в фразе символы

=cut

sub clear_template_phrase
{
    my $phrase = shift;

    if( $phrase ) {
        $phrase =~ s/[\"\+\!\[\]]//gsi;
        $phrase =~ s/(?:\s+)\-[$ALLOW_LETTERS]+(?=\s|$)//gsi;
    }

    return $phrase
}

=head2 validate_banner_template( HASHREF, ARRAYREF )

Для баннеров с шаблоном проверяет:
    1. Расположение шаблона относительно остальных фраз в заголовке и тексте.

    Если phrases передаются дополнительно, то проверяем с этими фразами, если нет - то из полей phrases или Phrases
=cut

sub validate_banner_template
{
    ## no critic (Freenode::DollarAB)
    my ($b, $phrases) = @_;

    my @result = ();
    my @phrases = ();
    my $b_phrases = $phrases || $b->{phrases} || $b->{Phrases};
    if ($b->{phr}) {
        @phrases = @{$b->{phr}}
    } elsif ($b_phrases && ref $b_phrases eq '') {
        @phrases = map {/^(.*?)::/ ? {phrase => $1} : ()} split(/,\s*/, $b_phrases);
    } elsif ($b_phrases && ref $b_phrases eq 'ARRAY') {
        @phrases = @{$b_phrases};
    }

    if (defined $b->{title} && $b->{title} =~ /$TEMPLATE_METKA_OLD/
        || defined $b->{title_extension} && $b->{title_extension} =~ /$TEMPLATE_METKA_OLD/
        || defined $b->{body} && $b->{body} =~ /$TEMPLATE_METKA_OLD/) {
        push @result, iget('Используется старый формат шаблона');
    } elsif (defined $b->{title} && $b->{title} =~ /^$TEMPLATE_METKA$/ && length($1) == 0
        || defined $b->{title_extension} && $b->{title_extension} =~ /^$TEMPLATE_METKA$/ && length($1) == 0
        || defined $b->{body} && $b->{body} =~ /^$TEMPLATE_METKA$/ && length($1) == 0) {
        # не допускаем текст или заголовок только из одного шаблона
        push @result, iget('Недопустимый заголовок или текст объявления');
    } else {
        if (my $default_phrases = get_default_template_banner_phrases($b)) {
            foreach my $def_phrase (@{$default_phrases}) {
                if ($def_phrase =~ /^шаблон[ы]?$/gsi) {
                    push @result, iget('Нельзя использовать в качестве фразы по умолчанию - "%s"', $def_phrase);
                    last;
                }
            }
        }
    }

    return \@result;
}

=head2 get_default_template_banner_phrases

    

=cut
sub get_default_template_banner_phrases
{
    my $banner = shift;
    my $banner_text = ($banner->{title}||'').' '.($banner->{title_extension}||'').' '.($banner->{body}||'');

    my @phrases = ();

    while ($banner_text =~ /$TEMPLATE_METKA/g) {
        push @phrases, $1;
    }

    return \@phrases;
}

1;
