##################################################
#
#  Direct.Yandex.ru
#
#  BannersCommon
#      функции для работы с баннерами, вынесено из Common
#
#
#  $Id$
#
# (c) 2010, Yandex
#
##################################################

=head1 NAME

BannersCommon

=head1 DESCRIPTION

  функции для работы с баннерами, часть вынесена из Common, 
  часть - следствие дробления get_user_banner и get_user_camp

=cut

package BannersCommon;

use strict;
use warnings;

use Settings;

use GeoTools;
use TextTools;
use Tools;
use BSAuction;
use BS::TrafaretAuction;
use BS::CheckUrlAvailability;
use Pokazometer;
use Primitives;
use PrimitivesIds;
use Sitelinks;
use TimeTarget;
use Stat::OrderStatDay;
use BannerTemplates;
use PhraseText;
use Sitelinks;
use URLDomain;
use VCards;
use MinusWords;

use Yandex::DBTools;
use Yandex::HashUtils;
use Yandex::IDN;
use Yandex::I18n;
use Yandex::Runtime qw/need_list_context/;
use Yandex::SendMail qw/send_alert/;
use Yandex::ScalarUtils;
use Yandex::DBShards;
use Yandex::Validate qw/is_uint is_valid_ip is_valid_id/;
use Yandex::URL;
use Yandex::Clone qw/yclone/;

use Direct::Validation::Keywords qw/base_validate_keywords/;
use Direct::Validation::Banners;
use Direct::ValidationResult;
use Direct::Banners;

use Models::AdGroup;
use Models::Banner;
use Models::Phrase;

use Data::Dumper;
use List::MoreUtils qw/uniq all none any firstidx/;
use List::Util qw/max min sum first/;
use Carp;

use utf8;

use base qw(Exporter);

our @EXPORT = qw(

    get_banners
    update_banner_status_moderate
    update_geo
    
    get_banner_phrases_count
    get_banner_relevance_matches_count
    get_banner_retargetings_count

    save_manual_prices

    is_banner_edited_by_moderator_multi
    change_banner_modedit_status

    check_json_ajax_update_struct
    check_json_ajax_apply_unglued_minus_words_struct

    get_banners_short_hash
    mass_archive_banners
);

# список полей в табл. banners
our @BANNER_FIELDS = qw/
    bid cid
    title title_extension body href
    BannerID
    statusShow statusActive statusModerate
    geoflag
    statusArch
    LastChange
    statusBsSynced
    domain
    reverse_domain
    statusPostModerate
    vcard_id
/;

our %MODERATE_STATUS = (
    "New"     => iget_noop("Черновик"),
    "Yes"     => iget_noop("Допущено модератором"),
    "No"      => iget_noop("Отклонено модератором"),
    "Ready"   => iget_noop("Ожидает модерации"),
    "Sending" => iget_noop("Ожидает модерации"),
    "Sent"    => iget_noop("Ожидает модерации"),
);

our %BS_STATUS = (
    "Yes"     => iget_noop("Идет активизация"),
    "No"      => "",
);

# переходить из Профи в чайники позволяем только если объявлений не больше $MAX_BANNER_LIMIT_EASY (т.к. они все окажутся на одной странице)
our $MAX_BANNER_LIMIT_EASY = 200; 

=head1 GLOBAL VARIABLES

=head2 get_banners

  Возвращает массив из двух элементов: ссылку на массив баннеров, и общее количество записей

  Входные параметры :
    $search_options {
        uid => , 
        cid => один или список номеров кампаний, 
        bid => ссылка на массив,
        context => ,
        phrase => ,
        pid => идентификатор группы,
        group_name => название группы,
        limit => ,
        offset => ,
        adgroup_types => массив типов групп (и баннеров соответственно),
                            которые необходимо выбирать (["base", "dynamic", "mobile_content"]),
                            по умолчанию выбираются "base"
    },
    $options {
        get_auction - bool получать ли данные из крутилки
        get_phrases_statuses - bool, получить статусы фраз из БК
        not_use_own_stat - bool не использовать данные из баннера ( транслируется в get_phrases )
        no_pokazometer_data - bool не получать данные из показометра
        get_add_camp_options - дополнительные параметры из кампании для каждого баннера (для get_user_banner),
        get_tags - bool получать ли данные о метках
        camp_spent_today - сумма в у.е., истраченная кампанией за сегодня (учитывается вместе с дневным бюджетом)
        опции для крутилки
        camp_bs_queue_status - bool, получить информацию о наличии кампании в очереди транспорта БК
        pass_empty_groups => 1 | 0 - выбирать группы без баннеров и представлять их как пустые баннеры {},
                торгов для таких баннеров не будет
        get_multiplier_stats => 1 | 0 - выбирать доп. статистику для различных корректировок. (детали в HierarchicalMultipliers::adjustment_bounds())
                и записывать её в поле multiplier_stats
        device_targeting => строка из поля camp_options.device_targeting БД, 
                нужна для того чтобы включить мобильные торги для кампаний таргетирванных на смартфоны
    }

=cut
sub get_banners {
    my ($search_options, $options) = @_;

    # проверяем на наличие хотя бы одного условия для поиска
    if ( ! scalar keys %{hash_cut $search_options, qw/uid cid bid pid/} ) {
        die "get_banners: Not all required params specified";
    }

    # возвращаем все запрошенные баннеры, в независмости от привязке к группе(используется в API) 
    my %banners_filter;
    %banners_filter = map {
        $_ => 1
    } ref $search_options->{bid} ? @{$search_options->{bid}} : $search_options->{bid} if $search_options->{bid};
    my $filter_by_banners = scalar keys %banners_filter;
    my %cid_cache;
    my $get_groups_options = yclone($options);
    $get_groups_options->{skip_banner_template} = 1 if $options->{get_auction};
    my ($groups, $total) = get_groups($search_options, $get_groups_options);
    Models::AdGroup::update_phrases_shows_forecast($groups); # showsForecast нужен для trafaret_auction
    my $banners = [];
    enrich_image_banners($groups);

    foreach my $group (@$groups) {
        my @group_banners;

        if ($filter_by_banners) {
            @group_banners = grep {$banners_filter{$_->{bid}}} @{$group->{banners}};
        }
        unless (@group_banners) {
            if (@{$group->{banners}}) {
                push @group_banners, Models::AdGroup::get_main_banner($group);    
            } elsif ($options->{pass_empty_groups}) {
                # пустые группы (без баннеров) представляем как не завершенный баннер
                push @group_banners, {};
            }
        }

        if (all { $_->{real_banner_type} eq 'image_ad' } @group_banners) {
            for my $ph (@{$group->{phrases}}) {
                $ph->{guarantee} = $ph->{premium} = [ ({ bid_price => 0.01, amnesty_price => 0.01 }) x 4 ];
            }
        }
    
        foreach my $banner (@group_banners) {
            hash_merge $banner,
                Models::Banner::_copy_to_banner($group, $Models::Banner::banner_fields{group}),
                Models::Banner::_copy_to_banner($group, $Models::Banner::banner_fields{camp}),
                $group->{adgroup_type} eq 'base'
                    ? Models::Banner::_copy_to_banner($group, $Models::Banner::banner_fields{adgroup_base})
                    : (),
                $group->{adgroup_type} eq 'dynamic'
                    ? Models::Banner::_copy_to_banner($group, $Models::Banner::banner_fields{adgroup_dynamic})
                    : (),
                $group->{adgroup_type} eq 'mobile_content'
                    ? Models::Banner::_copy_to_banner($group, $Models::Banner::banner_fields{adgroup_mobile_content})
                    : (),
                $group->{adgroup_type} eq 'performance'
                    ? Models::Banner::_copy_to_banner($group, $Models::Banner::banner_fields{adgroup_performance})
                    : (),
                $group->{adgroup_type} eq 'content_promotion_video'
                    ? Models::Banner::_copy_to_banner($group, $Models::Banner::banner_fields{adgroup_content_promotion_video})
                    : (),
                $group->{adgroup_type} eq 'content_promotion'
                    ? Models::Banner::_copy_to_banner($group, $Models::Banner::banner_fields{adgroup_content_promotion})
                    : ();
            $banner->{tags} = $group->{tags} if $group->{tags};
            $_->{BannerID} = $banner->{BannerID} foreach @{$banner->{phrases}}; 

            # TODO удалить когда перейдем к структуре campaign - group - banners
            # т.к. значения многих полей дублируются
            $banner->{banner_minus_words} = $group->{minus_words};
            
            if (!$cid_cache{$group->{cid}}) {
                my $vals = $cid_cache{$group->{cid}} = {};
                $vals->{timetarget_coef} = TimeTarget::timetarget_current_coef($group->{timeTarget}, $group->{timezone_id});
                $vals->{camp_rest} = round2s( $group->{sum} - $group->{sum_spent} );
                if ($options->{get_add_camp_options}) {
                    if (exists $options->{camp_spent_today}) {
                        $vals->{spent_today} = $options->{camp_spent_today};
                    } elsif($group->{day_budget} && $group->{day_budget} > 0 && $group->{OrderID} && $group->{OrderID} > 0) {
                        $vals->{spent_today} = Stat::OrderStatDay::get_order_spent_today($group->{OrderID});
                    }
                    $vals->{device_targeting} = $options->{device_targeting} if $options->{device_targeting};
                }
                if ($options->{camp_bs_queue_status}) {
                    $vals->{camp_in_bs_queue} = $group->{camp_in_bs_queue};
                }
            }
            hash_merge $banner, $cid_cache{$group->{cid}};

            $banner->{group_hierarchical_multipliers} = $group->{hierarchical_multipliers};
            $banner->{group_multiplier_stats} = $group->{multiplier_stats} if exists $group->{multiplier_stats};

            $banner->{minus_geo} = join ',', uniq @{$group->{minus_geo}//[]};

            push @$banners, $banner;
        }
    }

    # забираем данные из крутилки (торги)
    my $need_phrases_clicks = 0;
    if ($options->{get_auction}) {
        # для кампаний с отключенным поиском не ходим в "полные" торги, но ходим в bs_get_phrases_statuses за количеством кликов за 28 дней
        my $banners_with_search_on;
        if ($options->{get_auction_for_search_stop_too}) {
            $banners_with_search_on = $banners;
        } else {
            $banners_with_search_on = [ grep { ($_->{strategy} || '') ne 'different_places' || $_->{platform} ne 'context' } @$banners];
        }
        if ($banners_with_search_on && @$banners_with_search_on) {
            trafaret_auction($banners_with_search_on);
        }
        if (scalar(@$banners) != scalar(@$banners_with_search_on)) {
            # если хоть по одному баннеру не сходили, то заберём клики из bs_get_phrases_statuses
            $need_phrases_clicks = 1;
        }

        foreach my $banner (@$banners) {
            $banner->{fairAuction} = exists $options->{fairAuction} ? $options->{fairAuction}: $banner->{fairAuction};
            # если стратегия - отдельное размещение, то забираем contextCoverage, нужно для API
            $banner->{get_all_phrases} = $banner->{strategy} && $banner->{strategy} eq 'different_places' ? 1 : 0;
            # для кампаний с независимым управлением(different_places) цены в РСЯ устанавливаются вручную
            Models::AdGroup::calc_context_price($banner, currency => $banner->{currency}, ContextPriceCoef => $banner->{cContextPriceCoef}) unless $banner->{strategy} && $banner->{strategy} eq 'different_places';

            # Процессим шаблон после получения торгов
            hash_merge $banner, add_banner_template_fields($banner);
        }

        unless ($options->{no_pokazometer_data}) {
            my @take_pokazometer = grep { !$_->{is_bs_rarely_loaded} && !$_->{camp_is_arch} } @$banners;
            if (@take_pokazometer) {
                safe_pokazometer(\@take_pokazometer, net => 'context', map {$_ => $options->{$_}} qw/get_all_phrases/);
            }
        }
    }

    my $full_banners_stats = {};
    for my $banner (@$banners) {
        $full_banners_stats->{$_} ||= exists $banner->{$_} for qw/retargetings target_interests dynamic_conditions performance_filters relevance_match/;
    }
    for my $g (@$groups) {
        $full_banners_stats->{image_ad} ||= all { $_->{real_banner_type} eq 'image_ad' } @{$g->{banners}};
    }

    if ($need_phrases_clicks
        || $options->{ctx_from_bs}
        || $options->{get_phrases_statuses}
        || any { $full_banners_stats->{$_} } (qw/retargetings target_interests dynamic_conditions performance_filters relevance_match image_ad/)
       )
    {
        my $bs_get_phrases_statuses_options = {
            update_phrases => $full_banners_stats->{image_ad} || $options->{ctx_from_bs} || $options->{get_phrases_statuses},
            update_retargetings => ($full_banners_stats->{retargetings} || $full_banners_stats->{target_interests}),
            update_dynamic_conditions => $full_banners_stats->{dynamic_conditions},
            update_performance => $full_banners_stats->{performance_filters},
            update_relevance_matches => $full_banners_stats->{relevance_match},
        };

        bs_get_phrases_statuses($banners, $bs_get_phrases_statuses_options);
    }


    # Если у пользователя настроен БКшный мониторинг, то пересчитать флаги statusMetricaStop.
    # Практические тут обычно приходят баннеры одной кампании, но теоретически могут быть и разных даже клиентов. Учитываем это для справедливости.
    my %banners_by_cids;
    foreach (@$banners) {
        push @{$banners_by_cids{$_->{cid}}}, $_;
    }

    foreach my $cid (uniq keys %banners_by_cids) {
        my %monitoring_stopped_banners = map {$_->id => 1} @{BS::CheckUrlAvailability::get_monitoring_stopped_banners($cid, [map {$_->{bid}} @{$banners_by_cids{$cid}}])};
        foreach (@{$banners_by_cids{$cid}}) {
            $_->{statusMetricaStop} = ($monitoring_stopped_banners{$_->{bid}}) ? 'Yes' : 'No';
        }
    }
    return ($banners, $total);
}

=head2 enrich_image_banners($groups)

Функция заполняет в баннерах с типом image_ad поля title и body из других баннеров,
в которых эти поля есть

Парметры:
    $groups - [{pid => banners => []}]

=cut

sub enrich_image_banners {
    my ($groups) = @_;

    my %mising_groups;
    for my $group (@$groups) {
        my $banner_with_text = first { ($_->{real_banner_type}//'') ne 'image_ad' } @{$group->{banners}};
        if ($banner_with_text) {
            for my $banner (@{$group->{banners}}) {
                next if ($banner->{real_banner_type}//'') ne 'image_ad';
                hash_copy $banner, $banner_with_text, qw/title body/;
            }
        } elsif ($group->{pid}) {
            $mising_groups{$group->{pid}} = $group;
        }
    }

    if (my @pids = keys %mising_groups) {
        my $banners = get_all_sql(PPC(pid => \@pids), [
            "SELECT pid, title, body FROM banners",
            where => { pid => SHARD_IDS, banner_type__ne => 'image_ad' },
            "GROUP BY pid"
        ]);
        for my $banner (@$banners) {
            for my $image (@{ $mising_groups{$banner->{pid}}->{banners} }) {
                next if $image->{real_banner_type} ne 'image_ad';
                hash_copy $image, $banner, qw/title body/;
            }
        }
    }
}

=head2 update_banner_status_moderate

  update statusModerate on banners (+banner_images) and phrases after cmd_saveupdatedBanner(), cmd_bannersMultiSave()
  Если `Сохранить в черновиках`, или статус у кампании - `New` - то statusModerate = 'New'
  Если `Отправить на модерацию`/`Сохранить` и кампания не новая - то отправляем на модерацию только(!) НОВЫЕ баннеры
  
  Результат:
    $statusModerate - установленный статус модерации баннера (undef если статус не поменялся)

  Что-то похожее (но не 1 в 1) в некоторых случаях делается в Phrase::update_banner_phrases

=cut

sub update_banner_status_moderate
{
    my ($bid, $camp_statusModerate, $FORM_save_draft) = @_;

    my (%where, %banners, %phrases);
    if ($FORM_save_draft || $camp_statusModerate eq 'New') {
        %banners = (
            statusModerate => 'New',
            phoneflag => 'New',
            statusSitelinksModerate => 'New',
            statusPostModerate => 'No'
        );
        %phrases = (
            statusModerate => 'New',
            statusPostModerate => 'No'
        );
    } else {
        %banners = (
            statusModerate => 'Ready',
            phoneflag__dont_quote => "IF(vcard_id is not NULL, 'Ready', 'New')",
            statusSitelinksModerate__dont_quote => "IF(sitelinks_set_id is not NULL, 'Ready', 'New')"
        );
        %phrases = (statusModerate => 'Ready');
        %where = (statusModerate => 'New');
    }
    my %images = ( statusModerate => $banners{statusModerate} );
    
    my $is_affected = do_update_table(PPC(bid => $bid), 'banners', {
        LastChange__dont_quote => 'LastChange',
        %banners
    }, where => {bid => SHARD_IDS, %where});
    do_update_table(PPC(bid => $bid), 'banner_images', {
        date_added__dont_quote => 'date_added',
        %images
    }, where => { bid => SHARD_IDS, %where });

    # LastChange обновится естественным образом если statusModerate/statusPostModerate обновились в БД
    do_update_table(PPC(pid => get_pids(bid => $bid)), 'phrases', {
        %phrases
    }, where => {pid => SHARD_IDS, %where});

    clear_banners_moderate_flags([$bid]);

    if ($is_affected) {
        Direct::Banners::delete_minus_geo(bid => [$bid], status => $banners{statusModerate});
    }
    
    return $is_affected ? $banners{statusModerate} : undef;
}

=head2 validate_banner($banner, $options)

    Параметры:
        $banner - {}

        $options
            exists_banners_type     -- в случае обновления баннера (if $banner->{bid}) этот параметр обязателен,
                                        хранит типы существующих баннеров {bid => 'desktop', bid2 => 'mobile' ...}
            skip_changing_banner_type -- пропустить проверку на невозможность смены типа баннера, но непосредственно тип баннера проверяться будет
            skip_contactinfo        -- если установлен, то КИ будет проигнорирована
            skip_max_phrases_length -- если установлен, то не будет проводиться проверка суммарной максимальной длины фраз в баннере
            use_banner_with_flags   -- если установлен, то ссылка и КИ будут проверены в соответствии с $banner->{banner_with_(href|phone)}
            ClientID                -- ClientID клиента для гео ф-ций для выбора транслокально дерева регионов
            is_api                  -- вызов из API для выбора API-транслокально дерева регионов
            is_mediaplan_banner     -- проверяемый баннер является медиаплановым (отличия: вместо поля bid используется mbid)
    
            если установлен флаг use_banner_with_flags, то контактная информация должна быть в $banner->{vcard}
            
            Кстати, $b -- глобальная переменная для использования в sort. Как-то неаккуратно называть параметр функции так же...

    Результат:   
        {title => ['', ''], body => ['', '']} - хеш текстовых ошибок по полям        

=cut

# TODO навести порядок в проверке href и визитки. 
#   1. хорошо бы отовсюду присылать структурированное объявление (с $b->{vcard} и banner_with_*)
#   2. обязательно иметь два режима проверки: "мягкий" и "строгий", по умолчанию -- строгий
#      Мягкий -- проверять только те данные, которые соответствуют типу объявления ( !has_href => ссылку не проверять ),
#        применять для перехода со стр. редактирования объявления на страницу ставок 
#        (лишние поля нужны: пользователь заполнил визитку, переключился на "с адресом сайта", засабмитил форму, получил ошибку -- 
#        в этот момент хорошо помнить введенные визиточные значения)
#      Строгий -- проверять все данные, применять перед сохранением объявления 
#        (к этому моменту лишние поля уже не нужны, должны быть удалены, а если остались -- опасны: вдруг они все-таки сохраняются)
#

sub validate_banner
{
    ## no critic (Freenode::DollarAB)
    my ($b, $options) = @_;
    need_list_context();
    my $errors={};
    my $bid = $options->{is_mediaplan_banner} ? $b->{mbid} : $b->{bid};
    $bid ||= 0;
    # strip bad symbols.
    smartstrip($b->{$_}, dont_replace_angle_quotes => 1) for qw/body title title_extension/;
    smartstrip($b->{href});
    
    my %params = $options->{error_prefix} ? (prefix => $options->{error_prefix}) : ();

    my $keyword_field_name = $options->{keyword_field_name} || 'Phrase'; # В API4 фраза приходит в поле Phrase, нужно для ошибки про не заданное поле, в интерфейсе по идее вообще не должна фигурировать.
    
    if( !defined $b->{phrases} ) {
        $b->{phrases} = '';
    } elsif(ref $b->{phrases} eq 'ARRAY') {

        for (@{$b->{phrases}}) {
            smartstrip $_;
        }
        strip_phrases(@{$b->{phrases}});
    } else {
        smartstrip($b->{phrases});
        strip_phrases($b->{phrases});
    }
    
    if (my $title_error = validate_banner_title($b->{title})) {
        add_banner_error_text($errors, $bid, 'title', $title_error, %params);
    }

    if (my $title_extension_error = validate_banner_title_extension($b->{title_extension})) {
        add_banner_error_text($errors, $bid, 'title_extension', $title_extension_error, %params);
    }
    
    if (my $banner_error = validate_banner_body($b->{body})) {
        add_banner_error_text($errors, $bid, 'body', $banner_error, %params)
    }
    
    my $type_error = Models::Banner::validate_banner_type($b, $options->{exists_banners_type},
                        $options->{skip_changing_banner_type}, $options->{is_mediaplan_banner});    
    if ($type_error) {
    	add_banner_error_text($errors, $bid, banner_type => $type_error, %params)
    }
    
    my $translocal_opt = $options->{is_api} ? {tree => 'api'} : {ClientID => $options->{ClientID}};
    Models::Banner::check_geo_restrictions($b, $errors, %params, %$translocal_opt) unless $options->{skip_geo_restrictions};

    unless ($options->{skip_phrases}) {
        my $brackets_errors;
        if ( ! $options->{no_phrase_brackets} ) {
            my $br_result = process_phrase_brackets($b->{phrases});
            if ($br_result) {
                add_banner_error_text($errors, $bid, 'phrases', $br_result, %params);
                $brackets_errors = 1;
            }
        }
        
        my @phrases = $b->{phrases}
                        ? ref $b->{phrases} eq 'ARRAY' ? @{$b->{phrases}} : split(/\s*,[\s,]*/, $b->{phrases})
                        : ();
        # TODO: перейти на validate_add_keywords удалив brackets_errors                        
        my $phrase_vr = base_validate_keywords(\@phrases, {brackets_errors => $brackets_errors});
        $phrase_vr->process_objects_descriptions( keyword => { field => $keyword_field_name } );
        if (!$phrase_vr->is_valid) {
            add_banner_error_text($errors, $bid, 'phrases', $phrase_vr->one_error_description_by_objects, %params);
        } elsif ( !$options->{skip_max_phrases_length} && Models::AdGroup::is_group_oversized($b, client_id => $options->{ClientID}) ) {
            # суммарную длину фраз проверяем только при отсутствии ошибок в индивидуальных фразах
            add_banner_error_text($errors, $bid, 'phrases', iget('Превышено максимальное количество ключевых фраз'), %params);
        }
        
        # проверяем параметры ссылки (a.k.a. Param1 и Param2 в API)
        if ( $options->{i_know_href_params} ) {
            my $phrase_array;
            for my $possible_phrase_array($b->{phrases}, $b->{Phrases}) {
               if ($possible_phrase_array && ref($possible_phrase_array) eq 'ARRAY' && @$possible_phrase_array && $possible_phrase_array->[0] && ref($possible_phrase_array->[0]) eq 'HASH') {
                   $phrase_array = $possible_phrase_array;
               } 
            }
            if ($phrase_array) {
                for my $phrase(@$phrase_array) {
                    for my $param_name(@Models::Phrase::BIDS_HREF_PARAMS) {
                        my $param_val = exists $phrase->{$param_name} ? $phrase->{$param_name} : $phrase->{lc($param_name)};
                        my $err = Models::Phrase::validate_phrase_one_href_param( $param_val );
                        if ($err) {
                            add_banner_error_text($errors, $bid, $param_name, $err, %params);
                        }
                    }
                }
            } else {
                send_alert('BannersCommon::validate_banner was unable to found phrases array in the following banner: ' . Dumper($b), 'validate_banner: no phrase array');
            }
        }
    }

    my $geo_error = validate_geo($b->{geo}, undef, $translocal_opt);
    if ($geo_error) {
        add_banner_error_text($errors, $bid, 'geo', iget('Регионы указаны некорректно'), %params);
    }

    add_banner_error_text($errors, $bid, 'body', validate_banner_template( $b ), %params);

    my %sitelinks_options;
    # Проверяем ссылку...
    if( defined $b->{href} && (!$options->{use_banner_with_flags} || $b->{has_href}) ) {
        $b->{href} = clear_banner_href($b->{href}, $b->{url_protocol});
        my @href_errors = validate_banner_href( $b->{href} );
        add_banner_error_text($errors, $bid, 'href', \@href_errors, %params);
    }

    # ... и сайтлинки
    if (!$options->{ignore_sitelinks} && ref $b->{sitelinks} eq 'ARRAY' && scalar @{$b->{sitelinks}}) {
        foreach (@{$b->{sitelinks}}) {
            $_->{href} = clear_banner_href($_->{href}, $_->{url_protocol});
        }
        my @sitelinks_errors = Sitelinks::validate_sitelinks_set($b->{sitelinks}, $b->{href}, ClientID => $options->{ClientID}, %sitelinks_options);
        add_banner_error_text($errors, $bid, 'sitelinks', \@sitelinks_errors, %params);
    }

    # ... отображаемую ссылку...
    if ($b->{display_href}) {
        my @dh_errors;
        push @dh_errors, iget("Задана отображаемая ссылка, но не указана основная ссылка в объявлении")  if !$b->{href};
        if (my $dh_error = Direct::Validation::Banners::validate_banner_display_href($b->{display_href} || undef)) {
            # создаём ValidationResult для обработки шаблонов
            my $vr = Direct::ValidationResult->new();
            $vr->add_generic($dh_error);
            $vr->process_descriptions(__generic => {field => iget(q/'Отображаемая ссылка'/)});
            push @dh_errors, $vr->get_first_error_description;
        }
        add_banner_error_text($errors, $bid, 'display_href', \@dh_errors, %params) if @dh_errors;
    }

    # Проверяем минус-слова
    if (defined $b->{banner_minus_words}) {
        if (my @x = @{MinusWords::check_minus_words($b->{banner_minus_words}, type => 'banner')}) {
            add_banner_error_text($errors, $bid, 'banner_minus_words', \@x, %params);
        }
    }

    # Нельзя указываать сайтлинки, если отсутствует основная ссылка
    if( (!$options->{use_banner_with_flags} || $b->{has_href}) &&
        (!defined $b->{href} || $b->{href} eq '') &&
        (!$options->{ignore_sitelinks} && Sitelinks::need_save_sitelinks_set($b->{sitelinks}))
    ) {
        add_banner_error_text($errors, $bid, 'href', iget("Отсутствует основная ссылка при наличии быстрых"), %params);
    }

    # ... и контактную информацию 
    if ($options->{use_banner_with_flags}) {
        # со временем из всего if'а должна остаться только эта ветка, а $options->{use_banner_with_flags} станет не нужна 
        if ( $b->{banner_with_phone} &&  !$options->{skip_contactinfo}) {
             add_banner_error_text($errors, $bid, 'contactinfo', [ validate_contactinfo($b->{vcard}) ], %params);
        }
        add_banner_error_text($errors, $bid, 'href', iget('Не введена ссылка на сайт'), %params) if ( $b->{has_href} && ( !defined $b->{href} || $b->{href} !~ /\S/ ) );
        add_banner_error_text($errors, $bid, 'href', iget('Не введена ссылка на сайт'), %params)
            if ( !$b->{has_href} && ! ($b->{banner_with_phone} || $b->{turbolanding} || $b->{tl_id}) );
    } elsif ( defined $b->{phone}&& $b->{phone} ne '' 
        || defined $b->{city} && $b->{city} ne '' 
        || defined $b->{country} && $b->{country} ne '' 
        || defined $b->{name} && $b->{name} ne ''
        || defined $b->{contactperson} && $b->{contactperson} ne '' 
        || defined $b->{country_code} && $b->{country_code} ne '' 
        || defined $b->{city_code} && $b->{city_code} ne '' 
        || defined $b->{ext} && $b->{ext} ne '' 
        || defined $b->{street} && $b->{street} ne ''
        || defined $b->{house} && $b->{house} ne ''
        || defined $b->{build} && $b->{build} ne '' 
        || defined $b->{apart} && $b->{apart} ne ''
        || defined $b->{metro} && $b->{metro} ne ''
        || defined $b->{im_client} && $b->{im_client} ne '' 
        || defined $b->{im_login} && $b->{im_login} ne '' 
        || defined $b->{extra_message} && $b->{extra_message} ne '' 
        || defined $b->{contact_email} && $b->{contact_email} ne '' 
    ) {
        # TODO избавиться от этой ветки (для этого отовсюду должен приходить корректный banner_with_(href|phone))

        if (not $options->{skip_contactinfo}) {
            add_banner_error_text($errors, $bid, 'contactinfo', [ validate_contactinfo($b) ], %params);
        }
    } else {
       add_banner_error_text($errors, $bid, 'href', iget('Не введена ссылка на сайт'), %params) if ( (!defined $b->{href} || $b->{href} !~ /\S/) && !($b->{turbolanding} || $b->{tl_id}));
    }


    if (!$options->{ignore_image} && $b->{image_url}) {
        unless ($b->{image_url} =~ m!^https?://! && (Yandex::IDN::is_valid_domain(get_host($b->{image_url})) || Yandex::Validate::is_valid_ip(get_host($b->{image_url}))) ) {
            add_banner_error_text($errors, $bid, 'image_url', iget('Некорректная ссылка на изображение'), %params);
        }
    }
    
    if ($options->{use_multierrors_format}) {
        return $errors;
    } else {
        my @result = convert_banner_error_text_to_list($errors);
        return @result;
    }
}

=head2 get_banner_phrases_count

    Функция возвращает число фраз в переданном баннере.
    Если передано что-то, не похожее на баннер, возвращает undef или пустой список в скалярном и списочном контекстах соответственно.

    $banner = {
        phrases => [
            [...]
        ],
    };
    $phrases_count = get_banner_phrases_count($banner);

=cut

sub get_banner_phrases_count {
    my ($banner) = @_;

    return unless $banner && ref($banner) eq 'HASH' && exists $banner->{phrases} && ref($banner->{phrases}) eq 'ARRAY';

    return scalar @{$banner->{phrases}};
}

=head2 get_banner_relevance_matches_count

    Функция возвращает число условий беcфразного таргетинга в переданном баннере.
    Если передано что-то, не похожее на баннер, возвращает undef или пустой список в скалярном и списочном контекстах соответственно.

    $banner = {
        relevance_match => [
            [...]
        ],
    };
    $relevance_matches_count = get_banner_relevance_matches_count($banner);

=cut

sub get_banner_relevance_matches_count {
    my ($banner) = @_;

    return unless $banner && ref($banner) eq 'HASH' && exists $banner->{relevance_match} && ref($banner->{relevance_match}) eq 'ARRAY';

    return scalar @{$banner->{relevance_match}};
}

=head2 get_banner_retargetings_count

    Функция возвращает число условий ретаргетинга в переданном баннере.
    Если передано что-то, не похожее на баннер, возвращает undef или пустой список в скалярном и списочном контекстах соответственно.

    $banner = {
        retargetings => [
            [...]
        ],
    };
    $retargetings_count = get_banner_retargetings_count($banner);

=cut

sub get_banner_retargetings_count {
    my ($banner) = @_;

    return 0 unless $banner && ref($banner) eq 'HASH' && exists $banner->{retargetings} && ref($banner->{retargetings}) eq 'ARRAY';

    return scalar @{$banner->{retargetings}};
}

=head2 save_manual_prices

    Сохранить цены перед началом работы автобюджета

=cut
sub save_manual_prices {
    my ($cid) = @_;
    do_sql(PPC(cid => $cid), "INSERT INTO bids_manual_prices (id, cid, price, price_context)
                        SELECT id, cid, price, price_context
                          FROM bids bi WHERE bi.cid = ?
                               ON DUPLICATE KEY UPDATE price = values(price), price_context=values(price_context)
                        ", $cid);
}

=head2 update_geo

    Обновляет регион в баннере и фразах (если переданное значение
    отличается от сохранённого в БД). Также при изменении ставит
    e-mail-нотификацию в очередь. Возвращает 1 если geo действительно
    поменялось, иначе - undef.

    Поддержка транслокальности, при сохранении таргетинга учитываем страну рекламодателя:
    - Рекламодатель из Украины выбирает регион "Украина" - выставляем ему таргетинг "Украины + Крым"
    - Рекламодатель из Украины выбирает регион "Россия" - выставляем ему таргетинг "Россия"
    - Рекламодатель из России выбирает регион "Украина" - выставляем ему таргетинг "Украина"
    - Рекламодатель из России выбирает регион "Россия" - выставляем ему таргетинг "Россия + Крым"
     https://jira.yandex-team.ru/browse/DIRECT-30368

     Именованные опции:
        is_api => 1 -- использовать API транслокальное дерево регионов (с Крымом в корне)

=cut
sub update_geo($$$;%) {
    my ($pid, $geo, $uid, %opt) = @_;
    my $old_geo;
    if (!$pid && $opt{bid}) {
        ($pid, $old_geo) = get_one_line_array_sql(PPC(bid => $opt{bid}),
         "SELECT p.pid, p.geo FROM banners b JOIN phrases p ON b.pid = p.pid WHERE b.bid = ?", 
         $opt{bid});
    } else {
        $old_geo = get_one_field_sql(PPC(pid => $pid), "SELECT p.pid, p.geo FROM phrases p WHERE p.pid = ?", $pid);
    }
    my $ClientID = get_clientid(uid => $uid);
    my $geoflag;

    if (! $opt{is_api}) {
        $geo = GeoTools::modify_translocal_region_before_save($geo, {ClientID => $ClientID});
    }

    my $translocal_opt = $opt{is_api}
                         ? {tree => 'api'}
                         : {ClientID => $ClientID};
    $geo = refine_geoid($geo, \$geoflag, $translocal_opt);

    return undef if $old_geo eq $geo;

    # Т.к. geo гарантированно поменялось, то LastChange обновится естественным образом.
    do_update_table(PPC(pid => $pid), 'phrases', {
        geo => $geo,
        statusBsSynced => 'No',
        statusShowsForecast => 'New'
    }, where => {pid => $pid});
    
    do_update_table(PPC(pid => $pid), 'banners', {
        geoflag => $geoflag,
        opts__smod => {geoflag => $geoflag},
        statusBsSynced => 'No',
        LastChange__dont_quote => 'LastChange'
    }, where => {pid => $pid});

    # TODO: Раскомментировать, когда выложится новый тип уведомлений
    #mail_notification('adgroup', 'adgr_geo', $pid, $old_geo, $geo, $uid);
    return 1;
}


=head2 is_banner_edited_by_moderator_multi(bids)

    Проверка, был ли указанные баннеры отредактированы модератором
    Возвращает ссылку на хэш:
      для редактированных баннеров - bid -> 1,
      для нередактированных ключа не будет

=cut

sub is_banner_edited_by_moderator_multi {
    my ($bids) = @_;

    return {}  if !@$bids;

    return get_hash_sql(PPC(bid => $bids), [
                                "SELECT id, 1
                                   FROM mod_edit",
                                  WHERE => { 
                                    type => 'banner',
                                    id => SHARD_IDS, 
                                    _TEXT => "createtime > DATE_SUB(current_timestamp(), interval $Settings::MOD_EDIT_EXPIRATION)"
                            }]);
}


=head2 fill_in_text_before_moderation

Добавляет к баннерам структуру before_moderation: тексты до исправления на модерации.

=cut

sub fill_in_text_before_moderation {
    my ($banners) = @_;
    return $banners  if !$banners && !@$banners;

    my $original_data = mass_get_banner_before_moderation([map {$_->{bid} || ()} @$banners]);
    for my $banner (@$banners) {
        my $original = $original_data->{$banner->{bid}};
        next if !$original;
        $banner->{before_moderation} = $original;
    }

    return $banners;
}



=head2 mass_get_banner_before_moderation(bids)

    Возвращает пользовательскую версию баннера, отредактированного модератором

=cut

sub mass_get_banner_before_moderation($) {
    my ($bids) = @_;

    my $banners = get_hashes_hash_sql(PPC(bid => $bids), [
            'SELECT id as bid, old,
                unix_timestamp(createtime) as edit_createtime,
                statusShow as showModEditNotice
            FROM mod_edit',
            WHERE => {
                type => 'banner',
                id => SHARD_IDS,
            },
        ]);

    for my $banner (values %$banners) {
        next if !$banner->{old};
        hash_merge $banner, YAML::Syck::Load(delete $banner->{old});
    }

    return $banners;
}

=head2 change_banner_modedit_status(bid, statusShow)

    Меняет statusShow для сущности, редактирвоанной модератором

=cut
sub change_banner_modedit_status($$) {
    my ($bid, $statusShow) = @_;

    return do_update_table(PPC(bid => $bid), "mod_edit",
                           {statusShow => $statusShow},
                           where => {type=>'banner', id => $bid}
        );
}


=head2 check_json_ajax_update_struct

    Проверка, что хеш, приходящий при сохранении ключевых слов со страницы просмотра кампании, имеет правильную структуру. 

=cut
sub check_json_ajax_update_struct {
    my $struct = shift;
    return 0  if !defined $struct   ||
              ref $struct ne 'HASH';

    foreach my $bid (keys(%{$struct})) {
        return 0 unless is_uint($bid);
        my $param = $struct->{$bid};
        if (exists($param->{edited})) {
            return 0 if ref $param->{edited} ne 'HASH';
            foreach my $pid (keys(%{$param->{edited}})) {
                return 0 unless is_uint($pid);
            }
        }
        if (exists($param->{added})) {
            return 0 if ref $param->{added} ne 'ARRAY';
            foreach my $ph (@{$param->{added}}) {
                return 0 if ref $ph ne 'HASH';
            }
        }
        if (exists($param->{deleted})) {
            return 0 if ref $param->{deleted} ne 'ARRAY';
            foreach my $pid (@{$param->{deleted}}) {
                return 0 unless is_uint($pid);
            }
        }
        if (exists $param->{main_bid}) {
            return 0 if ref $param->{main_bid};
            return 0 unless is_valid_id($param->{main_bid});
        }
    }
    return 1;
}

=head2 check_json_ajax_apply_unglued_minus_words_struct

    Проверка, что хеш, приходящий при утверждении или отклонении клиентом автоматической расклейки ключевиков 
    на странице просмотра кампании, имеет правильную структуру.
    Правильный хеш:
    json_minus_words = {<bid> => {<id1> => "-minus -words", 
                                  <id2> => "-more -words"},
                        ...
                       }

=cut

sub check_json_ajax_apply_unglued_minus_words_struct {
    my $struct = shift;
    return 0  if !defined $struct   ||
              ref $struct ne 'HASH' ||
              !scalar(keys(%{$struct}));
    foreach my $bid (keys(%{$struct})) {
        return 0 unless is_uint($bid);
        my $param = $struct->{$bid};
        return 0 if ref $param ne 'HASH';
        foreach my $pid (keys(%{$param})) {
            return 0 unless is_uint($pid);
            return 0 if ref \($param->{$pid}) ne "SCALAR" and ref $param->{$pid} ne 'ARRAY';
        }
    }
    return 1;
}

=head2 get_banners_short_hash

    Возвращает данные из таблицы banners и, если нужно, из campaigns
    
=cut

sub get_banners_short_hash($$;$) {
    my ($bids, $banner_fields, $campaign_fields) = @_;

    my $banner_fields_sql = join ", ", map {'b.'.$_} grep {$_ ne 'bid'} @$banner_fields;
    my $campaign_fields_sql;
    $campaign_fields_sql = join ", ", map {'c.'.$_} @$campaign_fields if $campaign_fields;

    my $fields_sql = $banner_fields_sql . ($campaign_fields ? ", $campaign_fields_sql" : "");
    my $sql = "SELECT b.bid, $fields_sql FROM banners b" . ($campaign_fields ? " JOIN phrases p ON p.pid = b.pid JOIN campaigns c ON c.cid = p.cid" : "");
    
    return $bids && scalar @$bids ? get_hashes_hash_sql(PPC(bid => $bids), [$sql, where => {"b.bid" => SHARD_IDS}]) : {};
}

# --------------------------------------------------------------------

=head2 modify_groups_geo_for_translocal_before_show

    Модифицируем гео-таргетинг (geo, geo_names) по списку групп.
    Преобразуем из API-дерева в транслокальное дерево клиента
    Опции:
        ClientID -- клиент для выбора транслокального дерева
        tree -- предопределенное транслокальное дерево

    modify_groups_geo_for_translocal_before_show($groups, {ClientID => 12345});

=cut

sub modify_groups_geo_for_translocal_before_show($$) {
    my ($groups, $opt) = @_;

    for my $group (@$groups) {
        my $geo = GeoTools::modify_translocal_region_before_show($group->{geo}, $opt);
        $group->{geo} = $geo // '0';
        $group->{geo_names} = get_geo_names($geo, ', ');
    }
}

# --------------------------------------------------------------------

=head2 get_banners_type($kind => $bids)

    Возвращает тип баннеров.
    
    Параметры:
        $kind - mediaplan|text
            mediaplan - получить тип медиаплановых баннеров 
            text - получить тип текстовых баннеров
        $bids - массив [] id баннеров
        
    Результат:
        хеш {mbid => type, ...}
            ключ - id баннера, значение - тип (тестовая строка)

=cut

sub get_banners_type {
    
    my ($kind, $bids) = @_;
    
    confess 'unknown banner kind ' . ($kind || '') unless defined $kind && $kind =~ /^(mediaplan|text)$/;
    if ($kind eq 'text') {
        return get_hash_sql(PPC(bid => $bids), ["SELECT bid, type FROM banners", WHERE => {bid => SHARD_IDS}]);
    } else {
        return get_hash_sql(PPC(mediaplan_bid => $bids), ["SELECT mbid, type FROM mediaplan_banners", WHERE => {mbid => SHARD_IDS}]);
    }
}


=head2 mass_archive_banners(%)

    Архивирует остановленные объявления, либо объявления еще не отправленные в БК.
    Принимает параметры:
      pids | bids  - идентификаторы групп|баннеров, которые необходимо заархивировать.

    Возвращает информацию о заархивированных объектах в виде хеша:
   {
     updated_bids => { bid => 1 } - идентификаторы обработанных баннеров (bid)
     updated_pids => { pid => banners_count } - идентификаторы обработанных групп (pid) и количество обновленных баннеров в группе.
                                                Поле возвращается только в случае, когда метод обрабатывает массив групп (pids)
   }
=cut

sub mass_archive_banners(%)
{
    my (%params) = @_;
    my ($key, $result) = (undef, {});

    my $ids;
    if ($params{bids}) {
        $ids = $params{bids};
        $key = 'bid';
    } elsif ($params{pids}) {
        $ids = $params{pids};
        $key = 'pid';
    } else {
        return $result;
    }
    my $update_before = Models::Banner::get_update_before();

    $ids = [$ids] unless ref $ids eq 'ARRAY';
    my $bids_to_archive = get_all_sql(PPC($key => $ids), ['SELECT b.pid, b.bid FROM banners b
            JOIN campaigns c ON b.cid = c.cid
            LEFT JOIN adgroup_priority ap on ap.pid = b.pid', WHERE => {
        'b.' . $key => SHARD_IDS,
            'b.statusArch' => 'No',
            'b.statusShow' => 'No',
            'c.archived'=> 'No',
            _OR => {
                'ap.priority__is_null' => 1,
                'ap.priority__ne' => $Settings::DEFAULT_CPM_PRICE_ADGROUP_PRIORITY
            }
    }]) || [];

    foreach (@$bids_to_archive) {
        $result->{updated_bids}->{$_->{bid}} = 1;
        $result->{updated_pids}->{$_->{pid}}++ if ($params{pids});
    }
    my $banner_ids = [map {$_->{bid}} @$bids_to_archive];
    my $adgroup_ids = [uniq map {$_->{pid}} @$bids_to_archive];

    do_update_table(PPC( bid => $banner_ids), 'banners', {statusArch => 'Yes', statusShow => 'No', statusBsSynced => 'No', LastChange__dont_quote => 'NOW()'}, where => {bid => SHARD_IDS});

    Models::Banner::update_banner_statuses_is_obsolete($banner_ids, $update_before);
    Models::Banner::update_adgroup_statuses_is_obsolete($adgroup_ids, $update_before);

    return $result;
}

1;
