package BillingAggregateTools;

use Direct::Modern;

use Carp;
use JSON;

use Yandex::DBTools;
use Yandex::DBShards;

use Settings;

use Yandex::DBShards;
use Yandex::DBQueue;

use Client;
use BS::ResyncQueue;
use Direct::BillingAggregates;
use Direct::Campaigns;
use Direct::Wallets qw//;


=head2 on_save_new_camp($class, $client_id, $cid, $operator_uid)

    Обработка события "создание новой кампании".

    Читает из базы ОС клиента и передает управление в функцию check_and_add_new_aggregates

=cut
sub on_save_new_camp {
    my ($client_id, $cid, $operator_uid) = @_;

    my $camp = Direct::Campaigns->get_by(campaign_id => $cid)->items_by->{$cid};
    if ($camp->client_id != $client_id) {
        croak 'invalid cid';
    }
    if (!$camp->wallet_id) {
        return;
    }
    my $wallet = Direct::Wallets->get_by(campaign_id => $camp->wallet_id, no_additional => 1)->items->[0];
    if (!$wallet) {
        die 'wallet '.$camp->wallet_id.' not found';
    }
    if ($wallet->client_id != $client_id) {
        die 'invalid wallet_id';
    }

    my $created = check_and_add_new_aggregates($client_id, $wallet, [$camp->campaign_type], $operator_uid);

    # если клиент еще не в новой схеме зачислений, и старым клиентам разрешено автоматически
    # переходить в новую схему, добавляем в DBQueue задание на перевод.
    if ($created && $wallet->is_sum_aggregated ne 'Yes' && Client::ClientFeatures::has_aggregated_sums_for_old_clients($client_id)) {
        _add_wallet_sum_migration_job($wallet, $operator_uid);
    }
}

=head2 check_and_add_new_aggregates($client_id, $wallet, \@camp_types, $operator_uid, %O)

    Проверяет, есть ли под ОС биллинговые агрегаты тех продуктов, которые могут быть
    показаны в кампаниях типов $camp_types
    Если нужно (включен ОС, клиент попал под фичу, включено автосоздание для типов продукта),
    досоздает агрегаты.
    Если какие-то агрегаты были созданы, кладет кампании, которые могут использовать этот агрегат,
    в ленивую очередь переотправки в БК.

    $wallet - модель Direct::Model::Wallet общего счета, под которым нужно создать агрегаты

    Именованные необязательные параметры
        log - объект Yandex::Log, если хочется что-то записать в лог

        preloaded_feature_by_clients - хешик с флагами, где указано, есть ли у клиента фича,
                                        включающая автосоздание биллинговых агрегатов.
                                        Если не указать, функция сама проверит, есть ли фича у $client_id

        preloaded_existing_aggs - хешик, который возвращает Direct::BillingAggregates->get_by(...)->items_by_wallet_and_product();

    Возвращает 1, если клиенту были созданы новые биллинговые агрегаты

=cut
sub check_and_add_new_aggregates {
    my ($client_id, $wallet, $camp_types, $operator_uid, %O) = @_;
    my $log = delete $O{log} // Yandex::Log->new(no_log => 1);;
    my $preloaded_feature_by_clients = delete $O{preloaded_feature_by_clients};
    my $preloaded_existing_aggs = delete $O{preloaded_existing_aggs};
    if (keys %O) { die "unknown params ".join(', ', keys %O); };

    my $wallet_id = $wallet->id;

    if (Direct::BillingAggregates->is_autocreate_disabled($client_id, $wallet->user_id, $wallet->currency
        , preloaded_feature_by_clients => $preloaded_feature_by_clients
    )) {
        return;
    }

    my $to_create = Direct::BillingAggregates->get_missing_product_types($client_id, $wallet_id, $camp_types
        , preloaded_existing_aggs => $preloaded_existing_aggs
    );
    if (@$to_create) {
        $log->out("Client has camps with types ".join(', ', @$camp_types)
            ." under wallet, creating billing aggregates with product types ".join(', ', @$to_create)
        );

        Direct::BillingAggregates->make_new_aggregates_for_client($client_id, $to_create, $wallet)
                ->create($operator_uid);

        my $camp_types_to_resync = Direct::BillingAggregates::get_relevant_camp_types_by_product_types($to_create);
        $log->out("Resyncing camps with types ".join(', ', @$camp_types_to_resync));
        _bs_resync_camps_with_types($wallet_id, $camp_types_to_resync);
    }

    return @$to_create > 0 ? 1 : 0;
}

sub _bs_resync_camps_with_types {
    my ($wallet_id, $camp_types_to_resync) = @_;

    my $cids = get_one_column_sql(PPC(cid => $wallet_id), ["
        SELECT cid
        FROM campaigns",
        WHERE => [
            wallet_cid => SHARD_IDS,
            statusEmpty__ne => 'Yes',
            type => $camp_types_to_resync,
        ],
    ]);

    BS::ResyncQueue::bs_resync_camps($cids
        , priority => BS::ResyncQueue::PRIORITY_ON_BILLING_AGGREGATE_ADDED
    );
}

=head2 _add_wallet_sum_migration_job($wallet, $operator_uid)

    Добавление в DBQueue задания на перевод клиента в новую схему зачислений.
    Без новой схемы биллинговые агрегаты не имеют смысла, т.к. не отправляются в БК.

=cut
sub _add_wallet_sum_migration_job {
    my ($wallet, $operator_uid) = @_;

    my $ClientID = $wallet->client_id;
    my $queue = Yandex::DBQueue->new(PPC(ClientID => $ClientID), 'convert_campaigns_aggr_schema');

    $queue->insert_job( {
        job_id => get_new_id('job_id'),
        ClientID => $ClientID,
        uid => $operator_uid,
        namespace => undef,
        args => {
            walletId => $wallet->id + 0,
            rollback => JSON::false,
            ignoreBsQueue => JSON::true,
        },
    });
}

1;
