package CairoGraph;
## no critic (TestingAndDebugging::RequireUseWarnings)

use strict;

use Data::Dumper;
use Cairo;
use Date::Calc;
use List::Util qw/max min sum/;

use Yandex::HashUtils;

use Carp();

use POSIX qw(ceil floor log log10);

use utf8;	

our @ISA = qw(Exporter);
our @EXPORT = qw(cr_draw_multi_plot cr_err_list cr_err_flush);
our $MINIMAL_SCALE = 1e-3;
our $CR_DEBUG = 0;
our $CR_ERROR = '';
our @CR_ERROR_LIST=();
our $USE_INT_CRDS = 1;
our $always_draw_month = 1; #   Draw month pie-layer on timeline even if there is only one month to draw
our %cr_get_x_by_date_hash = ();

# $SIG{__WARN__} = \&Carp::cluck if $CR_DEBUG;

our $default_style = {
    line_color  =>[0.7, 0.7, 0.7, 1.0],
#     fill_color  =>[[0.8, 0.8, 0.8, 1.0],[1.0, 1.0, 1.0, 1.0]],
#     wall_color  =>[[0.0, 0.0, 0.0, 0.15],[0.0, 0.0, 0.0, 0.0]],
    color       => [0.0, 0.0, 0.0, 1.0],
    line_color  => [0.0, 0.0, 0.0, 1.0],
    #fill_color  =>sub{($_[0]||0)%2 ? [0.8, 0.8, 0.8, 1.0]:[1.0, 1.0, 1.0, 1.0]},
    pie_line_color  => [0.8, 0.8, 0.8, 1.0],
    fill_color      => sub{($_[0]||0)%2 ? [0.0, 0.0, 0.0, 0.15]:[0.0, 0.0, 0.0, 0.0]},
    wall_color      => sub{($_[0]||0)%2 ? [0.0, 0.0, 0.0, 0.15]:[0.0, 0.0, 0.0, 0.0]},
    text_font   =>[qw/Arial normal normal/],
    blend_color =>[0.85, 0.85, 0.9, 0.8],
    blend_line_color =>[0.0, 0.0, 1.0, 1.0],
    blend_font  =>[qw/Arial normal bold/],
    font_size   => 12,
    font_color  => [0.0, 0.0, 0.0, 1.0],
    legend_font_size    => 10,
    legend_font => [qw/Arial normal normal/],
    legend_color=> [0.95, 0.95, 1.0, 1.0],
    legend_line_color   => [0.7, 0.7, 0.7, 1.0],
    title_font_color    => [0,0,0,1],
    title_font          => [qw/Arial normal bold/],
    title_font_size     => 15,
    y_axis_font         => [qw/Arial normal normal/],
    y_axis_font_size    => 10,
    warn_color          => [1.0, 1.0, 1.0, 1.0],
    warn_line_color     => [0.5, 0.5, 0.5, 1.0],
    warn_font           => [qw/Arial normal normal/],
    warn_font_size      => 14,
    warn_font_color     => [0.5, 0.5, 0.5, 1.0],
    grid_color          => [0.0, 0.0, 0.0, 0.2],
    bar_fill_color   => sub{
            [
                [0.0, 1.0, 0.0, 1.0],
                [0.0, 0.7, 0.0, 1.0],
                [0.0, 0.4, 0.0, 1.0],
                [0.0, 0.2, 0.0, 1.0]
            ]->[floor($_[0]||0)%4]
        },

    space_strength  => 1.0,
    bar_strength  => 5.0,
    edge_strength  => 1.0,
    edge_width     => 12.0,
    plot_padding    => 10
};

#my $cairo_graph_source = <<'CREOM';

=head2 cr_style($vars, $name)

    $name =~ /[bar|wall|day|week|month|season|year][fill|line]color | [text][font|color]/;

    take all styles U need in convenient form
    U don't have to know exact style name, just try to guess

=cut

sub cr_style{
    my $vars = shift;
    my $name = shift;
    return cr_err("In cr_style USAGE: cr_style(\$vars, \$name [, ...args])") unless defined $vars and defined $name;
=pod    
    my @tags = split /\_/, $name;
    return cr_err('In cr_style $name must match /\w+(_\w+)*/') unless scalar @tags;

    my $style;
    while ( scalar @tags) {
        $style = $vars->{style}->{join('_', @tags)};
        last if defined $style;
        shift @tags;
    }
=cut
    my $style;
    while ($name) {
        last if $style = $vars->{style}->{$name};
        $name =~ s/^[^_]+_?//;
    }
    
    if ( defined $style ) {
        return $style if ref $style eq '';
        return $style if ref $style eq 'ARRAY';
        return $style->(@_) if ref $style eq 'CODE';
        return $style;
    } else {
        return cr_err( "Can't find style like [$name]\n");
    }
}

=head2 cr_err([$err_str])

    log error or return last error if nothing passed

=cut

sub cr_err{
    my ($err_str) = @_;

    if ( $err_str ) {
        $CR_ERROR = $err_str;
        push @CR_ERROR_LIST, $err_str;

        print STDERR "CR_ERROR:>>:$err_str\n" if $CR_DEBUG;
        Carp::cluck if $CR_DEBUG;
        return undef;
    } else {
        return $CR_ERROR;
    }
}

sub cr_err_flush{
    $CR_ERROR = undef;
    @CR_ERROR_LIST = ();
}

sub cr_err_list{
    return @CR_ERROR_LIST;
}

sub month_to_season($){
    return {
        12=>1, 1=>1, 2=>1,
        3=>2, 4=>2, 5=>2,
        6=>3, 7=>3, 8=>3,
        9=>4,10=>4,11=>4
    }->{int $_[0]};
}
sub month_of_season($){
    return {
        12=>0, 1=>1, 2=>2,
        3=>0, 4=>1, 5=>2,
        6=>0, 7=>1, 8=>2,
        9=>0,10=>1,11=>2
    }->{int $_[0]};
}
sub month_full_name($$){
    my ($vars, $i) = @_;
    return $vars->{month_full_name}->[$i-1];
}
sub season_name($$){
    my ($vars, $i) = @_;
    return $vars->{season_name}->[$i-1];
}

sub cast_int{
    return @_ unless $USE_INT_CRDS;
    my @inted = map { floor $_} @_;
    return @inted;
}

sub cast_half_int{
    return map { $_ + 0.5 } @_ unless $USE_INT_CRDS;
    my @inted = map { floor($_) + 0.5} @_;
    return @inted;
}

=head2 cr_nice_rect

    draws a rectangle for topRight/bottomLeft corners crds given
    declared in order not to pass haight/width to cr_rectangle

=cut

sub cr_nice_rect {
    my ($cr, $x1, $y1, $x2, $y2)=@_;
    $cr->new_path();
    $cr->move_to($x1, $y1);
    $cr->line_to($x2, $y1);
    $cr->line_to($x2, $y2);
    $cr->line_to($x1, $y2);
    $cr->close_path();
    return 1;
}

=head2 cr_get_date($date)

    return valid [YYYY,mm,dd] or undef

=cut

sub cr_get_date($) {
    my $date = shift;
    if(! defined $date) {
        cr_err "undefined value couldn't be treated as date";
        return undef;
    } elsif (ref $date eq 'ARRAY' ) {
    } elsif (  $date =~ /^\D*(\d{4})\D*(\d{1,2})\D*(\d{1,2})\D*/) {
        $date = [$1,$2,$3];
    } else {
        cr_err "invalid date format:[$date]";
        return undef;
    }
    #   $date is assumed to be ARRAYREF
    return $date if Date::Calc::check_date(@$date);

    cr_err "invalid date value:[".join('.', @$date)."]";
    return undef;
}

=head2 cr_date_floor($date, $period)

    returns the beginning of period (day|week|month|season|quarter|year) containing the date

=cut

sub cr_date_floor($$){
    my ($date, $period) = @_;

    return undef unless defined $date and ref $date eq 'ARRAY';
    return undef unless defined $period;

    if ( $period eq 'day' ) {
        return [@$date];
    } elsif ( $period eq 'week' ) {
        return [Date::Calc::Monday_of_Week(Date::Calc::Week_of_Year(@$date))];
    } elsif ( $period eq 'month' ) {
        return [$date->[0],$date->[1],1];
    } elsif ( $period eq 'season' ) {
        return [Date::Calc::Add_Delta_YM($date->[0],$date->[1],1,0,-month_of_season($date->[1]))];
    } elsif ($period eq 'quarter') {
        return [$date->[0],(int(($date->[1] - 1)/3)*3 + 1),1];
    } elsif ( $period eq 'year' ) {
        return [$date->[0],1,1];
    } else {
        return undef;
    }
}

=head2 cr_date_add($date, $period)

    returns the beginning of the period (day|week|month|season|quarter|year)
    following the period containing the date

=cut

sub cr_date_add($$){
    my ($date, $period) = @_;

    return $date unless defined $date and ref $date eq 'ARRAY';
    return $date unless defined $period;

    if ( $period eq 'day' ) {
        $date = [Date::Calc::Add_Delta_Days(@$date,1)];
    } elsif ( $period eq 'week' ) {
        return [Date::Calc::Add_Delta_Days(Date::Calc::Monday_of_Week(Date::Calc::Week_of_Year(@$date)),7)];
    } elsif ( $period eq 'month' ) {
        return [Date::Calc::Add_Delta_YM($date->[0],$date->[1],1,0,1)];
    } elsif ( $period eq 'season' ) {
        return [Date::Calc::Add_Delta_YM($date->[0],$date->[1],1,0,3-month_of_season($date->[1]))];
    } elsif ( $period eq 'quarter' ) {
        return [Date::Calc::Add_Delta_YM($date->[0],(int(($date->[1] - 1)/3)*3 + 1),1,0,3)];
    } elsif ( $period eq 'year' ) {
        return [Date::Calc::Add_Delta_YM($date->[0],1,1,1,0)];
    } else {
        return $date;
    }
}


sub cr_make_markup($$){
    my ($cr, $vars) = @_;

    $vars->{markup_date}=[];
    my $markup = $vars->{markup};

    for (my $date = $vars->{date_from}; Date::Calc::Delta_Days(@$date, @{$vars->{date_to}})>=0;) {

        my $date_start = cr_date_floor($date, $markup);
        my $date_end = cr_date_add($date, $markup);
        $date = $date_end;

        push @{$vars->{markup_date}}, {date_start=>$date_start, date_end=>$date_end};
    }

    %cr_get_x_by_date_hash=();
    return $vars->{markup_date};
}

=head2 cr_check_vars($cr, $vars)

    $cr - Cairo context;
    $vars - all plot parametrs

=cut

sub cr_check_vars($$){
    my ($cr, $vars) = @_;

    return cr_err "No variable hash providev"
        unless defined $vars and ref $vars eq 'HASH';

    return cr_err "Empty or wrong title\n".Dumper($vars->{title})
        unless defined $vars->{title} and ref $vars->{title} eq '';

    #   timeline options
    return cr_err "Empty or wrong markup\n".Dumper($vars->{markup})
        unless defined $vars->{markup} and $vars->{markup} =~ /day|week|month|quarter|year/;

    $vars->{date_from} = cr_get_date($vars->{date_from});
    return cr_err "Empty or wrong date_from\n"
        unless defined $vars->{date_from};

    $vars->{date_to} = cr_get_date($vars->{date_to});
    return cr_err "Empty or wrong date_to\n"
        unless defined $vars->{date_to};

    return cr_err "End_date should be grater then Start_date\n" unless
        Date::Calc::Delta_Days(@{$vars->{date_from}}, @{$vars->{date_to}})>=0;

    cr_make_markup($cr, $vars);

    #data options
#     return cr_err "No name list is specified" unless
#         defined $vars->{names} and ref $vars->{names} eq 'ARRAY';

    unless ( defined $vars->{plots} and ref $vars->{plots} eq 'ARRAY' ) {
        $vars->{plots} =
            [
                {
                    points  =>  $vars->{points},
                    names   =>  $vars->{names}
                }
            ];
    }

    for my $plot (@{$vars->{plots}}) {
        return undef unless cr_check_points($cr, hash_merge(
            $plot,
            {
                markup_date =>  $vars->{markup_date},
                markup      =>  $vars->{markup}
            }
        ));
    }

    unless ( defined $vars->{number_postfix} and
        ref $vars->{number_postfix} eq 'ARRAY') {
        $vars->{number_postfix}=[];
    }

    for my $m (0..5) {
        $vars->{number_postfix}->[$m] ||= (undef, qw/тыс млн млрд трлн млнмлрд/)[$m];
    }

    # coordinate options
    for my $key (qw/x y width height/) {
        return cr_err "cr_check_vars: $key showld be numeric"
            unless defined $vars->{$key} and $vars->{$key} =~ m/\d*[\.\,]\d+|\d+(?:[\.\,]\d*)?/;
    }

    unless (defined $vars->{month_full_name} and ref $vars->{month_full_name} eq 'ARRAY') {
        $vars->{month_full_name} = [];
    }
    for my $m (0..11) {
        $vars->{month_full_name}->[$m] ||= (qw/январь февраль март апрель май июнь июль август сентябрь октябрь ноябрь  декабрь/)[$m];
    }

    unless (defined $vars->{season_name} and ref $vars->{season_name} eq 'ARRAY') {
        $vars->{season_name} = [];
    }
    for my $m (0..3) {
        $vars->{season_name}->[$m] ||= (qw/зима весна лето осень/)[$m];
    }

    $vars->{style} = hash_merge({}, $default_style, hash_dump($vars->{style})||{} );

    $vars->{width_scale} = ($vars->{width} - 2*cr_style($vars, 'edge_width')) / (2*cr_style($vars, 'edge_strength') + cr_style($vars, 'bar_strength') + (cr_style($vars, 'space_strength')+cr_style($vars, 'bar_strength'))*(-1 + scalar @{$vars->{markup_date}}));
    return 1;
}

=head2 cr_check_points($cr, $vars)

    $

=cut

sub cr_check_points($$){
    my ($cr, $vars) = @_;

    return cr_err "No point list is specified" unless
        defined $vars->{points} and ref $vars->{points} eq 'ARRAY';

        #   Изменение структуры данных для построения
    my @points = map { ## no critic (ProhibitComplexMappings)
            my $e=$_;
            if (ref $e eq 'ARRAY') {
                {date=>$e->[0], y=>$e->[1]};
            } elsif (ref $e eq 'HASH') {
                $e;
            } else {
                undef;
            }
        } @{$vars->{points}};

    #   Отсеивание плохих точек
    $vars->{sort_points} =
        [
            ## здесь у Freenode::DollarAB ложное срабатывание:
            sort { 0 <=> Date::Calc::Delta_Days(@{$a->{date}}, @{$b->{date}}) } ## no critic (Freenode::DollarAB)
                map { ## no critic (ProhibitComplexMappings)
                    my $e=$_;
                    my @date = map { sprintf "%d", $_} $e->{date} =~ /^\D*(\d{4})\D*(\d{1,2})\D*(\d{1,2})\D*$/;
                    $e->{date}=\@date;
                    $e->{ys} = (ref $e->{y} eq 'ARRAY') ? $e->{y} : [$e->{y}];
                    $e->{y_sum} = sum(@{$e->{ys}}) ;
                    $e
                }
                grep {
                    my $e=$_;
                    defined $e and ref $e eq 'HASH' and
                    defined $e->{date} and $e->{date} =~ /^\D*(\d{4})\D*(\d{1,2})\D*(\d{1,2})\D*$/ and
                    defined $e->{y} and (
                        ( ref $e->{y} eq '' and $e->{y} =~ /^\-?(\d+\.?|\d*[\.\,]\d+)$/) or
                        ( ref $e->{y} eq 'ARRAY' and
                            not scalar grep { not defined $_ or  $_ !~ /^\-?(\d+\.?|\d*[\.\,]\d+)$/} @{$e->{y}})
                        )
                } @points
        ];

#     return cr_err "No valid points left" unless scalar @{$vars->{sort_points}};

    for my $point (@{$vars->{sort_points}}) {

        $point->{date} = cr_date_floor($point->{date}, $vars->{markup});
        $point->{x} = cr_get_x_by_date($point->{date}, $vars->{markup_date});
    }
    return 1;
}

=head2 cr_draw_multi_plot($cr, $vars)

    Darws multiple pltos on one image with the same timeline

    Different plots have different y_scales

=cut

sub cr_draw_multi_plot($$){
    my ($cr, $vars) = @_;

    cr_err_flush;

    return undef unless cr_check_vars($cr, $vars);

    #   fill bg with white
    unless ( grep {not defined $cr->get_target->can($_)} qw/get_height get_width/ ) {

        $cr->rectangle(0,0,$cr->get_target->get_width, $cr->get_target->get_height);
        $cr->set_source_rgb (1.0, 1.0, 1.0);
        $cr->fill;
    }

    my $plot_paddding = cr_style($vars, 'plot_padding');
    my $plots_cnt = scalar @{$vars->{plots}};

    my @sub_vars_arr=();
    my $cnt=0;

    for my $plot (@{$vars->{plots}}) {
        my $sub_vars = hash_merge(
            {},
            $vars,
            $plot,
            {
                y       =>  $vars->{y}-($vars->{height}+$plot_paddding)*$cnt/$plots_cnt,
                height  =>  ($vars->{height}+$plot_paddding)/$plots_cnt - $plot_paddding
            }
        );

        cr_draw_gist($cr, $sub_vars);
        cr_draw_y_grid($cr, $sub_vars);
        cr_draw_axis_y($cr, $sub_vars);

        #   Запоминаем текущие параметры, чтобы отрисовать легенду на верхнем слое
        push @sub_vars_arr, $sub_vars;
        $cnt++;
    }

    cr_draw_date_bar($cr, $vars);
    #cr_draw_axis_x($cr, $vars);
    #cr_draw_box($cr, $vars);
    cr_draw_title($cr, $vars);

    $cnt=0;
    for my $sub_vars (@sub_vars_arr) {
        cr_draw_box($cr, $sub_vars);
        cr_draw_legend($cr, $sub_vars);

        cr_draw_warning($cr, $sub_vars);

        #   Не рисуем белую отбивку под нижним графиком
        next if $cnt++ == 0;#-1 +scalar @sub_vars_arr;

        cr_nice_rect($cr, cast_int
            $sub_vars->{x},
            $sub_vars->{y}+1,
            $sub_vars->{x}+$sub_vars->{width}+1,
            $sub_vars->{y}+$plot_paddding);

        $cr->set_source_rgba (1,1,1,1);
        $cr->fill;
    }

    return 1;
}

=head2 cr_draw_gist($cr,$x,$y,$width,$height, $points, $markup, $plot_type, $title, $vars)

    $points - array ref containing data to plot:
        [
            {   date    => 'YYYY-MM-DD',
                {y}     => [d1,d2,d3,d4... ]|d1
            },
            {   date    => 'YYYY-MM-DD',
                {y}     => [d1,d2,d3,d4... ]|d1
            },...
        ]
        dN is considered not to be negative

=cut

sub cr_draw_gist($$){
    my ($cr, $vars) = @_;

    #my ($min_x, $max_x);
    my ($min_y, $max_y);

    my @sort_points = @{$vars->{sort_points}};

    for my $point (@sort_points) {

        $min_y = (! defined $min_y or $min_y > $point->{y_sum}) ? $point->{y_sum} : $min_y;
        $max_y = (! defined $max_y or $max_y < $point->{y_sum}) ? $point->{y_sum} : $max_y;
    }
    $min_y = 0;
    $max_y ||= 0;

    my $big_y = max(abs($max_y), abs($min_y), $max_y - $min_y, $MINIMAL_SCALE||1e-10);

    my $lg = floor( log10($big_y));

    my $y_scale;
    if ( $big_y/ 10**$lg < 2) {
        $y_scale = 10**$lg * 0.5;
    } elsif($big_y/ 10**$lg < 5) {
        $y_scale = 10**$lg * 1.0;
    } else {
        $y_scale = 10**$lg * 2.0;
    }
    $vars->{y_scale} = $y_scale;

    $vars->{Min_y} = 0;  #$y_scale*floor($min_y/$y_scale);

    $vars->{Max_y} = $y_scale*ceil(max($max_y, $min_y + $MINIMAL_SCALE||1e-10)/$y_scale * 1.24);

    my ($ew, $es, $bs, $ss) = map {cr_style($vars, $_)} qw(edge_width edge_strength bar_strength space_strength);

    my $no_holes = 0;#($#sort_points >= 31) ? 1 : 0;
    my ($left_hole_size)  = map { $USE_INT_CRDS ? (floor $_) : $_ } ( $ss*$vars->{width_scale} * 0.5 );
    my ($hole_size)       = map { $USE_INT_CRDS ? (floor $_): $_ } ( $ss*$vars->{width_scale} );


    for my $point (@sort_points) {

        my ($x1) = map {cr_get_x_by_date($_, $vars->{markup_date})} ($point->{date});

        my $base_y=0;
        my $layer_id=0;
        for my $ycrd (@{$point->{ys}}) {

            cr_nice_rect ($cr, map { $USE_INT_CRDS ? floor($_) : $_ }
                $vars->{x} + $ew + $vars->{width_scale}*($es+$x1*($ss+$bs) ),
                $vars->{y}-$vars->{height}*($base_y-$vars->{Min_y})/($vars->{Max_y}-$vars->{Min_y}),
                $vars->{x} + $ew + $vars->{width_scale}*($es+($x1+1)*($bs+$ss) ) - $hole_size,
                $vars->{y}-$vars->{height}*($base_y+$ycrd-$vars->{Min_y})/($vars->{Max_y}-$vars->{Min_y}) );


            $cr->set_source_rgba ( @{cr_style($vars, 'bar_fill_color', $layer_id)} );
            $cr->fill;

            cr_nice_rect ($cr, map { 0.5 + ( $USE_INT_CRDS ? floor($_) : $_ ) }
                $vars->{x} + $ew + $vars->{width_scale}*($es+$x1*($ss+$bs) )-1,
                $vars->{y}-$vars->{height}*($base_y-$vars->{Min_y})/($vars->{Max_y}-$vars->{Min_y}),
                $vars->{x} + $ew + $vars->{width_scale}*($es+($x1+1)*($bs+$ss) )-$hole_size-1,
                $vars->{y}-$vars->{height}*($base_y+$ycrd-$vars->{Min_y})/($vars->{Max_y}-$vars->{Min_y}) );

            $cr->set_source_rgba ( @{cr_style($vars, 'bar_line_color', $layer_id)} );
            $cr->set_line_width (1.0);
            $cr->stroke;

            $base_y += $ycrd;
            $layer_id ++;
        }
    }
    return 1;
}

sub cr_draw_title($$){
    my ($cr, $vars) = @_;

    $cr->select_font_face (@{cr_style($vars, 'title_font')});
    $cr->set_font_size (cr_style($vars, 'title_font_size'));
    $cr->set_source_rgba (@{cr_style($vars, 'title_font_color')});
    my $ext = $cr->text_extents($vars->{title});
    $cr->move_to(
        cast_int $vars->{x},
        $vars->{y}-$vars->{height}-$ext->{height}*0.5 );

    $cr->show_text ($vars->{title});
    return 1;
}

sub cr_draw_warning($$){
    my ($cr, $vars) = @_;

    my $text = $vars->{warn_text} or return;

    my $fsize = cr_style($vars, 'warn_font_size');

    cr_nice_rect($cr, cast_int
        $vars->{x}+$vars->{width}*0.2,
        $vars->{y}-$vars->{height}*0.7,
        $vars->{x}+$vars->{width}*0.8,
        $vars->{y}-$vars->{height}*0.3);

    $cr->set_source_rgba( @{cr_style($vars, 'warn_color')} );
    $cr->fill();

    cr_nice_rect($cr, cast_half_int
        $vars->{x}+$vars->{width}*0.2,
        $vars->{y}-$vars->{height}*0.7-1,
        $vars->{x}+$vars->{width}*0.8-1,
        $vars->{y}-$vars->{height}*0.3);

    $cr->set_source_rgba ( @{cr_style($vars, 'warn_line_color')} );
    $cr->stroke();

    $cr->select_font_face (@{cr_style($vars, 'legend_font')});
    $cr->set_font_size ($fsize);
    
    $cr->select_font_face (@{cr_style($vars, 'warn_font')});
    $cr->set_font_size ($fsize);
    $cr->set_source_rgba (@{cr_style($vars, 'warn_font_color')});

    my $ext = $cr->text_extents($text);
    $cr->move_to(cast_int
        $vars->{x}+$vars->{width}/2-$ext->{width}/2-$ext->{x_bearing},
        $vars->{y}-$vars->{height}/2+$fsize*0.5 );


    $cr->show_text ($text);
    return 1;
}

sub cr_draw_legend($$){
    my ($cr, $vars) = @_;

    my $fsize = cr_style($vars, 'legend_font_size');

    my ($x, $y, $width, $height) =
        (
            $vars->{x}+$vars->{width}*0.7,
            $vars->{y}-$vars->{height}+(0.5 + 1.5 * scalar grep {$_} @{$vars->{names}})*$fsize+4,
            $vars->{width}*0.3-20,
            (0.5 + 1.5*scalar grep {$_} @{$vars->{names}})*$fsize
        );

    #   Пропускаем значения undef, они нужны что-бы явно указывать неподписываемые данные
    return undef unless ref $vars->{names} eq 'ARRAY' and (scalar grep {$_} @{$vars->{names}}) > 0;

    cr_nice_rect($cr, cast_int
        $x,
        $y,
        $x+$width,
        $y-$height);

    $cr->set_source_rgba( @{cr_style($vars, 'legend_color')} );
    $cr->fill();

    cr_nice_rect($cr, cast_half_int
        $x,
        $y-1,
        $x+$width-1,
        $y-$height);

    $cr->set_source_rgba ( @{cr_style($vars, 'legend_line_color')} );
    $cr->stroke();

    $cr->select_font_face (@{cr_style($vars, 'legend_font')});
    $cr->set_font_size ($fsize);

    my $cnt=0;
    my $color_cnt=-1;
    for my $name (@{$vars->{names}}) {
        $color_cnt++;
        next unless $name;
        cr_nice_rect($cr, cast_int
            $x+$fsize*1.0,# + ($width-$fsize*2) * $cnt/(scalar @{$vars->{names}}),
            $y-$fsize*0.5 - (1.0+1.5*$cnt)*$fsize,
            $x+$fsize*2.0,# + ($width-$fsize*2) * $cnt/(scalar @{$vars->{names}}),
            $y+$fsize*0.5 - (1.0+1.5*$cnt)*$fsize);

        $cr->set_source_rgba ( @{cr_style($vars, 'bar_fill_color', $color_cnt)} );
        $cr->fill();

        cr_nice_rect($cr, cast_half_int
            $x+$fsize*1.0,# + ($width-$fsize*2) * $cnt/(scalar @{$vars->{names}}),
            $y-$fsize*0.5 - (1.0+1.5*$cnt)*$fsize,
            $x+$fsize*2.0 -1,# + ($width-$fsize*2) * $cnt/(scalar @{$vars->{names}})-1,
            $y+$fsize*0.5 - (1.0+1.5*$cnt)*$fsize -1);

        $cr->set_source_rgba ( @{cr_style($vars, 'bar_line_color', $color_cnt)} );
        $cr->stroke();

        my $ext = $cr->text_extents($name);
        $cr->move_to(cast_int
            $x+$fsize*3.0,# + ($width-$fsize*2) * $cnt/(scalar @{$vars->{names}}),
            $y+$fsize*0.5 - (1.1+1.5*$cnt)*$fsize,
            #$y-$height/2+$fsize*0.5,
            );

        $cr->show_text ($name);
        $cnt++;
    }
    return 1;
}

sub cr_draw_box($$){
    my ($cr, $vars) = @_;
    #   Рисуем рамку
    $cr->set_source_rgb (0, 0, 0);
    $cr->set_line_width (1);
    $cr->new_path;
    $cr->move_to(cast_half_int $vars->{x},$vars->{y});
    $cr->line_to(cast_half_int $vars->{x},$vars->{y}-$vars->{height});
    $cr->line_to(cast_half_int $vars->{x}+$vars->{width},$vars->{y}-$vars->{height});
    $cr->line_to(cast_half_int $vars->{x}+$vars->{width},$vars->{y});
    $cr->line_to(cast_half_int $vars->{x},$vars->{y});
    $cr->close_path;

    $cr->stroke;
    return 1;
}

sub cr_draw_y_grid($$){
    my ($cr, $vars) = @_;
    return 0 if $vars->{just_null_y_axis};

    for (my $dy=0; $dy<=$vars->{Max_y}-$vars->{Min_y}; $dy+=$vars->{y_scale} ) {

        $cr->set_source_rgba ( @{cr_style($vars, 'y_grid_color')} );
        $cr->set_line_width (1);
        $cr->new_path;

        $cr->move_to(cast_half_int
            $vars->{x},
            $vars->{y}-$vars->{height}*$dy/($vars->{Max_y}-$vars->{Min_y}));
        $cr->line_to(cast_half_int
            $vars->{x}+$vars->{width}-1,
            $vars->{y}-$vars->{height}*$dy/($vars->{Max_y}-$vars->{Min_y}));
        $cr->close_path;

        $cr->stroke;
    }
    return 1;
}

=head2 cr_draw_logo($cr, $x, $y, $width, $height);

drawing nise logo from SVG file

sub cr_draw_logo($$$$$){
    my($cr, $x, $y, $width, $height)=@_;
    my $logofile = 'yalogo.png';
    return undef unless ( -e $logofile);

    my $rsvg = new Image::LibRSVG();
    $rsvg->loadImageHandle( "yalogo.svg" );

    my $dim = $rsvg->GetDimentions;

    if ( $dim ) {
        $rsvg->cairoDraw(
            $cr,
            $x + ($width - $dim->{width})/2,
            $y - $height +($height - $dim->{height})/2
        );
    }

    $rsvg->DESTROY;
    return 1;
}

=cut



=head2 cr_get_x_by_date($date, $vars)

    $date - ARRAYREF, VALID date
    $vars->{markup_date} - ARRAYREF of consequent periods:

        [
            {
                date_start  => [YYYY,mm,dd],
                date_end    => [YYYY,mm,dd]
            }, ...
        ]

      N.B. Delta_Days(
                @{$vars->{markup_date}}[$n+1]->{date_start},
                @{$vars->{markup_date}}[$n]->{date_end}
            ) == 0;

=cut

#my $S = 0;
sub cr_get_x_by_date($$) {
    my ($date, $markup_date) = @_;
    
    #my $ssstart = Time::HiRes::time();

    return cr_err("cr_get_x_by_date USAGE: cr_get_x_by_date(\$date ARRAYREF, \$markup_date ARRAYREF)") unless defined $markup_date and ref $markup_date eq 'ARRAY' and defined $date and ref $date eq 'ARRAY';

    my $date_hash_key = join('_',(scalar @$markup_date),@{$markup_date->[0]{date_start}},@{$markup_date->[-1]{date_end}},@$date);
    if (defined  $cr_get_x_by_date_hash{$date_hash_key} ) {
        #print STDERR "  HASHED:>>:$date_hash_key\t VALUE:>>:",$cr_get_x_by_date_hash{$date_hash_key},"\n";
        #$S += Time::HiRes::time() - $ssstart;
        return $cr_get_x_by_date_hash{$date_hash_key};
    }

    my $cnt = ceil(0.5 * ((scalar @$markup_date)-1) );
    my $i=$cnt;
    my $j=0;
    while (1) {
        die "Maximun iteration count exceeded, 'cr_get_x_by_date' seems to be broken" unless $j++<20;

        $cnt = ceil(0.5 * $cnt);
        my $delta;
        if (Date::Calc::Delta_Days(@{$markup_date->[int($i)]->{date_start}}, @$date) < 0) {
            $delta = -1;
        } elsif (Date::Calc::Delta_Days(@{$markup_date->[int($i)]->{date_end}}, @$date) <=0) {
            last;
        } else {
            $delta = 1;
        }

        if ( $delta > 0 ) {
            $i = min((scalar @$markup_date-1), $i+$cnt);
        } else {
            $i = max(0, $i-$cnt);
        }
    }

    my $ans =  $i+(Date::Calc::Delta_Days(@{$markup_date->[$i]->{date_start}}, @$date))/(Date::Calc::Delta_Days(@{$markup_date->[$i]->{date_start}}, @{$markup_date->[$i]->{date_end}}));

    $cr_get_x_by_date_hash{$date_hash_key} = $ans;
    #$S += Time::HiRes::time() - $ssstart;
    return $ans;
}

#END {print "elapsed: $S\n"}

=head2  cr_draw_date_bar($cr,$x,$y,$size,$data)

  $size = {    space_strength => 1.0,
                 bar_strength => 5.0,
                edge_strength => 2.0      };
  ($date_from, $date_to) = ("YYYY,mm,dd","YYYY,mm,dd");
  $mode =~ (day|week|month|quarter|year|season);
  ref $mode =~ (|ARRAY|HASH);

  $atr = {
      hole_l  => 1,
      hole_r  => 1
  }

=cut

sub cr_draw_date_bar($$){
    #my ($cr,$x,$y,$width,$height,$size,$date_from,$date_to,$mode,$markup, $atr) = @_;
    my ($cr, $vars) = @_;
    my $date_from = $vars->{date_from};
    my $date_to = $vars->{date_to};

    #   'Настоящие' граничные даты отображаемого диапазона
    my ($date_first ,$date_last)=($vars->{markup_date}[0]->{date_start}, $vars->{markup_date}[-1]->{date_end});

    my $date_last_1 = [Date::Calc::Add_Delta_Days(@$date_last,-1)];
    my $delta_date = {
        day => Date::Calc::Delta_Days(@$date_first, @$date_last_1),

       week => Date::Calc::Delta_Days (
                 Date::Calc::Monday_of_Week(Date::Calc::Week_of_Year(@$date_first)),
                 Date::Calc::Monday_of_Week(Date::Calc::Week_of_Year(@$date_last_1))
                                        ) / 7,

      month => ($date_last_1->[0] - $date_first->[0])*12
               + $date_last_1->[1] - $date_first->[1],

     season => ( ($date_last_1->[0] - $date_first->[0])*12
               + $date_last_1->[1] - $date_first->[1]
               - month_of_season($date_last_1->[1])
               + month_of_season($date_first->[1]) )/3,

       year => $date_last_1->[0] - $date_first->[0]
                        };

    my ($ord, $mcnt) = ({},0);
    for my $mt (split /\|/, 'day|week|month|season|year'){
        $ord->{$mt}=$mcnt++;    #   Номер слоя отображения подписей
    }

    my $pie_cnt=0;  #   номер текущего слоя подписей
#    for my $draw_mode (grep { $mode->{$_};1 } split /\|/, 'day|week|month|season|year'){
    my $wall_paint_level=undef; #   слой на котором произведена заливка
    my $is_drawn={};            #   отрисованные слои
    my $markup = $vars->{markup};

    my ($ew, $es, $bs, $ss) = map {cr_style($vars, $_)} qw(edge_width edge_strength bar_strength space_strength);

    #   пробегаем по всем слоям
    for my $draw_mode (split /\|/, 'day|week|month|season|year'){

        next if $ord->{$markup} > $ord->{$draw_mode} or $pie_cnt>2;
        next if $delta_date->{$draw_mode} > 31 or $delta_date->{$draw_mode} > 18 and $draw_mode ne 'day' ;
        next if $delta_date->{$draw_mode} < 2 and $draw_mode eq 'season';

        #   не отображать подписи периодов, если запрошенный интервал захватывает только один период данного типа
        #   Месяц - исключение, всегда отображать слой с месяцами
        next if not $delta_date->{$draw_mode}
            and $pie_cnt
            and not ($draw_mode eq 'month' and $always_draw_month);

        my $dont_paint_walls = ($delta_date->{$draw_mode} < 3
                                or $delta_date->{$draw_mode} > 18
                                or defined $wall_paint_level and $wall_paint_level ne $draw_mode
                                );

        if (  1
              #and ($draw_mode ne 'week' or $dont_paint_walls)
              and not ($delta_date->{$draw_mode} > 10 and $draw_mode eq 'season')
              and not ($is_drawn->{day} and $draw_mode eq 'week')
            ) {
            $pie_cnt++;
        }

        print STDERR "mode:$draw_mode:\n" if $CR_DEBUG;

        $cr->select_font_face ( @{cr_style($vars, 'text_font')} );
        $cr->set_font_size ( cr_style($vars, 'text_font_size') );
        my ($str_h, $day_w) = ($cr->text_extents('00')->{height},$cr->text_extents('00')->{width});

        my $cnt=0;  #   счётчик количества подписей в строке
        for (my $date = $date_first; Date::Calc::Delta_Days(@$date, @$date_last)>0;$cnt++) {

            my $date_start = $date;
            my $is_first = (Date::Calc::Delta_Days(@$date_start, @$date_first) == 0)?1:0;

            $date = cr_date_floor($date, $draw_mode);

            my $name='';
            if ( $draw_mode eq 'day' ) {
                $name = sprintf "%d", $date->[2];
            } elsif ( $draw_mode eq 'week' ) {
                if ( $is_first ){
                    $name = ($always_draw_month or $delta_date->{month}) ? sprintf("%02d", $date_first->[2]) : sprintf("%02d.%02d", $date_first->[2], $date_first->[1]);
                } else {
                    $name = ($always_draw_month or $delta_date->{month})
                    ? sprintf("%02d", $date->[2]) : sprintf("%02d.%02d", $date->[2], $date->[1]);
                }
            } elsif ( $draw_mode eq 'month' ) {
                $name = ($delta_date->{$draw_mode} < 10) ?
                    month_full_name($vars, $date->[1]) :
                    substr( month_full_name($vars, $date->[1]),0,3 );
            } elsif ( $draw_mode eq 'season' ) {
                $name = season_name($vars, month_to_season($date->[1]));
            } elsif ( $draw_mode eq 'quarter' ) {
                # для кварталов отдельной подписи периодов - нет!
                # $name = (int(($mm - 1)/3)*3 + 1)." ".iget("квартал")." ".$yyyy;
                # если понадобится - то нужно будет добавить логику про кварталы еще в несколько мест, по аналогии с месяцами или годами.
            } elsif ( $draw_mode eq 'year' ) {
                $name = sprintf "%d", $date->[0];
            }
            $date = cr_date_add($date, $draw_mode);

            my $date_end = (Date::Calc::Delta_Days(@$date, @$date_last)>0) ? $date : $date_last;

            my $is_last  = (Date::Calc::Delta_Days(@$date_end  , @$date_last ) == 0)?1:0;

            my $str = $name;
            my $ext = $cr->text_extents($str);

            my ($x1, $x2) = map {cr_get_x_by_date($_, $vars->{markup_date})} ($date_start, $date_end);

            if ( $ord->{$markup} < $ord->{$draw_mode} and
                 not $dont_paint_walls ) {
                 $wall_paint_level = $draw_mode;

                cr_nice_rect($cr, cast_int
                    $vars->{x} + $ew*(1-$is_first) + $vars->{width_scale}*($es + $x1*($ss+$bs)-($is_first ? $es : ($ss/2))),
                    $vars->{y},
                    $vars->{x} + $ew*(1+$is_last) + $vars->{width_scale}*($es + $x2*($ss+$bs)+($is_last ? $ss : ($ss/2))-$ss),
                    $vars->{y}-$vars->{height});

                $cr->set_source_rgba ( @{cr_style($vars, 'wall_color', $cnt)} );
                $cr->fill;
            }

            if ( 1
                 #and ($dont_paint_walls or $draw_mode ne 'week')
                 and not ($delta_date->{$draw_mode} > 10 and $draw_mode eq 'season')
                 and not ($is_drawn->{day} and $draw_mode eq 'week')
                ) {

                $is_drawn->{$draw_mode} = 1;

               cr_nice_rect($cr, cast_int
                    $vars->{x} + $ew*(1-$is_first) + $vars->{width_scale}*($es + $x1*($ss+$bs)-($is_first ? $es : ($ss/2))),
                    $vars->{y}+($pie_cnt-1)*$str_h*1.6,
                    $vars->{x} + $ew*(1+$is_last) + $vars->{width_scale}*($es + $x2*($ss+$bs)+($is_last ? $ss : ($ss/2))-$ss),
                    $vars->{y}+($pie_cnt)*$str_h*1.6);

                $cr->set_source_rgba ( @{cr_style($vars, 'pie_fill_color', $cnt)} );
                $cr->fill;

                cr_nice_rect($cr, cast_half_int
                    $vars->{x} + $ew*(1-$is_first) + $vars->{width_scale}*($es + $x1*($ss+$bs)-($is_first ? $es : ($ss/2))),
                    $vars->{y}+($pie_cnt-1)*$str_h*1.6,
                    $vars->{x} + $ew*(1+$is_last) + $vars->{width_scale}*($es + $x2*($ss+$bs)+($is_last ? $ss : ($ss/2))-$ss),
                    $vars->{y}+($pie_cnt)*$str_h*1.6);

                $cr->set_source_rgba ( @{cr_style($vars, 'pie_line_color', $cnt)} );
                $cr->stroke;

                if ( not $is_last and not $is_first or
                    $ext->{width} + $str_h*0.2 < $vars->{width_scale}* (cr_style($vars, 'bar_strength')+cr_style($vars, 'space_strength'))* (cr_get_x_by_date($date_end,$vars->{markup_date}) - cr_get_x_by_date($date_start,$vars->{markup_date})) ) {

                        $cr->set_source_rgba ( @{cr_style($vars, 'font_color', $cnt)} );

                        if ( $draw_mode eq 'week' ) {
                            $cr->move_to(
                                $vars->{x} + $ew + $vars->{width_scale}*($es + $x1*($ss+$bs))-$ext->{x_bearing},
                                $vars->{y} + $str_h*1.2 + ($pie_cnt-1)*$str_h*1.6);
                        } else {
                            $cr->move_to(
                                $vars->{x} + $ew + $vars->{width_scale}*($es+($x1+$x2)*0.5*($ss+$bs)-$ss/2)-$ext->{width}/2 - $ext->{x_bearing},
                                $vars->{y} + $str_h*1.2 + ($pie_cnt-1)*$str_h*1.6);
                        }

                        $cr->show_text ($str);

    #   Рисуем рамочку вокруг текста, нужно для центрирования надписей и подбора правильных отступов

    #                     $cr->rectangle($vars->{x} + $vars->{width_scale}*(cr_style($vars, 'edge_strength') + (cr_get_x_by_date($date_start,$vars->{markup_date})+cr_get_x_by_date($date_end,$vars->{markup_date}))*0.5*(cr_style($vars, 'space_strength')+cr_style($vars, 'bar_strength')) - cr_style($vars, 'space_strength')/2) - $ext->{width}/2 , $vars->{y} + $str_h*1.3 + ($pie_cnt-1)*$str_h*1.6,
    #                             $ext->{width}, -$ext->{height});
    #                             $cr->set_source_rgb (1.0, 0.0, 0.0);
    #                             $cr->stroke;

            #             print STDERR sprintf( "%.02f\t%.02f\t%.02f\tOK\n",
            #                 cr_get_x_by_date($date_start,$vars->{markup_date}),
            #                 $vars->{x} + $vars->{width_scale}*(cr_style($vars, 'edge_strength') + cr_get_x_by_date($date_start,$vars->{markup_date})*(cr_style($vars, 'space_strength')+cr_style($vars, 'bar_strength'))-($is_first ? cr_style($vars, 'edge_strength') : cr_style($vars, 'space_strength')/2)),
            #                             $vars->{y}+($pie_cnt-1)*$str_h*1.6);
                }
            }
        }

#         cr_nice_rect($cr, cast_int
#             $vars->{x},
#             $vars->{y}+($pie_cnt-1)*$str_h*1.6,
#             $vars->{x} + $ew + $vars->{width_scale}*$es,
#             $vars->{y}+($pie_cnt)*$str_h*1.6);
# 
#         $cr->set_source_rgba ( @{cr_style($vars, 'pie_fill_color', 0)} );
#         $cr->fill;
# 
#         cr_nice_rect($cr, cast_half_int
#             $vars->{x},
#             $vars->{y}+($pie_cnt-1)*$str_h*1.6,
#             $vars->{x} + $ew + $vars->{width_scale}*$es,
#             $vars->{y}+($pie_cnt)*$str_h*1.6);
# 
#         $cr->set_source_rgba ( @{cr_style($vars, 'pie_line_color', 0)} );
#         $cr->stroke;
# 
#         my $dx = scalar @{$vars->{markup_date}};
# 
#         cr_nice_rect($cr, cast_int
#             $vars->{x} + $ew + $vars->{width_scale}*($es + $dx*($ss+$bs)-$ss/2),
#             $vars->{y}+($pie_cnt-1)*$str_h*1.6,
#             $vars->{x} + $ew*2 + $vars->{width_scale}*($es*2 + ($dx+1)*($ss+$bs)),
#             $vars->{y}+($pie_cnt)*$str_h*1.6);
# 
#         $cr->set_source_rgba ( @{cr_style($vars, 'pie_fill_color', 0)} );
#         $cr->fill;
# 
#         cr_nice_rect($cr, cast_half_int
#             $vars->{x} + $ew + $vars->{width_scale}*($es + $dx*($ss+$bs)-$ss/2),
#             $vars->{y}+($pie_cnt-1)*$str_h*1.6,
#             $vars->{x} + $ew*2 + $vars->{width_scale}*($es*2 + ($dx+1)*($ss+$bs)),
#             $vars->{y}+($pie_cnt)*$str_h*1.6);
# 
#         $cr->set_source_rgba ( @{cr_style($vars, 'pie_line_color', 0)} );
#         $cr->stroke;
        
    }

    my $dd_start = Date::Calc::Delta_Days(@$date_first, @$date_from );
    if ( $dd_start >= 0 ) {

        my $pos = cr_get_x_by_date($date_from,$vars->{markup_date});

        my $right_x =  $vars->{x} + $ew + $vars->{width_scale}*($es + floor($pos) *($ss+$bs) + (($dd_start != 0)?($bs*($pos - floor($pos))):(-$ss/2))  );

        cr_nice_rect($cr, cast_int
            $vars->{x},
            $vars->{y},
            $right_x,
            $vars->{y}-$vars->{height});

        $cr->set_source_rgba ( @{cr_style($vars, 'blend_color')} );
        $cr->fill;

        $cr->new_path;
        $cr->move_to(map { 0.5 + ($USE_INT_CRDS ? (floor $_):$_) }
            $right_x-1,
            $vars->{y}-1);
        $cr->line_to(map { 0.5 + ($USE_INT_CRDS ? (floor $_):$_) }
            $right_x-1,
            $vars->{y}-$vars->{height});
        $cr->close_path;

        $cr->set_line_width (1);
        $cr->set_source_rgba ( @{cr_style($vars, 'blend_line_color')} );
        $cr->stroke;

        cr_draw_text($cr, {
                text => sprintf("%02d.%02d.%04d", reverse(@$date_from)),
                font_face   => cr_style($vars, 'blend_font'),
                font_size   => cr_style($vars, 'blend_font_size'),
                color       => cr_style($vars, 'blend_font_color'),
                x => $right_x,
                y => $vars->{y} - 20,
                align => 0.0,
                valign => 0.3,
                angle => -3.14159/2
            }
        );

    }

    my $dd_end = Date::Calc::Delta_Days(@$date_to, @$date_last_1 );
    if ( $dd_end >= 0 ) {

        my $pos = cr_get_x_by_date([Date::Calc::Add_Delta_Days(@$date_to,1)],$vars->{markup_date});

        my $left_x =  $vars->{x} + $ew + $vars->{width_scale}*($es + floor($pos) *($ss+$bs) + (($dd_end != 0)?($bs*($pos - floor($pos))):(-$ss/2)) );

        my $right_x =  $vars->{x} + $vars->{width};

        cr_nice_rect($cr, map { 0.5 + ($USE_INT_CRDS ? (floor $_):$_) }
            $left_x,
            $vars->{y},
            $right_x,
            $vars->{y}-$vars->{height});

        $cr->set_source_rgba ( @{cr_style($vars, 'blend_color')} );
        $cr->fill;

        $cr->new_path;
        $cr->move_to(map { 0.5 + ($USE_INT_CRDS ? (floor $_):$_) }
            $left_x,
            $vars->{y}-1);
        $cr->line_to(map { 0.5 + ($USE_INT_CRDS ? (floor $_):$_) }
            $left_x,
            $vars->{y}-$vars->{height});
        $cr->close_path;

        $cr->set_line_width (1);
        $cr->set_source_rgba ( @{cr_style($vars, 'blend_line_color')} );
        $cr->stroke;

        cr_draw_text($cr, {
                text => sprintf("%02d.%02d.%04d", reverse(@$date_to)),
                font_face   => cr_style($vars, 'blend_font'),
                font_size   => cr_style($vars, 'blend_font_size'),
                color       => cr_style($vars, 'blend_font_color'),
                x => $left_x,
                y => $vars->{y} - 20,
                align => 0.0,
                valign => -1.2,
                angle => -3.14159/2
            }
        );
    }
    return 1;
}

=head2 cr_draw_text($cr, $atr);

  Nice func for text rendering

  USAGE:  everything except font is optional
    cr_draw_text($cr, {
            text => join('.', @$date_to), #text
            font => 'Arial bold 12',      #'FName [italic] [bold] [size]'
            x => $left_x,                 #pix
            y => $y - 20,                 #pix
            align => 0.0,                 #
            valign => -1.3,               #
            angle => -3.14159/2,          #rad
            color => [0.5, 0.5, 0.5, 0.5] #rgba
        }
    );

=cut

sub cr_draw_text($$){
    my ($cr, $atr) = @_;

    $cr->save;

    my $text = $atr->{text}||'UNDEFINED';#join('.', @$date_from);

    $cr->set_font_size ($atr->{font_size});
    $cr->select_font_face (@{$atr->{font_face}});

    $cr->set_source_rgba(@{$atr->{color} || [0.0, 0.0, 0.0, 1.0]});
    $cr->move_to( $atr->{x}||0, $atr->{y}||0);
    $cr->rotate ( $atr->{angle} || 0 );
    my $ext = $cr->text_extents($text);

    my $x_shift=0;
    if ( ($atr->{x_align} || $atr->{align} || 'left') =~ /^(center|right|left|\-?\d+([\.\,]\d*))$/ ) {
        $x_shift = $1;
        $x_shift = {center=>0.5,    right=>1.0, left=>0.0}->{$x_shift} || 0 unless $x_shift=~/\-?\d+([\.\,]\d*)/;
    }

    my $y_shift=0;
    if ( ($atr->{y_align} || $atr->{valign} || '0.5') =~ /^(center|top|bottom|\-?\d+([\.\,]\d*))$/ ) {
        $y_shift = $1;
        $y_shift = {center=>-0.5,    top=>0.0, bottom=>-1.0}->{$x_shift} || 0 unless $y_shift=~/\-?\d+([\.\,]\d*)/;
    }

    $cr->rel_move_to(-$ext->{width} * $x_shift, -$ext->{height} * $y_shift );

    $cr->show_text ($text);

    $cr->restore;
    return 1;
}

sub cr_draw_axis_y($$){
    #my ($cr, $x, $y, $height, $text,$y_scale,$min_y,$max_y) = @_;
    my ($cr, $vars) = @_;

    my $min_y = $vars->{Min_y};
    my $max_y = $vars->{Max_y};

    $cr->set_source_rgb (0, 0, 0);
    $cr->set_line_width (1);
    $cr->new_path;
    $cr->move_to(cast_half_int $vars->{x},$vars->{y});
    $cr->line_to(cast_half_int $vars->{x},$vars->{y}-$vars->{height});
    $cr->close_path;

    $cr->stroke;

    my $lg = floor( log10($vars->{y_scale}) );

    for (my $dy=0; $dy<$max_y-$min_y; $dy+=$vars->{y_scale} ) {

        $cr->set_source_rgb (0.3, 0.3, 0.3);
        $cr->set_line_width (1);
        $cr->new_path;

        $cr->move_to(cast_half_int $vars->{x},   $vars->{y}-$vars->{height}*$dy/($max_y-$min_y) );
        $cr->line_to(cast_half_int $vars->{x}-5, $vars->{y}-$vars->{height}*$dy/($max_y-$min_y) );
        $cr->close_path;

        $cr->stroke;
        my @letters = @{$vars->{number_postfix}};
        my $fsize = cr_style($vars, 'y_axis_font_size');
        $cr->select_font_face (@{cr_style($vars, 'y_axis_font')});
        $cr->set_font_size ($fsize);

        if ( $vars->{y_scale}>1 or $dy+$min_y == 0 ) {
            my $digit = sprintf("%d",($dy+$min_y)/1000**floor($lg/3));

            if ( $dy+$min_y != 0 and defined $letters[floor($lg/3)] ) {
                my $ext = $cr->text_extents($digit);
                $cr->move_to(cast_half_int
                    $vars->{x}-$ext->{width}-$ext->{x_bearing}-$fsize*0.5,
                    $vars->{y}-$vars->{height}*$dy/($max_y-$min_y)-$fsize*1.5 );
                $cr->show_text ($digit);

                $ext = $cr->text_extents( $letters[floor($lg/3)] );
                $cr->rel_move_to(cast_half_int
                    -$ext->{width}-$ext->{x_bearing},
                    $fsize*1.0);
                $cr->show_text ( $letters[floor($lg/3)] );
            } else {
                my $ext = $cr->text_extents($digit);
                $cr->move_to(cast_half_int
                    $vars->{x}-$ext->{width}-$ext->{x_bearing}-$fsize*0.5,
                    $vars->{y}-$vars->{height}*$dy/($max_y-$min_y)-$fsize*0.5 );
                $cr->show_text ($digit)
            }
        } else {
            my $digit = sprintf(sprintf("%%.%02df",-$lg),$dy+$min_y);
            my $ext = $cr->text_extents($digit);
            $cr->move_to(cast_half_int
                $vars->{x}-$ext->{width}-$ext->{x_bearing}-$ext->{height}*0.5,
                $vars->{y}-$vars->{height}*$dy/($max_y-$min_y)-$ext->{height}*0.5 );

            $cr->show_text ($digit);
        }

        last if $vars->{just_null_y_axis};
    }
    return 1;
}

=head2 hash_dump

    transform a ftee-like hash structure to hash

    from

        a:
            b
            c
            d:
                q
                w
                e

    to

        a_b
        a_c
        a_d_q
        a_d_w
        a_d_e

=cut

sub hash_dump{
    my $h = shift;
    return $h unless $h and ref $h eq 'HASH';
    my $dump = {};

    while ( my($k,$v) = each(%$h) ) {
        if ( defined $v and ref $v eq 'HASH' ) {
            while ( my($kv,$vv) = each(%{hash_dump($v)}) ) {
                $dump->{$k."_".$kv}=$vv;
            }
        } else {
            $dump->{$k}=$v;
        }
    }
    return $dump;
}

=pod

CREOM


$cairo_graph_source = qq~use Time::HiRes qw(gettimeofday tv_interval);
~.$cairo_graph_source;

$cairo_graph_source =~ s/\#.*$//mg;

$cairo_graph_source =~ s/^=(head[0-6]|pod).*?^=cut//smg;
#$cairo_graph_source =~ s/[\n.]*\n=cut//mg;

$cairo_graph_source =~ s/(^\s*sub\s+([A-Za-z_]+)\s*(?:\(\$+\))?\s*\n*\{\n*)/$1
    my \$cr_func_name = qq~$2~;
    my \$cr_begin_time = [gettimeofday];
    /mxg;

$cairo_graph_source =~ s/(\s*)(return\s*[^\;]+?)(\bif\b|\bunless\b|\;)/
    do{
        if ( defined \$cr_begin_time ) {
            my \$cr_current_time = [gettimeofday];
            my \$cr_elapsed_time = tv_interval ( \$cr_begin_time, \$cr_current_time );
            print STDERR "CR_TIME_REPORT:>>:\$cr_func_name:>>:\$cr_elapsed_time\\n" if \$cr_elapsed_time > 1e-07;
        }
        $2;
    }$3/mxg;

# $cairo_graph_source =~ s/(^\}\s*sub\s*)/
#     if ( defined \$cr_begin_time ) {
#         my \$cr_current_time = [gettimeofday];
#         my \$cr_elapsed_time = tv_interval ( \$cr_begin_time, \$cr_current_time );
#         print STDERR "CR_TIME_REPORT:>>:\$cr_func_name:>>:\$cr_elapsed_time\\n" if \$cr_elapsed_time > 1e-07;
#     }
# $1/mxg;


# $cairo_graph_source =~ s/(\s*)(return\s*[^\;]+?)(\;)/if ( defined \$cr_begin_time ) {my \$cr_current_time = [gettimeofday];my \$cr_elapsed_time = tv_interval ( \$cr_begin_time, \$cr_current_time );print STDERR "CR_TIME_REPORT:>>:\$cr_func_name:>>:\$cr_elapsed_time\\n" if \$cr_elapsed_time > 1e-07;}$1$2$3/mxg;

#$cairo_graph_source =~ s/(\s*)(return\s*[^\;]+?)(\;|unless|if)/do{if ( defined \$cr_begin_time ) {my \$cr_current_time = [gettimeofday];my \$cr_elapsed_time = tv_interval ( \$cr_begin_time, \$cr_current_time );print STDERR "CR_TIME_REPORT:>>:\$cr_func_name:>>:\$cr_elapsed_time\\n" if \$cr_elapsed_time > 1e-07;}$1$2;}$3/mxg;


print STDERR "###################\n###################\n",$cairo_graph_source,"###################\n###################\n";

eval($cairo_graph_source);
if($@){
    print STDERR "CR_EVAL:>>:\n$@\n$!\n";
}

=cut

1;
