use Direct::Modern;

package CampAutoPrice::Common;

use JSON;
use List::MoreUtils qw/none/;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::I18n;
use Yandex::Compress;
use Yandex::HashUtils;

use Settings;
use HashingTools;
use PhrasePrice;
use PlacePrice;
use Yandex::Validate qw/is_valid_int/;

use base qw/Exporter/;

our @EXPORT = qw/
    add_camp_auto_price_queue
    validate_camp_auto_price_params
    clear_auto_price_queue
/;
our @EXPORT_OK = qw/
    convert_base_place_from_templates
/;

=head2 add_camp_auto_price_queue

  добавить в очередь задание на изменение цен в кампании

  add_camp_auto_price_queue($cid, $UID, {price_base        => $FORM{price_base}
                                       , proc              => $FORM{proc}
                                       , proc_base         => $FORM{proc_base}
                                       , max_price         => $FORM{max_price}
                                       , update_phrases    => ($FORM{update_phrases} ? 1 : 0)
                                        });

  add_camp_auto_price_queue($cid, $UID, {single_price => $FORM{single_price});

=cut

sub add_camp_auto_price_queue
{
    my ($cid, $operator_uid, $params) = @_;

    # allow "," instead "."
    for (qw/single_price proc max_price/) {
        $params->{$_} =~ s/,/./ if $params->{$_};
    }

    my $count = 0;

    if ($params->{update_phrases}  || ! $params->{update_retargetings}) {
        # сколько обновлений фраз или рубрик
        $count += get_one_field_sql(PPC(cid => $cid), "select count(*)
                                          from bids bi
                                          where bi.cid = ?
                                         ", $cid) || 0;
    }

    if (# если явно обновляем условия ретаргетинга
    	$params->{update_retargetings}
    	# или устанавливаем "единую цену на всё". copy-paste из set_camp_auto_price
    	|| $params->{single_price} && $params->{platform} && $params->{platform} =~ /^(network|both)$/
        || $params->{cpm_banner}
        || $params->{cpm_deals}
    ) {
        # сколько обновлений условий
        $count += get_one_field_sql(PPC(cid => $cid), q/
                SELECT COUNT(DISTINCT br.ret_id)
                  FROM bids_retargeting br
                       JOIN phrases p ON p.pid = br.pid
                       JOIN banners b ON b.pid = p.pid
                 WHERE p.cid = ?
                       AND b.statusArch = 'No'
            /, $cid
        ) || 0;
    }

    # если обновляем цены на фильтрах (ДМО-кампаний)
    if ($params->{single_price_CPC} || $params->{single_price_CPA}) {
        $count += get_one_field_sql(PPC(cid => $cid), q/
                SELECT COUNT(*)
                  FROM bids_performance bp
                    JOIN phrases p ON p.pid = bp.pid
                    JOIN banners b ON b.pid = p.pid
                 WHERE p.cid = ?
                   AND b.statusArch = 'No'
                   AND bp.is_deleted = 0
            /, $cid
        ) || 0;
    }

    my $params_json = to_json($params);
    return do_insert_into_table(PPC(cid => $cid), 'auto_price_camp_queue', {cid => $cid
                                                             , number_of_phrases => $count
                                                             , operator_uid => $operator_uid
                                                             , params_hash => url_hash($params_json)
                                                             , params_compressed => mysql_compress($params_json)
                                                              });
}

# ---------------------------------------------------------------------------------------------------------------------------

# валидация мастера цен для РСЯ
sub _auto_price_network
{
    my $params=shift;
    my %errors;

    my $price = $params->{max_price};

    my $error = validate_phrase_price($price, $params->{currency});
    $errors{max_price} = $error if $error;

    unless (defined $params->{scope} && $params->{scope} =~ m/^\d+$/ && $params->{scope} > 0 && $params->{scope} <= 100) {
        $errors{scope} = iget("Некорректное значение охвата аудитории");
    }
    
    return \%errors;
}

# валидация мастера цен для поиска
sub _auto_price_params_search
{
    my ($params, $currency) = @_;

    my %errors; 

    die 'no currency given' unless $currency;

    my $price = $params->{max_price};

    my $error = validate_phrase_price($price, $currency);
    $errors{max_price} = $error if $error;

    $params->{proc} =~ s/,/./ if defined $params->{proc};
        
    if ($params->{use_position_ctr_correction}) {
        my $value = $params->{position_ctr_correction};
        unless (defined($value) && ($value eq 'max' || is_valid_int($value, 1, 100))) {
            $errors{position_ctr_correction} = iget("Некорректное значение объёма трафика");
        }
    } else {
        if (! defined $params->{price_base} || none { $_ == $params->{price_base} } @{PhrasePrice::get_available_places($params)}){
                $errors{price_base} = iget("PriceBase должно обозначать позицию, относительно которой будет выставлена ставка");
        }
    
        if (! defined $params->{proc_base} || $params->{proc_base} !~ /^(?:value|diff)$/){
                $errors{proc_base} = iget("ProcBase должно иметь значение value или diff");
        }        
    }
    
    if (! defined $params->{proc} || $params->{proc} !~ /^\d+(?:[\.,]\d+)?$/ || $params->{proc} < 0 || $params->{proc} > 1000){
            $errors{proc} = iget("Неправильно указаны проценты от цены");
    }
    
    if (! defined $params->{update_phrases} || ! $params->{update_phrases} ){
         $errors{update_phrases}    = iget("Необходимо указать, менять ли цены для фраз");
    }                

    return \%errors;
}

# валидация мастера цен в РСЯ для камании со стратегией независимое управление
sub _auto_price_context {
    my ($params, $currency) = @_;

    my %errors;

    die 'no currency given' unless $currency;

    my $err = validate_phrase_price($params->{max_price}, $currency);
    $errors{max_price} = $err if $err;
    
    $errors{scope} = iget('Некорректное значение охвата аудитории')
        unless (defined $params->{scope} && $params->{scope} =~ m/^\d+$/ && $params->{scope} > 0 && $params->{scope} <= 100);
    
    $errors{proc} = iget('Неправильно указаны проценты от цены')
        if !defined $params->{proc} || $params->{proc} !~ /^\d+(?:[\.,]\d+)?$/ 
            || $params->{proc} < 0 || $params->{proc} > 1000;

    if ((! defined $params->{update_phrases} && ! defined $params->{update_retargetings})
        ||
        (! $params->{update_phrases} && ! $params->{update_retargetings})
       )
    {
         $errors{update_phrases} = iget("Необходимо указать менять ли цены для фраз или условий подбора аудитории");
    }                
            
    return \%errors;
}

# валидация мастера цен для кампании со стратегией независимое управление
sub _auto_price_different_places {

    my $params = shift;
    my %errors;
    my $cnt = 0;

    my $currency = $params->{currency};
    die 'no currency given' unless $currency;

    if ($params->{single}) {
        foreach (qw/price price_ctx/) {
            ++$cnt,next unless defined $params->{single}->{$_};
            my $err;
            if ($params->{cpm_banner} || $params->{cpm_deals}) {
                $err = validate_cpm_price($params->{single}->{$_}, $currency);
            } elsif ($params->{cpm_yndx_frontpage}) {
                $err = validate_frontpage_price($params->{single}->{$_}, $currency, $params->{frontpage_types}, $params->{geo}, $params->{client_id});
            } else {
                $err = validate_phrase_price($params->{single}->{$_}, $currency);
            }
            $errors{"single_$_"} = $err if $err;
        }
    } else {
        if ($params->{on_search} || $params->{on_ctx}) {
            hash_merge \%errors, _auto_price_params_search($params->{on_search}, $currency) if $params->{on_search};
            hash_merge \%errors, _auto_price_context($params->{on_ctx}, $currency) if $params->{on_ctx};
        } else {
            $cnt = 2;
        }
    }
    
    $errors{strange} = iget('Задайте параметры в мастере цен') if $cnt == 2;
    
    return keys %errors ? \%errors: undef;
}

=head2 validate_camp_auto_price_params

    validate_camp_auto_price_params(
    {
      single_price => 0.10
    , price_base => $PlacePrice::PLACES{PREMIUM4}
    , proc       => 10              # +10% от price_base
    , proc_base  => 'value|diff'    # +10% 'value' - от цены; 'diff' - от разницы
    , max_price  => 30              # но не выше 30 y.e.
    , update_phrases => 1           # изменять цены фраз
    , update_retargetings => 1      # изменять цены для условий ретаргетинга
    , platform => both              # менять цены и для поиска, и для РСЯ
    , scope => 80                   # охватывать 80% аудитории РСЯ
    , change_all_banners => 1       # изменять все объявления
    , currency => 'RUB'|'USD'|...   # валюта кампании
    , geo                           # для cpm_yndx_frontpage
    , frontpage_types               # массив типов главной страницы(allowed_frontpage_types) для cpm_yndx_frontpage
   });
=cut

sub validate_camp_auto_price_params{
    my $params = shift;
    my %errors;

    my $currency = $params->{currency};
    die 'no currency specified for validate_camp_auto_price_params' unless $currency;

    if ($params->{cpm_yndx_frontpage} && (!$params->{frontpage_types} || !$params->{geo})) {
        die 'for cpm_yndx_frontpage "frontpage_types" and "geo" is required';
    } elsif (!$params->{cpm_yndx_frontpage} && ($params->{frontpage_types} || $params->{geo})) {
        die '"frontpage_types" and "geo" need only for cpm_yndx_frontpage';
    }

    if ($params->{for_different_places}) {
        hash_merge \%errors, _auto_price_different_places($params);
    } elsif (defined $params->{single_price}) {
        my $price = $params->{single_price};

        my $error = validate_phrase_price($price, $currency);
        $errors{single_price} = $error if $error;

        $errors{platform} = iget("Неверный параметр типа фраз") unless $params->{platform} =~ m/^(search|network|both)$/;

    } elsif (defined $params->{single_price_CPC} || defined $params->{single_price_CPA}) {
        for my $price_type (qw/single_price_CPC single_price_CPA/) {
            if (defined $params->{$price_type}) {
                my $error = validate_phrase_price($params->{$price_type}, $currency);
                $errors{$price_type} = $error if $error;
            }
        }
    } 
    elsif($params->{platform} eq 'network' ) {
        hash_merge \%errors, _auto_price_network($params, $currency);
    } 
    elsif($params->{platform} eq 'search' ) {
        hash_merge \%errors, _auto_price_params_search($params, $currency);
    } 
    elsif($params->{platform} eq 'both') {
        hash_merge \%errors, _auto_price_params_search($params, $currency);
        hash_merge \%errors, _auto_price_network($params, $currency);
    }
    elsif ($params->{platform} !~ m/^(search|network|both)$/) {
        $errors{platform} = iget("Неверный параметр типа фраз");
    } 
    else {    
        $errors{strange} = iget("Ошибка!");
    }

    return keys %errors ? \%errors: undef; 
}


=head2 clear_auto_price_queue

    очистить очередь на установку цены по кампании (ppc.auto_price_camp_queue),
    используется при изменении цен, фраз на кампании

    clear_auto_price_queue($cid);
    clear_auto_price_queue([$cid1, $cid2]);
    clear_auto_price_queue([$cid1, $cid2], error_str => $clear_reason);

=cut
sub clear_auto_price_queue {
    my ($cids, %O) = @_;
    $cids = [$cids] if ref $cids ne 'ARRAY';

    foreach_shard cid => $cids, sub {
        my ($shard, $cids_chunk) = @_;
        my $ids = get_one_column_sql(PPC(shard => $shard), [
                                         "SELECT id FROM auto_price_camp_queue", 
                                         WHERE => {status => "Wait", cid => $cids_chunk}]
            ) || [];
        if (@$ids) {
            do_update_table(PPC(shard => $shard), 'auto_price_camp_queue'
                       , {status => 'Error'
                          , send_time => 'NOW()'
                          , error_str => $O{error_str} || 'phrases or bids already changed from interface/api'
                         }
                       , where => {id => $ids}
                       , dont_quote => ['send_time']
                   );
        }
    };
}

=head2 convert_base_place_from_templates($base_place)

    Конвертирует относительную ставку для мастера цен из формата, используемого в шаблонах,
    в значение из %PlacePrice::PLACES

=cut

sub convert_base_place_from_templates {
    my ($base_place) = @_;

    state $base_place_map = {
        min => PlacePrice::get_guarantee_entry_place(),
        max => $PlacePrice::PLACES{GUARANTEE1},
        pmin => PlacePrice::get_premium_entry_place(),
        pmax2 => $PlacePrice::PLACES{PREMIUM2},
        pmax => $PlacePrice::PLACES{PREMIUM1},
    };

    return $base_place_map->{$base_place};
}

1;
