package CampAutoPrice::Process;

use Direct::Modern;

use Yandex::I18n;
use Yandex::HashUtils;
use Yandex::ListUtils qw/chunks/;
use Yandex::DBTools;

use Settings;
use PhrasePrice;
use PhrasesAuction;
use Retargeting;
use Models::AdGroup;
use Primitives qw//;
use PrimitivesIds;
use Campaign;
use Common;
use Currencies qw/
    currency_price_rounding
    get_currency_constant
    CPM_PRICE_CONSTANT_NAMES
/;

use Direct::Bids;
use Direct::DynamicConditions;
use Direct::PerformanceFilters;
use Direct::Model::BidRelevanceMatch::Helper;

use parent qw/Exporter/;
our @EXPORT = qw/
    set_camp_auto_price
/;

# скоько фраз должно попадать в каждую пачку при получении данных из торгов и показометра (по факту бьется на пачки по группам баннеров)
our $PROCESS_PHRASES_CHUNK_SIZE = 8000;

# ---------------------------------------------------------------------------------------------------------------------------

=head2 set_camp_auto_price

    Подготовка цен для автоматической установки
    set_camp_auto_price($cid, $params, $options)
                        , );
    $params = {price_base => $PlacePrices::PLACES{...} # позиция, относительно которой нужно выставить ставку
                , proc       => 10             # +10% от price_base
                , proc_base  => 'value|diff'   # +10% 'value' - от цены; 'diff' - от разницы
                , max_price  => 30             # но не выше 30 y.e.
                , update_phrases => 1          # изменять цены фраз
                , update_retargetings => 1     # изменять цены условий ретаргетинга
                , change_all_banners => 1      # менять цены на всех баннерах, включая черновики, иначе только на активных (используется при загрузке баннеров из csv или установке цен через API)
                , adgroup_ids => [], # обновлять только указанные группы
                , keyword_ids => []  # обновлять только указанные ключевые слова
                , for_different_places # если заранее известна стратегия
                , for => 'API|front'    # откуда вызывается мастер цен
                , mobile_content => 1/0 # выставить в 1, если редактируются ставки в мобильных объявлениях
    }

=cut

sub set_camp_auto_price {
    my ($campaign_id, $params, $options) = @_;
    die "set_camp_auto_price: error cid is not valid: $campaign_id\n" if ! $campaign_id || $campaign_id !~ /^\d+$/;
    my $campaign = get_camp_info($campaign_id, undef, short => 1);
    return iget('Кампания %d не существует', $campaign_id) unless $campaign;
    $params->{currency} = $campaign->{currency};

    my $client_chief_uid = $campaign->{uid};

    my $adgroup_ids;
    if (exists $params->{adgroup_ids}) {
        $adgroup_ids = filter_arch_groups($params->{adgroup_ids});
    } else {
        $adgroup_ids = filter_arch_groups(get_pids(cid => $campaign_id));
    }
    my $keyword_ids = exists $params->{keyword_ids} ? $params->{keyword_ids} : undef;

    return 'no_active_ads_flag'
        if ($campaign->{archived} && $campaign->{archived} eq 'Yes')
        || ! scalar @$adgroup_ids;

    my $strategy = campaign_strategy($campaign_id);
    
    my $has_extended_relevance_match = Campaign::has_context_relevance_match_feature($campaign->{type}, $campaign->{ClientID});
    
    # Отдельная обработка для смарт-кампаний (доступно при любой стратегии, кроме оптимизации конверсий с ограничением по недельному бюджету и по ROI и ДРР)
    if ($campaign->{type} eq 'performance' && $strategy->{name} ne 'autobudget' && $strategy->{name} ne 'autobudget_roi' && $strategy->{name} ne 'autobudget_crr') {
        my $new_price_cpc = currency_price_rounding($params->{single_price_CPC}, $campaign->{currency});
        my $new_price_cpa = currency_price_rounding($params->{single_price_CPA}, $campaign->{currency});

        my $perf_filters = Direct::PerformanceFilters->get_by(adgroup_id => $adgroup_ids);
        for my $row (@{$perf_filters->items}) {
            $row->price_cpc($new_price_cpc) if defined $new_price_cpc;
            $row->price_cpa($new_price_cpa) if defined $new_price_cpa;
        }
        $perf_filters->update(log_price__type => 'update2');

        return '';
    }

    return iget('Не допускается изменение цены клика, если у кампании установлена автобюджетная стратегия')
        if $strategy->{is_autobudget};
    
    my $is_different_places = $strategy->{name} eq 'different_places';
    die 'incorrect price params for strategy different_places'
        if $params->{for_different_places} && !$is_different_places;
    
    $params->{strategy} = $strategy;

    # разбиваем на чанки, поскольку данные показометра съедают очень много памяти
    # чтоб избежать чрезмерного разбиения - бьем по среднему количеству фраз в группах
    my $phrases_qty = get_one_field_sql(PPC(cid => $campaign_id), ["select count(id) from bids", where => {pid => $adgroup_ids}]) // 0;
    my $avg_phrases_in_adgroup_qty = $phrases_qty / scalar(@$adgroup_ids);

    my $adgroups_chunk_size = scalar(@$adgroup_ids);
    if ($avg_phrases_in_adgroup_qty > 0) {
        $adgroups_chunk_size = int($PROCESS_PHRASES_CHUNK_SIZE / $avg_phrases_in_adgroup_qty) || 1;
    }
    
    my ($need_update_phrases);
    my $prices = {};

    if (Campaign::is_cpm_campaign($campaign->{type})) {
        hash_merge $params, CPM_PRICE_CONSTANT_NAMES;
        $params->{single}{price_ctx} = currency_price_rounding($params->{single}{price_ctx}, $campaign->{currency}, %{CPM_PRICE_CONSTANT_NAMES()});
    }

    for my $adgroup_ids_chunk (chunks($adgroup_ids, $adgroups_chunk_size)) {
        # Отдельная обработка для динамических кампаний
        if ($campaign->{type} eq 'dynamic') {
            # Пока поддерживается установка цен только на поиске
            croak 'incorrent params for dynamic campaign' unless $params->{platform} eq 'search' && $params->{single_price};

            my $new_price = currency_price_rounding($params->{single_price}, $campaign->{currency});

            my $dyn_conds_obj = Direct::DynamicConditions->get_by(adgroup_id => $adgroup_ids_chunk);
            $_->price($new_price) for @{$dyn_conds_obj->items};
            $dyn_conds_obj->update(log_price__type => 'update2');

            return '';
        }
      
        if ($params->{update_phrases} || ! $params->{update_retargetings}) {
            my %options = (
                pid              => $adgroup_ids_chunk,
                phid             => $keyword_ids,
                only_auction_phrases => !($params->{for_different_places} && exists $params->{single} || $params->{single_price}),
                skip_pokazometer => $params->{single_price} || ($params->{for_different_places} && exists $params->{single}),
            );
            if ($params->{single_price}) {
                $options{exclude_price} = $options{exclude_price_context} = $params->{single_price}
            } elsif ($params->{for_different_places} && exists $params->{single}) {
                $options{exclude_price} = $params->{single}->{price} if exists $params->{single}->{price};
                $options{exclude_price_context} = $params->{single}->{price_ctx} if exists $params->{single}->{price_ctx};
            }

            my $phrases = PhrasesAuction::get_auction_phrases(%options) || [];

            if (@$phrases) {
                hash_merge $prices, $is_different_places
                    ? PhrasePrice::auto_price_for_different_places($phrases, $params)
                    : PhrasePrice::auto_price_for_other($phrases, $params);

                for my $phr (@$phrases) {
                    if ($phr->{pokazometer_failed}) {
                        return 'pokazometer_failed';
                    }

                    next unless exists $prices->{$phr->{id}};
                    # [] - фраза не торгуется в аукционе
                    $prices->{$phr->{id}}->{guarantee} = $phr->{guarantee} || [];
                    $prices->{$phr->{id}}->{premium} = $phr->{premium} || [];
                }
                $need_update_phrases //= 1;
            }

            # Расчет ставок беcфразного таргетинга
            if (%$prices) {
                my $min_price = get_currency_constant($campaign->{currency}, 'MIN_PRICE');
                my $max_price = get_currency_constant($campaign->{currency}, 'MAX_PRICE');
                if ($options{exclude_price} || $options{exclude_price_context}) {
                    # Если у нас единая ставка на всю кампанию
                    my $need_update = 0;
                    my $bids_relevance_match = Direct::Bids::BidRelevanceMatch->get_by(adgroup_id => $adgroup_ids_chunk);
                    for my $item (@{$bids_relevance_match->items}) {
                        $item->old($item->clone);
                        if ($options{exclude_price} && $min_price <= $options{exclude_price} && $options{exclude_price} <= $max_price) {
                            $item->price($options{exclude_price});
                            $need_update = 1;
                        }
                        if ($options{exclude_price_context} && $min_price <= $options{exclude_price_context} && $options{exclude_price_context} <= $max_price) {
                            $item->price_context($options{exclude_price_context});
                            $need_update = 1;
                        }
                    }
                    if ($need_update) {
                        $bids_relevance_match->update(uid => $client_chief_uid);
                    }
                } else {
                    # Если мастер-цен, то считаем 30ый перцентиль от ставки фраз
                    my %pid2phrases_ids;
                    foreach my $phrase (@$phrases) {
                        push @{ $pid2phrases_ids{$phrase->{pid}} //= [] }, $phrase->{id};
                    }
                    foreach my $pid (keys %pid2phrases_ids) {
                        my @prices = grep { $_ && $min_price <= $_ && $_ <= $max_price } map { $_->{price} } @$prices{@{$pid2phrases_ids{$pid}}};
                        my $context_prices = Direct::Model::BidRelevanceMatch::Helper::extract_ctx_prices([@$prices{@{$pid2phrases_ids{$pid}}}], $min_price, $max_price);

                        my ($price, $price_context);
                        if (@prices){
                            $price = Direct::Model::BidRelevanceMatch::Helper::calc_price(\@prices, $campaign->{currency});
                        }
                        if ($has_extended_relevance_match && @$context_prices){
                            $price_context = Direct::Model::BidRelevanceMatch::Helper::calc_average_price($context_prices, $campaign->{currency}, $campaign->{ContextPriceCoef});
                        }
                        $price_context //= $has_extended_relevance_match ? $price : 0;

                        next unless $price && $price_context;

                        my $bids_relevance_match = Direct::Bids::BidRelevanceMatch->get_by(adgroup_id => $pid);
                        foreach my $rlm (@{$bids_relevance_match->items}){
                            $rlm->old($rlm->clone);
                            $rlm->price($price) if defined $price;
                            $rlm->price_context($price_context) if defined $price_context;
                        }
                        $bids_relevance_match->update(uid => $client_chief_uid);
                    }
                }
            }
        }

        my $retargetings = Retargeting::get_group_retargeting(pid => $adgroup_ids_chunk, type => ['retargeting', 'interest']);
        if (($params->{update_retargetings}
             || (ref $params->{on_ctx} && $params->{on_ctx}->{update_retargetings})
             || $params->{single_price} && ($params->{strategy} && !$params->{strategy}->{is_net_stop}) && $params->{platform} && $params->{platform} =~ /^(search|both)$/
             || $params->{for_different_places} && $params->{single} && $params->{single}->{price_ctx}
            )
            && $retargetings && scalar keys %$retargetings
        )
        {
            my $main_banners = Primitives::get_main_banner_ids_by_pids(@$adgroup_ids_chunk);
            for my $adgroup_id (@$adgroup_ids_chunk) {
                my $group = get_groups({ pid => $adgroup_id },{ pure_groups => 1 })->[0];
                my @new_retargetings;
                my @retargetings = defined $retargetings->{$adgroup_id}
                    ? @{$retargetings->{$adgroup_id}} : ();
                if (scalar @retargetings) {
                    my $banner_id = $main_banners->{$adgroup_id};
                    for my $retargeting (@retargetings) {
                        my $new_ret_row = Retargeting::auto_price_for_retargetings($retargeting, $params, $group->{geo}, $campaign->{currency});
                        $new_ret_row->{cid} = $campaign_id; # для логов цен
                        $new_ret_row->{bid} = $banner_id;
                        $new_ret_row->{pid} = $adgroup_id;
                        push @new_retargetings, $new_ret_row;
                    }
                }
                if (scalar @new_retargetings) {
                    Retargeting::update_group_retargetings_params(\@new_retargetings, [],
                                                log_label => 'ret_update_auto',
                                                currency => $campaign->{currency},
                                                cid => $campaign_id);
                }
            }
        }
    }

    if (keys %$prices) {
        hash_merge $prices, {phrase_ids => $keyword_ids, cid => $campaign_id};
    }
    if ($need_update_phrases) {
        my $update_price_error = keys(%$prices) > 1
                                 ? Common::save_prices($client_chief_uid, $prices, hash_cut($options, qw/dont_clear_auto_price_queue/))
                                 : iget('Отсутствуют фразы, для которых можно применить цены');
        if ($update_price_error) {
            return $options->{ignore_no_active} ? 'no_active_phrases_flag' : iget('Отсутствуют фразы, для которых можно применить цены');
        }

        my $invalid_prices_cnt = 0;
        if ($options->{log}) {
            my $log = $options->{log};
            for my $price_type (qw/price price_context/) {
                if (exists $prices->{$price_type.'_errors'}) {
                    my $errors = $prices->{$price_type.'_errors'};
                    for my $bid_id (keys %$errors) {
                        $log->out("invalid $price_type. id: $bid_id, value: ".$errors->{$bid_id});
                        ++$invalid_prices_cnt;
                    }
                }
            }
        }
        if (ref $options->{out_invalid_prices_cnt} eq 'SCALAR') {
            ${$options->{out_invalid_prices_cnt}} = $invalid_prices_cnt;
        }
    }
    return '';
}

1;
