package CampStat;

=head1 NAME

CampStat - функции, необходимые для заказа оффлайновых отчетов статистики
           и выполнения этих заказов.

=head1 DESCRIPTION

=cut

use strict;
use warnings;
use utf8;

use Campaign::Types;
use RBACDirect;
use RBAC2::DirectChecks;
use RBACElementary;
use JSON;

use Settings;
use Yandex::DateTime;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::Overshard;
use Yandex::Validate;
use Yandex::I18n;
use DBStat;
use Stat::CustomizedArray;
use Stat::Const;
use Stat::Fields;
use Stat::OrderStatDay;
use Primitives;
use PrimitivesIds;
use List::Util qw/min/;
use List::MoreUtils qw/uniq any none/;
use Yandex::HashUtils;
use Yandex::ListUtils;
use Yandex::TimeCommon qw/mysql2unix/;
use Currencies;
use Client;
use Agency;
use Manager;
use Direct::Storage;
use LogTools qw/log_stacktrace/;

use Direct::Validation::Domains qw//;


# Максимальное количество строк в отчете по клиентам агентства, которое мы можем показать в интерфейсе
# Если нам кажется, что их будет больше, мы не строим отчет и предлагаем клиенту выгрузить его в экселе
our $MAX_EXPECTED_LINES_IN_AGENCY_CLIENTS_REPORT = 10_000;

my $json = JSON->new;


=head2 get_camp_stat($rbac, $data, $dbstat, $translocal_params, $client_uid)

    Получает ссылку на хеш переменных шаблона (vars) и на их основании
    собирает все необходимые данные для построения отчета, помещая их в новый
    хеш.
    В этой функции в идеале должны быть все потенциально долгие вызовы,
    необходимые для построения отчета.

    Параметры:
        $rbac   - объект RBAC2::Extended
        $data   - hashref с параметрами, из них используютя (список неполный):
            login_rights    - $rbac_login_rights из DoCmd
            UID             - UID оператора
            OrderID         - OrderID заказа, по которому запрашивается статистика
            date_from       - дата, с которой нужно получить статистику
            date_to         - дата, по которую нужно получить статистику
            stat_type       - тип отчета
            with_auto_added_phrases - учитывать "автоматически добавленные фразы"
            add_used_dynamic_conditions - флажок, что в результат нужно добавить по ключу
                                          used_dynamic_conditions словарь c условиями нацеливания
                                          формат описан в Stat::Tools::get_dynamic_data
            attribution_model - фильтр по модели атрибуции

        $dbstat - экземпляр класса DBStat для заказа $data->{OrderID}
        $translocal_params - хеш с параметрами для определения транслокальности.
            Возможные параметры хеша описаны в GeoTools::get_translocal_georeg
        $client_uid - текущий залогиненный представитель

=cut

sub get_camp_stat {
    my ($rbac, $data, $dbstat, $translocal_params, $client_uid) = @_;
    my $vars = {};
    my $campstat;

    # some init
    my $login_rights = $data->{login_rights};
    # оператор
    my $UID = $data->{UID};
    my $client_chief_uid = $login_rights->{client_chief_uid};
    my $client_id = get_clientid(uid => $client_chief_uid);
    my $operator_client_id = get_clientid(uid => $UID);

    $dbstat = Stat::CustomizedArray->new(OrderID => $data->{OrderID}, translocal_params => $translocal_params) unless $dbstat;

    my $date_from = $data->{date_from};
    my $date_to   = $data->{date_to};

    my $group = $data->{group};

    my %stat_options = (
        operator_ClientID => $operator_client_id,
    );

    if (Client::ClientFeatures::has_stat_4_digits_precision_feature($client_id)) {
        $stat_options{four_digits_precision} = 1;
    }

    if (any {$login_rights->{$_}} qw/super_control support_control superreader_control/) {
        $stat_options{use_page_id} = $data->{use_page_id};
    }
    if ($data->{stat_summary_only}) {
        $stat_options{summary_only} = 1;
    }
    hash_merge \%stat_options, hash_cut $data, qw/with_nds with_discount currency with_avg_position with_reach_stat/;

    if (exists $data->{single_currency} && $data->{single_currency}) {
        hash_merge \%stat_options, {single_currency => 1};
    }

    if (defined $data->{OrderID}) {
        my $camp_currency = get_one_field_sql(PPC(OrderID => $data->{OrderID}),
                ['SELECT IFNULL(c.currency, "YND_FIXED") FROM campaigns c', WHERE => {OrderID => $data->{OrderID}}]);
        $camp_currency ||= 'YND_FIXED'; # currency_defaults
        if ($camp_currency eq 'YND_FIXED') {
            $stat_options{with_nds} = $stat_options{with_discount} = 1;
        }
        $stat_options{currency} //= $camp_currency;

        my $order_ids = _get_linked_order_ids($data->{OrderID});
        my $sesnum_for_week_sum = 0;
        foreach my $sesnum_for_week (map { Stat::OrderStatDay::order_sessions_num_for_week($_) } @$order_ids) {
            $sesnum_for_week_sum += $sesnum_for_week;
        }
        $vars->{sesnum_for_week} = $sesnum_for_week_sum;
    }

    if (defined $data->{consumer}) {
        $stat_options{consumer} = $data->{consumer};
    }

    my @cids;
    if (defined $data->{cid}) {
        @cids = uniq grep {$_ && m/^\d+$/} split /\s*,\s*/, $data->{cid};
        $vars->{rights_on_campaigns} = rbac_get_rights_on_campaigns($UID, [@cids]) if @cids;

        # если статистика по одной кампании, выставляем флаг возможности редактирования
        # не относится к собственно статистике
        # TODO отделить "не-статистический" код в отдельные функции
        if ($data->{cid} =~ /^\d+$/) {
            $vars->{allow_edit_camp} = $vars->{rights_on_campaigns}->{ $data->{cid} }->{EditCamp} ? 1 : 0;
        }
    }
    
    my $ref = ref $data->{client_uids};
    my $client_uids_sql = $ref && $ref eq 'ARRAY' && @{$data->{client_uids}}
        ? 'AND c.uid IN (' . join(',', map {sql_quote($_)} @{$data->{client_uids}}) . ')'
        : '';
        
    my $hostname = $data->{hostname};
    # / some init end

    # эта ветка не должна исполняться
    # функция `CampStat::get_camp_stat` вызывается из
    # * `APIMethods::GetSummaryStat` (отключенный метод) с типом campdate
    # * `DoCmdStat::cmd_showCampStat` - оттуда не может прийти вызов с типом custom, т.к. это приводит к редиректу в МОК
    # * `ppcXLSReports.pl` - оттуда не может прийти вызов с типом custom, т.к. офлайн отчёты создаются в контроллере
    #       `DoCmdStat::cmd_showCampStat`, который не обрабатывает запросы с таким типом
    if ( defined $data->{stat_type} && $data->{stat_type} eq 'custom' ) {
        log_stacktrace("old-stat-unreachable", 'CampStat::get_camp_stat(stat_type=>custom)');
        # разбираемся с группировкой  
        my $phrases_group_by = join('|', @Stat::Const::ANY_PHRASES_GROUP_BY);
        my @group_by = grep {/^(?:banner|page|geo|phraseid|date|position|tag|image|adgroup|device_type|age|gender|detailed_device_type|connection_type|$phrases_group_by)$/} split /\s*,\s*/, $data->{group_by}||'';

        if (get_camp_type(OrderID => $data->{OrderID}) ne 'mobile_content') {
            @group_by = @{xminus \@group_by, [qw/detailed_device_type connection_type/]};
        }
        @group_by = qw/banner date/ if !@group_by;
        $vars->{group_hash} = { map { $_ => 1 } @group_by };

        # разбираемся с фильтрами
        my $filter = {};
        for my $name ( qw/page geo phraseid page_target position image adgroup device_type age gender detailed_device_type connection_type/, @Stat::Const::ANY_PHRASES_GROUP_BY ) {
            $vars -> {"filter_$name"} = $data->{"filter_$name"} || '';
            $filter->{$name} = lc( $vars->{"filter_$name"} );
            $filter->{$name} =~ s/^\s+|\s+$//g;
            if ($name =~ /^(geo|adgroup)$/ && $filter->{$name} =~ /^[\d\s,]+$/) {
                $filter->{$name} = [split /\D+/, $filter->{$name}] ;
            }
        }

        if (get_camp_type(OrderID => $data->{OrderID}) ne 'mobile_content') {
            delete $filter->{$_} for qw/detailed_device_type connection_type/;
        }

        my @bids;
        if ( $data->{filter_banner} ) {
            @bids = grep {/^\d+$/} split /\D+/, $data->{filter_banner};
            if ( @bids ) {
                $filter->{banner} = get_bannerids(bid => \@bids);
                $vars->{filter_banner} = $data->{filter_banner} || '';
            }
            $filter->{banner} ||= [];
        }
        # Если заполнено поле меток в статистике, то превращаем эти метки в массив id-шников
        if ( $data->{filter_tag} ) {
            my @tags = split /\s*,\s*/, $data->{filter_tag};
            $filter->{tag_ids} = Tag::get_tag_ids_by_tags_text(\@tags, \@cids, \@bids) || [] if @tags;
        }

        # Если заполнено поле групп объявлений в статистике, и не является массивом id-шников, то превращаем его в таковой
        if ($filter->{adgroup} && !ref $filter->{adgroup}) {
            my $cid = get_cid(OrderID => $data->{OrderID});
            if ($cid) {
                $filter->{adgroup} = [keys %{Stat::Tools::get_adgroups_info(cid => $cid, group_name => $filter->{adgroup})}];
            } else {
                $filter->{adgroup} = [];
            }
        }
        
        # Определяем сортировку и лимит
        my $limits = { order_by => [ map { { 'field' => $_ } } @group_by ] };
        unless ($data->{without_pager}) {
            # Для конструктора вычисляем количество результатов на странице
            $vars->{onpage} = $data->{onpage} && $data->{onpage} =~ /^\d+$/
                                && $data->{onpage} <= 10000 && $data->{onpage} >= 10
                                    ? $data->{onpage} : 100;
            $vars->{page} = $data->{page} && $data->{page} =~ /^\d+$/ ? $data->{page} : 1;
            
            $limits->{limit} = $vars->{onpage};
            $limits->{offset} = ($vars->{page} - 1) * $vars->{onpage};
        }

        if (  $data->{sort} ) {
            unshift @{$limits->{order_by}}, { 'field' =>  $data->{sort}, 'reverse' => $data->{reverse} };
        }
        $filter->{goal_id} = $data->{goals};

        my $dynamic_data;
        if ($data->{add_used_dynamic_conditions}) {
            $stat_options{dynamic_data_ref} = \$dynamic_data;
        }

        $campstat = $dbstat->set_report_parameters(
            oid => _get_linked_order_ids($data->{OrderID})
            , start_date => $date_from
            , end_date => $date_to
            , group_by => \@group_by
            , date_aggregation_by => $group
            , filter => $filter
            , limits => $limits
            , options => \%stat_options
            , translocal_params => $translocal_params // $dbstat->{translocal_params}
        );
        $campstat = $dbstat->generate_with_metrika;

        if ($data->{add_used_dynamic_conditions}) {
            $vars->{used_dynamic_conditions} = $dynamic_data;
        }

    } elsif ( defined $data->{stat_type} && $data->{stat_type} eq 'geo' ) {
        if ($data->{through_list} || $data->{FORM}{through_list}) {
            $stat_options{flat} = 1;
        }
        if ( !is_media_camp(OrderID => $data->{OrderID}) ) {
            $stat_options{no_spec_and_all_prefix} = 0;
        }
        my $filters = {
            single_goal_id => $data->{goals},
            $data->{attribution_model} ? (attribution_model => $data->{attribution_model}) : (),
        };
        $stat_options{external_countable_fields_override} = Stat::Fields::get_countable_fields_for_override($data->{stat_type});
        if ($data->{show_banners_stat}) {
            $campstat = $dbstat->get_stat_customized( $data->{OrderID}, $date_from, $date_to, ['banner', 'geo' ], '', $filters, %stat_options, stat_type => 'geo');
        } else {
            $campstat = $dbstat->get_stat_customized( $data->{OrderID}, $date_from, $date_to, ['geo' ], '', $filters, %stat_options, stat_type => 'geo');
        }

    } elsif ( defined $data->{stat_type} && $data->{stat_type} eq 'pages' ) {
        $stat_options{no_spec_and_all_prefix} = 0;
        if (is_media_camp(OrderID => $data->{OrderID}) || is_media_camp(cid => $data->{cid})) {
            log_stacktrace("mcb", '$dbstat->get_bad_pages_by_order');
            die "unsupported, DIRECT-98105";
        }
        if (Client::ClientFeatures::has_mol_page_name_support($client_id)
            || Client::ClientFeatures::has_mol_page_name_support($operator_client_id)
        ) {
            if ($stat_options{use_page_id}) {
                # сообщаем DBStat, что группировать придётся не по `page_id`, а по сочетанию `page_id` и `page_name`
                $stat_options{group_by_page_name_detailed} = 1;
            }
        }
        my $filters = {
            page_target => $data->{target_type},
            single_goal_id => $data->{goals},
            $data->{attribution_model} ? (attribution_model => $data->{attribution_model}) : (),
        };
        $stat_options{external_countable_fields_override} = Stat::Fields::get_countable_fields_for_override($data->{stat_type});
        if ( $data->{xls} || $data->{without_pager} ) {
            $campstat = $dbstat->get_stat_customized(
                _get_linked_order_ids($data->{OrderID}), $date_from, $date_to,
                ['page','date'], 'day',
                $filters, %stat_options, stat_type => 'pages'
            );
        } else {
                # для баяна оставляем все как было - выбирается вся статистики, сортировка и пейджинг - на стороне шаблонизатора
                $stat_options{onpage} = is_valid_int($data->{onpage}, 10, 10000) ? $data->{onpage} : 100;
                $stat_options{page} = is_valid_int($data->{page}, 1) ? $data->{page} : 1;
                hash_merge $vars, hash_cut(\%stat_options, qw/onpage page/);
                $stat_options{order_by} = [{field => 'page'}];
                if ($data->{sort}) {
                    unshift @{$stat_options{order_by}}, {field => $data->{sort} eq 'sorting' ? 'page' : $data->{sort}, reverse => $data->{reverse}};
                } else {
                    # сортировка по-умолчанию
                    unshift @{$stat_options{order_by}}, {field => 'clicks', reverse => 1};
                }
                $campstat = $dbstat->get_stat_customized(_get_linked_order_ids($data->{OrderID}), $date_from, $date_to, ['page'], '', $filters, %stat_options, stat_type => 'pages');
        }
        my $client_limits = get_client_limits(get_clientid(OrderID => $data->{OrderID}));
        $vars->{max_limit_dont_show} = $client_limits->{general_blacklist_size_limit};

    } elsif ( defined $data->{stat_type} && $data->{stat_type} eq 'page_dates' ) {
        if ( !is_media_camp(OrderID => $data->{OrderID}) ) {
            $stat_options{no_spec_and_all_prefix} = 0;
        }
        my $filters = {
            page_group => $data->{page_group},
            page_id => $data->{page_id},
            page_target => $data->{target_type},
            goal_id=>$data->{goals},
            $data->{attribution_model} ? (attribution_model => $data->{attribution_model}) : (),
        };
        if (Client::ClientFeatures::has_mol_page_name_support($client_id)
            || Client::ClientFeatures::has_mol_page_name_support($operator_client_id)
        ) {
            # МОЛ может выдать на один `page_id` несколько строк с разными `page_name`.
            # поэтому добавляем фильтр по названию площадки
            $filters->{page_name} = $data->{pageName};

            # статистика по площадкам (`stat_type == page`) больше не запрашивает группировку по `PageGroupID`,
            # а делает группировку по названию площадки (`PageName`), поэтому больше не нужно добавлять фильтр
            # `page_group` при разбивке площадки по дням. Но нужно добавлять фильтр по типам площадки, его выясняем из
            # `page_group`, т.к. фронт это поле может не прислать
            if (!$filters->{page_target} && $data->{page_group}) {
                if ($data->{page_group} =~ /\.3$/) {
                    $filters->{page_target} = 'context';
                } else {
                    $filters->{page_target} = 'search';
                }
            }
            delete $filters->{page_group};
        }
        $campstat = $dbstat->get_stat_customized(_get_linked_order_ids($data->{OrderID}), $date_from, $date_to, ['date'], 'day',
            $filters, %stat_options, stat_type => 'page_dates');

    } elsif ( defined $data->{stat_type} && $data->{stat_type} eq 'campdate' ) {
        # статистика по всем кампаниям клиента
        my $type;
        if ($data->{is_geo}) {
            $type = 'geo';
        } elsif ($data->{is_direct}) {
            $type = get_camp_kind_types('web_edit_base');
        } else {
            $type = get_camp_kind_types('media');
        }
        $vars->{mediaType} = ($data->{is_direct} || $data->{is_geo})
                             ? get_camp_kind_types('web_edit_base')
                             : get_camp_kind_types('media');

        my %conds = (
            'c.OrderID__ne' => 0,
            'c.type' => $type,
            'c.uid' => $data->{uid},
            'sc.master_cid__is_null' => 1, # подлежащие кампании будут учтены позже, пока что отбираем нужных мастеров
        );
        if ($data->{work_currency} && $data->{work_currency} ne 'YND_FIXED'
            && camp_kind_in(type => $type, 'web_edit_base')
        ) {
            if ($data->{currency_archive}) {
                $conds{_OR} = {'c.currency' => 'YND_FIXED', 'c.currency__is_null' => 1};
            } else {
                # мультивалютный клиент просит показать ему кампании НЕ из мультивалютного архива
                $conds{'c.currency__is_not_null'} = 1;
                $conds{'c.currency__ne'} = 'YND_FIXED';
            }
        }

        # фильтр по определенным кампаниям
        if (@cids) {
            $conds{'c.cid'} = \@cids;
        }

        # фильтр по кампаниям для построения графиков (отдельно от @cids чтоб не ломать общую логику )
        if ($data->{filter_campaign}) {
            my @filter_cids = uniq grep { /^\d+$/ } split /\s*,\s*/, $data->{filter_campaign};
            if (@filter_cids) {
                $conds{'c.cid'} = \@filter_cids;
            }
        }

        my $favorite_camps = {};
        if ($client_uid) {
            $favorite_camps = get_hash_sql(PPC(uid => $client_uid), "select cid, 1 from user_campaigns_favorite where uid = ?", $client_uid);
        }
        
        my @orders = 
            map { $_->{is_favorite_cid} = 1 if $favorite_camps->{ $_->{cid} };
                  $_->{can_view_details} = !RBAC2::DirectChecks::rbac_cmd_showCampStat( $rbac, {UID => $UID, cid => $_->{cid}});
                  $_
                }
            @{ get_all_sql( PPC(uid => $conds{'c.uid'}), ["select c.OrderID, c.name, c.cid, c.ManagerUID, c.metatype
                                        , IFNULL(c.currency, 'YND_FIXED') AS currency
                                     from campaigns c
                                     left join subcampaigns sc on c.cid = sc.cid",
                                    WHERE => \%conds
                                 ] ) };
        enrich_data(\@orders, using => 'ManagerUID', sub {
            my $manager_uids = shift;
            return get_hashes_hash_sql(PPC(uid => $manager_uids), 
                ["select m.uid as ManagerUID, m.fio as ManagerFio from users m", where => { uid => SHARD_IDS } ]);
        });

        # отфильтровываем кампании по которым текущий UID может смотреть статистику
        my $allow_show_stat = rbac_check_allow_show_stat_camps($rbac, $UID, [map {$_->{cid}} @orders]);
        @orders = grep {$allow_show_stat->{ $_->{cid} }} @orders;

        $vars->{show_manager_filter} = $login_rights->{manager_control} && scalar(grep {! $_->{ManagerUID} || $_->{ManagerUID} != $UID} @orders) > 0;
        # если есть и важные и не важные кампании, то показываем фильтр "по важным"
        $vars->{show_favorites_filter} = scalar(grep {$_->{is_favorite_cid}} @orders) > 0
                                      && scalar(grep {! $_->{is_favorite_cid}} @orders) > 0;

        if ($data->{show_for_manager} && $login_rights->{manager_control}) {
            @orders = grep {$_->{ManagerUID} && $_->{ManagerUID} == $UID} @orders;
        }
        if ($data->{show_favorites}) {
            @orders = grep {$_->{is_favorite_cid}} @orders;
        }
        
        my $by_target = $data->{target_0} || $data->{target_1};
        my $currency = $data->{currency};   # закладываемся на то, что можно будет выбирать валюту статистики в интерфейсе статистики
        if (!$currency && @orders) {
            my @currencies = map { $_->{currency} } @orders;
            $currency = Currencies::get_dominant_currency(\@currencies);
        }
        if (!$currency) {
            $currency = 'YND_FIXED';  # currency_defaults
            $data->{no_data_found} = 1;
        }

        if ($currency eq 'YND_FIXED' && !$data->{no_nds_and_discount_fix}) {
            # когда конвертируем в фишки, прячем галки про НДС и скидку и показываем данные с НДС
            $data->{had_nds} = $data->{had_discounts} = 0;
            $data->{with_nds} = $data->{with_discount} = 1;
            $stat_options{with_nds} = $stat_options{with_discount} = 1;
        }

        # вне зависимости от выбранной группировки по времени, запрашиваем статистику из CustomizedArray сгруппированную по дням
        # потом $dbstat->get_orders_stat_date_format догруппирует по нужному периоду
        $dbstat->set_report_parameters(
                oid => _get_linked_order_ids([map { $_->{OrderID} } @orders]),
                start_date => $date_from,
                end_date => $date_to,
                date_aggregation_by => 'day',
                group_by => ['campaign', 'date'], # в группировке по кампаниям учитываются подлежащие кампании
                ClientID_for_stat_experiments => $translocal_params->{ClientID},
                stat_type => $data->{stat_type},
                options => {
                        %stat_options,
                        single_currency => 1,
                        totals_separate => 1,
                        currency => $currency,
                        countable_fields_by_targettype => ($by_target // 0),
                        no_spec_and_all_prefix => 1,
                        external_countable_fields_override => Stat::Fields::get_countable_fields_for_override($data->{stat_type}),
                    },
            );
        my $ca_stat = $dbstat->generate_with_metrika({no_need_days_num => $data->{no_need_days_num}});

        my $campstatdata = DBStat::convert_stat_stream_to_orders_stat_date($ca_stat->{data_array}, 'is_customized_array_names');
        $campstat = $dbstat->get_orders_stat_date_format(\@orders, $date_from, $date_to, $group, $campstatdata,
            by_target => $by_target,
            four_digits_precision => $stat_options{four_digits_precision}
        );
        $campstat->{currency} = $campstat->{work_currency} = $currency;

        # разделение статистики на поиск/контекст будет в случае её наличия, если данные отсутствуют то всегда
        unless ($campstat->{shows_1}) {            
            my $show_target;
            while (my ($order_id, $date) = each %$campstatdata) {
                next unless $order_id =~ /^\d$/;
                $show_target = 0,last if scalar keys %$date;
            }
            $campstat->{is_target_separate} = defined $show_target ? $show_target : 1; 
        } else {
            $campstat->{is_target_separate} = 1;
        }

        if ($login_rights->{manager_control} && $UID != $client_chief_uid) {
            $vars->{client_info} = get_one_line_sql(PPC(uid => $client_chief_uid), "select login, FIO from users where uid = ?", $client_chief_uid);
            $vars->{manager_info} = get_one_line_sql(PPC(uid => $UID), "select login, FIO from users where uid = ?", $UID);
        }

    # by clients -----------------------------------------------------------------------------------------------------
    } elsif (defined $data->{stat_type} && $data->{stat_type} eq 'by_clients') {
        my $type = ($data->{is_direct}) ? get_camp_kind_types('web_edit_base') : get_camp_kind_types('media');

        do_sql(PPC(shard => 'all'), "SET SESSION group_concat_max_len = 100000");
        my @orders =
            @{ overshard group => 'login', order => 'login', group_concat => 'OrderID',
               get_all_sql(PPC(shard => 'all'),
                                            ["select GROUP_CONCAT(c.OrderID SEPARATOR ',') as OrderID,
                                              c.ManagerUID, u.login, u.FIO
                                            , u.ClientID
                                            , IFNULL(c.currency, 'YND_FIXED') AS currency
                                             from campaigns c
                                               join users u on u.uid = c.uid
                                             where c.OrderID != 0
                                               $client_uids_sql
                                               and c.shows > 0
                                               and c.ManagerUID > 0
                                               and c.ManagerUID = ?
                                               and lastShowTime >= ?
                                               and ", {'c.type' => $type}, "
                                             group by u.login"], $client_chief_uid, $date_from)};
        enrich_data(\@orders, using => 'ManagerUID', sub {
            my $manager_uids = shift;
            return get_hashes_hash_sql(PPC(uid => $manager_uids), 
                [ "select m.uid as ManagerUID, m.fio as ManagerFio from users m", where => { uid => SHARD_IDS } ]);
        });

        my $currency = $data->{currency};   # закладываемся на то, что можно будет выбирать валюту статистики в интерфейсе статистики
        if (!$currency) {
            if (@orders) {
                my $clientids;
                # если выбран ограниченный набор клиентов, отчёт показываем в их превалирующей валюте
                if ($data->{client_uids} && ref($data->{client_uids}) eq 'ARRAY' && @{$data->{client_uids}}) {
                    $clientids = get_clientids(uid => $data->{client_uids});
                } else {
                    # если клиенты не выбраны, отчет показываем в валюте, в которой работает большинство неархивных прямых клиентов менеджера
                    # неархивность клиента определяем также как в DoCmd::cmd_showManagerMyClients
                    # по наличию сервисируемой кампании в текущем сервисе (директ/баян) и неархивности пользователя
                    # используем только активных клиентов (см. логику определения is_active в cmd_showManagerMyClients по OrderID и sum)
                    $clientids = get_one_column_sql(PPC(shard => 'all'), [q/
                        SELECT DISTINCT u.ClientID
                        FROM campaigns c
                        INNER JOIN users u ON c.uid = u.uid
                     /, WHERE => {
                            'c.ManagerUID' => $client_chief_uid,
                            'c.type' => $type,
                            'c.statusEmpty' => 'No',
                            _OR => {'c.OrderID__gt' => 0, 'c.sum__gt' => $Currencies::EPSILON},
                            'u.statusArch' => 'No',
                    }]) || [];
                }
                if (@$clientids) {
                    $currency = _get_clients_dominant_currency($clientids);
                } else {
                    $currency = 'YND_FIXED';  # currency_defaults
                    $data->{no_data_found} = 1;
                }
            } else {
                $currency = 'YND_FIXED';  # currency_defaults
                $data->{no_data_found} = 1;
            }
        }

        if ($currency eq 'YND_FIXED') {
            # когда конвертируем в фишки, прячем галки про НДС и скидку и показываем данные с НДС
            $data->{had_nds} = $data->{had_discounts} = 0;
            $data->{with_nds} = $data->{with_discount} = 1;
            $stat_options{with_nds} = $stat_options{with_discount} = 1;
        }

        if (!$data->{is_direct}) {
            log_stacktrace("mcb", '$dbstat->get_orders_stat_date');
            die "unsupported";
        }
        $stat_options{apply_3_year_limit} = 0;
        $stat_options{external_countable_fields_override} = Stat::Fields::get_countable_fields_for_override($data->{stat_type});

        my $by_target = $data->{target_0} || $data->{target_1};
        $campstat = $dbstat->get_orders_stat_date(\@orders, $date_from, $date_to, $group, by_target => $by_target, %stat_options, single_currency => 1, currency => $currency);
        $campstat->{currency} = $campstat->{work_currency} = $currency;

        my @found_currencies = uniq map { $_->{currency} } @orders;
        $campstat->{found_currencies} = _get_found_currencies($currency, \@found_currencies);

    # by agencies ----------------------------------------------------------------------------------------------------
    } elsif (defined $data->{stat_type} && $data->{stat_type} eq 'by_agencies') {

        my $all_my_agencies_uids = rbac_get_my_agencies($rbac, $client_chief_uid);

        my $currency = $data->{currency};   # закладываемся на то, что можно будет выбирать валюту статистики в интерфейсе статистики
        if (@$all_my_agencies_uids) {

            my $type = ($data->{is_direct}) ? get_camp_kind_types('web_edit_base') : get_camp_kind_types('media');
            do_sql(PPC(shard => 'all'), "SET SESSION group_concat_max_len = 100000");
            my @orders =
                @{ overshard group => 'AgencyClientID', group_concat => 'OrderID',
                   get_all_sql(PPC(shard => 'all'), ["select GROUP_CONCAT(c.OrderID SEPARATOR ',') AS OrderID
                                                , c.AgencyID AS AgencyClientID
                                                , GROUP_CONCAT(DISTINCT IFNULL(c.currency, 'YND_FIXED')) AS currencies
                                           from campaigns c
                                           where c.OrderID != 0
                                             and ", {'c.AgencyUID' => $all_my_agencies_uids}, "
                                             and c.shows > 0
                                             and lastShowTime >= ?
                                             and ", {'c.type' => $type}, "
                                           group by c.AgencyID"], $date_from ) };
            enrich_data(\@orders, using => 'AgencyClientID', sub {
                my $agency_clientids = shift;
                return get_hashes_hash_sql(PPC(ClientID => $agency_clientids), [q(
                    SELECT cl.ClientID, cl.name AS AgencyName
                    FROM clients cl
                 ), WHERE => { 'cl.ClientID' => SHARD_IDS },
                ]);
            });

            my %found_currencies;   # собираем список встреченных валют, чтобы показать примечание о курсах у.е. к реальной валюте
            if (@orders) {
                my @agency_client_ids = map {$_->{AgencyClientID}} @orders;
                my $ag_chiefs = rbac_get_chief_reps_of_agencies(\@agency_client_ids);
                my @agency_uids = values %$ag_chiefs;
                my $ag_chiefs_info = get_hashes_hash_sql(PPC(uid => \@agency_uids), 
                    ["select ClientID, FIO, login from users", where => {uid => SHARD_IDS}]);
                for my $row (@orders) {
                    my $AgencyClientID = $row->{AgencyClientID};
                    $row->{AgencyLogin} = $ag_chiefs_info->{$AgencyClientID}->{login};
                    $row->{AgencyName} = $ag_chiefs_info->{$AgencyClientID}->{FIO} unless $row->{AgencyName};
                    $found_currencies{$_} = 1 for split ',', $row->{currencies};
                }
                # Отсортируем по логину агенства
                @orders = xsort { lc($_->{AgencyLogin}) } @orders;
            } else {
                $currency = 'YND_FIXED';  # currency_defaults
                $data->{no_data_found} = 1;
            }

            if (!$currency) {
                # отчет показываем в валюте, в которой работает большинство неархивных агентств менеджера
                # неархивное агентство определяем также как в DoCmd::cmd_showManagerMyClients:
                # должна быть кампания в текущей системе (директ/баян) + представитель агентства должен быть неархивен
                my $agency_clientids_with_campaigns = get_one_column_sql(PPC(shard => 'all'), [q(
                        SELECT DISTINCT c.AgencyID
                        FROM campaigns c
                     ), WHERE => {
                            'c.AgencyUID' => $all_my_agencies_uids,
                            'c.type' => $type,
                            'c.statusEmpty' => 'No',
                        },
                ]) || [];
                if (@$agency_clientids_with_campaigns) {
                    my @agency_currencies = values %{ mass_get_agency_currency($rbac, $agency_clientids_with_campaigns) };
                    $currency = Currencies::get_dominant_currency(\@agency_currencies);
                }
                if (!$currency) {
                    $currency = 'YND_FIXED';  # currency_defaults
                }
            }

            if ($currency eq 'YND_FIXED') {
                # когда конвертируем в фишки, прячем галки про НДС и скидку и показываем данные с НДС
                $data->{had_nds} = $data->{had_discounts} = 0;
                $data->{with_nds} = $data->{with_discount} = 1;
                $stat_options{with_nds} = $stat_options{with_discount} = 1;
            }

            if (!$data->{is_direct}) {
                log_stacktrace("mcb", '$dbstat->get_orders_stat_date');
                die "unsupported";
            }
            $stat_options{apply_3_year_limit} = 0;
            my $by_target = $data->{target_0} || $data->{target_1};
            $stat_options{external_countable_fields_override} = Stat::Fields::get_countable_fields_for_override($data->{stat_type});
            $campstat = $dbstat->get_orders_stat_date(\@orders, $date_from, $date_to, $group, by_target => $by_target, %stat_options, single_currency => 1, currency => $currency);

            $campstat->{found_currencies} = _get_found_currencies($currency, [keys %found_currencies]);
        } else {
            $currency = 'YND_FIXED'; # currency_defaults
            $data->{no_data_found} = 1;
        }
        $campstat->{currency} = $campstat->{work_currency} = $currency;

    # by managers ----------------------------------------------------------------------------------------------------
    } elsif (defined $data->{stat_type} && $data->{stat_type} eq 'by_managers') {

        my $all_my_managers_uids = rbac_get_my_managers_with_idm($rbac, $client_chief_uid);
        my $all_my_agencies_uids = rbac_get_my_agencies($rbac, $client_chief_uid);

        my %agency2manager;
        if (@$all_my_agencies_uids) {
            for my $ag_uid (@$all_my_agencies_uids) {
                # TODO performance: хорошо бы сделать массовую версию rbac_get_manager_of_agency_clientid
                my $manager_of_agency = rbac_get_manager_of_agency($rbac, $ag_uid, $data->{is_direct} ? 'text' : 'mcb');
                $agency2manager{$ag_uid} = $manager_of_agency if $manager_of_agency;
            }
        }

        my $currency = $data->{currency};   # закладываемся на то, что можно будет выбирать валюту статистики в интерфейсе статистики
        if (@$all_my_managers_uids || @$all_my_agencies_uids) {
            my $type = ($data->{is_direct}) ? get_camp_kind_types('web_edit_base') : get_camp_kind_types('media');

            # выбираем двумя запросами, чтобы использовать индексы по ManagerUID и AgencyUID
            do_sql(PPC(shard => 'all'), "SET SESSION group_concat_max_len = 100000");
            my @orders =
                @{ get_all_sql(PPC(shard => 'all'), ["SELECT OrderID
                                                  , c.ManagerUID
                                                  , c.AgencyUID
                                                  , c.uid
                                                  , IFNULL(c.currency, 'YND_FIXED') AS currency
                                                  , u.ClientID
                                           from campaigns c
                                             left join users u ON c.uid = u.uid
                                           where c.OrderID != 0
                                             and c.shows > 0
                                             and ", {'c.ManagerUID' => $all_my_managers_uids}, "
                                             and lastShowTime >= ?
                                             and ", {'c.type' => $type}, "
                                         UNION ALL
                                           SELECT c.OrderID
                                                , c.ManagerUID
                                                , c.AgencyUID
                                                , c.uid
                                                , IFNULL(c.currency, 'YND_FIXED') AS currency
                                                , u.ClientID
                                           from campaigns c
                                             left join users u ON c.uid = u.uid
                                           where c.OrderID != 0
                                             and c.shows > 0
                                             and ", {'c.AgencyUID' => $all_my_agencies_uids}, "
                                             and lastShowTime >= ?
                                             and ", {'c.type' => $type}
                                          ], $date_from, $date_from) || [] };
            enrich_data(\@orders, using => 'ManagerUID', sub {
                my $manager_uids = shift;
                return get_hashes_hash_sql(PPC(uid => $manager_uids), 
                    [ "select m.uid as ManagerUID, m.fio as ManagerFio, m.login as ManagerLogin from users m", where => { uid => SHARD_IDS } ]);
            });

            # объединяем агентсткие и сервисируемые заказы одного менеджера
            my (%grouped_orders, $order2uid, %client_ids);
            my %found_currencies;   # собираем список встреченных валют, чтобы показать примечание о курсах у.е. к реальной валюте
            for my $row (@orders) {
                $order2uid->{$row->{OrderID}} = $row->{uid};
                $client_ids{$row->{ClientID}} = 1;
                $found_currencies{$row->{currency}} = 1;

                my $ManagerUID = $row->{AgencyUID} ? $agency2manager{$row->{AgencyUID}} : $row->{ManagerUID};
                #next unless $ManagerUID; # без этой строки на бете иногда вылезают строчки «менеджер: ()» - что с этим делать? есть ли они в продакшн?..

                if ($grouped_orders{$ManagerUID}) {
                    $grouped_orders{$ManagerUID}->{OrderID} .= "," . $row->{OrderID};
                } else {
                    $grouped_orders{$ManagerUID} = $row;
                }
                $grouped_orders{$ManagerUID}->{ManagerLogin} ||= $row->{ManagerLogin};
                $grouped_orders{$ManagerUID}->{ManagerFio}   ||= $row->{ManagerFio};

                $grouped_orders{$ManagerUID}->{exists_agency_camp} = 1 if $row->{AgencyUID};
                $grouped_orders{$ManagerUID}->{exists_manager_camp} = 1 if $row->{ManagerUID};
            }

            if (!$currency) {
                if (@orders) {
                    my @manager_uids = grep {$_ != $client_chief_uid} @$all_my_managers_uids;
                    my $manager2currency = Manager::mass_get_manager_currencies($rbac, \@manager_uids, $type);
                    my @manager_currencies = values %$manager2currency;
                    $currency = Currencies::get_dominant_currency(\@manager_currencies);
                } else {
                    $currency = 'YND_FIXED';  # currency_defaults
                    $data->{no_data_found} = 1;
                }
            }

            if ($currency eq 'YND_FIXED') {
                # когда конвертируем в фишки, прячем галки про НДС и скидку и показываем данные с НДС
                $data->{had_nds} = $data->{had_discounts} = 0;
                $data->{with_nds} = $data->{with_discount} = 1;
                $stat_options{with_nds} = $stat_options{with_discount} = 1;
            }

            @orders = values %grouped_orders;
            if (!$data->{is_direct}) {
                log_stacktrace("mcb", '$dbstat->get_orders_stat_date');
                die "unsupported";
            }
            $stat_options{apply_3_year_limit} = 0;
            my $by_target = $data->{target_0} || $data->{target_1};
            $stat_options{external_countable_fields_override} = Stat::Fields::get_countable_fields_for_override($data->{stat_type});
            $campstat = $dbstat->get_orders_stat_date(\@orders, $date_from, $date_to, $group, by_target => $by_target, %stat_options, single_currency => 1, currency => $currency);

            for my $o (@{$campstat->{orders}}) {
                $o->{ManagerClientsCount} = count_manager_clients($o, $campstat->{ordinfo}, $order2uid);
                unless ($o->{ManagerLogin} && $o->{ManagerFio}) {
                    my $ManagerUID = $o->{ManagerUID} || $agency2manager{$o->{AgencyUID}};
                     ( $o->{ManagerLogin}, $o->{ManagerFio} ) = get_one_line_array_sql(PPC(uid => $ManagerUID),
                         "select login, FIO from users where uid = ?", $ManagerUID);
                     unless ($o->{ManagerLogin} && $o->{ManagerFio}) {
                         #die "can not fetch login&fio for (ManagerUID = $o->{ManagerUID}; AgencyUID = $o->{AgencyUID}; a2m uid = $ManagerUID)\n";
                         # can not fetch login&fio for (ManagerUID = ; AgencyUID = 13749072; a2m uid = )
                     }
                }
            }

            $campstat->{found_currencies} = _get_found_currencies($currency, [keys %found_currencies]);
        } else {
            $currency = 'YND_FIXED';  # currency_defaults
            $data->{no_data_found} = 1;
        }
        $campstat->{currency} = $campstat->{work_currency} = $currency;

    # by agency clients (for manager) --------------------------------------------------------------------------------
    } elsif (defined $data->{stat_type} && $data->{stat_type} eq 'by_agency_clients') {

        my $cids = rbac_get_agcamps($rbac,$client_chief_uid);

        my $camp_kind = $data->{is_direct} ? 'web_edit_base' : 'media';
        do_sql(PPC(cid => $cids), "SET SESSION group_concat_max_len = 100000");
        my @orders =
            @{ overshard group => 'login', order => 'login', group_concat => 'OrderID',
            get_all_sql(PPC(cid => $cids), ["select GROUP_CONCAT(DISTINCT c.OrderID SEPARATOR ',') as OrderID,
                                                    u.login, u.FIO, u.uid
                                             , u.ClientID
                                             , GROUP_CONCAT(DISTINCT IFNULL(c.currency, 'YND_FIXED')) AS currencies
                                             from campaigns c
                                               join users u on u.uid = c.uid
                                             where c.OrderID != 0
                                               $client_uids_sql
                                               and c.AgencyUID > 0
                                               and c.shows > 0
                                               and lastShowTime >= ?
                                               and ", {'c.type' => get_camp_kind_types($camp_kind)}, "
                                               and ", { 'c.cid' => SHARD_IDS },
                                               "group by u.login"], $date_from)};
        # fetch agency limited info about clients
        my %found_currencies;   # собираем список встреченных валют, чтобы показать примечание о курсах у.е. к реальной валюте
        my $currency = $data->{currency};   # закладываемся на то, что можно будет выбирать валюту статистики в интерфейсе статистики
        if (@orders) {
            my $limited_agencies_of_client = rbac_get_limited_agencies_reps_of_clients_list($client_chief_uid, [map { $_->{uid} } @orders]);
            my $limited_agencies_of_client_info = %$limited_agencies_of_client ? get_users_list_info([uniq map { @$_ } values %$limited_agencies_of_client]) : {};
            for my $row (@orders) {
                my $limited_agency_uid = $limited_agencies_of_client->{ $row->{uid} };
                $row->{limited_agency} = $limited_agencies_of_client_info->{$limited_agency_uid->[0]} if $limited_agency_uid;
                $found_currencies{$_} = 1 for split /,/, $row->{currencies};
            }
        } else {
            $currency = 'YND_FIXED'; # currency_defaults
            $data->{no_data_found} =1;
        }

        if (!$currency) {
            # отчет показываем в валюте, в которой работает большинство неархивных клиентов
            # неархивных клиентов определяем как в DoCmd::cmd_showClients
            # если выбрана только часть клиентов, то превалирующую валюту определяем только по ним
            my ($subclient_uids, $need_archive_filtering);
            if ($data->{client_uids} && ref($data->{client_uids}) eq 'ARRAY' && @{$data->{client_uids}}) {
                $subclient_uids = $data->{client_uids};
                $need_archive_filtering = 0;
            } else {
                $subclient_uids = rbac_get_subclients_all_reps_uids($rbac, $client_chief_uid);
                $need_archive_filtering = 1;
            }
            if ($subclient_uids && @$subclient_uids) {
                my $subclient_clientids = get_one_column_sql(PPC(uid => $subclient_uids),
                    ["SELECT DISTINCT u.ClientID
                        FROM users u
                             JOIN campaigns c ON c.uid = u.uid
                        ", WHERE => {'u.uid' => SHARD_IDS, 'c.type' => get_camp_kind_types($camp_kind)}]);
                if (@$subclient_clientids) {
                    if ($need_archive_filtering) {
                        my $agency_client_id = get_clientid(uid => $client_chief_uid);
                        my $relations_data = get_agency_client_relations($agency_client_id, $subclient_clientids);
                        if ($relations_data && %$relations_data) {
                            $subclient_clientids = [ grep { !$relations_data->{$_}->{client_archived} } keys %$relations_data ];
                        }
                    }
                    if (@$subclient_clientids) {
                        $currency = _get_clients_dominant_currency($subclient_clientids);
                    }
                }
            }
        }
        if (!$currency) {
            $currency = 'YND_FIXED';  # currency_defaults
            $data->{no_data_found} = 1;
        }

        if ($currency eq 'YND_FIXED') {
            # когда конвертируем в фишки, прячем галки про НДС и скидку и показываем данные с НДС
            $data->{had_nds} = $data->{had_discounts} = 0;
            $data->{with_nds} = $data->{with_discount} = 1;
            $stat_options{with_nds} = $stat_options{with_discount} = 1;
        }

        if (!$data->{is_direct}) {
            log_stacktrace("mcb", '$dbstat->get_orders_stat_date');
            die "unsupported";
        }
        $stat_options{apply_3_year_limit} = 0;
        my $by_target = $data->{target_0} || $data->{target_1};

        # Если это запрос статистики из интерфейса - скипаем получение, если в результате будет слишком много строк.
        if ($data->{from_interface} && !$data->{xls}) {
            my $lines_cnt = 0;
            if ( @{$data->{client_uids} // []} && defined $date_from && defined $date_to )
            {
                # считаем верхнюю теоретическую границу количества строк в отчете
                # умножаем количество дней в заданнном промежутке времени на количество клиентов
                # +1 к количеству клиентов - для учета строк общей статистики по всем клиентам
                $lines_cnt = (mysql2unix($date_to) - mysql2unix($date_from)) / (24 * 3600) * (scalar @{$data->{client_uids}} + 1);
            }
            if ($lines_cnt > $MAX_EXPECTED_LINES_IN_AGENCY_CLIENTS_REPORT) {
                $data->{report_by_agency_clients_too_big} = 1;
            }
        }

        $stat_options{external_countable_fields_override} = Stat::Fields::get_countable_fields_for_override($data->{stat_type});
        unless ($data->{report_by_agency_clients_too_big}) {
            $campstat = $dbstat->get_orders_stat_date(\@orders, $date_from, $date_to, $group, by_target => $by_target, %stat_options, single_currency => 1, currency => $currency, no_spec_and_all_prefix => 1);
        }
        $campstat->{found_currencies} = _get_found_currencies($currency, [keys %found_currencies]);
        $campstat->{currency} = $campstat->{work_currency} = $currency;

    } elsif (defined $data->{detail}) {
        # вкладка Статистика по дням
        die "unsupported, DIRECT-97949";

    } elsif (defined $data->{phrasedate}) {
        # Фразы по дням
        my $limits = {};
        if (  $data->{sort} ) {
            unshift @{$limits->{order_by}}, { 'field' =>  $data->{sort}, 'reverse' => $data->{reverse} };
        }
        unless ( is_media_camp ( cid => $data->{cid} ) || $data->{without_pager} || $data->{xls}) {
            $vars->{banners_on_page} = is_valid_int($data->{banners_on_page}, 1, 100)
                                                  ? $data->{banners_on_page} : 30;
            $vars->{page} = is_valid_int($data->{page}, 1) ? $data->{page} : 1;
            hash_merge \%stat_options, hash_cut($vars, qw/banners_on_page page/) unless $data->{allpages};
        }
        my $dynamic_data;
        if ($data->{add_used_dynamic_conditions}) {
            $stat_options{dynamic_data_ref} = \$dynamic_data;
        }
        if ( is_media_camp(OrderID => $data->{OrderID}) ) {
            log_stacktrace("mcb", '$dbstat->get_stat_phrase_date');
            die "unsupported";
        }

        $stat_options{external_countable_fields_override} = Stat::Fields::get_countable_fields_for_override('phrasedate');

        $campstat = $dbstat->get_stat_phrase_date( _get_linked_order_ids($data->{OrderID}), $date_from, $date_to, $group,
                                                   {single_goal_id=>$data->{goals},
                                                    $data->{tag} ? (tag_ids => [$data->{tag}]) : (),
                                                    $data->{attribution_model} ? (attribution_model => $data->{attribution_model}) : ()},
                                                   $limits, %stat_options );

        my $linked_cids = _get_linked_cids($data->{cid});
        $vars->{real_phrases} = DBStat::associate_phrases_from_stat($linked_cids, $campstat->{banners});
        $vars->{real_retargetings} = DBStat::associate_retargetings_from_stat($linked_cids, $campstat->{banners});
        $vars->{real_relevance_match} = DBStat::associate_relevance_match_from_stat($linked_cids, $campstat->{banners});
        if ($data->{add_used_dynamic_conditions}) {
            $vars->{used_dynamic_conditions} = $dynamic_data;
        }

    } elsif (defined $data->{plot}) {

        my $filter = {};
        if (camp_kind_in(cid => $data->{cid}, 'base')) {
            if ( $data->{filter_banner} ) {
                my @bids = grep {/^\d+$/} split /\D+/, $data->{filter_banner};
                if ( @bids ) {
                    $filter->{banner} = get_one_column_sql( PPC(cid => $data->{cid}), 
                        ["SELECT BannerID FROM banners", WHERE => { bid => \@bids } ]) || [];
                    $vars->{filter_banner} = $data->{filter_banner} || '';
                }
            }
        } elsif (is_media_camp( cid => $data->{cid} ) ) {
            if ( $data->{filter_banner} ) {
                my @bids = grep {/^\d+$/} split /\D+/, $data->{filter_banner};
                $filter->{banner} = get_one_column_sql( PPC(cid => $data->{cid}), 
                    ["SELECT BannerID FROM media_banners", WHERE => { mbid => \@bids }]) || [];
                $vars->{filter_banner} = $data->{filter_banner} || '';
            }
        }

        $campstat = $dbstat->get_stat_customized( _get_linked_order_ids($data->{OrderID}), $date_from, $date_to, ['date'], $group, {goal_id=>$data->{goals}, banner => $filter->{banner}}, %stat_options );

    } elsif ($data->{OrderID}) {
        # вкладка Общая статистика
        my $dynamic_data;
        if ( is_media_camp(OrderID => $data->{OrderID}) ) {
            log_stacktrace("mcb", '$dbstat->get_common_stat');
            die "unsupported";
        }
        my $linked_orders = get_all_sql(PPC(OrderID => $data->{OrderID}), ["SELECT c.OrderID, c.cid
                                                                            FROM campaigns c_m
                                                                              JOIN subcampaigns sc ON sc.master_cid = c_m.cid
                                                                              JOIN campaigns c ON sc.cid = c.cid",
                                                                            WHERE => {
                                                                              'c_m.OrderID' => $data->{OrderID},
                                                                              'c.OrderID__gt' => 0
                                                                            }]);
        push @$linked_orders, {
            OrderID => $data->{OrderID},
            cid => $data->{cid},
        };
        my @linked_order_ids = map { $_->{OrderID} } @$linked_orders;
        my @linked_cids = map { $_->{cid} } @$linked_orders;
        $stat_options{with_avg_position} = 1;
        $stat_options{attribution_model} = $data->{attribution_model};
        $stat_options{external_countable_fields_override} = Stat::Fields::get_countable_fields_for_override('common');
        if ( $stat_options{'summary_only'} ) {
            my $data = $dbstat->get_orders_stat_date($linked_orders, $date_from, $date_to, undef, %stat_options, use_stat_stream => 1);
            $campstat->{"t$_"} = $data->{$_} for qw/shows clicks ctr sum bonus av_sum adepth aconv agoalcost asesnum agoalnum agoalroi agoalincome/;
            hash_copy $campstat, $data, qw(stat_stream_ts);
        } else {
            unless ( $data->{without_pager} || $data->{xls}) {
                $vars->{banners_on_page} = is_valid_int($data->{banners_on_page}, 1, 100)
                                                      ? $data->{banners_on_page} : 30;
                $vars->{page} = is_valid_int($data->{page}, 1) ? $data->{page} : 1;
                hash_merge \%stat_options, hash_cut($vars, qw/banners_on_page page/) unless $data->{allpages};
            }
            $stat_options{tag_ids} = [$data->{tag}] if $data->{tag};
            if ($data->{add_used_dynamic_conditions}) {
                $stat_options{dynamic_data_ref} = \$dynamic_data;
            }
            if (!$data->{no_phrase_data}) {
                $stat_options{with_phrases} = 1;
            }
            $campstat = $dbstat->get_common_stat(\@linked_order_ids, $date_from, $date_to, %stat_options);
        }

        if (!$data->{no_phrase_data}) {
            $vars->{real_phrases} = DBStat::associate_phrases_from_stat(\@linked_cids, $campstat->{banners});
            $vars->{real_retargetings} = DBStat::associate_retargetings_from_stat(\@linked_cids, $campstat->{banners});
            if ($data->{add_used_dynamic_conditions}) {
                $vars->{used_dynamic_conditions} = $dynamic_data;
            }
        }

        $campstat->{OrderID} = $data->{OrderID};
        unless ($data->{without_pager} || $data->{xls}) {
            hash_merge $campstat, Stat::OrderStatDay::get_fraud_clicks_total(\@linked_order_ids, $date_from, $date_to);
        }

        # В этой вкладке пока нельзя посмотреть статистику по выбранной модели атрибуции, поэтому показываем
        # пользователю, модель атрибуции по которой фактически запрашивалась статистика
        if ($data->{campaign_type}) {
            $vars->{attribution_model} = $data->{attribution_model};
        }
    }

    hash_merge $vars, $campstat;

    return $vars;
}

=head2 _get_linked_order_ids($order_ids)

    Возвращает список OrderID, состоящий из переданных + OrderID кампаний, подчиненных кампаниям с переданными OrderID

=cut

sub _get_linked_order_ids {
    my ($order_ids) = @_;

    my @linked_order_ids = ref($order_ids) eq 'ARRAY' ? @{ $order_ids } : ($order_ids);
    push @linked_order_ids, @{ get_one_column_sql(PPC(OrderID => $order_ids), ["SELECT c.OrderID
                                                                                FROM campaigns c_m
                                                                                  JOIN subcampaigns sc ON c_m.cid = sc.master_cid
                                                                                  JOIN campaigns c ON sc.cid = c.cid",
                                                                                WHERE => {
                                                                                  'c_m.OrderID' => SHARD_IDS,
                                                                                  'c.OrderID__gt' => 0,
                                                                                }]) };
    return \@linked_order_ids;
}


=head2 _get_linked_cids($cids)

    Возвращает список id кампаний, состоящий из переданных + id кампаний, подчиненных кампаниям с переданными id

=cut

sub _get_linked_cids {
    my ($cids) = @_;

    my @linked_cids = ref($cids) eq 'ARRAY' ? @{ $cids } : ($cids);
    push @linked_cids, @{ get_one_column_sql(PPC(cid => $cids), ["SELECT sc.cid
                                                                  FROM subcampaigns sc",
                                                                  WHERE => { 'sc.master_cid' => SHARD_IDS }]) };
    return \@linked_cids;
}

=head2 _get_clients_dominant_currency



=cut

sub _get_clients_dominant_currency {
    my ($client_ids) = @_;

    return undef unless $client_ids && @$client_ids;

    my $client_currencies = mass_get_client_currencies($client_ids);
    my @currencies = map {$_->{work_currency}} values %$client_currencies;
    return Currencies::get_dominant_currency(\@currencies);
}

=head2 _get_found_currencies



=cut

sub _get_found_currencies {
    my ($currency, $found_currencies) = @_;

    return [grep {$_ ne $currency} @$found_currencies];
}

use constant {
    STATUS_FAILED => 'failed',
    STATUS_NEW => 'new',
    STATUS_PROCESSING => 'processing',
    STATUS_READY => 'ready',
    STATUS_OPENED => 'opened'
};

{
    my $allowed_keys = {
        av               => 1,
        av_day           => 1,
        autobudget_sum   => 1,
        campaign => {
            currency     => 1,
            cname        => 1,
        },
        cid              => 1,
        show_banners_stat => 1,
        clicks           => 1,
        client_uids      => 1,
        cname            => 1,
        ctr              => 1,
        d1               => 1,
        d2               => 1,
        date_from        => 1,
        date_to          => 1,
        detail           => 1,
        FIO              => 1,
        geo              => 1,
        goals            => 1,
        group            => 1,
        group_by         => 1,
        hostname         => 1,
        id               => 1,
        interface        => 1,
        is_direct        => 1,
        lastShowTime     => 1,
        login_rights     => {
            client_chief_uid    => 1,
            role                => 1,
            manager_control     => 1,
            super_control       => 1,
            support_control     => 1,
            superreader_control => 1,
            placer_control      => 1,
        },
        m1               => 1,
        m2               => 1,
        name             => 1,
        onpage           => 1,
        OrderID          => 1,
        without_pager    => 1,
        page             => 1,
        page_id          => 1,
        page_group       => 1,
        pageName         => 1,
        phrasedate       => 1,
        plot             => 1,
        plot_group       => 1,
        pseudo_currency  => {
            rate => 1,
            name => 1,
            id   => 1,
        },
        rate             => 1,
        reverse          => 1,
        role             => 1,
        show_favorites   => 1,
        show_for_manager => 1,
        shows            => 1,
        sort             => 1,
        start_time       => 1,
        stat_type        => 1,
        status           => 1,
        sum              => 1,
        sum_spent        => 1,
        super_control    => 1,
        xls_target_all   => 1,
        xls_target_0     => 1,
        xls_target_1     => 1,                
        target_all       => 1,
        target_0         => 1,
        target_1         => 1,        
        target_type      => 1,
        total            => 1,
        tshows_1         => 1,
        type             => 1,
        with_auto_added_phrases => 1,
        with_nds         => 1,
        with_discount    => 1,
        uid              => 1,
        UID              => 1,
        use_page_id      => 1,
        y1               => 1,
        y2               => 1,
        (map {("filter_$_" => 1)} qw/page geo phraseid page_target position image banner tag/, @Stat::Const::ANY_PHRASES_GROUP_BY)
    };

=head2 _clean_hash(hashref, rules)

    Удаляет из ссылки на хеш все ключи, которые не описаны в "правилах".
    Правила, собственно, тоже являются ссылкой на хеш и повторяют его
    структуру.

=cut
    sub _clean_hash {
        my ($h, $r) = @_;

        foreach my $k (keys %$h) {
            # если ключ не описан в правиле - clean && next
            unless ($r->{$k}) {
                delete $h->{$k};
                next;
            }

            # $r->{$k} или ref, или undef (для разрешения ссылок и скаляров соотв.)
            my ($data_ref, $r_ref) = (ref $h->{$k}, ref $r->{$k});

            # у нас не скаляр, а правило предполагает скаляр. Clean && next
            if ($data_ref && !$r_ref) {
                delete $h->{$k};
                next;
            }
            # скаляры пропускаем (даже если правило предполагает ссылку)
            next unless ($data_ref);

            if ($data_ref eq 'HASH' && $r_ref eq 'HASH') {
                _clean_hash($h->{$k}, $r->{$k});
            } elsif ($data_ref eq 'ARRAY' && $r_ref eq 'ARRAY') {
                # если согласно правилам внутри массива не ссылки на хеш, то
                # просто все пропускаем, мы не умеем такое.
                # Сейчас (18.12.10) нет таких хитрых правил.
                next unless (ref $r->{$k}->[0] eq 'HASH');

                foreach (@{$h->{$k}}) {
                    # такого быть не должно, но проверим, чтобы не спятисотить
                    next unless (ref($_) eq 'HASH');

                    _clean_hash($_, $r->{$k}->[0]);
                }
            } else {
                warn "Rules broken: data ref is $data_ref, rule ref is $r_ref";
                delete $h->{$k};
                next;
            }
        }
    }

=head2 _clean_vars(vars)

    Минимальная обертка вокруг _clean_hash. Копирует хеш с переменными шаблона
    в другую переменную и оставляет там только то, что может понадобиться
    для генерации отчета.

=cut
    sub _clean_vars {
        my ($vars) = @_;

        my $clean; %$clean = %$vars;
        _clean_hash($clean, $allowed_keys);

        return $clean;
    }
}

=head2 create_order(vars)

    Создает заказ на оффлайновое создание отчета из полученных переменных шаблона

=cut
sub create_order {
    my ($vars) = @_;

    my ($uid, $cid, $date_from, $date_to) = @$vars{qw/uid cid date_from date_to/};

    my $clean_vars = _clean_vars($vars);

    my $report_name;
    if ( defined $clean_vars->{stat_type} && $clean_vars->{stat_type} eq 'geo' ) {
        $report_name = iget_noop('По регионам');
    } elsif ( defined $clean_vars->{stat_type} && $clean_vars->{stat_type} eq 'custom' ) {
        $report_name = iget_noop('Мастер отчетов');
    } elsif ( defined $clean_vars->{stat_type} && $clean_vars->{stat_type} eq 'pages' ) {
        $report_name = iget_noop('По площадкам');
    } elsif ( defined $clean_vars->{stat_type} && $clean_vars->{stat_type} eq 'by_agency_clients' ) {
        $report_name = iget_noop('По клиентам агентств');
    } elsif ( defined $clean_vars->{stat_type} && $clean_vars->{stat_type} eq 'campdate' ) {
        $report_name = ''; # I dunno
    } elsif (defined $clean_vars->{detail}) {
        $report_name = iget_noop('Статистика по дням');
    } elsif (defined $clean_vars->{phrasedate}) {
        $report_name = iget_noop('Условия показа по дням');
    } elsif ( defined $clean_vars->{stat_type} && $clean_vars->{stat_type} eq 'by_clients' ) {
        $report_name = iget_noop('По клиентам');
    } elsif ( defined $clean_vars->{stat_type} && $clean_vars->{stat_type} eq 'by_managers' ) {
        $report_name = iget_noop('По менеджерам');
    } elsif (defined $clean_vars->{stat_type} && $clean_vars->{stat_type} eq 'by_agencies') {
        $report_name = iget_noop('По агентствам');
    } else {
        $report_name = iget_noop('Общая статистика');
    }
    $report_name .= '.xls';
    
    return do_sql(
        PPCDICT,
        "INSERT INTO xls_reports (uid, cid, date_from, date_to, vars, report_name)
        VALUES(?, ?, ?, ?, ?, ?)",
        $uid, $cid, $date_from, $date_to, $json->encode($clean_vars), $report_name
    );
}

=head2 release_abandoned_orders

    Находит в таблице с заказами отчеты, находящиеся в статусе обработки,
    чье соединение с базой уже отвалилось, и переводит их в статус
    необработанных, увеличивая счетчик ошибок.

=cut
sub release_abandoned_orders {

=comment
    Нижеследующий код закомментирован, потому что он совместимее
    незакомментированного, но не так надежен.
    
    Но если вероятность появления такого же connection_id, как было у кого-то
    ещё (причем относительно недавно было), мала, то сойдет и так...
=cut
#    my $processlist = get_all_sql(PPCFILES, "show full processlist")
#        or return err('Failed to get processlist');
#
#    my $conns = join ',', map { $_->{Id} } @$processlist;
#
#    do_sql(
#        PPCFILES,
#        "UPDATE xls_reports
#        SET status = ?, connect_id = 0, retries = retries + 1
#        WHERE status = ? AND retries < ? AND connect_id NOT IN ($conns)",
#        STATUS_NEW(),
#        STATUS_PROCESSING(), $Settings::OFFLINE_STAT_MAX_RETRIES
#    );

    # на сайте мускла пишут, что т-ца information_schema.processlist была добавлена
    # в 5.1.7, однако на test-mysql.yandex.ru в версии 5.0.84 она уже есть.
    # Слава Перконе!
    my $res = get_all_sql(
        PPCDICT,
        "SELECT id, connect_id FROM xls_reports WHERE status = ? AND NOT EXISTS (
            SELECT ID FROM information_schema.processlist WHERE ID = connect_id
        )",
        STATUS_PROCESSING
    );
    return 1 unless ($res && @$res);

    foreach (@$res) {
        do_sql(
            PPCDICT,
            "UPDATE xls_reports
            SET status = IF(retries >= ?, ?, ?), connect_id = 0, retries = retries + 1
            WHERE id = ? AND connect_id = ?",
            $Settings::OFFLINE_STAT_MAX_RETRIES,
            STATUS_FAILED, STATUS_NEW, 
            $_->{id}, $_->{connect_id}, 
        );
    }
    
    return 1;
}

=head2 get_order

    Возвращает наиболее ранний новый заказ на создание отчета
    Умеет возвращать заказы для конкретных кампаний (удобно для отладки) при заданном именованном параметре:
        cids -- номера кампаний, заказы для которых выбирать

=cut

sub get_order {
    my (%O) = @_;

    my $connection_id = get_one_field_sql(PPCDICT, 'SELECT CONNECTION_ID()');
    my $cids = get_one_column_sql(
        PPCDICT,
        [
            "SELECT cid FROM xls_reports",
            WHERE => { status => STATUS_PROCESSING }
        ]
    ) || [];

    my $affected = do_sql(
        PPCDICT,
        ["UPDATE xls_reports
        SET status = ?, connect_id = ?
        WHERE", {status => STATUS_NEW, ( @$cids ? ('cid__not_in' => $cids) : () ), ($O{cids} ? (cid => $O{cids}) : ())}, "
        ORDER BY create_time ASC LIMIT 1"],
        STATUS_PROCESSING, $connection_id,
    );

    if ($affected > 0) {
        return get_one_line_sql(
            PPCDICT,
            ["SELECT id, cid, vars, report_name, create_time
            FROM xls_reports
            WHERE", {status => STATUS_PROCESSING, connect_id => $connection_id, ($O{cids} ? (cid => $O{cids}) : ())}, "
            LIMIT 1", # "вот она пришла веснааа, как паранойяяя..."
            ],
        );
    } else {
        return undef;
    }
}

=head2 finish_order(order)

    Записывает содержимое сформированного отчета и переводит его в разряд
    обработанных

   %opt:
      uid - user id владельца отчета

=cut
sub finish_order {
    my ($order, %opt) = @_;

    my $report_name = $order->{report_name} =~ s/(\.[^\.]+)?$/\.$order->{file_type}/ir;

    my $storage = Direct::Storage->new();
    $storage->save('xls_reports', \$order->{xls_data}, filename => $report_name, uid => $opt{uid});
    
    do_sql(
        PPCDICT,
        "UPDATE xls_reports
        SET status = ?, ready_time = now(), connect_id = 0, report_name = ?
        WHERE id = ? AND status != ?",
        STATUS_READY, $report_name, $order->{id}, STATUS_READY,
    );

    return 1;
}

=head2 clear_old_orders

    удаляет оффлайновые отчёты старше 2 недель

=cut
sub clear_old_orders {
    do_sql(
        PPCDICT,
        "DELETE from xls_reports where create_time < NOW() - INTERVAL 2 WEEK"
    );
    # сами файлы из хранилища удаляются в ppcClearOldReports.pl
}

=head2 count_manager_clients($order, $ordinfo, \%order2uid)

 Подсчитать количество клиентов, у которых были показы
 $order = {
  ...
  OrderID => "111,222,333",
  ...
  }
  $ordinfo = {
    111 => 5, # OrderID => Shows
    222 => 10,
    ...
  }
  $order2uid = {
    $OrderID => $uid,
    ...
  }

=cut

sub count_manager_clients
{
    my ($order, $ordinfo, $order2uid) = @_;
    my %client_count = ();
    my @oid = grep { is_valid_id($_) } split /,/, ($order->{OrderID} || '');
    return 0 unless @oid;
    for my $oid (@oid) {
        next unless $ordinfo->{$oid} and $order2uid->{$oid};
        $client_count{$order2uid->{$oid}} += $ordinfo->{$oid};
    }
    return scalar grep { $_ > 0 } values %client_count;
}

1;
