package Campaign::Copy;

=head1 NAME

Campaign::Copy - копирование кампании

=head1 DESCRIPTION

=cut

use Direct::Modern;

use JSON;
use POSIX qw/strftime/;
use List::Util qw/max/;
use List::MoreUtils qw/pairwise uniq none any zip/;

use Yandex::HashUtils;
use Yandex::ListUtils;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::TimeCommon;
use Yandex::Log;

use Settings;
use PrimitivesIds;
use Primitives;
use LogTools;
use Tools qw/make_copy_sql_strings/;
use RBACElementary;
use Rbac;
use Currencies;
use HierarchicalMultipliers qw/copy_hierarchical_multipliers/;

use User qw/get_user_data/;
use BillingAggregateTools;
use Client qw/create_update_client get_client_data/;
use Campaign;
use Campaign::Types;
use Retargeting qw//;
use Phrase qw//;
use VCards qw/copy_vcard/;
use Models::AdGroup qw//;
use Models::Phrase;
use Common qw/send_banners_to_moderate/;

use Direct::Model::BannerPerformance;

use Direct::AdGroups2::Dynamic;
use Direct::DynamicConditions;
use Direct::Banners::Performance;
use Direct::PerformanceFilters;
use Direct::Validation::Campaigns qw//;
use Direct::Strategy::Tools;

use JavaIntapi::GenerateObjectIds;

use parent qw/Exporter/;
our @EXPORT = qw/
        copy_camp
/;

use constant ORDER_ID_OFFSET => 100_000_000;

=head3 get_price_convert_sql

    Возвращает кусок SQLя для конвертации ставок в указанную валюту с соблюдением
    минимальной и максимальной ставки

    $price_convert_sql = get_price_convert_sql($rate, $new_currency, $field_name, min_const => 'MIN_PRICE', max_const => 'MAX_PRICE');

=cut

sub get_price_convert_sql {
    my ($rate, $new_currency, $field_name, %consts) = @_;

    my $rate_quoted = sql_quote($rate);
    my $min_price_quoted = sql_quote(get_currency_constant($new_currency, $consts{min_const} || 'MIN_PRICE'));
    my $max_price_quoted = sql_quote(get_currency_constant($new_currency, $consts{max_const} || 'MAX_PRICE'));
    my $field_name_quoted = sql_quote_identifier($field_name);

    return "GREATEST(LEAST($field_name_quoted * $rate_quoted, $max_price_quoted), $min_price_quoted)";
}

=head2 copy_camp

 $new_cid = copy_camp($rbac, $FORM{cid}, $new_uid, $manager_uid, $agency_uid, %O);

 copy campaign to new with all banners, phrases, bids
 $O{flags} => { # флаги, уточняющие что нужно скопировать
    copy_stopped => 1|0 - копировать остановленные баннеры тоже (если copy_stopped = false, то добавляется условие statusShow == 'Yes', иначе копируются все баннеры)
    copy_archived => 1|0 - копировать архивированные баннеры тоже (если copy_archived = false, то добавляется условие statusArch == 'No', иначе копируются все баннеры)
    copy_ctr => 1|0 - копировать накопленный CTR
    copy_moderate_status => 1|0 - копировать статусы объявлений
    copy_archived_campaigns => 1|0 - копировать архивные кампании (без него будет ругаться на архивные)
    copy_phrase_status => 1|0 - копировать статус фраз (приостановка)
    copy_statusMail => 1|0 - копировать признак отправки писем об окончании средств
    copy_statusNoPay => 1|0 - копировать флаг запрещения оплаты кампании
    copy_autobudget_goal_id => 1|0, копировать ID цели в Метрике для автобюджета по конверсии
    save_conversion_strategy => 1|0, копировать стратегию и цели как есть (имеет смысл передавать только для копирования между логинами)
    copy_sms_settings => 1|0 - копировать настройки уведомлений по SMS
    copy_auto_optimize_request => 1|0 - копировать флажок автоматически созданной заявки на оптимизацию кампании
    copy_mediaplan_status => 1|0 - копировать поле mediaplan_status из camp_options
    copy_camp_description => 1|0 - копировать примечания к кампании
    copy_notification_settings => 1|0 - копировать настройки уведомлений кампании (FIO email valid sendNews sendWarn sendAccNews)
    copy_bid_place => 1|0 - копировать позицию, на которую попадала фраза при сохранении в интерфейсе
    copy_bid_warn_status => 1|0 - копировать признаки отправки предупреждений о низком CTR и потере позиции
    copy_manager => 1|0 - копировать кампанию под того же менеджера, что и у исходной кампании (аргумент $manager_uid должен быть undef)
    copy_agency => 1|0 - копировать кампанию под тоже агентство, что и у исходной кампании (аргумент $agency_uid должен быть undef)
    copy_retargetings => 1|0 - копировать условия ретаргетинга
    copy_bids_href_params => 1|0 — копировать параметры ссылок (Param1, Param2, ...)
    copy_camp_secondary_options => 1|0 — копировать дополнительные параметры кампани (таблицу camp_secondary_options)
    copy_favorite => 1|0 - копировать признак избранности ("самая важная") кампании
    skip_library_minus_words => 1|0 - не копировать библиотечные элементы
    keep_empty => 1|0 - оставить кампанию после копирования с statusEmpty = "Yes"
    stop_copied_campaigns => 1|0 - остановить кампании после копирования
    price_convert_rate => - множитель для ставок на фразы и условия ретаргетина
                            используется при копировании в аккаунт с другой валютой и конвертации копированием
    new_currency => 'RUB'|'YND_FIXED'|'EUR'|... - новая валюта, используется только совместно с price_convert_rate для определения минимальных/максимальных ставок
 }
 $O{override} => { # потабличный список значений полей, которые надо установить у скопированной кампании (если хочется, например, скопировать кампанию в другой валюте или остановленной)
     campaigns => {
         currency => 'RUB',
         statusShow => 'No',
     },
     camp_options => {...},
     banners => {...},
     phrases => {...},
     bids => {...},
     vcards => {...},
 }
 $O{new_wallet_cid} — на какой общий счёт повесить скопированную кампанию (указанный cid должен заранее существовать и быть общим счётом)
                      может быть == 0, если требуется принудительно создать кампанию, не связанную с общим счётом, у клиента с подключенным общим счётом
 $O{ids_mapping} — при наличии ссылки на хеш он будет заполнен скопированными старыми и новыми идентификаторами (cid, bid, pid, phrase_id).
     $O{ids_mapping} = {
        old_groups_banners => {
           <pid> => [
             <bid>,
             ...
           ],
           ...
        },
        new_groups_phrases => {
          <pid> => [
            <phrase_id>,
            ...,
          ],
          ...
        },
        bid_old2new => {
          <old_bid> => <new_bid>,
          ...
        },
        phrase_id_old2new => {
          <old_phrase_id> => <new_phrase_id>,
          ...
        },
        pid_old2new => {
          <old_pid> => <new_pid>,
          ...
        },
        old_pid2ret_cond_ids => {
          <old_pid> => [
            <ret_cond_id>,
            ...,
          ],
          ...
        },
        ret_cond_ids_old2new => {
          <old_ret_cond_id> => <new_ret_cond_id>,
          ...
        },
        old_gid2dyn_cond_ids => {
          old_adgroup_id = [
            <old_dyn_cond_id>,
            ...,
          ],
          ...
        },
        dyn_cond_ids_old2new => {
           <old_dyn_cond_id> => <new_dyn_cond_id>,
          ...
        },
     }

 $O{UID} - при логировании операции использовать указанный UID инициатора действия

 save RBAC info
 return new_cid

=cut

sub copy_camp
{
    my ($rbac, $old_cid, $new_uid, $manager_uid, $agency_uid, %O) = @_;

    my $UID = $O{UID};

    my $new_chief_uid = rbac_get_chief_rep_of_client_rep($new_uid);
    my $flags = $O{flags} || {};
    my $info = $O{info};
    die 'copy_manager flag and $manager_uid argument cannot be specified simultaneously' if $flags->{copy_manager} && $manager_uid;
    die 'copy_agency flag and $agency_uid argument cannot be specified simultaneously' if $flags->{copy_agency} && $agency_uid;

    my $old_camp_info = Campaign::get_camp_info($old_cid, undef, short => 1);
    die "no cid $old_cid found to copy" unless $old_camp_info && $old_camp_info->{cid};

    my $campaign_type = $old_camp_info->{type};
    my $strategy_id = $old_camp_info->{strategy_id};
    my $old_meaningful_goals = $old_camp_info->{meaningful_goals};
    croak "not allowed to copy campaign $old_cid" unless camp_kind_in(type => $campaign_type, 'copyable');
    my $is_cpm_yndx_frontpage = $campaign_type eq 'cpm_yndx_frontpage' ? 1 : 0;

    my $is_cpm_campaign = Campaign::is_cpm_campaign($campaign_type);

    my $old_chief_uid = $old_camp_info->{uid} or die "not found uid for cid=$old_cid";
    my $uid2clientid = rbac_get_client_clientids_by_uids([$new_uid, $old_chief_uid]);
    my $client_id = $uid2clientid->{$new_uid};
    die "ClientID for $new_uid not found" unless $client_id;
    my $old_client_id = $uid2clientid->{$old_chief_uid};

    my $new_client_perminfo = Rbac::get_perminfo(ClientID => $client_id);
    # Если не агентскому клиенту, на которого копируется кампания назначен персональный менеджер - создавать кампанию будем под персональным менеджером
    if (!($flags->{copy_agency} || $agency_uid) && $new_client_perminfo->{primary_manager_set_by_idm} && $new_client_perminfo->{primary_manager_uid}) {
        $manager_uid = $new_client_perminfo->{primary_manager_uid};
    }
    my $is_cpm_banner_campaign_disabled = Client::ClientFeatures::is_feature_cpm_banner_campaign_disabled_enabled($old_client_id) ||
        Client::ClientFeatures::is_feature_cpm_banner_campaign_disabled_enabled($client_id);

    # про копирование смартов еще есть такая же подпорка в InternalReports::multicurrency_copy_to_new_logins
    die "can't copy performance campaigns between clients" if $campaign_type eq 'performance' && $client_id != $old_client_id;
    die "can't copy performance content_promotion between clients" if $campaign_type eq 'content_promotion' && $client_id != $old_client_id;
    die "can't copy cpm campaigns between clients" if $is_cpm_campaign && $is_cpm_banner_campaign_disabled;
    # не копируем между клиентами, если в кампании есть креативы или ДО по фиду
    die "can't copy campaigns with feeds or canvas-creatives between clients" if ($client_id != $old_client_id && get_one_field_sql(PPC(ClientID => $old_client_id), ['
                SELECT c.cid
                FROM campaigns c
                LEFT JOIN phrases p ON p.cid = c.cid
                LEFT JOIN banners b on b.pid = p.pid
                LEFT JOIN adgroups_dynamic gd ON gd.pid = p.pid
                LEFT JOIN banners_performance b_perf ON b_perf.cid = c.cid AND b_perf.pid = p.pid
                LEFT JOIN perf_creatives perfc on perfc.creative_id = b_perf.creative_id
            ', WHERE => {
                'c.cid' => $old_cid,
                _OR => [
                    _AND => {
                        'c.type' => 'dynamic',
                        'gd.feed_id__is_not_null' => 1
                    },
                    _AND => {
                        'b_perf.banner_creative_id__is_not_null' => 1,
                        _OR => {
                            _AND => {
                                'c.type' => [qw/text mobile_content cpm_deals cpm_yndx_frontpage/],
                                'perfc.creative_type' => [qw/canvas html5_creative/],
                            },
                            _AND => {
                                _OR => [ 'c.type' => [qw/cpm_banner/], 'b.banner_type' => 'cpc_video' ],
                                'perfc.creative_type' => [qw/canvas html5_creative video_addition bannerstorage/],
                            },
                        },
                    },
                ],
               },
            ]));
    # временно не копируем между клиентами гиперлокал
    die "can't copy adgroups with hypergeo between clients" if ($client_id != $old_client_id && get_one_field_sql(PPC(ClientID => $old_client_id), ['
                SELECT c.cid
                FROM campaigns c
                JOIN phrases p ON p.cid = c.cid
                JOIN adgroup_additional_targetings aat ON aat.pid = p.pid
            ',
            WHERE => {
                'c.cid' => $old_cid,
                'aat.targeting_type' => 'auditorium_geosegments'
            },
            LIMIT => 1
            ]));

    my $agency_id = 0;

    # Если клиент на которого копируется кампания - агентский, по умолчанию будем создавать кампанию под тем же агенством
    if ($new_client_perminfo->{agency_client_id}) {
        $agency_id = $new_client_perminfo->{agency_client_id};
        $agency_uid = $new_client_perminfo->{agency_uid} if $new_client_perminfo->{agency_uid};
    }

    if ($flags->{copy_agency} || $flags->{copy_manager}) {
        if ($flags->{copy_agency}) {
            my $perminfo = Rbac::get_perminfo(ClientID => $client_id);
            $agency_uid = $perminfo && $perminfo->{agency_uid} || $old_camp_info->{AgencyUID};
            # если клиента, которому копируют кампанию, ведет ограниченный представитель, то компания скопируется под этого представителя
            $agency_id = $old_camp_info->{AgencyID};
        }
        if ($flags->{copy_manager}) {
            $manager_uid = $old_camp_info->{ManagerUID};
        }
    }

    my $banners_condition = {};
    if ($flags->{copy_stopped} && !$flags->{copy_archived}) {
        $banners_condition->{'b.statusArch'} = 'No';
    }
    if ($flags->{copy_archived} && !$flags->{copy_stopped}) {
        $banners_condition->{_NOT} = {
            'b.statusArch' => 'No',
            'b.statusShow' => 'No'
        };
    }
    if (!$flags->{copy_archived} && !$flags->{copy_stopped}) {
        $banners_condition->{'b.statusShow'} = 'Yes';
    }

    my %groups;
    my %bid_to_skip_moderation;
    my $has_banners_to_copy_mod = 0;
    my $banners = get_all_sql(PPC(cid => $old_cid), ["select b.bid, b.pid, b.banner_type from banners b",
        WHERE => {'b.cid' => $old_cid, 'b.pid__gt' => '0', %$banners_condition}]);
    foreach my $banner (@$banners) {
        $groups{$banner->{pid}} = [] if ! exists $groups{$banner->{pid}};
        if (
            $is_cpm_yndx_frontpage
            || $banner->{banner_type} eq 'cpm_outdoor'
            || $banner->{banner_type} eq 'cpm_indoor'
            || $banner->{banner_type} eq 'cpm_audio'
        ) {
            $bid_to_skip_moderation{$banner->{bid}} = 1;
        } else {
            $has_banners_to_copy_mod = 1;
        }
        push @{$groups{$banner->{pid}}}, $banner->{bid};
    }
    
    my $completed_groups = Models::AdGroup::is_completed_groups([keys %groups]);
    foreach (keys %groups) {
        delete $groups{$_} unless $completed_groups->{$_};
    }

    my $response;
    if (ref ($O{ids_mapping}) eq 'HASH') {
        $response = $O{ids_mapping};
        %$response = (
            old_groups_banners => \%groups,
            bid_old2new => {},
            new_groups_phrases => {},
            phrase_id_old2new => {},
            pid_old2new => {},
            old_pid2ret_cond_ids => {},
            ret_cond_ids_old2new => {},
            old_gid2dyn_cond_ids => {},
            dyn_cond_ids_old2new => {},
            old_gid2perf_filter_ids => {},
            perf_filter_ids_old2new => {}
        );
    }

    # copy campaigns
    my @campaign_extra_fields = qw/
        strategy
        strategy_id
        brand_survey_id
    /;

    my @campaigns_fields_to_copy = (
                   'name'
                  ,'start_time'
                  ,'finish_time'
                  ,'platform'
                  ,'strategy_name'
                  ,'strategy_data'
                  ,'autobudget'
                  ,'autobudget_date'
                  ,'geo'
                  ,'DontShow'
                  ,'disabled_ssp'
                  ,'disabled_video_placements'
                  ,'autoOptimization'
                  ,'dontShowCatalog'
                  ,'autobudgetForecastDate'
                  ,'autobudgetForecast'
                  ,'statusAutobudgetForecast'
                  ,'timeTarget'
                  ,'timezone_id'
                  ,'statusOpenStat'
                  ,'disabledIps'
                  ,'type'
                  ,'rf'
                  ,'rfReset'
                  ,'ContextLimit'
                  ,'ContextPriceCoef'
                  ,'currency'
                  ,'day_budget'
                  ,'day_budget_show_mode'
                  ,'opts'
                  ,'attribution_model'
                  ,'source'
                  ,@campaign_extra_fields
    );
    
    my $is_same_login_copy = $new_chief_uid == $old_chief_uid;

    if ($is_same_login_copy) {
        push @campaigns_fields_to_copy, qw/ab_segment_ret_cond_id ab_segment_stat_ret_cond_id brandsafety_ret_cond_id/;
    }

    if ($flags->{copy_moderate_status}) {
        push @campaigns_fields_to_copy, qw/statusShow statusModerate/;
    }
    if ($flags->{copy_archived_campaigns}) {
        push @campaigns_fields_to_copy, qw/archived/;
    }
    if ($flags->{copy_statusMail}) {
        push @campaigns_fields_to_copy, qw/statusMail/;
    }
    if ($flags->{copy_statusNoPay}) {
        push @campaigns_fields_to_copy, qw/statusNoPay/;
    }

    my $new_cid = get_new_id('cid', ClientID => $client_id);
    my $new_strategy_id = $strategy_id
                          ? Client::ClientFeatures::has_get_strategy_id_from_shard_inc_strategy_id_enabled($client_id)
                              ? get_new_id('strategy_id', ClientID => $client_id)
                              : ($new_cid + ORDER_ID_OFFSET)
                          : 0;
    my $is_intershard_copy = scalar(uniq values %{get_shard_multi(cid => [$old_cid, $new_cid])}) > 1;

    my $campaigns_default_override = {
        uid => $new_chief_uid,
        ClientID => $client_id,
        cid => $new_cid,
        statusEmpty => 'Yes',
        sum => 0,
        copiedFrom => $old_cid,
        strategy_id => $new_strategy_id,
    };
    $campaigns_default_override = hash_merge $campaigns_default_override, $O{override}->{campaigns};

    my ($campaigns_fields_str, $campaigns_values_str) = make_copy_sql_strings(\@campaigns_fields_to_copy, $campaigns_default_override);
    my $old_campaign = get_one_line_sql(PPC(cid => $old_cid), [
            "SELECT $campaigns_fields_str
            FROM campaigns
            JOIN camp_options USING(cid)
            LEFT JOIN campaigns_performance USING(cid)",
            where => {cid => $old_cid}
        ]);
    my $new_campaign = {%$old_campaign};

    my %old_camp_options = map {$_ => delete $new_campaign->{$_}} @campaign_extra_fields;
    hash_merge $new_campaign, $campaigns_default_override;
    my $today = strftime("%Y-%m-%d", localtime);
    if ($new_campaign->{start_time} lt $today) {
        $new_campaign->{start_time} = $today;
        if ($new_campaign->{finish_time} gt '0000-00-00' && $new_campaign->{finish_time} lt $today) {
            $new_campaign->{finish_time} = $today;
            $new_campaign->{statusShow} = 'No';
        }
    }
    if ($flags->{copy_moderate_status} && $new_campaign->{statusModerate} eq 'Sent') {
        $new_campaign->{statusModerate} = 'Ready';
    }
    if (    $flags->{copy_moderate_status}
         && $new_campaign->{statusModerate} ne 'New'
         && camp_kind_in(type => $old_camp_info->{type}, 'web_edit_base')
         && $has_banners_to_copy_mod == 0
    ) {
        $new_campaign->{statusModerate} = 'Ready';
    }
    # копии performance или internal_* кампаний (не черновиков) будут отправлены в модерацию сразу после создания
    if (camp_kind_in(type => $new_campaign->{type}, 'mod_export_campaigns_only') && $new_campaign->{statusModerate} ne 'New') {
        $new_campaign->{statusModerate} = 'Ready';
    }

    # заменяем currency == NULL (неявные у.е.) на YND_FIXED (явные у.е.)
    $new_campaign->{currency} ||= 'YND_FIXED';

    # подставляем модель атрибуции, если её не было
    $new_campaign->{attribution_model} ||= get_attribution_model_default();

    #Из опций удаляем is_virtual
    if ($new_campaign->{opts} =~ /(?:^|\W)is_virtual(?:\W|$)/){
        $new_campaign->{opts} = _remove_from_opts($new_campaign->{opts}, 'is_virtual');
    }

    # Не копируем опцию is_alone_trafaret_allowed между клиентами, если у нового клиента не включена фича alone_trafaret_option_enabled
    if ($new_campaign->{opts} =~ /(?:^|\W)is_alone_trafaret_allowed(?:\W|$)/
        && $client_id != $old_client_id
        && !Client::ClientFeatures::has_alone_trafaret_option_feature($client_id)
    ) {
        $new_campaign->{opts} = _remove_from_opts($new_campaign->{opts}, 'is_alone_trafaret_allowed');
    }

    # Не копируем опцию require_filtration_by_dont_show_domains между клиентами, если у нового клиента не включена фича can_require_filtration_by_dont_show_domains
    # и can_require_filtration_by_dont_show_domains_in_cpm
    if ($new_campaign->{opts} =~ /(?:^|\W)require_filtration_by_dont_show_domains(?:\W|$)/
        && $client_id != $old_client_id
        && !Client::ClientFeatures::has_can_require_filtration_by_dont_show_domains_feature($client_id)
        && !(Client::ClientFeatures::has_can_require_filtration_by_dont_show_domains_in_cpm_feature($client_id) && camp_kind_in(type => $old_camp_info->{type}, 'require_filtration_by_dont_show_domains_in_cpm'))
    ) {
        $new_campaign->{opts} = _remove_from_opts($new_campaign->{opts}, 'require_filtration_by_dont_show_domains');
    }

    # Не копируем опцию has_turbo_app между клиентами, если у нового клиента не включена фича turbo_app_allowed
    if ($new_campaign->{opts} =~ /(?:^|\W)has_turbo_app(?:\W|$)/
        && $client_id != $old_client_id
        && !Client::ClientFeatures::has_turbo_app_allowed($client_id)
    ) {
        $new_campaign->{opts} = _remove_from_opts($new_campaign->{opts}, 'has_turbo_app');
    }
    
    # Не копируем опцию s2s_tracking_enabled между клиентами, если у нового клиента не включена фича is_s2s_tracking_enabled
    if ($new_campaign->{opts} =~ /(?:^|\W)s2s_tracking_enabled(?:\W|$)/
        && $client_id != $old_client_id
        && !Client::ClientFeatures::has_s2s_tracking_feature($client_id)
    ) {
        $new_campaign->{opts} = _remove_from_opts($new_campaign->{opts}, 's2s_tracking_enabled');
    }
    
    # !!! стратегию оптимизации по конверсии меняем на недельный бюджет
    # !!! стратегию рентабельность рекламы меняем также на недельный бюджет, в случае НЕ копирования goal_id
    my $need_copy_strategy = $new_campaign->{type} ne 'wallet';
    
    my $need_copy_autobudget_goals = ($flags->{copy_autobudget_goal_id} || $is_same_login_copy);
    my $need_to_check_access_to_goals = 0;
    
    # Если true, то нужно сбросить стратегию в AutobudgetWeekSum и не копировать цели, иначе копируем стратегию и цели.
    my $drop_strategy_to_default = 0;

    # Не копируем аб-эксперименты, если на кампании есть Brand Lift
    if ($old_camp_options{brand_survey_id}) {
        delete $new_campaign->{ab_segment_ret_cond_id} if exists $new_campaign->{ab_segment_ret_cond_id};
        delete $new_campaign->{ab_segment_stat_ret_cond_id} if exists $new_campaign->{ab_segment_stat_ret_cond_id};
    }
    
    if ($need_copy_strategy) {
        my $strategy = Direct::Strategy::Tools::strategy_from_strategy_hash(from_json $new_campaign->{strategy_data});
        my $has_goal = $strategy->can("goal_id") && defined $strategy->goal_id;
        
        if ($has_goal && $strategy->goal_id == $Settings::ALL_GOALS) {
            # GoalId=0 все цели, устаревшая мета цель, поэтому не копируем ее. 
            $drop_strategy_to_default = 1;
        }
        
        my $copy_conversion_strategy_between_logins_allowed = Client::ClientFeatures::has_copy_conversion_strategy_between_logins_feature($client_id);
        
        if (($strategy->name eq 'autobudget_avg_cpa' || $strategy->name eq 'autobudget_avg_cpi') && !$is_same_login_copy) {
            if ($copy_conversion_strategy_between_logins_allowed && $flags->{save_conversion_strategy}) {
                $need_to_check_access_to_goals = 1;
            } else {
                $drop_strategy_to_default = 1;
            }
        }
        
        if (($strategy->name eq 'autobudget_roi' || $strategy->name eq 'autobudget_crr') && !$need_copy_autobudget_goals) {
            if ($copy_conversion_strategy_between_logins_allowed && $flags->{save_conversion_strategy}) {
                $need_to_check_access_to_goals = 1;
            } else {
                $drop_strategy_to_default = 1;
            }
        }
        
        if ($flags->{save_conversion_strategy} && !$drop_strategy_to_default && $need_to_check_access_to_goals) {
            my @goal_ids;
            if ($has_goal) {
                if ($strategy->{goal_id} == $Settings::MEANINGFUL_GOALS_OPTIMIZATION_GOAL_ID) {
                    my @meaningful_goal_ids = @{get_campaign_meaningful_goal_ids($old_cid)};
                    @goal_ids = @meaningful_goal_ids;
                } else {
                    push(@goal_ids, $strategy->goal_id);
                }
            }
            if (!@goal_ids || client_has_access_to_goals(\@goal_ids, $client_id)) {
                $drop_strategy_to_default = 0;
            } else {
                $drop_strategy_to_default = 1;
            }
        }
        
        if ($drop_strategy_to_default) {
            
            my $min_ab = get_currency_constant($new_campaign->{currency} || 'YND_FIXED', 'MIN_AUTOBUDGET');
            my $old_bid = $strategy->can('bid') && $strategy->bid;
            my $old_sum = $strategy->can('sum') && $strategy->sum;
            my $new_sum = $old_sum || max($min_ab, $old_bid || 0);

            my %param = (sum => $new_sum);
            $param{bid} = $old_bid  if defined $old_bid;
            $param{goal_id} = $strategy->goal_id  if $need_copy_autobudget_goals && $strategy->can("goal_id") && $strategy->goal_id;

            my $new_strategy = Direct::Strategy::AutobudgetWeekSum->new(%param);
            $new_campaign->{strategy_name} = $new_strategy->name;
            $new_campaign->{strategy_data} = $new_strategy->get_strategy_json;
        }
        
        # сбрасываем счетчик изменений при копировании
        if ($strategy->name eq 'autobudget_avg_cpv_custom_period' ||
            $strategy->name eq 'autobudget_max_impressions_custom_period' ||
            $strategy->name eq 'autobudget_max_reach_custom_period') {
            my $new_strategy =
                Direct::Strategy::Tools::strategy_from_strategy_hash(from_json $new_campaign->{strategy_data});
            $new_strategy->daily_change_count(1);
            $new_strategy->last_update_time(human_datetime());
            $new_campaign->{strategy_data} = $new_strategy->get_strategy_json;
        }
        
        # https://st.yandex-team.ru/DIRECT-134829 обновляем время биддера
        if ($strategy->has_last_bidder_restart_time_option()) {
            my $new_strategy =
                Direct::Strategy::Tools::strategy_from_strategy_hash(from_json $new_campaign->{strategy_data});
            $new_strategy->last_bidder_restart_time(human_datetime());
            $new_campaign->{strategy_data} = $new_strategy->get_strategy_json;
        }
    }

    my $client_info = get_client_data($client_id, [qw/is_using_quasi_currency feature_payment_before_moderation/]);
    my $product = product_info(type => $new_campaign->{type}, currency => $new_campaign->{currency}, quasi_currency => $client_info->{is_using_quasi_currency});
    die "no suitable product found for type = $new_campaign->{type} and currency $new_campaign->{currency}" unless $product && $product->{ProductID};
    $new_campaign->{ProductID} = $product->{ProductID};

    if ($new_campaign->{ProductID} == 503149) {
        # региональные кампании при копировании превращаем в обычные МКБ
        $new_campaign->{ProductID} = 2584;
    }

    if ($O{campaign_name_prefix} && length($O{campaign_name_prefix} . $new_campaign->{name}) <= $Direct::Validation::Campaigns::MAX_TITLE_LENGTH) {
        $new_campaign->{name} = $O{campaign_name_prefix} . $new_campaign->{name};
    }

    my $new_strategy_values = {
        strategy_id                 => $new_strategy_id,
        ClientID                    => $client_id,
        wallet_cid                  => $new_campaign->{wallet_cid} // 0,
        strategy_data               => $new_campaign->{strategy_data},
        type                        => $new_campaign->{strategy_name},
        day_budget                  => $new_campaign->{day_budget},
        day_budget_show_mode        => $new_campaign->{day_budget_show_mode},
        enable_cpc_hold             => ($new_campaign->{opts} =~ /(?:^|\W)enable_cpc_hold(?:\W|$)/) ? 'Yes' : 'No',
        attribution_model           => $new_campaign->{attribution_model},
        meaningful_goals            => undef,
    };

    log_copy_camp("START", $new_cid, {from => $old_cid});

    do_insert_into_table(PPC(cid => $new_cid), 'campaigns', $new_campaign);

    if ($need_copy_strategy) {
        my $old_strategy = Campaign::define_strategy($old_campaign);
        Campaign::mark_strategy_change(
            $new_cid, $new_uid,
            Campaign::define_strategy({%$new_campaign, strategy => $old_camp_options{strategy}}),
            $old_strategy, $UID,
        );
    }

    # логируем факт копирования кампании и то, откуда и с какими параметрами она была скопирована
    # делаем это сразу после появления нового cid'а, т.к. копирование может произойти не до конца и
    # кампания не должна остаться без записи о факте своего копирования
    EventLog::log_event(
        slug => 'campaign_copied',
        ClientID => $client_id,
        cid => $new_cid,
        params => {
            from_cid => $old_cid,
            flags => $flags,
            manager_uid => $manager_uid,
            agency_uid => $agency_uid,
        },
    );

    # copy camp_options
    my $camp_options_default_override = {
        cid => $new_cid,
    };

    my @camp_options_fields_to_copy = (
                               'contactinfo'
                              ,'money_warning_value'
                              ,'banners_per_page'
                              ,'warnPlaceInterval'
                              ,'statusMetricaControl'
                              ,'fairAuction'
                              ,'offlineStatNotice'
                              ,'email_notifications'
                              ,'minus_words'
                              ,'broad_match_flag'
                              ,'broad_match_limit'
                              ,'statusContextStop'
                              ,'strategy'
                              ,'manual_autobudget_sum'
                              ,'competitors_domains'
                              ,'device_targeting'
                              ,'status_click_track'
                              ,'content_lang'
                              ,'mobile_app_goal'
                              ,'impression_standard_time'
                              ,'eshows_banner_rate'
                              ,'eshows_video_rate'
                              ,'eshows_video_type'
                              ,'placement_types'
    );

    if (!$drop_strategy_to_default) {
        push @camp_options_fields_to_copy, 'meaningful_goals';
        $new_strategy_values->{meaningful_goals} = Direct::Model::Campaign::_serialize_meaningful_goals($old_meaningful_goals);
    }
    if ($flags->{copy_moderate_status}) {
        push @camp_options_fields_to_copy, 'statusPostModerate';
    }
    if ($flags->{copy_sms_settings}) {
        push @camp_options_fields_to_copy, qw/sms_time sms_flags/;
    } else {
        $camp_options_default_override->{sms_time} = '09:00:21:00';
    }
    if ($flags->{copy_auto_optimize_request}) {
        push @camp_options_fields_to_copy, qw/auto_optimize_request/;
    }
    if ($flags->{copy_mediaplan_status}) {
        push @camp_options_fields_to_copy, 'mediaplan_status';
    }
    if ($flags->{copy_camp_description}) {
        push @camp_options_fields_to_copy, 'camp_description';
    }
    my $new_login_info = get_user_data($new_uid, [qw/fio email valid sendNews sendWarn sendAccNews ClientID/]);
    if ($flags->{copy_notification_settings}) {
        push @camp_options_fields_to_copy, qw/FIO email valid sendNews sendWarn sendAccNews/;
    } else {
        hash_copy $camp_options_default_override, $new_login_info, qw/email valid sendNews sendWarn sendAccNews/;
        $camp_options_default_override->{FIO} = $new_login_info->{fio};
        if ($is_cpm_campaign) {
            $camp_options_default_override->{sendWarn} = 'No';
        }
    }

    $camp_options_default_override = hash_merge $camp_options_default_override, $O{override}->{camp_options};
    if ($campaign_type eq 'mcbanner') {
        # see %DoCmd::Checks::CAMPAIGN_MCBANNER__DISABLED_PARAMS
        $camp_options_default_override->{sendWarn} = 'No';
    }

    do_in_transaction {
            # copy camp minus_words
            if ($old_chief_uid == $new_chief_uid) {
                # копирование в пределах одного клиента - можно просто копировать ID'шник
                push @camp_options_fields_to_copy, 'mw_id';
            } else {
                # копирование другому клиенту - ему нужно сохранить "собственный" экземпляр минус-слов
                my $old_camp_mw = get_one_line_sql(PPC(cid => $old_cid), [
                        "SELECT mw.mw_id, IFNULL(mw.mw_text, co.minus_words) as minus_words
                   FROM camp_options co
                        left join minus_words mw on mw.mw_id = co.mw_id
                ", WHERE => { cid => $old_cid } ]);

                if ($old_camp_mw) {
                    # сохранение с дедупликацией
                    my $new_mw_id = MinusWords::save_minus_words(
                        MinusWordsTools::minus_words_str2array($old_camp_mw->{minus_words})
                        , $client_id);
                    $camp_options_default_override->{mw_id} = $new_mw_id;
                }
            }

            my ($camp_options_fields_str, $camp_options_values_str) = make_copy_sql_strings(
                \@camp_options_fields_to_copy, $camp_options_default_override);
            do_insert_select_sql(PPC(cid => $old_cid), "INSERT INTO camp_options ($camp_options_fields_str) VALUES %s",
                [ "SELECT $camp_options_values_str FROM camp_options", where => { cid => $old_cid } ],
                dbw => PPC(cid => $new_cid) );
        };

    if ($new_strategy_id) {
        do_insert_into_table(PPC(cid => $new_cid), 'strategies', $new_strategy_values);
        do_insert_into_table(PPCDICT, 'shard_inc_strategy_id', { strategy_id => $new_strategy_id, ClientID => $client_id }, ignore => 1);
    }


    if (exists $old_campaign->{placement_types}) {
        Campaign::mark_placement_types_change(
            $new_cid, $new_uid, $UID,
            $old_campaign->{placement_types}
        );
    }

    my $old_campaign_mobile_content_id;
    if ($campaign_type eq 'mobile_content') {
        my $old_mobile_app_id = get_one_field_sql(PPC(cid => $old_cid),
            [ 'SELECT mobile_app_id FROM campaigns_mobile_content', WHERE => { cid => $old_cid } ]);
        my $new_mobile_app_id;

        if ($client_id != $old_client_id && $old_mobile_app_id != 0) {
            $new_mobile_app_id = find_same_mobile_app($old_client_id, $client_id, $old_mobile_app_id);
            if (!defined $new_mobile_app_id) {
                # если не нашли, нужно создать (возможно скопировать домены между шардами)
                $new_mobile_app_id = copy_mobile_app(
                    $old_client_id, $client_id, $old_mobile_app_id, $is_intershard_copy, $flags);
            }
        } else {
            $new_mobile_app_id = $old_mobile_app_id;
        }

        $old_campaign_mobile_content_id = get_mobile_content_id_by_mobile_app_id($old_client_id, $old_mobile_app_id);

        # полученный mobile_app_id нового нужно записать в новую campaigns_mobile_content
        my ($fields, $values) = make_copy_sql_strings(
            [qw/is_installed_app device_type_targeting network_targeting mobile_app_id/],
            {cid => $new_cid, mobile_app_id => $new_mobile_app_id});
        do_insert_select_sql(PPC(cid => $old_cid), "INSERT INTO campaigns_mobile_content ($fields) VALUES %s",
                                              ["SELECT $values FROM campaigns_mobile_content", where => {cid => $old_cid}],
                                              dbw => PPC(cid => $new_cid) );
    }
    if ($old_chief_uid == $new_chief_uid) { #копируем промоакции только при внутриклиентском копировании
        my ($fields, $values) = make_copy_sql_strings([qw/promoaction_id/],{cid => $new_cid});
        do_insert_select_sql(PPC(cid => $old_cid), "INSERT INTO campaign_promoactions ($fields) VALUES %s",
                                              ["SELECT $values FROM campaign_promoactions", where => {cid => $old_cid}],
                                              dbw => PPC(cid => $new_cid) );
    }
    if ($is_cpm_yndx_frontpage) {
        my ($fields, $values) = make_copy_sql_strings([qw/allowed_frontpage_types/],{cid => $new_cid});
        do_insert_select_sql(PPC(cid => $old_cid), "INSERT INTO campaigns_cpm_yndx_frontpage ($fields) VALUES %s",
                                              ["SELECT $values FROM campaigns_cpm_yndx_frontpage", where => {cid => $old_cid}],
                                              dbw => PPC(cid => $new_cid) );
    }

    # копируем данные из таблицы camp_additional_data
    my @camp_additonal_data_fields_fields_to_copy = ('href', 'camp_aim_type', 'company_name', 'business_category', 'budget_display_format');
    my ($camp_additonal_data_fields_str, $camp_additonal_data_values_str) = make_copy_sql_strings(\@camp_additonal_data_fields_fields_to_copy, {cid => $new_cid});
    do_insert_select_sql(PPC(cid => $old_cid), "INSERT INTO camp_additional_data ($camp_additonal_data_fields_str) VALUES %s", 
                                              ["SELECT $camp_additonal_data_values_str FROM camp_additional_data", where => {cid => $old_cid}],
                                              dbw => PPC(cid => $new_cid) );

    #Копируем "в довесок", не подтягивая в кампанию дополнительные поля
    my %camp_additional_table = (
        performance => [
            ['now_optimizing_by'],
            {cid => $new_cid, now_optimizing_by => 'CPC'}
        ]
    );

    if (exists $camp_additional_table{$campaign_type}) {
        my ($source_fields, $new_values) = @{$camp_additional_table{$campaign_type}};
        my $table = "campaigns_$campaign_type";
        # сконвертированные значения bid/cpa для performance
        hash_merge $new_values, $O{override}->{$table} if exists $O{override}->{$table};
        my ($fields, $values) = make_copy_sql_strings($source_fields, $new_values);
        do_insert_select_sql(PPC(cid => $old_cid), "INSERT INTO $table ($fields) VALUES %s", 
                                              ["SELECT $values FROM $table", where => {cid => $old_cid}],
                                              dbw => PPC(cid => $new_cid) );
    }


    # copy camp_secondary_options
    if ($flags->{copy_camp_secondary_options}) {
        my $camp_secondary_options_default_override = {
            cid => $new_cid,
        };
        my @camp_secondary_options_fields_to_copy = ('options', 'key');
        $camp_secondary_options_default_override = hash_merge $camp_secondary_options_default_override, $O{override}->{camp_secondary_options};
        my ($camp_secondary_options_fields_str, $camp_secondary_options_values_str) = make_copy_sql_strings(\@camp_secondary_options_fields_to_copy, $camp_secondary_options_default_override);
        do_insert_select_sql(PPC(cid => $old_cid), "INSERT INTO camp_secondary_options ($camp_secondary_options_fields_str) VALUES %s", 
                                                  ["SELECT $camp_secondary_options_values_str FROM camp_secondary_options",
                                                    where => {cid => $old_cid, key => 'default'}],
                                                  dbw => PPC(cid => $new_cid) );
    }

    # copy metrika_counters
    do_insert_select_sql(PPC(cid => $old_cid), "INSERT INTO metrika_counters (cid, metrika_counter, source, is_deleted) VALUES %s", 
                                               "SELECT ?, metrika_counter, source, is_deleted FROM metrika_counters where cid =?",
                                               binds => [$new_cid, $old_cid],
                                               dbw => PPC(cid => $new_cid) );

    if ($new_strategy_id){
        do_insert_select_sql(PPC(cid => $old_cid), "INSERT INTO strategy_metrika_counters (strategy_id, metrika_counter, source, is_deleted) VALUES %s",
            "SELECT ?, metrika_counter, source, is_deleted FROM strategy_metrika_counters where strategy_id =?",
            binds => [$new_strategy_id, $strategy_id],
            dbw => PPC(cid => $new_cid) );

        do_insert_select_sql(PPC(cid => $old_cid), "INSERT INTO strategy_inventori_data (strategy_id, general_auction_probability) VALUES %s",
            "SELECT ?, general_auction_probability FROM strategy_inventori_data where strategy_id =?",
            binds => [$new_strategy_id, $strategy_id],
            dbw => PPC(cid => $new_cid) );
    }

    # copy camp_metrika_counters
    do_insert_select_sql(PPC(cid => $old_cid), "INSERT INTO camp_metrika_counters (cid, metrika_counters) VALUES %s", 
                                               "SELECT ?, metrika_counters FROM camp_metrika_counters where cid =?",
                                               binds => [$new_cid, $old_cid],
                                               dbw => PPC(cid => $new_cid) );

    # copy camp_dialogs and client_dialogs
    my $dialog = get_one_line_sql(PPC(cid => $old_cid),[
        qq/
        SELECT cd.client_dialog_id, ClientID, cld.skill_id, cld.bot_guid, cld.name, cld.is_active, last_sync_time/,
            FROM => 'camp_dialogs cd',
            JOIN => 'client_dialogs cld ON (cd.client_dialog_id = cld.client_dialog_id)',
            WHERE => {cid => $old_cid}
    ]);
    if ($dialog && keys %$dialog){
        if ($new_chief_uid != $old_chief_uid) {
            $dialog->{client_dialog_id} = get_new_id('client_dialog_id');
            $dialog->{ClientID} = $new_login_info->{ClientID};
            do_insert_into_table(PPC(cid => $new_cid), 'client_dialogs', $dialog, ignore => 1);

            #Возможно, у нового клиента уже был такой диалог в базе и наш INSERT проигнорировался.
            #Поэтому перезапросим client_dialog_id после сохранения.
            $dialog->{client_dialog_id} = get_one_field_sql(PPC(ClientID => $dialog->{ClientID}),[
                qq/SELECT client_dialog_id/,
                FROM => 'client_dialogs',
                WHERE => {ClientID => $dialog->{ClientID}, skill_id => $dialog->{skill_id}},
            ]);
        }
        do_insert_into_table(PPC(cid => $new_cid), 'camp_dialogs', {client_dialog_id => $dialog->{client_dialog_id}, cid => $new_cid});
    }

    # для performance кампании $need_copy_autobudget_goals == true
    if ($need_copy_autobudget_goals || ($flags->{save_conversion_strategy} && !$drop_strategy_to_default)) {
        my @fields_to_copy = qw/goal_id goals_count stat_date context_goals_count/;
        my ($fields_str, $values_str) = make_copy_sql_strings(\@fields_to_copy, {cid => $new_cid});
        do_insert_select_sql(PPC(cid => $old_cid),
            "INSERT INTO camp_metrika_goals ($fields_str) VALUES %s",
            ["SELECT $values_str FROM camp_metrika_goals", WHERE => {cid => $old_cid}],
            dbw => PPC(cid => $new_cid),
        );
    }

    # copy user_campaigns_favorite
    if ($flags->{copy_favorite}) {
        if (get_one_field_sql(PPC(cid => $old_cid), ['SELECT 1 FROM user_campaigns_favorite', WHERE => {cid => $old_cid}])) {
            do_insert_into_table(PPC(cid => $new_cid), 'user_campaigns_favorite', {uid => $new_chief_uid, cid => $new_cid});
        }
    }

    copy_hierarchical_multipliers($old_client_id, $client_id, [ [{cid => $old_cid}, {cid => $new_cid}] ]);

    # хеш для новых vcard_id. Нужен, чтобы одинаковые визитки из старой кампании стали одинаковыми визитками в новой кампании
    my %NEW_VCARD_ID;
    # хеш соответствия старый => новый sitelinks_set_id
    my %NEW_SITELINKS_SET_ID;
    # хэш соответствий старый => новый mw_id
    my %NEW_MINUSWORDS_ID;
    # хэш соответствий старый => новый mw_id библиотечных элементов
    my %NEW_LIBRARY_MINUSWORDS_ID;
    my $new_bids_to_mod_queue;

    my @phrases_fields_to_copy = qw/
        adgroup_type
        geo
        statusShowsForecast
        forecastDate
        group_name
    /;
    # в реальности значения этих полей не копируются а заменяются переопределениями ниже по коду
    push @phrases_fields_to_copy, qw/
        pid
        mw_id
        cid
    /;
    if ($flags->{copy_moderate_status}) {
        push @phrases_fields_to_copy, qw(
            statusModerate
            statusPostModerate
        );
    }

    my @group_params_fields_to_copy = qw(
        has_phraseid_href
    );

    my @adgroup_bs_tags_fields_to_copy = qw(
        page_group_tags_json
        target_tags_json
    );

    my @adgroup_hypergeo_fields_to_copy = qw(
        ret_cond_id
    );

    my @adgroups_mobile_content_fields_to_copy = qw(
        store_content_href
        device_type_targeting
        network_targeting
        min_os_version
    );
    push @adgroups_mobile_content_fields_to_copy, 'mobile_content_id'  if $is_same_login_copy;
    my @banners_mobile_content_fields_to_copy = qw(
        reflected_attrs
        primary_action
    );

    my @banners_content_promotion_video_fields_to_copy = qw(
        content_promotion_video_id
        packshot_href
    );

    my @banners_content_promotion_fields_to_copy = qw(
        content_promotion_id
        visit_url
    );

    my @adgroups_content_promotion_fields_to_copy = qw(
        content_promotion_type
    );

    my @mod_reasons_fields_to_copy = qw(
        type
        statusModerate
        statusPostModerate
        reason
    );

    my @bids_fields_to_copy = qw(
        phrase
        norm_phrase
        numword
        showsForecast
        autobudgetPriority
    );
    my %price_constant_names;
    %price_constant_names = %{Currencies::CPM_PRICE_CONSTANT_NAMES()} if ($is_cpm_campaign);
    my %bids_price_override;
    if (defined $O{price_convert_rate}) {
        die "no new_currency given" unless defined $O{new_currency};
        %bids_price_override = (
            price__sql => get_price_convert_sql($O{price_convert_rate}, $O{new_currency}, 'price'),
            price_context__sql => get_price_convert_sql($O{price_convert_rate}, $O{new_currency}, 'price_context', %price_constant_names),
        );
    } else {
        push @bids_fields_to_copy, qw(
            price
            price_context
        );
    }
    if ($flags->{copy_phrase_status}) {
        push @bids_fields_to_copy, qw/is_suspended/;
    }
    if ($flags->{copy_bid_place}) {
        push @bids_fields_to_copy, qw/place/;
    }
    if ($flags->{copy_bid_warn_status}) {
        push @bids_fields_to_copy, qw/warn/;
    }
    if ($flags->{copy_moderate_status}) {
        push @bids_fields_to_copy, qw/statusModerate/;
    }

    my @bids_tables_to_copy = qw/bids/;
    if ($flags->{copy_archived_campaigns}) {
        push @bids_tables_to_copy, 'bids_arc';
    }

    my @bids_href_params_fields_to_copy = map { lc($_) } @Models::Phrase::BIDS_HREF_PARAMS;

    my @bids_base_fields_to_copy = qw(
        bid_type
        autobudgetPriority
        relevance_match_categories
    );
    if ($flags->{copy_phrase_status}) {
        push @bids_base_fields_to_copy, qw/opts/;
    }
    my %bids_base_price_override;
    if (defined $O{price_convert_rate}) {
        die "no new_currency given" unless defined $O{new_currency};
        %bids_base_price_override = (
            price__sql => get_price_convert_sql($O{price_convert_rate}, $O{new_currency}, 'price'),
            price_context__sql => get_price_convert_sql($O{price_convert_rate}, $O{new_currency}, 'price_context', %price_constant_names),
        );
    } else {
        push @bids_base_fields_to_copy, qw(
            price
            price_context
        );
    }
    my @bids_base_tables_to_copy = qw/bids_base/;

    my @banners_fields_to_copy = qw(
        type
        banner_type
        title
        title_extension
        body
        language
        href
        domain
        domain_id
        reverse_domain
        geoflag
        opts
    );
    # в реальности значения этих полей не копируются а заменяются переопределениями ниже по коду
    push @banners_fields_to_copy, qw/
        bid
        cid
        pid
        vcard_id
        sitelinks_set_id
    /;
    if ($flags->{copy_moderate_status}) {
        push @banners_fields_to_copy, qw(
            statusShow
            statusModerate
            statusPostModerate
            statusArch
            statusSitelinksModerate
            phoneflag
            flags
        );
    }

    my @banner_display_hrefs_fields_to_copy = qw(bid display_href);
    if ($flags->{copy_moderate_status}) {
        push @banner_display_hrefs_fields_to_copy, 'statusModerate';
    }

    my @banners_additions_fields_to_copy = qw(additions_type sequence_num);
    push @banners_additions_fields_to_copy, 'additions_item_id'  if $is_same_login_copy;
    my @banner_prices_fields_to_copy = qw(bid prefix price price_old currency);
    my @banner_permalinks_fields_to_copy = qw(bid permalink chain_id permalink_assign_type);
    my @organizations_fields_to_copy = qw(ClientID permalink_id chain_id status_publish);
    my @campaign_permalinks_fields_to_copy = qw(cid permalink_id chain_id);
    my @campaign_phones_fields_to_copy = qw(cid client_phone_id);
    my @camp_measurers_fields_to_copy = qw(cid measurer_system params);
    my @additions_callouts_fields_to_copy = qw(hash callout_text flags is_deleted);
    $flags->{copy_callouts_moderate_status} = $is_same_login_copy;
    if ($flags->{copy_callouts_moderate_status}) {
        push @additions_callouts_fields_to_copy, 'statusModerate';
    }

    my @catalogia_banners_rubrics_fields_to_copy = qw(
        categories_bs
    );

    my %mobile_content_old2new;
    my %additions_old2new;
    my @old_pids = nsort keys %groups;
    for my $old_pids_chunk (chunks \@old_pids, 1_000) {
        my $pids_cnt = scalar @$old_pids_chunk;
        my $new_pids;
        my %pid_old2new;

        # Копирование динамических и перфоманс групп
        if ($campaign_type eq 'dynamic' || $campaign_type eq 'performance') {
            my $class = $campaign_type eq 'dynamic' ? 'Direct::AdGroups2::Dynamic' : 'Direct::AdGroups2::Performance'; 
            my $src_pid2dst_pid = $class->copy(
                [adgroup_id => $old_pids_chunk, with_tags => 1, with_adgroup_bs_tags => 1],
                $old_client_id, $client_id,
                sub {
                    my $old_group = shift;
                    my $new_group = $old_group->new();

                    if ($campaign_type eq 'dynamic') {
                        $new_group->main_domain($old_group->main_domain);
                        $new_group->feed_id($old_group->feed_id);
                        $new_group->relevance_match_categories($old_group->relevance_match_categories);
                    } elsif ($campaign_type eq 'performance') {
                        $new_group->feed_id($old_group->feed_id);
                        $new_group->field_to_use_as_name($old_group->field_to_use_as_name);
                        $new_group->field_to_use_as_body($old_group->field_to_use_as_body);
                    }

                    $new_group->client_id($client_id);
                    $new_group->campaign_id($new_cid);
                    $new_group->status_bs_synced('No');
                    for my $field (qw/adgroup_name geo status_shows_forecast forecast_date minus_words tags adgroup_bs_tags href_params/) {
                        $new_group->$field(scalar $old_group->$field);
                    }
                    if ($flags->{copy_moderate_status}) {
                        if (any { $old_group->status_moderate eq $_ } qw/Sending Sent/) {
                            $new_group->status_moderate('Ready');
                        } else {
                            $new_group->status_moderate($old_group->status_moderate);
                        }
                        if ($old_group->status_post_moderate eq 'Sent') {
                            $new_group->status_post_moderate('Ready');
                        } else {
                            $new_group->status_post_moderate($old_group->status_post_moderate);
                        }

                        $new_group->status_bl_generated('Processing') if $new_group->status_moderate ne 'New';
                    }

                    return $new_group;
                },
                primary_key => 'pid',
                with_multipliers => 1,
                with_mod_reasons => $flags->{copy_moderate_status},
            );

            %pid_old2new = %$src_pid2dst_pid;
            $new_pids = [values %pid_old2new];
        } else {
            # Старое копирование
            $new_pids = JavaIntapi::GenerateObjectIds->new(object_type => 'adgroup',
                        count => $pids_cnt, client_id => $client_id)->call();
            %pid_old2new = map { $old_pids_chunk->[$_] => $new_pids->[$_] } 0 .. $pids_cnt - 1;

            do_in_transaction {
                    # copy minus_words
                    my $old_pid2new_mw_id = { };
                    if ($old_chief_uid == $new_chief_uid) {
                        # копирование в пределах одного клиента - можно просто копировать ID'шники
                        $old_pid2new_mw_id = get_hash_sql(PPC(cid => $old_cid),
                            [ "SELECT pid, mw_id FROM phrases", WHERE => { pid => $old_pids_chunk, mw_id__gt => 0 } ]);
                    } else {
                        # копирование другому клиенту - ему нужно сохранить "собственные" экземпляры минус-слов
                        my $old_pid2old_mw = get_hashes_hash_sql(PPC(cid => $old_cid), [
                                "SELECT p.pid, mw.mw_id, mw.mw_text
                           FROM phrases p 
                                join minus_words mw using (mw_id)
                        ", WHERE => { pid => $old_pids_chunk } ]);

                        while (my ($old_pid, $old_mw) = each %$old_pid2old_mw) {
                            # сохранение с кешированием и дедупликацией
                            $old_pid2new_mw_id->{$old_pid} =
                                $NEW_MINUSWORDS_ID{$old_mw->{mw_id}} //=
                                MinusWords::save_minus_words(MinusWordsTools::minus_words_str2array($old_mw->{mw_text}),
                                    $client_id);
                        }
                    }

                    # copy phrases
                    my %phrases_override = (
                        pid                     => \%pid_old2new,
                        mw_id                   => $old_pid2new_mw_id,
                        cid                     => $new_cid,
                        statusModerate__sql     => "IF(adgroup_type IN ('cpm_outdoor', 'cpm_indoor', 'cpm_yndx_frontpage'), IF(statusModerate != 'New', 'Ready', 'New'), IF(statusModerate IN ('Sent', 'Sending'), 'Ready', statusModerate))",
                        statusPostModerate__sql => "IF(adgroup_type IN ('cpm_outdoor','cpm_indoor',  'cpm_yndx_frontpage'), 'No', IF(statusPostModerate IN ('Sent'), 'Ready', statusPostModerate))",
                    );
                    my ($phrases_fields_str, $phrases_values_str) = make_copy_sql_strings(\@phrases_fields_to_copy,
                        \%phrases_override, by => 'pid', override_fields_to_copy_only => 1);
                    do_insert_select_sql(PPC(cid => $old_cid), "INSERT INTO phrases ($phrases_fields_str) VALUES %s",
                        [ "SELECT $phrases_values_str FROM phrases", WHERE => { pid => $old_pids_chunk } ],
                        dbw => PPC(cid => $new_cid));
                };

            # copy group_params
            my %group_params_override = (
                pid => \%pid_old2new,
            );
            my ($group_params_fields_str, $group_params_values_str) = make_copy_sql_strings(\@group_params_fields_to_copy, \%group_params_override, by => 'pid');
            do_insert_select_sql(PPC(cid => $old_cid), "INSERT IGNORE INTO group_params ($group_params_fields_str) VALUES %s", 
                                                       ["SELECT $group_params_values_str FROM group_params", WHERE => {pid => $old_pids_chunk}],
                                                      dbw => PPC(cid => $new_cid));

            # copy adgroup_bs_tags
            my %adgroup_bs_tags_override = (
                pid => \%pid_old2new,
            );
            my ($adgroup_bs_tags_fields_str, $adgroup_bs_tags_values_str) = make_copy_sql_strings(\@adgroup_bs_tags_fields_to_copy, \%adgroup_bs_tags_override, by => 'pid');
            do_insert_select_sql(PPC(cid => $old_cid), "INSERT IGNORE INTO adgroup_bs_tags ($adgroup_bs_tags_fields_str) VALUES %s", 
                                                       ["SELECT $adgroup_bs_tags_values_str FROM adgroup_bs_tags", WHERE => {pid => $old_pids_chunk}],
                                                      dbw => PPC(cid => $new_cid));

            # copy hypergeo
            my %adgroup_hypergeo_override = (
                pid => \%pid_old2new,
            );
            my ($adgroup_hypergeo_fields_str, $adgroup_hypergeo_values_str) = make_copy_sql_strings(\@adgroup_hypergeo_fields_to_copy, \%adgroup_hypergeo_override, by => 'pid');
            do_insert_select_sql(PPC(cid => $old_cid), "INSERT IGNORE INTO adgroups_hypergeo_retargetings ($adgroup_hypergeo_fields_str) VALUES %s",
                                                       ["SELECT $adgroup_hypergeo_values_str FROM adgroups_hypergeo_retargetings", WHERE => {pid => $old_pids_chunk}],
                                                      dbw => PPC(cid => $new_cid));
            
            # copy adgroup_additional_targetings
            my $old_targeting_id_to_old_pid = get_hash_sql(PPC(cid => $old_cid), [
                "SELECT id, pid FROM adgroup_additional_targetings",
                WHERE => { pid => $old_pids_chunk },
            ]);
            my @old_targetings_ids = keys %$old_targeting_id_to_old_pid;

            my $targetings_cnt = scalar @old_targetings_ids;
            my $new_targetings_ids = get_new_id_multi('id', $targetings_cnt);

            my %old_targeting_id_to_new = zip @old_targetings_ids, @$new_targetings_ids;

            my @targetings_fields_to_copy = qw(
                id
                pid
                targeting_type
                targeting_mode
                value_join_type
                value
            );
            my %old_targeting_id_to_new_pid = map { $_ => $pid_old2new{ $old_targeting_id_to_old_pid->{$_} } } @old_targetings_ids;
            my $targetings_override = {
                id => \%old_targeting_id_to_new,
                pid => \%old_targeting_id_to_new_pid
            };

            my ($targetings_fields_str, $targetings_values_str) = make_copy_sql_strings(\@targetings_fields_to_copy, $targetings_override, by => 'id');

            do_insert_select_sql(PPC(cid => $old_cid),
                "INSERT INTO adgroup_additional_targetings ($targetings_fields_str) VALUES %s",
                ["SELECT $targetings_values_str FROM adgroup_additional_targetings", where => {id => \@old_targetings_ids}],
                dbw => PPC(cid => $new_cid)
            );

            if (!$flags->{skip_library_minus_words}) {
                do_in_transaction {
                    if ($old_chief_uid == $new_chief_uid) {
                        my ($adgroups_minus_words_fields_str, $adgroups_minus_words_values_str) = make_copy_sql_strings([ 'mw_id' ], { pid => \%pid_old2new }, by => 'pid');
                        do_insert_select_sql(PPC(cid => $old_cid), "INSERT IGNORE INTO adgroups_minus_words ($adgroups_minus_words_fields_str) VALUES %s",
                            [ "SELECT $adgroups_minus_words_values_str FROM adgroups_minus_words", WHERE => { pid => $old_pids_chunk } ],
                            dbw => PPC(cid => $new_cid));
                    }
                    else {
                        my $old_pid2new_lib_mw_id = {};
                        #существующие библ. элементы у старого клиента
                        my $old_mw_id2mw = get_hashes_hash_sql(PPC(ClientID => $old_client_id), [
                            "SELECT mw.mw_id, mw.mw_text, mw.mw_name, mw.mw_hash
                           FROM minus_words mw
                        ", WHERE => { ClientID => $old_client_id, is_library => 1 } ]);

                        #существующие библ. элементы у нового клиента
                        my $new_mw_id2mw = get_all_sql(PPC(ClientID => $client_id), [
                            "SELECT mw.mw_id, mw.mw_text, mw.mw_name, mw.mw_hash
                           FROM minus_words mw
                        ", WHERE => { ClientID => $client_id, is_library => 1 } ]);
                        my $new_client_libr_count = scalar @$new_mw_id2mw;

                        my %new_mw_hash_with_name2mw = map {$_->{mw_hash} . '||||' . $_->{mw_name} => 1} @$new_mw_id2mw;
                        for my $old_mw (values %$old_mw_id2mw) {
                            my $complex_hash = $old_mw->{mw_hash} . '||||' . $old_mw->{mw_name};
                            if (!$new_mw_hash_with_name2mw{$complex_hash}) {
                                $new_client_libr_count++;
                            }
                        }

                        #библиотечных элементов может быть не больше N
                        if ($new_client_libr_count > $Settings::LIBRARY_MINUS_WORDS_COUNT_LIMIT) {
                            croak "not allowed to copy campaign when client have more than 30 library minus_words"
                        }

                        while (my ($old_mw_id, $old_mw) = each %$old_mw_id2mw) {
                            $NEW_LIBRARY_MINUSWORDS_ID{$old_mw_id} //=
                                MinusWords::save_library_minus_words({ name => $old_mw->{mw_name}, text => MinusWordsTools::minus_words_str2array($old_mw->{mw_text}) }, $client_id);
                        }

                        # копирование другому клиенту - ему нужно сохранить "собственные" экземпляры библиотечных минус-слов
                        my $old_libr_group_data = get_all_sql(PPC(cid => $old_cid), [
                            "SELECT amw.pid, mw.mw_id, mw.mw_text, mw.mw_name
                           FROM adgroups_minus_words amw
                                join minus_words mw using (mw_id)
                        ", WHERE => { pid => $old_pids_chunk, mw_id => [keys %$old_mw_id2mw] } ]);

                        for my $old_data (@$old_libr_group_data) {
                            # сохранение с кешированием и дедупликацией
                           push @{$old_pid2new_lib_mw_id->{$old_data->{pid}}},
                                $NEW_LIBRARY_MINUSWORDS_ID{$old_data->{mw_id}} //=
                                MinusWords::save_library_minus_words( {name => $old_data->{mw_name} , text => MinusWordsTools::minus_words_str2array($old_data->{mw_text}) }, $client_id);
                        }

                        my @insert_minus_words = ();
                        for my $old_pid (keys %$old_pid2new_lib_mw_id) {
                            my $new_pid = $pid_old2new{$old_pid};
                            push @insert_minus_words, map { [$new_pid, $_] } @{$old_pid2new_lib_mw_id->{$old_pid}};
                        }
                        do_mass_insert_sql(PPC(cid => $new_cid), "INSERT IGNORE INTO adgroups_minus_words (pid, mw_id) VALUES %s", \@insert_minus_words);
                    }
                };
            }

            if ($campaign_type eq 'cpm_price') {
                my ($adgroup_project_params_fields_str, $adgroup_project_params_values_str) = make_copy_sql_strings(['project_param_conditions'], { pid => \%pid_old2new}, by => 'pid');
                do_insert_select_sql(PPC(cid => $old_cid), "INSERT IGNORE INTO adgroup_project_params ($adgroup_project_params_fields_str) VALUES %s",
                                                           ["SELECT $adgroup_project_params_values_str FROM adgroup_project_params", WHERE => {pid => $old_pids_chunk}],
                                                          dbw => PPC(cid => $new_cid));
            }
            if ($campaign_type eq 'cpm_banner' || $campaign_type eq 'cpm_deals') {
                my ($adgroups_cpm_banner_fields_str, $adgroups_cpm_banner_values_str) = make_copy_sql_strings(['criterion_type', 'has_in_banner_creatives'], { pid => \%pid_old2new}, by => 'pid');
                do_insert_select_sql(PPC(cid => $old_cid), "INSERT IGNORE INTO adgroups_cpm_banner ($adgroups_cpm_banner_fields_str) VALUES %s",
                                                           ["SELECT $adgroups_cpm_banner_values_str FROM adgroups_cpm_banner", WHERE => {pid => $old_pids_chunk}],
                                                          dbw => PPC(cid => $new_cid));
            }
            if ($campaign_type eq 'cpm_banner' || $campaign_type eq 'cpm_deals') {
                my ($adgroups_cpm_video_fields_str, $adgroups_cpm_video_values_str) = make_copy_sql_strings(['is_non_skippable'], { pid => \%pid_old2new}, by => 'pid');
                do_insert_select_sql(PPC(cid => $old_cid), "INSERT IGNORE INTO adgroups_cpm_video ($adgroups_cpm_video_fields_str) VALUES %s",
                                                           ["SELECT $adgroups_cpm_video_values_str FROM adgroups_cpm_video", WHERE => {pid => $old_pids_chunk}],
                                                          dbw => PPC(cid => $new_cid));
            }
            if ($campaign_type eq 'cpm_banner') {
                #copy page targets
                my ($adgroups_cpm_banner_fields_str, $adgroups_cpm_banner_values_str) = make_copy_sql_strings(['page_blocks', 'display_areas'], { pid => \%pid_old2new}, by => 'pid');
                do_insert_select_sql(PPC(cid => $old_cid), "INSERT IGNORE INTO adgroup_page_targets ($adgroups_cpm_banner_fields_str) VALUES %s",
                    ["SELECT $adgroups_cpm_banner_values_str FROM adgroup_page_targets", WHERE => {pid => $old_pids_chunk}],
                    dbw => PPC(cid => $new_cid));

                #copy video_segment_goals
                my %video_segment_goals_override = (
                    pid                      => \%pid_old2new,
                    external_audience_status => 'few_data',
                    internal_status          => 'New',
                    external_audience_id     => 0,
                    segment_owner_uid        => 0,
                    segment_error_count      => 0,
                );
                my ($video_segment_goals_fields_str, $video_segment_goals_values_str) = make_copy_sql_strings(['audience_type', 'is_disabled'], \%video_segment_goals_override, by => 'pid');
                do_insert_select_sql(PPC(cid => $old_cid), "INSERT IGNORE INTO video_segment_goals ($video_segment_goals_fields_str) VALUES %s",
                    ["SELECT $video_segment_goals_values_str FROM video_segment_goals", WHERE => {pid => $old_pids_chunk}],
                    dbw => PPC(cid => $new_cid));
            }

            copy_hierarchical_multipliers(
                $old_client_id, $client_id,
                [
                    map { [ {cid => $old_cid, pid => $_},
                            {cid => $new_cid, pid => $pid_old2new{$_}}
                        ] } keys %pid_old2new,
                ]
            );

            # copy mobile content
            if (camp_kind_in(type => $new_campaign->{type}, 'store_content')) {
                my %amc_override = ( pid => \%pid_old2new );

                # при копировании другому клиенту делаем копию mobile_content
                if ($new_chief_uid != $old_chief_uid) {
                    my $chunk_pid2mcid = get_hash_sql(PPC(cid => $old_cid), [
                            'SELECT DISTINCT pid, mobile_content_id FROM adgroups_mobile_content',
                            WHERE => { pid => $old_pids_chunk },
                        ]);
                    my @chunk_old_mcids = uniq values %$chunk_pid2mcid;
                    my @chunk_missed_mcids = grep {!$mobile_content_old2new{$_}} @chunk_old_mcids;

                    # ищем уже заполненные mcid, чтобы избежать конфликта по UNIQUE KEY
                    if (@chunk_missed_mcids) {
                        my $same_mcid_old2new = find_same_mobile_contents($old_client_id, $client_id, \@chunk_missed_mcids);
                        hash_merge \%mobile_content_old2new, $same_mcid_old2new;
                        @chunk_missed_mcids = grep {!$mobile_content_old2new{$_}} @chunk_missed_mcids;
                    }

                    # создаём новые записи
                    if (@chunk_missed_mcids) {
                        my $new_mcid_map = copy_mobile_contents(
                            $old_client_id, $client_id, \@chunk_missed_mcids, $is_intershard_copy, $flags);
                        hash_merge \%mobile_content_old2new, $new_mcid_map;
                    }

                    # нужно обработать случай с разными приложениями
                    # в группах -- такое может быть после миграции старой кампании
                    my @extra_mc_ids = grep {$_ != $old_campaign_mobile_content_id} @chunk_old_mcids;
                    if (@extra_mc_ids) {
                        my $mc_to_app_id = MobileApps::get_mobile_app_ids_by_mobile_content_ids($old_client_id, \@extra_mc_ids);
                        my @mobile_app_ids = uniq values %$mc_to_app_id;

                        foreach my $app_id (@mobile_app_ids) {
                            my $new_app_id = find_same_mobile_app($old_client_id, $client_id, $app_id);
                            if (!defined $new_app_id) {
                                # mobile_content соотвествующий мобильному приложению уже был скопирован выше
                                # поэтому тут копироваться могут только сами приложения и домены
                                copy_mobile_app($old_client_id, $client_id, $app_id, $is_intershard_copy, $flags);
                            }
                        }
                    }

                    my $chunk_mcid_map = hash_cut \%mobile_content_old2new, @chunk_old_mcids;
                    my %chunk_pid2new_mcid = map {$_ => $chunk_mcid_map->{$chunk_pid2mcid->{$_}}} keys %$chunk_pid2mcid;
                    $amc_override{mobile_content_id} = \%chunk_pid2new_mcid;
                }

                my ($amc_fields_str, $amc_values_str) = make_copy_sql_strings(
                    \@adgroups_mobile_content_fields_to_copy,
                    \%amc_override,
                    by => 'pid',
                );
                do_insert_select_sql(PPC(cid => $old_cid),
                    "INSERT IGNORE INTO adgroups_mobile_content ($amc_fields_str) VALUES %s",
                    ["SELECT $amc_values_str FROM adgroups_mobile_content", WHERE => { pid => $old_pids_chunk }],
                    dbw => PPC(cid => $new_cid),
                );
            }

            # copy mod_reasons for phrases
            if ($flags->{copy_moderate_status}) {
                my %mod_reasons_phrases_override = (
                    id => \%pid_old2new,
                    timeCreated => unix2mysql(time()),
                    statusSending => 'Yes',
                    ClientID => $client_id,
                    cid => $new_cid,
                );
                my ($mod_reasons_phrases_fields_str, $mod_reasons_phrases_values_str) = make_copy_sql_strings(\@mod_reasons_fields_to_copy, \%mod_reasons_phrases_override, by => 'id');
                do_insert_select_sql(PPC(cid => $old_cid), "INSERT INTO mod_reasons ($mod_reasons_phrases_fields_str) VALUES %s", 
                                                           ["SELECT $mod_reasons_phrases_values_str FROM mod_reasons", WHERE => {id => $old_pids_chunk, type => 'phrases'}],
                                                          dbw => PPC(cid => $new_cid));
            }
        }

        my @old_bids = nsort map { @{ $groups{$_} || [] } } @$old_pids_chunk;
        my (%bid_old2new, %newpid2newfirstbid, %oldpid2newfirstbid);

        my @old_perf_bids;
        if ($campaign_type eq 'performance') {
            @old_perf_bids = @{ get_one_column_sql(PPC(cid => $old_cid),
                [ "SELECT bid FROM banners", WHERE => { bid => \@old_bids, banner_type => "performance" } ]) };
            @old_bids = @{ xminus(\@old_bids, \@old_perf_bids) };
        }

        # copy old performance banners
        if (@old_perf_bids && !Client::ClientFeatures::creative_free_interface_enabled($client_id)) {
            # copy without sitelinks, vcards, images
            my ($src2dst, $new_bid2new_pid) = Direct::Banners::Performance->copy(
                [banner_id => \@old_perf_bids],
                $old_client_id, $client_id,
                sub {
                    my $old_banner = shift;
                    my $new_status_moderate = $old_banner->status_moderate;
                    my $new_status_post_moderate = $old_banner->status_post_moderate;
                    if (any { $old_banner->status_moderate eq $_ } qw/Sending Sent/) {
                        $new_status_moderate = 'Ready';
                    }
                    if ($old_banner->status_post_moderate eq 'Sent') {
                        $new_status_post_moderate = 'Ready';
                    }
                    return Direct::Model::BannerPerformance->new(
                        campaign_id => $new_cid,
                        adgroup_id => $pid_old2new{$old_banner->adgroup_id},
                        creative_id => $old_banner->creative_id,
                        status_bs_synced => 'No',
                        ($flags->{copy_moderate_status} 
                            ? (
                                status_moderate => $new_status_moderate,
                                status_post_moderate => $new_status_post_moderate,
                            ) : ()),
                    );
                },
                primary_key => 'bid',
                with_mod_reasons => $flags->{copy_moderate_status}
            );

            %bid_old2new = %$src2dst; 
            my %pid_new2old = map { ($pid_old2new{$_} => $_) } keys %pid_old2new;
            while (my ($bid, $pid) = each %$new_bid2new_pid) {
                # bid записываем случайный
                $newpid2newfirstbid{$pid} = $bid; 
                $oldpid2newfirstbid{$pid_new2old{$pid}} = $bid;
            }
        }
        # copy other banners
        if (@old_bids) {
            # copy vcards, addresses, org_details
            my %oldbid2newvcardid;
            my $bids_vcards_data = get_all_sql(PPC(cid => $old_cid), [
                   "SELECT      b.bid, b.vcard_id, vc.address_id, vc.org_details_id
                    FROM        banners b
                    INNER JOIN  vcards vc USING(vcard_id)
                 ", WHERE => {'b.bid' => \@old_bids, 'b.vcard_id__gt' => 0},
                   'ORDER BY b.bid']) || [];
            for my $bid_vcard (@$bids_vcards_data) {
                my ($old_bid, $old_vcard_id, $old_address_id, $old_org_details_id) = @{$bid_vcard}{
                  qw(bid       vcard_id       address_id       org_details_id)};
                if ($old_vcard_id){
                    unless ($NEW_VCARD_ID{$old_vcard_id}){
                        # если копируем другому клиенту то нужно скопировать адрес из addresses
                        # copy addresses
                        my $new_address_id;
                        if ($old_chief_uid == $new_chief_uid || ! $old_address_id) {
                            # один и тот же клиент или адрес не задан
                            # тот же адрес что и в старой кампании
                            $new_address_id = $old_address_id;
                        } else {
                            # адрес есть и клиенты отличаются, клонируем строку в addresses. если такой адрес уже есть, то его оставляем (не перезаписываем map_id, map_id_auto...)
                            $new_address_id = CommonMaps::copy_address($old_address_id, $old_chief_uid, $client_id);
                        }
    
                        # copy org_details
                        my $new_org_details_id;
                        if ($old_chief_uid == $new_chief_uid || ! $old_org_details_id) {
                            # один и тот же клиент или огрн не задан
                            # тот же огрн что и в старой кампании
                            $new_org_details_id = $old_org_details_id;
                        } else {
                            # огрн есть и клиенты разные - нужно использовать другую (потенциально новую) запись в org_details
                            $new_org_details_id = OrgDetails::copy_org_details($old_org_details_id, $old_chief_uid, $new_chief_uid);
                        }
    
                        my $vcards_override = {
                            cid => $new_cid,
                            uid => $new_chief_uid,
                            address_id => $new_address_id,
                            org_details_id => $new_org_details_id,
                        };
                        $NEW_VCARD_ID{$old_vcard_id} = copy_vcard($old_vcard_id, $vcards_override);
                    }
                    $oldbid2newvcardid{$old_bid} = $NEW_VCARD_ID{$old_vcard_id};
                }
            }

            # copy organizations
            if ($campaign_type eq 'text' || $campaign_type eq 'cpm_banner') {
                my $organization_permalink_ids_to_copy = get_one_column_sql(PPC(cid => $old_cid),
                    ["SELECT DISTINCT permalink FROM banners b JOIN banner_permalinks bp ON b.bid = bp.bid AND bp.permalink_assign_type = 'manual'", WHERE => { cid => $old_cid }],
                );
                my $default_permalink_id = get_one_field_sql(PPC(cid => $old_cid),
                    ["SELECT permalink_id FROM campaign_permalinks", WHERE => { cid => $old_cid }],
                );
                if ($default_permalink_id) {
                    push @$organization_permalink_ids_to_copy, $default_permalink_id;
                }

                if (@$organization_permalink_ids_to_copy) {
                    $organization_permalink_ids_to_copy = [ uniq @$organization_permalink_ids_to_copy ];

                    my ($organizations_fields_str, $organizations_values_str) = make_copy_sql_strings(
                        \@organizations_fields_to_copy,
                        {
                            ClientID => { $old_client_id => $client_id },
                        },
                        by => 'ClientID',
                        override_fields_to_copy_only => 1,
                    );

                    do_insert_select_sql(PPC(cid => $old_cid),
                        "INSERT IGNORE INTO organizations ($organizations_fields_str) VALUES %s",
                        ["SELECT $organizations_values_str FROM organizations", WHERE => { permalink_id => $organization_permalink_ids_to_copy }],
                        dbw => PPC(cid => $new_cid),
                    );
                }
            }

            # copy campaign_permalinks
            my ($campaign_permalinks_fields_str, $campaign_permalinks_values_str) = make_copy_sql_strings(
                \@campaign_permalinks_fields_to_copy,
                {
                    cid => { $old_cid => $new_cid },
                },
                by => 'cid',
                override_fields_to_copy_only => 1,
            );
            do_insert_select_sql(PPC(cid => $old_cid),
                "INSERT IGNORE INTO campaign_permalinks ($campaign_permalinks_fields_str) VALUES %s",
                ["SELECT $campaign_permalinks_values_str FROM campaign_permalinks", WHERE => { cid => $old_cid }],
                dbw => PPC(cid => $new_cid),
            );
            
            # copy campaign_phones
            my ($campaign_phones_fields_str, $campaign_phones_values_str) = make_copy_sql_strings(
                \@campaign_phones_fields_to_copy, 
                { cid => { $old_cid => $new_cid } }, 
                by => 'cid', 
                override_fields_to_copy_only => 1
            );
            do_insert_select_sql(PPC(cid => $old_cid),
                "INSERT IGNORE INTO campaign_phones ($campaign_phones_fields_str) VALUES %s",
                ["SELECT $campaign_phones_values_str FROM campaign_phones", WHERE => {cid => $old_cid }],
                dbw => PPC(cid => $new_cid)
            );

            # copy camp_measurers
            my ($camp_measurers_fields_str, $camp_measurers_values_str) = make_copy_sql_strings(
                \@camp_measurers_fields_to_copy,
                {
                    cid => { $old_cid => $new_cid },
                },
                by => 'cid',
                override_fields_to_copy_only => 1,
            );
            do_insert_select_sql(PPC(cid => $old_cid),
                "INSERT IGNORE INTO camp_measurers ($camp_measurers_fields_str) VALUES %s",
                ["SELECT $camp_measurers_values_str FROM camp_measurers", WHERE => {cid => $old_cid }],
                dbw => PPC(cid => $new_cid)
            );
            # copy sitelinks
            my $old_bid2new_sitelinks_set_id = {};
            my $old_bid2old_sitelinks_set_id = get_hash_sql(PPC(cid => $old_cid), [
                   "SELECT b.bid, b.sitelinks_set_id
                      FROM banners b
                  ", WHERE => {'b.bid' => \@old_bids, 'b.sitelinks_set_id__gt' => 0},
            ]);

            my $old_bid2old_status_sitelinks_moderate = get_hash_sql(PPC(cid => $old_cid), [
                "SELECT b.bid, b.statusSitelinksModerate
                      FROM banners b
                  ", WHERE => {'b.bid' => \@old_bids},
            ]);
            my $old_bid2new_status_sitelinks_moderate = {};
            foreach my $old_bid (@old_bids) {
                # "IF(statusSitelinksModerate IN ('Sent', 'Sending'), 'Ready', statusSitelinksModerate)",
                my $status_sitelinks_moderate = $old_bid2old_status_sitelinks_moderate->{$old_bid};
                if ($status_sitelinks_moderate eq 'Sent' || $status_sitelinks_moderate eq 'Sending') {
                    $status_sitelinks_moderate = 'Ready';
                }
                $old_bid2new_status_sitelinks_moderate->{$old_bid} = $status_sitelinks_moderate;
            }

            if ($old_chief_uid == $new_chief_uid) {
                # копирование в пределах одного клиента - можно просто копировать ID'шники сетов
                $old_bid2new_sitelinks_set_id = $old_bid2old_sitelinks_set_id;
            } else {
                # копирование другому клиенту - ему нужно сохранить "собственные" экземпляры сайтлинков
                my $bids_sitelinks_data = Sitelinks::get_sitelinks_by_set_id_multi([values %$old_bid2old_sitelinks_set_id]);
                while (my ($old_bid, $old_sitelinks_set_id) = each %$old_bid2old_sitelinks_set_id) {
                    my $sitelinks_set = $bids_sitelinks_data->{$old_sitelinks_set_id};
                    #при копировании между клиентами привязанные к сайтлинкам турболендинги очищаем т.к. турболендинг привязан к клиенту
                    my $valid_sitelinks = [];
                    foreach my $sitelink (@$sitelinks_set) {
                        if ($sitelink->{turbolanding} || $sitelink->{tl_id}) {
                            delete $sitelink->{turbolanding};
                            delete $sitelink->{tl_id};
                            $info->{sitelink_tl_ids_purged} //= 1;
                        }
                        #у сайтлинка могло не быть основной ссылки - только турболендинг
                        #в этом случае после удаления турболендинга получится невалидный сайтлинк, его копировать не будем
                        if ($sitelink->{href}) {
                            push @$valid_sitelinks, $sitelink;
                        } else {
                            $info->{empty_sitelinks_removed} //= 1;
                        }
                    }
                    if (@$valid_sitelinks) {
                        $bids_sitelinks_data->{$old_sitelinks_set_id} = $valid_sitelinks;
                        # если у нового клиента есть такой же сет - вернется его id
                        $NEW_SITELINKS_SET_ID{$old_sitelinks_set_id} //= Sitelinks::save_sitelinks_set($valid_sitelinks, $client_id);
                        $old_bid2new_sitelinks_set_id->{$old_bid} = $NEW_SITELINKS_SET_ID{$old_sitelinks_set_id};
                    } else {
                        delete $bids_sitelinks_data->{$old_sitelinks_set_id};
                        $old_bid2new_sitelinks_set_id->{$old_bid} = undef;
                        # если удаляем sitelinks_set_id, то сбрасываем статус в New DIRECT-132618
                        $old_bid2new_status_sitelinks_moderate->{$old_bid} = 'New';
                    }
                }
            }

            # copy banners
            if (@old_bids) {
                my (%oldbid2oldpid);
                for my $old_pid (@$old_pids_chunk) {
                    for my $old_bid ( @{ $groups{$old_pid} || [] } ) {
                        $oldbid2oldpid{$old_bid} = $old_pid;
                    }
                }

                for my $old_bids_chunk (chunks \@old_bids, 1_000) {
                    my %part_bid_old2new;
                    my %copy_statuses_part_bid_old2new;
                    my $bids_cnt = scalar @$old_bids_chunk;
                    my $new_bids_ids = JavaIntapi::GenerateObjectIds->new(object_type => 'banner',
                            count => $bids_cnt, client_id => $client_id)->call();
                    my %oldbid2newpid;
                    for my $i (0 .. $bids_cnt - 1) {
                        my $old_bid = $old_bids_chunk->[$i];
                        my $new_bid = $new_bids_ids->[$i];
                        $bid_old2new{$old_bid} = $new_bid;
                        $part_bid_old2new{$old_bid} = $new_bid;
                        $copy_statuses_part_bid_old2new{$old_bid} = $new_bid if ! $bid_to_skip_moderation{$old_bid};
                        my $old_pid = $oldbid2oldpid{$old_bid};
                        my $new_pid = $pid_old2new{$old_pid};
                        $oldpid2newfirstbid{$old_pid} ||= $new_bid;
                        $newpid2newfirstbid{$new_pid} ||= $new_bid;
                        $oldbid2newpid{$old_bid} = $new_pid;
                        $new_bids_to_mod_queue->{$old_bid} = {new_bid => $new_bid, new_pid => $new_pid, old_pid => $old_pid} if ! $bid_to_skip_moderation{$old_bid};
                    }
                    my %banners_override = (
                        bid => \%part_bid_old2new,
                        cid => $new_cid,
                        pid => \%oldbid2newpid,
                        vcard_id => \%oldbid2newvcardid,
                        sitelinks_set_id => $old_bid2new_sitelinks_set_id,
                        statusSitelinksModerate => $old_bid2new_status_sitelinks_moderate,
                        phoneflag__sql => "IF(phoneflag IN ('Sent', 'Sending'), 'Ready', phoneflag)",
                    );
                    if ($is_cpm_yndx_frontpage) {
                        $banners_override{statusModerate__sql} = "IF(statusModerate != 'New', 'Ready', 'New')";
                        $banners_override{statusPostModerate} = "No";
                        $banners_override{flags} = undef;
                    } else {
                        $banners_override{statusModerate__sql} = "IF(banner_type IN ('cpm_outdoor', 'cpm_indoor', 'cpm_audio'), IF(statusModerate != 'New', 'Ready', 'New'), IF(statusModerate IN ('Sent', 'Sending'), 'Ready', statusModerate))";
                        $banners_override{statusPostModerate__sql} = "IF(banner_type IN ('cpm_outdoor', 'cpm_indoor', 'cpm_audio'), 'No', IF(statusPostModerate IN ('Sent'), 'Ready', statusPostModerate))";
                        $banners_override{flags__sql} = "IF(banner_type IN ('cpm_outdoor', 'cpm_indoor', 'cpm_audio'), null, flags)";
                    }

                    if ($is_intershard_copy) {
                        my $domain_ids_to_copy = get_one_column_sql(PPC(cid => $old_cid), [
                            'SELECT DISTINCT domain_id FROM banners',
                            where => {bid => $old_bids_chunk}
                        ]);
                        _copy_ppc_domains($domain_ids_to_copy, PPC(cid => $old_cid), PPC(cid => $new_cid));
                    }
                    my ($banners_fields_str, $banners_values_str) = make_copy_sql_strings(\@banners_fields_to_copy, \%banners_override, by => 'bid', override_fields_to_copy_only => 1);
                    do_insert_select_sql(PPC(cid => $old_cid), "INSERT INTO banners ($banners_fields_str) VALUES %s", 
                                                              ["SELECT $banners_values_str FROM banners", where => {bid => $old_bids_chunk}],
                                                              dbw => PPC(cid => $new_cid));

                    # copy performance main banners
                    my ($bpm_fields_str, $bpm_values_str) = make_copy_sql_strings([qw/bid pid/], \%banners_override, by => 'bid', override_fields_to_copy_only => 1);
                    do_insert_select_sql(PPC(cid => $old_cid), "INSERT INTO banners_performance_main ($bpm_fields_str) VALUES %s",
                                                              ["SELECT $bpm_values_str FROM banners_performance_main", where => {bid => $old_bids_chunk}],
                                                              dbw => PPC(cid => $new_cid));

                    # copy turbolandings
                    if ($is_same_login_copy) {
                        my %turbolandings_override = (
                            bid => \%part_bid_old2new, 
                            cid => $new_cid, 
                        );
                        if ($is_cpm_yndx_frontpage) {
                            $turbolandings_override{statusModerate__sql} = "IF(statusModerate != 'New', 'Ready', 'New')";
                        }
                        else{
                            $turbolandings_override{statusModerate__sql} = "IF(statusModerate IN ('Sent', 'Sending'), 'Ready', statusModerate)";
                        }
                        my ($btl_fields_str, $btl_values_str) = make_copy_sql_strings(
                            [qw/cid bid tl_id statusModerate/], \%turbolandings_override, by => 'bid');
                        do_insert_select_sql(PPC(cid => $old_cid),
                            "INSERT IGNORE INTO banner_turbolandings ($btl_fields_str) VALUES %s",
                            ["SELECT $btl_values_str FROM banner_turbolandings", WHERE => {cid => $old_cid, bid => $old_bids_chunk}],
                            dbw => PPC(cid => $new_cid),
                        );
                    }

                    # copy turbolanding_href_params
                    if ( $campaign_type eq 'text' ) {
                        my ($btlh_fields_str, $btlh_values_str) = make_copy_sql_strings(
                            [qw/href_params/],
                            { bid => \%part_bid_old2new},
                            by => 'bid',
                        );
                        do_insert_select_sql(PPC(cid => $old_cid),
                            "INSERT IGNORE INTO banner_turbolanding_params ($btlh_fields_str) VALUES %s",
                            ["SELECT $btlh_values_str FROM banner_turbolanding_params", WHERE => {bid => $old_bids_chunk}],
                            dbw => PPC(cid => $new_cid),
                        );
                    }

                    # copy banner_multicards
                    my $multicards = get_hash_sql(PPC(cid => $old_cid),
                        ["SELECT multicard_id, bid from banner_multicards", WHERE => { bid => $old_bids_chunk, }]);
                    my @multicard_ids = keys $multicards;
                    my $new_multicard_ids = get_new_id_multi('banner_multicards_multicard_id', scalar @multicard_ids);

                    my %multicard_id_old2new = map { 
                        ($multicard_ids[$_] => $new_multicard_ids->[$_]) 
                    } (0 .. $#multicard_ids);

                    my %multicard_id_old2new_bid = map {
                        ($_ => $part_bid_old2new{$multicards->{$_}})
                    } @multicard_ids;

                    # картинки копируются в mass_copy_banner_image
                    my ($bm_fields_str, $bm_values_str) = make_copy_sql_strings(
                        [qw/order_num text image_hash href/],
                        {
                            bid => \%multicard_id_old2new_bid, 
                            multicard_id => \%multicard_id_old2new,
                        },
                        by => 'multicard_id',
                    );

                    do_insert_select_sql(PPC(cid => $old_cid),
                        "INSERT IGNORE INTO banner_multicards ($bm_fields_str) VALUES %s",
                        ["SELECT $bm_values_str FROM banner_multicards", WHERE => {bid => $old_bids_chunk}],
                        dbw => PPC(cid => $new_cid),
                    );

                    # copy banner_multicard_sets
                    my ($bms_fields_str, $bms_values_str) = make_copy_sql_strings(
                        [qw/statusModerate/],
                        { 
                            bid => \%part_bid_old2new,
                            statusModerate__sql => "IF(statusModerate = 'New', 'New', 'Ready')",
                        },
                        by => 'bid',
                    );
                    do_insert_select_sql(PPC(cid => $old_cid),
                        "INSERT IGNORE INTO banner_multicard_sets ($bms_fields_str) VALUES %s",
                        ["SELECT $bms_values_str FROM banner_multicard_sets", WHERE => {bid => $old_bids_chunk}],
                        dbw => PPC(cid => $new_cid),
                    );
                    
                    # copy banner_logos
                    my ($bl_fields_str, $bl_values_str) = make_copy_sql_strings(
                        [qw/image_hash statusModerate/],
                        {
                            bid => \%part_bid_old2new,
                            statusModerate__sql => "IF(statusModerate = 'New', 'New', 'Ready')",
                        },
                        by => 'bid',
                    );
                    do_insert_select_sql(PPC(cid => $old_cid),
                        "INSERT IGNORE INTO banner_logos ($bl_fields_str) VALUES %s",
                        ["SELECT $bl_values_str FROM banner_logos", WHERE => {bid => $old_bids_chunk}],
                        dbw => PPC(cid => $new_cid),
                    );

                    # copy banner_buttons
                    my ($bb_fields_str, $bb_values_str) = make_copy_sql_strings(
                        [qw/key caption href statusModerate/],
                        {
                            bid => \%part_bid_old2new,
                            statusModerate__sql => "IF(statusModerate = 'New', 'New', 'Ready')",
                        },
                        by => 'bid',
                    );
                    do_insert_select_sql(PPC(cid => $old_cid),
                        "INSERT IGNORE INTO banner_buttons ($bb_fields_str) VALUES %s",
                        ["SELECT $bb_values_str FROM banner_buttons", WHERE => {bid => $old_bids_chunk}],
                        dbw => PPC(cid => $new_cid),
                    );

                    # copy display hrefs
                    my ($bdh_fields_str, $bdh_values_str) = make_copy_sql_strings(
                        \@banner_display_hrefs_fields_to_copy,
                        {
                            bid => \%part_bid_old2new,
                            statusModerate__sql => "IF(statusModerate IN ('Sent', 'Sending'), 'Ready', statusModerate)",
                        },
                        by => 'bid',
                        override_fields_to_copy_only => 1,
                    );
                    do_insert_select_sql(PPC(cid => $old_cid),
                        "INSERT IGNORE INTO banner_display_hrefs ($bdh_fields_str) VALUES %s",
                        ["SELECT $bdh_values_str FROM banner_display_hrefs", WHERE => { bid => $old_bids_chunk }],
                        dbw => PPC(cid => $new_cid),
                    );

                    # copy banner prices
                    my ($bp_fields_str, $bp_values_str) = make_copy_sql_strings(
                        \@banner_prices_fields_to_copy,
                        {
                            bid => \%part_bid_old2new,
                        },
                        by => 'bid',
                        override_fields_to_copy_only => 1,
                    );
                    do_insert_select_sql(PPC(cid => $old_cid),
                        "INSERT IGNORE INTO banner_prices ($bp_fields_str) VALUES %s",
                        ["SELECT $bp_values_str FROM banner_prices", WHERE => { bid => $old_bids_chunk }],
                        dbw => PPC(cid => $new_cid),
                    );

                    # copy manual permalinks
                    if ($campaign_type eq 'text' || $campaign_type eq 'cpm_banner') {
                        my ($banner_permalinks_fields_str, $banner_permalinks_values_str) = make_copy_sql_strings(
                            \@banner_permalinks_fields_to_copy,
                            {
                                bid => \%part_bid_old2new,
                            },
                            by => 'bid',
                            override_fields_to_copy_only => 1,
                        );
                        do_insert_select_sql(PPC(cid => $old_cid),
                            "INSERT IGNORE INTO banner_permalinks ($banner_permalinks_fields_str) VALUES %s",
                            ["SELECT $banner_permalinks_values_str FROM banner_permalinks", WHERE => { bid => $old_bids_chunk, permalink_assign_type => 'manual' }],
                            dbw => PPC(cid => $new_cid),
                        );
                    }

                    # copy additions
                    if (camp_kind_in(type => $new_campaign->{type}, 'callouts')) {
                        my %ba_override = (bid => \%part_bid_old2new);

                        if ($new_chief_uid != $old_chief_uid) {
                            my $chunk_old_callout = get_hash_sql(PPC(cid => $old_cid), [
                                    'SELECT DISTINCT additions_item_id, hash
                                    FROM banners_additions ba
                                    JOIN additions_item_callouts aic using(additions_item_id)',
                                    WHERE => {
                                        bid => $old_bids_chunk,
                                        additions_type => 'callout',
                                    },
                                ]);
                            my @chunk_old_aids = keys %$chunk_old_callout;
                            my @chunk_missed_aids = grep {!$additions_old2new{$_}} @chunk_old_aids;

                            if (@chunk_missed_aids) {
                                my $existing_aid = get_hash_sql(PPC(ClientID => $client_id), [
                                        'SELECT hash, additions_item_id
                                        FROM additions_item_callouts',
                                        WHERE => {
                                            ClientID => $client_id,
                                            hash => [map {$chunk_old_callout->{$_}} @chunk_missed_aids],
                                            is_deleted => 0,
                                        },
                                    ]);
                                if (%$existing_aid) {
                                    for my $old_aid (@chunk_missed_aids) {
                                        my $new_aid = $existing_aid->{$chunk_old_callout->{$old_aid}};
                                        next if !$new_aid;
                                        $additions_old2new{$old_aid} = $new_aid;
                                    }
                                    @chunk_missed_aids = grep {!$additions_old2new{$_}} @chunk_missed_aids;
                                }
                            }

                            if (@chunk_missed_aids) {
                                my $new_aids = get_new_id_multi('additions_item_id', scalar @chunk_missed_aids);
                                my %new_aid_map = map {($chunk_missed_aids[$_] => $new_aids->[$_])} (0 .. $#chunk_missed_aids);
                                hash_merge \%additions_old2new, \%new_aid_map;

                                my %ai_override = (
                                    additions_item_id => \%new_aid_map,
                                    ClientID => $client_id,
                                );
                                $ai_override{statusModerate} = 'Ready'  if !$flags->{copy_callouts_moderate_status};

                                my ($ai_fields_str, $ai_values_str) = make_copy_sql_strings(
                                    \@additions_callouts_fields_to_copy,
                                    \%ai_override,
                                    by => 'additions_item_id',
                                );
                                do_insert_select_sql(PPC(cid => $old_cid),
                                    "INSERT INTO additions_item_callouts ($ai_fields_str) VALUES %s ON DUPLICATE KEY UPDATE is_deleted = 0, last_change = now()",
                                    ["SELECT $ai_values_str FROM additions_item_callouts", WHERE => {additions_item_id => \@chunk_missed_aids}],
                                    dbw => PPC(cid => $new_cid),
                                );
                            }

                            my $chunk_aid_map = hash_cut \%additions_old2new, @chunk_old_aids;
                            $ba_override{additions_item_id__sql} = sql_case(additions_item_id => $chunk_aid_map);
                        }

                        my ($ba_fields_str, $ba_values_str) = make_copy_sql_strings(
                            \@banners_additions_fields_to_copy,
                            \%ba_override,
                            by => 'bid',
                        );
                        do_insert_select_sql(PPC(cid => $old_cid),
                            "INSERT IGNORE INTO banners_additions ($ba_fields_str) VALUES %s",
                            [
                                "SELECT $ba_values_str FROM banners_additions",
                                WHERE => { additions_type => 'callout', bid => $old_bids_chunk },
                            ],
                            dbw => PPC(cid => $new_cid),
                        );
                    }

                    # copy canvas-creatives & video addition
                    my $copy_perf_creative_types = [qw/canvas html5_creative bannerstorage/];
                    if ($old_chief_uid == $new_chief_uid) {
                        # кампании с канвас-креативами не копируются между разными пользователями вообще (умираем в самом начале)
                        # кампании с видео-дополнениями копируются, но в-д пропускаются
                        push @$copy_perf_creative_types, 'video_addition';
                    }

                    my $old_banner_creatives = get_all_sql(
                        PPC(cid => $old_cid), [
                            "select bp.banner_creative_id, bp.bid, pc.creative_type,
                                IF(bp.statusModerate IN ('Sent', 'Sending'), 'Ready', bp.statusModerate) as statusModerate
                            from banners_performance bp
                            inner join perf_creatives pc using(creative_id)",
                            where => { 'bp.bid' => $old_bids_chunk, 'pc.creative_type' => $copy_perf_creative_types }
                        ]
                    );

                    if (@$old_banner_creatives) {
                        my $new_banner_creative_ids = JavaIntapi::GenerateObjectIds->new(object_type => 'banner_creative',
                                count => scalar @$old_banner_creatives)->call();
                        my (%banner_creative_id_old2new, %old_bid2new_banner_creative_id, %new_creative_modstatus, %old_bid2new_extracted_text_sql);
                        for my $idx (0 .. $#$old_banner_creatives) {
                            my $old_banner_creative = $old_banner_creatives->[$idx];
                            my $new_banner_creative_id = $new_banner_creative_ids->[$idx];
                            my $old_bid = $old_banner_creative->{bid};
                            #причины отклонения outdoor не копируем
                            $banner_creative_id_old2new{$old_banner_creative->{banner_creative_id}} = $new_banner_creative_id if !$bid_to_skip_moderation{$old_bid};
                            $old_bid2new_banner_creative_id{$old_bid} = $new_banner_creative_id;
                            # для креатива outdoor-а не копируем статус
                            if ($flags->{copy_moderate_status}) {
                                if ($bid_to_skip_moderation{$old_bid}) {
                                    $new_creative_modstatus{$old_bid} = $old_banner_creative->{statusModerate} ne 'New' ? 'Ready' : 'New';
                                } else {
                                    $new_creative_modstatus{$old_bid} = $old_banner_creative->{statusModerate} =~ /Sent|Sending/ ? 'Ready' : $old_banner_creative->{statusModerate};
                                }
                            } else {
                                $new_creative_modstatus{$old_bid} = 'New';
                            }
                            $old_bid2new_extracted_text_sql{$old_bid} = $bid_to_skip_moderation{$old_bid} ? 'NULL' : 'extracted_text';
                        }
                        my @copy_banners_performance_fields = qw/banner_creative_id cid pid bid creative_id statusModerate show_title_and_body/;
                        push @copy_banners_performance_fields, qw/extracted_text/  if $flags->{copy_moderate_status};

                        my ($banners_performance_fields_str, $banners_performance_values_str) = make_copy_sql_strings(
                            \@copy_banners_performance_fields,
                            {
                                banner_creative_id  => \%old_bid2new_banner_creative_id,
                                cid                 => $new_cid,
                                pid                 => \%oldbid2newpid,
                                bid                 => \%part_bid_old2new,
                                statusModerate      => \%new_creative_modstatus,
                                extracted_text__sql => sql_case(bid => \%old_bid2new_extracted_text_sql, dont_quote_value => 1),
                            },
                            by => 'bid',
                            override_fields_to_copy_only => 1,
                        );
                        do_insert_select_sql(PPC(cid => $old_cid),
                            "INSERT INTO banners_performance ($banners_performance_fields_str) values %s",
                            ["SELECT $banners_performance_values_str from banners_performance
                                LEFT JOIN perf_creatives USING(creative_id)", where => { bid => $old_bids_chunk }],
                            dbw => PPC(cid => $new_cid),
                        );
                        if ($flags->{copy_moderate_status}) {
                            my %banners_performance_override = (
                                id => \%banner_creative_id_old2new,
                                timeCreated => unix2mysql(time()),
                                statusSending => 'Yes',
                                ClientID => $client_id,
                                cid => $new_cid,
                            );
                            my ($fields, $values) = make_copy_sql_strings(\@mod_reasons_fields_to_copy, \%banners_performance_override, by => 'id');
                            do_insert_select_sql(PPC(cid => $old_cid), "INSERT INTO mod_reasons ($fields) VALUES %s",
                                ["SELECT $values FROM mod_reasons", WHERE => {id => [keys %banner_creative_id_old2new], type => $copy_perf_creative_types}],
                                dbw => PPC(cid => $new_cid));
                        }
                    }

                    # copy image_ads
                    my $old_image_ad_ids = get_all_sql(PPC(cid => $old_cid), [ "select image_id, bid from images", where => { bid => $old_bids_chunk } ]);
                    if (@$old_image_ad_ids) {
                        my $new_image_ad_ids = JavaIntapi::GenerateObjectIds->new(object_type => 'image',
                                count => scalar @$old_image_ad_ids)->call();
                        my (%image_id_old2new, %old_bid2new_image_ad_id);
                        for (my $idx = 0; $idx <= $#$old_image_ad_ids; $idx++) {
                            $image_id_old2new{$old_image_ad_ids->[$idx]->{image_id}} = $new_image_ad_ids->[$idx];
                            $old_bid2new_image_ad_id{$old_image_ad_ids->[$idx]->{bid}} = $new_image_ad_ids->[$idx];
                        }
                        my @copy_images_fields = qw/image_hash/;
                        # в реальности значения этих полей не копируются а заменяются переопределениями ниже по коду
                        push @copy_images_fields, qw/image_id cid pid bid/;

                        push @copy_images_fields, qw/image_text disclaimer_text statusModerate/ if $flags->{copy_moderate_status};

                        my ($images_fields_str, $images_values_str) = make_copy_sql_strings(
                            \@copy_images_fields,
                            {
                                image_id => \%old_bid2new_image_ad_id,
                                cid => $new_cid,
                                pid => \%oldbid2newpid,
                                bid => \%part_bid_old2new,
                                statusModerate__sql => "IF(statusModerate IN ('Sent', 'Sending'), 'Ready', statusModerate)",
                            },
                            by => 'bid',
                            override_fields_to_copy_only => 1,
                        );
                        do_insert_select_sql(PPC(cid => $old_cid),
                            "INSERT INTO images ($images_fields_str) values %s",
                            ["SELECT $images_values_str from images", where => { bid => $old_bids_chunk }],
                            dbw => PPC(cid => $new_cid),
                        );
                        # записи из banner_images_formats и banner_images_pool скопируются в mass_copy_banner_image
                        if ($flags->{copy_moderate_status}) {
                            my %image_override = (
                                id => \%image_id_old2new,
                                timeCreated => unix2mysql(time()),
                                statusSending => 'Yes',
                                ClientID => $client_id,
                                cid => $new_cid,
                            );
                            my ($fields, $values) = make_copy_sql_strings(\@mod_reasons_fields_to_copy, \%image_override, by => 'id');
                            do_insert_select_sql(PPC(cid => $old_cid), "INSERT INTO mod_reasons ($fields) VALUES %s", 
                                                                      ["SELECT $values FROM mod_reasons", WHERE => {id => [keys %image_id_old2new], type => ['image_ad']}],
                                                                      dbw => PPC(cid => $new_cid));
                        }
                    }

                    # copy audio/video
                    my $old_with_media_resources = get_one_column_sql(PPC(cid => $old_cid), ['SELECT bid FROM banner_resources', WHERE =>{bid => $old_bids_chunk }]);
                    if (@$old_with_media_resources){
                        my $bid_old2new_slice = hash_cut(\%part_bid_old2new, @$old_with_media_resources);

                        my $ids = get_new_id_multi('resource_id', scalar @$old_with_media_resources);
                        my %bid2resource_id = map {($_ => shift @$ids)} @$old_with_media_resources;
                        my ($resources_fields_str, $resources_values_str) = make_copy_sql_strings(
                            [qw/used_resources/],
                            {
                                bid => $bid_old2new_slice,
                                resource_id => \%bid2resource_id, 
                            },
                            by => 'bid',
                        );
                        do_insert_select_sql(
                                PPC(cid => $old_cid),
                                'INSERT INTO banner_resources ('.$resources_fields_str.') VALUES %s',
                                ['SELECT '.$resources_values_str.' FROM banner_resources', WHERE => {bid => [keys %$bid_old2new_slice]}],
                                dbw => PPC(cid => $new_cid),
                        );
                    }

                    # copy mobile content
                    if (camp_kind_in(type => $new_campaign->{type}, 'store_content')) {
                        my %bmc_override = (
                            bid => \%part_bid_old2new,
                        );
                        my ($bmc_fields_str, $bmc_values_str) = make_copy_sql_strings(
                            \@banners_mobile_content_fields_to_copy,
                            \%bmc_override,
                            by => 'bid',
                        );
                        do_insert_select_sql(PPC(cid => $old_cid),
                            "INSERT IGNORE INTO banners_mobile_content ($bmc_fields_str) VALUES %s",
                            ["SELECT $bmc_values_str FROM banners_mobile_content", WHERE => { bid => $old_bids_chunk }],
                            dbw => PPC(cid => $new_cid),
                        );
                    }

                    # copy content promotion
                    if ($new_campaign->{type} eq 'content_promotion') {
                        my %bcpv_override = (
                            bid => \%part_bid_old2new,
                        );
                        my ($bcpv_fields_str, $bcpv_values_str) = make_copy_sql_strings(
                            \@banners_content_promotion_video_fields_to_copy,
                            \%bcpv_override,
                            by => 'bid',
                        );
                        do_insert_select_sql(PPC(cid => $old_cid),
                            "INSERT IGNORE INTO banners_content_promotion_video ($bcpv_fields_str) VALUES %s",
                            ["SELECT $bcpv_values_str FROM banners_content_promotion_video", WHERE => { bid => $old_bids_chunk }],
                            dbw => PPC(cid => $new_cid),
                        );
                        my %bcp_override = (
                            bid => \%part_bid_old2new,
                        );
                        my ($bcp_fields_str, $bcp_values_str) = make_copy_sql_strings(
                            \@banners_content_promotion_fields_to_copy,
                            \%bcp_override,
                            by => 'bid',
                        );
                        do_insert_select_sql(PPC(cid => $old_cid),
                            "INSERT IGNORE INTO banners_content_promotion ($bcp_fields_str) VALUES %s",
                            ["SELECT $bcp_values_str FROM banners_content_promotion", WHERE => { bid => $old_bids_chunk }],
                            dbw => PPC(cid => $new_cid),
                        );
                        my %acp_override = (
                            pid => \%pid_old2new,
                        );
                        my ($acp_fields_str, $acp_values_str) = make_copy_sql_strings(
                            \@adgroups_content_promotion_fields_to_copy,
                            \%acp_override,
                            by => 'pid',
                        );
                        do_insert_select_sql(PPC(cid => $old_cid),
                            "INSERT IGNORE INTO adgroups_content_promotion ($acp_fields_str) VALUES %s",
                            ["SELECT $acp_values_str FROM adgroups_content_promotion", WHERE => { pid => $old_pids_chunk }],
                            dbw => PPC(cid => $new_cid),
                        );
                    }
    
                    # copy catalogia_banners_rubrics
                    my %catalogia_banners_rubrics_override = (
                        bid => \%part_bid_old2new,
                    );
                    my ($catalogia_banners_rubrics_fields_str, $catalogia_banners_rubrics_values_str) = make_copy_sql_strings(\@catalogia_banners_rubrics_fields_to_copy, \%catalogia_banners_rubrics_override, by => 'bid');
                    do_insert_select_sql(PPC(cid => $old_cid), "INSERT INTO catalogia_banners_rubrics ($catalogia_banners_rubrics_fields_str) VALUES %s", 
                                                              ["SELECT $catalogia_banners_rubrics_values_str FROM catalogia_banners_rubrics", where => {bid => $old_bids_chunk}],
                                                              dbw => PPC(cid => $new_cid));
    
    
                    # copy mod_reasons for banner, contactinfo, sitelinks_set, image
                    if ($flags->{copy_moderate_status}) {
                        my %mod_reasons_banners_override = (
                            id => \%copy_statuses_part_bid_old2new,
                            timeCreated => unix2mysql(time()),
                            statusSending => 'Yes',
                            ClientID => $client_id,
                            cid => $new_cid,
                        );
                        my ($mod_reasons_banners_fields_str, $mod_reasons_banners_values_str) = make_copy_sql_strings(\@mod_reasons_fields_to_copy, \%mod_reasons_banners_override, by => 'id');
                        do_insert_select_sql(PPC(cid => $old_cid), "INSERT INTO mod_reasons ($mod_reasons_banners_fields_str) VALUES %s", 
                                                                  ["SELECT $mod_reasons_banners_values_str FROM mod_reasons", WHERE => {id => [keys %copy_statuses_part_bid_old2new], type => [qw(banner contactinfo sitelinks_set image display_href callout turbolanding)]}],
                                                                  dbw => PPC(cid => $new_cid));
                    }
    
                    # copy banner_images
                    BannerImages::mass_copy_banner_image($old_cid, $new_cid, \%part_bid_old2new, %$flags, old_client_id => $old_client_id, client_id => $client_id,
                        image_ad_ids => $old_image_ad_ids);

                    # copy banners_minus_geo
                    if ($flags->{copy_moderate_status} && $old_chief_uid == $new_chief_uid) {
                        my ($bmg_fields_str, $bmg_values_str) = make_copy_sql_strings(
                            [qw/bid minus_geo/],
                            {
                                bid => \%copy_statuses_part_bid_old2new,
                            },
                            by => 'bid'
                        );
                        do_insert_select_sql(PPC(cid => $old_cid), "INSERT INTO banners_minus_geo ($bmg_fields_str) VALUES %s",
                            ["SELECT $bmg_values_str FROM banners_minus_geo", where => { bid => [keys %copy_statuses_part_bid_old2new], type => 'current' }],
                            dbw => PPC(cid => $new_cid)
                        );
                    }

                    # copy pixels
                    if (Campaign::is_cpm_campaign($new_campaign->{type})) {
                        my ($bp_fields_str, $bp_values_str) = make_copy_sql_strings([qw/pixel_id pixel_url/], { bid => \%part_bid_old2new }, by => 'bid');
                        do_insert_select_sql(PPC(cid => $old_cid),
                            "INSERT IGNORE INTO banner_pixels ($bp_fields_str) VALUES %s",
                            [
                                "SELECT $bp_values_str FROM banner_pixels",
                                WHERE => { bid => $old_bids_chunk },
                            ],
                            dbw => PPC(cid => $new_cid),
                        );
                    }

                    my ($measurers_fields, $measurers_values) = make_copy_sql_strings([qw/measurer_system params has_integration/], { bid => \%part_bid_old2new }, by => 'bid');
                    do_insert_select_sql(PPC(cid => $old_cid),
                        "INSERT IGNORE INTO banner_measurers ($measurers_fields) VALUES %s",
                        ["SELECT $measurers_values FROM banner_measurers", WHERE => { bid => $old_bids_chunk }],
                        dbw => PPC(cid => $new_cid),
                    );

                    # copy aggregator domains
                    my ($ad_fields_str, $ad_values_str) = make_copy_sql_strings(
                        [ 'pseudo_domain' ], # fields_to_copy
                        { bid => \%part_bid_old2new }, # override
                        by => 'bid',
                    );
                    do_insert_select_sql(PPC(cid => $old_cid),
                        "INSERT IGNORE INTO aggregator_domains ($ad_fields_str) VALUES %s",
                        ["SELECT $ad_values_str FROM aggregator_domains", WHERE => { bid => $old_bids_chunk }],
                        dbw => PPC(cid => $new_cid),
                    );

                }
            }
        }

        # и в banners и в phrases есть bid и pid. развязываем узел, отдельно записывая phrases.bid после создания обоих объектов
        # performance: т.к. айдишники заранее выделяем из метабазы, этого update'а можно избежать, если заранее раскидать связку старых и новых bid и pid
        my $bid_sql_case = sql_case('pid', \%newpid2newfirstbid);
        my @new_pids = values %pid_old2new;
        do_update_table(PPC(cid => $new_cid), 'phrases', {LastChange__dont_quote => 'LastChange', bid__dont_quote => $bid_sql_case},
                                               where => {pid => \@new_pids});

        # copy bids and bids_arc
        my %bids_override = (
            cid => $new_cid,
            pid => \%pid_old2new,
            %bids_price_override,
        );
        my %old_bids_id2new;
        my %new_groups_phrases;
        _copy_bids($old_cid, $new_cid, $old_pids_chunk, \@bids_tables_to_copy, \@bids_fields_to_copy, \%bids_override, \%old_bids_id2new,
                \%new_groups_phrases, $new_campaign->{currency}, $client_id);

        # copy bids_base
        my %bids_base_override = (
            cid => $new_cid,
            pid => \%pid_old2new,
            %bids_base_price_override,
        );
        my (undef, $bids_base_values_str) = make_copy_sql_strings(\@bids_base_fields_to_copy, \%bids_base_override, by => 'pid');
        for my $table (@bids_base_tables_to_copy) {
            my $bids_base = get_all_sql(
                PPC(cid => $old_cid),
                [
                    "SELECT bid_id, $bids_base_values_str FROM $table",
                    WHERE => {pid => $old_pids_chunk, cid => $old_cid, bid_type__ne => 'keyword',  _TEXT => 'NOT FIND_IN_SET("deleted", opts)'},
                    'ORDER BY pid, bid_id'
                ]
            );
            my $new_id_count = grep {!$old_bids_id2new{$_->{bid_id}}} @$bids_base;
            my $new_bids_ids = JavaIntapi::GenerateObjectIds->new(object_type => 'phrase',
                        count => $new_id_count)->call();
            my (@bids_base_fields, @bids_base_values, @bids_base_log_price);
            foreach my $bid (@$bids_base) {
                my $old_bid_id = delete $bid->{bid_id};
                # Т.к. у bids и bids_base общее простарнство id'шников - пишем пары id-id в тот же хеш что и bids
                # это позволит скопировать bids_href_params вместе с bids
                my $new_bid_id;
                if ($old_bids_id2new{$old_bid_id}) {
                    # новый id уже получили для bids - берём его
                    $new_bid_id = $old_bids_id2new{$old_bid_id};
                }
                else {
                    $new_bid_id = shift @$new_bids_ids;
                    $old_bids_id2new{$old_bid_id} = $new_bid_id;
                    push @{$new_groups_phrases{$bid->{pid}} //= []}, $old_bid_id;
                }

                $bid->{bid_id} = $new_bid_id;
                @bids_base_fields = keys %$bid unless @bids_base_fields;
                push @bids_base_values, [ map {$bid->{$_}} @bids_base_fields ];
                push @bids_base_log_price, {
                    ( map { $_ => $bid->{$_} } qw(cid pid price) ),
                    id => $bid->{bid_id},
                    price_ctx => $bid->{price_context},
                    type => 'insert1',
                    currency => $new_campaign->{currency},
                };
            }
            if (@bids_base_values) {
                my $bids_base_fields_str = join ', ', map { sql_quote_identifier($_) } @bids_base_fields;
                log_copy_camp("copy bids_base", $new_cid, {id => [map {$_->{bid_id}} @$bids_base]});
                LogTools::log_price(\@bids_base_log_price);
                do_mass_insert_sql(PPC(cid => $new_cid), "INSERT INTO bids_base ($bids_base_fields_str) VALUES %s", \@bids_base_values);
            }
        }

        # copy bids_href_params
        if ($flags->{copy_bids_href_params} && %old_bids_id2new) {
            my %bids_href_params_override = (
                id => \%old_bids_id2new,
                cid => $new_cid,
            );
            my ($bids_href_params_fields_str, $bids_href_params_values_str) = make_copy_sql_strings(\@bids_href_params_fields_to_copy, \%bids_href_params_override, by => 'id');
            do_insert_select_sql(PPC(cid => $old_cid), "INSERT INTO bids_href_params ($bids_href_params_fields_str) VALUES %s", 
                                                       ["SELECT $bids_href_params_values_str FROM bids_href_params", WHERE => {cid => $old_cid, id => [keys %old_bids_id2new]}],
                                                      dbw => PPC(cid => $new_cid));
        }

        my %old_pid2ret_cond_ids;
        my %ret_cond_ids_old2new;

        # copy bids_retargeting
        if ($flags->{copy_retargetings}) {
            # performance: сделать массовую версию copy_retargetings_between_groups. пока же считаем, что ретаргетинг встречается редко
            my $pids_with_retargeting = get_one_column_sql(PPC(cid => $old_cid), ['SELECT DISTINCT pid FROM bids_retargeting', WHERE => {pid => $old_pids_chunk}]) || [];
            for my $old_pid (@$pids_with_retargeting) {
                my $new_pid = $pid_old2new{$old_pid};
                my $new_first_bid = $oldpid2newfirstbid{$old_pid};
                my $ret_ids = Retargeting::copy_retargetings_between_groups(
                    new_cid => $new_cid
                    , old_cid => $old_cid
                    , new_pid => $new_pid
                    , old_pid => $old_pid
                    , new_bid => $new_first_bid    # TODO adgroup: удалить при переходе на bids_retargeting.pid
                    , copy_suspended_status => $flags->{copy_phrase_status}
                    , price_convert_rate => $O{price_convert_rate}
                    , new_currency => $new_campaign->{currency},
                    %price_constant_names,
                );
                hash_merge \%old_pid2ret_cond_ids, $ret_ids->{old_pid2ret_cond_ids};
                hash_merge \%ret_cond_ids_old2new, $ret_ids->{ret_cond_ids_old2new};
            }
        }

        my ($dyn_cond_ids_old2new, $old_gid2dyn_cond_ids);
        my ($perf_filter_ids_old2new, $old_gid2perf_filter_ids);
        # Копирование условий нацеливания (для динамической кампании)
        if ($campaign_type eq 'dynamic') {
            ($dyn_cond_ids_old2new, $old_gid2dyn_cond_ids) = Direct::DynamicConditions->copy(
                [adgroup_id => $old_pids_chunk, with_additional => 1],
                $old_client_id, $client_id,
                sub {
                    my ($dyn_cond) = @_;
                    $dyn_cond->adgroup_id($pid_old2new{$dyn_cond->adgroup_id});
                    if ($dyn_cond->price > 0) {
                        $dyn_cond->price(currency_price_rounding($dyn_cond->price * ($O{price_convert_rate} // 1), $new_campaign->{currency}));
                    }
                    if ($dyn_cond->price_context > 0) {
                        $dyn_cond->price_context(currency_price_rounding($dyn_cond->price_context * ($O{price_convert_rate} // 1), $new_campaign->{currency}));
                    }
                    $dyn_cond->is_suspended(0) unless $flags->{copy_phrase_status};
                    $dyn_cond->status_bs_synced('No');
                    return $dyn_cond;
                }, primary_key => 'dyn_id');
        } elsif ($campaign_type eq 'performance') {
            ($perf_filter_ids_old2new, $old_gid2perf_filter_ids) = Direct::PerformanceFilters->copy(
                [adgroup_id => $old_pids_chunk, with_additional => 1],
                $old_client_id, $client_id,
                sub {
                    my $filter = shift;
                    $filter->campaign_id($new_cid);
                    $filter->adgroup_id($pid_old2new{$filter->adgroup_id});
                    if ($filter->price_cpc > 0) {
                        $filter->price_cpc(
                            currency_price_rounding(
                                $filter->price_cpc * ($O{price_convert_rate} // 1),
                                $new_campaign->{currency}, min_const => 'MIN_CPC_CPA_PERFORMANCE'
                            )
                        );
                    } 
                    if ($filter->price_cpa > 0) {
                        $filter->price_cpa(
                            currency_price_rounding(
                                $filter->price_cpa * ($O{price_convert_rate} // 1),
                                $new_campaign->{currency}, min_const => 'MIN_CPC_CPA_PERFORMANCE'
                            )
                        );
                    }
                    $filter->is_suspended(0) unless $flags->{copy_phrase_status};
                    $filter->status_bs_synced('No');

                    return $filter;
                }, primary_key => 'perf_filter_id');
        }

        # добавляем маппинги для отчета по сокопированым объектам
        if ($response) {
            # добавляем к ответу маппинг старых pid в новые
            hash_merge $response->{pid_old2new}, \%pid_old2new;
            # добавляем в ответ маппинг старых bid в новые
            hash_merge $response->{bid_old2new}, \%bid_old2new;
            # добавляем в ответ соответствие фраз группе в старой кампании
            hash_merge $response->{new_groups_phrases}, \%new_groups_phrases;
            # добавляем в ответ маппинг старых фраз в новые
            hash_merge $response->{phrase_id_old2new}, \%old_bids_id2new;
            # добавляем в ответ соответствие старых ретаргетингов группе
            hash_merge $response->{old_pid2ret_cond_ids}, \%old_pid2ret_cond_ids;
            # добавляем в ответ маппинг старых ретаргетингов в новые
            hash_merge $response->{ret_cond_ids_old2new}, \%ret_cond_ids_old2new;
            # добавляем в ответ соответствие старых условий нацеливания группе (для динамической кампании)
            hash_merge $response->{old_gid2dyn_cond_ids}, $old_gid2dyn_cond_ids || {};
            # добавляем в ответ маппинг новых условий нацеливания в старые (для динамической кампании)
            hash_merge $response->{dyn_cond_ids_old2new}, $dyn_cond_ids_old2new || {};
            hash_merge $response->{old_gid2perf_filter_ids}, $old_gid2perf_filter_ids || {};
            hash_merge $response->{perf_filter_ids_old2new}, $perf_filter_ids_old2new || {};
        }

        # copy bids_phraseid_history
        my $old_OrderID = $old_camp_info->{OrderID};
        if ($flags->{copy_ctr} && %old_bids_id2new) {
            my $old_phrases = get_all_sql(PPC(cid => $old_cid), ['
                SELECT bi.pid
                     , bi.id, bi.PhraseID
                     , b.bid, b.BannerID
                     , bim.BannerID AS img_BannerID
                     , bph.phraseIdHistory
                FROM bids bi
                JOIN banners b ON bi.pid = b.pid
           LEFT JOIN banner_images bim ON bim.bid = b.bid
           LEFT JOIN bids_phraseid_history bph ON bi.cid = bph.cid AND bi.id = bph.id
            ', WHERE => {
                'bi.id' => [keys %old_bids_id2new],
                _TEXT => '(IFNULL(bph.phraseIdHistory, "") != "" OR (bi.PhraseID > 0 AND (b.BannerID > 0 OR bim.BannerID > 0)))',
            }]);

            if ($old_phrases && @$old_phrases) {
                my %newid2history;
                for my $old_phrase (@$old_phrases) {
                    my $new_id = $old_bids_id2new{ $old_phrase->{id} };
                    next unless $new_id;    # фраза могла добавиться после начала копирования, ломаться на этом не хочется

                    if ($old_phrase->{PhraseID} && ($old_phrase->{BannerID} || $old_phrase->{img_BannerID})) {
                        my $history = $newid2history{ $new_id } ||= {
                            new_cid => $new_cid,
                            OrderID => $old_OrderID,
                            GroupID => $old_phrase->{pid},
                            phrases => [],
                            banners => {},
                        };
                        push @{$history->{phrases}}, $old_phrase->{PhraseID};
                        my $new_bid = $bid_old2new{ $old_phrase->{bid} };
                        if ($old_phrase->{BannerID}) {
                            $history->{banners}->{$new_bid} = $old_phrase->{BannerID};
                        }
                        if ($old_phrase->{img_BannerID}) {
                            $history->{banners}->{ BS::History::get_img_bid($new_bid) } = $old_phrase->{img_BannerID};
                        }
                    } elsif ($old_phrase->{phraseIdHistory}) {
                        my %parsed_history = BS::History::parse_bids_history($old_phrase->{phraseIdHistory});
                        $newid2history{ $new_id } = hash_cut \%parsed_history, qw(OrderID GroupID phrases);
                        $newid2history{ $new_id }->{banners} = hash_kmap { $bid_old2new{ $_[0] =~ s/^im//r } } $parsed_history{banners};
                    }
                }
                if (%newid2history) {
                    my @insert_phraseid_hist;
                    my %new_bids_id2old = reverse %old_bids_id2new;
                    while (my ($new_id, $history) = each %newid2history) {
                        my $serialized_history = BS::History::serialize_bids_history($history);
                        push @insert_phraseid_hist, [$new_cid, $new_id, $serialized_history];
                        log_copy_camp("copy bid history", $new_cid,
                            hash_merge({ new_id => $new_id, old_id => $new_bids_id2old{$new_id} }, $history)
                        );
                    }
                    do_mass_insert_sql(PPC(cid => $new_cid), "INSERT INTO bids_phraseid_history (cid, id, phraseIdHistory) VALUES %s", \@insert_phraseid_hist);
                }
            }
        }

        if ($flags->{copy_moderate_status} && ($old_camp_info->{statusModerate} ne 'New')) {
            # resend to moderation objects with Sending/Sent statuses
            send_banners_to_moderate([values %bid_old2new], {moderate_whole_group => 1,
                                                             filter_by_status_moderate => [qw/Sending Sent/]});
        }

        # copy tag_group
        Tag::copy_campaign_tags($old_cid, $new_cid, \%pid_old2new);
    }

    #media_groups with media_banners
    if (is_media_camp(type => $new_campaign->{type})) {
        my $mgids = get_one_column_sql(PPC(cid => $old_cid),
            ["select distinct(mgid) from media_banners b join media_groups g using(mgid)",
            WHERE => {'g.cid' => $old_cid, %$banners_condition}]);
        for my $mgid (@$mgids) {
            # clone media_groups
            my $new_mgid = get_new_id('pid', ClientID => $client_id);
            do_insert_select_sql(PPC(cid => $old_cid), "INSERT INTO media_groups (mgid, cid, geo, mcb_theme_id, mcbPhrasesChanged, categories, last_month_shows) VALUES %s", 
                                  "SELECT ?, ?, geo, mcb_theme_id, mcbPhrasesChanged, categories, last_month_shows FROM media_groups where mgid = ?",
                                  binds => [$new_mgid, $new_cid, $mgid],
                                  dbw => PPC(cid => $new_cid),
                );

            my @phrase_fields = qw/phrase norm_hash month_shows/;
            my $phrase_fields_sql = sql_fields(@phrase_fields);
            my $mcb_phrases = get_all_sql(PPC(cid => $old_cid), "SELECT $phrase_fields_sql FROM mcb_phrases where mgid = ?", $mgid);
            my $new_phrases_ids = get_new_id_multi('mcb_phrases_id', scalar(@$mcb_phrases));

            my @phrases_rows = pairwise 
                                    { [$a, $new_mgid, @{$b}{@phrase_fields}] }
                                    @$new_phrases_ids, @$mcb_phrases;
            do_mass_insert_sql(PPC(ClientID => $client_id), "INSERT INTO mcb_phrases (id, mgid, $phrase_fields_sql) VALUES %s", \@phrases_rows);

            # clone media_banner
            my @mb_fields = qw/alt href domain reverse_domain format_id md5_flash name_flash md5_picture name_picture name flash_href counter_href/;
            my $mb_fields_sql = join ', ', @mb_fields;
            my $old_media_banners = get_all_sql(PPC(cid => $old_cid),
                ["SELECT $mb_fields_sql FROM media_banners b",
                WHERE => {'b.mgid' => $mgid, %$banners_condition}]);
            next if !@$old_media_banners;
            my $new_mbids = get_new_id_multi('mbid', scalar(@$old_media_banners), ClientID => $client_id);

            ## здесь тоже pairwise, см. комментарий выше
            ## no critic (Freenode::DollarAB)
            my @rows = pairwise 
                            { [$a, $new_mgid, @{$b}{@mb_fields}] }
                            @$new_mbids, @$old_media_banners;
            do_mass_insert_sql(PPC(cid => $new_cid), "INSERT INTO media_banners (mbid, mgid, $mb_fields_sql) VALUES %s", \@rows);
        }
    }
  
    my $error = Campaign::can_create_camp(
      client_chief_uid => $new_chief_uid,
      ( $agency_uid  ? ( agency_uid  => $agency_uid  ) : () ),
      ( $manager_uid ? ( manager_uid => $manager_uid ) : () ),
    );
    # Если падает с ошибкой NOT_ENOUGH_FREEDOM при копировании на агентского субклиента, то
    # нужно проверить не менялся ли у субклиента представитель (см. https://st.yandex-team.ru/DIRECT-91627)
    die "Can\t create camp: $error" if $error;

    Client::on_campaign_creation($client_id, $agency_uid, $manager_uid);

    if ($agency_uid) {
        $agency_id ||= PrimitivesIds::get_clientid(uid => $agency_uid);
    }

    my $new_wallet_cid = 0;
    if (defined $O{new_wallet_cid} && camp_kind_in(type => $old_camp_info->{type}, 'web_edit_base')) {
        $new_wallet_cid = $O{new_wallet_cid};
    } elsif (camp_kind_in(type => $old_camp_info->{type}, 'web_edit_base')) {
        my $new_currency = $new_campaign->{currency};
        my $wallet_camp = get_wallet_camp($new_chief_uid, $agency_id, $new_currency);
        # is_enabled будет 0 если клиент попал в эксперимент оплаты до модерации и не успел создать ни одной кампании
        # поэтому в этом случае привязываем скопированную кампанию на существующий кошелек даже если он is_enabled=0
        $new_wallet_cid = $wallet_camp->{wallet_cid} && ($wallet_camp->{is_enabled} || $client_info->{feature_payment_before_moderation})
                         ? $wallet_camp->{wallet_cid}
                         : 0;
    }

    do_update_table(PPC(cid => $new_cid), "campaigns"
                       , {ManagerUID => $manager_uid
                          , AgencyUID => $agency_uid
                          , AgencyID => $agency_id
                          , wallet_cid => $new_wallet_cid
                          , statusBsSynced => 'No'
                         }
                       , where => {cid => $new_cid});

    if ($new_strategy_id) {
        do_update_table(PPC(strategy_id => $new_strategy_id), "strategies"
                           , {wallet_cid => $new_wallet_cid}
                           , where => {strategy_id => $new_strategy_id});
    }
                       
    if ($flags->{stop_copied_campaigns}) {
        Campaign::update_campaigns_for_stop($new_cid);
    }

    # во время конвертации клиента в валюту не получается создавать биллинговые агрегаты:
    # * новый общий счет создан не полностью (нет записи в wallet_campaigns)
    # * взят MySQL лок
    # а т.к. клиентов скоро перестанем конвертировать, просто не делаем биллинговые агрегаты
    # в этом случае
    if (!camp_kind_in(type => $old_camp_info->{type}, 'without_billing_aggregates') && !$O{converting_client_currency}) {
        BillingAggregateTools::on_save_new_camp($client_id, $new_cid, $UID);
    }

    do_in_transaction {
        if (!$O{keep_empty}) {
            do_update_table(PPC(cid => $new_cid), "campaigns", { statusEmpty => 'No' }, where => { cid => $new_cid });
        }
    };

    log_copy_camp("FINISH", $new_cid, {});

    return $new_cid;
}

sub client_has_access_to_goals {
    my ($goals, $client_id) = @_;
    my $client_goal_ids = MetrikaCounters::get_client_goal_ids(
            $client_id,
            goal_ids_for_check_ecom => $goals,
            extra_goal_ids => [map { $_ } keys %Stat::Const::MOBILE_APP_SPECIAL_GOALS]
        );
    
    my %client_goals_hash = map {$_ => 1} @$client_goal_ids;
    
    for my $goal (@$goals) {
        unless (exists($client_goals_hash{$goal})) {
            return 0;
        }
    }
    return 1;
}

sub get_campaign_meaningful_goal_ids {
    my ($cid) = @_;
    my @meaningful_goals = @{get_one_column_sql(PPC(cid => $cid), [ "
        SELECT co.meaningful_goals
          FROM camp_options co",
        WHERE => { cid => $cid } ])};
    my @meaningful_goal_ids;
    for my $m_goals (@meaningful_goals) {
        next unless $m_goals;
        push(@meaningful_goal_ids, map {$_->{goal_id}} @{decode_json($m_goals)});
    }
    return \@meaningful_goal_ids;
}

=head2 _copy_ppc_domains($domain_ids, $dbh1, $dbh2)

Скопировать запись ppc.domains для выбранных domain_ids из dbh1 в dbh2

=cut

sub _copy_ppc_domains {
    my ($domain_ids, $dbh1, $dbh2) = @_;
    return unless ($domain_ids && @$domain_ids);
    my @domains_fields_to_copy = qw(
        domain_id
        domain
        reverse_domain
    );
    my ($dom_fields_str, $dom_values_str) = make_copy_sql_strings(\@domains_fields_to_copy);
    do_insert_select_sql($dbh1,
        "INSERT IGNORE INTO domains ($dom_fields_str) VALUES %s",
        ["SELECT $dom_values_str FROM domains", WHERE => {domain_id => $domain_ids}],
        dbw => $dbh2,
    );
}

=head2 find_same_mobile_contents($old_client_id, $client_id, $old_mobile_content_ids)

    Найти у клиента $client_id записи в mobile_content идентичные записям с
    идентификаторами $old_mobile_content_ids у клиента $old_client_id

    Для идентификаторов, которым не соотвествуют ни один mobile_content у $client_id,
    записей в результате не будет

=cut
sub find_same_mobile_contents {
    my ($old_client_id, $client_id, $old_mobile_content_ids) = @_;

    my $old_keys = get_all_sql(PPC(ClientID => $old_client_id), [
            'SELECT mobile_content_id, `store_content_id`,`store_country`,`os_type`,`content_type`
            FROM mobile_content',
            WHERE => { mobile_content_id => $old_mobile_content_ids },
        ]);
    my $old_to_new_mobile_content_ids = get_hash_sql(PPC(ClientID => $client_id), [
            'SELECT CASE',
            (map {(
                    WHEN => hash_cut($_, qw/store_content_id store_country os_type content_type/),
                    THEN => $_->{mobile_content_id},
                )} @$old_keys),
            'END, mobile_content_id
            FROM mobile_content',
            WHERE => {
                ClientID => $client_id,
                _OR => [
                    map {( _AND => hash_cut($_, qw/store_content_id store_country os_type content_type/)
                    )} @$old_keys
                ],
            },
        ]);
    return $old_to_new_mobile_content_ids;
}

=head2 copy_mobile_contents($old_client_id, $client_id, $old_mobile_content_ids)

    Создать у клиента $client_id записи в mobile_content идентичные записям с
    идентификаторами $old_mobile_content_ids у клиента $old_client_id

=cut
sub copy_mobile_contents {
    my ($old_client_id, $client_id, $old_mobile_content_ids, $is_intershard_copy, $flags) = @_;

    if ($is_intershard_copy) {
        my $missed_domains = get_one_column_sql(PPC(ClientID => $old_client_id), [
                'SELECT DISTINCT publisher_domain_id FROM mobile_content',
                WHERE => {
                    mobile_content_id => $old_mobile_content_ids,
                    publisher_domain_id__is_not_null => 1,
                },
            ]);
        _copy_ppc_domains($missed_domains, PPC(ClientID => $old_client_id), PPC(ClientID => $client_id));
    }

    my $old_mobile_content_ids_cnt = scalar @$old_mobile_content_ids;

    my $new_mobile_content_ids = get_new_id_multi('mobile_content_id', $old_mobile_content_ids_cnt);
    my $old_to_new_mobile_content_ids = {map {($old_mobile_content_ids->[$_] => $new_mobile_content_ids->[$_])} (0 .. ($old_mobile_content_ids_cnt - 1))};

    my @mobile_content_fields_to_copy = qw(
        store_content_id
        store_country
        os_type
        content_type
        bundle_id
        is_available
        name
        prices_json
        rating
        rating_votes
        icon_hash
        min_os_version
        app_size_bytes
        available_actions
        publisher_domain_id
        genre
        age_label
    );
    # в реальности значения этих полей не копируются а заменяются переопределениями ниже по коду
    push @mobile_content_fields_to_copy, qw/
        mobile_content_id
        ClientID
        statusBsSynced
    /;
    if ($flags->{copy_moderate_status}) {
        push @mobile_content_fields_to_copy, qw/statusIconModerate/;
    }

    my %mc_override = (
        mobile_content_id => $old_to_new_mobile_content_ids,
        ClientID => $client_id,
        statusBsSynced => 'No',
        statusIconModerate => 'Ready',
    );
    my ($mc_fields_str, $mc_values_str) = make_copy_sql_strings(
        \@mobile_content_fields_to_copy,
        \%mc_override,
        by => 'mobile_content_id',
        override_fields_to_copy_only => 1,
    );
    do_insert_select_sql(PPC(ClientID => $old_client_id),
        "INSERT INTO mobile_content ($mc_fields_str) VALUES %s",
        ["SELECT $mc_values_str FROM mobile_content", WHERE => {mobile_content_id => $old_mobile_content_ids}],
        dbw => PPC(ClientID => $client_id),
    );
    return $old_to_new_mobile_content_ids;
}



=head2 get_mobile_content_id_by_mobile_app_id($client_id, $mobile_app_id)

    получить идентификатор mobile_content_id в приложении $mobile_app_id клиента $client_id

=cut
sub get_mobile_content_id_by_mobile_app_id {
    my ($client_id, $mobile_app_id) = @_;
    return get_one_field_sql(PPC(ClientID => $client_id), [
        "SELECT mobile_content_id FROM mobile_apps",
        WHERE => {ClientID => $client_id, mobile_app_id => $mobile_app_id}
    ]);
}


=head2 find_same_mobile_app($old_client_id, $client_id, $old_mobile_app_id)

    найти на клиенте $client_id приложение с такими же данными

=cut
sub find_same_mobile_app {
    my ($old_client_id, $client_id, $old_mobile_app_id) = @_;

    # TODO: эта конструкция сломается, если в mobile_app_trackers станет можно больше одного трекера на приложение
    my $old_key = get_one_line_sql(PPC(ClientID => $old_client_id), [
        "SELECT a.store_href, a.domain_id, t.url
        FROM mobile_apps a
        LEFT JOIN mobile_app_trackers t USING(mobile_app_id)",
        WHERE => {
            'a.ClientID' => $old_client_id,
            'a.mobile_app_id' => $old_mobile_app_id,
        },
        'ORDER BY t.mobile_app_tracker_id',
        LIMIT => 1
    ]);

    my $where = {
        'a.ClientID' => $client_id,
        'a.store_href' => $old_key->{store_href},
    };

    if (defined $old_key->{domain_id}) {
        $where->{'a.domain_id'} = $old_key->{domain_id};
    } else {
        $where->{'a.domain_id__is_null'} = 1;
    }

    if (defined $old_key->{url}) {
        $where->{'t.url'} = $old_key->{url};
    } else {
        $where->{'t.url__is_null'} = 1;
    }

    return get_one_field_sql(PPC(ClientID => $client_id), [
        "SELECT a.mobile_app_id
        FROM mobile_apps a
        LEFT JOIN mobile_app_trackers t USING(mobile_app_id)",
        WHERE => $where,
        'ORDER BY a.mobile_app_id',
        LIMIT => 1
    ]);

}

=head2 copy_mobile_app($old_client_id, $client_id, $old_mobile_app_id, $is_intershard_copy, $flags)

    скопировать клиенту $client_id приложение $old_mobile_app_id от $old_client_id

=cut
sub copy_mobile_app {
    my ($old_client_id, $client_id, $old_mobile_app_id, $is_intershard_copy, $flags) = @_;

    my $data = get_one_line_sql(PPC(ClientID => $old_client_id), [
        "SELECT mobile_content_id, domain_id
        FROM mobile_apps",
        WHERE => {mobile_app_id => $old_mobile_app_id},
    ]);

    # скопировать домен
    if ($is_intershard_copy && $data->{domain_id}) {
        _copy_ppc_domains([$data->{domain_id}], PPC(ClientID => $old_client_id), PPC(ClientID => $client_id));
    }

    # скопировать mobile_content
    my $old_to_new_mobile_content = find_same_mobile_contents($old_client_id, $client_id, [$data->{mobile_content_id}]);
    if (!exists $old_to_new_mobile_content->{$data->{mobile_content_id}}) {
        $old_to_new_mobile_content = copy_mobile_contents($old_client_id, $client_id, [$data->{mobile_content_id}], $is_intershard_copy, $flags);
    }
    my $new_mobile_content_id = $old_to_new_mobile_content->{$data->{mobile_content_id}};

    # скопировать приложение
    my $new_mobile_app_id = get_new_id('mobile_app_id');

    my @fields = qw(
        mobile_app_id
        ClientID
        mobile_content_id
        store_type
        store_href
        name
        device_type_targeting
        network_targeting
        min_os_version
        displayed_attributes
        primary_action
        domain_id
    );
    my $override = {
        mobile_app_id => $new_mobile_app_id,
        ClientID => $client_id,
        mobile_content_id => $new_mobile_content_id,
    };

    my ($fields_str, $values_str) = make_copy_sql_strings(\@fields, $override, by => 'mobile_app_id');
    do_insert_select_sql(PPC(ClientID => $old_client_id),
        "INSERT INTO mobile_apps ($fields_str) VALUES %s",
        ["SELECT $values_str FROM mobile_apps", where => {mobile_app_id => $old_mobile_app_id}],
        dbw => PPC(ClientID => $client_id)
    );

    # скопировать трекеры
    my $old_tracker_ids = get_one_column_sql(PPC(ClientID => $old_client_id), [
        "SELECT mobile_app_tracker_id from mobile_app_trackers",
        WHERE => { mobile_app_id => $old_mobile_app_id },
    ]);
    my $old_tracker_cnt = scalar @$old_tracker_ids;
    my $new_tracker_ids = get_new_id_multi('mobile_app_tracker_id', $old_tracker_cnt);

    my %tracker_ids_old_to_new = zip @$old_tracker_ids, @$new_tracker_ids;

    @fields = qw(
        mobile_app_tracker_id
        ClientID
        mobile_app_id
        tracking_system
        tracker_id
        url
        user_params
    );
    $override = {
        mobile_app_tracker_id => \%tracker_ids_old_to_new,
        mobile_app_id => $new_mobile_app_id,
        ClientID => $client_id,
    };

    ($fields_str, $values_str) = make_copy_sql_strings(\@fields, $override, by => 'mobile_app_tracker_id');
    do_insert_select_sql(PPC(ClientID => $old_client_id),
        "INSERT INTO mobile_app_trackers ($fields_str) VALUES %s",
        ["SELECT $values_str FROM mobile_app_trackers", where => {mobile_app_tracker_id => $old_tracker_ids}],
        dbw => PPC(ClientID => $client_id)
    );

    return $new_mobile_app_id;
}

=head2 _copy_bids

Скопировать bids и bids_arc (опционально), заполнить хеши соответствия id.

=cut

sub _copy_bids {
    my ($old_cid, $new_cid, $old_pids_chunk, $bids_tables_to_copy, $bids_fields_to_copy, $bids_override, $old_bids_id2new, $new_groups_phrases, $currency, $client_id) = @_;
    my %pid_remoderate_triggers;
    my (undef, $bids_values_str) = make_copy_sql_strings($bids_fields_to_copy, $bids_override, by => 'pid');

    for my $table (@$bids_tables_to_copy) {
        my $bids = get_all_sql(PPC(cid => $old_cid), ["SELECT id, $bids_values_str FROM $table", WHERE => {pid => $old_pids_chunk, cid => $old_cid}, 'ORDER BY pid, id']);
        my $new_phids = JavaIntapi::GenerateObjectIds->new(object_type => 'phrase',
                        count => scalar(@$bids))->call();
        my (@bids_fields, @bids_values, @bids_log_price);
        foreach my $ph (@$bids) {
            if ($ph->{statusModerate} && ($ph->{statusModerate} eq 'New') && !$pid_remoderate_triggers{$ph->{pid}}) {
                $pid_remoderate_triggers{$ph->{pid}} = $ph->{id};
            }
            $new_groups_phrases->{$ph->{pid}} = [] unless exists $new_groups_phrases->{$ph->{pid}};
            push @{$new_groups_phrases->{$ph->{pid}}}, $ph->{id};
            my $old_bids_id = delete $ph->{id};
            my $new_bids_id = shift @$new_phids;
            $old_bids_id2new->{$old_bids_id} = $new_bids_id;
            $ph->{id} = $new_bids_id;
            @bids_fields = keys %$ph unless @bids_fields;
            push @bids_values, [map {$ph->{$_}} @bids_fields];
            push @bids_log_price, {
                (map { $_ => $ph->{$_} } qw(cid pid bid id price)),
                price_ctx => $ph->{price_context},
                type => 'insert1',
                currency => $currency,
            };
        }
        if (@bids_values) {
            my $bids_fields_str = join ', ', map { sql_quote_identifier($_) } @bids_fields;
            log_copy_camp("copy bids", $new_cid, {id => [map {$_->{id}} @$bids]});
            LogTools::log_price(\@bids_log_price);
            do_mass_insert_sql(PPC(cid => $new_cid), "INSERT INTO bids ($bids_fields_str) VALUES %s", \@bids_values);
        }
    }

    # перемодерировать группы в нецелостном состоянии модерации
    if (%pid_remoderate_triggers) {
        _remoderate_inconsistent_adgroups($new_cid, \%pid_remoderate_triggers);
    }
}

=head2 _remoderate_inconsistent_adgroups

Переотправить на модерацию промодерированные группы с новыми фразами.

=cut

sub _remoderate_inconsistent_adgroups {
    my ($new_cid, $pid_remoderate_triggers) = @_;
    # явно выберем id групп, чтобы записать их в лог
    my $pids_to_resend = get_one_column_sql(PPC(cid => $new_cid),
        ["SELECT pid FROM phrases", where => {pid => [keys %$pid_remoderate_triggers], statusModerate__not_in => ['New', 'Ready']}]);
    if (@$pids_to_resend) {
        my $log_struct = [map {"pid $_, triggered by bid " . $pid_remoderate_triggers->{$_}} @$pids_to_resend];
        LogTools::log_messages("resend adgroups (phrases)", $log_struct);
        do_update_table(PPC(cid => $new_cid), 'phrases', {statusModerate => 'Ready'},
            where => {pid => $pids_to_resend, statusModerate__not_in => ['New', 'Ready']});
    }
}

=head2 _remove_from_opts

Удалить из поля campaign->opts опцию, которую не нужно копировать.

=cut

sub _remove_from_opts
{
    my ($opts, $excluded_name) = @_;

    return join(',', grep {defined $_ && lc($_) ne $excluded_name} split(/\,\s*/, $opts));
}

=head2 log_copy_camp

=cut

sub log_copy_camp
{
    my ($action, $cid, $values) = @_;

    state $log_syslog = Yandex::Log->new(use_syslog => 1, no_log => 1, syslog_prefix => 'COPY_CAMP', log_file_name => 'copy_camp.log');
    
    my $log_data = {
        action => $action,
        cid => $cid,
        values => Tools::trim_log_data($values),
    };

    eval { $log_syslog->out($log_data); 1 }
    or do { print STDERR "failed to syslog: $@\n" };

    return;
}

1;
