package Campaign::CopyWithinClient;

=head1 NAME

CopyClient

=head1 DESCRIPTION

Временный модуль для копирования кампаний внутри клиента, на момент переезда копирования на java.

=cut

use Direct::Modern;

use List::Util qw/sum/;
use Yandex::DBShards;
use Yandex::DBTools;
use Yandex::I18n;
use POSIX qw/ceil/;

use Campaign;
use Client::ConvertToRealMoney;
use Direct::Validation::Campaigns;
use Settings;

=head2 copy_camps

    Добавление задачи копирования кампаний внутри клиента в очередь. Предварительно валидирует, что кампанию можно копировать.

    Принимает именованые параметры:
        sync => 1|0 - копировать синхронно, без очереди копирования

    Возвращает
        result => 1|0 - успешно поставлена задача в очередь (либо успешно скопирована кампания, при sync => 1)

        result_time_estimate => 3.6 - ожидаемое время в очереди в минутах при успехе (при sync => 0)
        cid => ... - id новой кампании (при sync => 1)

        status => 'error' - при неуспехе
        validation_error - текстовое описание ошибок, при неуспешной валидации
        copy_error - текстовое описание ошибок, при неуспешном копировании (при sync => 1)

=cut

sub copy_camps {
    my ($cids, $operator_uid, $client_id, %O) = @_;

    my ($override, $campaign_name_prefix) = ({campaigns => {statusShow => 'No'}}, iget("(копия)") . " ");

    my @campaigns;
    my $result;

    my $rows = get_all_sql(PPC(cid => $cids), [
                "select c.cid, c.ManagerUID, c.AgencyUID, c.uid, c.ClientID, c.archived, c.type, q.cid AS is_in_camp_operations_queue,
                     qc.cid AS is_in_camp_operations_queue_copy,
                     exists (select bid from banners b where b.cid = c.cid) as has_banners
                from campaigns c
                left join camp_operations_queue q using (cid)
                left join (select cid from camp_operations_queue_copy",
                    where => { "cid" => SHARD_IDS },
                    " GROUP BY cid) AS qc on qc.cid = c.cid",
                where => { "c.cid" => SHARD_IDS } ]);

    for my $row (@$rows) {
        $row->{is_in_camp_operations_queue} = $row->{is_in_camp_operations_queue} ? 1 : 0;
        $row->{is_in_camp_operations_queue_copy} = $row->{is_in_camp_operations_queue_copy} ? 1 : 0;
        my $model = Direct::Model::Campaign->from_db_hash($row, \{}, with => ["CampQueue", "HasBanners"]);
        push @campaigns, $model;
    }

    my $perminfo = Rbac::get_perminfo(ClientID => $client_id);

    my $vr = validate_copy_campaigns_for_client(\@campaigns, $client_id, $perminfo);
    if (!$vr->is_valid) {
        return {
            result => 0,
            status => 'error',
            validation_error => join("\n" => @{ $vr->get_error_descriptions }),
        };
    }

    my %flags = (
        copy_agency           => 1,
        copy_manager          => 1,
        copy_stopped          => 1,
        copy_archived         => 1,
        copy_moderate_status  => 1,
        copy_phrase_status    => 1,
        copy_retargetings     => 1,
        copy_bids_href_params => 1,
    );

    if ($O{sync}) {
        if (scalar @$cids != 1) {
            return { result => 0, status => 'error', validation_error => 'Only one campaign can be copied synchronously at a time' };
        }

        my $rbac = RBAC2::Extended->get_singleton($operator_uid);

        my $cid = $cids->[0];

        my $new_cid = eval { 
            Client::ConvertToRealMoney::copy_camp_converting_currency(
                $rbac, 
                $cid,
                $perminfo->{chief_uid},
                undef,
                undef,
                flags => \%flags,
                override => $override,
                campaign_name_prefix => $campaign_name_prefix,
                UID => $operator_uid,
            ); 
        };

        if (defined $new_cid) {
            return { result => 1, cid => $new_cid + 0 };
        } else {
            return { result => 0, status => 'error', copy_error => "Failed to copy campaign: $@" };
        }
    } else {
        my $queue_camp_params = {
            UID     => $operator_uid,
            new_uid => $perminfo->{chief_uid},
            flags   => \%flags,
            override => $override,
            campaign_name_prefix => $campaign_name_prefix
        };

        queue_camp_operation('copy' => $cids, params => $queue_camp_params);

        # оцениваем время на копирование кампаний, уже стоящих в очереди, и только что добавленных кампаний (по 0.1 минуты на кампанию)
        my $time_estimate = 0.1 * (sum 0, @{ get_one_column_sql(PPC(cid => $cids),
                'SELECT COUNT(*) FROM (
                select cid from camp_operations_queue
                    UNION
                    select cid from camp_operations_queue_copy
                ) q') || [ ] });

        return {
            result               => 1,
            result_time_estimate => ceil($time_estimate)
        };
    }
}

1;
