package Campaign::Types;

=head2 NAME

    Campaign::Types - информация о типах и "надтипах" кампаний

=cut

use Direct::Modern;

use Carp qw/croak/;
use Readonly;

use Yandex::HashUtils;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::Validate;

use Settings;
use RbacSimpleCache;

use parent qw/Exporter/;
our @EXPORT = qw/
    camp_kind_in
    get_camp_kind_types
    get_camp_kind_types_and
    expand_camp_type

    camp_type_in
    get_camp_type
    is_media_camp
    is_wallet_camp

    get_camp_supported_adgroup_types
    get_camp_supported_adgroup_types_with_geo
/;
our @EXPORT_OK = qw/
    get_camp_type_multi
    camp_type_to_param
/;


=head2 %TYPES

    хэш из существующих типов кампаний

=cut

our %TYPES = map {$_ => 1} qw/text mcb geo wallet dynamic mobile_content performance mcbanner cpm_banner cpm_deals billing_aggregate internal_distrib internal_free internal_autobudget cpm_yndx_frontpage content_promotion cpm_price/;

=head2 %DEFAULT_PRODUCT_TYPES_BY_CAMP_TYPE

    Соответствие типа кампании типу продукта в Биллинге.
    При работе с биллинговыми агрегатами, этот тип продукта определяет, на какой
    из агрегатов будут записываться открутки по умолчанию.

=cut
Readonly my %DEFAULT_PRODUCT_TYPES_BY_CAMP_TYPE => (
    (map { $_ => 'text' } qw/text wallet dynamic mobile_content performance mcbanner/),
    (map { $_ => $_ } qw/cpm_banner cpm_deals internal_autobudget internal_distrib internal_free cpm_yndx_frontpage content_promotion cpm_price/),
);


=head2 %KINDS

    Хэш из "надтипов" и соответствующих им типов

=cut
our %KINDS = (
    # все типы кампаний
    'all' => {map {$_ => 1} qw/text mcb geo wallet dynamic mobile_content performance mcbanner cpm_banner cpm_deals billing_aggregate
        internal_distrib internal_free internal_autobudget cpm_yndx_frontpage content_promotion cpm_price/},

    # "стандартная" кампания - c adgroup's в phrases и banners, к ней применяется большинство стратегий
    'base' => {map {$_ => 1} qw/text geo dynamic mobile_content performance mcbanner cpm_banner cpm_deals internal_distrib internal_free
        internal_autobudget cpm_yndx_frontpage content_promotion cpm_price/},

    # типы кампаний, доступные для редактирования в web-интерфейсе
    'web_edit' => {map {$_ => 1} qw/text mcb dynamic mobile_content performance mcbanner cpm_banner cpm_deals internal_distrib internal_free
        internal_autobudget cpm_yndx_frontpage content_promotion cpm_price/},

    # типы кампаний, доступные для редактирования в web-интерфейсе, базового типа
    'web_edit_base' => {map {$_ => 1} qw/text dynamic mobile_content performance mcbanner cpm_banner cpm_deals internal_distrib internal_free
        internal_autobudget cpm_yndx_frontpage content_promotion cpm_price/},

    # типы кампаний, которые не имеют полной поддержки в старом интерфейсе
    'old_web_no_support' => {map {$_ => 1} qw/cpm_price/},

    # типы кампаний, у которых может быть валюта
    'with_currency' => {map {$_ => 1} qw/text dynamic mobile_content wallet performance mcbanner cpm_banner cpm_deals billing_aggregate
        internal_distrib internal_free internal_autobudget cpm_yndx_frontpage content_promotion cpm_price/},

    # типы кампаний, доступные для просмотра в web-интерфейсе супер-пользователю
    'web_view_super' => {map {$_ => 1} qw/text mcb geo dynamic mobile_content performance mcbanner cpm_banner cpm_deals internal_distrib
        internal_free internal_autobudget cpm_yndx_frontpage content_promotion cpm_price/},

    # типы кампаний, которые можно редактировать обычному клиенту в АПИ
    'api_edit' => {map {$_ => 1} qw/text/},

    # типы кампаний, которые можно редактировать клиенту в АПИ, которому разрешено работать с кампаниями типа geo
    'api_edit_geo' => {map {$_ => 1} qw/text geo/},

    # типы кампаний, по которым можно запрашивать отчёт по ДТО
    'dynamic_text_ads_report' => {map {$_ => 1} qw/dynamic/},
    # типы кампаний, по которым можно запрашивать «медленный» отчёт АПИ
    'api_stat' => {map {$_ => 1} qw/text geo dynamic mobile_content performance/}, # TODO DIRECT-67003 +mcbanner?
    # типы кампаний, в которых можно применять фильтр по фразам в АПИ
    'api_stat_phrase_filter' => {map {$_ => 1} qw/text geo mobile_content/},

    # типы кампаний, которые в принципе считаются кампаниями в API5 (остальных мы просто не видим с ошибокой NotFound, а из этого списка либо работаем, либо возвращаем NotSupportedType)
    'api5_visible' => {map {$_ => 1} qw/text mobile_content dynamic performance mcbanner cpm_banner cpm_deals content_promotion/},
    # объекты таких типов кампаний можно редактировать клиенту в АПИ5
    'api5_edit' => {map {$_ => 1} qw/text mobile_content performance dynamic cpm_banner cpm_deals content_promotion/},
    # объекты таких типов кампаний будут отдаваться в API5 через сервис changes
    'api5_changes' => {map {$_ => 1} qw/text mobile_content performance dynamic cpm_banner cpm_deals content_promotion/},
    # типы кампаний, которые можно редактировать клиенту в сервисе API5 Сampaigns
    'api5_edit_campaigns' => {map {$_ => 1} qw/text mobile_content performance dynamic cpm_banner/},
    # типы кампаний, которые можно редактировать клиенту в сервисе API5 СampaignsExt
    'api5_edit_campaigns_ext' => {map {$_ => 1} qw/text mobile_content performance dynamic cpm_banner content_promotion/},
    # типы кампаний, живущих под кошельком
    # сюда не включен billing_aggregate, хотя кампании этого типа всегда привязаны к ОС
    # не включен, т.к. в большинстве мест, где используется этот надтип, агрегаты не применимы,
    # а где применимы - там указаны явно
    # поэтому следует интерпретировать этот надтип как "кампании под кошельком, в которых могут быть рекламные материалы"
    'under_wallet' => {map {$_ => 1} qw/text dynamic mobile_content performance mcbanner cpm_banner cpm_deals
        cpm_yndx_frontpage content_promotion cpm_price/},

    # типы кампаний, для которых доступен перенос средств
    'money_transfer' => {map {$_ => 1} qw/wallet text dynamic mobile_content performance mcbanner cpm_banner cpm_deals internal_autobudget
        cpm_yndx_frontpage content_promotion cpm_price/},

    # типы кампаний, для которых доступна inplace конвертация валют
    'currency_convert' => {map {$_ => 1} qw/wallet text dynamic mobile_content performance mcbanner cpm_banner cpm_deals cpm_yndx_frontpage content_promotion cpm_price/},
    # типы кампаний, которые остаются в фишках и не проходят конвертацию в валюту
    'non_currency_convert' => {map {$_ => 1} qw/geo mcb/},
    # типы кампаний, которые НЕ учитываем в лимите на количество кампаний клиента
    'skip_in_client_campaign_count' => {map {$_ => 1} qw/wallet mcb geo billing_aggregate/ },
    # типы кампаний по рекламе контента в сторах
    'store_content' => {map {$_ => 1} qw/mobile_content/},
    # типы медийных кампаний с оплатой за показы
    'cpm' => { map {$_ => 1} qw/cpm_banner cpm_deals cpm_yndx_frontpage cpm_price/ },

    # типы кампаний, которые можно копировать
    # ВНИМАНИЕ! на текущий момент не умеем копировать между клиентами фиды и креативы, поэтому
    # динамические кампании могут быть скопированы только если в них нет фидов,
    # текстовые и кампании мобильного контента могут быть скопированы только если в них нет ГО с креативом из конструктора креативов
    'copyable' => {map {$_ => 1} qw/mcb text wallet dynamic mobile_content performance mcbanner cpm_banner cpm_deals cpm_yndx_frontpage content_promotion internal_free internal_distrib internal_autobudget/},
    # типы кампаний, которые может копировать клиент
    'copyable_by_client' => {map {$_ => 1} qw/text dynamic mobile_content performance mcbanner cpm_banner cpm_yndx_frontpage internal_free internal_distrib internal_autobudget/},

    # типы кампаний, для которых работает загрузка/выгрузка из excel
    'xls' => {map {$_ => 1} qw/text mobile_content/},   # TODO DIRECT-67003 +mcbanner
    # типы кампаний, для которых доступно создание PDF-отчета
    'pdf_report' => {map {$_ => 1} qw/text dynamic mobile_content performance mcbanner/},
    # типы кампаний, для которых можно создавать медиаплан
    'mediaplan' => {'text' => 1},
    # типы кампаний, в которых могут быть виртуальные визитки
    'vcards' => {map {$_ => 1} qw/text dynamic/},

    # типы кампаний, доступные для оплаты клиентами
    'payable' => {map {$_ => 1} qw/text wallet mcb dynamic mobile_content performance mcbanner cpm_banner cpm_deals cpm_yndx_frontpage content_promotion cpm_price/},
    # типы кампаний, доступные для оплаты клиентами через терминал
    'payable_by_terminal' => {map {$_ => 1} qw/text wallet dynamic mobile_content mcbanner cpm_banner cpm_deals cpm_yndx_frontpage content_promotion cpm_price/},
    # у кого может работать автобюджет
    'autobudget' => {map {$_ => 1} qw/text geo dynamic mobile_content/},    # TODO DIRECT-67003 непонятно, нужен ли mcbanner
    # у кого может работать дневной бюджет
    'day_budget' => {map {$_ => 1} qw/text dynamic mobile_content mcbanner wallet cpm_banner cpm_deals internal_autobudget cpm_yndx_frontpage content_promotion cpm_price/},

    # типы кампаний, платежи по которым триггерят проверку "можно ли сделать заявку на ПП"
    'payment_on_types_for_fa' => {map {$_ => 1} qw/text wallet/},

    # Типы кампаний, представленные как `text` в Балансе
    'text_campaign_in_balance' => {map {$_ => 1} qw/text wallet dynamic mobile_content performance mcbanner content_promotion/},

    # Типы кампаний, представленные как `text` в Модерации
    # Для динамических кампаний считаем что тип кампании - текстовая. оторвать после DIRECTMOD-4232
    # Для mcbanner также отправляем тип кампании - текстовая. Отрывать пока не планируем
    'text_campaign_in_moderation' => {map {$_ => 1} qw/text dynamic mcbanner cpm_banner cpm_deals cpm_price/},
    # кампании, отправляемые обычным (moderateExportMaster + moderateClientNew) экспортом в модерацию
    'mod_export' => {'text' => 1, 'geo' => 1, 'dynamic' => 1, 'mobile_content' => 1, 'mcbanner' => 1, 'cpm_banner' => 1, 'cpm_deals' => 1, 'internal_distrib' => 1, 'internal_free' => 1, 'internal_autobudget' => 1, 'cpm_yndx_frontpage' => 1, 'content_promotion' => 1, 'cpm_price' => 1, 'performance' => 1},
    # кампании из набора 'mod_export', для которых не генерируется событие copyCampModerateInfo в moderation_cmd_queue
    'mod_export_copy_disabled' => {'performance' => 1, 'internal_distrib' => 1, 'internal_free' => 1, 'internal_autobudget' => 1},
    # кампании, отправляемые в старый транспорт без других объектов групп/баннеров/...
    # отправляем в модерацию только для того, чтобы можно было привязывать к ним документы на стороне Модерации
    'mod_export_campaigns_only' => {'performance' => 1, 'internal_distrib' => 1, 'internal_free' => 1, 'internal_autobudget' => 1},
    # кампании, переотправляемые через внутренний отчет "Ленивая перепосылка в модерацию"
    'lazy_moderate_resync' => {'text' => 1, 'mobile_content' => 1, 'dynamic' => 1, 'mcbanner' => 1, 'cpm_banner' => 1, 'cpm_deals' => 1, 'cpm_yndx_frontpage' => 1, 'content_promotion' => 1, 'cpm_price' => 1},

    # кампании, которые можно перемещать только в std и nosend очереди транспорта
    'bs_export_limited_queues' => {'geo' => 1, 'mcb' => 1},
    # кампании, отправляемые обычным (bsExportMaster + bsClientData) экспортом в БК
    'bs_export' => {'text' => 1, 'geo' => 1, 'wallet' => 1, 'dynamic' => 1, 'mobile_content' => 1, 'performance' => 1, 'mcbanner' => 1, 'cpm_banner' => 1,
        'cpm_deals' => 1, 'internal_distrib' => 1, 'internal_free' => 1, 'internal_autobudget' => 1, 'cpm_yndx_frontpage' => 1, 'content_promotion' => 1, 'cpm_price' => 1},

    # кампании для которых доступны звездочки Маркета
    'market_ratings' => {map {$_ => 1} qw/text dynamic/},

    # кампании средства на которых учитываются при рассчете дневного лимита баллов в API5
    'api5_camp_sum_rest_for_units'  => {map {$_ => 1} qw/text wallet dynamic mobile_content performance mcbanner cpm_banner cpm_deals cpm_yndx_frontpage content_promotion cpm_price/},

    # кампании, для которых можно установить дату окончания
    'camp_finish' => { map { $_ => 1 } qw/text mcb mobile_content dynamic performance mcbanner cpm_banner cpm_deals internal_distrib
        internal_free internal_autobudget cpm_yndx_frontpage content_promotion cpm_price/ },

    # кампании статистика по которым поддерживается в МОЛ (Stat::StreamExtended)
    'stat_stream_ext' => {map {$_ => 1} qw/text dynamic mobile_content performance geo mcbanner cpm_banner cpm_deals cpm_yndx_frontpage content_promotion internal_distrib internal_free internal_autobudget cpm_price/},

    # кампании статистика по которым собирается в офлайн отчетах по агентствам
    'agency_offline_stat' => {map {$_ => 1} qw/text dynamic mobile_content performance/},   # TODO DIRECT-67003 mcbanner?

    # кампании для которых не отображаются показы и CTR в статистике
    'stat_no_shows' => {map {$_ => 1} qw/performance/}, # TODO DIRECT-67003 mcbanner?

    # кампании, для которых выполняем автоархивацию баннеров (после месяца неактивности)
    auto_archive_banners => { map {$_ => 1} qw/text/ },
    # кампании, для которых НЕ выполняем автоматическую архивацию (подробнее в ppcArchiveOldCampaigns.pl)
    # кампании этих типов также не удаляются как старые неоплаченные (ppcClearOldUnpaidCampaigns.pl)
    skip_auto_archive_campaigns => { map {$_ => 1} qw/geo wallet billing_aggregate internal_distrib internal_free internal_autobudget content_promotion/ },

    # кампании, которым всегда выставляем флаг no_title_substitute
    no_title_substitute => { map {$_ => 1} qw/mobile_content/ },

    # кампании, которым можно настраивать расширенный геотаргетинг
    with_extended_geotargeting => { map {$_ => 1} qw/text dynamic mobile_content mcbanner cpm_banner cpm_deals content_promotion/ },

    # кампании, у баннеров которых есть заголовок 1
    'with_title_on_banners' => { map {$_ => 1} qw/text mobile_content/ },

    # кампании, у баннеров которых есть заголовок 2
    'with_title_extension_on_banners' => { map {$_ => 1} qw/text/ },

    # Кампании, для которых ограничено управление показами в РСЯ
    'context_limited' => {map {$_ => 1} qw/dynamic mcbanner content_promotion/},

    # Кампании, у которых принудительно включена разметка ссылок для метрики
    'yclid_enabled' => {map {$_ => 1} qw/performance internal_distrib internal_free internal_autobudget/},

    # кампании, у которых возможны уточнения на баннерах
    'callouts' => {map {$_ => 1} qw/text dynamic/},

    # кампании, для которых разрешены авто-видео
    'auto_resources' => { map { $_ => 1 } qw/text mobile_content/ },

    # кампании, которым можно проставлять язык кампании
    'camp_lang' => {map {$_ => 1} qw/text mobile_content dynamic mcbanner cpm_banner cpm_deals cpm_yndx_frontpage content_promotion/},

    # кампании с автоматически генерируемыми дочерними баннерами
    'sub_banners_generated' => {map {$_ => 1} qw(dynamic performance)},

    # кампании, для которых можно использовать Я.Аудитории в ретаргетинге
    'allow_audience' => { map { $_ => 1 } qw/text mobile_content dynamic performance mcbanner cpm_banner cpm_deals internal_distrib
        internal_free internal_autobudget cpm_yndx_frontpage content_promotion cpm_price/ },


    # кампании, для которых доступен мониторинг баннеров по доступности домена.
    'allow_domain_monitoring' => { map { $_ => 1 } qw/text mcbanner cpm_banner cpm_deals internal_distrib internal_free internal_autobudget dynamic
        cpm_yndx_frontpage/ },

    # кампании хранящие ставки в таблице bids
    'has_bids' => { map { $_ => 1 } qw/text geo mobile_content content_promotion/ },
    # кампании хранящие ставки в таблице bids_dynamic
    'has_bids_dynamic' => { map { $_ => 1 } qw/dynamic/ },
    # кампании хранящие ставки в таблице bids_performance
    'has_bids_performance' => { map { $_ => 1 } qw/performance/ },

    # кампании, которым можно включить и выключить показ дополнительных данных из Справочника в Яндекс.Картах
    'has_permalink_info' => { map { $_ => 1 } qw/text dynamic/ },

    # кампании, в которых можно включить матчинг на самую точную фразу в рамках кампании
    'campaign_exact_phrase_matching_allowed' => { map { $_ => 1 } qw/text dynamic mobile_content/ },

    # кампании, на которых можно включать использование трафаретов с вытеснением других участников
    'alone_trafaret_allowed' => { map { $_ => 1 } qw/text dynamic mobile_content performance/ },

    # кампании, на которых можно включать фильтрацию по списку запрещенных доменов на внутренних площадках
    'require_filtration_by_dont_show_domains' => { map { $_ => 1 } qw/text dynamic cpm_banner performance mobile_content/ },

    # кампании, на которых можно включать фильтрацию по списку запрещенных доменов на внутренних площадках по второй фиче
    'require_filtration_by_dont_show_domains_in_cpm' => { map { $_ => 1 } qw/cpm_banner/ },

    # кампании, на которых можно включать продвижение турбо-аппов
    'has_turbo_app' => { map { $_ => 1 } qw/text performance dynamic/ },

    # кампании, по которым требуется проверять соблюдение уникальности домена при активированном промокоде
    # баннеры в них непосредственно имеют ссылки на сайт рекламодателя
    # или содержат "главный домен", по которому BannerLand сгенерирует объявления
    'antifraud_promocodes' => { map { $_ => 1 } qw/text dynamic mcbanner cpm_banner cpm_deals cpm_yndx_frontpage cpm_price/ },

    # кампании, ссылки из которых обычно ведут на сайт рекламодателя, кроме неподдерживаемых типов
    'direct_links_in_ads' => { map { $_ => 1 } qw/text dynamic performance mcbanner cpm_banner cpm_deals cpm_yndx_frontpage cpm_price/ },

    # кампании, для которых не нужно создавать биллинговые агрегаты. Либо у них не бывает
    # рекламных материалов, либо их больше не создают
    'without_billing_aggregates' => { map { $_ => 1 } qw/mcb geo wallet billing_aggregate internal_distrib internal_free internal_autobudget/},

    # кампании с внутренними объявлениями
    'internal' => { map {$_ => 1} qw/internal_distrib internal_free internal_autobudget/ },

    # кампании, которые не отправляются в биллинг
    'no_send_to_billing' => { map {$_ => 1} qw/internal_distrib internal_free/ },

    # это совсем-совсем уникальные типы, у которых всё не так, по-хорошему их в campaigns не должно быть
    # для них делаем отдельные "надтипы"
    # по этой же причине для них отдельные функции is_xxx_camp
    'media' => {'mcb' => 1},
    'wallet' => {'wallet' => 1},
    'billing_aggregate' => {'billing_aggregate' => 1},
    'geo' => {'geo' => 1},
);

# кеш типов кампаний по cid/OrderID
my $CACHE;


=head2 $CACHE_OPTIONS

    Ссылка на хэш с опциями для кеша

=cut
our $CACHE_OPTIONS ||= {
    size => 10_000,
    ttl => 60,
};


=head2 get_camp_kind_types($kind1, ...)

Получить ссылку на массив из типов, входящих в надтип по условию ИЛИ

=cut
sub get_camp_kind_types(@) {
    my (@kinds) = @_;

    my %types;
    for my $kind (@kinds) {
        croak "Incorrect kind $kind" unless exists $KINDS{$kind};
        hash_merge \%types, $KINDS{$kind};
    }
    return [sort keys %types];
}

=head2 get_camp_kind_types_and($kind1, ...)

Получить ссылку на массив из типов, входящих в надтип по условию И

=cut

sub get_camp_kind_types_and(@) {
    my (@kinds) = @_;

    my %types;
    for my $kind (@kinds) {
        croak "Incorrect kind $kind" unless exists $KINDS{$kind};
        $types{$_}++ for keys %{$KINDS{$kind}};
    }

    return [sort grep { $types{$_} == @kinds } keys %types];
}

=head2 camp_kind_in(valtype, val|vals, kind1, ...)

    camp_kind_in(type => 'mcb', 'web_edit')
    camp_kind_in(cid => [23123, 124], 'web_edit')
    camp_kind_in(OrderID => 123, 'all')

    Проверка того, что все переданные кампании или типы входят в надтип

=cut
sub camp_kind_in($$$;@) {
    my ($key, $vals, @kinds) = @_;

    my @camps_types = _get_camp_types($key, $vals);

    my $check_hash = @kinds == 1
        ? ($KINDS{$kinds[0]} || croak "Incorrect kind $kinds[0]")
        : {map {$_ => 1} @{get_camp_kind_types(@kinds)}};

    for my $type (@camps_types) {
        return 0 unless defined $type && $check_hash->{$type};
    }

    return 1;
}


=head2 camp_type_in()

    camp_type_in(type => 'mcb', 'text')
    camp_type_in(cid => 23123, 'geo')
    camp_type_in(OrderID => 123, '')

=cut
sub camp_type_in($$$;@) {
    my ($key, $vals, @types) = @_;

    my @camps_types = _get_camp_types($key, $vals);
    my %check_hash = map {$_ => 1} @types;

    for my $type (@camps_types) {
        return 0 unless defined $type && $check_hash{$type};
    }

    return 1;
}


=head2 get_camp_type(cid => 768)

    Получить тип кампании по cid или OrderID
    Для несуществующей кампании (или cid=undef|0) - undef
    Для невалидных id будет выброшено исключение

=cut
sub get_camp_type($$) {
    my ($key, $id) = @_;
    return undef unless $id;
    return get_camp_type_multi($key, [$id])->{$id};
}


=head2 get_camp_type_multi(cid => [1,2,3])

    Получить ссылку на хэш - тип кампании по cid или OrderID
    Для несуществующих id в хэше будет undef
    Если в массиве встретится невалидный id - будет выброшено исключение

=cut
sub get_camp_type_multi($$) {
    my ($key, $ids) = @_;

    croak "Incorrect key: $key" unless $key eq 'cid' || $key eq 'OrderID';
    if (my @invalids = grep {!is_valid_id($_)} @$ids) {
        croak "Incorrect ids: ".join(',', @invalids);
    }

    $CACHE //= RbacSimpleCache->new(%$CACHE_OPTIONS);

    my %ret = map {$_ => $CACHE->get("$key:$_")} @$ids;
    if (my @uncached_ids = grep {!defined $ret{$_}} @$ids) {
        my $types = get_hash_sql(PPC($key => \@uncached_ids), [
                                     "SELECT $key, type FROM campaigns",
                                     WHERE => {"${key}__int" => SHARD_IDS}
                                 ]
            );
        while(my ($id, $type) = each %$types) {
            $CACHE->set("$key:$id", $type);
            $ret{$id} = $type;
        }
    }

    return \%ret;
}


=head2 clear_cache()

    очистка кеша типов

=cut
sub clear_cache() {
    $CACHE->clear() if $CACHE;
}


# внутренняя функция для реализации is_media_camp/is_wallet_camp
sub _is_type_camp {
    my ($expected_type, $key, $val) = @_;
    my $type;
    if ($key eq 'type') {
        $type = $val;
    } else {
        $type = get_camp_type($key, $val);
    }
    return defined $type && $type eq $expected_type ? 1 : 0;
}


=head2 is_media_camp(key => val)

    по cid/OrderID/type определить, является ли кампания Баяновской

=cut
sub is_media_camp($$) {
    return _is_type_camp('mcb', @_);
}


=head2 is_wallet_camp(key => val)

    по cid/OrderID/type определить, является ли кампания кошельком

=cut
sub is_wallet_camp($$) {
    return _is_type_camp('wallet', @_);
}


# получить массив и типов кампаний
# для несуществующих кампаний отдаётся undef
sub _get_camp_types($$) {
    my ($key, $vals) = @_;

    $vals = [$vals] if ref $vals ne 'ARRAY';

    if ($key eq 'type') {
        return @$vals;
    } elsif ($key eq 'cid' or $key eq 'OrderID') {
        return values %{get_camp_type_multi($key => $vals)};
    } else {
        croak "Incorrect key: $key";
    }
}


=head2 expand_camp_type

Расширить один тип кампаний до всей группы типов, проверяем в порядке переданных надтипов.
Используем, при запросах по типу, если исходно есть только тип кампании

    $types = expand_camp_type("text", ["web_edit_base", "media"]); # ["text", "mobile_content"]
    $types = expand_camp_type("mcb", ["web_edit_base", "media"]); # ["mcb"]

=cut

sub expand_camp_type($$) {
    my ($type, $kinds) = @_;

    for my $kind (@$kinds) {
        if (camp_kind_in(type => $type, $kind)) {
            return get_camp_kind_types($kind);
        }
    }
    # если не нашли надтип, возвращаем исходный тип
    return [$type];
}

=head2 get_camp_supported_adgroup_types($key, $val)
    Возвращает массив типов групп которые поддерживает данный тип кампании.

    Параметры:
        Параметры взаимоисключающие
        type => text|dynamic|mobile_content - по типу кампании
        cid => 123 - по номеру кампании

    Результат:
        [] - массив типов групп (или ошибка, если кампания имеет неизвестный тип)

=cut

{
#Если у группы есть гео, то также добавить ее в ADGROUP_TYPES_WITH_GEO_BY_CAMP_TYPE
my %ADGROUP_TYPES_BY_CAMP_TYPE = (
    text => ['base'],
    dynamic => ['dynamic'],
    mobile_content => ['mobile_content'],
    performance => ['performance'],
    mcb => undef,
    mcbanner => ['mcbanner'],
    cpm_banner => ['cpm_banner', 'cpm_video', 'cpm_outdoor', 'cpm_indoor', 'cpm_audio', 'cpm_geoproduct', 'cpm_geo_pin'],
    cpm_deals => ['cpm_banner'],
    cpm_yndx_frontpage => ['cpm_yndx_frontpage'],
    internal_distrib => ['internal'],
    internal_free => ['internal'],
    internal_autobudget => ['internal'],
    content_promotion => ['content_promotion_video', 'content_promotion'],
    cpm_price => ['cpm_yndx_frontpage', 'cpm_video', 'cpm_banner', 'cpm_audio'],
);

sub get_camp_supported_adgroup_types {
    my ($key, $val) = @_;

    my $camp_type;
    if ($key eq 'type') {
        $camp_type = $val;
    } elsif ($key eq 'cid') {
        $camp_type = get_camp_type(cid => $val);
    } else {
        croak "Incorrect key: $key";
    }

    croak "Don't know about adgroups of campaign type: $camp_type" if !exists $ADGROUP_TYPES_BY_CAMP_TYPE{$camp_type};

    return $ADGROUP_TYPES_BY_CAMP_TYPE{$camp_type};
}
}

=head2 get_camp_supported_adgroup_types_with_geo($key, $val)
    Возвращает массив типов групп которые поддерживают гео для данного типа кампании.

    Параметры:
        Параметры взаимоисключающие
        type => text|dynamic|mobile_content - по типу кампании
        cid => 123 - по номеру кампании

    Результат:
        [] - массив типов групп (или ошибка, если кампания имеет неизвестный тип)

=cut

{
    my %ADGROUP_TYPES_WITH_GEO_BY_CAMP_TYPE = (
    text => ['base'],
    dynamic => ['dynamic'],
    mobile_content => ['mobile_content'],
    performance => ['performance'],
    mcb => undef,
    mcbanner => ['mcbanner'],
    cpm_banner => ['cpm_banner', 'cpm_video', 'cpm_audio'],
    cpm_deals => ['cpm_banner'],
    cpm_yndx_frontpage => ['cpm_yndx_frontpage'],
    internal_distrib => ['internal'],
    internal_free => ['internal'],
    internal_autobudget => ['internal'],
    content_promotion => ['content_promotion_video', 'content_promotion'],
    cpm_price => ['cpm_yndx_frontpage'],
);

sub get_camp_supported_adgroup_types_with_geo {
    my ($key, $val) = @_;

    my $camp_type;
    if ($key eq 'type') {
        $camp_type = $val;
    } elsif ($key eq 'cid') {
        $camp_type = get_camp_type(cid => $val);
    } else {
        croak "Incorrect key: $key";
    }

    croak "Don't know about adgroups of campaign type: $camp_type" if !exists $ADGROUP_TYPES_WITH_GEO_BY_CAMP_TYPE{$camp_type};

    return $ADGROUP_TYPES_WITH_GEO_BY_CAMP_TYPE{$camp_type};
}
}

=head2 camp_type_to_param

    Превращает тип кампании в параметр для подстановки в url:
    text                => type1 - Текстово-графические объявления
    mobile_content      => type2 - Реклама мобильных приложений
    dynamic             => type3 - Динамические текстовые объявления
    performance         => type4 - Динамические медийные объявления (зарезервированно, но не используется)
    geo                 => type5 - Кампании Геоконтекста
    mcbanner            => type6 - Графические объявления на поиске
    cpm_banner          => type7 - охватный продукт, оплата за показы, баннер в сети
    cpm_deals           => type8 - частные сделки
    internal_distrib    => type9 - дистрибуционные кампании внутренней рекламы
    internal_free       => type10 - бесплатные кампании внутренней рекламы
    internal_autobudget => type11 - автобюджетные кампании внутренней рекламы
    cpm_yndx_frontpage  => type12 - медийные кампании на главной
    content_promotion   => type13 - продвижение контента в Вертикалях
    cpm_price           => type14 - Прайсовые продажи

=cut

sub camp_type_to_param {
    my $camp_type = shift;

    my $type2param = {
        text                => 'type1',
        mobile_content      => 'type2',
        dynamic             => 'type3',
        performance         => 'type4',
        geo                 => 'type5',
        mcbanner            => 'type6',
        cpm_banner          => 'type7',
        cpm_deals           => 'type8',
        internal_distrib    => 'type9',
        internal_free       => 'type10',
        internal_autobudget => 'type11',
        cpm_yndx_frontpage  => 'type12',
        content_promotion   => 'type13',
        cpm_price           => 'type14',
    };

    my $param = $type2param->{$camp_type};

    die "unknown camp_type $camp_type" unless $param;

    return $param;
}

=head2 default_product_type_by_camp_type

    Получить тип продукта "по умолчанию" для заданного типа кампании.
    Внутри кампании одного типа могут быть рекламные материалы со специфичным типом продукта.
    К примеру, в кампаниях типа cpm_banner могут быть медийные видео объявления.
    Если произошли открутки по видео, они должны быть записаны на заказ (биллинговый агрегат)
    с типом продукта cpm_video. Иначе, открутки записываются на заказ с типом продукта по умолчанию.

=cut
sub default_product_type_by_camp_type {
    my ($camp_type) = @_;

    if (!exists($DEFAULT_PRODUCT_TYPES_BY_CAMP_TYPE{$camp_type})) {
        croak "unknown camp type $camp_type";
    }
    return $DEFAULT_PRODUCT_TYPES_BY_CAMP_TYPE{$camp_type};
}

1;
