package CampaignQuery;
use strict;
use warnings;

# $Id$

=head1 NAME

CampaignQuery

=head1 SYNOPSIS

 my $fields = [ qw( cid OrderID name ) ];

 # совсем простой способ:
 my $campaign = CampaignQuery->get_campaign_data( cid => 123, [ qw( cid OrderID) ] );
 # в $campaign либо хэш с cid и OrderID, либо undef

 # когда нужно несколько кампаний:
 my $filter = { cid_range => [ 1, 100_000 ], require_OrderID => 1 };
 my $campaign_data = CampaignQuery->get_campaign_data_multi( $filter, $fields );
 # в campaign_data массив хэшей: [ { cid => ..., OrderID => ..., name => ... }, ... ]

 # более подробный вариант предыдущего:
 my $query = CampaignQuery->new( filter => $filter, fields => $fields );
 my $campaign_data = $query->result;

=head1 DESCRIPTION

Обёртка (Data Mapper) над SQL-запросами для получения данных кампаний.
Нужна для составления запросов на выборку определённых данных из
таблиц кампаний (их может быть несколько) по нужным фильтрам.
(Фильтры необязательны: если понадобится, можно выбрать из таблицы
все кампании.)

Решение о том, какие таблицы нужно подключить для выполнения запроса,
объект принимает сам.

Список параметров: см. @VALID_SELECTORS и @VALID_FILTERS
разница в том, что фильтр из VALID_SELECTORS отвечает за выборку
по идентификатору (cid или OrderiD), и такой можно задать только один.

Список полей: см. %IS_FIELD_SUPPORTED

=cut

use parent 'Class::Accessor';

use Carp qw( croak );
use List::MoreUtils qw( all any uniq );
use Readonly;
use Storable qw( dclone );

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::Validate;

use Settings;

Readonly my @TABLES => qw( campaigns camp_options camp_metrika_counters users strategies);

Readonly my %TABLE_FIELDS => (
    campaigns => [ qw(
        cid
        uid
        ManagerUID
        AgencyUID
        name
        LastChange
        start_time
        OrderID
        AgencyID
        currency
        sum_to_pay
        sum
        sum_spent
        sum_last
        sum_spent_units
        sum_units
        wallet_cid
        statusModerate
        statusShow
        statusActive
        shows
        clicks
        statusEmpty
        statusMail
        archived
        balance_tid
        platform
        autobudget
        autobudget_date
        statusBsSynced
        statusNoPay
        geo
        DontShow
        autoOptimization
        dontShowCatalog
        autobudgetForecastDate
        autobudgetForecast
        statusAutobudgetForecast
        timeTarget
        timezone_id
        lastShowTime
        statusBsArchived
        statusOpenStat
        disabledIps
        type
        ProductID
        myPagesOnly
        rf
        rfReset
        ContextPriceCoef
        crm_inquiry_id
        finish_time
        day_budget
        day_budget_show_mode
        paid_by_certificate
        currencyConverted
        copiedFrom
        opts
    ) ],
    camp_options => [ qw(
        FIO
        email
        valid
        lastnews
        sendNews
        sendWarn
        sendAccNews
        stopTime
        contactinfo
        money_warning_value
        enabled_sms
        banners_per_page
        sms_time
        sms_flags
        warnPlaceInterval
        statusMetricaControl
        status_click_track
        last_pay_time
        auto_optimize_request
        mediaplan_status
        manual_autobudget_sum
        camp_description
        statusPostModerate
        fairAuction
        offlineStatNotice
        minus_words
        broad_match_flag
        broad_match_limit
        statusContextStop
        create_time
        strategy
        day_budget_daily_change_count
        day_budget_stop_time
        day_budget_notification_status
        email_notifications
        competitors_domains
        device_targeting
        content_lang
        eshows_video_type
    ) ],
    camp_metrika_counters => [ qw( metrika_counters ) ],
    users => [ qw( ClientID ) ],
    strategies => [ qw(
        strategy_data
        ContextLimit
    ) ]
);

Readonly my %TABLE_SUPPORTED    => map { $_ => 1 } @TABLES;
Readonly my @VALID_SELECTORS    => qw( cid cid_range OrderID OrderID_range );
Readonly my @VALID_FILTERS      => qw( require_OrderID require_metrika_counters );
Readonly my %FIELD_MODIFIERS    => {
    currency => sub { return $_[0] || 'YND_FIXED' },
};
Readonly my $DB_CHUNK_SIZE      => 20_000;

my ( %FIELD_TO_TABLE_MAP, @SUPPORTED_FIELDS, %IS_FIELD_SUPPORTED );
foreach my $table (@TABLES) {
    foreach my $field ( @{ $TABLE_FIELDS{$table} } ) {
        $FIELD_TO_TABLE_MAP{$field} = $table;

        push @SUPPORTED_FIELDS, $field;
        $IS_FIELD_SUPPORTED{$field} = 1;
    }
}

=head1 SUBROUTINES/METHODS

=head2 CampaignQuery->get_campaign_data( $id_type => $id, [ $field, $field, ... ] )

    Поддерживаемые варианты $id_type: cid, OrderID
    fields - ARRAYREF полей из тамблиц, связанных с кампаниями (TABLE_FIELDS)
    params - именные опциональные параметры:
        get_also_empty_campaigns - 1 - не фильтровать кампании по statusEmpty, 0 - только непустые кампании.

=cut

sub get_campaign_data {
    my ( $class, $id_type, $id, $fields, %params ) = @_;

    if ( !defined $id_type ) {
        croak 'Missing id_type';
    }

    if ( $id_type ne 'cid' && $id_type ne 'OrderID' ) {
        croak 'Invalid params: must select a campaign using either cid or OrderID';
    }

    if ( !is_valid_id($id) ) {
        croak "Invalid ID ($id): must be an ID-like number";
    }

    my $data_multi = $class->get_campaign_data_multi( { $id_type => $id }, $fields, %params);

    my ($campaign_data) = @$data_multi;
    return $campaign_data;
}

=head2 CampaignQuery->get_campaign_data_multi( $filter, [ $field, $field, ... ] )

=cut

sub get_campaign_data_multi {
    my ( $class, $filter, $fields, %params ) = @_;

    my $query = $class->new( filter => $filter, fields => $fields, %params );
    return $query->result;
}

=head2 CampaignQuery->new

Конструктор: получает именованные параметры filter и fields и создаёт объект.
Валидация данных производится здесь.

=cut

sub new {
    my ( $class, %params ) = @_;

    my $filter = delete $params{filter};
    $filter = {} unless defined $filter;

    my $fields = delete $params{fields};

    my $get_also_empty_campaigns = delete $params{get_also_empty_campaigns} // 0;

    my $self = bless { filter => $filter, fields => $fields, get_also_empty_campaigns => $get_also_empty_campaigns}, $class;

    $self->_validate_filter;
    $self->_validate_fields;

    return $self;
}

=head2 $query->filter
=head2 $query->fields

Методы доступа к параметрам запроса; параметры доступны только для чтения.

=cut

__PACKAGE__->mk_ro_accessors(qw( filter fields get_also_empty_campaigns));

=head2 $query->result

=cut

sub result {
    my ($self) = @_;

    my $sql_queries = $self->_construct_sql_queries;

    my $rows = [];
    foreach my $query (@$sql_queries) {
        my $chunk_rows = get_all_sql( PPC( %{ $query->{shard_selector} } ), $query->{sql} ) || [];
        push @$rows, @$chunk_rows;
    }

    my $result = $self->_map_rows_for_result($rows);
    return $result;
}

=head2 $self->_validate_filter
=head2 $self->_validate_fields

Проверить параметры запроса. Исключения "ваш запрос неправильный"
порождаются только здесь.

=cut

sub _validate_filter {
    my ($self) = @_;

    my $filter = dclone $self->filter;

    if ( defined $filter && ref $filter ne 'HASH' ) {
        croak 'Invalid filter';
    }

    my $selector = $self->_selector;

    if (%$selector) {
        if ( keys %$selector > 1 ) {
            my $valid_selectors_out = join ', ', @VALID_SELECTORS;
            croak "Only one of those can be specified in a query: $valid_selectors_out";
        }

        my $selector_type  = $self->_selector_type;
        my $selector_param = $self->_selector_param;

        if ( all { $_ ne $selector_type } @VALID_SELECTORS ) {
            croak "Invalid selector type: $selector_type";
        }

        if ( $selector_type eq 'cid' || $selector_type eq 'OrderID' ) {
            my $selector_ok = is_valid_id($selector_param)
                || ( ref $selector_param eq 'ARRAY' && all { is_valid_id($_) } @$selector_param );

            if ( !$selector_ok ) {
                croak "Invalid value for selector=$selector_type, need either an ID or an array of IDs";
            }
        }

        if ( $selector_type eq 'cid_range' || $selector_type eq 'OrderID_range' ) {
            my $selector_ok
                = ref $selector_param eq 'ARRAY'
                && @$selector_param == 2
                && all { defined $_ } @$selector_param
                && all { $_ eq '0' || is_valid_id($_) } @$selector_param;

            if ( !$selector_ok ) {
                croak "Invalid value for selector=$selector_type, need an array with the lower and the higher bounds";
            }
        }
    }

    delete $filter->{$_} foreach @VALID_SELECTORS, @VALID_FILTERS;

    if ( my @remaining_filters = keys %$filter ) {
        my $filters_out = join ', ', @remaining_filters;
        croak "Unsupported filters: $filters_out";
    }

    return;
}

sub _validate_fields {
    my ($self) = @_;

    my $fields = dclone $self->fields;

    if ( !$fields || ref $fields ne 'ARRAY' ) {
        croak 'Missing or invalid fields';
    }

    if ( my @unsupported_fields = grep { !$IS_FIELD_SUPPORTED{$_} } @$fields ) {
        my $fields_out = join ', ', @unsupported_fields;
        croak "Unsupported fields: $fields_out";
    }

    return;
}

=head2 _selector
=head2 _selector_type
=head2 _selector_param

"Селектор" - это часть фильтра, которая относится к идентификаторам.
Он может быть только один (либо отсутствовавать, если потребителю нужны
все кампании), поэтому считается, что у него есть тип (_selector_type)
и аргумент (_selector_param).

Для упрощения проверок в случае отсутствия селектора _selector_type
представляет собой пустую строку (а не undef).

=cut

sub _selector {
    my ($self) = @_;

    my $filter = $self->filter || {};

    my %selector;
    foreach my $selector_type (@VALID_SELECTORS) {
        next unless exists $filter->{$selector_type};
        $selector{$selector_type} = $filter->{$selector_type};
    }

    return \%selector;
}

sub _selector_type {
    my ($self) = @_;

    my $selector = $self->_selector;
    return unless $selector && %$selector;

    my ($type) = keys %$selector;
    $type ||= '';

    return $type;
}

sub _selector_param {
    my ($self) = @_;

    my $selector = $self->_selector;
    return unless $selector && %$selector;

    my ($param) = values %$selector;
    return $param;
}

=head2 _required_tables
=head2 _sql_conditions_and_shards

Эти функции отвечают за части запроса к базе:

_required_tables                дальше превращается в список таблиц в запросе, с правильными join, если требуется
_sql_conditions_and_shards      подставляются в SQL-запрос в where => ... и PPC(...)

=cut

sub _required_tables {
    my ($self) = @_;

    my $filter = $self->filter;
    my $fields = $self->fields;

    my @required_tables;

    if ( $filter->{require_metrika_counters} ) {
        push @required_tables, 'camp_metrika_counters';
    }

    push @required_tables, @FIELD_TO_TABLE_MAP{@$fields};

    @required_tables = uniq @required_tables;
    @required_tables = sort @required_tables;

    return \@required_tables;
}

sub _sql_conditions_and_shards {
    my ($self) = @_;

    my $selector_type  = $self->_selector_type;
    my $selector_param = $self->_selector_param;

    my $filter = $self->filter;

    my %sql_where_base = ();

    unless ( $self->get_also_empty_campaigns ) {
        $sql_where_base{'campaigns.statusEmpty'} = 'No';
    }

    if ( $filter->{require_OrderID} ) {
        $sql_where_base{'campaigns.OrderID__ne'} = 0;
    }

    if ( $filter->{require_metrika_counters} ) {
        $sql_where_base{'camp_metrika_counters.metrika_counters__is_not_null'} = 1;
    }

    if ( $selector_type eq 'cid_range' ) {
        return [
            {
                conditions => { %sql_where_base, 'campaigns.cid__between' => $selector_param },
                shards => { shard => 'all' },
            }
        ];
    }

    if ( $selector_type eq 'OrderID_range' ) {
        return [
            {
                conditions => { %sql_where_base, 'campaigns.OrderID__between' => $selector_param },
                shards => { shard => 'all' },
            }
        ];
    }

    if ( $selector_type eq 'cid' ) {
        if ( ref $selector_param eq 'ARRAY' ) {
            my @conditions_and_shards;
            foreach_shard(
                cid        => $selector_param,
                chunk_size => $DB_CHUNK_SIZE,
                sub {
                    my ( $shard, $ids ) = @_;

                    my $query = {
                        conditions => { %sql_where_base, 'campaigns.cid' => $ids },
                        shards => { shard => $shard },
                    };

                    push @conditions_and_shards, $query;
                }
            );

            return \@conditions_and_shards;
        } else {
            return [
                {
                    conditions => { %sql_where_base, 'campaigns.cid' => $selector_param },
                    shards => { cid => $selector_param },
                }
            ];
        }
    }

    if ( $selector_type eq 'OrderID' ) {
        if ( ref $selector_param eq 'ARRAY' ) {
            my @conditions_and_shards;
            foreach_shard(
                OrderID    => $selector_param,
                chunk_size => $DB_CHUNK_SIZE,
                sub {
                    my ( $shard, $ids ) = @_;

                    my $query = {
                        conditions => { %sql_where_base, 'campaigns.OrderID' => $ids },
                        shards => { shard => $shard },
                    };

                    push @conditions_and_shards, $query;
                }
            );

            return \@conditions_and_shards;
        } else {
            return [
                {
                    conditions => { %sql_where_base, 'campaigns.OrderID' => $selector_param },
                    shards => { OrderID => $selector_param },
                }
            ];
        }
    }

    return [
        {
            conditions => \%sql_where_base,
            shards     => { shard => 'all' },
        }
    ];
}

=head2 _construct_sql_queries

Превращает пользовательский запрос в массив хэшей, который можно использовать
для запросов в базу:
[ { shard_selector => ..., sql => ... }, ... ]

Здесь определяется, каким образом объединять таблицы и какие поля в них нужны.
Таблица campaigns и поле cid нужны всегда.

=cut

sub _construct_sql_queries {
    my ($self) = @_;

    my $required_tables = $self->_required_tables;

    if ( my @unsupported_tables = grep { !$TABLE_SUPPORTED{$_} } @$required_tables ) {
        my $tables_out = join ', ', @unsupported_tables;
        croak "Internal error: unsupported tables $tables_out";
    }

    my $fields = $self->fields;

    my @fields_to_select = qw( campaigns.cid );
    foreach my $field (@$fields) {
        my $field_table = $FIELD_TO_TABLE_MAP{$field};
        push @fields_to_select, "$field_table.$field";
    }

    my $conditions_and_shards = $self->_sql_conditions_and_shards;

    my $queries = [];

    my $table_spec = 'campaigns';
    foreach my $table (@$required_tables) {
        next if $table eq 'campaigns';

        if ($table eq 'users') {
            $table_spec .= " left join $table on campaigns.uid = $table.uid";
        } elsif ($table eq 'strategies') {
            $table_spec .= " left join $table on campaigns.strategy_id = $table.strategy_id";
        } else {
            $table_spec .= " left join $table on campaigns.cid = $table.cid";
        }
    }

    foreach my $shard_query_params (@$conditions_and_shards) {
        push @$queries, {
            shard_selector => $shard_query_params->{shards},

            sql => [
                select => sql_fields(@fields_to_select),
                "from $table_spec",
                where => $shard_query_params->{conditions},
            ]
        };
    }

    return $queries;
}

=head2 _map_rows_for_result

Выбирает из результатов запроса те поля, которые запросил пользователь.

На момент на самом деле как-то меняет результат запроса только в том случае, если
пользователю не нужно поле cid. Тем не менее, если в будущем, например, названия
полей в базе и для пользователя будут отличаться, логику преобразования можно
будет добавить сюда.

=cut

sub _map_rows_for_result {
    my ( $self, $rows ) = @_;

    my $fields = $self->fields;

    my %is_field_needed = map { $_ => 1 } @$fields;

    my @result;
    foreach my $row (@$rows) {
        my $row_out = {};

        foreach my $field_name (@SUPPORTED_FIELDS) {
            if ( $is_field_needed{$field_name} ) {
                if ($FIELD_MODIFIERS{$field_name}) {
                    $row->{$field_name} = $FIELD_MODIFIERS{$field_name}->($row->{$field_name});
                }
                $row_out->{$field_name} = $row->{$field_name};
            }
        }

        push @result, $row_out;
    }

    return \@result;
}

1;

__END__

=head1 TODO

Посмотреть на следующие функции, попробовать использовать этот конструктор вместо них:

    * get_camp_info
    * get_user_camp
    * get_user_camps
    * get_user_camps_by_sql
    * get_user_camps_for_yamoney
    * get_user_camps_lite
    * get_user_camps_name_only
