package CampaignTools;

# вытащено из Campaigns, чтобы избежать циклических зависимостей

use Direct::Modern;

use JSON;
use List::Util qw/max/;
use List::MoreUtils qw/uniq any none/;
use Yandex::DateTime qw/now/;
use Yandex::I18n qw/iget iget_noop/;
use Yandex::TVM2;
use Yandex::HTTP qw/http_parallel_request/;

use BalanceQueue qw//;
use Campaign::Types qw/get_camp_type/;
use Client;
use Currencies qw/get_currency_constant/;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::TimeCommon qw/mysql2unix unix2human ts_round_day get_distinct_dates tomorrow today str_round_day/;
use TextTools qw/get_num_array_by_str/;
use MetrikaCounters qw//;
use User;
use Stat::Const qw/:base :enums/;
use Stat::OrderStatDay;
use Yandex::Validate qw/is_valid_id/;

use Settings;

use base qw/Exporter/;
our @EXPORT_OK = qw/
    validate_autobudget_goal_id
    get_campaigns_goals
    get_campaign_goals
    actualize_metrika_goals
    is_performance_allowed
    has_performance_campaigns
    is_dynamic_allowed
    is_cpv_strategies_enabled
    has_dynamic_campaigns
    get_metrika_goals
    camp_metrika_counters
    camp_metrika_counters_multi
    set_status_empty_to_no
    campaign_manager_changed
    change_status_moderate_pay
    get_deals_by_cids
    get_deals_by_deal_ids
    get_cpm_deals_is_yandex_page_flag
    enrich_camp_dialog
    campaigns_with_rejected_creatives
    get_content_promotion_content_type
    mass_get_content_promotion_content_type
    mass_get_is_uac_campaign_by_order_ids
/;



=head2 validate_autobudget_goal_id($which, $goal_id, $camp, %opts)

Проверить что выбранная цель метрики доступна для оптимизации
Статистику достижений цели больше не учитываем (DIRECT-98056).
Проверяем:
  - доступность целей
  - доступность оптимизации "по всем целям" (goal_id=0)

$err = validate_autobudget_goal_id($goal_id, {cid => 123, type => 'text', metrika_counters => '78831,877123' });

Параметры:
    $goal_id - id цели
    $camp - хеш кампании (должен содержать cid, type, metrika_counters, strategy)
    %opts
        check_if_enough_stat - проверять ли достаточность статистики, по-умолчанию -- да
        meaningful_goals - ключевые цели

=cut

sub validate_autobudget_goal_id {
    my ($goal_id, $camp, %opts) = @_;
    $opts{check_if_enough_stat} //= 1;

    return '' unless defined $goal_id; # цель не выбрали, это допустимо

    my $cid = $camp->{cid};
    my $camp_type = $cid ? get_camp_type(cid => $cid) : $camp->{type};
    croak "unknown campaign type " . ($camp_type // '')  unless $camp_type;

    my $counters = get_num_array_by_str($camp->{metrika_counters});
    if ($camp_type eq 'mobile_content') {
        return iget('Цель %d не соответствует выбранной стратегии. Допустимые цели: %s', $goal_id, join(', ', keys %MOBILE_APP_SPECIAL_GOALS) ) 
            if none { $_ == $goal_id } keys %MOBILE_APP_SPECIAL_GOALS;

        return ''; 
    } elsif ($camp_type eq 'performance') {
        return iget('Необходимо задать счетчик метрики') unless @$counters;
    }

    my $client_id = $camp->{ClientID};
    my $goals = get_campaign_goals($cid, include_multi_goals => Client::ClientFeatures::is_allowed_step_goals_in_strategies($client_id));

    # выбрано "по всем целям"
    if ($goal_id == 0 && $camp_type ne 'performance') {
        return '';
    }

    # специальная цель из РМП для ТГО-кампании
    if (($opts{has_mobile_app_goals_for_text_campaign_allowed}
        || $opts{has_mobile_app_goals_for_text_campaign_strategy_enabled})
        && $camp_type eq 'text'
        && (any { $_ == $goal_id } keys %MOBILE_APP_SPECIAL_GOALS)) {
        return '';
    }


    # выбрано "по ключевым целям"
    my $strategy_name = $camp->{strategy_name};
    if ($goal_id == $Settings::MEANINGFUL_GOALS_OPTIMIZATION_GOAL_ID
        && ($strategy_name eq 'autobudget_roi'
            || $strategy_name eq 'autobudget'
            || $strategy_name eq 'autobudget_crr' && !$opts{pay_for_conversion}
            || $opts{has_all_meaningful_goals_for_pay_for_conversion_strategies_allowed}
                && $strategy_name eq 'autobudget_crr' && $opts{pay_for_conversion})
    ) {
        if ($opts{meaningful_goals}
            && any {$_->{goal_id} != $Settings::ENGAGED_SESSION_GOAL_ID } @{$opts{meaningful_goals}}) {
            return '';
        } else {
            return iget('Для оптимизации по ключевым целям необходимо указать хотя бы одну ключевую цель, отличную от вовлеченных сессий');
        }
    }

    if (!exists $goals->{campaign_goals}->{$goal_id}) {
        my $counter_goals = $opts{prefetched_goals} // MetrikaCounters::get_counters_goals($counters);
        for my $goal (map { @{$_ || []} } values %$counter_goals) {
            if ($goal->goal_id == $goal_id) {
                $goals->{campaign_goals}->{$goal_id} = $goal->to_hash;
            }
        }
    }
    # выбрана конкретная цель
    return iget('Указанная цель не найдена.') unless exists $goals->{campaign_goals}->{$goal_id};
    return iget('Указанная цель не найдена.') if $goals->{campaign_goals}->{$goal_id}->{goal_status} eq 'Deleted' || $goals->{campaign_goals}->{$goal_id}->{counter_status} eq 'Deleted';

    return '';
}

# -----------------------------------------------------------------------------

=head2 get_campaigns_goals

  по номеру кампании (или массиву номеров кампаний) возвращаем все цели из метрики (id, название, кол-во переходов, статусы счетчика и цели)
  и сумму переходов по кампании с максимальным кол-вом переходов

  get_campaigns_goals([$cid], %options):
  (
    campaigns_goals => {cid1  => {
            goal_id1 => {goal_name => 'name1', goals_count => 12, goal_status => 'Deleted', counter_status => 'Active'},
            goal_id2 => {goal_name => 'name2', goals_count => 3, goal_status => 'Active', counter_status => 'Deleted'},
            ...
          },
         cid2  => {
           goal_id1 => {goal_name => 'name1', goals_count => 12, goal_status => 'Deleted', counter_status => 'Active'},
           goal_id2 => {goal_name => 'name2', goals_count => 3, goal_status => 'Active', counter_status => 'Deleted'},
           ...
         },
        },
    max_goals => 15,
    max_context_goals => 25
  )

  в %options возможны ключи:
    get_goal_types - включить в выборку только конкретные типы целей.
                     Eсли не указано, то выберутся все типы целей, кроме составных
    include_multi_goals - включить в выборку составные цели

=cut

sub get_campaigns_goals($;%) {
    my ($cids, %O) = @_;

    if (! $cids || ! (ref($cids) eq 'ARRAY' && @$cids)) {
        return ({}, 0);
    }
    my %get_goal_types;
    if (defined $O{get_goal_types} && !(ref($O{get_goal_types}) eq 'ARRAY')) {
        die "get_goal_types must be ARRAY ref!";
    } elsif (defined $O{get_goal_types}) {
        %get_goal_types = map { $_ => 1 } @{$O{get_goal_types}};
    }

    my $camp_metrika_goals = get_all_sql(PPC(cid => $cids), [
        "select cid, goal_id, goals_count, context_goals_count from camp_metrika_goals", 
        where => {cid => SHARD_IDS}]);
    my $goals_data = get_metrika_goals(where => {goal_id => [uniq map {$_->{goal_id}} @$camp_metrika_goals]});

    my $result = {};
    my %campaigns = (context => {}, total => {});

    for my $row (@$camp_metrika_goals) {
        my $cid = $row->{cid};
        my $goal_id = $row->{goal_id};

        # составные цели не используем в оптимизации по конверсии, пропускаем
        next if !$O{include_multi_goals} && exists $goals_data->{$goal_id}
                && ($goals_data->{$goal_id}->{goal_type} eq 'step'
                    || $goals_data->{$goal_id}->{parent_goal_id}
                    || $goals_data->{$goal_id}->{subgoal_index}
                   );
        # если указан конкретный список типов целей, то пропускаем все не входящие в него типы
        my $goal_type_key = $goals_data->{$goal_id}->{goal_type} // '';
        next if exists $O{get_goal_types} && !$get_goal_types{$goal_type_key};
        
        $result->{$cid}->{$goal_id} = {
           goal_id => $goal_id,
           goals_count => $row->{goals_count},
           context_goals_count => $row->{context_goals_count},
           goal_name => $goals_data->{$goal_id}->{name} || "",
           goal_status => $goals_data->{$goal_id}->{goal_status} || 'Deleted',
           counter_status => $goals_data->{$goal_id}->{counter_status} || 'Deleted'
        };

        if ($O{show_goal_types}) {
            my $goal_type = $goals_data->{$goal_id}->{goal_type};
            $result->{$cid}->{$goal_id}->{goal_type} = $goal_type;
            $result->{$cid}->{$goal_id}->{parent_goal_id} = $goals_data->{$goal_id}->{parent_goal_id};
        }
        
        if ($result->{$cid}->{$goal_id}->{goal_status} eq 'Active'
            && $result->{$cid}->{$goal_id}->{counter_status} eq 'Active') {
                
            $campaigns{total}->{$cid} = ($campaigns{total}->{$cid} || 0) + $row->{goals_count};
            $campaigns{context}->{$cid} = ($campaigns{context}->{$cid} || 0) + $row->{context_goals_count};
        }
    }

    return {campaigns_goals => $result,
            count_all_goals => max(values %{$campaigns{total}}),
            count_all_context_goals => max(values %{$campaigns{context}})};
}

# -----------------------------------------------------------------------------

=head2 get_campaign_goals

  Получить все цели метрики для конкретной кампании. См. get_campaigns_goals

=cut

sub get_campaign_goals
{
    my ($cid, %O) = @_;
    return {} unless $cid;

    my $goals = get_campaigns_goals([$cid], %O);
    # Выбираем только данные на одну конкретную кампанию.
    $goals->{campaign_goals} = delete($goals->{campaigns_goals})->{$cid} || {};
    return $goals;
}

=head2 actualize_metrika_goals($camp)

Добавить к списку целей кампании, цели актуальные на текущий момент

Параметры:
$camp - хеш {} кампании (обязательные поля mediaType, metrika_counters, campaign_goals)
skip_errors - не падать на ошибках из Метрики
    (если переданное значение - ссылка на скаляр - то при ошибке он будет разыменован и присвоен единице)

Результат:
В исходном $camp добавляет новые цели в $camp->{campaign_goals}

=cut

sub actualize_metrika_goals {
    my ($camp, %opts) = @_;

    if ($camp->{mediaType} eq 'performance') {
        $opts{skip_errors} //= 1;
        my $counters = MetrikaCounters::get_counters_goals(get_num_array_by_str($camp->{metrika_counters}), skip_errors => $opts{skip_errors});
        my $campaign_goals = $camp->{campaign_goals};
        for my $goal (map { @{$_ || []} } values %$counters) {
            next if exists $campaign_goals->{$goal->goal_id};
            $campaign_goals->{$goal->goal_id} = $goal->to_hash;
        }
    }
}

=head2 is_performance_allowed

Проверка, доступна ли юзеру работа со смарт-кампаниями

%O:
    ignore_eixsting - не учитывать существующие смарт кампании

=cut

sub is_performance_allowed {
    my ($client_id, $login_rights, $domain, %O) = @_;

    return 1 if $login_rights->{super_control};
    if (User::is_turkish_client($client_id, domain => $domain)) {
        return 1 if $login_rights->{manager_control};
        return 0 if $O{ignore_eixsting};
        return has_performance_campaigns($client_id);
    }
    return 1;
}

=head2 has_performance_campaigns

Есть ли у пользователя смарт кампании?

=cut

sub has_performance_campaigns
{
    my ($client_id) = @_;
    return get_one_field_sql(PPC(ClientID => $client_id), [
        "SELECT 1 FROM campaigns c JOIN users u using(uid)",
        where => {
            'u.ClientID' => SHARD_IDS,
            'c.type' => 'performance',
            'c.statusEmpty' => 'No',
        },
        "LIMIT 1"
    ]) || 0;

}

=head2 is_dynamic_allowed

Проверка, доступна ли юзеру работа с динамическими-кампаниями

=cut

sub is_dynamic_allowed {
    my ($client_id, $login_rights, $domain) = @_;

    return 1 if $login_rights->{super_control};
    if (User::is_turkish_client($client_id, domain => $domain)) {
        return has_dynamic_campaigns($client_id);
    }
    return 1;
}

=head2 is_cpv_strategies_enabled

    cpv стратегии разрешены в медийной кампании, только если в кампании все группы типа cpm_video

=cut
sub is_cpv_strategies_enabled {
    my $cid = shift;
    my $ppc_shard = PPC(cid => $cid);
    return get_one_field_sql($ppc_shard, "select if(count(1) > 0, 0, 1) from phrases where adgroup_type <> 'cpm_video' and cid = ?", $cid);
}

=head2 has_dynamic_campaigns

Есть ли у пользователя динамические кампании?

=cut

sub has_dynamic_campaigns
{
    my ($client_id) = @_;
    return get_one_field_sql(PPC(ClientID => $client_id), [
            "SELECT 1 FROM campaigns c JOIN users u using(uid)",
            where => {
                'u.ClientID' => SHARD_IDS,
                'c.type' => 'dynamic',
                'c.statusEmpty' => 'No',
            },
            "LIMIT 1"
        ]) || 0;

}

=head2 get_metrika_goals

    возвращаем по условию цели метрики
    $goals = get_metrika_goals(where => {goal_id => [1461, 1462]});
    $goals = get_metrika_goals(where => {goal_id => [1461, 1462]}, OR => {parent_goal_id => [129]});

    на входе условия:
        where => {goal_id => [...]}
        where => {parent_goal_id => [...]}

    на выходе:
    {
      '1461' => {
                  'counter_status' => 'Active',
                  'goal_id' => '1461',
                  'goal_status' => 'Active',
                  'goal_type' => 'number',
                  'name' => "Цель 1",
                  'parent_goal_id' => 2333,
                  'subgoal_index' => undef
                },
      '1462' => {
                  'counter_status' => 'Active',
                  'goal_id' => '1462',
                  'goal_status' => 'Deleted',
                  'goal_type' => 'number',
                  'name' => 'Goal name',
                  'parent_goal_id' => undef,
                  'subgoal_index' => undef
                },
    }

=cut

sub get_metrika_goals {
    my @where = @_;

    my $goals = get_hashes_hash_sql(PPCDICT, ["SELECT goal_id, name, goal_status, counter_status, goal_type, parent_goal_id, subgoal_index
                                               FROM metrika_goals", @where]);
    foreach my $goal_id (keys %$goals) {
        if ($goals->{$goal_id}->{goal_type} eq 'ecommerce') {
            $goals->{$goal_id}->{ecommerce_counter_id} = $goals->{$goal_id}->{name};
            # для виртуальных ecommerce целей заменяем название на переведенное на язык пользователя
            $goals->{$goal_id}->{name} = iget("eCommerce: Покупка (счетчик № %s)", $goals->{$goal_id}->{name});
        }
    }
    return $goals;
}


=head2 camp_metrika_counters($cid)

    Получить список счетчиков метрики для кампании,
    Результат - ссылка на массив, возможно пустой

=cut

sub camp_metrika_counters
{
    my ($cid) = @_;
    return camp_metrika_counters_multi([$cid])->{$cid};
}

=head2 camp_metrika_counters_multi($cids)

    Получить список счетчиков метрики для списка кампаний
    Результат:
    ссылка на хэш, ключи - id кампаний, значения - ссылки на массивы с id счётчиков метрики
    для кампаний, у которых нет счётчиков - ссылки на пустые массивы

=cut

sub camp_metrika_counters_multi
{
    my ($cids) = @_;

    return {} if !@$cids;

    my %ret = map {$_ => []} @$cids;
    my $metrika_counters_recs = get_all_sql(PPC(cid => $cids), [ "select cid, metrika_counters from camp_metrika_counters", where => { cid => SHARD_IDS } ]);
    for my $m (@$metrika_counters_recs) {
        $ret{$m->{cid}} = [ split /\s*,\s*/, $m->{metrika_counters} ];
    }
    return \%ret;
}

# --------------------------------------------------------------------

=head2 campaign_manager_changed

 функция вызывается после смены менеджера на кампании
 ManagerUID = 0 - кампанию рассервисировали

=cut

sub campaign_manager_changed {
    my (undef, $UID, $cid, $ManagerUID) = @_;

    # accept pay by real money without blocking
    change_status_moderate_pay($cid);

    # need for servicing campaign in balance,
    #   and for accepted pay not only money without blocking
    BalanceQueue::add_to_balance_info_queue($UID, 'cid', $cid, BalanceQueue::PRIORITY_CAMP_ON_MANAGER_CHANGED);

    my $camp_info = get_one_line_sql(PPC(cid => $cid),
       "SELECT u.uid,
               u.ClientID,
               u.login,
               c.name,
               c.ManagerUID,
               c.type AS campaign_type
          FROM campaigns c
               JOIN users u USING(uid)
         WHERE c.cid = ?", $cid
    );

    my $vars = {
        campaign_type   => $camp_info->{campaign_type}
        , client_id     => $camp_info->{ClientID}
        , client_login  => $camp_info->{login}
        , campaign_id   => $cid
        , campaign_name => $camp_info->{name}
        , manager_uid   => $camp_info->{ManagerUID}
    };
}

=head2 change_status_moderate_pay($cid)

    Set flag statusPostModerate to state for pay without blocking
        and if it not decline send order to balance

=cut

sub change_status_moderate_pay($) {
    my $cid = shift;

    my $statusPostModerate = get_one_field_sql(PPC(cid => $cid), ['SELECT statusPostModerate FROM camp_options', where => {cid => SHARD_IDS}]);

    if ($statusPostModerate && $statusPostModerate eq 'Yes') {
        my $camp_stat = get_one_line_sql(PPC(cid => $cid), [q{
            SELECT COUNT(*) AS cnt,
                   SUM(IF(
                        b.statusModerate IN ('Ready', 'Sent', 'Sending') OR
                        p.statusModerate IN ('Ready', 'Sent', 'Sending') OR
                        (b.vcard_id IS NOT NULL AND b.phoneflag IN ('Ready', 'Sent', 'Sending')),
                    1, 0)) AS cnt_on_moderation,
                    SUM(IF(
                        b.statusModerate IN ('Yes', 'No') OR
                        p.statusModerate IN ('Yes', 'No') OR
                        b.phoneflag IN ('Yes', 'No'),
                    1, 0)) AS cnt_moderated,
                    SUM(IF(b.statusModerate = 'New' OR p.statusModerate = 'New', 1, 0)) AS cnt_draft
            FROM banners b
            LEFT JOIN phrases p ON (p.pid = b.pid)
        }, where => {'p.cid' => SHARD_IDS}]);

        # if not exist moderated banner and some banners on moderation now
        # small hack for change campaign's status
        if ($camp_stat->{cnt_on_moderation} && !$camp_stat->{cnt_moderated}) {
            do_update_table(PPC(cid => $cid),
                'campaigns',
                {statusModerate => 'Ready'},
                where => {cid => $cid, statusModerate => 'Yes'},
            );
        }

        # change statusPostModerate to Accepted for accepted on momentary moderate order
        do_update_table(PPC(cid => $cid),
            'camp_options',
            {statusPostModerate => 'Accepted'}, 
            where => {cid => $cid, statusPostModerate => 'Yes'}
        );
    }

    return 1;
}

=head2 set_status_empty_to_no

    Меняет statusEmpty кампании на No

=cut

sub set_status_empty_to_no {
    my $cid = shift;

    return unless $cid;

    do_update_table(PPC(cid => $cid),
        'campaigns',
        {statusEmpty => 'No'},
        where => {cid => $cid, statusEmpty => 'Yes'},
    );
}

=head2 get_deals_by_cids

    по списку cid'ов возвращает сделки, привзяанные к данным кампаниям, в формате [{deal_id => 123, name => "name"}, ...]

=cut

sub get_deals_by_cids {
    my ($cids) = @_;

    my $deal_ids = get_one_column_sql(PPC(cid => $cids), ["
                                                      SELECT cd.deal_id
                                                        FROM campaigns_deals cd",
            WHERE => {'cd.cid' => SHARD_IDS,'cd.is_deleted' => 0}]);
    return get_deals_by_deal_ids($deal_ids);
}

=head2 get_deals_by_deal_ids

    по списку id сделок возвращает сделки в формате [{deal_id => 123, name => "name"}, ...]

=cut

sub get_deals_by_deal_ids {
    my ($deal_ids) = @_;

    my $deals_from_ppcdict = get_hashes_hash_sql(PPCDICT, ["
                                                 SELECT da.deal_id, da.adfox_name as name, da.ClientID
                                                   FROM deals_adfox da",
            WHERE => {'da.deal_id__in' => $deal_ids}]);
    my $deals;
    foreach_shard ClientID => [values $deals_from_ppcdict], sub {
            my ($shard, $chunk) = @_;
            push @$deals, @{get_all_sql(PPC(shard => $shard), ["SELECT deal_id, direct_name as name, direct_deal_status FROM deals", WHERE => {deal_id => [map {$_->{deal_id}} @$chunk]}])}
        };

    for my $deal (@$deals){
        if (!defined $deal->{name}){
            $deal->{name} = $deals_from_ppcdict->{$deal->{deal_id}}->{name};
        }
    }
    return $deals;
}


=head2 get_cpm_deals_is_yandex_page_flag($cid)

    определяем для сделочной кампании признак яндексовой площадки

=cut

sub get_cpm_deals_is_yandex_page_flag {
    my ($cid) = @_;

    my $is_yandex_page_flag;
    my $deal_ids = get_one_column_sql(PPC(cid => $cid), ["SELECT deal_id FROM campaigns_deals", WHERE => { cid => $cid, is_deleted => 0 }]);
    my $placements = get_one_column_sql(PPCDICT(), ["SELECT placements FROM deals_adfox", WHERE => { deal_id => $deal_ids, placements__is_not_null => 1 }]);
    my @page_ids = uniq(map { map { $_->{pageId} } @{from_json($_)} } @$placements);
    my $is_yandex_page_flags = get_one_column_sql(PPCDICT(), ["SELECT is_yandex_page FROM placements", WHERE => { PageID => \@page_ids }]);
    if (@$is_yandex_page_flags == @page_ids && uniq(@$is_yandex_page_flags) == 1) { # флаги для всех площадок получены и они все одинаковые
        $is_yandex_page_flag = $is_yandex_page_flags->[0];
    }

    return $is_yandex_page_flag;
}


=head2 _is_valid_dialog_id

    Проверяем что переданное значение похоже на идентификатор диалога

=cut
sub _is_valid_dialog_id {
    my ($dialog_id) = @_;

    return $dialog_id && $dialog_id =~ /^[0-9a-f]{8}-(?:[0-9a-f]{4}-){3}[0-9a-f]{12}$/
}


=head2 enrich_camp_dialog

    Пользователь указывает только dialog_id (в терминологии чатов - skill_id), но нам так же требуется имя и bot_guid,
    за этими данными мы ходим в ручку Диалогов.

    На входе:
        client_id, { id => 'uuid-диалога' }

    На выходе:
        изменяем тот же объект что был на входе, добавляем необходимые знания о диалоге

        в случае успеха
        { id => 'uuid-диалога', bot_guid => 'uuid-бота', name => 'Название диалога обрезанное до 255 символов' }

        в случае ошибки
        { id => 'uuid-диалога', error => 'Описание ошибки для фронта' }

=cut
sub enrich_camp_dialog {
    my ($client_id, $dialog) = @_;

    unless (_is_valid_dialog_id($dialog->{id})) {
        $dialog->{error} = iget('Некорректный идентификатор чата');
        return $dialog;
    }

    #Проверим, нет ли уже у клиента такого диалога. Если есть - возьмем данные из базы
    my $existing_dialog = get_one_line_sql(PPC(ClientID => $client_id),[
        'SELECT client_dialog_id, skill_id as id, bot_guid, name, is_active FROM client_dialogs',
        WHERE => {ClientID => $client_id, skill_id => $dialog->{id}}
    ]);

    if ($existing_dialog && keys %$existing_dialog){
        $existing_dialog->{error} = iget('Указанный идентификатор чата не активен') unless $existing_dialog->{is_active};
        return $existing_dialog;
    }

    my $ticket = eval{ Yandex::TVM2::get_ticket($Settings::PASKILLS_TVM2_ID) }
        or die "Cannot get ticket for $Settings::PASKILLS_TVM2_ID: $@";
    my $url = $Settings::PASKILLS_URL;
    my $profile = Yandex::Trace::new_profile('paskills:get_skill');
    my $response = Yandex::HTTP::http_parallel_request(
        POST => {
            1 => {
                url => $url,
                body => encode_json({ skillIds => [ $dialog->{id}, ], })
            }
        },
        timeout => 5,
        headers => {
            'Content-Type' => 'application/json',
            'X-Ya-Service-Ticket' => $ticket,
        },
    )->{1};
    undef $profile;

    if (
        $response->{is_success}
        && $response->{content}
        && (my $skill = eval {
            my $content = decode_json($response->{content});
            return ref($content) eq 'ARRAY' ? $content->[0] : 0;
        })
    ) {
        if ($skill->{botGuid} && $skill->{id} eq $dialog->{id}) {
            if ($skill->{onAir}) {
                $dialog->{is_active} = 1;
                $dialog->{bot_guid} = $skill->{botGuid};
                $dialog->{name} = length($skill->{name}) < 256 ? $skill->{name} : substr($skill->{name}, 0, 252) . '...';

                return $dialog;
            } else {
                $dialog->{error} = iget('Указанный идентификатор чата не активен');

                return $dialog;
            }
        }

        if ($skill->{error}) {
            $dialog->{error} = iget('Указан некорректный идентификатор чата');

            return $dialog;
        }
    }

    my $error = $response->{content} || ('Status: ' . $response->{headers}->{Status} . ' (' . $response->{headers}->{Reason} . ')');
    utf8::decode($error);
    warn "paskills request failed $url: response $error";

    $dialog->{error} = iget('Не удалось сохранить идентификатор чата, пожалуйста попробуйте позже');

    return $dialog;
}

=head2 campaigns_with_rejected_creatives($cids)

    Возвращает хеш с ID кампаний, в которых есть заблокированные креативы.

=cut

sub campaigns_with_rejected_creatives {
    my ($cids) = @_;

    if (!@$cids) {
        return {};
    }

    return get_hash_sql(PPC(cid => $cids), [
        'SELECT bp.cid, 1
         FROM banners_performance bp
         JOIN perf_creatives pc ON bp.creative_id = pc.creative_id',
        WHERE => {
            "bp.cid"            => SHARD_IDS,
            "pc.statusModerate" => "AdminReject"
        },
        'GROUP BY bp.cid'
    ]);
}


=head2 calc_min_available_budget_for_strategy_with_custom_period

    Расчитывает минимальный бюджет, который можно установить на периодной стратегии во время ее выполнения.

    Для этого определяем минимально допустимый бюджет для текущей даты:
    $min_budget = max(real_spent, assumed_spent) + (min_daily_budget * days_remain),
      где:
      - real_spent - расходоы из статистики БК до текущей даты (включительно)
      - assumed_expenses  - рассчетные траты до текущего дня (включительно)
      - min_daily_budget - минимальный дневной бюджет для текущей валюты
      - days_remain - кол-во дней до конца периода

    Параметры:
        camp_strategy -  объект стратегии из которой берем параметры start, finish, budget для расчета расходов до текущего момента.
        currency - валюта кампании
        bs_order_id - id кампании в БК для получения статистики
        new_finish_time - новая дата окончания стратегии, если она изменилась. Нужна чтобы правильно рассчитать минимальный бюджет до окончания периода.

=cut
sub calc_min_available_budget_for_strategy_with_custom_period {
    my ($camp_strategy, $client_id, $currency, $bs_order_id, $new_finish_time) = @_;

    if ($camp_strategy
        && $camp_strategy->is_period_strategy_in_progress()) {

        my $days_count_in_period = scalar get_distinct_dates($camp_strategy->start, $camp_strategy->finish);
        #сколько прошло дней, включая сегодня
        my $days_count_till_now = scalar get_distinct_dates($camp_strategy->start, str_round_day(today()));

        #оставшиеся дни до конца периода. Учитываем, что дата окончания периода могла измениться.
        my $finish_time = (defined $new_finish_time)? $new_finish_time : $camp_strategy->finish;
        my $days_remain = scalar get_distinct_dates(str_round_day(today()), $finish_time) - 1;

        #Получаем траты по кампании из статистики БК с даты начала стратегии до текущего момента
        my $real_spent = ($bs_order_id) ?
            Stat::OrderStatDay::get_orders_sum_spent([ $bs_order_id ],
                { 'from_start_till_now' => { from => $camp_strategy->start, to => str_round_day(today()) } },
                $currency) :
            {};

        if ($real_spent->{from_start_till_now} && $currency ne 'YND_FIXED') {
            my $client_nds = get_client_NDS($client_id, fetch_missing_from_balance => 1);
            $real_spent->{from_start_till_now} = Currencies::remove_nds($real_spent->{from_start_till_now}, $client_nds);
        }

        #статстика БК может отставать или отсутстовать вовсе.
        #Поэтому дополнительно рассчитывем средний дневной расход, сравнивем со статистикой и выбирем большее значение.
        my $assumed_expenses = $camp_strategy->budget / $days_count_in_period * $days_count_till_now;
        my $money_spent = max($real_spent->{from_start_till_now} // 0, $assumed_expenses);
        #Определяем минимальный бюджет для текущей даты
        my $min_budget = $money_spent + get_currency_constant($currency, 'MIN_DAILY_BUDGET_FOR_PERIOD') * $days_remain;
        return $min_budget;
    }
    return undef;
}

=head2 get_content_promotion_content_type

    Функция достаёт из базы тип контента кампании продвижения контента

=cut
sub get_content_promotion_content_type {
    my $cid = shift;
    return mass_get_content_promotion_content_type([$cid])->{$cid};
}

=head2 mass_get_content_promotion_content_type

    Функция массово достаёт из базы тип контента кампаний продвижения контента

=cut
sub mass_get_content_promotion_content_type {
    my $cids = shift;
    return {} if !@$cids;
    return get_hash_sql(PPC(cid => $cids),
        ["SELECT DISTINCT p.cid, acp.content_promotion_type",
         "FROM phrases p LEFT JOIN adgroups_content_promotion acp ON p.pid = acp.pid",
         WHERE => {
             'p.cid' => $cids,
             'p.adgroup_type' => 'content_promotion'
         }]
    );
}

=head2 is_uac_campaign_by_order_id($order_id)

    По order_id определить является ли кампания универсальной

=cut

sub is_uac_campaign_by_order_id($) {
    my $order_id = shift;

    if (!is_valid_id($order_id)) {
        return 0;
    }
    return mass_get_is_uac_campaign_by_order_ids([$order_id])->{$order_id} ? 1 : 0;
}

=head2 mass_get_is_uac_campaign_by_order_ids

    Определить, какие кампании из набора являются универсальными

=cut

sub mass_get_is_uac_campaign_by_order_ids {
    my $oids = shift;
    return {} unless @$oids;
    my $result = get_one_column_sql(PPC(OrderID => $oids->[0]), [#статистика всегда по кампаниям в одном шарде
        "SELECT OrderID FROM campaigns", WHERE => {OrderID => $oids, source => ['uac', 'widget']}
    ]);
    return { map { $_ => 1 } @$result };
}


1;
