package Captcha;

=head1 NAME

Captcha

=head1 DESCRIPTION

Расчет лимитов для показа капчи

=cut

use strict;
use warnings;
use utf8;

use List::Util qw/min/;

use Client qw/mass_client_total_sums/;
use Direct::ResponseHelper qw/detect_respond_format/;
use Forecast;
use Plack::CaptchaChecker;
use RBACElementary;
use Settings;
use Yandex::HashUtils qw/hash_cut/;
use Yandex::Memcached;
use EnvTools;

=head2 need_show_captcha

    Вычисляем, надо ли показать капчу пользователю

=cut

sub need_show_captcha
{
    my %params = @_;
    my $cur_step = $params{cur_step};

    my %checkers;
    if ($params{user_captcha_freq}
            && (detect_respond_format($params{request}) ne 'json' # для всего остального есть :NoCaptcha
                || $cur_step =~ /^(ajaxGetSuggestion|ajaxDataForBudgetForecast)$/)
    ) {
        my $captcha_checker = Plack::CaptchaChecker->new(
            md_servers => $Settings::MEMCACHED_SERVERS,
            prefix => 'uid_manual',
            captcha_key_sub => sub {$params{UID}},
            captcha_freq => $params{user_captcha_freq},
            captcha_interval => 86400,
            captcha_max_req => $params{user_captcha_freq},
        );
        $checkers{uid_manual} = $captcha_checker;
    }
    my $request_params = hash_cut \%params, qw/ClientID form user_role user_passport_karma/;
    for my $cmd_captcha (@{$params{cmd_captcha}}) {
        my %par_vars = (IP => $params{IP}, UID => $params{UID});
        my $key = $cmd_captcha->{Key};
        my @value_parts;
        for my $key_part (@$key) {
            my $value_part;
            if ($key_part =~ /^cb:(.*)$/) {
                no strict 'refs';
                $value_part = *{$1}->(%$request_params);
            } elsif (exists $par_vars{$key_part}) {
                $value_part = $par_vars{$key_part};
            } else {
                die "unknown Key part <$key_part> in :Captcha for $cur_step";
            }
            push @value_parts, $value_part;
        }
        my $value = join '/', @value_parts;

        my $captcha_limits;
        if (my $func = $cmd_captcha->{DynamicLimits}) {
            no strict 'refs';
            $captcha_limits = *{$func}->(%$request_params);
            return undef unless $captcha_limits;
        } else {
            $captcha_limits = hash_cut $cmd_captcha, qw/Freq Interval MaxReq/;
        }
        my $captcha_checker = Plack::CaptchaChecker->new(
            md_servers => $Settings::MEMCACHED_SERVERS,
            prefix => $cur_step,
            captcha_key_sub => sub {$value},
            captcha_freq => $captcha_limits->{Freq},
            captcha_interval => $captcha_limits->{Interval},
            captcha_max_req => $captcha_limits->{MaxReq},
            captcha_incr => $captcha_limits->{Incr} || 1,
        );
        $checkers{"$cur_step/$value"} = $captcha_checker;
    }

    for my $checker_name (keys %checkers) {
        my $captcha_checker = $checkers{$checker_name};
        if ($captcha_checker->check) {
            if ($params{is_captcha_ok}) {
                $captcha_checker->on_recognition();
            } else {
                return $checker_name;
            }
        }
    }
    return undef;
}


sub calc_forecast
{
    my %params = @_;
    return undef if $params{user_role} eq 'agency' || rbac_is_internal_user(undef, role => $params{user_role});
    return undef if $params{form}->{validate_phrase_only};

    my $max_req = 6000;
    if (has_paid($params{ClientID})) {
        $max_req *= 10;
    } elsif ($params{user_passport_karma} && $params{user_passport_karma} >= 85) {
        $max_req = 200;
    }
    my $phrases = Forecast::prepare_phrases_for_forecast($params{form}->{new_phrases} || $params{form}->{phrases} || '');
    return {
        Freq => 200,
        Interval => 86400,
        MaxReq => $max_req * ($params{coef} || 1),
        Incr => scalar(@$phrases) || 1,
    };
}

sub calc_forecast_ip
{
    if (my $limits = calc_forecast(@_, coef => 10)) {
        $limits->{Interval} = 3600;
        return $limits;
    } else {
        return undef;
    }
}


=head2 advanced_forecast

    Вычисление лимитов использования "нового" прогнозатора
    (применяются на UID)
    
=cut
sub advanced_forecast
{
    my %params = @_;
    return undef if $params{user_role} eq 'agency' || rbac_is_internal_user(undef, role => $params{user_role});

    my $max_req = 1000;
    if (has_paid($params{ClientID})) {
        $max_req *= 60;
    } elsif ($params{user_passport_karma} && $params{user_passport_karma} >= 85) {
        $max_req = 100;
    }
    my $phrases = Forecast::prepare_phrases_for_forecast($params{form}->{phrases});
    return {
        Freq => min($max_req, 300),
        Interval => 86400,
        MaxReq => $max_req * ($params{coef} || 1),
        Incr => scalar(@$phrases),
    };
}


=head2 advanced_forecast_ip

    Лимиты использования нового прогнозатора на IP
    
=cut
sub advanced_forecast_ip
{
    if (my $limits = advanced_forecast(@_, coef => 10)) {
        $limits->{Interval} = 3600;
        return $limits;
    } else {
        return undef;
    }
}


sub get_cached_sum_total
{
    my $ClientID = shift;

    my $md = Yandex::Memcached->new(servers => $Settings::MEMCACHED_SERVERS);
    my $key = "sum_total2client/$ClientID";
    my $cached = $md->get($key);
    unless (defined $cached) {
        my $sum_total = mass_client_total_sums(ClientIDs => [$ClientID], type => 'text', currency => 'YND_FIXED', without_bonus => 1)->{$ClientID}->{sum};

        $md->set($key, $sum_total, 3600);
        $cached = $sum_total;
    }
    return $cached;
}


{
my %cache;
sub has_paid
{
    my $ClientID = shift or return;

    unless (exists $cache{$ClientID}) {
        $cache{$ClientID} = get_cached_sum_total($ClientID);
    }
    my $sum_total = $cache{$ClientID};
    return $sum_total && $sum_total > 100;
}
}


sub has_paid_key_part
{
    my %params = @_;
    return has_paid($params{ClientID}) ? 'money' : 'no_money';
}

sub wordstat
{
    my %params = @_;
    return undef if $params{user_role} eq 'agency'
                    || is_beta()
                    || rbac_is_internal_user(undef, role => $params{user_role});

    my $max_req = 20;
    if (has_paid($params{ClientID})) {
        $max_req *= 50;
    } elsif ($params{user_passport_karma} && $params{user_passport_karma} >= 85) {
        $max_req = 2;
    }

    return {
        MaxReq => $max_req,
        Freq => min($max_req, 10),
        Interval => 86400,
    };
}

1;
