package CheckAdv;
## no critic (TestingAndDebugging::RequireUseWarnings)

# $Id$

=head1 NAME

    CheckAdv

=head1 DESCRIPTION

    Проверка рекламных объявлений 

=cut

use strict;

use GeoTools;
use geo_regions;
use BannerFlags;

use base qw/Exporter/;

our @EXPORT = qw/
        check_banner_for_regions
    /;

our %BANLIST = (
    'kz' => [
        'alcohol',
        'drugs',
        'weapon',
        'milk-substitute',
    ],
    'by' => [
        'alcohol',
        'milk-substitute',
        'med_equipment',
        'work_abroad',
        'realtor',
        'dietarysuppl',
    ],
    'ru' => [
        'alcohol',
        'abortion',
    ],
    'ua' => [
    ],
);


# символьные обозначения некоторых важных регионов
my %GEO_ID = (
    kz => 159,
    by => 149,
    ru => 225,
    ua => 187,
);

# регионы, для которых разрешены показы при условии полного вхождения таргетинга в данный регион
my @ALLOW_SHOWS = ('kz');

=head2 check_banner_for_regions($flags_str, %opt)

    Проверка, может ли баннер, содержащий слова из списков $lists показываться по таргетингу $opt{geo} 
    в регионе $opt{region}(если не указан - выполняются проверки для всех известных регионов)

    Параметры позиционные 
        $flags_str -- строка флагов, приписанных к объявлению, через запятую

    Параметры именованные
        geo -- таргетинг объявления
        region -- относительно какого региона с ограничениями надо проверить таргетинг. Необязательно, по умолчанию -- все, что записано в %BANLIST
        ClientID -- клиент, для определения транслокального дерева регионов
    
    Результат: 
        ссылка на двухуровневый хэш из ошибок - на первом уровне регион, на втором список подсветки
        если баннер можно показывать во всех регионах таргетинга - возвращается ссылка на пустой хэш

      Примеры:
      - если исключены не все регионы (объявление все-таки может показываться)
        {
            kz => {alcohol => 1}, 
            by => {alcohol => 1},
        }
      - если исключены вообще все регионы (объявление совсем не может показываться)
        {
            all => {
                kz => {alcohol => 1}, 
                by => {alcohol => 1},
            }
        }

=cut

sub check_banner_for_regions
{
    my $flags_str = shift;
    my %opt = @_;
    
    return {} unless defined $flags_str;
    my $banner_has_flag = BannerFlags::get_banner_flags_as_hash($flags_str, no_children => 1);
    
    my @restricted_regions = $opt{region} ? ($opt{region}) : keys %BANLIST;

    # разбираем строку с геотаргетингом в удобный хеш: { $id => { id => $id, plus => 1 }, $id2 => { id => $id2, minus => 1 }, }
    my %geo;
    for my $r (split ',', $opt{geo}){
        (my $id = $r) =~ s/^-//;
        $geo{$id} = { id => $id };
        $geo{$id}->{plus} = 1 unless $r =~ /^-/;
        $geo{$id}->{minus} = 1 if $r =~ /^-/;
    }
    
    # проставляем отношения "содержит/содержится"
    for my $restricted_reg (@restricted_regions){
        for my $r (values %geo){
            # флаг: текущий элемент из geo СОДЕРЖИТСЯ в текущем регионе с ограничениями 
            $r->{subregion_of}->{$restricted_reg} = 1 if GeoTools::is_targeting_in_region($r->{id}, $GEO_ID{$restricted_reg}, {ClientID => $opt{ClientID}});
        }
    }

    my $res = {};
    # Главный цикл: перебираем ограничивающие регины один за другим 
    for my $restricted_reg (@restricted_regions) {
        # вычисляем, какие списки/флаги могут ограничить показ объявления в текущем ограничивающем регионе
        my $lists_for_reg = { map { $banner_has_flag->{$_} ? ( $_ => 1 ) : () } @{$BANLIST{$restricted_reg}} };
        # если флаги объявления не влияют на показы в текущем ограничивающем регионе -- переходим к следующему 
        next unless keys %$lists_for_reg;

        # если есть плюс-регионы внутри ограничивающего региона -- срабатывает ограничение
        $res->{$restricted_reg} = $lists_for_reg, next if grep { $_->{plus} && $_->{subregion_of}->{$restricted_reg} } values %geo;

        # от ограничивающего региона идем вверх по списку (несобственных) родителей. Если первым встречается плюс-регион -- ограничение срабатывает. Если минус -- нет.
        for my $p ( $GEO_ID{$restricted_reg}, reverse @{$geo_regions::GEOREG{$GEO_ID{$restricted_reg}}->{parents}} ){
            next unless exists $geo{$p};     # Забавная ошибка получается, если эту строчку не указывать...
            last if $geo{$p}->{minus};
            $res->{$restricted_reg} = $lists_for_reg, last if $geo{$p}->{plus};
        }
    }
    
    foreach my $rr (@ALLOW_SHOWS) {
        # если таргетинг полностью входит в спец регион, то не учитываем его далее
        my $envolve = GeoTools::is_targeting_in_region($opt{geo}, $GEO_ID{$rr}, {ClientID => $opt{ClientID}});
        if ($envolve) {
            delete $res->{$rr};
        }
    }
    
    # Проверяем, остались ли регионы, в которых можно показываться
    my $has_allowed_geo = 0;
    for my $r ( grep {$_->{plus}} values %geo ){
        # если текущий плюс-регион является подрегионом хотя бы для одного регина, в котором сработало ограничение -- он полностью отклонен. 
        # Иначе -- этот регион отклонен не полностью, и объявление может показываться хоть где-то
        $has_allowed_geo = 1, last unless grep { $r->{subregion_of}->{$_} } keys %$res;
    }
    
    $res = { all => $res } unless $has_allowed_geo;

    return $res;
}



1;
