
package Client;

# $Id$

=head1 NAME

    Client

=head1 DESCRIPTION

    Работа с сущностью "клиент" (ClientID)

=cut

use Direct::Modern;
use Cache::SizeAwareMemoryCache;
use Carp qw/croak/;
use List::Util qw/min max/;
use List::MoreUtils qw/uniq any none all lastval/;
use Time::Local;
use Readonly;
use Yandex::Log;
use Yandex::HashUtils;
use Yandex::Balance;
use Yandex::IDN qw(is_valid_email);
use Yandex::I18n;
use Yandex::TimeCommon;
use Yandex::SendMail qw/send_alert/;
use Yandex::Overshard;
use BS::ResyncQueue;

use Settings;
use Yandex::DBTools;
use Yandex::DBShards;
use ShardingTools;

use Rbac qw/:const add_manager_in_hierarchy get_perminfo/;
use Currency::Rate;
use Currencies;
use Client::NDSDiscountSchedule;
use Client::CustomOptions;
use Client::ClientFeatures;
use Campaign::Types;
use Primitives ();
use PrimitivesIds;
use Yandex::Validate qw/is_valid_int is_valid_id/;
use Yandex::ScalarUtils;
use Yandex::ListUtils qw/xminus nsort chunks/;
use GeoTools;
use LogTools qw/log_bs_resync_market_ratings log_stacktrace log_messages/;
use JSON;

use base qw/Exporter/;
our @EXPORT = qw/
    create_client_id
    update_client_id
    get_client_id_list
    create_client_in_balance
    update_deleted_reps
    create_update_client
    get_overdraft_info
    get_mass_overdraft_info
    _get_autooverdraft_params

    mass_get_client_discount
    get_client_discount

    get_client_limits
    validate_client_limits
    set_client_limits
    delete_client_limits

    check_add_client_campaigns_limits
    check_add_client_groups_limits
    check_add_client_creatives_limits
    check_add_library_minus_words_limits

    mass_get_client_currencies
    get_client_currencies

    get_client_NDS
    mass_get_client_NDS

    mass_get_client_nds_schedule

    get_client_data
    mass_get_clients_data
    update_client_table

    mass_client_total_sums
    client_total_sums

    is_client_exists

    mass_is_universal_campaign_client
    is_universal_campaign_client

    is_new_wallet_warnings_client
    is_send_sms_despite_sms_flags_for_new_wallet_warnings_client

    mass_get_clients_brand_ids
    get_client_brand_id

    mass_is_new_moderate_send_warn_enabled_for_client
    mass_is_new_stat_rollbacks_enabled_for_client

    is_assessor_offer_accepted
    is_business_unit_client
/;
our @EXPORT_OK = qw(
    client_must_convert
    mass_client_must_convert
    is_any_client_must_convert
    mass_is_client_converting_soon
);

my $CACHE = Cache::SizeAwareMemoryCache->new({namespace => 'ClientLimits', default_expires_in => 60});

# Список "плохих" клиентов для автоовердрафтов. см DIRECT-87190
Readonly::Array our @BAD_AUTO_OVERDRAFT_CLIENTS_LIST => (656873, 958950, 1026231, 1627773, 3131528,
    3205125, 4081112, 1703366, 6870093, 15814128, 6652704, 2995358, 5508797,
    6833998, 7841354, 7817363, 8247544, 8877584, 9293020, 10970263, 14722902);

# Тот же список, но в виде hash map. см DIRECT-87190
Readonly::Hash our %BAD_AUTO_OVERDRAFT_CLIENTS => map { $_ => 1 } @BAD_AUTO_OVERDRAFT_CLIENTS_LIST;

# Список доступных валют для Порога отключения. см DIRECT-91643
Readonly::Array our @AUTOOVERDRAFT_CURRENCY_AVAILABLE => ("RUB", "BYN", "KZT");

Readonly my %ROLES => map {$_ => 1} qw/
    client agency manager super superreader support limited_support placer media empty
    internal_ad_admin internal_ad_manager internal_ad_superreader
/;
Readonly my %SUBROLE2ROLE => (
        superplacer     => 'placer',
        supermedia      => 'media',
        superteamleader => 'manager',
        teamleader      => 'manager',
    );

Readonly my %new_feature_names => (
    PREVIEW_ACCESS_TO_NEW_FEATURE_CLIENTS_IDS_MEMBERS_4 => 1,   # turbolandings
    PREVIEW_ACCESS_TO_NEW_FEATURE_CLIENTS_IDS_MEMBERS_5 => 1,   # aprgoodmultigoal/Умные цели; устарело, т.к. они открыты всем
);

# максимальная длина описания на клиенте
our $MAX_USER_DESCRIPTION_LENGTH = 4096;

my $NO_SUPERVISOR = 0;

=head2 %CLIENT_TABLES

    Хэш с именами таблиц с клиентскими данными и списком полей в каждой.
    При добавлении новых таблиц/полей дополнить юнит-тест mass_get_clients_data.t

=cut

our %CLIENT_TABLES = (

        clients =>
                { order => 1,
                  fields => [ qw /
                        ClientID
                        role
                        subrole
                        chief_uid
                        agency_client_id
                        agency_uid
                        name
                        work_currency
                        create_date
                        report_name
                        deleted_reps
                        agency_url
                        agency_status
                        primary_manager_uid
                        primary_manager_set_by_idm
                        primary_bayan_manager_uid
                        primary_geo_manager_uid
                        allow_create_scamp_by_subclient
                        country_region_id
                        is_favicon_blocked
                    /],
                  boolean_fields => [ qw/
                        allow_create_scamp_by_subclient
                    /],
                },
        clients_options =>
                { order => 2,
                  fields => [ qw /
                        hide_market_rating
                        subregion_id
                        non_resident
                        is_business_unit
                        common_metrika_counters
                        is_ya_agency_client
                        is_using_quasi_currency
                        gdpr_agreement_accepted_time
                        overdraft_lim
                        auto_overdraft_lim
                        nextPayDate
                        debt
                        is_brand
                        can_manage_price_packages
                        can_approve_price_packages
                        cashback_bonus
                        cashback_awaiting_bonus
                        default_disallowed_page_ids
                        default_allowed_domains
                        social_advertising
                        tin
                        tin_type
                    /],
                  set_fields => {
                      client_flags => [qw/
                          no_text_autocorrection
                          no_display_hrefs
                          not_agreed_on_creatives_autogeneration
                          not_convert_to_currency
                          can_copy_ctr
                          auto_video
                          feature_access_auto_video
                          suspend_video
                          create_without_wallet
                          feature_context_relevance_match_allowed
                          cant_unblock
                          feature_payment_before_moderation
                          is_touch
                          is_pro_strategy_view_enabled
                          videohints_enabled
                      /],
                  },

                },
);
{
    # в каждой таблице создаём служебное поле _fields_hash
    # ключ - имя аттибута,
    # значение - undef, если поле самостоятельное, или имя set-поля
    for my $table (keys %CLIENT_TABLES) {
        my $data = $CLIENT_TABLES{$table};
        $data->{_fields_hash} = {
            map {$_ => undef} @{$data->{fields}}
        };
        if ($data->{set_fields}) {
            for my $sf (keys %{$data->{set_fields}}) {
                hash_merge $data->{_fields_hash}, {map {$_ => $sf} @{$data->{set_fields}->{$sf}}};
            }
        }
    }
}

our %CLIENT_LIMIT_FIELDS = (
    'camp_count_limit' => {
        bounds => [0],
        name => iget_noop('общее число кампаний')
    },
    'unarc_camp_count_limit' => {
        bounds => [0],
        name => iget_noop('число незаархивированных кампаний')
    },
    'banner_count_limit' => {
        bounds => [1000, 15000],
        name => iget_noop('число объявлений в каждой кампании')
    },
    'keyword_count_limit' => {
        bounds => [10, 10000],
        name => iget_noop('число ключевых фраз в каждой группе')
    },
    'feed_count_limit' => {
        bounds => [0],
        name => iget_noop('количество фидов'),
    },
    'feed_max_file_size' => {
        bounds => [0],
        name => iget_noop('размер файла фида'),
    },
    'general_blacklist_size_limit' => {
        bounds => [10, 20000],
        name => iget_noop('максимальное количество отключенных площадок для для не-видео объявлений')
    },
    'video_blacklist_size_limit' => {
        bounds => [10, 1000],
        name => iget_noop('максимальное количество отключенных площадок для видео')
    },
);

=head2 $NOT_GEOCONTEXT_HAVING_CONDITION

    Условие, отсекающее клиентов геоконтекста при выборке из базы с группировкой.
    Клиент Геоконтекста == клиент, имеющий только гео-кампании.
    Условие предполагает что в запросе джоинится таблица campaigns с алиасом c.

    Типичное использование:
        SELECT [...]
        FROM users u
        JOIN campaigns c using(uid)
        WHERE uid IN ([...])
        GROUP BY uid
        HAVING $Client::NOT_GEOCONTEXT_CLIENT_CONDITION_FOR_HAVING

=cut

our $NOT_GEOCONTEXT_HAVING_CONDITION = '(COUNT(NULLIF(c.type, "geo")) > 0 OR COUNT(c.cid) = 0)';

my $SQL_LOCK_NAME_PREFIX = 'CONVERT_TO_REAL_MONEY_LOCK_FOR_CLIENT_';

our $BEGIN_OF_TIME_FOR_MULTICURRENCY_CLIENT = '20000101';

our $CURRENCY_CONVERT_TEASER_DISABLED_PROPERTY_NAME = 'currency_convert_teaser_disabled';

our $SHOW_BELARUS_BANK_CHANGE_WARNING_PROPERTY_NAME = 'show_belarus_bank_change_warning';

our $SHOW_BELARUS_OLD_RUB_WARNING_PROPERTY_NAME = 'show_belarus_old_rub_warning';

our $FEATURE_ACCESS_SAVE_ADGROUP_WITHOUT_TWO_STEP_PERCENT = 'feature_skip_two_step_percent';

our $HIDE_BELARUS_BANK_CHANGE_WARNING_CLIENT_IDS_PROPERTY_NAME = 'hide_belarus_bank_change_warning_client_ids';

our $IS_FREELANCER_PROPERTY_NAME = 'is_freelancer';

our $AUTO_OVERDRAFT_MIN_VALUE_PROPERTY_NAME = 'AUTOOVERDRAFT_MIN_VALUE';

our $AUTO_OVERDRAFT_MIN_DEFAULT_VALUE = 1;

our $BUSINESS_UNIT_CLIENT_IDS_PROPERTY_NAME = 'business_unit_client_ids';

=head2 create_client_id

    Создаёт запись о клиенте в Балансе.
    Принимает позиционными параметрами:
        - ссылку на хеш с данными о клиенте. используются следующие ключи:
          + name
          + phone
          + fax
          + email
          + city
          + url
          + country_region_id -- страна клиента (код региона страны из геобазы; 225 == Россия и т.п.)
          + subregion_id -- город/регион/страна клиента, с максимально доступной точностью (код из геобазы)
          + currency -- валюта клиента ('RUB'/'EUR'/'USD'/...)
          + agency -- 'yes' => агентство определим по $UID
          + from -- 'createAgency' => создаём агентство
        - UID пользователя, создающего запись

    Возвращает 1, если создание не удалось и 0 в противном случае.
    ClientID созданного клиента записывает в ключ new_clientID переданной ссылки на хеш.

    $params = {name => , phone =>, fax =>, email =>, city =>, url =>, country_region_id =>, subregion_id =>, currency =>, agency =>, from =>};
    create_client_id($params, $UID);
    # $params->{new_clientID} => 987654

=cut

sub create_client_id($$);
sub create_client_id($$) {
    my ($vars, $UID) = @_;

    my $hash = { NAME    => ($vars->{name}  ? $vars->{name}  : '-'),
                 PHONE   => ($vars->{phone} ? $vars->{phone} : '-'),
                 FAX     => ($vars->{fax}   ? $vars->{fax}   : '-'),
                 EMAIL   => ($vars->{email} ? $vars->{email} : '-'),
                 CITY    => ($vars->{city}  ? $vars->{city}  : '-'),
                 URL     => ($vars->{url}   ? $vars->{url}   : '-'),
                 # передаём всегда директовский ServiceID, т.к. клиенты по валюте общие во всех сервисах, живущих у нас
                 SERVICE_ID => $Settings::SERVICEID{direct},
               };

    return 1 unless $hash->{NAME} && $hash->{PHONE} && $hash->{FAX} && $hash->{EMAIL} && $hash->{URL};

    return 'email' unless is_valid_email($hash->{EMAIL});

    $hash->{CLIENT_TYPE_ID} = 0;
    $hash->{IS_AGENCY} = (defined $vars->{from} && $vars->{from} eq 'createAgency') ? 1 : 0;
    if (defined $vars->{agency} &&  $vars->{agency} eq 'yes') {
        $hash->{AGENCY_ID} = get_clientid(uid => $UID);
    }

    # раньше этих полей не было. для сохранения старого поведения ключ добавляем только если у них есть значения.
    $hash->{REGION_ID} = $vars->{country_region_id} if $vars->{country_region_id};
    $hash->{SUBREGION_ID} = $vars->{subregion_id} if $vars->{subregion_id};
    if ($vars->{currency} && $vars->{currency} ne 'YND_FIXED') {
        $hash->{CURRENCY} = $vars->{currency};
        # для новых клиентов отправляем давно прошедшую дату перехода. она нужна Балансу,
        # чтобы понять по новой (мультивалютной) или старой (в фишках) схеме работает клиент
        $hash->{MIGRATE_TO_CURRENCY} = $Client::BEGIN_OF_TIME_FOR_MULTICURRENCY_CLIENT;
    }

    my ($error, $client_id) = balance_create_client($UID, $hash);

    return 1 if $error;

    $vars->{new_clientID} = $client_id;

    return 0;
}


=head2 update_client_id

    update info about client in balance by ClientID

    $is_error = update_client_id($operator_uid, $client_id, $new_info);

    $new_info = {
        NAME => '...',
        PHONE => '...',
        URL => '...',
        ...
    };

=cut

sub update_client_id
{
    my ($operator_uid, $client_id, $new_info) = @_;

    # get old info
    my $hash = {
        AgencySelectPolicy => 1,
        ClientID => $client_id,
        PrimaryClients => 1
    };

    my $clients = eval{ balance_find_client($hash) };
    if ($@){
        return 1;
    }

    # оставляем только те поля, что принимает CreateClient. всякие SERVICE_DATA выкидываем.
    my @fields = qw(
        CLIENT_ID
        CLIENT_TYPE_ID
        NAME
        EMAIL
        PHONE
        FAX
        URL
        CITY
        IS_AGENCY
        AGENCY_ID
        REGION_ID
        SUBREGION_ID
        SERVICE_ID
        CURRENCY
        MIGRATE_TO_CURRENCY
    );
    my $old_info = $clients->[0] ? hash_cut($clients->[0], @fields) : {};

    # update
    my $changed_flag = 0;
    for my $key (keys %$new_info) {
        if (defined $new_info->{$key} && (! defined $old_info->{$key} || $old_info->{$key} ne $new_info->{$key})) {
            $old_info->{$key} = $new_info->{$key};
            $changed_flag++;
        }
    }
    return 0 unless $changed_flag;

    # передаём всегда директовский ServiceID, т.к. клиенты по валюте общие во всех сервисах, живущих у нас
    $old_info->{SERVICE_ID} = $Settings::SERVICEID{direct};

    my ($error, undef) = balance_create_client($operator_uid, $old_info);

    return $error ? 1 : 0;
}


=head2 get_client_id_list

 from client.yandex.ru/Functions.pm/ShowClientIdList()

 return 0 - if all ok
 return 1 - if error

=cut

sub get_client_id_list {
    my $vars = shift;
    my $policy = 1; # все
    if ( defined $vars->{from} && $vars->{from} eq 'createAgency' ) {
        $policy = 3; # только агентства
    } elsif (defined $vars->{for_agency}) {
        $policy = 1; # all (with subclients)
    } elsif ( defined $vars->{from} && $vars->{from} eq 'stepZeroProcess' ) {
        $policy = 2; # прямые клиенты
    }
    my $hash = { AgencySelectPolicy => $policy,
                Name =>  $vars->{'name'} || undef,
                Phone => $vars->{'phone'} || undef,
                Fax   => $vars->{'fax'} || undef,
                Email => $vars->{'email'} || undef,
                Url   => $vars->{'url'} || undef,
                ClientID => $vars->{'ClientID'} || undef,
                PrimaryClients => 1
            };
    return 1 unless $hash->{Name} || $hash->{Phone} || $hash->{Fax} || $hash->{Email} || $hash->{Url} || $hash->{ClientID};

    my $clients = eval { balance_find_client($hash); };

    if ($@) {
        warn "get_client_id_list: balance_fine_client fault - $@";
        return 1;
    }

    foreach my $el ( @$clients ) {
        push @{$vars->{client}}, hash_cut($el, qw/NAME PHONE FAX URL EMAIL AGENCY_ID IS_AGENCY CLIENT_ID/);
    }
    $hash->{Name} = $hash->{Name};
    $vars->{clients} = $hash;

    return 0;
}


=head2 create_client_in_balance

    создаем клиента в балансе
    create_client_in_balance($UID, $uid, %O);

    при создании нового клиента в Директе, нельзя передавать в эту функцию ClientID, даже если он уже известен
    иначе мы не отправим в Баланс страну и валюту

=cut

sub create_client_in_balance($$;%);
sub create_client_in_balance($$;%)
{
    my ($UID, $uid, %O) = @_;

    my $data = hash_cut \%O, qw/ClientID FIO phone email initial_country initial_currency initial_subregion gdpr_agreement_accepted_time is_touch/;

    my $main_client_id;
    my $user_is_not_created = 1;

    if (!$data->{ClientID}) {

        $main_client_id = get_clientid_by_uid($uid);

        if (! $main_client_id) {
            my $params = {
                name => $data->{FIO},
                phone => $data->{phone},
                email => $data->{email},
                client_uid => $uid,
            };

            if ($data->{initial_country}) {
                $params->{country_region_id} = $data->{initial_country};
            }
            # multicurrency: при полном открытии мультивалютности (когда перестанем давать создавать уешных клиентов в том числе для Баяна и Геоконтекста) нужно будет умирать при отсутствии валюты у нового клиента, т.к. это нештатный случай
            $params->{currency} = $data->{initial_currency} if $data->{initial_currency};
            $params->{subregion_id} = $data->{initial_subregion} if $data->{initial_subregion};

            $params->{agency} = $O{agency} if $O{agency};

            create_client_id($params, $UID) && croak "Can't create client";
            $main_client_id = $params->{new_clientID} || croak "Undefined main_client_id";
            create_client_id_association($uid, $main_client_id, $UID) || croak "Can't create client-uid association";

        } elsif ($main_client_id && ($data->{initial_currency} || $data->{initial_country} || $data->{initial_subregion})) {
            # если у нас есть initial_(country|currency|subregion) и при этом нет записи в clients,
            # обновляем страну и валюту в Балансе вне зависимости от наличия ClientID
            # такое может быть, если клиент имел денежные отношения с Балансом вне Директа (клиент Маркета, например)
            my $client_data = get_client_data($main_client_id, [qw/country_region_id subregion_id work_currency/]);
            if (!$client_data ||
                    ( !$client_data->{country_region_id} && $data->{initial_country} ) ||
                    ( !$client_data->{work_currency} && $data->{initial_currency} ) ||
                    ( !$client_data->{subregion_id} && $data->{initial_subregion} )
            ) {
                my %new_params;
                $new_params{REGION_ID} = $data->{initial_country} if $data->{initial_country};
                $new_params{SUBREGION_ID} = $data->{initial_subregion} if $data->{initial_subregion};
                if ($data->{initial_currency} && $data->{initial_currency} ne 'YND_FIXED') {
                    $new_params{CURRENCY} = $data->{initial_currency};
                    $new_params{MIGRATE_TO_CURRENCY} = get_client_migrate_to_currency($main_client_id);
                }
                my $is_error = update_client_id($UID, $main_client_id, \%new_params);
                die "Error updating ClientID $main_client_id in Balance" if $is_error;
            }
        }

    } else {
        $main_client_id = $data->{ClientID};
        $user_is_not_created = undef;
    }

    if (!defined get_shard(ClientID => $main_client_id) ) {
        my $shard = get_new_available_shard($main_client_id, $UID);
        save_shard(ClientID => $main_client_id, shard => $shard);
    }

    if ($data && ($data->{initial_country} || $data->{initial_currency} || $data->{initial_subregion})) {
        # фиксируем в таблице clients валюту клиента при создании его в Балансе
        my %new_client_data = (
            ClientID => $main_client_id,
            name     => $data->{FIO},
            is_touch => !!$data->{is_touch},
            tin      => $O{tin},
            tin_type => $O{tin_type},
        );
        $new_client_data{role} = $O{role} if exists $O{role};
        if ($O{chief_uid}) {
            $new_client_data{chief_uid} = $O{chief_uid};
        } elsif (! get_one_field_sql(PPC(ClientID => $main_client_id), "SELECT 1 FROM clients WHERE ClientID = ?", $main_client_id)) {
            # если клиент только создаётся - прописываем первого uid как шефа
            $new_client_data{chief_uid} = $uid;
        }
        $new_client_data{country_region_id} = $data->{initial_country} if defined $data->{initial_country};
        $new_client_data{work_currency} = $data->{initial_currency} if defined $data->{initial_currency};
        # устанавливаем флаг квазивалюты для клиентов в тенге, чтобы кампании создавались с правильным ProductID
        $new_client_data{is_using_quasi_currency} = 1 if (defined $data->{initial_currency} && $data->{initial_currency} eq 'KZT');
        $new_client_data{subregion_id} = $data->{initial_subregion} if defined $data->{initial_subregion};
        $new_client_data{user_is_not_created} = $user_is_not_created;
        $new_client_data{gdpr_agreement_accepted_time} = $data->{gdpr_agreement_accepted_time} if $data->{gdpr_agreement_accepted_time};
        create_update_client({client_data => \%new_client_data});
    }

    do_insert_into_table(PPC(ClientID => $main_client_id), 'clients_to_fetch_nds', {ClientID => $main_client_id}, ignore => 1);

    Yandex::Log
      ->new(%Yandex::Balance::BALANCE_CALLS_LOG_SETTINGS, msg_prefix => 'create_new_client_balance:')
      ->out("save ClientID: $main_client_id, uid: $uid, operator_uid: $UID");

    return $main_client_id;
}

=head2 update_deleted_reps

    Обновляет список удаленных представителей.
    Если передано add - добавляет в список, если remove - удаляет из списка удаленных представителей.

    На входе:
        client_id - ClientID главного представителя
        what - действия, которые требуется совершить:
            add - добавить в список, значение - массив HASHREF с данными о представителях. Данные должны содержать
                  поля uid, login, fio, email, phone
            remove - удалить из списка, значение - массив UIDов.

    Пример:
        update_deleted_reps($ClientID, {add => [$user_data1, $user_data2]})    # add uid to deleted_reps, after delete rep
        update_deleted_reps($ClientID, {remove => [$uids]}) # remove uid from deleted_reps, after re-added rep

=cut

sub update_deleted_reps
{
    my ($client_id, $what) = @_;

    my %deleted_reps =  map {$_->{uid} => $_} @{(get_client_data($client_id, [qw/deleted_reps/]) || {})->{deleted_reps}};

    while (my ($cmd, $uids) = each %$what) {
        if($cmd eq 'add'){
            foreach(@$uids){
                $deleted_reps{$_->{uid}} = $_;
            }
        }
        if($cmd eq 'remove'){
            foreach(@$uids){
                delete $deleted_reps{$_} if $deleted_reps{$_};
            }
        }
    }

    create_update_client({
                          client_data => {
                                ClientID => $client_id,
                                deleted_reps => to_json([values %deleted_reps])
                          }
                        });
}

=head2 update_role(ClientID, role, $subrole)

    Безусловно прописать роль на ClientID в табличке clients

=cut
sub update_role($$$) {
    my ($ClientID, $role, $subrole) = @_;
    _check_role_subrole($role, $subrole);

    create_update_client({client_data => { ClientID => $ClientID, role => $role, subrole => $subrole }});
}

# проверяем соответствие роли и под-роли
sub _check_role_subrole($$) {
    my ($role, $subrole) = @_;
    if (!defined $role) {
        croak "Subrole should be undefined for undefined role: $subrole" if defined $subrole;
    } elsif (!exists $ROLES{$role}) {
        croak "Unknown role: $role";
    } else {
        if (defined $subrole) {
            if (!defined $SUBROLE2ROLE{$subrole}) {
                croak "Unknown subrole: $subrole";
            } else {
                croak "Incorrect subrole: $subrole for role: $role" if $SUBROLE2ROLE{$subrole} ne $role;
            }
        }
    }
}

=head2 on_campaign_creation(ClientID, agency_uid, manager_uid)

    Вызывается после создания кампании.
    Если роль клиента empty - проставляем client
    Если кампания агентская - прописываем клиенту агентство
    Если кампания сервисируемая - прописываем клиенту менеджера

=cut
sub on_campaign_creation {
    my ( $client_id, $agency_uid, $manager_uid ) = @_;

    my %up;

    my $perminfo = Rbac::get_perminfo(ClientID => $client_id);

    if (!$perminfo->{role} || $perminfo->{role} eq 'empty') {
        $up{role} = 'client';
    }

    if ($agency_uid && (!defined $perminfo->{agency_uid} || $perminfo->{agency_uid} != $agency_uid)) {
        my $ag_perminfo = Rbac::get_perminfo(uid => $agency_uid);
        $up{agency_client_id} = $ag_perminfo->{ClientID};
        $up{agency_uid} = $ag_perminfo->{rep_type} eq 'limited' ? $agency_uid : $ag_perminfo->{chief_uid};
    }

    if (%up) {
        create_update_client({
            client_data => {
                ClientID => $client_id,
                %up
            }
        });
    }
}

=head2 update_client_chief(ClientID, new_chief_uid)

    Безусловно прописать нового шефа клиенту (или агентству)
    Старому шефу проставится тип main

=cut
sub update_client_chief($$) {
    my ($ClientID, $new_chief_uid) = @_;
    my $new_chief_ClientID = get_clientid(uid => $new_chief_uid);
    if (!$new_chief_ClientID) {
        croak "New chief must be created before call of update_client_chief";
    } elsif ($new_chief_ClientID != $ClientID) {
        croak "New chief must be rep of the same client: ($ClientID != $new_chief_ClientID)";
    }
    my $old_chief_uid = get_chief(ClientID => $ClientID);
    do_in_transaction {
            if ($old_chief_uid != $new_chief_uid) {
                do_update_table(PPC(ClientID => $ClientID), "users", {rep_type => 'main'}, where => {uid => $old_chief_uid});
            }
            do_update_table(PPC(ClientID => $ClientID), "users", {rep_type => 'chief'}, where => {uid => $new_chief_uid});
            do_update_table(PPC(ClientID => $ClientID), "clients", {chief_uid => $new_chief_uid}, where => {ClientID => $ClientID});
        };
}

=head2 create_update_client

  Добавляет, или обновляет запись в таблицах clients и agency_client_relations,
  обновляет поле lastChange в таблице users, если надо

    create_update_client({
        client_data => {
            ClientID =>
            role =>
            [...]
        },
        agency_client_relations_data => {
            agency_client_id =>
            client_client_id =>
            [...]
        }
    });

    В случае, если эту процедуру будут менять в будущем, следует иметь в виду, что
    массовая архивация клиентов агентства в обход этой процедуры есть в InternalReports::mass_archive_clients_for_agency

=cut

sub create_update_client
{

    my ($client_data, $ac_relation_data) = @{$_[0]}{qw/client_data agency_client_relations_data/};
    if ($client_data && scalar keys %{$client_data}) {

        my $client_id = $client_data->{ClientID};
        if( $client_id ){
            if (defined $client_data->{role} || defined $client_data->{subrole}) {
                _check_role_subrole($client_data->{role}, $client_data->{subrole});
            }
            if (!defined $client_data->{role} && !is_client_exists($client_id)) {
                $client_data->{role} = 'empty';
            }

            foreach ( keys %CLIENT_TABLES ) {
                update_client_table($client_id, $_, $client_data);
            }

            if (($client_data->{role} // '') eq 'manager') {
                my $uid = $client_data->{chief_uid} // get_chief(ClientID => $client_id);
                add_manager_in_hierarchy($client_id, $uid);
            }

            if ($client_data->{user_is_not_created} ) {
                do_update_table(PPC(ClientID => $client_id), 'users', {LastChange__dont_quote =>'NOW()'}, where => {ClientID => SHARD_IDS});
            }
        }
    }
    if ($ac_relation_data) {
        my $ac_relation_data = hash_copy({}, $ac_relation_data, qw/
                                                                  agency_client_id
                                                                  client_client_id
                                                                  bind
                                                                  client_archived
                                                                  client_description
                                                                  /);
        if (scalar keys %{$ac_relation_data} && $ac_relation_data->{client_client_id} && $ac_relation_data->{agency_client_id}) {
            do_insert_into_table(PPC(ClientID => $ac_relation_data->{client_client_id}), 'agency_client_relations'
                            , $ac_relation_data
                            , on_duplicate_key_update => 1
                            , key => ['agency_client_id', 'client_client_id']);

            do_update_table(PPC(ClientID => $ac_relation_data->{client_client_id}), 'users', {LastChange__dont_quote =>'NOW()'}, where =>{ClientID => SHARD_IDS});
        }
    }
}

# --------------------------------------------------------------------

=head2 get_overdraft_info

    Получаем информацию об овердрафте, бюджете и порогам скидок по единственному клиенту

    $res = get_overdraft_info(
        $client_id,
        client_discount => get_client_discount($client_id),
        client_nds => get_client_NDS($client_id),
        client_currencies => get_client_currencies($client_id),
    );
    $res->{debt} -- сколько клиент уже должен; в валюте клиента без НДС
    $res->{overdraft_rest} -- размер доверительного платежа, который может внести клиент; в валюте клиента без НДС
    $res->{nextPayDate} -- дата следующего платежа (2007-10-21 00:00:00)
    $res->{nextPayDateText} -- дата следующего платежа словами (21 октября 2007г.)
    $res->{dateFlag} -- 'Future', 'Present' или 'Past' в зависимости от того, находится ли nextPayDate в будущем, настоящем или прошлом; для того, чтобы в интерфейсе выводить особое сообщение для просроченных платежей
    $res->{budget}
    $res->{discount}
    $res->{border_next}
    $res->{discount_next}
    $res->{border_prev}

=cut

sub get_overdraft_info {
    my ($client_id, %O) = @_;

    return get_mass_overdraft_info(
        [$client_id],
        clients_discount => {$client_id => $O{client_discount}},
        clients_nds => {$client_id => $O{client_nds}},
        clients_currencies => {$client_id => $O{client_currencies}},
    )->{$client_id};
}

sub _get_autooverdraft_params {
    my ($client_id) = @_;

    my $auto_overdraft_params = get_one_line_sql(PPC(ClientID => $client_id),
        ["SELECT overdraft_lim, debt, auto_overdraft_lim, is_brand, IF(statusBalanceBanned = 'Yes' OR overdraft_lim = 0, 1, 0) AS is_banned FROM clients_options",
          where => { ClientID => $client_id }]);
    $auto_overdraft_params->{auto_overdraft_limit_default_min_value} =
        Property->new($AUTO_OVERDRAFT_MIN_VALUE_PROPERTY_NAME)->get() // $AUTO_OVERDRAFT_MIN_DEFAULT_VALUE;
    return $auto_overdraft_params;
}

=head2 get_mass_overdraft_info

    Получаем информацию об овердрафте, бюджете и порогам скидок по нескольким клиентам

    $clientid2overdraft = get_mass_overdraft_info(
        \@client_ids,
        clients_discount => mass_get_client_discount(\@client_ids),
        clients_nds => mass_get_client_NDS(\@client_ids),
        clients_currencies => mass_get_client_currencies(\@client_ids),
    );
    $clientid2overdraft => {
        $client_id1 => {
            debt => # сколько клиент уже должен; в валюте клиента без НДС
            overdraft_rest => # размер доверительного платежа, который может внести клиент; в валюте клиента без НДС
            nextPayDate => # дата следующего платежа (2007-10-21 00:00:00)
            nextPayDateText => # дата следующего платежа словами (21 октября 2007г.)
            dateFlag => # 'Future', 'Present' или 'Past' в зависимости от того, находится ли nextPayDate в будущем, настоящем или прошлом; для того, чтобы в интерфейсе выводить особое сообщение для просроченных платежей
            budget
            discount
            border_next
            discount_next
            border_prev
        },
        ...
    };

=cut

sub get_mass_overdraft_info {
    my ($client_ids, %O) = @_;

    my %result;

    my $clientid2info = get_hashes_hash_sql(PPC(ClientID => $client_ids),
                                      ["SELECT ClientID
                                           , overdraft_lim - debt AS overdraft_rest
                                           , overdraft_lim
                                           , auto_overdraft_lim
                                           , debt
                                           , nextPayDate
                                           , budget, discount
                                           , border_next, discount_next
                                           , border_prev
                                        FROM clients_options",
                                        where => { ClientID => SHARD_IDS }]);

    my $client_discounts = $O{clients_discount};
    die 'no clients_discount given' unless $client_discounts;

    my $clients_nds = $O{clients_nds};
    die 'no clients_nds given' unless $clients_nds;

    my $clients_currencies = $O{clients_currencies};
    die 'no client currency given' unless $clients_currencies;

    foreach my $client_id (@$client_ids) {
        my $res={overdraft_rest => 0,
                debt           => 0,
                nextPayDate    => {dd => 0, mm => 0, yyyy => 0000},
                nextPayDateText => '',
                discount => ($client_discounts->{$client_id} || 0),
                };

        if (!exists $clientid2info->{$client_id}) {
            $result{$client_id} = $res;
            next;
        }

        $res = $clientid2info->{$client_id};
        if ( ($res->{discount} || 0) != ($client_discounts->{$client_id} || 0) ) {
            # если скидка в overdraft отличается от сегодняшней скидки в графике, то
            # считаем данные в overdraft устаревшими и не отдаём их
            # такое может быть в начале суток при изменении скидки, т.к. график приезжает заранее,
            # а данные в overdraft забираем по состоянию на момент забора
            hash_merge $res, {
                budget => undef,
                discount => ($client_discounts->{$client_id} || 0),
                border_next => undef,
                discount_next => undef,
                border_prev => undef,
            };
        }

        # в базе овердрафт и долг хранятся с НДС, а клиентам показываем без НДС
        if ($clients_currencies->{$client_id}->{work_currency} ne 'YND_FIXED') {
            for my $field (qw/overdraft_rest debt overdraft_lim auto_overdraft_lim/) {
                next unless $res->{$field};
                $res->{$field} = Currencies::remove_nds($res->{$field}, $clients_nds->{$client_id} // 0); # отсутствие НДС у клиента здесь нормальная ситуация
            }
        }

        #Если долг -- целое число, то показывать надо без дробной части (200 у.е.). Если долг дробный -- с двумя знаками (199.55 у.е., 200.50 у.е.)
        # multicurrency: очень подозрительная, но пока приемлемая логика
        my $debt  = $res->{debt}||0;
        $res->{debt} = abs($debt - int($debt)) > 0.001 ? sprintf("%.2f", $debt) : int($debt);

        my ($yyyy, $mm, $dd) = (0, 0, 0);
        if ($res->{nextPayDate} && $res->{nextPayDate} =~ m/^(\d\d\d\d)-(\d\d)-(\d\d)/) {
            ($yyyy, $mm, $dd) = ($1+0, $2+0, $3+0);
        }

        #Дальше обрабатываем только правильную дату
        if ( $yyyy && $mm && $dd ) {
            $res->{nextPayDateText} = sprintf("%02d.%02d.%04d", $dd, $mm, $yyyy);

            #Определяем, дата оплаты уже прошла, еще будет, или она -- сегодня
            my ($cur_sec, $cur_min, $cur_hour, $cur_day, $cur_mon, $cur_year, $cur_wday, $cur_yday, $cur_isdst) = localtime();
            $cur_year += 1900;
            my $last_midnight = timelocal( 0, 0, 0,$cur_day,$cur_mon,$cur_year);
            my $next_midnight = timelocal(59,59,23,$cur_day,$cur_mon,$cur_year);

            my $next_pay = timelocal(0, 0, 10, $dd, $mm-1, $yyyy);
            if ($next_pay > $next_midnight) {
                $res->{dateFlag} = 'Future';
            } elsif ($next_pay < $last_midnight) {
                $res->{dateFlag} = 'Past';
            } else {
                $res->{dateFlag} = 'Present';
            }
        }

        $result{$client_id} = $res;
    }

    return \%result;
}

=head2 mass_get_client_currencies

    Возвращает информацию о валютах сразу нескольких клиентов.
    Не умеет работать с валютами пользователей, которые ещё не созданы в Балансе.

    $client2currencies = mass_get_client_currencies([$client_id1, $client_id2, ...]);
    $client2currencies => {
        $client_id1 => {
            work_currency => 'USD',
        },
        $client_id2 => {},
        ...
    };

=cut

sub mass_get_client_currencies {
    my ($client_ids) = @_;

    die 'invalid client_ids array' unless $client_ids && ref($client_ids) eq 'ARRAY';
    return {} unless @$client_ids;

    for my $client_id (@$client_ids) {
        die 'invalid ClientID: ' . str($client_id) unless $client_id && is_valid_int($client_id, 0);
    }

    my $client2currencies = {};
    if (@$client_ids) {
        my $clients = get_all_sql(PPC(ClientID => $client_ids),
            ['SELECT ClientID, IFNULL(work_currency, "YND_FIXED") AS work_currency FROM clients',
                WHERE => {ClientID => SHARD_IDS}]);
        for my $client (@$clients) {
            $client2currencies->{ $client->{ClientID} } = hash_cut $client, qw/work_currency/;
        }
        for my $client_id (@$client_ids) {
            if (!exists $client2currencies->{ $client_id }) {
                # currency_defaults: проставляем YND_FIXED тем клиентам, про которых нет записей
                $client2currencies->{ $client_id } = {
                    work_currency => 'YND_FIXED',
                };
            }
        }
    }

    return $client2currencies;
}

=head2 get_client_currencies

    Возвращает информацию о валютах пользователя.
    Берёт информацию из clients.

    Принимает обязательный позиционный параметр:
        ClientID

    Возвращает ссылку на хеш с ключами:
        work_currency -- валюта, в которой работает клиент

    $client_currencies = get_client_currencies($client_id); # определяем валюту существующего в Балансе клиента
    $client_currencies => {
        work_currency => 'USD',
    };

=cut

sub get_client_currencies
{
    my ($client_id, %O) = @_;

    my $client_currencies;
    my @currency_fields = qw/work_currency/;

    if ($client_id && is_valid_int($client_id, 0)) {
        $client_currencies = mass_get_client_currencies([$client_id])->{$client_id};
    } elsif ($O{allow_initial_currency}) {
        # TODO: После успешной выкладки DIRECT-21051 этот блок if следует удалить.  И удалить все использования флага allow_initial_currency в коде.
        die 'Invalid uid given: ' . str($O{uid}) unless is_valid_int($O{uid}, 0);
        my $user_options = User::get_user_options($O{uid});
        if ($user_options->{initial_currency}) {
            $client_currencies->{$_} = $user_options->{initial_currency} for @currency_fields;
        } else {
            # Для того чтобы понимать, есть ли у нас такие дыры, что можно попасть в этот блок.
            my $msg = "User without ClientID and without initial_currency in user options was found";
            send_alert(Carp::longmess($msg), 'get_client_currencies without currency');
        }
    } else {
        die 'Invalid ClientID given: ' . str($client_id);
    }

    $client_currencies->{$_} ||= 'YND_FIXED' for @currency_fields; # currency_defaults
    return $client_currencies;
}

=head2 get_limited_clients

    Список клиентов, на которых распространяются персональные ограничения

=cut

sub get_limited_clients {

    my $limited = get_all_sql(PPC(shard => 'all'), [
        'SELECT', sql_fields('u.login', 'l.ClientID', map {"l.$_"} keys %CLIENT_LIMIT_FIELDS),
        'FROM client_limits l
        LEFT JOIN users u ON l.ClientID = u.ClientID'
    ]);

    my %clients;
    foreach my $limit (@$limited) {
        unless (exists $clients{$limit->{ClientID}}) {
            $clients{$limit->{ClientID}} = {
                users => [],
                map {
                    ($_ => $limit->{$_})
                }
                ('ClientID', keys %CLIENT_LIMIT_FIELDS)
            }
        }
        push @{$clients{$limit->{ClientID}}->{users}}, $limit->{login}
    }

    return [values %clients]
}

=head2 get_client_NDS

    Возвращает информацию о значении НДС клиента.
    Берёт информацию из client_nds + опционально умеет полчать отсутствующее из Баланса по ключу fetch_missing_from_balance.

    Возвращает значение НДС в процентрах в виде скаляра (например, 18).
    Если об НДС совсем ничего не известно, возвращает undef.

    $client_nds_value = get_client_NDS($client_id);
    $client_nds_value = get_client_NDS($client_id, fetch_missing_from_balance => 1, rbac => $rbac, fetch_for_ynd_fixed_too => 1);
    # $client_nds_value == 18

=cut

sub get_client_NDS
{
    my ($client_id, %O) = @_;

    die 'Wrong ClientID: '.str($client_id) unless is_valid_int($client_id, 0);

    return mass_get_client_NDS([$client_id], %{hash_cut \%O, qw/fetch_missing_from_balance rbac fetch_for_ynd_fixed_too/})->{$client_id};

}

=head2 mass_get_client_NDS

    Возвращает информацию о значениях НДС в процентах для нескольких клиентов.
    Берёт информацию из client_nds + опционально умеет полчать отсутствующее из Баланса по ключу fetch_missing_from_balance.

    $client_nds_values = mass_get_client_NDS([$client_id1, $client_id2, ...]);
    $client_nds_values = mass_get_client_NDS([$client_id1, $client_id2, ...], fetch_missing_from_balance => 1, fetch_for_ynd_fixed_too => 1);
    $client_nds_values => {
        $client_id1 => $nds_value1,
        $client_id2 => $nds_value2,
        ...
    }

=cut

sub mass_get_client_NDS
{
    my ($client_ids, %O) = @_;

    die 'wrong ClientIDs array' unless $client_ids && ref($client_ids) eq 'ARRAY';
    return {} unless @$client_ids;

    my $clientid2agencyid = _get_clientid2agencyid($client_ids);
    # выбираем из базы НДС для неагентстких клиентов + агентств
    my $client_ids_to_fetch = xminus($client_ids, [keys %$clientid2agencyid]);
    push @$client_ids_to_fetch, values %$clientid2agencyid;

    my $current_date = today();
    my $resulting_client_nds = get_hash_sql(PPC(ClientID => $client_ids_to_fetch),
        ['SELECT ClientID, nds FROM client_nds', WHERE => {ClientID => SHARD_IDS, 'date_to__ge' => $current_date, 'date_from__le' => $current_date}]);
    if ($O{fetch_missing_from_balance}) {
        my $missing_clientids = xminus $client_ids_to_fetch, [keys %$resulting_client_nds];
        if ($missing_clientids && @$missing_clientids) {
            $missing_clientids = [uniq @$missing_clientids];
            my $missing_client_currencies = mass_get_client_currencies($missing_clientids);
            if (!$O{fetch_for_ynd_fixed_too}) {
                # графики НДС по уешным клиентам нас не интересуют — у них НДС ни на что не влияет
                $missing_clientids = [grep { $missing_client_currencies->{$_}->{work_currency} ne 'YND_FIXED' } @$missing_clientids];
            }
            if ($missing_clientids && @$missing_clientids) {
                my $log = Yandex::Log::Messages->new();
                $log->msg_prefix("[mass_get_client_NDS]");
                $log->out('mass_get_client_NDS', 'Fetching missing NDS from mass_get_client_NDS for following ClientIDs: '.$missing_clientids);
                Client::NDSDiscountSchedule::sync_nds_schedule_for_clients($missing_clientids, log => $log);
                my $missing_client_nds_data = mass_get_client_NDS($missing_clientids, fetch_missing_from_balance => 0);
                hash_merge $resulting_client_nds, $missing_client_nds_data;
            }
        }
    }

    my %result;
    for my $client_id (@$client_ids) {
        my $client_id_for_nds = (exists $clientid2agencyid->{$client_id}) ? $clientid2agencyid->{$client_id} : $client_id;
        if (exists $resulting_client_nds->{$client_id_for_nds}) {
            $result{$client_id} = $resulting_client_nds->{$client_id_for_nds};
        }
    }

    return \%result;
}

sub _get_clientid2agencyid {
    my ($client_ids) = @_;

    # multicurrency-agency-nds: по информации от ppalex, это не костыль, а исторически устоявшееся решение
    # исходно предполагалось, что у клиента один НДС. но клиент может быть свободен и НДС у него
    # будет зависеть от кампании (и того, кто оплачивает эту кампанию). пока не готова
    # полноценная поддержка определения НДС по кампании, поставили костыли:
    # - запретили образовываться мультивалютным свободным клиентам;
    # - если у клиента есть хоть одна агентская кампаний, возвращаем по нему НДС агентства
    # Эти костыли "чинят" учёт НДС для несвободных клиентов: за менеджерские и самостоятельные
    # кампании клиент платит сам со своим НДСом, за агентские достаём НДС агентства костылём.
    # Смеси из разных типов обслуживания не может быть, т.к. для этого требуется свобода, а её не даём.

    return get_hash_sql(PPC(ClientID => $client_ids), [q/
        SELECT DISTINCT u.ClientID, c.AgencyID
        FROM users u
        INNER JOIN campaigns c ON u.uid = c.uid
        LEFT JOIN clients_options co ON u.ClientID = co.ClientID
     /, WHERE => {
           'u.ClientID' => SHARD_IDS,
           'c.type' => get_camp_kind_types('with_currency'),
           'c.statusEmpty' => 'No',
           'c.AgencyID__gt' => 0,
           '_OR' => {'co.non_resident__is_null' => 1, 'co.non_resident' => 0},
        },
    ]);
}

=head2 mass_get_client_nds_schedule

    Массовый метод получения графика НДС для указанных клиентов
    Для каждого клиента возвращается массив хэшей вида
    {
        date_from  => '20190101',
        date_to    => '20380119',
        nds        => '20.0000'
    }
    с данными из таблицы client_nds.

=cut

sub mass_get_client_nds_schedule {
    my ($client_ids) = @_;

    my $clientid2agencyid = _get_clientid2agencyid($client_ids);
    # выбираем из базы НДС для неагентстких клиентов + агентств
    my $client_ids_to_fetch = xminus($client_ids, [keys %$clientid2agencyid]);
    push @$client_ids_to_fetch, values %$clientid2agencyid;

    my $client_nds_data = get_all_sql(PPC(ClientID => $client_ids_to_fetch), [
           "SELECT ClientID
                 , DATE_FORMAT(date_from, '%Y%m%d') AS date_from
                 , DATE_FORMAT(date_to,'%Y%m%d') AS date_to
                 , nds
              FROM client_nds
          ", WHERE => { ClientID => SHARD_IDS },
         "ORDER BY ClientID, date_from"
    ]);

    my %client_nds_schedules;
    foreach my $row (@$client_nds_data) {
        push @{ $client_nds_schedules{ $row->{ClientID} } }, hash_cut($row, qw/date_from date_to nds/);
    }

    my %result;
    for my $client_id (@$client_ids) {
        my $client_id_for_nds = (exists $clientid2agencyid->{$client_id}) ? $clientid2agencyid->{$client_id} : $client_id;
        $result{$client_id} = $client_nds_schedules{$client_id_for_nds} if exists $client_nds_schedules{$client_id_for_nds};
    }

    return \%result;
}

=head2 get_client_discount

    Скидки устарели (DIRECT-70057), данные не обновляются - значения могут быть только 0 или отсутствовать

    Возвращает информацию о текущем значении скидки клиента.
    Берёт информацию из client_discounts.

    Возвращает значение скидки в процентрах в виде скаляра (например, 10).
    Если об скидке ничего не известно или она нулевая, возвращает undef.

    $client_discount_value = get_client_discount($client_id);
    # $client_discount_value == 10

=cut

sub get_client_discount
{
    my ($client_id) = @_;

    die 'Wrong ClientID: '.str($client_id) unless is_valid_int($client_id, 0);

    return mass_get_client_discount([$client_id])->{$client_id};
}

=head2 mass_get_client_discount

    Скидки устарели (DIRECT-70057), данные не обновляются - значения могут быть только 0 или отсутствовать

    Возвращает информацию о текущих значениях скидок нескольких клиентов.
    Берёт информацию из client_discounts.

    $client_discount_values = mass_get_client_discount([$client_id1, $client_id2, ...]);
    $client_discount_values = mass_get_client_discount([$client_id1, $client_id2, ...], date => '20150831');
    $client_discount_values => {
        $client_id1 => $discount_value1,
        $client_id2 => $discount_value2,
        ...
    }

=cut

sub mass_get_client_discount
{
    my ($client_ids, %O) = @_;

    die 'wrong ClientIDs array' unless $client_ids && ref($client_ids) eq 'ARRAY';
    return {} unless @$client_ids;

    my $current_date = $O{date} || today();
    return get_hash_sql(PPC(ClientID => $client_ids), ['SELECT ClientID, discount FROM client_discounts', WHERE => {ClientID => SHARD_IDS, 'date_to__ge' => $current_date, 'date_from__le' => $current_date}]) || {};
}

=head2 get_client_limits

    Возвращает, ограничения, накладываемы на указанного клиента (по ClientID).

    $client_limits = get_client_limits($client_id);
    $clint_limit = {
        camp_count_limit => 1000, # максимальное количество кампаний
        unarc_camp_count_limit => 3000, # максимальное количество незаархивированных кампаний
        feed_count_limit => 50, # максимальное  количество фидов
        feed_max_file_size => 200000000,
    };

    ВАЖНО:
    banner_count_limit # максимальное количество групп на кампанию

=cut

sub get_client_limits {
    my ($client_id) = @_;
    $client_id //= 0;

    my $limits = $CACHE->get($client_id);
    return $limits  if $limits;

    $limits = get_one_line_sql(PPC(ClientID => $client_id), [
        'SELECT', sql_fields(keys %CLIENT_LIMIT_FIELDS),
        'FROM client_limits',
        WHERE => { ClientID => $client_id },
    ]);

    {
        no strict 'refs';
        foreach my $f (keys %CLIENT_LIMIT_FIELDS) {
           $limits->{$f} ||= ${"Settings::DEFAULT_" . uc $f};
        }
    }

    $CACHE->set($client_id => $limits);

    return $limits;
}


=head2 validate_client_limits($limits)

    Валидация ограничений на клиента

=cut

{

sub validate_client_limits {

    my $limits = shift;

    my @errors;
    my $text = iget('неверно указано ограничение на %s');
    my $bound = iget('необходимый диапазон от %i до %i');
    foreach my $f (keys %CLIENT_LIMIT_FIELDS) {
        next unless exists $limits->{$f};
        unless (defined $limits->{$f}
                && (is_valid_int($limits->{$f}, @{$CLIENT_LIMIT_FIELDS{$f}->{bounds}})
                    || is_valid_int($limits->{$f}) && $limits->{$f} == 0)) {

            my $t = sprintf $text, iget ($CLIENT_LIMIT_FIELDS{$f}->{name});
            $t .= '; ' . sprintf $bound, @{$CLIENT_LIMIT_FIELDS{$f}->{bounds}} if @{$CLIENT_LIMIT_FIELDS{$f}->{bounds}} > 1;
            push @errors, $t;
        }
    }

    if (!@errors && exists($limits->{camp_count_limit})
        && exists($limits->{unarc_camp_count_limit})
        && $limits->{unarc_camp_count_limit} > $limits->{camp_count_limit}) {

        push @errors, iget('ограничение на число незаархивированных кампаний не должно превышать ограничение на общее число кампаний')
    }

    return @errors;
}
}

=head2 set_client_limits($clientid, $limits)

    Сохранение органичений для клиента

    $limits:
        camp_count - число кампаний
        unarc_camp_count - число незаархивированных кампаний
        banner_count - число объявлений в каждой кампании

=cut

sub set_client_limits {

    my ($clientid, $limits) = @_;

    my %row = (ClientID => $clientid);
    foreach (keys %CLIENT_LIMIT_FIELDS) {
        $row{$_} = $limits->{$_} if defined $limits->{$_}
    }

    $CACHE->remove($clientid);

    do_insert_into_table(PPC(ClientID => $clientid), 'client_limits', \%row,
        on_duplicate_key_update => 1, key => 'ClientID',
    );
}


=head2 delete_client_limits($clientid)

Удаление ограничений для клиента

=cut

sub delete_client_limits {
    my ($clientid) = @_;

    $CACHE->remove($clientid);
    do_delete_from_table(PPC(ClientID => $clientid), 'client_limits', where => {ClientID => SHARD_IDS});
    return;
}


=head2 get_client_campaign_count

    Возвращает общее число кампаний и число незаархивированных кампаний клиента(ов) (по ClientID)
    БЕЗ учета:
        кампаний надтипа skip_in_client_campaign_count
        старых кампаний в у.е. (до конвертации)

    $client_camp_counts = get_client_campaign_count( $client_id );
    $client_camp_counts = get_client_campaign_count( [$client_id1, $client_id2] );
    $client_camp_counts = {
        total_count => 15,    # общее число кампаний
        unarc_count => 9,     # число незаархивированных кампаний
    };

=cut

sub get_client_campaign_count {
    my ($client_ids) = @_;

    my $counts = get_all_sql(PPC(ClientID => $client_ids), [ q/
        SELECT
            COUNT(*) AS total_count,
            COUNT(IF(c.archived = 'No', 1, NULL)) AS unarc_count
        FROM users u
        INNER JOIN campaigns c ON u.uid = c.uid
     /, WHERE => {
            'u.ClientID' => $client_ids,
            'c.statusEmpty' => 'No',
            _NOT => [
                _OR => [
                    'c.type' => get_camp_kind_types('skip_in_client_campaign_count'),
                    _AND => [
                        # NB! рублевые, сконвертированные некопированием - тоже currencyConverted=Yes
                        'c.currencyConverted' => "Yes",
                        _TEXT => 'IFNULL(c.currency, "YND_FIXED") = "YND_FIXED"',
                    ],
                ],
            ],
        }
    ] );
    push @$counts, {total_count => 0, unarc_count => 0} unless @$counts; #на случай если для $client_ids не найдено ни одного шарда
    return overshard(group => 1,
                     sum => [qw/total_count unarc_count/],
                     $counts)->[0];
}

=head2 check_add_client_campaigns_limits

    Проверяет, что клиент может создать ещё N кампаний без нарушения своих ограничений.

    Именованные параметры:
    ClientID         — идентификатор клиента
    uid              — идентификатор пользователя
    unarchived_count — количество создаваемых незаархивированных кампаний
    archived_count   — количество создаваемых заархивированных кампаний

    Обязательно должен быть указан или ClientID или uid. Одновременное их указание не допускается.
    Если не указано ни unarchived_count, ни archived_count, считается, что добавляется одна незаархивированная кампания.
    Возвращает текст с причиной, не позволяющей создать кампании с указанными параметрами. Или undef, если создать можно.

    $can_add_campaign = check_add_client_campaigns_limits(ClientID => $client_id);
    $can_add_campaign = check_add_client_campaigns_limits(uid => $uid);
    $error = check_add_client_campaigns_limits(ClientID => $client_id, count => 3);

=cut

sub check_add_client_campaigns_limits {
    my %O = @_;

    die 'ClientID and uid cannot both be specified' if exists $O{ClientID} && exists $O{uid};

    my $client_id;
    if ( exists $O{ClientID} ) {
        $client_id = $O{ClientID};
    } elsif (exists $O{uid}) {
        die 'cannot handle false uid' unless $O{uid};
        $client_id = get_clientid( uid => $O{uid} );
    } else {
        die 'nor ClientID nor uid given';
    }

    if ( !$client_id ) {
        # у нового пользователя, создающего свою первую кампанию, может ещё не быть ClientID
        # лимит на число кампаний в этом случае не проверяем
        return;
    }

    # получаем текущее количество кампаний клиента (общее + неархивных)
    my $cur_camp_stat = get_client_campaign_count( $client_id );

    my $unarchived_count = $O{unarchived_count} || 0;
    my $archived_count = $O{archived_count} || 0;
    $unarchived_count = 1 if !$unarchived_count && !$archived_count;
    my $client_limits = get_client_limits( $client_id );

    # проверяем, что с появлением кампаний не выйдем за лимит общего количества кампаний
    if ( ($cur_camp_stat->{total_count} + $unarchived_count + $archived_count) > $client_limits->{camp_count_limit} ) {
        return iget('Превышено максимальное количество кампаний — %d', $client_limits->{camp_count_limit});
    }

    # проверяем, что с появлением кампаний не выйдем за лимит количества незаархивных кампаний
    if ( ($cur_camp_stat->{unarc_count} + $unarchived_count) > $client_limits->{unarc_camp_count_limit} ) {
        return iget('Превышено максимальное количество незаархивированных кампаний — %d', $client_limits->{unarc_camp_count_limit});
    }
}

=head2 check_add_client_groups_limits($options)

    Проверяет, что клиент может создать ещё N групп без нарушения своих ограничений.

    $options:
    cid - номер кампании куда добавляются баннеры
    media_plan - проверка для медиаплана?
    new_banners - количество добавляемых баннеров

    Возвращает текст с причиной, не позволяющей создать кампании с указанными параметрами. Или undef, если создать можно.

=cut

sub check_add_client_groups_limits {
    my $options = shift;
    my $client_id;
    my $cid = '';
    my $cids = [];

    if ($options->{cid}) {
        $cid = $options->{cid};
        $cids = [$cid];
        $client_id = get_one_field_sql(PPC(cid => $cid), 'SELECT u.ClientID FROM campaigns c JOIN users u USING (uid) WHERE c.cid = ?', $cid);
    } else {
        die 'not uid given' unless exists $options->{uid};
        $client_id = get_clientid(uid => $options->{uid});
    }

    my $check = check_add_client_groups_limits_mass($client_id, $cids, $options);
    return exists $check->{$cid} ? $check->{$cid} : $check->{''};
}


=head2 check_add_client_groups_limits_mass($client_id, $cids, $options)

    Проверяет, что клиент может создать ещё N групп без нарушения своих ограничений.

    Возвращает хеш {
        cid => текст с причиной, не позволяющей создать кампании с указанными параметрами. Или undef, если создать можно
        '' => то же самое для новой кампании
    }

=cut

sub check_add_client_groups_limits_mass {
    my ($client_id, $cids, $options) = @_;
    $options ||= {};

    my $new_groups = $options->{new_groups} || 0;
    my $limit = get_client_limits($client_id)->{banner_count_limit};

    my $quantity_by_cid = {};
    if (@$cids) {
        my $table = $options->{media_plan} ? 'mediaplan_banners' : 'phrases';
        $quantity_by_cid = get_hash_sql(PPC(cid => $cids), [
                "SELECT cid, COUNT(*)
                FROM $table",
                WHERE => { cid => $cids },
                'GROUP BY cid',
            ]);
    }
    $quantity_by_cid->{''} = 0;

    my $error = iget('Достигнуто максимальное количество групп объявлений в кампании - %d', $limit);
    my %result = map {($_ => $quantity_by_cid->{$_} + $new_groups > $limit ? $error : undef)} keys %$quantity_by_cid;
    return \%result;
}


=head2 check_add_client_creatives_limits($options)

    Проверяет, что клиент может создать ещё N креативов в группе без нарушения своих ограничений.

    $options:
    pid - номер группы
    new_creatives - количество добавляемых креативов

    Возвращает текст с ошибкой или undef, если создать можно.

=cut

sub check_add_client_creatives_limits {
    my $options = shift;

    my $quantity = $options->{pid}
        ? (get_one_field_sql(PPC(pid => $options->{pid}), 'SELECT COUNT(*) FROM banners WHERE pid = ?', $options->{pid}) || 0)
        # || 0 - для несуществующих групп, чтобы не было предупреждения про undef + число
        : 0; # in new groups

    return $quantity + ($options->{new_creatives} || 0) > $Settings::DEFAULT_CREATIVE_COUNT_LIMIT
        ? iget('Достигнуто максимальное количество объявлений в группе - %d', $Settings::DEFAULT_CREATIVE_COUNT_LIMIT)
        : undef;
}

=head2 check_add_library_minus_words_limits($options)

    Проверяет, что клиент может создать ещё N библиотечный элементов без нарушения своих ограничений.

    $options:
    src_client_id - идентификатор пользователя с которого копируем
    dst_client_id - идентификатор пользователя на который копируем

    Возвращает текст с ошибкой или undef, если добавить можно.

=cut

sub check_add_library_minus_words_limits {
    my %O = @_;

    my $src_client_id = $O{src_client_id};
    my $dst_client_id = $O{dst_client_id};

    if ( $src_client_id == $dst_client_id || !$dst_client_id ) {
        # у нового пользователя, создающего свою первую кампанию, может ещё не быть ClientID
        # лимит на число библиотечных элементов в этом случае не проверяем
        return;
    }

    my $old_mw_id2mw = get_hashes_hash_sql(PPC(ClientID => $src_client_id), [
        "SELECT mw.mw_id, mw.mw_text, mw.mw_name, mw.mw_hash
                           FROM minus_words mw
                        ", WHERE => { ClientID => $src_client_id, is_library => 1 } ]);

    #существующие библ. элементы у нового клиента
    my $new_mw_id2mw = get_all_sql(PPC(ClientID => $dst_client_id), [
        "SELECT mw.mw_id, mw.mw_text, mw.mw_name, mw.mw_hash
                           FROM minus_words mw
                        ", WHERE => { ClientID => $dst_client_id, is_library => 1 } ]);
    my $new_client_libr_count = scalar @$new_mw_id2mw;

    my %new_mw_hash_with_name2mw = map {$_->{mw_hash} . '||||' . $_->{mw_name} => 1} @$new_mw_id2mw;
    for my $old_mw (values %$old_mw_id2mw) {
        my $complex_hash = $old_mw->{mw_hash} . '||||' . $old_mw->{mw_name};
        if (!$new_mw_hash_with_name2mw{$complex_hash}) {
            $new_client_libr_count++;
        }
    }

    #библиотечных элементов может быть не больше 30-ти
    if ($new_client_libr_count > $Settings::LIBRARY_MINUS_WORDS_COUNT_LIMIT) {
        return iget('Превышено максимальное количество библиотечных элементов — %d', $Settings::LIBRARY_MINUS_WORDS_COUNT_LIMIT);
    }

    return;
}

=head2 client_currency_changes

    $changes = client_currency_changes($client_id);
    $changes => {
        'YND_FIXED' => {
            currency_from => 'YND_FIXED',
            currency_to => 'RUB',
            date => '20121217',
    };

=cut

sub client_currency_changes
{
    my ($client_id) = @_;

    my $changes = mass_client_currency_changes([$client_id]);

    return $changes->{$client_id};
}

=head2 mass_client_currency_changes

    $changes = mass_client_currency_changes([$client_id1, $client_id2, ...]);
    $changes => {
        $client_id1 => {
            'YND_FIXED' => {
                currency_from => 'YND_FIXED',
                currency_to => 'RUB',
                date => '20121217',
            }
        },
        $client_id2 => {...},
        ...
    };

=cut

sub mass_client_currency_changes {
    my ($client_ids) = @_;

    my %client_currency_changes = ();
    my $changes = get_all_sql(PPC(ClientID => $client_ids), ['SELECT ClientID, currency_from, currency_to, date FROM client_currency_changes',
                              WHERE => {ClientID => SHARD_IDS}]) || [];
    for my $row (@$changes) {
        $client_currency_changes{$row->{ClientID}}->{$row->{currency_from}} = $row;
    }

    return \%client_currency_changes;
}

=head2 mass_client_total_sums

    Возвращает суммарные данные по суммам на кампаниях нескольких клиентов.
    Если результирующая валюта это у.е., то суммы получаются с НДС.
    Для реальных валют суммы получаются без НДС с включённым скидочным бонусом.
    Если к кампаниям привязан общий счет, то суммируем суммы с него (указывать отдельно type => [... 'wallet'] не нужно)

    Параметры именованные:
        ClientIDs -- список ClientID клиентов, по которым надо получить данные; обязательный, если не указано agency_client_id, agency_uidm manager_uid или cids
        currency -- валюта, к которой надо привести суммы
            по умолчанию валюта, в которой работает клиент
        type -- (обязательно) условие на тип кампании, которые надо учитывать
            разумные значения: 'text', 'mcb', ['text', 'mcb'] (ссылка на массив)
        agency_client_id -- ClientID агентства, кампании которого надо учитывать
            по умолчанию учитываются все кампании без ограничения по агентству
        agency_uid -- UIDы представителей агентств; будут учитываться только кампании, связанные с этими UIDами
        manager_uid -- UIDы менеджеров; будут учитываться только кампании, связанные с этими UIDами
        cids -- смотреть только на часть кампаний
            по умолчанию учитываются все кампании
        without_nds -- 1, если требуется получить суммы без НДС
        without_bonus -- 1, если требуется получить суммы без скидочного бонуса
        wallet_sums_only -- 1, если требуется считать суммы только по средствам самого общего счета (без кампаний под ним), требуется при переносе средств
        sums_available_for_transfer -- 1, если требуется получить суммы, которые можно перевести со счета в балансе.
                                       В этом случае для общего счета возвращается сумма не больше, чем campaigns.sum_balance

    $sums = mass_client_total_sums(
        ClientIDs => [$clientid1, $clientid2, ...],
        type => ['text', 'dynamic', 'performance'] | 'mcb' | ...,
        agency_client_id => $agency_client_id,
        agency_uid => [$uid1, $uid2, ...],
        manager_uid => [$uid1, $uid2, ...],
        currency => $currency,
        cids => [$cid1, $cid2, ...],
    );
    $sums = {
        $client_id1 => {
            sum       => 123.45,  # "Было"
            sum_spent => 23.4,    # "Потрачено"
            total     => 100.05,  # "Осталось"
            bonus     => 12.13,   # "из них скидочный бонус"
            currency  => 'RUB',   # Валюта
            count     => 123,     # Количество кампаний
            shows     => 789,     # Количество показов на кампаниях
            clicks    => 456,     # Количество кликов на кампаниях
        },
        $client_id1 => {...},
    };

=cut

sub mass_client_total_sums {
    my (%O) = @_;

    die 'nor ClientIDs, nor agency_client_id specified' if none {defined $O{$_}} qw/ClientIDs agency_client_id agency_uid manager_uid cids/;
    die "client_total_sums: 'type' missed" unless $O{type};

    # Список всех кампаний: две суммы, валюта
    $O{type} = [$O{type}] if ref($O{type}) ne 'ARRAY';

    my $cond = {
        'c.statusEmpty' => 'No',
        'c.type' => $O{type},
    };

    # Для типов кампаний, живущих под ОС, выбираем еще и кошелек
    my $sql_count_cond = 'cid';
    if (any { camp_kind_in(type => $_, 'under_wallet') } @{$O{type}}) {
        push @{$cond->{'c.type'}}, 'wallet';
        $sql_count_cond = 'IF(c.type = "wallet", NULL, c.cid)';
    }

    $cond->{'c.ClientID'} = SHARD_IDS if $O{ClientIDs};
    $cond->{'c.AgencyID'} = $O{agency_client_id} if $O{agency_client_id};
    $cond->{'c.AgencyUID'} = $O{agency_uid} if $O{agency_uid};
    $cond->{'c.ManagerUID'} = $O{manager_uid} if $O{manager_uid};
    $cond->{'c.cid'} = $O{cids} if $O{cids};
    $cond->{'c.archived'} = 'No' if $O{no_archived};

    my @shard_keys = qw/ClientID cid/;
    my %shard_cond = choose_shard_param({ map { $_ => $O{$_.'s'} } @shard_keys }, \@shard_keys, allow_shard_all => 1);
    my $camp_types_under_wallet_in = join ",", map { sql_quote($_) } @{get_camp_kind_types('under_wallet')};
    my $sum_sql = 'IF(IFNULL(c.currency, "YND_FIXED") = IFNULL(cl.work_currency, "YND_FIXED"), c.sum, c.sum_spent)';

    my $camps = get_all_sql(PPC(%shard_cond), [
        select => "ifnull(c.currency, 'YND_FIXED') currency
                 , SUM(IF(c.type = 'wallet', c.sum, 0)) wallet_sum
                 , SUM(IF(c.type = 'wallet', c.sum_spent, 0)) wallet_sum_spent
                 , SUM(IF(c.type = 'wallet', c.sum - c.sum_spent, 0)) AS wallet_total
                 , SUM(IF(c.type = 'wallet', IF(IFNULL(wwc.is_sum_aggregated, 'No') = 'Yes', c.sum_balance, c.sum), 0)) AS wallet_sum_balance

                 , SUM(IF(c.type IN ($camp_types_under_wallet_in) AND c.wallet_cid = 0, $sum_sql, 0)) text_sum
                 , SUM(IF(c.type IN ($camp_types_under_wallet_in) AND c.wallet_cid = 0, c.sum_spent, 0)) text_sum_spent
                 , SUM(IF(c.type IN ($camp_types_under_wallet_in) AND c.wallet_cid = 0, $sum_sql - c.sum_spent, 0)) AS text_total

                 , SUM(IF(c.type IN ($camp_types_under_wallet_in) AND c.wallet_cid > 0, $sum_sql, 0)) text_on_wallet_sum
                 , SUM(IF(c.type IN ($camp_types_under_wallet_in) AND c.wallet_cid > 0, c.sum_spent, 0)) text_on_wallet_sum_spent
                 , SUM(IF(c.type IN ($camp_types_under_wallet_in) AND c.wallet_cid > 0, $sum_sql - c.sum_spent, 0)) AS text_on_wallet_total

                 , SUM($sum_sql) sum
                 , SUM(c.sum_spent) sum_spent
                 , SUM($sum_sql - c.sum_spent) AS total

                 , SUM(c.shows) AS shows
                 , SUM(c.clicks) AS clicks
                 , c.ClientID
                 , count($sql_count_cond) AS count
                 , c.type
                  ",
        from => 'campaigns c
                 JOIN clients cl ON cl.ClientID = c.ClientID
                 LEFT JOIN wallet_campaigns wwc ON wwc.wallet_cid = c.cid',
        where => $cond,
        "group by 1, c.ClientID",
        ]
    );

    return {} unless $camps && @$camps;

    my @all_client_ids = grep {$_} uniq map {$_->{ClientID}} @$camps;
    my $client_currencies = mass_get_client_currencies(\@all_client_ids);
    my $client_currency_changes = mass_client_currency_changes(\@all_client_ids);
    my $client_nds_data = mass_get_client_NDS(\@all_client_ids);
    my $client_discount_data = {};
    if (!$O{without_bonus}) {
        $client_discount_data = mass_get_client_discount(\@all_client_ids);
    }

    my (%client_sums);

    # для каждой кампании
    for my $camp (@$camps){
        my $i = 0;

        if ($O{wallet_sums_only}) {
            # прибавляем к общему счету отрицательные остатки на кампаниях под счетом,
            # имеет смысл только при фильтрации по agency_client_id (по одному типу сервисирования)
            if (defined $O{agency_client_id}) {
                for my $suff (qw/sum sum_spent total/) {
                    $camp->{"wallet_$suff"} += $camp->{"text_on_wallet_$suff"} if $camp->{text_on_wallet_total} < 0;
                }
                if ($O{sums_available_for_transfer}) {
                    for (qw/sum total/) {
                        $camp->{"wallet_$_"} = min($camp->{"wallet_$_"}, $camp->{"wallet_sum_balance"});
                    }
                }
            }

            # если запросили суммы по кошелькам
            # то берем сумму только с кошельков + сумму с кампаний без кошелька
            $camp->{$_} = $camp->{"wallet_$_"} + $camp->{"text_$_"} for qw/sum sum_spent total/;
        }

        # удаляем промежуточные данные
        for my $suff (qw/sum sum_spent total/) {
            delete $camp->{"wallet_$suff"};
            delete $camp->{"text_$suff"};
            delete $camp->{"text_on_wallet_$suff"};
        }

        my $client_id = $camp->{ClientID};
        $camp->{currency} ||= 'YND_FIXED';  # currency_defaults

        if ($camp->{currency} ne 'YND_FIXED' && $O{without_nds}) {
            for my $f (qw/sum sum_spent/) {
                my $client_nds = $client_nds_data->{$client_id} || 0;
                $camp->{$f} /= (1 + $client_nds/100) if defined $camp->{$f};
            }
        }

        # Рабочая валюта клиента
        my $work_currency = $client_currencies->{$client_id}->{work_currency};
        my $target_currency = $O{currency} || $work_currency;

        # Список всех дат перехода на новые валюты
        my $currency_change = $client_currency_changes->{$client_id};
        # во время конвертации на некоторое время возможна ситуация, когда у клиента ещё старая валюта (у.е.), а часть кампаний уже в новой валюте
        # конвертировать все кампании одной транзакцией не хочется, интерфейс в это время заблокирован, а данные используются только для
        # суммарных цифр по клиенту/агентству (т.е. не 100% точных из-за различий в валютах)
        # поэтому ставим костыль: кампании в новой валюте приводим к старой валюте тоже по той же дате перехода, если не было последующего перехода
        if ($currency_change && exists $currency_change->{YND_FIXED} && !exists $currency_change->{$currency_change->{YND_FIXED}->{currency_to}}) {
            $currency_change->{$currency_change->{YND_FIXED}->{currency_to}} = {
                currency_to => $currency_change->{YND_FIXED}->{currency_from},
                currency_from => $currency_change->{YND_FIXED}->{currency_to},
                date => $currency_change->{YND_FIXED}->{date},
            }
        }

        # конвертируем в принудительно указанную валюту по сегодняшнему курсу
        if ($work_currency ne $target_currency){
            $currency_change->{$work_currency} = {
                currency_to => $target_currency,
                currency_from => $work_currency,
                date => today(),
            };
        }

        #убираем НДС с мультивалютных кампаний
        campaign_remove_nds_and_add_bonus($camp, client_nds => $client_nds_data->{$client_id}, client_discount => $client_discount_data->{$client_id});

        #пока текущая валюта не равна целевой
        while ($camp->{currency} ne $target_currency){
            #берем дату, когда клиент перешел с текущей валюты на новую (или умираем, если даты нет)
            my $change = $currency_change->{$camp->{currency}};
            if (!$change) {
                # если по причине какого-то бага у клиента есть директовская кампании в несвоей валюте, но нет даты конвертации,
                # ругаемся в ppc-admin@ и конвертируем сразу в целевую валюту по сегодняшнему курсу
                # Баян и геоконтекст продолжают работать в у.е., для них это не проблема. при необходимости конвертируем тоже по сегодняшнему курсу
                if (camp_kind_in(type => $camp->{type}, 'web_edit_base')) {
                    my $msg = "missed currency change data for currency $camp->{currency} on client $client_id";
                    send_alert(Carp::longmess($msg), 'mass_client_total_sums error: no currency change date');
                }
                $change = {
                    currency_to => $target_currency,
                    currency_from => $camp->{currency},
                    date => today(),
                };
            }
            #переводим обе суммы из старой валюты в новую по курсу на эту дату
            for my $f (qw/sum sum_spent total/){
                $camp->{$f} = convert_currency($camp->{$f}, $camp->{currency}, $change->{currency_to}, date => $change->{date}, with_nds => !$O{without_nds});
            }
            #записываем новую валюту вместо текущей
            $camp->{currency} = $change->{currency_to};

            die "suspicious long currency change history" if $i++ > 100; # против бесконечного цикла
        }

        ## суммируем суммы
        my @summable_fields = qw/sum sum_spent total bonus shows clicks count/;
        $client_sums{$client_id} ||= {
            currency => $target_currency,
            (map {$_ => 0} @summable_fields),
            total => 0,
        };

        #умираем, если валюта кампании не равна требуемой
        die unless $camp->{currency} eq $target_currency;
        #добавляем суммы кампании к результату
        for my $f(@summable_fields){
             if ($camp->{$f}) {
                $client_sums{$client_id}->{$f} += $camp->{$f};
             }
        }
    }

    return \%client_sums;
}

=head2 client_total_sums

    Возвращает суммарные данные по суммам на кампаниях одного клиента.
    Если результирующая валюта это у.е., то суммы получаются с НДС.
    Для реальных валют суммы получаются без НДС с включённым скидочным бонусом.

    Параметры позиционные
        $ClientID -- id клиента

    Параметры именованные
        currency -- валюта, к которой надо привести суммы
            по умолчанию валюта, в которой работает клиент
        type -- (обязательно) условие на тип кампании, которые надо учитывать
            разумные значения: 'text', 'mcb', ['text', 'mcb'] (ссылка на массив)
        agency_client_id -- ClientID агентства, кампании которого надо учитывать
            по умолчанию учитываются все кампании без ограничения по агентству
        no_archived -- 1|0 не использовать в подсчете архивные кампании

    Возвращаемое значение
    {
        sum       => 123.45,  # "Было"
        sum_spent => 23.4,    # "Потрачено"
        total     => 100.05,  # "Осталось"
        bonus     => 12.13,   # "из них скидочный бонус"
        currency  => 'RUB',   # Валюта
        count     => 123,     # Количество кампаний
        shows     => 789,     # Количество показов на кампаниях
        clicks    => 456,     # Количество кликов на кампаниях
    }

=cut

sub client_total_sums
{
    my ($client_id, %O) = @_;

    die "client_total_sums: ClientID missed" unless $client_id;
    die "client_total_sums: 'type' missed" unless $O{type};

    my $sums = mass_client_total_sums(ClientIDs => [$client_id], type => $O{type}, currency => $O{currency}, agency_client_id => $O{agency_client_id}, no_archived => $O{no_archived});
    return $sums->{$client_id};
}

=head2 check_pay_yamoney

    Проверяет, может ли клиент платить яндекс.деньгами. Если не может -- возвращает ошибку.

    # $c -- объект DirectContext
    $error = check_pay_yamoney($c, $client_id, $yandex_domain, $work_currency);
    $error => undef|'client'|'domain'|'currency'|'shows'|'country'

=cut

sub check_pay_yamoney {
	my ($c, $client_id, $yandex_domain, $work_currency) = @_;

    if (!$c->login_rights->{is_any_client} || $c->login_rights->{client_have_agency}) {
    	return 'client';
    }

    if ($yandex_domain !~ /\.ru$/) {
    	return 'domain';
    }

    if (none {$work_currency eq $_} qw/YND_FIXED RUB/) {
    	return 'currency';
    }

    my $show_in_270_last_days =
        get_one_field_sql(PPC(uid => $c->client_chief_uid), "select 1
                                from campaigns
                                where uid = ?
                                  and statusEmpty = 'No'
                                  and type = 'text'
                                  and lastShowTime >= DATE_SUB(NOW(), INTERVAL 270 DAY)
                                LIMIT 1
                               ", $c->client_chief_uid);
	return 'shows' unless $show_in_270_last_days;

    # оплата Я.Деньгами возможна только если страна не выбрана или выбрана Россия
    my $client_data = get_client_data($client_id, [qw/country_region_id/]);
    if ($client_data && $client_data->{country_region_id} && $client_data->{country_region_id} != $geo_regions::RUS) {
        return 'country';
    }

    return undef;
}

=head2 get_per_client_convert_lock_guard

    Получает guard-объект эксклюзивного поклиентского лока конвертации в реальную валюту.
    Используется для предотвращения одновременного изменений валютных данных у клиента
    (ставок, например) во время его конвертации в реальную валюту.

    $lock_guard = get_per_client_convert_lock_guard($client_id, timeout => 3);

=cut

sub get_per_client_convert_lock_guard {
    my ($client_id, %O) = @_;

    my $lock_name = $SQL_LOCK_NAME_PREFIX . $client_id;
    my $timeout = (defined $O{timeout}) ? $O{timeout} : 1;
    # пытаемся захватить поклиентный лок в БД (таймаут по умолчанию -- 1 секунда)
    return sql_lock_guard(PPC(ClientID => $client_id), $lock_name, $timeout);
}

=head2 mass_skip_converting_clients

    Отсеивает из списка ClientID клиентов, которые уже конвертируются в валюту или
    скоро начнут конвертироваться.
    См. описание $Settings::STOP_OPERATION_MINUTES_BEFORE_CONVERT.

    $client_ids_not_converting = mass_skip_converting_clients(\@client_ids);
    $client_ids_not_converting = mass_skip_converting_clients(\@client_ids, include_states => ['OVERDRAFT_WAITING']);

=cut

sub mass_skip_converting_clients {
    my ($client_ids, %O) = @_;

    return [] unless $client_ids && @$client_ids;

    my @include_states = ('DONE');
    if ($O{include_states}) {
        push @include_states, @{$O{include_states}};
    }
    my $client_ids_converting = get_one_column_sql(PPC(ClientID => $client_ids), [q/
        SELECT ClientID
        FROM currency_convert_queue
        WHERE state != "DONE" AND NOW() > start_convert_at - INTERVAL ? MINUTE
        AND/, {ClientID => SHARD_IDS, state__not_in => \@include_states},
    ], $Settings::STOP_OPERATION_MINUTES_BEFORE_CONVERT) || [];
    return xminus($client_ids, $client_ids_converting);
}

=head2 is_client_converting_soon

    Проверяет, будет ли клиент в ближайшее время конвертироваться (или уже конвертируется).

    $is_client_converting_soon = is_client_converting_soon($client_id);
    $is_client_converting_soon = is_client_converting_soon($client_id, include_states => ['OVERDRAFT_WAITING']);
    $is_client_converting_soon => 1|0

=cut

sub is_client_converting_soon {
    my ($client_id, %O) = @_;

    my $rest_clientids = mass_skip_converting_clients([$client_id], include_states => $O{include_states});
    return scalar(@$rest_clientids) > 0 ? 0 : 1;
}

=head2 is_any_client_converting_soon(\@client_ids)

    Аналог is_client_converting_soon, но по списку клиентов

=cut

sub is_any_client_converting_soon {
    my ($client_ids, %O) = @_;

    my $rest_clientids = mass_skip_converting_clients($client_ids, include_states => $O{include_states});

    return scalar(@$rest_clientids) == scalar(@$client_ids) ? 0 : 1;
}


=head2 mass_client_must_convert

    Определяет по списку ClientID нужно ли принудительно заставить клиента
    конвертироваться до того, как сможет что-то сделать.
    См. https://st.yandex-team.ru/DIRECT-53676

    $client_id2must_convert = mass_client_must_convert(\@client_ids);
    $client_id2must_convert => {
        $client_id1 => 1,
        ...
    }

=cut

sub mass_client_must_convert {
    my ($client_ids) = @_;

    return {} unless $client_ids && @$client_ids;

    return get_hash_sql(PPC(ClientID => $client_ids), ['
        SELECT ctfmt.ClientID, 1
        FROM clients_to_force_multicurrency_teaser ctfmt
        LEFT JOIN force_currency_convert fcc ON ctfmt.ClientID = fcc.ClientID
        LEFT JOIN clients cl ON ctfmt.ClientID = cl.ClientID
        LEFT JOIN clients_options clo ON cl.ClientID = clo.ClientID
        LEFT JOIN currency_convert_queue q ON ctfmt.ClientID = q.ClientID
        INNER JOIN users u ON ctfmt.ClientID = u.ClientID
        WHERE
            IFNULL(cl.work_currency, "YND_FIXED") = "YND_FIXED"
            AND fcc.ClientID IS NULL -- не принял оферту
            AND IFNULL(FIND_IN_SET("not_convert_to_currency", clo.client_flags), 0) = 0 -- не установлен флаг не конвертироваться
            AND q.ClientID IS NULL -- конвертация ещё не заказана
            AND ', {
                'ctfmt.ClientID__int' => SHARD_IDS,
            }, '
        GROUP BY 1
          HAVING SUM(IF(u.hidden = "Yes", 1, 0)) = 0  -- не тестовый
        ORDER BY NULL
    ']);
}

=head2 client_must_convert_for_role($client_id, $login_rights)

Функция проверяет, что клиента нужно средиректить на принудительную конвертацию,
только если текущая роль для этого подходит

=cut

sub client_must_convert_for_role
{
    my ($client_id, $login_rights) = @_;
    return client_must_convert($client_id)
        && ! any { $login_rights->{$_."_control"} } qw/support super manager superreader media/;
}


=head2 client_must_convert

    Определяет по ClientID клиента нужно ли принудительно заставить клиента
    конвертироваться до того, как сможет что-то сделать.
    См. https://st.yandex-team.ru/DIRECT-53676

    $must_convert = Client::client_must_convert($client_id);
    $must_convert => 1

=cut

sub client_must_convert {
    my ($client_id) = @_;

    return is_any_client_must_convert([$client_id]);
}

=head2 is_any_client_must_convert

    Проверяет, должен ли кто-то из переданного списка ClientID обязательно сконвертироваться
    До этого надо блокировать ему интерфейс

    $any_client_must_convert = Client::is_any_client_must_convert(\@client_ids);
    $any_client_must_convert => 1/0

=cut

sub is_any_client_must_convert {
    my ($client_ids) = @_;

    return 0 unless @$client_ids;

    my $client_id2must_convert = mass_client_must_convert($client_ids);
    return any { $_ } values %$client_id2must_convert;
}

=head2 mass_is_client_converting_soon

    По списку ClientID возвращает ссылку на хеш ClientID => 1/0
    1 — скоро будет конвертироваться или уже в процессе конвертации
    0 — всё хорошо

=cut

sub mass_is_client_converting_soon {
    my ($client_ids) = @_;

    return {} unless @$client_ids;

    my $client_ids_not_converting = mass_skip_converting_clients($client_ids);
    my %client_id_not_converting = map {$_ => undef} @$client_ids_not_converting;

    my %is_client_converting_soon;
    for my $client_id (@$client_ids) {
        $is_client_converting_soon{$client_id} = (exists $client_id_not_converting{$client_id}) ? 0 : 1;
    }
    return \%is_client_converting_soon;
}

=head2 can_convert_to_real_money

    Проверяет, может ли клиент перевестись хотя бы в какую-нибудь реальную валюту.

    Перейти даём, если выполняются все условия:
    - клиент работает в у. е.
    - мы знаем НДС клиента
    - ещё нет заявки на переход (если не указана опция ignore_request_in_queue)
    - у клиента стоит галочка "показывать тизер перехода в реальную валюту"
    - у клиента не установлен флаг not_convert_to_currency (ppc.clients_options.client_flags)

    Наличие доступных для перехода валют и наше знание о них НЕ проверяется. Это надо делать
    в вызывающем коде, т.к. можно проверять запросом в Баланс или по таблице client_firm_country_currency.

    Возвращает undef, если переходить можно, и одну из строк, характеризующую ошибку, если нельзя:
    - already_converted
    - no_NDS_known
    - no_country_currency_known
    - request_in_queue
    - no_checkbox
    - free_client (multicurrency-agency-nds: EXPIRES 2013-09-01)
    - not_convert_to_currency

    $client_currencies = get_client_currencies($client_id);
    $client_nds = get_client_NDS($client_id);
    my @possible_currencies;

    $error = Client::can_convert_to_real_money(
        ClientID => $client_id,
        NDS => $client_nds,
        client_currencies => $client_currencies,
        ignore_request_in_queue => 0,
        ignore_nds_absence => 0,
        ignore_checkbox => 0,
        client_chief_uid => $c->client_chief_uid,
        country_currency_ref = [{region_id => 225, currency => 'RUB'}, ...],
        agency_id => <ClientID агентства> 
    );
    $error => undef|'already_converted'|'no_NDS_known'|...

=cut

sub can_convert_to_real_money {
    my (%O) = @_;

    my ($client_id, $client_nds, $client_currencies, $client_cheif_uid, $country_currency_ref) = @{\%O}{qw/ClientID NDS client_currencies client_chief_uid country_currency_ref/};

    die 'invalid ClientID: ' . str($client_id) unless is_valid_int($client_id, 0);
    die 'invalid client_currencies' unless $client_currencies && ref($client_currencies) eq 'HASH' && exists $client_currencies->{work_currency};

    return 'already_converted' unless $client_currencies->{work_currency} eq 'YND_FIXED';

    if (!$O{ignore_checkbox} && !get_one_field_sql(PPC(ClientID => $client_id), 'SELECT 1 FROM clients_to_force_multicurrency_teaser WHERE ClientID = ?', $client_id) ) {
        return 'no_checkbox';
    }

    if ( $O{ignore_country_currency_existance} ) {
        return 'no_NDS_known' if !defined $client_nds && !$O{ignore_nds_absence};
    } else {
        my $country_currencies = get_all_sql(PPC(ClientID => $client_id), 'SELECT country_region_id AS region_id, currency FROM client_firm_country_currency WHERE ClientID = ?', $client_id) || [];
        if ($country_currencies && @$country_currencies) {
            # НДС проверяем только для тех, о ком знает Баланс
        } elsif (!$country_currencies || ($country_currencies && scalar @$country_currencies == 0)) {
            # пустой список доступных для перехода стран/валют из Баланса означает, что они ещё ничего не знают про этого клиента (ни страны, ни оплат)
            # в этом случае пользователь может выбирать любую страну/валюту
            # но на всякий случай проверяем нет ли у них денег на кампании и признаков выставления счёта
            if (!get_one_field_sql(PPC(uid => $client_cheif_uid), 'SELECT 1 FROM campaigns WHERE (sum > 0 OR sum_to_pay > 0) AND uid = ? LIMIT 1', $client_cheif_uid)) {
                my $agency_id;
                if (exists $O{agency_id}) {
                    $agency_id = $O{agency_id};
                } else {
                    $agency_id = Primitives::get_client_first_agency($client_id);
                }
                my $country_currencies_hashref = $agency_id ? Currencies::get_currencies_by_country_hash_for_agency(undef, $agency_id)
                    : Currencies::get_currencies_by_country_hash_not_for_agency($client_cheif_uid);
                #конвертация в плоский список пар
                $country_currencies = [map { my $region_local = $_; map {[$region_local, $_]} @{$country_currencies_hashref->{$region_local}} } keys %$country_currencies_hashref];
            } else {
                return 'no_country_currency_known';
            }
        }
        if (defined $country_currency_ref) {
            @$country_currency_ref = @$country_currencies;
        }
    }

    if ( !$O{ignore_request_in_queue} ) {
        if (get_one_field_sql(PPC(ClientID => $client_id), 'SELECT 1 FROM currency_convert_queue WHERE ClientID = ?', $client_id)) {
            return 'request_in_queue';
        }
    }

    # multicurrency-agency-nds: EXPIRES 2013-09-01: не даём получить мультивалютного свободного клиента
    # подробности смотри в Client::mass_get_client_NDS
    my $client_data = get_client_data($client_id, [qw/allow_create_scamp_by_subclient not_convert_to_currency/]);
    if ($client_data->{allow_create_scamp_by_subclient}) {
        return 'free_client';
    }
    if ($client_data->{not_convert_to_currency}) {
        return 'not_convert_to_currency';
    }

    return undef;
}

=head2 get_client_firm_id

    Определяет балансовый идентификатор фирмы, в которую клиент платит деньги
    Названия фирм хранятся в %Currencies::FIRM_NAME

=cut

sub get_client_firm_id {
    my $client_id = shift;

    my $country_region_id;

    if ($client_id) {
        my $client_data = get_client_data($client_id, [qw/country_region_id/]);
        $country_region_id = $client_data->{country_region_id};
    } else {
        my $msg = 'get_client_firm_id called without ClientID';
        send_alert(Carp::longmess($msg), 'invalid ClientID in get_client_firm_id');
        return undef;
    }

    if ($country_region_id) {
        my $firm_by_country_hash = get_firm_by_country_hash();
        return $firm_by_country_hash->{$country_region_id};
    } else {
        return undef;
    }
}

=head2 need_country_currency

    $res = need_country_currency(
        client_id => $client_id,    # ClientID нужно брать не только из нашей базы, но и из Баланса
        client_role => rbac_who_is($rbac, $client_chief_uid), # роль нужно определять по главному представителю, т.к. это может быть новый [для нас] представитель известного нам клиента, добавленный в Балансе
        is_direct => $c->is_direct,     # сейчас - не используется, ранее меняло поведение для баяна на "ничего не спрашивать, работа только в фишках"
        is_agency_client => 1|0,
    );
    $res => {
        need_country_currency => 1|0,   # у клиента нужно спросить страну ИЛИ валюту
        need_only_currency => 1|0,
    };

=cut

sub need_country_currency {
    my (%O) = @_;

    my %res = (
        need_country_currency => 1,
        # при создании нового клиента под агентство спрашиваем только валюту, т.к. страна
        # определяет способы оплаты самого клиента, а платить за него будет агентство
        # фиксировать способы оплаты клиента по стране агентства, которое за него платит — неправильно
        need_only_currency => (($O{is_agency_client}) ? 1 : 0),
    );

    if ($O{client_id}) {
        if ($O{client_role} eq 'empty') {
            my $client_data = get_client_data($O{client_id}, [qw/work_currency country_region_id/]);
            if ($client_data && $client_data->{work_currency} && ($O{is_agency_client} || $client_data->{country_region_id})) {
                # если у нас уже выбрана валюта (и страна для неагентских), то ничего дополнительно спрашивать не надо
                $res{need_country_currency} = 0;
            }
        } else {
            $res{need_country_currency} = 0;
        }
    }

    return \%res;
}

=head2 get_client_migrate_to_currency

    Возвращает дату перехода клиента на мультивалютность, пригодную для
    отправки в Баланс в виде параметра MIGRATE_TO_CURRENCY

    my $migrate_to_currency = get_client_migrate_to_currency($client_id);

=cut

sub get_client_migrate_to_currency {
    my ($client_id) = @_;

    my $currency_changes = client_currency_changes($client_id);
    return $currency_changes->{YND_FIXED}->{date} || $Client::BEGIN_OF_TIME_FOR_MULTICURRENCY_CLIENT;
}

=head2 is_client_exists

    Проверяет существование в БД пользователей с указанным ClientID

=cut
sub is_client_exists {
    my $client_id = shift;
    return $client_id && get_one_field_sql(PPC(ClientID => $client_id), ['SELECT 1 FROM users', WHERE => {ClientID => SHARD_IDS}, 'LIMIT 1']) ? 1 : 0;
}

=head2 update_client_table

    Обновляет или добавляет запись в одну из таблиц с клиентскими данными,
    если есть что обновлять

=cut

sub update_client_table
{
    my ($client_id, $table, $data) = @_;

    my $table_data = {};
    my $_fh = $CLIENT_TABLES{$table}{_fields_hash};
    for my $f (grep {exists $_fh->{$_}} keys %$data) {
        if (defined $_fh->{$f}) {
            # поле типа set
            $table_data->{"$_fh->{$f}__smod"}->{$f} = $data->{$f};
        } else {
            $table_data->{$f} = $data->{$f};
        }
    }
    $table_data->{ClientID} = $client_id;

    return if scalar(keys(%$table_data)) <= 1; # не вставляем запись если есть только ClientID

    my $old_data = get_client_data($client_id, [qw/ role hide_market_rating /]);
    unless (%$old_data) {
        $old_data = { role => $Rbac::ROLE_EMPTY, hide_market_rating => 0 };
    }

    # Нельзя перезаписывать любую роль кроме empty потому что cломаем доступы клиенту
    if (defined $table_data->{role}
        && $table_data->{role} ne $old_data->{role}
        && !($old_data->{role} eq $Rbac::ROLE_EMPTY || $table_data->{role} eq $Rbac::ROLE_EMPTY ))
    {
        die "invalid role update for ClientID $client_id: db role $old_data->{role}, new role $table_data->{role}";
    }

    # Если изменили изменился флаг hide_market_rating, добавляем все кампании клиента на переотправку
    if (defined $table_data->{hide_market_rating} && $table_data->{hide_market_rating} != $old_data->{hide_market_rating}) {
        my $uid = get_one_column_sql(PPC(ClientID => $client_id), "select uid from users where ClientID = ?", $client_id);
        my $cids = get_one_column_sql(PPC(uid => $uid), ["select cid from campaigns", where => { 'uid' => SHARD_IDS, 'OrderID__gt' => 0, 'archived' => 'No', 'type' => get_camp_kind_types('market_ratings')}]);

        # Пишем в лог номера кампаний, которые добавили в ленивую очередь
        log_bs_resync_market_ratings({
            ClientID => $client_id,
            cid => $cids,
            old_hide_market_rating_value => $old_data->{hide_market_rating},
            new_hide_market_rating_value => $table_data->{hide_market_rating},
        });

        BS::ResyncQueue::bs_resync_camps($cids, priority => BS::ResyncQueue::PRIORITY_UPDATE_DOMAIN_RATINGS);
    }

    do_insert_into_table(PPC(ClientID => $client_id), $table
                             , $table_data
                             , on_duplicate_key_update => 1
                             , key => 'ClientID');
}

=head2 get_client_data

    Возвращает хеш с данными о клиенте.
    На вход принимает позиционными параметрами ClientID и ссылку на список полей из таблиц clients, clients_options.
    Если клиент с указанным ClientID не существует, возвращает undef.
    Возвращает ссылку на хеш с запрошенными полями.

    $ClientID = 12345;
    $fields = [qw/name allow_create_scamp_by_subclient/];
    $client_data = get_client_data($ClientID, $fields);
    $client_data => {
        ClientID => 12345,
        name => 'test client name',
        allow_create_scamp_by_subclient => 1,
    }

=cut

sub get_client_data
{
    my ($client_id, $fields) = @_;
    my $client_data = mass_get_clients_data([$client_id], $fields)->{ $client_id };

    if (exists $client_data->{work_currency} && !$client_data->{work_currency}) {
        $client_data->{work_currency} = 'YND_FIXED';
    }
    return $client_data;
}

=head2 mass_get_clients_data(\@client_ids, \@fields)

    На вход принимает ссылку на массив ClientID и ссылку на список полей из таблицы clients, clients_options.
    Возвращает ссылку на хеш, где ключами являются ClientID, а значениями хэши с запрошенными полями.

    $сlid2data = mass_get_clients_data(\@client_ids, [qw/ClientID hide_market_rating/]);
    # $clid2data => { 123123 => {ClientID => 12345, hide_market_rating => 0} };

=cut

sub mass_get_clients_data {
    my ($client_ids, $fields) = @_;

    return {} unless @$client_ids;
    return {} unless @$fields;

    my (@tables, @clean_fields, @sql_fields, %sets2fields, $client_id_required);

    my $is_can_use_day_budget_required;
    foreach my $table (keys %CLIENT_TABLES) {
        my $_fh = $CLIENT_TABLES{$table}->{_fields_hash};
        my @searched_fields;
        for my $f (@$fields) {
            if (exists $_fh->{$f}) {
                push @searched_fields, $f;
            } elsif ($f eq 'ClientID') {
                $client_id_required = 1;
            } elsif ($f eq 'can_use_day_budget') {
                $is_can_use_day_budget_required = 1;
            }
        }
        if (@searched_fields) {
            push @tables, $table;
            push @clean_fields, @searched_fields;
            push @sql_fields, uniq map {sql_quote_identifier("$table.".($_fh->{$_} // $_))} @searched_fields;
            for my $f (grep {defined $_fh->{$_}} @searched_fields) {
                push @{$sets2fields{$_fh->{$f}}}, $f;
            }
        }
    }

    if (scalar @clean_fields || $is_can_use_day_budget_required) {
        # чтобы гарантировать, что выберутся все существующие поля, в случае если надо обработать более 1 таблицы,
        # джойним clients и сортируем таблицы для join в порядке, определенном в CLIENT_TABLES
        push @tables, 'clients' if @tables > 1 && (none {$_ eq 'clients'} @tables);
        @tables = sort {$CLIENT_TABLES{$a}{order} <=> $CLIENT_TABLES{$b}{order}} @tables;
        @tables = ('clients')  if !@tables;
        my $tables_sql = join(' ', $tables[0], map {"left join $_ using (ClientID)"} @tables[1..$#tables]);

        my $fields_sql = join ',' => 'ClientID', @sql_fields;
        my $result = get_hashes_hash_sql(PPC(ClientID => $client_ids),
            [ "select $fields_sql from $tables_sql", where => {ClientID =>  SHARD_IDS} ]);

        return {} unless keys %$result;

        my %clean_fields_set = map {$_ => 1} @clean_fields;

        foreach my $client_id (@$client_ids) {
            my $data = $result->{$client_id} //= {map {$_ => undef} @clean_fields};
            for my $f (keys %$data) {
                # превращаем set-поля в реальные поля
                if (exists $sets2fields{$f}) {
                    my %set_data = map {$_ => 1} split /,/, $data->{$f}//'';
                    for my $sf (@{$sets2fields{$f}}) {
                        $data->{$sf} = $set_data{$sf};
                    }
                }
                delete $data->{$f} if !$clean_fields_set{$f} && !($client_id_required && $f eq 'ClientID');
                # для полей boolean_fields реальные значения Yes/No нужно заменить на 1/0
                my $boolean_fields = [];
                foreach my $table (keys %Client::CLIENT_TABLES) {
                    foreach my $field (@{$Client::CLIENT_TABLES{$table}{boolean_fields}}) {
                        push @$boolean_fields, $field;
                    }
                }
                if (@$boolean_fields && any {$_ eq $f} @$boolean_fields) {
                    $data->{$f} = ($data->{$f} && $data->{$f} eq 'Yes') ? 1 : 0;
                }

                if ($f eq 'deleted_reps') {
                    if ($data->{$f}) {
                        $data->{$f} = from_json($data->{$f});
                    } else {
                        delete $data->{$f};
                    }
                }

                # DIRECT-56254: подставляем флаг для обратной совместимости
                $data->{can_use_day_budget} = 1  if $is_can_use_day_budget_required;
            }
        }
        return $result;
    } else {
        die "no such fields: " . join(',', @$fields);
    }

    return {};
}

=head2 is_valid_client_country

    Проверяет можно ли использовать указанный ID региона в качестве страны клиента/агентства

    $is_valid = Client::is_valid_client_country($region_id);

=cut

sub is_valid_client_country {
    my ($region_id) = @_;

    return 0 unless $region_id && $region_id > 0;

    state $allowed_region_ids;
    $allowed_region_ids //= { map { $_->{region_id} => undef } @geo_regions::COUNTRY_REGIONS };
    return (exists $allowed_region_ids->{$region_id}) ? 1 : 0;
}


=head2 mass_get_clients_brand_ids

    Массовое получение ClientID главного клиента в бренде по ClientID

=cut

sub mass_get_clients_brand_ids {
    my ($client_ids) = @_;

    die 'invalid client_ids array' unless $client_ids && ref($client_ids) eq 'ARRAY';
    return {} unless @$client_ids;

    return get_hash_sql( PPC(ClientID => $client_ids), [
        'SELECT ClientID, brand_ClientID FROM client_brands',
        WHERE => {ClientID => SHARD_IDS}
    ]);
}

=head2 get_client_brand_id

    Получение ClientID главного клиента в бренде по ClientID

=cut

sub get_client_brand_id {
    my ($client_id) = @_;

    die 'invalid client_id' unless $client_id;
    return mass_get_clients_brand_ids([$client_id])->{$client_id};
}

=head2 get_count_perf_creatives

    get_count_perf_creatives(ClientID);

    Возвращает количество performance креативов клиента :

=cut
sub get_count_perf_creatives {
    my ($ClientID) = @_;
    my $count = get_one_field_sql(PPC(ClientID => $ClientID),
        ['SELECT count(*)
            FROM perf_creatives',
            WHERE => {ClientID => $ClientID, creative_type => 'performance'}]);
    return $count;
}

=head2 need_show_belarus_old_rub_warning($client_id)

Находится ли клиент в списке кому показываем предупреждение о смене валюты на BYN DIRECT-69501

=cut

sub need_show_belarus_old_rub_warning {
    my $ClientID = shift;

    my $currency = get_client_currencies($ClientID);
    return if ($currency eq 'BYN' || $currency eq 'YND_FIXED');

    my $flag = get_one_field_sql(
        PPC(ClientID => $ClientID),
        [
            "select 1 from clients_custom_options",
            where => {keyname => $SHOW_BELARUS_OLD_RUB_WARNING_PROPERTY_NAME, ClientID => $ClientID},
            limit => 1
       ]
    ) || 0;

    return $flag;
}

=head2 need_show_belarus_bank_change_warning($client_id)

    Клиента нет в списке кому НЕ показываем тизер про изменение реквизитов счетов Белорусов DIRECT-69950

=cut

sub need_show_belarus_bank_change_warning {
    my $ClientID = shift;

    my $flag = get_one_field_sql(
        PPC(ClientID => $ClientID),
        [
            "select 1 from clients_custom_options",
            where => {keyname => $HIDE_BELARUS_BANK_CHANGE_WARNING_CLIENT_IDS_PROPERTY_NAME, ClientID => $ClientID},
            limit => 1
       ]
    ) || 0;

    return $flag ? 0 : 1;
}

=head2 bind_manager($ClientID, $manager_uid)

    Записать в ppc привязку клиент-менеджер

=cut

sub bind_manager($$) {
    my ($ClientID, $manager_uid) = @_;
    do_insert_into_table(PPC(ClientID => $ClientID),
                         "client_managers", {
                             ClientID => $ClientID,
                             manager_uid => $manager_uid,
                         },
                         ignore => 1,
        );
}

=head2 unbind_manager($ClientID, $manager_uid)

    удалить в ppc привязку клиент-менеджер

=cut

sub unbind_manager($$) {
    my ($ClientID, $manager_uid) = @_;
    do_delete_from_table(PPC(ClientID => $ClientID),
                         "client_managers",
                         where => {
                             ClientID => $ClientID,
                             manager_uid => $manager_uid,
                         }
        );
}

=head2 bind_manager_to_agency($agency_client_id, $manager_uid)

    Записать в ppc привязку менеджера к агентству

=cut

sub bind_manager_to_agency {
    my ($agency_client_id, $manager_uid) = @_;

    my $manager_perms = get_perminfo(uid => $manager_uid) || die "Perminfo for manager $manager_uid not found";
    die "User with $manager_uid not a manager!" if ($manager_perms->{role} // '') ne $ROLE_MANAGER;

    my $manager_client_id = $manager_perms->{ClientID} || die "ClientID for manager $manager_uid not found";

    do_insert_into_table(PPC(ClientID => $agency_client_id),
        'agency_managers',
        {
            agency_client_id  => $agency_client_id,
            manager_client_id => $manager_client_id,
            manager_uid       => $manager_uid,
        },
        ignore => 1,
    );
}

=head2 unbind_manager_from_agency($agency_client_id, $manager_uid)

    удалить из ppc привязку менеджер к агентству

=cut

sub unbind_manager_from_agency {
    my ($agency_client_id, $manager_uid) = @_;

    my $manager_perms = get_perminfo(uid => $manager_uid) || die "Perminfo for manager $manager_uid not found";
    die "User with $manager_uid not a manager!" if ($manager_perms->{role} // '') ne $ROLE_MANAGER;

    my $manager_client_id = $manager_perms->{ClientID} || die "ClientID for manager $manager_uid not found";

    do_delete_from_table(PPC(ClientID => $agency_client_id),
        'agency_managers',
        where => {
            agency_client_id  => $agency_client_id,
            manager_client_id => $manager_client_id,
        }
    );
}

=head2 is_assessor_offer_accepted($client_id)

    Принял ли клиент оферту на изменение его кампаний асессорами
    При условии, что этот клиент уже заведён в Директе

=cut

sub is_assessor_offer_accepted {
    my ($client_id) = @_;
    return 0 unless $client_id;
    return get_user_custom_options($client_id)->{'assessor_offer_accepted'} ? 1 : 0;
}

=head2 get_related_users_by_client_id([$client_id1, $client_id2 ...])

    По каждому из переданных client_id получить список логинов, имеющих доступ к ресурсам клиента

=cut

sub get_related_users_by_client_id {
    my ($client_ids) = @_;

    my $result = {};

    my $client_uids = get_all_sql(
        PPC( ClientID => $client_ids ),
        [
            q/SELECT u.ClientID, u.uid, u.login, c.agency_client_id, c.agency_uid,
            c.chief_uid, 1 as is_rep
                FROM users u JOIN clients c ON (u.ClientID = c.ClientID)
            /,
            WHERE => { 'u.ClientID' => SHARD_IDS },
            q/UNION DISTINCT SELECT u.ClientID, u.uid, u.login, c.agency_client_id, alrc.agency_uid,
            c.chief_uid, 1 as is_rep
            FROM agency_lim_rep_clients alrc JOIN clients c ON (c.ClientID = alrc.ClientID) JOIN users u ON (u.ClientID = c.ClientID)/,
            WHERE => { 'alrc.ClientID' => SHARD_IDS },
        ]
    );

    my @agency_client_ids = uniq grep {$_} map {$_->{agency_client_id}} @$client_uids;
    my (%agency_reps_by_agency_id, %agency_limited_reps);
    if (@agency_client_ids) {
        my $agencies_reps = get_all_sql(
            PPC( ClientID => \@agency_client_ids ),
                [
                    q/SELECT DISTINCT a.ClientID, u.uid, u.login, u.rep_type, 1 as by_agency
                        FROM users u JOIN clients a ON (u.ClientID = a.ClientID)
                    /,
                    WHERE => { 'u.ClientID' => SHARD_IDS }
                ]
        );
        foreach my $rep (@$agencies_reps) {
            if ($rep->{rep_type} eq 'limited') {
                #Ограниченные представители агенства привязываются к клиенту по uid представителя
                $agency_limited_reps{$rep->{uid}} = $rep;
            }
            else {
                #Остальные - по ClientID агенства
                push @{$agency_reps_by_agency_id{$rep->{ClientID}}}, $rep;
            }
        }
    }

    my $manager_uids = get_all_sql(
        PPC( ClientID => $client_ids ),
        [
            q/SELECT DISTINCT ClientID, ManagerUID
                FROM campaigns
                /,
            WHERE => {
                'ClientID' => SHARD_IDS,
                'ManagerUID__gt' => 0,
            }
        ]
    );
    my $managers;
    if (@$manager_uids) {
        my @uids = uniq map {$_->{ManagerUID}} @$manager_uids;
        $managers = get_hashes_hash_sql(
            PPC(uid => \@uids),
            [
                q/SELECT uid, login, 1 as by_manager FROM users/, WHERE => {uid => SHARD_IDS},
            ]
        );
    }

    my $byUidHash = {};
    foreach my $row (@$client_uids) {
        _add_row_to_result($row => $result,undef,  $byUidHash);
        if ($row->{agency_client_id}) {
            _add_row_to_result( $_ => $result, $row->{ClientID}, $byUidHash ) foreach @{$agency_reps_by_agency_id{$row->{agency_client_id}}}
        }
        if ($row->{agency_uid}) {
            my $limited_rep = $agency_limited_reps{$row->{agency_uid}};
            _add_row_to_result( $limited_rep => $result, $row->{ClientID}, $byUidHash ) if $limited_rep;
        }
    }

    foreach my $row (@$manager_uids) {
        my $manager = $managers->{$row->{ManagerUID}};
        _add_row_to_result( $manager => $result, $row->{ClientID}, $byUidHash);
    }

    my $related_freelancers = get_all_sql(
            PPC( ClientID => $client_ids ),
            [
                q/SELECT DISTINCT client_id_to as ClientID, client_id_from as freelancer FROM clients_relations/,
                WHERE => {client_id_to => SHARD_IDS, type => 'freelancer'}
            ]);
    if (@$related_freelancers) {
        my %clients_by_freelancer;
        foreach my $row (@$related_freelancers){
            push @{$clients_by_freelancer{$row->{freelancer}}}, $row->{ClientID};
        }
        my $freelancers_uids = get_all_sql(
            PPC(ClientID => [keys %clients_by_freelancer]),
            [
                q/SELECT uid, ClientID as frClientID, login FROM users/,
                WHERE => {ClientID => SHARD_IDS},
            ]);
        foreach my $fr_info (@$freelancers_uids) {
            foreach my $client_id (@{$clients_by_freelancer{$fr_info->{frClientID}}}) {
                _add_row_to_result( $fr_info => $result, $client_id, $byUidHash);
            }
        }
    }

    return $result;
}

# Вспомогательный метод для get_related_users_by_client_id
sub _add_row_to_result {
    my ($row, $result, $ClientID, $byUidHash) = @_;

    my $uid      = $row->{uid};
    $ClientID //= $row->{ClientID};

    return if $byUidHash->{$ClientID}->{$uid};

    my $login    = $row->{login};

    $byUidHash->{$ClientID}->{$uid} = 1;
    my $uid_info = { uid => $uid, login => $login,};
    $uid_info->{is_agency_uid}  = JSON::true if $row->{by_agency};
    $uid_info->{is_manager_uid} = JSON::true if $row->{by_manager};
    $uid_info->{is_main_rep}    = JSON::true if $uid == $row->{chief_uid};
    $uid_info->{is_rep}         = JSON::true if $row->{is_rep};

    push @{$result->{$ClientID}}, $uid_info;

    return
}

=head2 add_client_reminder_letters_subscription($ClientID)

    Подписать клиента на получение мотивирующих писем

=cut
sub add_client_reminder_letters_subscription {
    my $ClientID = shift;
    do_insert_into_table(PPC(ClientID => $ClientID), 'client_reminder_letters', {ClientID => $ClientID});
}

=head2 get_count_received_deals

    get_count_received_deals(ClientID);

    Возвращает количество полученных сделок агентства

=cut
sub get_count_received_deals {
    my ($ClientID) = @_;
    my $count = get_one_field_sql(PPC(ClientID => $ClientID),
        ['SELECT count(*)
            FROM deals',
            WHERE => {ClientID => $ClientID, direct_deal_status => 'Received'}]);
    return $count;
}


=head2 get_clients_auto_overdraft_info($client_ids)

    Получает всю информацию о клиентах, которая необходима для рассчёта доступного лимита автоовердрафта.

=cut
sub get_clients_auto_overdraft_info {
    my ($client_ids) = @_;

    return get_hashes_hash_sql(PPC(ClientID => $client_ids), [q{
                    SELECT
                        cl.ClientID as clientID,
                        IFNULL(clo.debt, 0) as debt,
                        IFNULL(clo.overdraft_lim, 0) as overdraft_lim,
                        IFNULL(clo.auto_overdraft_lim, 0) as auto_overdraft_lim,
                        IFNULL(clo.statusBalanceBanned, 'No') as statusBalanceBanned
                    FROM clients cl
                        LEFT JOIN clients_options clo ON clo.ClientID = cl.ClientID
                    },
        where => {
            'cl.ClientID' => SHARD_IDS
        }
    ]);
}

=head2 clear_cache()

    Очистить все определённые в модуле кеши

=cut
sub clear_cache {
    $CACHE->clear();
}

=head2 enable_touch_direct_feature($client_id)

    Включает клиенту фичу touch_direct_enabled с помощью записи в таблицу clients_features

=cut
sub enable_touch_direct_feature {
    my ($client_id) = @_;

    my $feature_id = get_one_field_sql(PPCDICT, "
        SELECT feature_id
        FROM features
        WHERE feature_text_id = 'touch_direct_enabled'
    ");
    if (!$feature_id) {
        die "no feature 'touch_direct_enabled' in PPCDICT";
    }
    do_insert_into_table(PPC(ClientID => $client_id),
        'clients_features',
        {ClientID   => $client_id, feature_id => $feature_id, is_enabled => 1},
        on_duplicate_key_update => 1,
        key => [qw/ClientID feature_id/]
    );
    clear_cache();
    Client::ClientFeatures::clear_cache();
}

=head2 show_support_chat

Нужно ли пользователю показать чат поддержки. Предполагается, что фичу support_chat проверили заранее.
Вход:
    rbac_login_rights из rbac_login_check с дополнительно подмешанным ключом client_is_serviced
    country_region_id из ppc.clients
    lang из lang_auto_detect
Выход: показывать чат или нет.

=cut

sub show_support_chat {
    my ($rbac_login_rights, $country_region_id, $lang) = @_;
    return 0 if ($lang ne 'ru' || $country_region_id != $geo_regions::RUS);
    return ($rbac_login_rights->{is_any_client}
        && !$rbac_login_rights->{client_is_serviced} && !$rbac_login_rights->{client_have_agency}) ? 1 : 0;
}

=head2 is_new_wallet_warnings_client

    Для переданного ClientID опредяет, разрешено ли ему отправлять новые письма об остатке на Общем счете

=cut

sub is_new_wallet_warnings_client {
    my $client_id = shift;

    return Client::ClientFeatures::has_new_wallet_warnings_enabled($client_id);
}

=head2 is_send_sms_despite_sms_flags_for_new_wallet_warnings_client

    Для переданного ClientID опредяет, разрешено ли ему принудительно отправлять новые письма об остатке на Общем счете

=cut

sub is_send_sms_despite_sms_flags_for_new_wallet_warnings_client {
    my $client_id = shift;

    return Client::ClientFeatures::has_send_sms_despite_sms_flags_for_new_wallet_warnings($client_id);
}

=head2 is_universal_campaign_client

    Для переданного ClientID опредяет, являестя ли клиент UC-клиентом (т.е. клиентом у которого только UC-кампании)

=cut

sub is_universal_campaign_client {
    my $client_id = shift;

    return mass_is_universal_campaign_client([$client_id])->{$client_id};
}

=head2 mass_is_universal_campaign_client

    Принимает на вход список ClientID. Для каждого переданного ClientID возвращает признак того, являестя ли клиент UC-клиентом (т.е. клиентом у которого только UC-кампании):
        1 - UC-клиент
        0 - не UC-клиент

=cut

sub mass_is_universal_campaign_client {
    my ($client_ids) = @_;

    return {} unless @$client_ids;

    my $feature_states = {};
    foreach my $client_ids_chunk (chunks($client_ids, 100)) {
        hash_merge($feature_states, Client::ClientFeatures::is_universal_campaigns_enabled_for_client_ids($client_ids_chunk));
    }

    my $client_ids_with_feature_enabled = [ grep { defined $_ && exists $feature_states->{$_} && $feature_states->{$_} == 1 } keys %$feature_states ];

    my $clients_data = get_hashes_hash_sql(PPC(ClientID => $client_ids_with_feature_enabled), [
       'SELECT c.ClientID, COUNT(cid) AS camps_count, SUM(CASE WHEN c.opts LIKE "%is_universal%" THEN 1 ELSE 0 END) AS universal_camps_count
        FROM campaigns c',
        where => {ClientID => SHARD_IDS, type => get_camp_kind_types("base"), statusEmpty => 'No'}, 'GROUP BY c.ClientID']);
    my $result = {};
    for my $client_id (@$client_ids) {
        if(    exists $feature_states->{$client_id} && $feature_states->{$client_id} == 1
            && exists $clients_data->{$client_id} && $clients_data->{$client_id}->{camps_count} == $clients_data->{$client_id}->{universal_camps_count}) {
            $result->{$client_id} = 1;
        } else {
            $result->{$client_id} = 0;
        }
   }

   return $result;
}


=head2 mass_is_new_moderate_send_warn_enabled_for_client

    Принимает на вход список ClientID. Для каждого переданного ClientID возвращает признак того, нужно ли ему отправлять письмо moderate_send_warn:
        1 - не отправляем письмо moderate_send_warn
        0 - отправляем

=cut

sub mass_is_new_moderate_send_warn_enabled_for_client {
    my ($client_ids) = @_;

    return {} unless @$client_ids;

    my $feature_states = {};
    foreach my $client_ids_chunk (chunks($client_ids, 100)) {
        hash_merge($feature_states, Client::ClientFeatures::is_new_moderate_send_warn_enabled_for_client_ids($client_ids_chunk));
    }

   return $feature_states;
}

=head2 has_cashbacks_available

    Определяет, что клиенту доступны кешбэки "в общем", то есть в интерфейсе будут показываться кешбэчные элементы.
    Признак не включает в себя добровольный выход пользователя из программы лояльности.

=cut

sub has_cashbacks_available {
    # Исправление тут требует исправления в логике GraphQL:
    # https://a.yandex-team.ru/arc/trunk/arcadia/direct/libs-internal/grid-processing/src/main/java/ru/yandex/direct/grid/processing/service/client/ClientDataService.java
    # (см. метод hasCashbackAvailable)

    my ($c, $client_data) = @_;

    my @available_regions = ($geo_regions::RUS,  $geo_regions::BY, $geo_regions::KAZ);
    my @available_currencies = ('RUB', 'BYN', 'KZT');

    my $isRuClientWithRuCurrency = (
        ($client_data->{country_region_id} eq $geo_regions::RUS) || !$client_data->{country_region_id}
    ) && ($client_data->{work_currency} eq 'RUB');
    my $isAvailableRegionAndCurrency = (
        (any {$_ eq $client_data->{country_region_id}} @available_regions) ||
        !$client_data->{country_region_id}
    ) && (any {$_ eq $client_data->{work_currency}} @available_currencies);
    my $isCashbackAvailable = Client::ClientFeatures::has_cashback_page_for_by_and_kz_feature_enabled($c->client_client_id) ?
        $isAvailableRegionAndCurrency : $isRuClientWithRuCurrency;

    return Client::ClientFeatures::has_cashback_page_feature_enabled($c->client_client_id)
        && $isCashbackAvailable;
}


=head2 mass_is_new_stat_rollbacks_enabled_for_client

    Принимает на вход список ClientID. Для каждого переданного ClientID возвращает признак того, можно ли ему отправлять новое письмо об откатах статистики:
        1 - отправляем новое письмо
        0 - отправляем старое письмо

=cut

sub mass_is_new_stat_rollbacks_enabled_for_client {
    my ($client_ids) = @_;

    return {} unless @$client_ids;

    my $feature_states = {};
    foreach my $client_ids_chunk (chunks($client_ids, 100)) {
        hash_merge($feature_states, Client::ClientFeatures::is_new_stat_rollbacks_enabled_for_client_ids($client_ids_chunk));
    }

   return $feature_states;
}


=head2 is_business_unit_client
    Проверяет является ли ClientID бизнес юнитом
=cut
sub is_business_unit_client {
    my $client_id = shift;
    state $business_unit_client_ids_property = Property->new($BUSINESS_UNIT_CLIENT_IDS_PROPERTY_NAME);
    my %client_ids = map {$_ => 1} split ",", $business_unit_client_ids_property->get(60) || "";
    return ($client_ids{$client_id}) ? 1 : 0
}

=head2 validate_tin_and_type

Валидация ИНН и его типа. На входе необязательная строка из цифр $tin и строка с типом "physical" | "legal", обязательная, если указан $tin.
Тип может быть указан без ИНН.
На выходе непереведённая строка с ошибкой или undef, если значения и их сочетание валидны.
Дублирует проверки фронта, так что проверка должна реагировать только на скрафченные запросы.

=cut

sub validate_tin_and_type {
    my ($tin, $tin_type) = @_;
    if (defined $tin && !defined $tin_type) {
        return iget_noop("Не сделан выбор между физическим и юридическим лицом");
    }
    if (!defined $tin_type) {
        return undef; # совсем нечего проверять
    }
    if ($tin_type ne "physical" && $tin_type ne "legal") {
        return iget_noop("Ошибка в выборе между физическим и юридическим лицом");
    }
    if (!defined $tin) {
        return undef; # больше нечего проверять
    }

    my $msg = iget_noop("Неверный формат ИНН");
    if ($tin_type eq "physical") {
        if ($tin !~ /^[0-9]{12}$/ || _check_physical_tin($tin)) {
            return $msg;
        }
    }
    if ($tin_type eq "legal") {
        if ($tin !~ /^[0-9]{10}$/ || _check_legal_tin($tin)) {
            return $msg;
        }
    }
    return undef;
}

{
    # списано с market/mbi/mbi/mbi-model/src/main/java/ru/yandex/market/api/cpa/yam/validation/InnValidator.java
    my $COEFFICIENTS_9 = [2, 4, 10, 3, 5, 9, 4, 6, 8];
    my $COEFFICIENTS_10 = [7, 2, 4, 10, 3, 5, 9, 4, 6, 8];
    my $COEFFICIENTS_11 = [3, 7, 2, 4, 10, 3, 5, 9, 4, 6, 8];

    sub _check_tin_sum {
        my ($digits, $coeff) = @_;
        my @n = map {0 + $_} split //, $digits;

        my $sum = 0;
        for my $i (0 .. $#$coeff) {
            $sum += $n[$i] * $coeff->[$i];
        }
        $sum %= 11;
        $sum %= 10;

        return $n[$#$coeff + 1] != $sum;
    }

    sub _check_physical_tin {
        my $digits = shift;
        return _check_tin_sum($digits, $COEFFICIENTS_10) || _check_tin_sum($digits, $COEFFICIENTS_11);
    }

    sub _check_legal_tin {
        my $digits = shift;
        return _check_tin_sum($digits, $COEFFICIENTS_9);
    }
}

1;

