package Client::ClientFeatures;

=head1 NAME

package ClientFeatures;
    Client

=head1 DESCRIPTION

    Работа клиентскими фичами

=cut
use Direct::Modern;
use Carp qw/croak/;

use Settings;

use Direct::Feature;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::Validate qw/is_valid_id/;

use List::MoreUtils qw/any/;

use Rbac qw/get_perminfo/;
use Tools;
use PrimitivesIds qw/ get_clientid /;

use base qw/Exporter/;
our @EXPORT = qw/
    has_feature
    get_features_enabled_for_client
/;
our $CACHE_CLIENT_FEATURE_ACCESS = {};

=head2 has_feature

    Проверяет наличие заданной фичи у клиента.
    Возвращает 1|0/

=cut
sub has_feature {
    my ($client_id, $new_feature_name) = @_;

    return has_access_to_new_feature_from_java($client_id, $new_feature_name) // 0;
}

=head2 allowed_features

    вернуть ссылку на список включённых фич по ClientID

=cut
sub allowed_features {
    my ($client_id) =  @_;
    if (!is_valid_id($client_id)) {
        return [];
    }
    if (!defined $CACHE_CLIENT_FEATURE_ACCESS->{$client_id}) {
        $CACHE_CLIENT_FEATURE_ACCESS->{$client_id} = get_abt_info($client_id)->{features};
    }
    return [ @{$CACHE_CLIENT_FEATURE_ACCESS->{$client_id}} ];
}

=head2 get_abt_info

    Получить информацию о AB-эксперименте, в котором участвует клиент и включённых фичах
    Пишет полученные фичи в кеш.

=cut
sub get_abt_info {
    my ($ClientID) = @_;
    unless (defined $ClientID) {
        return {
            boxes         => '',
            boxes_crypted => '',
            features      => [],
        };
    }
    my $res = get_abt_info_parallel([$ClientID]);
    return $res->{$ClientID};
}

=head2 get_abt_info_parallel

    Версия get_abt_info для нескольких клиентов.
    Получить информацию о AB-экспериментах, в котором участвуют клиенты и включённых фичах
    Пишет полученные фичи в кеш.

=cut
sub get_abt_info_parallel {
    my @client_ids = @{ $_[0] };

    my %result;
    for my $invalid_client_id (grep { !is_valid_id($_) } @client_ids) {
        $result{$invalid_client_id} = {
            boxes         => '',
            boxes_crypted => '',
            features      => [],
        };
    }

    my $resp_hash = Direct::Feature::abt_info_parallel([grep {!exists $result{$_}} @client_ids]);
    for my $ClientID (keys %$resp_hash) {
        my $resp = $resp_hash->{$ClientID};
        if (!$resp->{error}) {
            $CACHE_CLIENT_FEATURE_ACCESS->{$ClientID} = [@{$resp->{features}}];
            $result{$ClientID} = {
                boxes         => $resp->{boxes},
                boxes_crypted => $resp->{boxes_crypted},
                features      => $resp->{features},
            };
        } else {
            utf8::decode($resp->{error}) unless utf8::is_utf8($resp->{error});
            die sprintf('java intapi/abt_info error: client_id %s, %s', $ClientID, $resp->{error});
        }
    }
    return \%result;
}

=head2 clear_cache()

    Очистить все определённые в модуле кеши

=cut
sub clear_cache {
    $CACHE_CLIENT_FEATURE_ACCESS = {};
}

=head2 has_access_to_new_feature_from_java($client_id, $feature_name)

    Включена ли у клиента такая фича?

=cut

sub has_access_to_new_feature_from_java {
    my ($client_id, $new_feature_name) = @_;

    if (grep {$_ eq $new_feature_name} @{allowed_features($client_id)}) {
        return 1;
    } else {
        return 0;
    }
}

sub _get_features_allowed_for_client_ids {
    my ($client_ids) = @_;
    return {} unless $client_ids && @$client_ids;

    my $features_resp = Direct::Feature::get_clients_uids_features($client_ids);

    if ($features_resp->{error}) {
        die sprintf('java intapi error: client_ids %s, error: %s', join(", ", @$client_ids), $features_resp->{error});
    }
    return $features_resp->{per_client_id_features};
}

sub _is_feature_allowed_for_client_ids {
    my ($client_ids, $new_feature_name) = @_;
    return {} unless $client_ids && @$client_ids;

    my $allowed_features_by_client_id = _get_features_allowed_for_client_ids($client_ids);

    my $result = {};
    foreach my $client_id (@$client_ids) {
        if (my $allowed_features = $allowed_features_by_client_id->{$client_id}) {
            $result->{ $client_id } = (any { $_ eq $new_feature_name } @$allowed_features) ? 1 : 0;
        }
    }
    return $result;
}

# unified wrapper for java feature request
sub _does_client_have_feature {
    my ($client_id, $feature) = @_;
    if (!$client_id) {
        local $Carp::CarpLevel = $Carp::CarpLevel + 1;
        croak 'client_id is required';
    }
    return has_access_to_new_feature_from_java($client_id, $feature) ? 1 : 0;
}

my %FEATURE_NAME_FUNCTION_MAP = (
    social_advertising => \&social_advertising_feature,
    cpv_strategies_enabled => \&cpv_strategies_enabled_feature,
    cpc_device_modifiers => \&has_cpc_device_modifiers_allowed_feature,
    cpm_video_device_modifiers => \&has_cpm_video_device_modifiers_allowed_feature,
    mobile_os_bid_modifier_enabled => \&is_mobile_os_bid_modifier_enabled,
    canvas_range_ratio_cpc => \&is_canvas_range_ratio_cpc,
    in_app_events_in_rmp_enabled => \&has_in_app_events_in_rmp_enabled_feature,
    crr_strategy_allowed => \&has_crr_strategy_feature,
    default_autobudget_avg_cpa => \&has_default_avg_cpa_feature,
    default_autobudget_avg_click_with_week_budget => \&has_default_autobudget_avg_click_with_week_budget_feature,
    default_autobudget_roi => \&has_default_autobudget_roi_feature,
    alone_trafaret_option_enabled => \&has_alone_trafaret_option_feature,
    can_require_filtration_by_dont_show_domains => \&has_can_require_filtration_by_dont_show_domains_feature,
    require_filtration_by_dont_show_domains => \&has_can_require_filtration_by_dont_show_domains_feature,
    can_require_filtration_by_dont_show_domains_in_cpm => \&has_can_require_filtration_by_dont_show_domains_in_cpm_feature,
    require_filtration_by_dont_show_domains_in_cpm => \&has_can_require_filtration_by_dont_show_domains_in_cpm_feature,
    turbo_app_allowed => \&has_turbo_app_allowed,
    is_demography_bid_modifier_unknown_age_allowed => \&get_is_feature_demography_bid_modifier_unknown_age_allowed,
    mobile_content_cpc_video_allowed => \&has_mobile_content_cpc_video_allowed,
    support_chat => \&has_support_chat_allowed_feature,
    checked_support_chat => \&has_checked_support_chat_allowed_feature,
    cpm_deals => \&has_cpm_deals_allowed_feature,
    enable_sidebar_optimize => \&is_enable_sidebar_optimize,
    content_promotion_video => \&has_content_promotion_video_allowed_feature,
    content_promotion_collection => \&is_feature_content_promotion_collection_enabled,
    billing_order_domains_offline_report_enabled => \&has_billing_order_domains_offline_report_enabled_feature,
    agency_kpi_offline_report_enabled => \&has_agency_kpi_offline_report_enabled_feature,
    b2b_balance_cart => \&has_b2b_balance_cart_feature,
    show_daily_budget_recommendations_in_old_interface => \&has_show_daily_budget_recommendations_in_old_interface_feature,
    show_aggregated_status_open_beta => \&has_show_aggregated_status_open_beta_feature,
    is_universal_campaigns_enabled => \&has_universal_campaigns_allowed,
    new_campaign_info_enabled => \&has_new_campaign_info_enabled_feature,
    new_campaign_page_enabled => \&has_new_campaign_page_enabled_feature,
    edit_avg_cpm_without_restart => \&has_edit_avg_cpm_without_restart_feature,
    cpm_yndx_frontpage_profile => \&has_cpm_yndx_frontpage_profile_allowed_feature,
    cpm_audio => \&has_cpm_audio_feature,
    has_mobile_app_goals_for_text_campaign_allowed => \&has_mobile_app_goals_for_text_campaign_allowed,
    mobile_app_goals_for_text_campaign_statistics_enabled => \&has_mobile_app_goals_for_text_campaign_statistics_enabled,
    mobile_app_goals_for_text_campaign_strategy_enabled => \&has_mobile_app_goals_for_text_campaign_strategy_enabled,
    cpm_geoproduct_enabled => \&has_cpm_geoproduct_enabled,
    mass_edit_regions_for_dna_enabled => \&has_mass_edit_regions_for_dna_enabled_feature,
    cpm_banner_enable_empty_ret_cond_json => \&has_cpm_banner_enable_empty_ret_cond_json_allowed_feature,
    cpm_outdoor_forecast => \&has_cpm_outdoor_forecast_feature,
    minus_words_lib => \&has_minus_words_lib_feature,
    banner_prices => \&has_banner_prices_enabled,
    indoor_segments => \&has_indoor_segments_feature,
    cpm_video_several_pixels_enabled => \&has_cpm_video_several_pixels_enabled_feature,
    relevance_match_for_new_groups_enabled => \&relevance_match_for_new_groups_enabled_feature,
    tns_enabled => \&has_tns_counter_feature,
    landings_wizard_allowed => \&has_landings_wizard_allowed_feature,
    disable_any_domains_allowed => \&has_disable_any_domains_allowed_feature,
    disable_mail_ru_domain_allowed => \&has_disable_mail_ru_domain_allowed_feature,
    can_have_internal_dont_show_domains => \&has_can_have_internal_dont_show_domains_feature,
    cpm_banner_sov_in_stat => \&has_cpm_banner_sov_in_stat_feature,
    cpm_yndx_frontpage_sov_in_stat => \&has_cpm_yndx_frontpage_sov_in_stat_feature,
    cpm_price_sov_in_stat => \&has_cpm_price_sov_in_stat_feature,
    goals_only_with_campaign_counters_used => \&has_goals_only_with_campaign_counters_used,
    turbo_page_types_enabled_in_mol => \&has_turbo_page_types_enabled_in_mol_feature,
    new_master_token_authentication => \&has_new_master_token_authentication_feature,
    is_cpa_pay_for_conversions_extended_mode_allowed => \&has_cpa_pay_for_conversions_extended_mode_allowed,
    is_has_cpa_pay_for_conversions_mobile_apps_allowed => \&has_cpa_pay_for_conversions_mobile_apps_allowed,
    has_all_meaningful_goals_for_pay_for_conversion_strategies_allowed => \&has_all_meaningful_goals_for_pay_for_conversion_strategies_allowed,
    impression_standard_time => \&has_impression_standard_time_feature,
    stat_4_digits_precision => \&has_stat_4_digits_precision_feature,
    lal_segments_enabled => \&has_lal_segments_feature,
    retargeting_only_lal_enabled => \&has_retargeting_only_lal_feature,
    disable_all_goals_optimization_for_dna => \&has_disable_all_goals_optimization_for_dna_feature,
    is_hide_old_show_camps => \&has_hide_old_show_camps_for_dna_feature,
    is_grid_enabled => \&has_grid_enabled_for_dna,
    is_show_dna_by_default => \&has_soft_hide_show_camps,
    collecting_verified_phones => \&collecting_verified_phones_feature,
    facelift_disabled_for_dna => \&facelift_disabled_for_dna_feature,
    enabled_change_offer_for_clients_from_turkey => \&enabled_change_offer_for_clients_from_turkey_dna_feature,
    simplified_strategy_view_enabled =>\&has_simplified_strategy_view_enabled_feature,
    loading_dna_scripts_before_old_interface_scripts_enabled =>\&loading_dna_scripts_before_old_interface_scripts_enabled_feature,
    conversion_strategy_learning_status_enabled => \&has_conversion_strategy_learning_status_enabled,
    increased_cpa_limit_for_pay_for_conversion => \&has_increased_cpa_limit_for_pay_for_conversion,
    pay_with_cash_disabled => \&pay_with_cash_disabled,
    new_cash_payment_enabled => \&new_cash_payment_enabled,
    creative_frontpage_728_90_disabled => \&is_creative_frontpage_728_90_disabled,
    brandsafety_base_categories_stat => \&has_brandsafety_base_categories_stat,
    pay_for_conversion_visible_in_filter_optimization_entity_for_dna => \&pay_for_conversion_visible_in_filter_optimization_entity_for_dna,
    cpm_fixed_data_in_mol => \&cpm_fixed_data_in_mol,
    cpm_additional_data_in_mol => \&cpm_additional_data_in_mol,
    do_not_check_passport_karma_for_xls_access => \&do_not_check_passport_karma_for_xls_access,
    telegram_enabled => \&is_telegram_enabled,
    disable_number_id_and_short_bundle_id_allowed => \&has_disable_number_id_and_short_bundle_id_allowed_feature,
    allow_copy_conversion_strategy_between_logins => \&has_copy_conversion_strategy_between_logins_feature,
    avg_nshow_for_strategy_in_moc => \&has_avg_nshow_for_strategy_in_moc_feature,
    avg_nshow_for_cpm_campaigns_in_moc => \&has_avg_nshow_for_cpm_campaigns_in_moc_feature,
    avg_nshow_complete_for_strategy_in_moc => \&has_avg_nshow_complete_for_strategy_in_moc_feature,
    avg_nshow_complete_for_cpm_campaigns_in_moc => \&has_avg_nshow_complete_for_cpm_campaigns_in_moc_feature,
    auction_stats_for_all_campaigns => \&has_auction_stats_for_all_campaigns_feature,
    disable_autobudget_week_bundle => \&has_disable_autobudget_week_bundle_in_api_feature,
    cross_device_attribution_types_enabled => \&cross_device_attribution_types_enabled,
    has_s2s_tracking_feature => \&has_s2s_tracking_feature,
    flat_cpc_disabled => \&has_flat_cpc_disabled,
    flat_cpc_adding_disabled => \&has_flat_cpc_adding_disabled
);

=head2 get_features_enabled_for_client (client_id, ArrayRef features, Hash opt)

    Возвращает значения для фич, переданных в features

=cut

sub get_features_enabled_for_client  {
    my $client_id = shift;
    my $features = shift;
    my (%opt) = @_;
    my %result = ();

    foreach my $feature_name (@$features) {
        my $fresult = $FEATURE_NAME_FUNCTION_MAP{$feature_name}($client_id);
        if ($opt{put_any_result} || !$opt{put_any_result} && $fresult) {
            $result{$feature_name} = $fresult;
        }
    }
    return \%result;
}

=head2 get_is_featureTurboLandingEnabled(client_id => $client_id, rights => $rbac_login_rights)

    Возвращает признак того, что клиенту доступны турболендинги.

=cut

sub get_is_featureTurboLandingEnabled {
    my (%opt) = @_;

    my ($client_id, $rbac_login_rights) = @opt{qw/client_id rights/};

    croak 'cliend_id required' unless ($client_id);

    #Турболендинги
    return 1 if $rbac_login_rights && $rbac_login_rights->{super_control};

    return 1 if get_one_field_sql(PPC(ClientID =>$client_id), [q/SELECT 1 FROM turbolandings/, WHERE => {ClientID => SHARD_IDS}, LIMIT => 1]);

    return 1 if has_access_to_new_feature_from_java($client_id, 'mobile_landing');

    return 0;
}

=head2 get_is_feature_cpm_banner_enabled(client_id => $client_id)

    Возвращает признак того, что клиенту доступны кампании медийного охвата.

=cut

sub get_is_feature_cpm_banner_enabled {
    return _does_client_have_feature(+{@_}->{client_id}, 'cpm_banner');
}

=head2 has_cpm_banner_enable_empty_ret_cond_json_allowed_feature($client_id)

    Возвращает признак того, что клиенту доступна возможность сохранять пустое условие в кампаниях с типом cpm_banner.

=cut

sub has_cpm_banner_enable_empty_ret_cond_json_allowed_feature {
    return _does_client_have_feature(shift, 'cpm_banner_enable_empty_ret_cond_json');
}

=head2 has_recommendation_availability_feature($client_id)

    Возвращает признак того, что клиенту доступна возможность показа рекомендаций.

=cut

sub has_recommendation_availability_feature {
    return _does_client_have_feature(shift, 'recommendation_availability');
}

=head2 has_show_goals_stat_in_grid_feature($client_id)

    Возвращает признак того, что клиенту доступна возможность показа статистики по целям в гридах.

=cut

sub has_show_goals_stat_in_grid_feature {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "show_goals_stat_in_grid") ? 1 : 0;
}

=head2 has_universal_campaigns_allowed($client_id)

    Возвращает признак доступности универсальных кампаний

=cut

sub has_universal_campaigns_allowed {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "universal_campaigns_enabled") ? 1 : 0;
}

=head2 has_universal_campaigns_enabled_for_uac($client_id)

    Возвращает признак доступности универсальных кампаний

=cut

sub has_universal_campaigns_enabled_for_uac {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "universal_campaigns_enabled_for_uac") ? 1 : 0;
}

=head2 is_universal_campaigns_enabled_for_client_ids

    Возвращает список клиентов с признаком доступности универсальных кампаний

=cut
sub is_universal_campaigns_enabled_for_client_ids {
    my $client_ids = shift;
    return _is_feature_allowed_for_client_ids($client_ids, 'universal_campaigns_enabled_for_uac');
}

=head2 has_unavailable_auto_goals_allowed($client_id)

    Возвращает признак возможности использовать автоцели (source=auto, ecom-цели, цели на технических счетчиках) без доступа

=cut

sub has_unavailable_auto_goals_allowed {
    my $client_id =  shift;
    return 0 unless $client_id;
    return ( has_access_to_new_feature_from_java($client_id, "direct_unavailable_auto_goals_allowed")
             || has_access_to_new_feature_from_java($client_id, "uac_unavailable_auto_goals_allowed")
           ) ? 1 : 0
}

=head2 has_unavailable_goals_allowed($client_id)

    Возвращает признак возможности использовать недоступные цели

=cut

sub has_unavailable_goals_allowed {
    my $client_id =  shift;
    return 0 unless $client_id;
    return ( has_access_to_new_feature_from_java($client_id, "uac_unavailable_goals_allowed")
             || has_direct_unavailable_goals_allowed($client_id)
           ) ? 1 : 0
}

=head2 has_direct_unavailable_goals_allowed($client_id)

    Возвращает признак возможности использовать цели без доступа в профи интерфейсе и АПИ

=cut

sub has_direct_unavailable_goals_allowed {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "direct_unavailable_goals_allowed") ? 1 : 0
}

=head2 is_send_metrika_counters_with_autogoals_in_perl_enabled($client_id)

    Возвращает признак того, что клиенту доступна отправка всех счетчиков метрики с автоцелями из недоступных счетчиков

=cut

sub is_send_metrika_counters_with_autogoals_in_perl_enabled {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "send_metrika_counters_with_autogoals_in_perl_enabled") ? 1 : 0;
}

=head2 is_send_unavailable_goals_in_perl_enabled($client_id)

    Возвращает признак того, что клиенту доступна отправка всех целей в perl (в МОЛ), включая недоступные

=cut

sub is_send_unavailable_goals_in_perl_enabled {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "send_unavailable_goals_in_perl_enabled") ? 1 : 0;
}

=head2 has_get_revenue_only_by_available_goals_in_perl($client_id)

    Возвращает признак того, что для клиента получаем доход из БК только по доступным целям

=cut

sub has_get_revenue_only_by_available_goals_in_perl {
    my $client_id = shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "get_revenue_only_by_available_goals_in_perl") ? 1 : 0;
}

=head2 is_paused_by_day_budget_warnings_enabled_for_client_ids

    Возвращает соответствие клиентов и признаком отправки уведомлений об остановки по дневному бюджету

=cut

sub is_paused_by_day_budget_warnings_enabled_for_client_ids {
    my $client_ids = shift;
    return _is_feature_allowed_for_client_ids($client_ids, 'paused_by_day_budget_warnings');
}

=head2 is_new_moderate_send_warn_enabled_for_client_ids

    Возвращает ссылку на хеш клиентов, которым нужно отправлять новые письма moderate_send_warn из Нирваны:
    {
        $client_id => [0|1],
        ...
    }

=cut

sub is_new_moderate_send_warn_enabled_for_client_ids {
    my $client_ids = shift;
    return _is_feature_allowed_for_client_ids($client_ids, 'new_moderate_send_warn_enabled');
}

=head2 is_new_stat_rollbacks_enabled_for_client_ids

    Возвращает список клиентов с признаком доступности новых писем об откатах статистики

=cut

sub is_new_stat_rollbacks_enabled_for_client_ids {
    my $client_ids = shift;
    return _is_feature_allowed_for_client_ids($client_ids, 'new_stat_rollback_notification_enabled');
}

=head2 has_new_wallet_warnings_enabled

    Возвращает признак того, что клиенту доступна отправка новых писем об остатке денег общем счете

=cut

sub has_new_wallet_warnings_enabled {
    my $client_id = shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, 'new_wallet_warnings_enabled') ? 1 : 0;
}

=head2 has_send_sms_despite_sms_flags_for_new_wallet_warnings

    Возвращает признак того, что клиенту доступна принудительная отправка новых писем об остатке денег на общем счете

=cut

sub has_send_sms_despite_sms_flags_for_new_wallet_warnings {
    my $client_id = shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, 'send_sms_despite_sms_flags_for_new_wallet_warnings') ? 1 : 0;
}

=head2 has_cpa_pay_for_conversions_extended_mode_allowed($client_id)

    Возвращает признак того, что клиенту доступна расширенная функциональность в оплате за конверсии.
    Работает только вместе с включенной фичей оплаты конверсий

=cut

sub has_cpa_pay_for_conversions_extended_mode_allowed {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "cpa_pay_for_conversions_extended_mode") ? 1 : 0;
}

=head2 has_all_meaningful_goals_for_pay_for_conversion_strategies_allowed($client_id)

    Возвращает признак того, что клиенту доступен выбор оптимизации "по нескольким целям" при выборе оплаты за конверсии в ДРР стратегии.
    Работает только вместе с включенной фичей оплаты конверсий

=cut

sub has_all_meaningful_goals_for_pay_for_conversion_strategies_allowed {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "allow_all_meaningful_goals_for_pay_for_conversion_strategies") ? 1 : 0;
}

=head2 has_cpa_pay_for_conversions_mobile_apps_allowed($client_id)

    Возвращает признак того, что клиенту доступна расширенная функциональность в оплате за конверсии для РМП.

=cut

sub has_cpa_pay_for_conversions_mobile_apps_allowed {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "cpa_pay_for_conversions_mobile_apps_allowed") ? 1 : 0;
}

=head2 get_is_feature_brand_lift_enabled(client_id => $client_id)

    Возвращает признак того, что клиенту доступны опросы Brand-lift.

=cut

sub get_is_feature_brand_lift_enabled {
    return _does_client_have_feature(+{@_}->{client_id}, 'brand_lift');
}

=head2 get_is_brand_lift_hidden_enabled($client_id)

    Возвращает признак того, что клиент проводит скрытые Brand-lift'ы.

=cut

sub get_is_brand_lift_hidden_enabled {
    my $client_id = shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "brand_lift_hidden") ? 1 : 0;
}

=head2 get_is_feature_demography_bid_modifier_unknown_age_allowed($client_id)

    Возвращает признак того, что для client_id доступна возможность использовать в соцдем корректировках
    "неизвестный" возраст

=cut
sub get_is_feature_demography_bid_modifier_unknown_age_allowed {
    my ($client_id) =  @_;
    return _does_client_have_feature($client_id, 'demography_bid_modifier_unknown_age_allowed') ? 1 : 0;
}


=head2 is_cpm_video_disabled($client_id)

    Клиенту запрешено охватное видео (adgroup_type == 'cpm_video')

=cut

sub is_cpm_video_disabled {
    return _does_client_have_feature(shift, 'disable_cpm_video');
}

=head2 get_is_feature_autooverdraft_enabled(client_id => $client_id)

    Возвращает признак того, что клиенту доступны автоовердрафты.

=cut

sub get_is_feature_autooverdraft_enabled {
    return _does_client_have_feature(+{@_}->{client_id}, 'autooverdraft_enabled');
}

=head2 get_is_feature_url_monitoring_online_request_enabled(client_id => $client_id)

    Онлайн запрос в YT таблицу url-мониторинга разрешен

=cut

sub get_is_feature_url_monitoring_online_request_enabled {
    return _does_client_have_feature(+{@_}->{client_id}, 'url_monitoring_online_request_enabled');
}

=head2 get_is_feature_cpc_video_banner_enabled(client_id => $client_id)

    Возвращает признак того, что клиенту доступны видео-объявления

=cut

sub get_is_feature_cpc_video_banner_enabled {
    return 1;
}


=head2 has_ab_segments_allowed_feature($client_id)

    Возвращает признак того, что для client_id доступна работа с экспериментальными сегментами (ab_segments)

=cut
sub has_ab_segments_allowed_feature($) {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "ab_segments") ? 1 : 0;
}

=head2 has_bid_modifiers_in_java_allowed_feature($client_id)

    Возвращает признак того, что для client_id доступна работа с корректировками из Java

=cut
sub has_bid_modifiers_in_java_allowed_feature($) {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "bid_modifiers_in_java") ? 1 : 0;
}

=head2 has_cpm_audio_feature($client_id)

    Возвращает признак того, что для client_id доступна работа c audio рекламой

=cut
sub has_cpm_audio_feature($) {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "cpm_audio") ? 1 : 0;
}

=head2 has_billing_order_domains_offline_report_enabled_feature($client_id)

    Возвращает признак того, что для client_id доступна работа c офлайн-отчётами по доменам

=cut
sub has_billing_order_domains_offline_report_enabled_feature($) {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "billing_order_domains_offline_report_enabled") ? 1 : 0;
}

=head2 has_agency_kpi_offline_report_enabled_feature($client_id)

    Возвращает признак того, что для client_id доступна работа c офлайн-отчётами по KPI

=cut
sub has_agency_kpi_offline_report_enabled_feature($) {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "agency_kpi_offline_report_enabled") ? 1 : 0;
}

=head2 has_increase_ad_text_limits_allowed_feature(client_id => $client_id)

    Возвращает признак того, что клиенту доступно увеличение текущих лимитов текстов в объявлениях

=cut

sub has_increase_ad_text_limits_allowed_feature {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "increase_ad_text_limits") ? 1 : 0;
}


=head2 is_webvisor_enabled_for_dna($)

    Возвращает признак того, включен ли вебвизор на стороне dna

=cut
sub is_webvisor_enabled_for_dna($) {
    my $client_id =  shift;
    return has_access_to_new_feature_from_java($client_id, "webvisor_enabled_for_dna") ? 1 : 0;
}

=head2 is_enable_long_term_caching($client_id)

    Возвращает признак того, что для client_id доступно долгосрочное кэширование

=cut
sub is_enable_long_term_caching($) {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "enable_long_term_caching") ? 1 : 0;
}

=head2 is_font_preloading_enabled($client_id)

    Включение предзагрузки основных шрифтов для DIRECT-154251
    https://st.yandex-team.ru/DIRECT-154251

=cut
sub is_font_preloading_enabled($) {
    my $client_id = shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "enable_font_preloading") ? 1 : 0;
}

=head2 is_velocity_fast_show_dna_enabled($client_id)

    Включить быструю версию контроллера showDna

=cut
sub is_velocity_fast_show_dna_enabled {
    my ($client_id) = @_;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "velocity_fast_show_dna") ? 1 : 0;
}

=head2 is_svg_sprite_loader_plugin_disabled($client_id)
    Отключает плагин svg-sprite-loader для DIRECT-153887 и DIRECT-154515
    https://st.yandex-team.ru/DIRECT-153887
    https://st.yandex-team.ru/DIRECT-154515
=cut
sub is_svg_sprite_loader_plugin_disabled {
    my ($client_id) = @_;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "disable_svg_sprite_loader_plugin") ? 1 : 0;
}

=head2 is_enable_sidebar_optimize($)

    Возвращает признак того, включена ли оптимизация сайдбара

=cut
sub is_enable_sidebar_optimize($) {
    my $client_id =  shift;
    return has_access_to_new_feature_from_java($client_id, "enable_sidebar_optimize") ? 1 : 0;
}

=head2 has_minus_words_lib_feature($client_id)

    Возвращает признак того, что для client_id доступна работа с библиотекой минус-фраз

=cut
sub has_minus_words_lib_feature($) {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "minus_words_lib") ? 1 : 0;
}

=head2 has_cpm_outdoor_forecast_feature($client_id)

    Возвращает признак того, что для client_id включен прогноз outdoor

=cut
sub has_cpm_outdoor_forecast_feature($) {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "cpm_outdoor_forecast") ? 1 : 0;
}

=head2 has_grid_feature($client_id)

    Возвращает признак того, что у клиента включено отображение списка кампаний в виде Grid

=cut
sub has_grid_feature($) {
    my $client_id =  shift;
    return has_access_to_new_feature_from_java($client_id, "grid") ? 1 : 0;
}

=head2 has_grid_allowed_feature($client_id)

    Возвращает признак того, что у клиента может использовать грид

=cut
sub has_grid_allowed_feature($) {
    my $client_id =  shift;
    return has_access_to_new_feature_from_java($client_id, "grid_allowed") ? 1 : 0;
}

=head2 has_cpm_deals_allowed_feature($client_id)

    Возвращает признак того, что для client_id доступна работа с частными сделками

=cut
sub has_cpm_deals_allowed_feature($) {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "cpm_deals") ? 1 : 0;
}


=head2 has_content_promotion_video_allowed_feature($client_id)

    Возвращает признак того, что для client_id доступна работа с кампаниями типа "Продвижение контента"

=cut
sub has_content_promotion_video_allowed_feature($) {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "content_promotion_video") ? 1 : 0;
}

=head2 has_support_chat_allowed_feature($client_id)

Возвращает признак того, что для client_id принудительно включен чат поддержки

=cut

sub has_support_chat_allowed_feature($) {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "support_chat") ? 1 : 0;
}

=head2 has_checked_support_chat_allowed_feature($client_id)

Возвращает признак того, что для client_id доступен чат поддержки (но для показа есть дополнитльные проверки)

=cut

sub has_checked_support_chat_allowed_feature($) {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "checked_support_chat") ? 1 : 0;
}

=head2 has_mobile_app_goals_for_text_campaign_allowed($client_id)

    Возвращает признак того, что для client_id доступны цели РМП в ТГО кампаниях (старая невключенная фича)

=cut
sub has_mobile_app_goals_for_text_campaign_allowed($) {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "mobile_app_goals_for_text_campaign_allowed") ? 1 : 0;
}

=head2 has_in_app_mobile_targeting_allowed($client_id)

    Возвращает признак того, что для client_id доступны цели событий мобильных приложений

=cut
sub has_in_app_mobile_targeting_allowed($) {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "in_app_mobile_targeting") ? 1 : 0;
}

=head2 has_mobile_app_goals_for_text_campaign_statistics_enabled($client_id)

    Возвращает признак того, что для client_id доступны цели РМП в статистике ТГО кампаний

=cut
sub has_mobile_app_goals_for_text_campaign_statistics_enabled($) {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "mobile_app_goals_for_text_campaign_statistics_enabled") ? 1 : 0;
}

=head2 has_mobile_app_goals_for_text_campaign_strategy_enabled($client_id)

    Возвращает признак того, что для client_id доступны цели РМП в cтратегиях ТГО кампаний

=cut
sub has_mobile_app_goals_for_text_campaign_strategy_enabled($) {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "mobile_app_goals_for_text_campaign_strategy_enabled") ? 1 : 0;
}

=head2 has_new_payment_workflow_enabled_feature($client_id)

    Возвращает признак того, что для client_id доступен новый попап оплаты

=cut
sub has_new_payment_workflow_enabled_feature($) {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "new_payment_workflow_enabled") ? 1 : 0;
}

=head2 relevance_match_for_new_groups_enabled_feature($client_id)

    Возвращает признак того, что для client_id при создании ТГО групп и кампаний автотаргетинг включен по умолчанию

=cut
sub relevance_match_for_new_groups_enabled_feature($) {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "relevance_match_for_new_groups_enabled") ? 1 : 0;
}

=head2 has_cpm_yndx_frontpage_allowed_feature($client_id)

    Возвращает признак того, что для client_id доступна работа с охватным продуктом на главной

=cut
sub has_cpm_yndx_frontpage_allowed_feature($) {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "cpm_yndx_frontpage") ? 1 : 0;
}

=head2 has_cpm_price_allowed_feature($client_id)
    Возвращает признак того, что для client_id доступна работа с прайсовым охватным продуктом на главной
    Завязываемся на фичу нового интерфейсе, т.к. в старом кампания не будет поддерживаться
=cut
sub has_cpm_price_allowed_feature($) {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "show_cpm_price_campaigns_in_grid") ? 1 : 0;
}

=head2 has_indoor_segments_feature($client_id)

    Возвращает признак того, что для client_id доступна работа indoor сегментами

=cut
sub has_indoor_segments_feature($) {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "indoor_segments") ? 1 : 0;
}

=head2 has_cpm_yndx_frontpage_profile_allowed_feature($client_id)

    Возвращает признак того, что для client_id доступен таргетинг на профиль(Медийный продукт на Главной

=cut
sub has_cpm_yndx_frontpage_profile_allowed_feature($) {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "cpm_yndx_frontpage_profile") ? 1 : 0;
}

=head2 has_new_campaign_strategy_enabled_feature($client_id)

    Возвращает признак того, что для client_id доступен новый компонент редактирования стратегий

=cut
sub has_new_campaign_strategy_enabled_feature($) {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "new_campaign_info_strategy_enabled") ? 1 : 0;
}

=head2 has_save_text_group_in_java_allowed_feature($client_id)

    Возвращает признак того, что для client_id доступно сохранение ТГО групп в Java

=cut
sub has_save_text_group_in_java_allowed_feature($) {
    my ($client_id) = @_;
    return has_access_to_new_feature_from_java($client_id, "save_text_group_in_java") ? 1 : 0;
}

=head2 has_new_campaign_info_enabled_feature($client_id)

    Возвращает признак доступности новой шапки на странице кампании

=cut

sub has_new_campaign_info_enabled_feature {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "new_campaign_info_enabled") ? 1 : 0;
}

=head2 has_new_campaign_page_enabled_feature($client_id)

    Возвращает признак доступности новой страницы кампании

=cut

sub has_new_campaign_page_enabled_feature {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "new_campaign_page_enabled") ? 1 : 0;
}

=head2 has_turbo_page_types_enabled_in_mol_feature($client_id)

    Возвращает признак того, что для client_id доступен срез турбо-площадка

=cut
sub has_turbo_page_types_enabled_in_mol_feature($) {
    my ($client_id) = @_;
    return has_access_to_new_feature_from_java($client_id, "turbo_page_types_enabled_in_mol") ? 1 : 0;
}

=head2 has_new_master_token_authentication_feature($client_id)

    Возвращает признак того, что все представители client_id авторизуются через смс для работы с мастер-токеном

=cut
sub has_new_master_token_authentication_feature($) {
    my ($client_id) = @_;
    return has_access_to_new_feature_from_java($client_id, "new_master_token_authentication") ? 1 : 0;
}

=head2 has_grid_goals_filtration_for_stat_feature($client_id)

    Возаращает состояние фичи отправки в ручку мастера отчетов списка целей, для получения значений по всем целям

=cut
sub has_grid_goals_filtration_for_stat_feature($) {
    my ($client_id) = @_;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "grid_goals_filtration_for_stat") ? 1 : 0;
}

=head2 has_disable_autobudget_week_bundle_in_api_feature($client_id)

    Возаращает состояние фичи отключения стратегии пакета кликов в api

=cut
sub has_disable_autobudget_week_bundle_in_api_feature($) {
    my ($client_id) = @_;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "disable_autobudget_week_bundle_in_api") ? 1 : 0;
}

=head2 get_measurers_features($client_id)

    Возвращает группу фичей доступности измерителей баннеров

=cut

sub get_measurers_features($) {

    my ($client_id) = @_;

    my %measurers = (map {
        $_ . "_measurer" => 0
    } qw/admetrica adloox adriver dv integral_ad_science mediascope moat sizmek weborama omi/);

    for my $feature (@{allowed_features($client_id)}) {
        $measurers{$feature} = 1 if exists $measurers{$feature};
    }
    return \%measurers;
}

=head2 has_avg_bid_in_mol_feature($client_id)

    Возвращает признак того, что для client_id доступно поле "Средняя ставка" в МОЛ

=cut
sub has_avg_bid_in_mol_feature($) {
    my ($client_id) =  @_;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "avg_bid_in_mol") ? 1 : 0;
}

=head2 has_virtual_campaigns_feature($client_id)

    Возвращает признак того, что для client_id доступна возможность создавать виртуальные кампании

=cut
sub has_virtual_campaigns_feature($) {
    my ($client_id) =  @_;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, 'virtual_campaigns_allowed') ? 1 : 0;
}

=head2 has_alone_trafaret_option_feature($client_id)

    Возвращает признак того, что для client_id доступна возможность включать опцию показа
    объявлений на Поиске в трафаретах с вытеснением других участников

=cut
sub has_alone_trafaret_option_feature($) {
    my ($client_id) =  @_;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, 'alone_trafaret_option_enabled') ? 1 : 0;
}

=head2 has_can_require_filtration_by_dont_show_domains_feature($client_id)

    Возвращает признак того, что для client_id доступна возможность включать фильтрацию по списку запрещенных доменов
    на внутренних площадках Яндекса

=cut
sub has_can_require_filtration_by_dont_show_domains_feature($) {
    my ($client_id) =  @_;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, 'can_require_filtration_by_dont_show_domains') ? 1 : 0;
}

=head2 has_can_require_filtration_by_dont_show_domains_in_cpm_feature($client_id)

    Возвращает признак того, что для client_id доступна возможность включать фильтрацию по списку запрещенных доменов
    на внутренних площадках Яндекса для CPM кампаний

=cut
sub has_can_require_filtration_by_dont_show_domains_in_cpm_feature($) {
    my ($client_id) =  @_;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, 'can_require_filtration_by_dont_show_domains_in_cpm') ? 1 : 0;
}

=head2 has_enable_dk2_in_api_feature($client_id)

    Возвращает признак того, что для client_id доступна работа с ДК2

=cut
sub has_enable_dk2_in_api_feature($) {
    my $client_id = shift;
    return 0 unless $client_id;

    return has_access_to_new_feature_from_java($client_id, "enable_dk2_in_api") ? 1 : 0;
}

=head2 has_new_lim_rep_schema_feature($client_id)

    Возвращает признак того, что для агентства с заданным client_id доступна расширенная схема ограниченных представителей

=cut
sub has_new_lim_rep_schema_feature($) {
    my $client_id = shift;
    return 0 unless $client_id;

    return has_access_to_new_feature_from_java($client_id, "new_lim_rep_schema") ? 1 : 0;
}

=head2 has_new_lim_rep_schema_share_client_feature($client_id)

    Возвращает признак того, что для агентства с заданным client_id доступна привязка клиента в разных группах в расширенной схеме ограниченных представителей

=cut
sub has_new_lim_rep_schema_share_client_feature($) {
    my $client_id = shift;
    return 0 unless $client_id;

    return has_access_to_new_feature_from_java($client_id, "new_lim_rep_schema_share_client") ? 1 : 0;
}

=head2 has_mobile_content_cpc_video_allowed($client_id)

    Возвращает признак того, что для client_id доступно создание видеобаннеров в РМП кампаниях

=cut
sub has_mobile_content_cpc_video_allowed($) {
    my $client_id = shift;
    return 0 unless $client_id;

    return has_access_to_new_feature_from_java($client_id, "mobile_content_cpc_video_allowed") ? 1 : 0;
}

=head2 can_upload_banner_experiments(client_id => $client_id, rights => $rbac_login_rights)


    Возвращает признак того, что для client_id доступна загрузка через внутренний отчет xls, xlsx файлов для экспериментов на СЕРПе и РСЯ

=cut
sub can_upload_banner_experiments {
    my (%opt) = @_;
    my ($client_id, $rbac_login_rights) = @opt{qw/client_id rights/};

    croak 'cliend_id required' unless ($client_id);

    return 1 if $rbac_login_rights && $rbac_login_rights->{super_control};

    return has_access_to_new_feature_from_java($client_id, "upload_banner_experiments") ? 1 : 0;
}

=head2 has_context_relevance_match_feature($client_id)

    Возвращает признак того, что для клиента доступно управление бесфразным таргетингом на сети

=cut
sub has_context_relevance_match_feature($) {
    my ($client_id) = @_;
    if ($client_id) {
        return (
                has_access_to_new_feature_from_java($client_id, "context_relevance_match_allowed")
                || has_access_to_new_feature_from_java($client_id, "context_relevance_match_interface_only")
        ) ? 1 : 0;
    } else {
        return 0;
    }
}

=head2 has_video_report_feature($client_id)

Разрешен ли клиенту доступ к отчетам по видео статистике

=cut

sub has_video_report_feature
{
    my ($client_id) = @_;
    if ($client_id) {
        return has_access_to_new_feature_from_java($client_id, 'video_report') ? 1 : 0;
    }
    return 0;
}

=head2 has_strategy_save_in_java_feature($client_id)

    Сохранение ТГО стратегий на Java

=cut

sub has_strategy_save_in_java_feature
{
    my ($client_id) = @_;
    if ($client_id) {
        return has_access_to_new_feature_from_java($client_id, "strategy_save_in_java");
    }

    return 0;
}

=head2 has_strategy_save_in_java_prevalidate_feature($client_id)

    Превалидация сохранения ТГО стратегий на Java

=cut

sub has_strategy_save_in_java_prevalidate_feature
{
    my ($client_id) = @_;
    if ($client_id) {
        return has_access_to_new_feature_from_java($client_id, "strategy_save_in_java_prevalidate");
    }

    return 0;
}

=head2 has_set_campaign_allowed_page_ids_feature($client_id)

    Возвращает признак того, что для client_id доступна запись списка page_ids на которых допустипо показывать объявления кампании
    Проверять на опертора

=cut

sub has_set_campaign_allowed_page_ids_feature($) {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "set_campaign_allowed_page_ids") ? 1 : 0;
}

=head2 has_new_wallet_with_aggregated_sums

    Разрешено ли новым клиентам создавать ОС в новой схеме зачислений.

=cut

sub has_new_wallet_with_aggregated_sums {
    my ($client_id) = @_;
    if ($client_id) {
        return has_access_to_new_feature_from_java($client_id, "new_wallet_with_aggregated_sums") ? 1 : 0;
    }
    return 0;
}

=head2 show_cashback_bonus

    Разрешено ли показывать клиенту доступную ему сумму кэшбека.

=cut

sub show_cashback_bonus {
    my ($client_id) = @_;
    if ($client_id) {
        return has_access_to_new_feature_from_java($client_id, "show_cashback_bonus") ? 1 : 0;
    }
    return 0;
}

=head2 has_aggregated_sums_for_old_clients

    Разрешено ли старым клиентам автоматически переходить на новую схему зачислений.
    Это может случиться при включении ОС

=cut

sub has_aggregated_sums_for_old_clients {
    my ($client_id) = @_;
    if ($client_id) {
        return has_access_to_new_feature_from_java($client_id, "aggregated_sums_for_old_clients") ? 1 : 0;
    }
    return 0;
}

=head2 has_dialog_allowed($client_id)

    Возвращает признак того, что для client_id доступно добавление чата (dialog_id) в параметрах кампании

=cut

sub has_dialog_allowed($) {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "dialogs_allowed") ? 1 : 0;
}

=head2 has_cpc_device_modifiers_allowed_feature($client_id)

    Возвращает признак того, что для client_id доступна работа с корректировками на устройства в cpc-кампаниях и группах

=cut
sub has_cpc_device_modifiers_allowed_feature {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "cpc_device_modifiers") ? 1 : 0;
}

=head2 has_cpm_video_device_modifiers_allowed_feature($client_id)

    Возвращает признак того, что для client_id доступна работа с корректировками на устройства в кампаниях с cpm-видео

=cut
sub has_cpm_video_device_modifiers_allowed_feature {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "cpm_video_device_modifiers") ? 1 : 0;
}

=head2 is_mobile_os_bid_modifier_enabled($client_id)

    Возвращает признак того, что для client_id доступна работа с корректировками на устройства в ТГО, СМАРТ,
    продвижение контента безвозможности отключения мобильных устройств полностью

=cut
sub is_mobile_os_bid_modifier_enabled {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "mobile_os_bid_modifier_enabled") ? 1 : 0;
}

=head2 has_payment_before_moderation_feature($client_id)

    Возвращает признак того, что для client_id доступна оплата до модерации первой кампании
    Проверять на опертора

=cut
sub has_payment_before_moderation_feature($) {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, 'payment_before_moderation') ? 1 : 0;
}

=head2 has_bid_correction_search_enabled($client_id)

    Возвращает признак того, что для client_id меняется название стратегии на
    Ручное управление ставками с оптимизацией

=cut
sub has_bid_correction_search_enabled($) {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "bid_correction_search_enabled") ? 1 : 0;
}

=head2 has_banner_update_allowed_for_dna($client_id)

    Возвращает признак того, что для client_id доступно редактирование баннеров

=cut
sub has_banner_update_allowed_for_dna($) {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "banner_update_in_java_for_dna") ? 1 : 0;
}

=head2 has_grid_enabled_for_dna($cliend_id)

    Возвращает признак того что для cliend_id доступены гриды

=cut
sub has_grid_enabled_for_dna($) {
    my ($client_id) = @_;
    return 0 unless $client_id;
    return _does_client_have_feature($client_id, "grid");
}

=head2 has_soft_hide_show_camps($cliend_id)

    Мягкое переключение клиента на новый интерфейс (с возможностью вернуться в старый)

=cut
sub has_soft_hide_show_camps {
    my ($client_id) = @_;

    # Фича 'grid' в новом интерфейсе используется как пользовательская настройка.
    # Для требуемой по ТЗ логики работы фичи show_dna_by_default нам нужно различать состояния "grid не задана для конкретного клиента"
    # и "grid выключена для конкретного клиента" (DIRECT-118848). Поэтому в обход основной логики проверяем заданность фичи "grid" для текущего клиента.
    my $grid_feature_id = get_one_field_sql(PPCDICT, 'select feature_id from features where feature_text_id = "grid"');
    my $grid_feature_client_value = get_one_field_sql(PPC(ClientID => $client_id),
        'select is_enabled from clients_features where ClientID=? AND feature_id=?', $client_id, $grid_feature_id);
    #Если grid для клиента не задана - условно редиректим на новый интерфейс
    $grid_feature_client_value //= 1;

    #Чтобы отработать кейс "grid включена на 100%" как альтернативу
    return has_feature($client_id, 'show_dna_by_default')
        && ($grid_feature_client_value || has_grid_enabled_for_dna($client_id));
}

=head2 has_first_expanded_sidebar_for_dna($cliend_id)

    Возвращает признак того что для cliend_id доступен открытый сайдбар

=cut
sub has_first_expanded_sidebar_for_dna($) {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "first_expanded_sidebar_for_dna") ? 1 : 0;
}

=head2 need_create_billing_aggregates($client_id)

    Нужно ли создавать клиенту биллинговые агрегаты при создании кампании.

    Проверяет отсутствие фичи disable_billing_aggregates у клиента и агентства, если оно есть.

=cut
sub need_create_billing_aggregates {
    my $client_id =  shift;
    return 0 unless $client_id;

    my $info = Rbac::get_perminfo(ClientID => $client_id);
    if ($info->{agency_client_id}) {
        if(has_access_to_new_feature_from_java($info->{agency_client_id}, "disable_billing_aggregates")) {
            return 0;
        }
    }
    if (has_access_to_new_feature_from_java($client_id, "disable_billing_aggregates")) {
        return 0;
    }

    return 1;
}

=head2 has_auction_stats_for_all_campaigns_feature($client_id)

    Доступны ли клиенту новые метрики по аукционам в статистике

=cut
sub has_auction_stats_for_all_campaigns_feature {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "auction_stats_for_all_campaigns") ? 1 : 0;
}

=head2 has_cpm_banner_sov_in_stat_feature($client_id)

    Доступны ли клиенту новые метрики в статистике для медийной кампании

=cut
sub has_cpm_banner_sov_in_stat_feature {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "cpm_banner_sov_in_stat") ? 1 : 0;
}

=head2 has_cpm_yndx_frontpage_sov_in_stat_feature($operator_id)

    Доступны ли клиенту новые метрики в статистике для медийной кампании на главной

=cut
sub has_cpm_yndx_frontpage_sov_in_stat_feature {
    my $operator_id =  shift;
    return 0 unless $operator_id;
    return has_access_to_new_feature_from_java($operator_id, "cpm_yndx_frontpage_sov_in_stat") ? 1 : 0;
}

=head2 has_cpm_price_sov_in_stat_feature($operator_id)

    Доступны ли клиенту новые метрики в статистике для прайсовой кампании

=cut
sub has_cpm_price_sov_in_stat_feature {
    my $operator_id =  shift;
    return 0 unless $operator_id;
    return has_access_to_new_feature_from_java($operator_id, "cpm_price_sov_in_stat") ? 1 : 0;
}

=head2 has_cpm_yndx_frontpage_frequency_cap_feature($client_id)

    Доступно ли клиенту ограничение частоты показов медийных кампаний на главной

=cut
sub has_cpm_yndx_frontpage_frequency_cap_feature {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "cpm_yndx_frontpage_frequency_cap") ? 1 : 0;
}

=head2 has_in_app_events_in_rmp_enabled_feature($client_id)

    Включена фича "in app события в РМП" в попапе стратегии

=cut
sub has_in_app_events_in_rmp_enabled_feature {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "in_app_events_in_rmp_enabled") ? 1 : 0;
}

=head2 has_cpm_video_several_pixels_enabled_feature($client_id)

    Возвращает признак того, что клиенту разрешено иметь несколько пикселей аудита на баннере CPM_VIDEO группы

=cut
sub has_cpm_video_several_pixels_enabled_feature {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "cpm_video_several_pixels_enabled") ? 1 : 0;
}

=head2 has_turbo_smarts($client_id)

    Доступна ли клиенту возможность турбировать смарт-кампании

=cut
sub has_turbo_smarts {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "turbo_smarts") ? 1 : 0;
}

=head2 has_s2s_tracking_feature($client_id)

    Доступна ли клиенту возможность включать межсерверное взаимодействие

=cut
sub has_s2s_tracking_feature {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "is_s2s_tracking_enabled") ? 1 : 0;
}

=head2 has_b2b_balance_cart_feature($client_id)

    Доступна ли клиенту функциональность единой корзины в балансе (DIRECT-97990)

=cut
sub has_b2b_balance_cart_feature {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "b2b_balance_cart") ? 1 : 0;
}

=head2 is_brandsafety_enabled($client_id, $camp_type)

    Доступна ли клиенту настройка brand safety

=cut

sub is_brandsafety_enabled {
    my ($client_id, $camp_type) = @_;
    return $camp_type eq 'cpm_banner'
        ? has_access_to_new_feature_from_java($client_id, 'brandsafety_base_categories_cpm')
        : has_access_to_new_feature_from_java($client_id, 'brandsafety_base_categories_cpc');
}

=head2 is_additional_brandsafety_enabled($client_id)

    Доступна ли клиенту дополнительные категории brand safety (для любого типа кампании)

=cut

sub is_additional_brandsafety_enabled {
    return has_access_to_new_feature_from_java(shift, 'brandsafety_additional_categories');
}

=head2 has_show_daily_budget_recommendations_in_old_interface_feature($client_id)

    Доступна ли клиенту фича "Отображение рекомендаций по дневному бюджету в старом интерфейсе"

=cut
sub has_show_daily_budget_recommendations_in_old_interface_feature {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "show_daily_budget_recommendations_in_old_interface") ? 1 : 0;
}

=head2 has_show_aggregated_status_open_beta_feature($client_id)

    Доступна ли клиенту фича "Отображение ess статуса вместо обычного"

=cut
sub has_show_aggregated_status_open_beta_feature {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "show_aggregated_status_open_beta") ? 1 : 0;
}

=head2 has_tycoon_organizations_feature($client_id)

    Доступна ли клиенту фича "Организации Яндекс.Справочника"

=cut
sub has_tycoon_organizations_feature {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "tycoon_organizations") ? 1 : 0;
}

=head2 has_default_avg_cpa_feature($client_id)

    Доступна ли клиенту фича "По умолчанию предлагать стратегию autobudget_avg_cpa (оптимизация конверсий)"

=cut
sub has_default_avg_cpa_feature {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "default_autobudget_avg_cpa") ? 1 : 0;
}

=head2 has_default_autobudget_avg_click_with_week_budget_feature($client_id)

    Доступна ли клиенту фича "По умолчанию предлагать стратегию autobudget_avg_click (оптимизация кликов)"

=cut
sub has_default_autobudget_avg_click_with_week_budget_feature {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "default_autobudget_avg_click_with_week_budget") ? 1 : 0;
}

=head2 has_default_autobudget_roi_feature($client_id)

    Доступна ли клиенту фича "По умолчанию предлагать стратегию autobudget_roi (оптимизация рентабельности)"

=cut
sub has_default_autobudget_roi_feature {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "default_autobudget_roi") ? 1 : 0;
}

=head2 has_crr_strategy_feature($client_id)

    Доступна ли клиенту фича "Доступность ДРР стратегии"

=cut
sub has_crr_strategy_feature {
    my $client_id = shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "crr_strategy_allowed") ? 1 : 0;
}

=head2 has_mass_edit_regions_for_dna_enabled_feature($client_id)

    Доступна ли клиенту фича "Редактировать гео массово для групп в dna"

=cut
sub has_mass_edit_regions_for_dna_enabled_feature {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "mass_edit_regions_in_java_for_dna") ? 1 : 0;
}


=head2 has_new_autopay_type

    Вычисляем доступность использования новых методов в автоплатеже

=cut
sub has_new_autopay_type {
    my $client_id = shift;
    return has_access_to_new_feature_from_java($client_id, "autopay_with_pay_request") ? 1 : 0;
}

=head2 has_goals_only_with_campaign_counters_used($client_id)

    Оставляет только цели, к которым есть доступ у пользовательских счетчиков

=cut
sub has_goals_only_with_campaign_counters_used {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "goals_only_with_campaign_counters_used") ? 1 : 0;
}

=head2 has_edit_avg_cpm_without_restart_feature($client_id)

    Доступна ли клиенту фича "Редактироване cредней цены без рестарта cpm-стратегии"

=cut
sub has_edit_avg_cpm_without_restart_feature {
    my $client_id = shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "edit_avg_cpm_without_restart") ? 1 : 0;
}

=head2 has_disable_any_domains_allowed_feature($client_id)

    Доступна ли клиенту фича, которая включает возможность запрещать любые домены

=cut
sub has_disable_any_domains_allowed_feature {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "disable_any_domains_allowed") ? 1 : 0;
}

=head2 has_disable_mail_ru_domain_allowed_feature($client_id)

    Доступна ли клиенту фича, которая включает возможность запрещать домены mail.ru

=cut
sub has_disable_mail_ru_domain_allowed_feature {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "disable_mail_ru_domain_allowed") ? 1 : 0;
}

=head2 has_disable_number_id_and_short_bundle_id_allowed_feature($client_id)

    Доступна ли клиенту фича, которая включает возможность запрещать числовые (233411) и короткие bundle (yandexbrowser)

=cut
sub has_disable_number_id_and_short_bundle_id_allowed_feature {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "disable_number_id_and_short_bundle_id_allowed") ? 1 : 0;
}

=head2 has_can_have_internal_dont_show_domains_feature($client_id)

    Доступна ли клиенту фича, которая включает возможность запрещать внутренние площадки

=cut
sub has_can_have_internal_dont_show_domains_feature {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "can_have_internal_dont_show_domains") ? 1 : 0;
}

=head2 has_campaign_new_min_days_limit($client_id)

    Доступна ли клиенту фича, которая включает новое правило валидации временного таргетинга

=cut
sub has_campaign_new_min_days_limit {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "campaign_new_min_days_limit") ? 1 : 0;
}

=head2 has_service_worker_allowed_for_dna($client_id)

    Возвращает признак того, что для client_id доступен service-worker

=cut
sub has_service_worker_allowed_for_dna($) {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "service_worker_allowed") ? 1 : 0;
}

=head2 has_smart_at_search_enabled_feature($client_id)

    Возвращает признак того, что для client_id доступны смарт-баннеры на поиске

=cut
sub has_smart_at_search_enabled_feature($) {
    my ($client_id) = @_;
    return has_access_to_new_feature_from_java($client_id, 'smart_at_search') ? 1 : 0;
}

=head2 has_banner_prices_enabled($client_id)

    Возвращает признак того, что для client_id доступно создание цен на товары на баннерах

=cut
sub has_banner_prices_enabled($) {
    my ($client_id) = @_;
    return has_access_to_new_feature_from_java($client_id, 'banner_prices') ? 1 : 0;
}

=head2 has_cpm_geoproduct_enabled($client_id)

    Возвращает признак того, что для client_id доступно создание баннеров для геопродукта

=cut
sub has_cpm_geoproduct_enabled($) {
    my ($client_id) = @_;
    return has_access_to_new_feature_from_java($client_id, 'cpm_geoproduct_enabled') ? 1 : 0;
}

=head2 has_nds20_alarm_feature($client_id)

    Возвращает признак того, что клиенту нужно выдавать тизер о переходе на 20%-й НДС

=cut
sub has_nds20_alarm_feature($) {
    my ($client_id) = @_;
    return has_access_to_new_feature_from_java($client_id, 'nds20_alarm') ? 1 : 0;
}

=head2 is_feature_content_promotion_collection_enabled($client_id)

    Возвращает признак того, что клиенту разрешено создание групп типа content_promotion_collection

=cut
sub is_feature_content_promotion_collection_enabled($) {
    my ($client_id) = @_;
    return has_access_to_new_feature_from_java($client_id, 'content_promotion_collection') ? 1 : 0;
}

=head2 has_content_promotion_services_allowed_feature

    Возвращает признак того, что разрешено редактирование продвижения Услуг в старом интерфейсе

=cut
sub has_content_promotion_services_allowed_feature($) {
    my ($client_id) = @_;
    return has_access_to_new_feature_from_java($client_id, 'content_promotion_services') ? 1 : 0;
}

=head2 has_eda_content_promotion_interface_feature

    Возвращает признак того, что клиенту включена фича на просмотр продвижения Еды в интерфейсе

=cut
sub has_eda_content_promotion_interface_feature($) {
    my ($client_id) = @_;
    return has_access_to_new_feature_from_java($client_id, 'content_promotion_eda_interface') ? 1 : 0;
}

=head2 has_content_promotion_collection_grid_allowed_feature

    Возвращает признак того, что разрешено создание продвижения коллекций в новом интерфейсе

=cut
sub has_content_promotion_collection_grid_allowed_feature($) {
    my ($client_id) = @_;
    return has_access_to_new_feature_from_java($client_id, 'content_promotion_collection_grid') ? 1 : 0;
}

=head2 has_content_promotion_video_grid_allowed_feature

    Возвращает признак того, что разрешено создание продвижения видео в новом интерфейсе

=cut
sub has_content_promotion_video_grid_allowed_feature($) {
    my ($client_id) = @_;
    return has_access_to_new_feature_from_java($client_id, 'content_promotion_video_grid') ? 1 : 0;
}

=head2 has_any_content_promotion_feature_enabled($client_id)

    Возвращает признак того, что хотя бы одна из фичей продвижения контента включена клиенту

=cut
sub has_any_content_promotion_feature_enabled($) {
    my ($client_id) = @_;
    return is_feature_content_promotion_collection_enabled($client_id) ||
        has_content_promotion_video_allowed_feature($client_id) ||
        has_content_promotion_collection_grid_allowed_feature($client_id) ||
        has_content_promotion_video_grid_allowed_feature($client_id);
}

=head2 has_uniq_completed_viewers_stat_enabled_feature($client_id)

    Возвращает признак того, что для client_id доступна колонка "Достроенный охват" в статистике

=cut
sub has_uniq_completed_viewers_stat_enabled_feature($) {
    my ($client_id) = @_;
    return has_access_to_new_feature_from_java($client_id, 'uniq_completed_viewers_stat') ? 1 : 0;
}

=head2 has_copy_conversion_strategy_between_logins_feature($client_id)

    Можно ли клиенту копировать кампании между логинами без сброса конверсионной стратегии.

=cut
sub has_copy_conversion_strategy_between_logins_feature($) {
    my ($client_id) = @_;
    return has_access_to_new_feature_from_java($client_id, 'allow_copy_conversion_strategy_between_logins') ? 1 : 0;
}

=head2 has_income_grade_bid_modifier_feature($client_id)

    Можно ли клиенту использовать корректировки по доходу.

=cut
sub has_income_grade_bid_modifier_feature($) {
    my ($client_id) = @_;
    return has_access_to_new_feature_from_java($client_id, 'income_grade_bid_modifier_allowed') ? 1 : 0;
}

=head2 has_tns_counter_feature($client_id)

    Возвращает признак того, клиенту доступны корректировки на пробки в наружной рекламе

=cut
sub has_tns_counter_feature {
    my ($client_id) = @_;
    return _does_client_have_feature($client_id, 'tns_enabled');
}

=head2 is_creative_frontpage_728_90_disabled($client_id)

    Креатив 728х90 для клиента становится не обязательным?

=cut

sub is_creative_frontpage_728_90_disabled {
    my ($client_id) = @_;
    return _does_client_have_feature($client_id, 'creative_frontpage_728_90_disabled');
}

=head2 has_brandsafety_base_categories_stat($client_id)

    Вкладка статистики по brand safety

=cut

sub has_brandsafety_base_categories_stat {
    my ($client_id) = @_;
    return _does_client_have_feature($client_id, 'brandsafety_base_categories_stat');
}


=head2 has_touch_direct_feature($client_id)

    Разрешаем ли клиенту получать на мобильных браузерах мобильный DNA интерфейс

=cut
sub has_touch_direct_feature {
    my ($client_id) = @_;

    return _does_client_have_feature($client_id, 'touch_direct_enabled');
}

=head2 has_hide_old_show_camps_for_dna_feature($client_id)

    Возвращает признак того, что для клиента скрыта страница showCamps старого интерфейса

=cut
sub has_hide_old_show_camps_for_dna_feature {
    my ($client_id) = @_;
    return 0 unless $client_id;
    return _does_client_have_feature($client_id, 'hide_old_show_camps_for_dna');
}

=head2 is_overview_enabled($client_id)

    Для клиента доступна страница "Обзор", и она используется в качестве домашней

=cut
sub is_overview_enabled {
    my ($client_id) = @_;
    return 0 unless $client_id;
    return _does_client_have_feature($client_id, 'overview_enabled');
}

=head2 homepage_use_grid($client_id)

    Домашней страницей показывать грид кампаний, а не "Обзор".
    Имеет смысл только в связке с фичей "overview_enabled"

=cut
sub homepage_use_grid {
    my ($client_id) = @_;
    return 0 unless $client_id;
    return _does_client_have_feature($client_id, 'homepage_use_grid');
}

=head2 collecting_verified_phones_feature($client_id)

    Включен ли сбор подтвержденных телефонов рекламодателей

=cut
sub collecting_verified_phones_feature {
    my ($client_id) = @_;

    return _does_client_have_feature($client_id, 'collecting_verified_phones_for_old_clients')
        || _does_client_have_feature($client_id, 'collecting_verified_phones_for_new_clients')
}

=head2 facelift_disabled_for_dna_feature($client_id)

    Включена ли обратная фича по фейслифту

=cut
sub facelift_disabled_for_dna_feature {
    my ($client_id) = @_;

    return _does_client_have_feature($client_id, 'facelift_disabled_for_dna')
}

=head2 enabled_change_offer_for_clients_from_turkey_dna_feature($client_id)

    Включена ли фича новой оферты для турецких клиентов

=cut
sub enabled_change_offer_for_clients_from_turkey_dna_feature {
    my ($client_id) = @_;

    return _does_client_have_feature($client_id, 'change_offer_for_clients_from_turkey')
}

=head2 social_advertising_payable_feature($client_id)

    Включена ли фича социальной рекламы

=cut
sub social_advertising_payable_feature {
    my ($client_id) = @_;

    return _does_client_have_feature($client_id, 'social_advertising_payable')
}

=head2 social_advertising_feature($client_id)

    Включена ли фича социальной рекламы

=cut
sub social_advertising_feature {
    my ($client_id) = @_;

    return _does_client_have_feature($client_id, 'social_advertising')
}

=head2 cpv_strategies_enabled_feature($client_id)

    Включена ли фича cpv стратегий

=cut
sub cpv_strategies_enabled_feature {
    my ($client_id) = @_;

    return _does_client_have_feature($client_id, 'cpv_strategies_enabled')
}


=head2 dynamic_groups_edit_allowed_for_dna_feature($client_id)

    Включает возможность редактировать динамические группы в DNA

=cut
sub dynamic_groups_edit_allowed_for_dna_feature {
    my ($client_id) = @_;

    return _does_client_have_feature($client_id, 'dynamic_groups_edit_allowed_for_dna')
}

=head2 dynamic_ads_edit_allowed_for_dna_feature($client_id)

    Включает возможность редактировать динамические объявления в DNA

=cut
sub dynamic_ads_edit_allowed_for_dna_feature {
    my ($client_id) = @_;

    return _does_client_have_feature($client_id, 'dynamic_ads_edit_allowed_for_dna')
}

=head2 dynamic_only_for_dna_feature($client_id)

    Закрытие динамических кампаний, объявлений и групп в старом интерфейсе

=cut
sub dynamic_only_for_dna_feature {
    my ($client_id) = @_;

    return _does_client_have_feature($client_id, 'dynamic_only_for_dna')
}

=head2 mobile_content_groups_edit_allowed_for_dna_feature($client_id)

    Включает возможность редактировать РМП группы в DNA

=cut
sub mobile_content_groups_edit_allowed_for_dna_feature {
    my ($client_id) = @_;

    return _does_client_have_feature($client_id, 'mobile_content_groups_edit_allowed_for_dna')
}

=head2 mobile_content_ads_edit_allowed_for_dna_feature($client_id)

    Включает возможность редактировать РМП объявления в DNA

=cut
sub mobile_content_ads_edit_allowed_for_dna_feature {
    my ($client_id) = @_;

    return _does_client_have_feature($client_id, 'mobile_content_ads_edit_allowed_for_dna')
}

=head2 mobile_content_only_for_dna_feature($client_id)

    Закрытие РМП кампаний, объявлений и групп в старом интерфейсе

=cut
sub mobile_content_only_for_dna_feature {
    my ($client_id) = @_;

    return _does_client_have_feature($client_id, 'mobile_content_only_for_dna')
}

=head2 loading_dna_scripts_before_old_interface_scripts_enabled_feature($client_id)

    Загружать скрипты большого директа после скриптов нового интерфейса

=cut
sub loading_dna_scripts_before_old_interface_scripts_enabled_feature {
    my ($client_id) = @_;
    return 0 unless $client_id;

    return has_access_to_new_feature_from_java($client_id, 'loading_dna_scripts_before_old_interface_scripts_enabled') ? 1 : 0;
}

=head2 cpm_yndx_frontpage_in_dna_feature($client_id)

    Закрытие Медийки на главной: кампаний, объявлений и групп в старом интерфейсе

=cut
sub cpm_yndx_frontpage_in_dna_feature {
    my ($client_id) = @_;

    return _does_client_have_feature($client_id, 'edit_cpm_yndx_frontpage_in_dna')
}

=head2 has_video_constructor_enabled_direct_feature($client_id)

    Разрешаем ли клиенту доступ в Видеоконструктор в DNA интерфейсe

=cut
sub has_video_constructor_enabled_direct_feature {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "video_constructor_enabled") ? 1 : 0;
}

=head2 has_video_constructor_create_from_scratch_enabled_direct_feature($client_id)

    Разрешаем ли клиенту доступ к созданию видео с нуля в Видеоконструкторе

=cut
sub has_video_constructor_create_from_scratch_enabled_direct_feature {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "video_constructor_create_from_scratch_enabled") ? 1 : 0;
}

=head2 has_video_constructor_feed_enabled_direct_feature($client_id)

    Разрешаем ли клиенту доступ к фидам в Видеоконструкторе

=cut
sub has_video_constructor_feed_enabled_direct_feature {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "video_constructor_feed_enabled") ? 1 : 0;
}

=head2 is_canvas_range_ratio_cpc($client_id)

    Разрешаем ли клиенту доступ к вертикальным шаблонам в cpc.

=cut
sub is_canvas_range_ratio_cpc {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "canvas_range_ratio_cpc") ? 1 : 0;
}

=head2 has_landings_wizard_allowed_feature($client_id)

    Возвращает признак того, что клиенту доступен визард создания тубо-страниц

=cut
sub has_landings_wizard_allowed_feature {
    my ($client_id) = @_;
    return _does_client_have_feature($client_id, 'landings_wizard_allowed');
}

=head2 has_touch_payment_page_feature($client_id)

    Возвращает признак того, что клиенту доступна страница оплаты для тачей

=cut
sub has_touch_payment_page_feature {
    my ($client_id) = @_;
    return has_access_to_new_feature_from_java($client_id, 'touch_payment_page') ? 1 : 0;
}

=head2 has_multi_clients_in_stat_allowed($client_id)

    Разрешаем ли клиенту доступ к Мультиклиентности в статистике

=cut
sub has_multi_clients_in_stat_allowed {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "multi_clients_in_stat_allowed") ? 1 : 0;
}

=head2 has_client_mcc($client_id)

    Разрешаем ли использовать клиентский МСС

=cut

sub has_client_mcc {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, 'client_mcc') ? 1 : 0;
}

=head2 has_postview_conversions_report($client_id)

    Разрешаем ли использовать клиентский МСС

=cut

sub has_postview_conversions_report {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, 'postview_conversions_report') ? 1 : 0;
}

=head2 has_edit_tin_in_user_settings($client_id)

    Разрешаем редактировать ИНН в настройках пользователя

=cut

sub has_edit_tin_in_user_settings {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, 'edit_tin_in_user_settings') ? 1 : 0;
}

=head2 has_client_reps_redesign_enabled_for_dna($client_id)

    Включен ли в DNA новый флоу представителей клиента

=cut

sub has_client_reps_redesign_enabled_for_dna {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, 'client_reps_redesign_enabled_for_dna') ? 1 : 0;
}

=head2 has_impression_standard_time_feature($client_id)

    Разрешаем клиенту выбор за сколько секунд (1-2) засчитывать показ ролика

=cut
sub has_impression_standard_time_feature {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "impression_standard_time") ? 1 : 0;
}

=head2 has_stat_4_digits_precision_feature

Показывать ли клиенту 4 знака после запятой в статистике

=cut

sub has_stat_4_digits_precision_feature {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "stat_4_digits_precision") ? 1 : 0;
}

=head2 has_lal_segments_feature($client_id)

    Включены ли LAL-сегменты в ретаргетинге

=cut
sub has_lal_segments_feature {
    my $client_id = shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "lal_segments_enabled") ? 1 : 0;
}

=head2 has_retargeting_only_lal_feature($client_id)

    Включена ли возможность указать только LAL-сегмент (без родителя) в условии ретаргетинга

=cut
sub has_retargeting_only_lal_feature {
    my $client_id = shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "retargeting_only_lal_enabled") ? 1 : 0;
}

=head2 has_disable_all_goals_optimization_for_dna_feature($client_id)

    Запрет переключения на оптимизацию по всем целям при сохранении стратегии

=cut
sub has_disable_all_goals_optimization_for_dna_feature {
    my $client_id = shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "disable_all_goals_optimization_for_dna") ? 1 : 0;
}

=head2 has_enable_phone_filter_in_click_places($client_id)

    Возвращает признак того, что для client_id в мастере отчетов в фильтрации по "Месту клика" доступно значение "Телефон"

=cut

sub has_enable_phone_filter_in_click_places {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "enable_phone_filter_in_click_places") ? 1 : 0;
}

=head2 has_turbo_app_allowed($client_id)

    Возвращает признак того, что для client_id доступна возможность продвигать турбо-аппы

=cut
sub has_turbo_app_allowed($) {
    my $client_id = shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, 'turbo_app_allowed') ? 1 : 0;
}

=head2 is_allowed_step_goals_in_strategies($client_id)

    Возвращает признак того, что для client_id доступны составные цели в стратегиях

=cut
sub is_allowed_step_goals_in_strategies {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "allow_step_goals_in_strategies") ? 1 : 0;
}

1;

=head2 has_mol_page_name_support($client_id)

    В МОЛ разрешается получать название площадки (`PageName`) напрямую, а не через словарь по `PageID`.
    Это позволяет получать имя площадки в случае, когда пейдж встраивается в страницу (берётся `TopAncestor`)

=cut
sub has_mol_page_name_support {
    my ($client_id) = @_;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "mol_page_name_support_enabled") ? 1 : 0;
}

=head2 has_simplified_strategy_view_enabled_feature($client_id)

    Разрешает simple и pro режимы редактирования стратегий в DNA

=cut
sub has_simplified_strategy_view_enabled_feature {
    my ($client_id) = @_;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "simplified_strategy_view_enabled") ? 1 : 0;
}

=head2 has_moderation_offer_enabled_for_dna($uid)

    Возвращает признак того, что для клиента скрыта оферта

=cut
sub has_moderation_offer_enabled_for_dna {
    my ($uid) = @_;
    return 0 unless $uid;
    return has_access_to_new_feature_from_java($uid, "moderation_offer_enabled_for_dna") ? 1 : 0;
}

=head2 has_goals_from_all_orgs_allowed($client_id)

    Разрешаем использовать цели из счетчиков любых организаций без проверки прав на счетчики

=cut
sub has_goals_from_all_orgs_allowed {
    my $client_id =  shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "goals_from_all_orgs_allowed") ? 1 : 0;
}

=head2 has_show_conversion_report_for_login

    Показываем отчет о конверсиях в МОЛ

=cut
sub has_show_conversion_report_for_login {
    my ($client_id) = @_;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "show_conversion_report_for_login") ? 1 : 0;
}

=head2 has_conversion_strategy_learning_status_enabled($client_id)

    Разрешает статус обучения у стратегий

=cut
sub has_conversion_strategy_learning_status_enabled {
    my ($client_id) = @_;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "conversion_strategy_learning_status_enabled") ? 1 : 0;
}

=head2 has_increased_cpa_limit_for_pay_for_conversion($client_id)

    Утроенная максимальная цена конверсий

=cut
sub has_increased_cpa_limit_for_pay_for_conversion {
    my ($client_id) = @_;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "increased_cpa_limit_for_pay_for_conversion") ? 1 : 0;
}

=head2 is_telegram_enabled($client_id)

    Доступность привязки телеграм-аккаунта

=cut
sub is_telegram_enabled {
    my ($client_id) = @_;
    return 0 unless $client_id;
    return has_feature($client_id, "telegram_enabled") ? 1 : 0;
}

=head2 has_cashback_page_feature_enabled($client_id)

    Доступность страницы бонусной программы

=cut
sub has_cashback_page_feature_enabled {
    my ($client_id) = @_;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "cashback_page_enabled") ? 1 : 0;
}

=head2 has_cashback_page_for_by_and_kz_feature_enabled($client_id)

    Доступность страницы бонусной программы для белорусских и казахских клиентов

=cut
sub has_cashback_page_for_by_and_kz_feature_enabled {
    my ($client_id) = @_;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "cashback_page_enabled_for_by_and_kz") ? 1 : 0;
}

=head2 has_uc_design_enabled($client_id)

    Доступность нового дизайна в DNA

=cut
sub has_uc_design_enabled {
    my ($client_id) = @_;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "uc_design_for_dna_edit_enabled") ? 1 : 0;
}

=head2 has_uc_grid_design_enabled($client_id)

    Доступность нового дизайна грида в DNA

=cut
sub has_uc_grid_design_enabled {
    my ($client_id) = @_;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "uc_design_for_dna_grid_enabled") ? 1 : 0;
}

=head2 has_enable_uc_dna_user_choice($client_id)

    Включить разводящую без таба Рекламной подписки

=cut

sub has_enable_uc_dna_user_choice {
    my ($client_id) = @_;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "enable_uc_dna_user_choice") ? 1 : 0;
}

=head2 pay_with_cash_disabled($client_id)

    Спрятан способ оплаты наличными

=cut
sub pay_with_cash_disabled {
    my ($client_id) = @_;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "pay_with_cash_disabled") ? 1 : 0;
}

=head2 new_cash_payment_enabled($client_id)

    Оплата наличиными через новое API yoomoney

=cut
sub new_cash_payment_enabled {
    my ($client_id) = @_;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "new_cash_payment_enabled") ? 1 : 0;
}

=head2 is_promocode_hint_enabled($client_id)

    Подсказка про промокод в попапе оплаты

=cut
sub is_promocode_hint_enabled {
    my ($client_id) = @_;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "enable_promocode_hint") ? 1 : 0;
}

=head2 is_preload_assets_enabled($client_id)

    Предзагружать ассеты для экранов c помощью тега link[rel=preload]

=cut
sub is_preload_assets_enabled {
    my ($client_id) = @_;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "enable_preload_assets") ? 1 : 0;
}

=head2 is_prefetch_assets_enabled($client_id)

    Предзагружать ассеты для экранов c помощью тега link[rel=prefetch]

=cut
sub is_prefetch_assets_enabled {
    my ($client_id) = @_;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "enable_prefetch_assets") ? 1 : 0;
}

=head2 has_avg_nshow_for_strategy_in_moc_feature($client_id)

    Включена ли возможность строить отчет по заметности показов по стратегии

=cut
sub has_avg_nshow_for_strategy_in_moc_feature {
    my $client_id = shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "avg_nshow_for_strategy_in_moc") ? 1 : 0;
}

=head2 has_avg_nshow_for_cpm_campaigns_in_moc_feature($client_id)

    Включена ли возможность строить отчет по заметности показов по типу кампании

=cut
sub has_avg_nshow_for_cpm_campaigns_in_moc_feature {
    my $client_id = shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "avg_nshow_for_cpm_campaigns_in_moc") ? 1 : 0;
}

=head2 has_avg_nshow_complete_for_strategy_in_moc_feature($client_id)

    Включена ли возможность строить отчет по заметности досмотров по стратегии

=cut
sub has_avg_nshow_complete_for_strategy_in_moc_feature {
    my $client_id = shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "avg_nshow_complete_for_strategy_in_moc") ? 1 : 0;
}

=head2 has_avg_nshow_complete_for_cpm_campaigns_in_moc_feature($client_id)

    Включена ли возможность строить отчет по заметности досмотров по типу кампании

=cut
sub has_avg_nshow_complete_for_cpm_campaigns_in_moc_feature {
    my $client_id = shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "avg_nshow_complete_for_cpm_campaigns_in_moc") ? 1 : 0;
}

=head2 has_flat_cpc_disabled($client_id)

    Возвращает значение фичи отключения стретегии совместного управления ставками

=cut
sub has_flat_cpc_disabled {
    my $client_id = shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "flat_cpc_disabled") ? 1 : 0;
}

=head2 has_flat_cpc_adding_disabled($client_id)

    Возвращает значение фичи отключения добавления стретегии совместного управления ставками

=cut
sub has_flat_cpc_adding_disabled {
    my $client_id = shift;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "flat_cpc_adding_disabled") ? 1 : 0;
}

=head2 pay_for_conversion_visible_in_filter_optimization_entity_for_dna($client_id)

    На смарт кампаниях в стратегиях для условия 'На каждый фильтр' доступна 'Оплата за конверсии'

=cut
sub pay_for_conversion_visible_in_filter_optimization_entity_for_dna {
    my ($client_id) = @_;
    return 0 unless $client_id;
    return has_feature($client_id, "pay_for_conversion_visible_in_filter_optimization_entity_for_dna") ? 1 : 0;
}

=head2 cpm_fixed_data_in_mol($client_id)

    Показывать все новые обязательные данные/срезы/фильтры в МОЛ

=cut
sub cpm_fixed_data_in_mol {
    my ($client_id) = @_;
    return 0 unless $client_id;
    return has_feature($client_id, "cpm_fixed_data_in_mol") ? 1 : 0;
}

=head2 cpm_additional_data_in_mol($client_id)

    Показывать данные/срезы/фильтры в МОЛ, которые требуют дополнительной проработки со стороны
    Движка/статистики/пкода etc

=cut
sub cpm_additional_data_in_mol {
    my ($client_id) = @_;
    return 0 unless $client_id;
    return has_feature($client_id, "cpm_additional_data_in_mol") ? 1 : 0;
}

=head2 do_not_check_passport_karma_for_xls_access($client_id)

    Не учитывать паспортную карму при проверке доступности XLS

=cut
sub do_not_check_passport_karma_for_xls_access {
    my ($client_id) = @_;
    return 0 unless $client_id;
    return has_feature($client_id, "do_not_check_passport_karma_for_xls_access") ? 1 : 0;
}

=head2 change_recurring_payment_sum_offer($client_id)

    Изменить предложенную сумму оплаты для повторного платежа (DIRECT-142057)

=cut
sub change_recurring_payment_sum_offer {
    my ($client_id) = @_;
    return 0 unless $client_id;
    return has_feature($client_id, "change_recurring_payment_sum_offer") ? 1 : 0;
}

=head2 change_first_payment_min_sum_recommendation($client_id)

    Изменить рекомендацию минимальной суммы первого платежа (DIRECT-142057)

=cut
sub change_first_payment_min_sum_recommendation {
    my ($client_id) = @_;
    return 0 unless $client_id;
    return has_feature($client_id, "change_first_payment_min_sum_recommendation") ? 1 : 0;
}

=head2 cross_device_attribution_types_enabled($client_id)

    Добавить кросс-девайс типы атрибуции

=cut
sub cross_device_attribution_types_enabled {
    my ($client_id) = @_;
    return 0 unless $client_id;
    return has_feature($client_id, "cross_device_attribution_types") ? 1 : 0;
}

=head2 creative_free_interface_enabled($client_id)

    Бескреативный интерфейс для смарт-баннеров

=cut
sub creative_free_interface_enabled {
    my ($client_id) = @_;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "creative_free_interface") ? 1 : 0;
}

=head2 has_package_strategies_stage_two_enabled($client_id)

    Пакетные стратегии, этап 2

=cut
sub has_package_strategies_stage_two_enabled {
    my ($client_id) = @_;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "package_strategies_stage_two") ? 1 : 0;
}

=head2 has_single_goal_cost_enabled($client_id)

    Отображать колонку "Цена цели" как одну

=cut
sub has_single_goal_cost_enabled {
    my ($client_id) = @_;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "single_goal_cost_enabled") ? 1 : 0;
}

=head2 use_new_page_for_copy_campaigns_between_clients($client_id)

    Новая страницы копирования кампаний

=cut
sub use_new_page_for_copy_campaigns_between_clients {
    my ($client_id) = @_;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "use_new_page_for_copy_campaigns_between_clients") ? 1 : 0;
}

=head2 has_get_strategy_id_from_shard_inc_strategy_id_enabled($client_id)

    Пакетные стратегии, этап 2

=cut
sub has_get_strategy_id_from_shard_inc_strategy_id_enabled {
    my ($client_id) = @_;
    return 0 unless $client_id;
    return has_access_to_new_feature_from_java($client_id, "get_strategy_id_from_shard_inc_strategy_id") ? 1 : 0;
}

=head2 get_operator_features($client_id)

    метод для получения списка фичей для сохраненного в запросе оператора.
    используется на страницы ошибок и в контроллерах, в которых фичи не передали, для инициализации dna

=cut
sub get_operator_features() {
    my $operator_uid = Tools::_get_operator_uid();
    my @operator_features = ();
    if (defined $operator_uid && $operator_uid > 0) {
        @operator_features = map {uc $_} @{allowed_features(get_clientid(uid => $operator_uid))};
    }
    return \@operator_features;
}

=head2 get_client_features($client_id)

    метод для получения списка фичей для сохраненного в запросе клиента.
    используется на страницы ошибок и в контроллерах, в которых фичи не передали, для инициализации dna

=cut
sub get_client_features() {
    my $client_id = Tools::_get_client_id();
    my @client_features = ();
    if (defined $client_id && $client_id > 0) {
        @client_features = map {uc $_} @{allowed_features($client_id)};
    }
    return \@client_features;
}

=head2 is_feature_cpm_banner_campaign_disabled_enabled($client_id)

    Возвращает признак того, что у клиента отключена возможность добавлять/изменять медийные кампании

=cut
sub is_feature_cpm_banner_campaign_disabled_enabled($) {
    my ($client_id) = @_;
    return has_access_to_new_feature_from_java($client_id, 'is_cpm_banner_campaign_disabled') ? 1 : 0;
}

1;
