package Client::ConvertToRealMoney;

use warnings;
use strict;

use base qw/Exporter/;
our @EXPORT_OK = qw/
    copy_old_campaign
    get_converted_campaign_params
    convert_campaign_bids
    fixate_client_currency
    get_convert_type
    get_closest_modify_convert_start_ts
    mass_queue_currency_convert
/;

use Settings;
use Yandex::DBTools;
use Yandex::DBShards;

use Yandex::ScalarUtils;
use Yandex::ListUtils;
use Yandex::TimeCommon;
use Yandex::HashUtils;
use Yandex::Memcached;

use Currencies;
use Currency::Rate;

use Common;
use Campaign;
use Campaign::Copy;
use Campaign::Types qw/camp_kind_in get_camp_kind_types get_camp_type_multi/;
use Primitives;
use PrimitivesIds;
use Client;
use TextTools;
use Property;

use List::Util qw/min max/;
use JSON;

use utf8;

# по сколько записей обновлять за один запрос в bids/bids_arc/bids_manual_prices
my $BIDS_UPDATE_CHUNK_SIZE = 1_000;

=head2 $SECONDS_BEFORE_MIDNIGHT_FOR_MODIFY_CONVERT

    За сколько секунд до полуночи перестаём приниматья заявки на перевод в полночь.
    Если до полуночи осталось меньше, чем указано здесь, то предлагаем переводиться только
    в следующую полночь.

=cut

our $SECONDS_BEFORE_MIDNIGHT_FOR_MODIFY_CONVERT = 4 * 60 * 60; # 4 часа (т.е. переход прекращаем в 20:00 по Москве)

=head2 $SECONDS_WE_ALLOW_TO_CHOOSE_CONVERT_DATE_IN_FUTURE

    На сколько максимум секунд вперёд можно оставить заявку на переход.

=cut

our $SECONDS_WE_ALLOW_TO_CHOOSE_CONVERT_DATE_IN_FUTURE = 16 * 24 * 60 * 60; # 16 дней

=head2 $COUNTRY_CURRENCY_LAST_UPDATE_INTERVAL_DAYS

    Актуальность данных про страны-валюты, когда позволяем себе ставить клиентов на принудительную конвертацию
    (писать уведомление и определять дату конвертации)

=cut

our $COUNTRY_CURRENCY_LAST_UPDATE_INTERVAL_DAYS = 7;

=head2 $COUNTRY_CURRENCY_LAST_UPDATE_INTERVAL_QUEUE_DAYS

    Актуальность данных про страны-валюты, когда позволяем себе начинать принудительно конвертацию
    (собственно, конвертировать ставим)

=cut

our $COUNTRY_CURRENCY_LAST_UPDATE_INTERVAL_QUEUE_DAYS = 14;


=head2 $MASS_CONVERT_BUTTON_CLIENTS_LIMIT

    Сколько клиентов за раз обрабатывает кнопка "massMoveToRealMoney"

=cut

our $MASS_CONVERT_BUTTON_CLIENTS_LIMIT = 4_000;

=head2 @KNOWN_BAD_CLIENTIDS

    Список проблемных клиентов, которым нельзя включать тизер конвертации

=cut


our @KNOWN_BAD_CLIENTIDS = (
    # список клиентов со смешанными НДС из DIRECT-25412
    1000315, 1009700, 1011808, 1014550, 1015265, 1015330, 1019161, 1029341,
    1032738, 1035787, 1039424, 1045944, 1046357, 1047093, 1048075, 1051757,
    1054212, 1058442, 1062349, 1064051, 1067556, 1073455, 1079323, 1079687,
    1091796, 1092192, 1095626, 1098098, 1103445, 1103480, 1106934, 1107485,
    1113614, 1116322, 1117119, 1122671, 1125669, 1126663, 1132307, 1137765,
    1146522, 1156882, 1161352, 1166098, 1167888, 1171529, 1175272, 1176882,
    1185299, 1196376, 1208045, 1208181, 1210227, 1218632, 1220674, 1223998,
    1224151, 1225569, 1230476, 1231945, 1233414, 1236785, 1237879, 1242639,
    1244577, 1244615, 1245699, 1249365, 1260727, 1261445, 1261758, 1264047,
    1265575, 1266179, 12687, 1272222, 1274549, 1274609, 1280175, 1283959,
    1287932, 1295483, 1295855, 1296090, 1301644, 1304387, 1306170, 1306498,
    1307191, 1308268, 1310657, 1311562, 1322682, 1332769, 1351580, 1360863,
    1371417, 1371605, 1371785, 1372231, 1386746, 1390709, 1394162, 1395150,
    1404553, 1404820, 1410183, 1419253, 1421522, 1424378, 1443695, 1445406,
    1446756, 1447900, 1448949, 1450191, 1461016, 1464923, 1465361, 1468733,
    1472518, 1475306, 1477905, 1480900, 1481746, 1484485, 1492618, 1497674,
    1498164, 1499669, 1508241, 1511707, 1517115, 1520834, 1523207, 1524746,
    1529151, 1532116, 1532235, 1534827, 1535101, 15397, 1545493, 1547320,
    1549961, 1556766, 1558917, 1559055, 1560638, 1561138, 1561756, 1566578,
    1569325, 1569333, 1573630, 1576107, 1580523, 1581835, 1584097, 1588216,
    1594216, 1595586, 1596644, 1597626, 1597985, 1598723, 1600211, 1603011,
    1603465, 1616019, 1616185, 1618103, 1619173, 1623104, 1623750, 1626357,
    1633060, 16333, 1636088, 16525, 1654505, 1657641, 1662325, 1681002, 1684953,
    1690489, 1691000, 1702330, 1712468, 1713443, 1723075, 1724149, 1727063,
    1728118, 1729008, 1731507, 1733690, 1739934, 1743441, 1751102, 1753456,
    1755930, 1756065, 1757807, 1769419, 1771326, 1775359, 1775716, 1783960,
    1785143, 1790961, 1792861, 1793244, 1799255, 1809344, 1814879, 1816627,
    1822262, 1825210, 1828167, 1830666, 1831035, 1833099, 1835399, 1836021,
    1836277, 1840121, 1841381, 1841981, 1844321, 1845265, 1845641, 1849369,
    1850811, 1851628, 1851768, 1852153, 1852208, 1853248, 1853513, 1853526,
    1855331, 1857828, 1862596, 1864882, 1867051, 1867602, 1867742, 1869996,
    1871059, 1871443, 1877572, 1879884, 1879902, 1882581, 1882678, 1885564,
    1887225, 1887925, 1892504, 1892632, 1893389, 1896438, 1897956, 1899310,
    1899888, 1901764, 1902968, 1903490, 1904150, 1904769, 1905853, 1906964,
    1907545, 1908045, 1909642, 1914055, 1914075, 1914171, 1914235, 1914942,
    1916360, 1923203, 1924431, 1924433, 1925580, 1927363, 1928759, 1932835,
    1933306, 1933816, 1934701, 1935547, 1939763, 1939967, 1940951, 1942258,
    1943862, 1945071, 1947059, 1951058, 1951429, 1952470, 1953273, 1956913,
    1957431, 1957610, 1958046, 1959236, 1959859, 1964603, 1965122, 1965136,
    1967681, 1968599, 1969950, 1970124, 1972395, 1973879, 1974876, 1974967,
    1979009, 1985607, 1985613, 1990507, 1994214, 1997952, 2004214, 2004535,
    2010958, 2012011, 2013382, 2013425, 2015110, 2016264, 2016279, 2016851,
    2018140, 2019791, 2020206, 2020261, 2020314, 2020689, 2021297, 2022595,
    2023151, 2023525, 2024125, 2025058, 2027884, 2029660, 2030986, 2030993,
    2030995, 2035873, 2036662, 2037795, 2039114, 2042737, 2044052, 2045385,
    2047870, 2050757, 2056835, 2062147, 2065484, 2067035, 2068076, 2072236,
    2074492, 2077253, 2081289, 2086122, 2089534, 2094919, 2102372, 2105853,
    2108412, 2112091, 2116588, 2121734, 2127295, 2137951, 2139861, 2139876,
    2139879, 2150699, 2155095, 2156737, 2161359, 2170705, 2173315, 2173776,
    2175010, 2177009, 2180815, 2185235, 2185450, 2185996, 2191767, 2196186,
    2196715, 2198711, 2198982, 2199304, 2201896, 2204660, 2209450, 2218308,
    2223176, 2238489, 2248864, 2249180, 2250562, 2273058, 2273817, 2275297,
    2286911, 2288736, 2290584, 2292814, 2296564, 2298807, 2307395, 2308943,
    2311632, 2313148, 2314126, 2314574, 2355860, 2357128, 2359664, 2364992,
    2366135, 2368498, 2387999, 2390507, 2395666, 2399676, 2401990, 2403392,
    2408681, 2415010, 2416480, 2420043, 2435462, 2446904, 2447863, 2452680,
    2460233, 2460247, 2460908, 2528378, 2550231, 2555040, 2687537, 2693526,
    2695625, 2697016, 2698116, 2706371, 2733321, 2755293, 2757510, 2757736,
    2758553, 2761639, 2777940, 2778740, 2790018, 2792617, 2793824, 2794199,
    2803580, 2805214, 2822963, 2849204, 2855353, 2865113, 2866145, 2866969,
    2868831, 2874996, 2882345, 2931609, 2936611, 2936733, 2943186, 2961351,
    2964905, 2965026, 2965196, 2973268, 2984528, 2990914, 2997666, 2999135,
    2999234, 3000623, 3010608, 3011912, 3015788, 3017687, 3022173, 3030473,
    303128, 3048860, 3073131, 3073712, 3081376, 3091445, 3118459, 3137190,
    313845, 3144484, 3149820, 3150171, 3159456, 3166927, 3177785, 3195463,
    3209714, 3223366, 3242926, 3247005, 3249313, 3256049, 326774, 3267860,
    3270412, 3271998, 327380, 3282899, 3306244, 3306683, 330889, 330904,
    3310001, 3313458, 3314219, 3328995, 3342013, 3345456, 3363605, 3367123,
    3370999, 3374268, 3385922, 3400335, 3400888, 3400969, 3410703, 3425133,
    3426378, 3445932, 3449746, 3449997, 3458424, 3462549, 3477992, 3488699,
    3489130, 3495930, 34979, 3501320, 3504574, 3509951, 3515618, 3532849,
    3535189, 3545580, 354800, 355111, 3566092, 358663, 3607947, 36430, 3656349,
    36579, 366586, 3685951, 3707810, 3726670, 3748871, 3818937, 38324, 3895486,
    389563, 390000, 3911790, 391936, 395278, 396738, 397267, 397543, 397962,
    401115, 401440, 4019626, 4068807, 4071367, 408609, 409029, 4103335, 410396,
    4113374, 4145196, 415056, 415676, 417269, 418251, 419751, 4211801, 4211859,
    421948, 422846, 4228690, 4230484, 4233341, 4241680, 426104, 428019, 4286930,
    4293704, 4294099, 4320340, 4323201, 4342970, 435207, 4364267, 437027,
    4370278, 4372111, 438580, 4392662, 4399150, 440766, 443643, 4445940, 446667,
    452668, 456092, 458183, 460524, 461347, 4638476, 469123, 4730946, 473771,
    473989, 476562, 476988, 47760, 479043, 479800, 4813434, 483596, 489125,
    489948, 493787, 496658, 4971346, 497652, 504215, 505854, 506106, 507239,
    50907, 509542, 510807, 512825, 514194, 515613, 520718, 521686, 525839,
    5271531, 5275753, 527696, 5319142, 532478, 5356280, 537240, 5389528, 540316,
    5419745, 5429776, 5443816, 544436, 544496, 5465631, 5511794, 552095, 555216,
    5588421, 560357, 560920, 5614965, 5632196, 5632210, 5632220, 5632226,
    564146, 5642261, 568005, 5693438, 57154, 573779, 579710, 582342, 5883169,
    5951848, 5965165, 5983533, 5983589, 5983599, 6042709, 6060181, 6060221,
    6109506, 6112030, 6152360, 619798, 6308292, 6311320, 6394796, 6530929,
    6530943, 6530963, 6563972, 6564472, 6564511, 6564535, 6599107, 6619985,
    674736, 6877697, 6991555, 776795, 779368, 779602, 782659, 787395, 787637,
    794246, 796096, 801773, 808746, 809930, 810561, 813182, 821578, 822195,
    823075, 825947, 827933, 832602, 842074, 848378, 852038, 864689, 865993,
    878554, 878866, 882860, 905978, 907620, 919212, 926033, 928613, 929327,
    932373, 940242, 949458, 951306, 951633, 952589, 956291, 957124, 964485,
    972409, 974062, 982017, 984955, 985391, 988588, 996396,

   # DIRECT-44438
   1103959,
);

=head2 @KNOWN_BAD_AGENCY_CLIENTIDS

    Список проблемных агентств, клиентам которых нельзя включать тизер конвертации

=cut


our @KNOWN_BAD_AGENCY_CLIENTIDS = (
);

=head2 copy_old_campaign

    Копирует старую кампанию в новую с сохранением максимально возможного числа параметров.
    Скопированная кампания останется с statusEmpty = 'Yes'.

    Принимает позиционные параметры:
        - ссылку на объект RBAC
        - ссылку на хеш, описывающий кампанию
        - ссылку на значения, которые нужно заменить в кампании, разбитые по кампаниям
          (аналог параметра override у copy_camp)
        - номер кампании-кошелька, к которому привязать скопированную кампанию [не обязательный]
    И опциональные параметры:
        - price_convert_rate - по какому курсу сконвертировать ставки в фразах/ретаргетинге
        - new_currency - какая новая валюта у кампании

    Возвращает номер скопированной кампании.

=cut

our %CURRENCY_CONVERT_COPY_FLAGS = (
    copy_stopped => 1,
    copy_archived => 1,
    copy_ctr => 1,
    copy_moderate_status => 1,
    copy_archived_campaigns => 1,
    copy_phrase_status => 1,
    copy_statusMail => 1,
    copy_statusNoPay => 1,
    copy_autobudget_goal_id => 1,
    copy_sms_settings => 1,
    copy_auto_optimize_request => 1,
    copy_mediaplan_status => 1,
    copy_camp_description => 1,
    copy_notification_settings => 1,
    copy_bid_place => 1,
    copy_bid_warn_status => 1,
    copy_retargetings => 1,
    copy_bids_href_params => 1,
    copy_camp_secondary_options => 1,
    copy_autobudget_goal_id => 1,
    save_conversion_strategy => 1,
    copy_favorite => 1,
);

sub copy_old_campaign {
    my ($rbac, $campaign, $override, $new_wallet_cid, %O) = @_;

    my $new_cid = copy_camp($rbac, $campaign->{cid}, $campaign->{uid}, undef, undef,
        flags => {
            %CURRENCY_CONVERT_COPY_FLAGS,
            copy_manager => 1,
            copy_agency => 1,
        },
        override => $override,
        keep_empty => 1,
        new_wallet_cid => $new_wallet_cid,
        price_convert_rate => $O{price_convert_rate},
        new_currency => $O{new_currency},
        UID => $O{UID},
        converting_client_currency => 1,
    );

    return $new_cid;
}

=head2 get_converted_campaign_params

    Конвертирует параметры кампании в у.е. в соответствующие параметры
    в реальной валюте.

    Принимает позиционные параметры:
        - ссылку на хеш, описывающий кампанию
        - код валюты, в которую происходит переход (RUB/USD/EUR/...)
        - курс у.е. к реальной валюты
    Принимает именованные параметры:
        - keep_ProductID -- не изменять ProductID кампании
        - copy_last_bill_sums -- скопировать данные о последнем выставленном/оплаченном счёте (sum_to_pay/sum_last)
    Возвращает ссылку на хеш с конвертированными параметрами вида {
        campaigns => {
            currency => ...,
            ProductID => ...,
            sum_to_pay => ...,
            sum_last => ...,
            currency => ...,
            autobudgetForecast => ...,
            day_budget => ...,
            strategy_data => ...,
        },
        camp_options => {
            manual_autobudget_sum => ...,
        },
    }

=cut
{
sub _convert_param {
    my ($campaign, $rate, $currency, $field_description, %opt) = @_;

    if (ref($field_description) ne 'HASH') {
        $field_description = {field => $field_description};
    }

    my $name = $field_description->{field};
    my $value = $campaign->{ $name };
    if (defined $value && $value > 0) {
        if (defined $field_description->{default_constant}) {
            my $current_currency = $opt{current_currency} || $campaign->{currency} || 'YND_FIXED'; # currency_defaults
            my $current_default_value = get_currency_constant($current_currency, $field_description->{default_constant});
            # если было дефолтное значение, то оставляем его же, но для новой валюты
            if (abs($value - $current_default_value) < $Currencies::EPSILON) {
                my $new_default_value = get_currency_constant($currency, $field_description->{default_constant});
                $value = $new_default_value;
            } else {
                $value *= $rate;
            }
        } else {
            $value *= $rate;
        }
        if ($field_description->{round}) {
            if ($field_description->{round} eq 'up' || $field_description->{round} eq 'down') {
                $value = Currencies::round_price_to_currency_step($value, $currency, $field_description->{round} => 1);
            } else {
                $value = round2s($value);
            }
        } else {
            # по умолчанию округляем до точности валюты
            $value = Currencies::round_to_currency_digit_count($value, $currency);
        }
        if (defined $field_description->{min_constant}) {
            my $min_value = get_currency_constant($currency, $field_description->{min_constant});
            $value = max($value, $min_value);
        }
        if (defined $field_description->{max_constant}) {
            my $max_value = get_currency_constant($currency, $field_description->{max_constant});
            $value = min($value, $max_value);
        }
    }
    return ($name => $value);
}

sub get_converted_campaign_params {
    my ($campaign, $currency, $rate, %O) = @_;

    my $old_strategy = $campaign->{strategy_decoded} || {};

    my $is_wallet = $campaign->{type} eq 'wallet';
    my $is_performance = $campaign->{type} eq 'performance';

    my ($bid_min_constant, $avg_bid_min_constant, $avg_cpa_min_constant);
    if ($is_performance) {
        $bid_min_constant = 'MIN_CPC_CPA_PERFORMANCE';
        $avg_bid_min_constant = 'MIN_CPC_CPA_PERFORMANCE';
        $avg_cpa_min_constant = 'MIN_CPC_CPA_PERFORMANCE';
    } else {
        $bid_min_constant = $O{save_conversion_strategy} && $old_strategy->{avg_cpa} && $old_strategy->{avg_cpa} > 0 ? 'MIN_AUTOBUDGET_AVG_CPA' : 'MIN_AUTOBUDGET_BID';
        $avg_bid_min_constant = 'MIN_AUTOBUDGET_AVG_PRICE';
        $avg_cpa_min_constant = 'MIN_AUTOBUDGET_AVG_CPA';
    }
    my @campaign_fields_to_change = (qw/
        autobudgetForecast
        /,
        {field => 'day_budget', min_constant => ($is_wallet ? 'MIN_WALLET_DAY_BUDGET' : 'MIN_DAY_BUDGET'), max_constant => 'MAX_DAILY_BUDGET_AMOUNT'},
    );

    if ($O{copy_last_bill_sums}) {
        push @campaign_fields_to_change, qw/sum_to_pay sum_last/;
    }

    my %strategy_field_to_change = map {($_->{field} => $_)} (
        {field => 'sum', min_constant => 'MIN_AUTOBUDGET', max_constant => 'MAX_AUTOBUDGET'},
        {field => 'avg_bid', min_constant => $avg_bid_min_constant, max_constant => 'MAX_AUTOBUDGET_BID'},
        {field => 'filter_avg_bid', min_constant => 'MIN_CPC_CPA_PERFORMANCE', max_constant => 'MAX_AUTOBUDGET_BID'},
        {field => 'bid', min_constant => $bid_min_constant, max_constant => 'MAX_AUTOBUDGET_BID'},
        {field => 'avg_cpa', min_constant => $avg_cpa_min_constant},
        {field => 'avg_cpi', min_constant => $avg_cpa_min_constant},
        {field => 'filter_avg_cpa', min_constant => 'MIN_CPC_CPA_PERFORMANCE'},
    );

    my %new_strategy =
        map {
            my $key = $_;
            my $val = $old_strategy->{$key};
            my $rule = $strategy_field_to_change{$key};
            $rule
                ? _convert_param($old_strategy, $rate, $currency, $rule, current_currency => $campaign->{currency})
                : ($key => $val)
        }
        keys %$old_strategy;

    my %changes = (
        campaigns => {
            currency => $currency,
            # write new strategy only if it is defined
            ($campaign->{strategy_decoded} ? (strategy_data => to_json(\%new_strategy)) : ()),
            map { _convert_param($campaign, $rate, $currency, $_) } @campaign_fields_to_change,
        },
    );


    if (!$O{keep_ProductID}) {
        my $type = $campaign->{type} || $campaign->{mediaType};
        die "no type known for campaign" unless $type;
        my $product = product_info(type => $type, currency => $currency);
        die "no suitable product found for type = $type and currency $currency" unless $product && $product->{ProductID};
        $changes{campaigns}->{ProductID} = $product->{ProductID};
    }

    my @camp_options_fields_to_change = (
        {field => 'manual_autobudget_sum', min_constant => 'MIN_AUTOBUDGET', max_constant => 'MAX_AUTOBUDGET'},
    );

    # camp_options трогаем только если задана manual_autobudget_sum
    if ($campaign->{manual_autobudget_sum} && $campaign->{manual_autobudget_sum} > 0) {
        $changes{camp_options} = {
            map { _convert_param($campaign, $rate, $currency, $_) } @camp_options_fields_to_change,
        };
    }

    return \%changes;
}

=head3 _convert_json_fields

    Конвертирует указанные поля в сериализованном в JSON хеше
    Используется для сохранённых настроек оффлайнового и онлайнового конструктора

    $new_json_data = _convert_json_fields($old_json_data, $rate, $new_currency, $fields_to_change_description);

=cut

sub _convert_json_fields {
    my ($json_data, $rate, $new_currency, $fields) = @_;

    my $data = eval { from_json($json_data) } || {};
    if (!$data || !%$data) {
        return undef;
    }
    my @existing_fields = grep { my $field = ref($_) ? $_->{field} : $_; defined $data->{$field} } @$fields;
    if (@existing_fields) {
        hash_merge $data, { map { _convert_param($data, $rate, $new_currency, $_) } @existing_fields };
        return to_json($data);
    } else {
        # менять нечего, оставляем тоже, что и было
        return $json_data;
    }
}

=head2 convert_campaign_secondary_options

    Конвертирует в валюту настройки для онлайнового и оффлайнового
    конструкторов цен в таблице camp_secondary_options.

    convert_campaign_secondary_options(\@cids, $old_currency, $new_currency);

=cut

sub convert_campaign_secondary_options {
    my ($cids, $old_currency, $new_currency) = @_;

    for my $cid (@$cids) {
        my $old_camp_options = Common::get_camp_options($cid);
        if ($old_camp_options) {
            my %new_camp_options;
            my $rate = convert_currency(1, $old_currency, $new_currency);

            if ($old_camp_options->{price_editor}) {
                my @fields_to_change = (
                    {field => 'price', min_constant => 'MIN_PRICE', max_constant => 'MAX_PRICE', default_constant => 'MIN_PAY', round => 'up'},
                    {field => 'price_context', min_constant => 'MIN_PRICE', max_constant => 'MAX_PRICE', default_constant => 'MIN_PAY', round => 'up'},
                    {field => 'price_search', min_constant => 'MIN_PRICE', max_constant => 'MAX_PRICE', default_constant => 'MIN_PAY', round => 'up'},
                    {field => 'price_performance', min_constant => 'MIN_PRICE', max_constant => 'MAX_PRICE', default_constant => 'MIN_PAY', round => 'up'},
                );
                $new_camp_options{price_editor} = _convert_json_fields($old_camp_options->{price_editor}, $rate, $new_currency, \@fields_to_change);
            }
            if ($old_camp_options->{offline_price_editor}) {
                my @fields_to_change = (
                    {field => 'price', min_constant => 'MIN_PRICE', max_constant => 'MAX_PRICE', default_constant => 'MIN_PAY', 'round' => 'up'},
                    {field => 'price_context', min_constant => 'MIN_PRICE', max_constant => 'MAX_PRICE', default_constant => 'MIN_PAY', 'round' => 'up'},
                    {field => 'price_search', min_constant => 'MIN_PRICE', max_constant => 'MAX_PRICE', default_constant => 'MIN_PAY', 'round' => 'up'},
                    {field => 'simple_price', min_constant => 'MIN_PRICE', max_constant => 'MAX_PRICE', default_constant => 'MIN_PAY', 'round' => 'up'},
                    {field => 'wizard_search_max_price', min_constant => 'MIN_PRICE', max_constant => 'MAX_PRICE', default_constant => 'MIN_PAY', 'round' => 'up'},
                    {field => 'wizard_network_max_price', min_constant => 'MIN_PRICE', max_constant => 'MAX_PRICE', default_constant => 'MIN_PAY', 'round' => 'up'},
                    {field => 'search_max_price', min_constant => 'MIN_PRICE', max_constant => 'MAX_PRICE', default_constant => 'MIN_PAY', 'round' => 'up'},
                    {field => 'ctx_max_price', min_constant => 'MIN_PRICE', max_constant => 'MAX_PRICE', default_constant => 'MIN_PAY', 'round' => 'up'},
                    {field => 'single_price_ctx', min_constant => 'MIN_PRICE', max_constant => 'MAX_PRICE', default_constant => 'MIN_PAY', 'round' => 'up'},
                    {field => 'single_price', min_constant => 'MIN_PRICE', max_constant => 'MAX_PRICE', default_constant => 'MIN_PAY', 'round' => 'up'},
                    {field => 'price_performance', min_constant => 'MIN_PRICE', max_constant => 'MAX_PRICE', default_constant => 'MIN_PAY', round => 'up'},
                );
                $new_camp_options{offline_price_editor} = _convert_json_fields($old_camp_options->{offline_price_editor}, $rate, $new_currency, \@fields_to_change);
            }
            if ($old_camp_options->{easy_week_bundle_sum_ue}) {
                my $field_description = {field => 'easy_week_bundle_sum_ue', , min_constant => 'MIN_PRICE', max_constant => 'MAX_PRICE', round => 1};
                hash_merge \%new_camp_options, { _convert_param($old_camp_options, $rate, $new_currency, $field_description) };
            }

            if (%new_camp_options) {
                update_camp_options($cid, \%new_camp_options);
            }
        }
    }
}
}

=head2 convert_campaign_payments_info

    Конвертирует в валюту последнюю зачисленную сумму в таблице camp_payments_info.

=cut
sub convert_campaign_payments_info {
    my ($cids, $old_currency, $new_currency) = @_;

    foreach_shard cid => $cids, sub {
        my ($shard, $cids_chunk) = @_;
        my $payments_sums = get_hash_sql(PPC(shard => $shard), 
                                         ["SELECT cid, last_payment_sum FROM camp_payments_info", 
                                          WHERE => {cid => $cids_chunk}]
            );

        for my $cid (keys %$payments_sums) {
            $payments_sums->{$cid} = convert_currency($payments_sums->{$cid}, $old_currency, $new_currency);;
        }

        do_mass_update_sql(PPC(shard => $shard), "camp_payments_info", "cid",
                           {map { $_ => {last_payment_sum => $payments_sums->{$_}} } keys %$payments_sums},
            );
    };
}

=head2 convert_campaign_bids

    Конвертирует ставки в кампании из у.е. в реальную валюту.

    Принимает позиционные параметры:
        - cid'ы конвертируемых кампаний
        - старую валюту кампании
        - новую валюту кампании

    convert_campaign_bids([$cid1, $cid2, ...], $old_currency, $new_currency);

=cut

sub convert_campaign_bids {
    my ($cids, $old_currency, $new_currency) = @_;

    my $campaign_type_by_cid = get_camp_type_multi(cid => $cids);
    my $rate = convert_currency(1, $old_currency, $new_currency);
    my $price_convert_sql = Campaign::Copy::get_price_convert_sql($rate, $new_currency, 'price');
    my $price_context_convert_sql = Campaign::Copy::get_price_convert_sql($rate, $new_currency, 'price_context');
    my $price_cpc_convert_sql = Campaign::Copy::get_price_convert_sql($rate, $new_currency, 'price_cpc', min_const => 'MIN_CPC_CPA_PERFORMANCE');
    my $price_cpa_convert_sql = Campaign::Copy::get_price_convert_sql($rate, $new_currency, 'price_cpa', min_const => 'MIN_CPC_CPA_PERFORMANCE');

    # в bids данные обновляем единственным запросом, т.к. там есть индекс по id
    foreach_shard cid => $cids, sub {
        my ($shard, $cids_chunk) = @_;
        my $bids_ids = get_one_column_sql(PPC(shard => $shard), ["SELECT id FROM bids", WHERE => {cid => $cids_chunk}]) || [];
        for my $ids_chunk(chunks $bids_ids, $BIDS_UPDATE_CHUNK_SIZE) {
            do_update_table(PPC(shard => $shard), 'bids', {
                price__dont_quote => $price_convert_sql,
                price_context__dont_quote => $price_context_convert_sql,
                statusBsSynced => 'No',
            }, where => {id => $ids_chunk});
        }
    };

    # в bids_arc обновляем отдельным запросом на каждое условие, т.к. индекс есть только по cid+(bid|pid)+id, а фразы привязаны к условию
    my $bids_arc_data = get_all_sql(PPC(cid => $cids), ['SELECT cid, pid, id FROM bids_arc', WHERE => {cid => SHARD_IDS}]);
    if ($bids_arc_data && @$bids_arc_data) {
        my %cid_pid_ids;
        for my $row(@$bids_arc_data) {
            push @{$cid_pid_ids{$row->{cid}}->{$row->{pid}}}, $row->{id};
        }
        while (my ($cid, $pid_ids) = each %cid_pid_ids) {
            while (my ($pid, $ids) = each %$pid_ids) {
                for my $ids_chunk(chunks $ids, $BIDS_UPDATE_CHUNK_SIZE) {
                    do_update_table(PPC(cid => $cid), 'bids_arc', {
                        price__dont_quote => $price_convert_sql,
                        price_context__dont_quote => $price_context_convert_sql,
                        statusBsSynced => 'No',
                    }, where => {
                        cid => $cid,
                        pid => $pid,
                        id => $ids_chunk,
                    });
                }
            }
        }
    }

    # конвертируем цены на условия ретаргетинга
    foreach_shard cid => $cids, sub {
        my ($shard, $cids_chunk) = @_;
        my $bids_rets_data = get_all_sql(PPC(shard => $shard), [q/
            SELECT p.cid, br.ret_id, br.pid, br.bid, br.price_context
            FROM phrases p
            INNER JOIN bids_retargeting br ON p.pid = br.pid
        /,  WHERE => {'p.cid' => $cids_chunk}]) || [];
        my $bids_ret_ids = [ map { $_->{ret_id} } @$bids_rets_data ];
        for my $ids_chunk(chunks $bids_ret_ids, $BIDS_UPDATE_CHUNK_SIZE) {
            do_update_table(PPC(shard => $shard), 'bids_retargeting', {
                price_context__dont_quote => $price_context_convert_sql,
                statusBsSynced => 'No',
                }, where => {ret_id => $ids_chunk});
        }
    };

    # Конвертируем цены на условиях нацеливания для динамических кампаний
    foreach_shard cid => [grep { $campaign_type_by_cid->{$_} eq 'dynamic' } @$cids], sub {
        my ($shard, $cids_chunk) = @_;
        my $bids_dyn_ids = get_one_column_sql(PPC(shard => $shard), [
            'SELECT bd.dyn_id FROM phrases g JOIN bids_dynamic bd ON (bd.pid = g.pid)',
            WHERE => { 'g.cid' => $cids_chunk },
        ]);
        for my $ids_chunk (chunks $bids_dyn_ids, $BIDS_UPDATE_CHUNK_SIZE) {
            do_update_table(PPC(shard => $shard), 'bids_dynamic', {
                price__dont_quote => $price_convert_sql,
                price_context__dont_quote => $price_context_convert_sql,
                statusBsSynced => 'No',
            }, where => {dyn_id => $ids_chunk});
        }
    };

    foreach_shard cid => [grep { $campaign_type_by_cid->{$_} eq 'performance' } @$cids], sub {
        my ($shard, $cids_chunk) = @_;
        my $perf_filter_ids = get_one_column_sql(PPC(shard => $shard), [
            'SELECT bp.perf_filter_id FROM phrases g JOIN bids_performance bp USING(pid)',
            WHERE => { 'g.cid' => $cids_chunk },
        ]);
        for my $ids_chunk (chunks $perf_filter_ids, $BIDS_UPDATE_CHUNK_SIZE) {
            do_update_table(PPC(shard => $shard), 'bids_performance', {
                price_cpc__dont_quote => "IF(price_cpc > 0, $price_cpc_convert_sql, 0)",
                price_cpa__dont_quote => "IF(price_cpa > 0, $price_cpa_convert_sql, 0)",
                statusBsSynced => 'No',
            }, where => {perf_filter_id => $ids_chunk});
        }
    };

    # обновляем старые ручные ставки в bids_manual_prices отдельным запросом на каждую кампанию
    my $bids_manual_prices_data;
    $bids_manual_prices_data = get_all_sql(PPC(cid => $cids), ['SELECT cid, id FROM bids_manual_prices', WHERE => {cid => SHARD_IDS}]);
    if ($bids_manual_prices_data && @$bids_manual_prices_data) {
        my %cid2ids;
        for my $row(@$bids_manual_prices_data) {
            push @{$cid2ids{$row->{cid}}}, $row->{id};
        }
        while (my ($cid, $ids) = each %cid2ids) {
            for my $ids_chunk(chunks $ids, $BIDS_UPDATE_CHUNK_SIZE) {
                do_update_table(PPC(cid => $cid), 'bids_manual_prices', {
                    price__dont_quote => $price_convert_sql,
                    price_context__dont_quote => $price_context_convert_sql,
                }, where => {
                    cid => $cid,
                    id => $ids_chunk,
                });
            }
        }
    }
}

=head2 copy_camp_converting_currency

    Версия copy_camp, которая умеет копировать кампании между клиентами в разных валютах.
    Если валюты разные, то параметры кампании и ставки будут сконвертированы как при переводе в валюту копированием.
    Все нужные данные получает из базы сама.

    Принимаемые параметры полностью совпадают с Campaign::Copy::copy_camp.
    Возвращает номер новой кампании или undef в случае ошибки.

    $new_cid = copy_camp_converting_currency($rbac, $FORM{cid}, $new_uid, $manager_uid, $agency_uid, %O);

=cut

sub copy_camp_converting_currency {
    my ($rbac, $old_cid, $new_uid, $manager_uid, $agency_uid, %O) = @_;

    # multicurrency-lock: брать shared-локи на исходного и целевого клиентов, чтобы избежать изменения их валют во время копирования

    my $camp_info = get_camp_info($old_cid, undef, short => 1);
    return undef unless $camp_info && $camp_info->{cid};
    my $old_currency = $camp_info->{currency};

    my $new_currency;
    if (camp_kind_in(type => $camp_info->{type}, 'with_currency')) {
        my $new_client_id = get_clientid(uid => $new_uid);
        return undef unless $new_client_id;

        my $new_client_currencies = get_client_currencies($new_client_id);
        $new_currency = $new_client_currencies->{work_currency};
    } else {
        # Баян и Геоконтекст остаются в у.е. вне зависимости от валюты клиента
        $new_currency = 'YND_FIXED';
    }

    if ($old_currency eq $new_currency) {
        return copy_camp($rbac, $old_cid, $new_uid, $manager_uid, $agency_uid, %O);
    } else {
        my $rate = convert_currency(1, $old_currency, $new_currency);
        my $converted_campaign_params = get_converted_campaign_params($camp_info, $new_currency, $rate, %{hash_cut($O{flags}, qw/copy_last_bill_sums save_conversion_strategy/)});
        my $override = hash_merge {}, $O{override}, $converted_campaign_params;
        my $new_cid = copy_camp($rbac, $old_cid, $new_uid, $manager_uid, $agency_uid, %O, override => $override, keep_empty => 1, price_convert_rate => $rate, new_currency => $new_currency);
        convert_campaign_secondary_options([$new_cid], $old_currency, $new_currency);
        if ( !$O{keep_empty} ) {
            do_update_table(PPC(cid => $new_cid), 'campaigns', {statusEmpty => 'No', statusBsSynced => 'No'}, where => {cid => $new_cid});
        }
        return $new_cid;
    }
}

=head2 fixate_client_currency

    Записывает в базу факт окончания перевода клиента в новую валюту.
    По факту записывает новую work_currency и дату перехода.
    Вызывать её надо в транзакции.

    Принимает позиционные параметры:
        - ClientID
        - новую валюту клиента (RUB/EUR/USD/...)
        - страну клиента (в виде id региона)

    do_in_transaction {
        fixate_client_currency($client_id, $new_currency, $new_country_region_id);
    };

=cut

sub fixate_client_currency {
    my ($client_id, $new_currency, $new_country_region_id) = @_;

    create_update_client({client_data => {
        ClientID => $client_id,
        work_currency => $new_currency,
        country_region_id => $new_country_region_id,
    }});
}

{
my @TABLES_TO_CLEANUP_BY_CLIENTID = qw/
    client_firm_country_currency
    clients_to_force_multicurrency_teaser
    force_currency_convert
    client_teaser_data_lastupdate
/;

=head2 cleanup_data_for_converted_client($client_id)

    Удаляет данные по клиенту, которые больше не требуются после конвертации в валюту.
    Функцию следует вызывать в одной транзакции с fixate_client_currency.

    Список очищаемых таблиц
        записан выше в @TABLES_TO_CLEANUP_BY_CLIENTID

=cut

sub cleanup_data_for_converted_client {
    my $client_id = shift;

    for my $table (@TABLES_TO_CLEANUP_BY_CLIENTID) {
        do_delete_from_table(PPC(ClientID => $client_id), $table, where => { ClientID => $client_id });
    }
}
}

=head2 get_next_convert_task_name

    my $task_name = get_next_convert_task_name($convert_type, $current_state);

=cut

sub get_next_convert_task_name {
    my ($convert_type, $current_state) = @_;

    return undef unless $convert_type && $current_state;

    if ($convert_type eq 'COPY') {
        return {
            NEW => 'notify_balance',
            BALANCE_NOTIFIED => 'copy_convert_bids_and_stop_campaigns',
            WAITING_TO_STOP => 'check_old_campaigns_stopped',
            STOPPED => 'transfer_money_and_archive_old_campaigns',
            FETCHING_BALANCE_DATA => 'fetch_balance_data',
            OVERDRAFT_WAITING => 'wait_for_overdraft',
            NOTIFY => 'notify_convert_done',
        }->{$current_state};
    } elsif ($convert_type eq 'MODIFY') {
        return {
            NEW => 'notify_balance',
            BALANCE_NOTIFIED => 'convert_client_inplace',
            CONVERTING_DETAILED_STAT => 'convert_detailed_stat',
            FETCHING_BALANCE_DATA => 'fetch_balance_data',
            OVERDRAFT_WAITING => 'wait_for_overdraft',
            NOTIFY => 'notify_convert_done',
        }->{$current_state};
    } else {
        return undef;
    }
}

=head2 get_clients_convert_duration_forecast

    Определяет примерное время в секундах, которое потребуется на конвертацию клиентов.
    Однажды посчитанное значение кешируется.

    my $chief_uid2duration = get_clients_convert_duration_forecast(\@client_chief_uids, $convert_type);
    $chief_uid2duration => {
        $client_chief_uid1 => $duration1,
        ...
    };

=cut

my $DURATION_FORECAST_CACHE_TTL = 30*60;

sub _get_duration_cache_name {
    my ($client_chief_uid, $convert_type) = @_;

    return "client_convert_duration_forecast/$convert_type/$client_chief_uid";
}

sub get_clients_convert_duration_forecast {
    my ($client_chief_uids, $convert_type) = @_;

    my (@uids_to_calculate, %uid2duration);
    my $mc = Yandex::Memcached->new(servers => $Settings::MEMCACHED_SERVERS);

    for my $client_chief_uid (@$client_chief_uids) {
        my $key = _get_duration_cache_name($client_chief_uid, $convert_type);
        my $forecast_duration = $mc->get($key);
        if ($forecast_duration) {
            $uid2duration{$client_chief_uid} = $forecast_duration;
        } else {
            push @uids_to_calculate, $client_chief_uid;
        }
    }

    if (@uids_to_calculate) {
        if ($convert_type eq 'COPY') {
            my $all_stat = get_all_sql(PPC(uid => \@uids_to_calculate), ['
                SELECT c.uid
                     , COUNT(DISTINCT c.cid) AS campaigns_cnt
                     , COUNT(DISTINCT p.pid) AS phrases_cnt
                     , COUNT(DISTINCT b.bid) AS banners_cnt
                     , COUNT(DISTINCT bi.id) AS bids_cnt
                     , COUNT(DISTINCT bi_arc.id) AS bids_arc_cnt
                     , COUNT(DISTINCT bi_ret.ret_id) AS bids_ret_cnt
                     , COUNT(DISTINCT bi_dyn.dyn_id) AS bids_dyn_cnt
                FROM campaigns c
                LEFT JOIN phrases p ON c.cid = p.cid
                LEFT JOIN banners b ON p.pid = b.pid
                LEFT JOIN bids bi ON p.pid = bi.pid
                LEFT JOIN bids_arc bi_arc ON c.cid = bi_arc.cid AND p.pid = bi_arc.pid
                LEFT JOIN bids_retargeting bi_ret ON p.pid = bi_ret.pid
                LEFT JOIN bids_dynamic bi_dyn ON (p.adgroup_type = "dynamic" AND p.pid = bi_dyn.pid)
             ', WHERE => {
                 'c.uid' => SHARD_IDS,
                 'c.type' => get_camp_kind_types('with_currency'),
                 'c.statusEmpty' => "No",
                 _TEXT => 'IFNULL(c.currency, "YND_FIXED") = "YND_FIXED"',
            }, 'GROUP BY c.uid']) || {};

            for my $stat (@$all_stat) {
                my $forecast_duration = 15*60
                    + 0.6*($stat->{campaigns_cnt} || 0)
                    + 0.05*(($stat->{phrases_cnt} || 0) + ($stat->{banners_cnt} || 0))
                    + 0.01*(($stat->{bids_cnt} || 0) + ($stat->{bids_arc_cnt} || 0) + ($stat->{bids_ret_cnt} || 0) + ($stat->{bids_dyn_cnt} || 0));
                my $key = _get_duration_cache_name($stat->{uid}, $convert_type);
                $mc->set($key, $forecast_duration, $DURATION_FORECAST_CACHE_TTL);
                $uid2duration{$stat->{uid}} = $forecast_duration;
            }
        } elsif ($convert_type eq 'MODIFY') {
            for my $uid (@uids_to_calculate) {
                $uid2duration{$uid} = 2*60*60;
            }
        } else {
            die "invalid convert type $convert_type";
        }
    }

    return \%uid2duration;
}

=head2 get_client_convert_duration_forecast

    Определяет примерное время в секундах, которое потребуется на конвертацию клиента.
    Однажды посчитанное значение кешируется.

    my $duration = get_client_convert_duration_forecast($client_chief_uid, $convert_type);

=cut

sub get_client_convert_duration_forecast {
    my ($client_chief_uid, $convert_type) = @_;

    return get_clients_convert_duration_forecast([$client_chief_uid], $convert_type)->{$client_chief_uid};
}

=head2 mass_enable_currency_convert_teaser

    Включает тизер конвертации указанному списку клиентов

    mass_enable_currency_convert_teaser(\@ClientIDs);
    mass_enable_currency_convert_teaser(\@ClientIDs, modify_convert_allowed => 1);

=cut

sub mass_enable_currency_convert_teaser {
    my ($clientids, %O) = @_;

    foreach_shard ClientID => $clientids, sub {
        my ($shard, $clientids) = @_;
        do_mass_insert_sql(PPC(shard => $shard),
            'INSERT INTO clients_to_force_multicurrency_teaser (ClientID, modify_convert_allowed) VALUES %s ON DUPLICATE KEY UPDATE modify_convert_allowed = GREATEST(modify_convert_allowed, VALUES(modify_convert_allowed))',
            [map {[$_, ($O{modify_convert_allowed}) ? 1 : 0]} @$clientids],
        );
    };
}

my $FORCE_CONVERT_DAILY_CLIENT_LIMIT_DEFAULT = 0;
my $FORCE_CONVERT_DAILY_CLIENT_LIMIT_PROP_NAME = 'FORCE_CONVERT_DAILY_CLIENT_LIMIT';

=head2 get_force_convert_daily_client_limit

    Возвращает количество клиентов, которых надо ставить на принудительную конвертацию ежедневно
    [писать письмо и назначать дату]

    $limit_cnt = Client::ConvertToRealMoney::get_force_convert_daily_client_limit();

=cut

sub get_force_convert_daily_client_limit {
    return Property->new($FORCE_CONVERT_DAILY_CLIENT_LIMIT_PROP_NAME)->get() // $FORCE_CONVERT_DAILY_CLIENT_LIMIT_DEFAULT;
}

=head2 set_force_convert_daily_client_limit

    Возвращает количество клиентов, которых надо ставить на принудительную конвертацию ежедневно
    [писать письмо и назначать дату]

    $limit_cnt = Client::ConvertToRealMoney::set_force_convert_daily_client_limit();

=cut

sub set_force_convert_daily_client_limit {
    my ($limit) = @_;

    die 'no limit given' unless defined $limit;

    Property->new($FORCE_CONVERT_DAILY_CLIENT_LIMIT_PROP_NAME)->set($limit);
}

my $FORCE_CONVERT_DAILY_QUEUE_CLIENT_LIMIT_DEFAULT = 0;
my $FORCE_CONVERT_DAILY_QUEUE_CLIENT_LIMIT_PROP_NAME = 'FORCE_CONVERT_DAILY_QUEUE_CLIENT_LIMIT';

=head2 get_force_convert_daily_queue_client_limit

    Возвращает количество клиентов, которых надо ставить на принудительную конвертацию ежедневно
    [добавлять в очередь конвертации]

    $limit_cnt = Client::ConvertToRealMoney::get_force_convert_daily_queue_client_limit();

=cut

sub get_force_convert_daily_queue_client_limit {
    return Property->new($FORCE_CONVERT_DAILY_QUEUE_CLIENT_LIMIT_PROP_NAME)->get() // $FORCE_CONVERT_DAILY_QUEUE_CLIENT_LIMIT_DEFAULT;
}

=head2 set_force_convert_daily_queue_client_limit

    Возвращает количество клиентов, которых надо ставить на принудительную конвертацию ежедневно
    [добавлять в очередь конвертации]

    $limit_cnt = Client::ConvertToRealMoney::set_force_convert_daily_queue_client_limit();

=cut

sub set_force_convert_daily_queue_client_limit {
    my ($limit) = @_;

    die 'no limit given' unless defined $limit;

    Property->new($FORCE_CONVERT_DAILY_QUEUE_CLIENT_LIMIT_PROP_NAME)->set($limit);
}

=head2 get_convert_type

    Определяет какой у клиента будет тип конвертации в валюту

    $convert_type = get_convert_type($client_id, $currency, $client_nds, \@country_currencies);
    $convert_type => 'COPY'|'MODIFY'

=cut

sub get_convert_type {
    my ($client_id, $currency, $client_nds, $country_currencies) = @_;

    # конвертировать кампании без остановки и копирования будем только рублёвых клиентов, которые платят НДС 18% и которым доступно только одно сочетание страны-валюты
    # т.к. конвертация статистики завязана на стоимость фишки 1 у.е. = 30 рублей с НДС 18%
    # и у которых нет общего счета - https://st.yandex-team.ru/DIRECT-70897#1507297674000
    my $modify_convert_allowed = get_one_field_sql(PPC(ClientID => $client_id),
            ['SELECT modify_convert_allowed FROM clients_to_force_multicurrency_teaser', WHERE => {ClientID => SHARD_IDS}]
        );
    if ($currency eq 'RUB' && $client_nds && $client_nds == 100*$Settings::NDS_RU_DEPRECATED && scalar(@$country_currencies) == 1) {
        if ($modify_convert_allowed) {
            return 'MODIFY';
        } else {
            return 'COPY';
        }
    } else {
        return 'COPY';
    }
}

=head2 get_closest_modify_convert_start_ts

    $convert_start_ts = Client::ConvertToRealMoney::get_closest_modify_convert_start_ts();

    $now_ts = time();
    $convert_start_ts = Client::ConvertToRealMoney::get_closest_modify_convert_start_ts($now_ts);

=cut

sub get_closest_modify_convert_start_ts {
    my ($now_ts) = @_;

    $now_ts //= time();

    # выставляем дату и время ближайшей полуночи по Москве, если до неё осталось больше 4 часов
    # иначе -- выставляем дату следующей полуночи
    # фактически, код работает по локальной таймзоне для сервера, но сейчас полагаемся на то, что это всегда Москва
    my $today_date = ts_to_str($now_ts, 'day');
    my $tomorrow_date = tomorrow($today_date);
    my $closest_midnight_ts = mysql2unix($tomorrow_date);
    if ($closest_midnight_ts - $now_ts >= $SECONDS_BEFORE_MIDNIGHT_FOR_MODIFY_CONVERT) {
        return $closest_midnight_ts;
    } else {
        return mysql2unix(tomorrow($tomorrow_date));
    }
}

=head2 mass_queue_currency_convert(\@requests, %O)

    @requests = (
        {
            ClientID => $client_id,
            uid => $requester_uid,
            convert_type => $convert_type,
            new_currency => $new_currency,
            country_region_id => $country_region_id,
            email => $notify_email,
            start_convert_ts => $convert_ts,
        },
        ...
    );
    %O:
        ignore => 1/0, делать ли INSERT IGNORE или падать на уже существующих записях
        AgencyID => ClientID агентства, которое ставит на конвертацию и которому
                    написать письмо по завершению конвертации всех поставленных клиентов
                    (см. таблицу subclient_convert_requests)

    Client::ConvertToRealMoney::mass_queue_currency_convert(\@requests);
    Client::ConvertToRealMoney::mass_queue_currency_convert(\@requests, ignore => 1);
    Client::ConvertToRealMoney::mass_queue_currency_convert(\@requests, AgencyID => $agency_id);

=cut

sub mass_queue_currency_convert {
    my ($requests, %O) = @_;

    foreach_shard ClientID => $requests, chunk_size => 100, sub {
        my ($shard, $requests_chunk) = @_;

        my $ignore_sql_str = ($O{ignore}) ? 'IGNORE' : '';
        do_in_transaction {
            do_mass_insert_sql(PPC(shard => $shard),
                "INSERT $ignore_sql_str INTO currency_convert_queue (ClientID, uid, convert_type, new_currency, country_region_id, email, start_convert_at, state, in_state_since) VALUES %s",
                [ map { [
                    (map { sql_quote($_) } (
                        $_->{ClientID},
                        $_->{uid},
                        $_->{convert_type},
                        $_->{new_currency},
                        $_->{country_region_id},
                        $_->{email},
                        unix2mysql($_->{start_convert_ts}),
                        'NEW',
                    )),
                    'NOW()'
                ] } @$requests_chunk ],
                { dont_quote => 1 },
            );
            do_mass_insert_sql(PPC(shard => $shard),
                "INSERT $ignore_sql_str INTO client_currency_changes (ClientID, currency_from, currency_to, date) VALUES %s",
                [ map { [
                    $_->{ClientID},
                    'YND_FIXED',
                    $_->{new_currency},
                    unix2mysql(ts_round_day($_->{start_convert_ts})),
                ] } @$requests_chunk ],
            );
            if ($O{AgencyID} && $O{AgencyID} > 0) {
                do_mass_insert_sql(PPC(ClientID => $O{AgencyID}),
                    "INSERT $ignore_sql_str INTO subclient_convert_requests (AgencyID, ClientID) VALUES %s",
                    [ map { [
                        $O{AgencyID},
                        $_->{ClientID},
                    ] } @$requests_chunk ],
                );
            }
        };
    };
}

1;
