#!/usr/bin/perl

=head1 NAME

Client::ConvertToRealMoneyTasks

=head1 DESCRIPTION

    Модуль, реализующий задания для перевода клиентов в настоящие валюты из у.е.
    Самостоятельно управляет очередью и отмечает в ней завершение соответствующей стадии

    Перед использованием нужно определить $log:
    
    $Client::ConvertToRealMoneyTasks::log = Yandex::Log->new( ... );

=cut

package Client::ConvertToRealMoneyTasks;

use Direct::Modern;

use List::Util qw(max);

use Settings;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::Trace;
use Yandex::HashUtils;
use Yandex::TimeCommon;
use Yandex::Balance qw(balance_get_completion_history balance_get_orders_info);
use Yandex::ListUtils qw(xsort chunks);

use RBAC2::Extended;
use RBACElementary;
use RBACDirect;
use Campaign;
use Campaign::Types;
use Client;
use Client::ConvertToRealMoney;
use Common;
use CampAutoPrice::Common;
use Currency::Rate;
use MoneyTransfer;
use PrimitivesIds;
use Notification;
use DBLog;
use Wallet;

our $OPERATOR_UID = 1;

sub _get_rbac_object {
    my $rbac = RBAC2::Extended->get_singleton($OPERATOR_UID);
    return $rbac;
}

=head2 $CONVERT_TO_REAL_MONEY_TIMEOUT

    Таймат на конвертирование клиента в реальную валюту (в секундах)

=cut

our $CONVERT_TO_REAL_MONEY_TIMEOUT = 10 * 60 * 60; # 10 часов

=head2 $TIME_TO_WAIT_FOR_CAMPAIGN_STOP

    Сколько минут ждать остановки кампании при конвертации копированием.
    Если за это время кампания не остановится в БК, то всё-равно уносим
    с кампании все деньги.

=cut

our $TIME_TO_WAIT_FOR_CAMPAIGN_STOP = 40;

=head2 $TIME_TO_WAIT_FOR_OVERDRAFT

    Сколько минут ждать нотификации от Баланса с пересчитанной в валюту суммой овердрафта
    Если за это время нотификация не придёт, продолжим конвертировать клиента с нулевым овердрафтом.

=cut

our $TIME_TO_WAIT_FOR_OVERDRAFT = 40;


our %ACTION_MAP = (
    copy_convert_bids_and_stop_campaigns => \&copy_convert_bids_and_stop_campaigns,
    check_old_campaigns_stopped => \&check_old_campaigns_stopped,
    transfer_money_and_archive_old_campaigns => \&transfer_money_and_archive_old_campaigns,
    notify_balance => \&notify_balance,
    convert_client_inplace => \&convert_client_inplace,
    notify_convert_done => \&notify_convert_done,
    convert_detailed_stat => \&convert_detailed_stat,
    fetch_balance_data => \&fetch_balance_data,
    wait_for_overdraft => \&wait_for_overdraft,
);

our $log;

=head2 copy_convert_bids_and_stop_campaigns

    Первый шаг перевода клиента в реальную валюту копированием.
    С точки зрения очереди это переход из состояния BALANCE_NOTIFIED в WAITING_TO_STOP:
        NEW -> копирование кампаний -> конвертация ставок и параметров кампании -> остановка кампаний в у.е. -> WAITING_TO_STOP

=cut

sub copy_convert_bids_and_stop_campaigns {
    my ($data) = @_;

    my $client_id = $data->{ClientID};
    my $currency = $data->{new_currency};

    my $client_msg_prefix = "[copy_convert_bids_and_stop_campaigns] [ClientID $client_id]";
    my $prefix_guard = $log->msg_prefix_guard($client_msg_prefix);

    my $profile = Yandex::Trace::new_profile('ConvertToRealMoney:copy_convert_bids_and_stop_campaigns', tags => 'main');
    $log->out('start');

    my $success = eval {
        my $sql_lock = Client::get_per_client_convert_lock_guard($client_id);

        # проверяем наличие в очереди задачи в правильном состоянии
        my %queue_cond = (ClientID => $client_id, state => 'BALANCE_NOTIFIED', convert_type => 'COPY', balance_convert_finished => 1);
        my ($is_task_correct, $cur_convert_started_at) = get_one_line_array_sql(PPC(ClientID => $client_id), ['SELECT 1, convert_started_at FROM currency_convert_queue', WHERE => \%queue_cond]);
        die "Incorrect task state in queue" unless $is_task_correct;

        # отмечаем время начала перехода только если его ещё нет
        if (!$cur_convert_started_at || !check_mysql_date($cur_convert_started_at)) {
            do_update_table(PPC(ClientID => $client_id), 'currency_convert_queue', {convert_started_at__dont_quote => 'CURRENT_TIMESTAMP'}, where => {%queue_cond, convert_started_at => $cur_convert_started_at});
        }

        $log->out('Start converting step 1');

        my $rbac = _get_rbac_object();
        my $client_chief_uid = rbac_get_chief_rep_of_client($client_id);

        # архивируем пустые кампании
        # после конвертации в них ничего добавить всё-равно нельзя будет (старая валюта), а
        # если бы скопировали, то номер всё-равно поменялся бы и всё-равно в них ничего не добавить
        $log->out('Archiving empty campaigns');
        my $empty_cids = get_one_column_sql(PPC(ClientID => $client_id), ['SELECT cid FROM campaigns', WHERE => {uid => $client_chief_uid, statusEmpty => 'Yes'}]);
        $log->out("Archiving empty cids:", $empty_cids);
        my $arc_res = Wallet::arc_camp($rbac, $OPERATOR_UID, $client_chief_uid, $empty_cids, archive_non_stopped => 1, force => 1);
        $log->out('Archiving result:', $arc_res);

        my $cond = {'c.uid' => $client_chief_uid,
                    'c.type' => get_camp_kind_types_and(qw/with_currency copyable/),
                    'c.currencyConverted' => "No",
                    _TEXT => 'IFNULL(c.currency, "YND_FIXED") = "YND_FIXED"',
        };
        # не передаём в get_user_camps_by_sql ключи client_nds/client_discount, т.к. до конвертации мультивалютных кампаний не может быть
        # по возрастанию номеров кампании сортируем специально, чтобы совпадал порядок кампаний на странице с их списком до и после конвертации (там сортируем по cid'у)
        my $campaigns_data = Common::get_user_camps_by_sql($cond, {shard => {ClientID => $client_id}});
        my $campaigns = $campaigns_data->{campaigns};
        $log->out('Got ' . scalar(@$campaigns) . ' campaigns from DB to convert');
        my $profile = Yandex::Trace::new_profile('ConvertToRealMoney:copy_convert_bids_and_stop_campaigns', obj_num => scalar(@$campaigns));

        my @cids = map {$_->{cid}} @$campaigns;
        clear_auto_price_queue(\@cids, error_str => 'currency convert');

        my $rate = convert_currency(1, 'YND_FIXED', $currency);

        $log->out('Begin copying old campaigns');
        my (%oldcid2newcid, %oldcid2data);
        for my $campaign (xsort {(($_->{mediaType} eq 'wallet') ? 0 : 1 ), int($_->{cid})} @$campaigns) {
            my $cid = $campaign->{cid};
            my $for_msg_prefix_guard = $log->msg_prefix_guard("$client_msg_prefix [cid $cid]");

            # копируем и конвертируем параметры
            $log->out("Starting copy");
            my $new_wallet_cid = 0;
            if ($campaign->{wallet_is_enabled}) {
                $new_wallet_cid = $oldcid2newcid{$campaign->{wallet_cid}};
                $log->die("No copied wallet campaign found wallet_cid $campaign->{wallet_cid}") unless $new_wallet_cid;
                $log->out("Campaign will be copied with new wallet_cid $new_wallet_cid");
            }
            my $converted_campaign_params = Client::ConvertToRealMoney::get_converted_campaign_params($campaign, $currency, $rate, copy_last_bill_sums => 1, save_conversion_strategy => 1);
            my $copied_cid = Client::ConvertToRealMoney::copy_old_campaign($rbac, $campaign, $converted_campaign_params, $new_wallet_cid, price_convert_rate => $rate, new_currency => $currency, UID => $OPERATOR_UID);
            $log->die('bad new cid after copying') unless $copied_cid;
            $log->out("Copied into campaign $copied_cid");
            # copy_old_campaign копирует кампанию с statusEmpty = 'Yes', поэтому даже если мы упадём в процесс конвертации,
            # скопированная кампания не будет ничему мешать, а со временем её удалит скрипт ppcClearEmptyCampaigns.pl
            # при следующей же конвертации после падения будет создана новая копия кампании

            $oldcid2newcid{$cid} = $copied_cid;
            $oldcid2data{$cid} = $campaign;
        }

        $log->out('Done copying campaigns');

        if (%oldcid2newcid) {
            # конвертируем ставки в скопированных кампаниях
            my @cids = values %oldcid2newcid;

            $log->out('Converting secondary options in campaigns:', \@cids);
            Client::ConvertToRealMoney::convert_campaign_secondary_options(\@cids, 'YND_FIXED', $currency);

            # ставим скопированным кампаниям statusEmpty = 'No', а исходным ставим currencyConverted = "Yes" и останавливаем их
            my @new_cids = values %oldcid2newcid;
            my @old_cids = keys %oldcid2newcid;
            my @campaign_correspondence_data;
            while (my ($old_cid, $new_cid) = each %oldcid2newcid) {
                my $old_camp = $oldcid2data{$old_cid};
                my $was_archived = ($old_camp->{archived} && $old_camp->{archived} eq 'Yes') ? 1 : 0;
                push @campaign_correspondence_data, [$client_id, $old_cid, $new_cid, $was_archived];
            }
            $log->out('Removing statusEmpty from campaigns:', \@new_cids);
            $log->out('Marking converting done campaigns:', \@old_cids);
            do_in_transaction {
                do_update_table(PPC(ClientID => $client_id), 'campaigns', {statusEmpty => 'No', statusBsSynced => 'No'}, where => {cid => \@new_cids});
                do_update_table(PPC(ClientID => $client_id), 'campaigns', {currencyConverted => 'Yes', statusShow__dont_quote => 'IF(type != "wallet", "No", statusShow)', statusBsSynced => 'No'}, where => {cid => \@old_cids});
                do_update_table(PPC(ClientID => $client_id), 'camp_options', {stopTime__dont_quote => 'CURRENT_TIMESTAMP'}, where => {cid => \@old_cids});
                do_mass_insert_sql(PPC(ClientID => $client_id), 'INSERT INTO currency_convert_money_correspondence (ClientID, old_cid, new_cid, was_archived) VALUES %s', \@campaign_correspondence_data);
            };
        }

        $log->out('Fixating converting step 1 result');
        # обновляем задание в очереди вне транзакции потому, что в этом шаге смотрим только на кампании с currencyConverted
        # а currencyConverted выставляем в транзакции. т.е. при падении до этого апдейта, повторное выполнение шага не вызовет
        # проблем и единственное что сделаем -- обновим currency_convert_queue
        do_update_table(PPC(ClientID => $client_id), 'currency_convert_queue', {state => 'WAITING_TO_STOP', in_state_since__dont_quote => 'NOW()'}, where => \%queue_cond);
        $log->out("Done converting step 1");

        return 1;
    };
    if ($success) {
        $log->out('finish');
    } else {
        $log->die("Error in copy_convert_bids_and_stop_campaigns:", $@);
    }
}

=head2 check_old_campaigns_stopped

    Второй шаг перевода клиента в реальную валюту копированием.
    С точки зрения очереди это переход из состояния WAITING_TO_STOP в STOPPED
        WAITING_TO_STOP -> проверка что все кампании остановились -> STOPPED
    Воркер только проверяет остановлены ли все кампании в у.е. у клиента.
    Если остановлены -- переводит задачу в состояние STOPPED.
    Если не остановлены -- умирает. В этом случае ppcCurrencyConvertMaster через
    некоторое время перепоставит задачу в очередь и проверка повторится.

=cut

sub check_old_campaigns_stopped {
    my ($data) = @_;

    my $client_id = $data->{ClientID};

    my $client_msg_prefix = "[check_old_campaigns_stopped] [ClientID $client_id]";
    my $prefix_guard = $log->msg_prefix_guard($client_msg_prefix);

    my $profile = Yandex::Trace::new_profile('ConvertToRealMoney:check_old_campaigns_stopped');
    $log->out('start');

    my $success = eval {
        my $sql_lock = Client::get_per_client_convert_lock_guard($client_id);

        # проверяем наличие в очереди задачи в правильном состоянии
        my %queue_cond = (ClientID => $client_id, state => 'WAITING_TO_STOP', convert_type => 'COPY');
        my ($is_task_correct) = get_one_line_array_sql(PPC(ClientID => $client_id), ['SELECT 1 FROM currency_convert_queue', WHERE => \%queue_cond]);
        die("Incorrect task state in queue") unless $is_task_correct;

        my $rbac = _get_rbac_object();
        my $client_chief_uid = rbac_get_chief_rep_of_client($client_id);
        my ($running_campaign_cid) = get_one_line_array_sql(PPC(ClientID => $client_id), ['
            SELECT c.cid
            FROM campaigns c
            LEFT JOIN camp_options co ON c.cid = co.cid
            WHERE
                IFNULL(c.currency, "YND_FIXED") = "YND_FIXED"
                AND ', { 'c.type' => get_camp_kind_types_and(qw/web_edit_base with_currency copyable/) }, '
                AND c.statusEmpty = "No"
                AND c.archived = "No"
                AND c.statusActive = "Yes"
                AND (co.stopTime IS NULL OR co.stopTime > NOW() - INTERVAL ? MINUTE)
                AND OrderID > 0
                AND', {uid => $client_chief_uid}, '
            LIMIT 1'], $TIME_TO_WAIT_FOR_CAMPAIGN_STOP);
        if ($running_campaign_cid) {
            $log->out("Found running campaign: $running_campaign_cid");
            return 0;
        } else {
            $log->out('No running campaigns found. Fixating converting step 2 result');
            do_update_table(PPC(ClientID => $client_id), 'currency_convert_queue', {state => 'STOPPED', in_state_since__dont_quote => 'NOW()'}, where => \%queue_cond);
            $log->out("Done waiting for campaigns to stop");
        }

        return 1;
    };

    if (defined $success) {
        if ($success) {
            $log->out('finish');
        } else {
            # блок eval не упал, но и успешно не закончился
            # значит, сейчас он успешно завершиться не может (например, кампании ещё не остановились)
            $log->out('task not completed, finishing');
        }
    } else {
        $log->die("Error:", $@);
    }
}

=head2 transfer_money_and_archive_old_campaigns

    Третий и последний шаг перевода клиента в реальную валюту копированием.
    С точки зрения очереди это переход из состояния STOPPED в FETCHING_BALANCE_DATA
        STOPPED -> смена валюты и страны клиенту в Балансе -> создание новых кампаний в Балансе -> перенос денег со старых кампаний на новые -> FETCHING_BALANCE_DATA

=cut

sub transfer_money_and_archive_old_campaigns {
    my ($data) = @_;

    my $client_id = $data->{ClientID};
    my $currency = $data->{new_currency};

    my $client_msg_prefix = "[transfer_money_and_archive_old_campaigns] [ClientID $client_id]";
    my $prefix_guard = $log->msg_prefix_guard($client_msg_prefix);

    my $profile = Yandex::Trace::new_profile('ConvertToRealMoney:transfer_money_and_archive_old_campaigns');
    $log->out('start');

    my $success = eval {
        my $sql_lock = Client::get_per_client_convert_lock_guard($client_id);

        # проверяем наличие в очереди задачи в правильном состоянии
        my %queue_cond = (ClientID => $client_id, state => 'STOPPED', convert_type => 'COPY');
        my ($is_task_correct) = get_one_line_array_sql(PPC(ClientID => $client_id), ['SELECT 1 FROM currency_convert_queue', WHERE => \%queue_cond]);
        die("Incorrect task state in queue") unless $is_task_correct;

        my $oldcid2newciddata = get_hashes_hash_sql(PPC(ClientID => $client_id), [q(
            SELECT ccmc.old_cid
                 , ccmc.new_cid
                 , ccmc.money_before
                 , ccmc.money_after
                 , cold.sum AS old_sum
                 , cnew.sum AS new_sum
                 , cold.sum - cold.sum_spent AS old_total
                 , cold.type AS type
                 , cold.sum_to_pay AS old_sum_to_pay
            FROM currency_convert_money_correspondence ccmc
            INNER JOIN campaigns cold ON ccmc.old_cid = cold.cid
            INNER JOIN campaigns cnew ON ccmc.new_cid = cnew.cid
         ), WHERE => {
                'ccmc.ClientID' => SHARD_IDS,
         }]);

        if ($oldcid2newciddata && %$oldcid2newciddata) {
            my $rbac = _get_rbac_object();
            my $client_chief_uid = rbac_get_chief_rep_of_client($client_id);

            my @cids_to_transfer = grep {$_} map { $_->{cid } } values %$oldcid2newciddata;
            my $cid2scampaign = (@cids_to_transfer) ? rbac_mass_is_scampaign($rbac, \@cids_to_transfer) : {};

            # отправляем в Баланс новые и старые кампании в единой группе (как в общем счёте)
            # нужно, чтобы при зачислениях/откатах по старой (уешной) кампании деньги приходили на новую (валютную кампанию)
            # сначала отправляем новые, т.к. в едином вызове создание заказа и включение существующего к нему в группу Баланс не принимает
            # автоматический перенос денег на общий счёт отменяем, т.к. хотим знать точную сумму, в которую превратились перенесённые фишки
            my @new_cids = map { $_->{new_cid} } xsort {( ($_->{type} eq 'wallet' ? 1 : 2), int($_->{new_cid}) )} values %$oldcid2newciddata;
            for my $new_cids_chunk (chunks \@new_cids, 100) {
                $log->out('Creating new campaigns in Balance:', $new_cids_chunk);
                my $new_create_res = create_campaigns_balance($rbac, $OPERATOR_UID, $new_cids_chunk, dont_create_wallets => 1);
                if (!$new_create_res || !$new_create_res->{balance_res}) {
                    $log->die("Error creating new campaigns in Balance:", $new_create_res);
                }
            }
            my @old_cids = map { $_->{old_cid} } xsort {( ($_->{type} eq 'wallet' ? 1 : 2), int($_->{old_cid}) )} values %$oldcid2newciddata;
            for my $old_cids_chunk (chunks \@old_cids, 100) {
                $log->out('Creating old campaigns in Balance:', $old_cids_chunk);
                my $old_create_res = create_campaigns_balance($rbac, $OPERATOR_UID, $old_cids_chunk, dont_create_wallets => 1);
                if (!$old_create_res || !$old_create_res->{balance_res}) {
                    $log->die("Error creating old campaigns in Balance:", $old_create_res);
                }
            }

            while (my ($old_cid, $new_cid_data) = each %$oldcid2newciddata) {
                my $new_cid = $new_cid_data->{new_cid};

                # переносим деньги (если потенциально есть что переносить)
                # кошельки пытаемся переносить всегда, т.к. для них в Балансе есть асинхронный фоновый перенос
                # если кошелёк подключили прямо сразу перед конвертацией, то о его деньгах мы можем и не успеть узнать до начала конвертации
                if (($new_cid_data->{old_sum} || 0) > 0 || ($new_cid_data->{old_sum_to_pay} || 0) > 0 || ($new_cid_data->{type} || '') eq 'wallet') {
                    # переносим деньги отдельным запросом для каждой кампании, т.к. важно перенести деньги между кампаниями
                    # ровно 1-к-1. массово переносить нельзя, т.к. если на старую кампанию зачислятся новые деньги, то
                    # при массовом переносе снимем старую (известную нам) сумму, и раскидаем её пропорционально по всем
                    # новым кампаниям
                    my $old_sum_our_db = $new_cid_data->{old_sum};

                    # записываем сумму, которая была на старой кампании
                    # если упадём после переноса денег, но до записи его результатов, эта сумма — единственное, что у нас останется
                    do_update_table(PPC(ClientID => $client_id), 'currency_convert_money_correspondence', {
                        money_before => $new_cid_data->{old_total},
                    }, where => {ClientID => SHARD_IDS, old_cid => $old_cid, new_cid => $new_cid});

                    my %transfer_params = (
                        campaigns_from => [{
                            cid => $old_cid,
                            # переносим все деньги, поэтому конкретные суммы здесь не важны
                            sum => $old_sum_our_db,
                            sum_move => $new_cid_data->{old_total},
                            NDS => 0,   # внутри клиента НДС одинаковый, не хочется портить суммы снятием/начислением НДС
                            currency => 'YND_FIXED',
                        }],
                        campaigns_to => [{
                            cid => $new_cid,
                            ClientID => $client_id,
                            # performance: хорошо бы иметь массовый аналог rbac_is_scampaign и использовать его в этом месте
                            serviced => $cid2scampaign->{$new_cid},
                            # переносим все деньги, поэтому конкретные суммы здесь не важны
                            sum_get => 1,
                            NDS => 0,
                            currency => $currency,
                        }],
                    );

                    $log->out("Transferring money for campaigns $old_cid => $new_cid:", \%transfer_params);
                    my $transfer_results = process_transfer_money($rbac, $data->{uid}, $OPERATOR_UID, \%transfer_params, detailed_response => 1, move_all_qty => 1, dont_create_campaigns => 1);
                    $log->die('Invalid transfer results: ', $transfer_results) unless $transfer_results && ref($transfer_results) eq 'ARRAY';

                    my ($new_sum, $old_sum);
                    if (@$transfer_results) {
                        $log->out("Transfer results for campaigns $old_cid => $new_cid:", $transfer_results);
                        for my $result (@$transfer_results) {
                            my $cid = $result->{ServiceOrderID};
                            my $sum = abs($result->{Qty});
                            if ($cid == $new_cid) {
                                $new_sum = $sum;
                            } elsif ($cid == $old_cid) {
                                $old_sum = $sum;
                            } else {
                                $log->die("Got strange result from money transfering:", $result);
                            }
                        }
                    } else {
                        # отсутствие результатов переноса при успешном запросе означает, что переносить было нечего
                        # такое возможно, если мы расходимся по суммам с Балансом (у нас деньги ещё есть, а у них уже нет)
                        $log->out('Empty transfer results, assuming there was nothing to transfer');
                        $old_sum = 0;
                        $new_sum = 0;
                    }

                    # нотификации во время конвертации не принимаем, поэтому сбрасываем деньги руками
                    # все новые приходящие деньги отправятся уже на новую кампанию засчёт связки аля общий счёт в Балансе
                    $log->out("Resetting money on old campaign $old_cid");
                    do_sql(PPC(ClientID => $client_id), 'UPDATE campaigns SET sum = sum_spent, balance_tid = 0, statusBsSynced="No", LastChange=LastChange WHERE cid = ?', $old_cid);

                    # записываем суммы до конвертации и после, чтобы показывать их на странице успеха
                    $log->out("Recording money correspondence for campaigns $old_cid ($old_sum) => $new_cid ($new_sum)");
                    do_sql(PPC(ClientID => $client_id), [q(
                        UPDATE currency_convert_money_correspondence
                        SET money_after = IFNULL(money_after,0) + IFNULL(?,0)
                     ), WHERE => {ClientID => SHARD_IDS, old_cid => $old_cid, new_cid => $new_cid}], $new_sum);
                }
                if ($new_cid_data->{new_sum} && $new_cid_data->{new_sum} > 0 && !$new_cid_data->{money_after}) {
                    # перенос денег не записали, а на новой кампании деньги есть
                    # это значит, что упали между переносом и записью в currency_convert_money_correspondence
                    # дозаписываем туда текущие данные, раз уж протеряли в сам момент перехода
                    # к сожалению, сколько денег было до переноса уже не определить и там будет ноль
                    my $new_sum = $new_cid_data->{new_sum};
                    $log->out("Adding lost data for tranfer $old_cid => $new_cid ($new_sum) to currency_convert_money_correspondence");
                    do_update_table(PPC(ClientID => $client_id), 'currency_convert_money_correspondence', {
                        money_after => $new_sum,
                    }, where => {ClientID => SHARD_IDS, old_cid => $old_cid, new_cid => $new_cid});
                }
            }

            while (my ($old_cid, $new_cid_data) = each %$oldcid2newciddata) {
                my $new_cid = $new_cid_data->{new_cid};

                if ($new_cid_data->{type} eq 'wallet') {
                    $log->out("Adding new wallet $new_cid");
                    do_insert_into_table(PPC(ClientID => $client_id), 'wallet_campaigns', {wallet_cid => $new_cid, onoff_date__dont_quote => 'NOW()'});
                } else {
                    # архивируем старые кампании
                    $log->out("Archiving old campaign with cid $old_cid");
                    # archive_non_stopped = 1, т.к. иначе архивация будет пытаться дожидаться часа с последнего показа. пренебрегаем долго едущей статистикой и архивируем сразу: деньги всё-равно уже все унесли с кампании.
                    my ($arc_result, $arc_error) = Common::_arc_camp($client_chief_uid, $old_cid, force => 1, archive_non_stopped => 1, archived_is_error => 0);
                    if ($arc_result) {
                        $log->out("Successfully archived campaign $old_cid");
                    } else {
                        $log->die("ERROR archiving campaign $old_cid:", {old_cid => $old_cid, arc_result => $arc_result, arc_error => $arc_error});
                    }
                }
            }
        } else {
            $log->out('No campaigns to process');
        }

        # multicurrency: monitor.agency_clients_month_stat

        do_in_transaction {
            # все кампании сконвертировали. теперь записываем клиенту новую валюту
            $log->out('fixating new client currency');
            Client::ConvertToRealMoney::fixate_client_currency($client_id, $currency, $data->{country_region_id});

            $log->out('cleanup multicurrency data');
            Client::ConvertToRealMoney::cleanup_data_for_converted_client($client_id);

            $log->out('All done! Fixating converting step 3 result');
            do_update_table(PPC(ClientID => $client_id), 'currency_convert_queue', {state => 'FETCHING_BALANCE_DATA', in_state_since__dont_quote => 'NOW()'}, where => \%queue_cond);
        };

        return 1;
    };

    if ($success) {
        $log->out('finish');
    } else {
        $log->die("Error:", $@);
    }
}

=head2 notify_convert_done

    Отправляем пользователю уведомление об окончании конвертации
    Работает со всеми типами перевода

    Переводит заявку на переход из состояния NOTIFY в DONE.

=cut

sub notify_convert_done {
    my ($data) = @_;

    my $client_id = $data->{ClientID};

    my $client_msg_prefix = "[notify_convert_done] [ClientID $client_id]";
    my $prefix_guard = $log->msg_prefix_guard($client_msg_prefix);

    my $profile = Yandex::Trace::new_profile('ConvertToRealMoney:notify_convert_done');
    $log->out('start');

    my $success = eval {
        my $sql_lock = Client::get_per_client_convert_lock_guard($client_id);

        # проверяем наличие в очереди задачи в правильном состоянии
        my %queue_cond = (ClientID => $client_id, state => 'NOTIFY');
        my ($is_task_correct) = get_one_line_array_sql(PPC(ClientID => $client_id), ['SELECT 1 FROM currency_convert_queue', WHERE => \%queue_cond]);
        die("Incorrect task state in queue") unless $is_task_correct;

        my $vars = hash_cut $data, qw/email ClientID uid convert_type new_currency/;

        my $rbac = _get_rbac_object();
        my $client_chief_uid = rbac_get_chief_rep_of_client($client_id);
        $vars->{client_chief_uid} = $client_chief_uid;
        $vars->{client_chief_login} = get_login(uid => $client_chief_uid);

        # отправляем письмо клиенту и всем его активным менеджерам/агентствам + SMS
        $log->out("Sending notification with vars", $vars);
        add_notification($rbac, 'currency_convert_finished', $vars);

        $log->out('Fixating notification results');
        do_update_table(PPC(ClientID => $client_id), "users", {statusBlocked => "No"}, where => {ClientID => SHARD_IDS});
        do_update_table(PPC(ClientID => $client_id), 'currency_convert_queue', {state => 'DONE', convert_finished_at__dont_quote => 'CURRENT_TIMESTAMP', in_state_since__dont_quote => 'NOW()'}, where => \%queue_cond);

        return 1;
    };

    if ($success) {
        $log->out('finish');
    } else {
        $log->die("Error:", $@);
    }
}


=head2 notify_balance

    Дёргаем ручку в Балансе, сообщая время перевода клиента.

    В целом процедура выглядит так:
        - клиент подтверждает переход
        - мы отправляем в БК и Баланс, что с полуночи такого-то дня клиент будет в валюте X [bsClientData.pl, notify_balance]
        - в полночь блокируем клиенту интерфейс целиком
        - меняем валюты на кампаниях, пересчитываем ставки и параметры кампаний [convert_client_inplace]
        - разблокируем интерфейс за исключением детальной статистики [convert_client_inplace]
        - перезабираем детальную статистику [convert_client_inplace]
        - разблокируем детальную статистику [convert_client_inplace]

    Переводит заявку на переход из состояния NEW в BALANCE_NOTIFIED.

=cut

sub notify_balance {
    my ($data) = @_;

    my $client_id = $data->{ClientID};
    my $currency = $data->{new_currency};

    my $client_msg_prefix = "[notify_balance] [ClientID $client_id]";
    my $prefix_guard = $log->msg_prefix_guard($client_msg_prefix);

    my $profile = Yandex::Trace::new_profile('ConvertToRealMoney:notify_balance');
    $log->out('start');

    my $success = eval {
        my $sql_lock = Client::get_per_client_convert_lock_guard($client_id);

        # проверяем наличие в очереди задачи в правильном состоянии
        my %queue_cond = (ClientID => $client_id, state => 'NEW');
        my ($is_task_correct, $convert_type) = get_one_line_array_sql(PPC(ClientID => $client_id), ['SELECT 1, convert_type FROM currency_convert_queue', WHERE => \%queue_cond]);
        die "Incorrect task state in queue" unless $is_task_correct;

        my $migrate_datetime = _get_migrate_date($data->{start_convert_at}, $data->{convert_type});

        $log->out('Notifying balance');
        my $is_error = update_client_id($OPERATOR_UID, $client_id, {
            REGION_ID => $data->{country_region_id},
            CURRENCY => $currency,
            MIGRATE_TO_CURRENCY => $migrate_datetime,
            CURRENCY_CONVERT_TYPE => $convert_type,
        });
        if ($is_error) {
            $log->die('Error updating client data in Balance');
        }
        $log->out('Done notifying Balance');

        if ($data->{convert_type} eq 'MODIFY') {
            my $rbac = _get_rbac_object();
            my $client_chief_uid = rbac_get_chief_rep_of_client($client_id);
            my $cids = get_one_column_sql(PPC(ClientID => $client_id), ['SELECT DISTINCT cid FROM campaigns', WHERE => {uid => $client_chief_uid, type => get_camp_kind_types('currency_convert')}]) || [];
            if ($cids && @$cids) {
                $log->out('Marking campaigns unsync to BS');
                do_update_table(PPC(ClientID => $client_id), 'campaigns', {statusBsSynced => 'No', LastChange__dont_quote => 'LastChange'}, where => {cid => $cids});
            }
        }

        do_update_table(PPC(ClientID => $client_id), 'currency_convert_queue', {state => 'BALANCE_NOTIFIED', in_state_since__dont_quote => 'NOW()'}, where => \%queue_cond);

        return 1;
    };

    if ($success) {
        $log->out('finish');
    } else {
        $log->die("Error:", $@);
    }
}

=head2 convert_client_inplace

    Переводит клиента в реальную валюту без копирования и остановки его кампаний.

    Переводит заявку на переход из состояния BALANCE_NOTIFIED в CONVERTING_DETAILED_STAT.

=cut

sub convert_client_inplace {
    my ($data) = @_;

    my $client_id = $data->{ClientID};
    my $currency = $data->{new_currency};

    my $client_msg_prefix = "[convert_client_inplace] [ClientID $client_id]";
    my $prefix_guard = $log->msg_prefix_guard($client_msg_prefix);

    my $profile = Yandex::Trace::new_profile('ConvertToRealMoney:convert_client_inplace');
    $log->out('start');

    my $success = eval {
        my $sql_lock = Client::get_per_client_convert_lock_guard($client_id);

        # проверяем наличие в очереди задачи в правильном состоянии
        my %queue_cond = (ClientID => $client_id, state => 'BALANCE_NOTIFIED', convert_type => 'MODIFY', balance_convert_finished => 1);
        my ($is_task_correct, $cur_convert_started_at) = get_one_line_array_sql(PPC(ClientID => $client_id), ['SELECT 1, convert_started_at FROM currency_convert_queue', WHERE => \%queue_cond]);
        die "Incorrect task state in queue" unless $is_task_correct;

        # отмечаем время начала перехода только если его ещё нет
        if (!$cur_convert_started_at || !check_mysql_date($cur_convert_started_at)) {
            do_update_table(PPC(ClientID => $client_id), 'currency_convert_queue', {convert_started_at__dont_quote => 'CURRENT_TIMESTAMP'}, where => {%queue_cond, convert_started_at => $cur_convert_started_at});
        }

        my $rbac = _get_rbac_object();
        my $client_chief_uid = rbac_get_chief_rep_of_client($client_id);
        my $cond = {'c.uid' => $client_chief_uid,
                    'c.type' => get_camp_kind_types('currency_convert'),
                    'c.currencyConverted' => "No",
                    _TEXT => 'IFNULL(c.currency, "YND_FIXED") = "YND_FIXED"',
        };

        my $campaigns_data = Common::get_user_camps_by_sql($cond, {shard => {ClientID => $client_id}, include_empty_campaigns => 1});
        my $campaigns = $campaigns_data->{campaigns};
        $log->out('Got ' . scalar(@$campaigns) . ' campaigns from DB to convert');

        my @cids = map {$_->{cid}} @$campaigns;
        clear_auto_price_queue(\@cids, error_str => 'currency convert');

        my $rate = convert_currency(1, 'YND_FIXED', $currency);
        my $rate_nds = convert_currency(1, 'YND_FIXED', $currency, with_nds => 1);

        my $migrate_datetime = _get_migrate_date($data->{start_convert_at}, $data->{convert_type});
        my $first_date_in_real_currency = mysql_round_day($migrate_datetime);

        $log->out('Fetching chips cost/spent from Balance to update in DB');
        for my $cids_chunk (chunks \@cids, 10) {
            my $cids_chunk_str = join(',', @$cids_chunk);
            $log->out("Fetching chips cost/spent for cids chunk: $cids_chunk_str");
            my $balance_info_chunk = balance_get_orders_info($cids_chunk);
            my @for_insert;
            for my $order (@$balance_info_chunk) {
                $log->out($order);
                my $cid = $order->{ServiceOrderID};
                my $chips_cost = $order->{CompletionFixedMoneyQty} // $order->{CompletionMoneyQty};
                my $chips_spent = $order->{CompletionFixedQty} // $order->{completion_qty} // 0;
                push @for_insert, [$cid, $chips_cost, $chips_spent];
            }
            $log->out("Updating chips cost/spent in DB for cids chunk $cids_chunk_str");
            do_mass_insert_sql(PPC(ClientID => $client_id), '
                INSERT INTO campaigns_multicurrency_sums
                    (cid, chips_cost, chips_spent) VALUES %s
                ON DUPLICATE KEY UPDATE
                    chips_cost = VALUES(chips_cost),
                    chips_spent = VALUES(chips_spent)
            ', \@for_insert);
        }

        my @wallets = grep { $_->{mediaType} eq 'wallet' } @$campaigns;
        if (@wallets) {
            $log->out('Making wallets locked in Balance');
            my $wallet_cids = [map { $_->{cid} } @wallets];
            create_campaigns_balance($rbac, $OPERATOR_UID, $wallet_cids, force_lock_wallet => 1);
        }

        $log->out('Begin converting campaigns');

        my $campaigns_multicurrency_sums_data = get_hashes_hash_sql(PPC(ClientID => $client_id), ['SELECT cid, sum, chips_cost FROM campaigns_multicurrency_sums', WHERE => {cid => \@cids}]);
        for my $campaign (@$campaigns) {
            my $cid = $campaign->{cid};
            my $for_msg_prefix_guard = $log->msg_prefix_guard("$client_msg_prefix [cid $cid]");

            my $converted_campaign_params = Client::ConvertToRealMoney::get_converted_campaign_params($campaign, $currency, $rate, keep_ProductID => 1, copy_last_bill_sums => 1, save_conversion_strategy => 1);

            # берём сохранённые суммы для кампании в валюте, которые получаем в нотификациях от Баланса
            my $campaign_multicurrency_sums_data = $campaigns_multicurrency_sums_data->{$cid};
            my $new_sum_spent = ($campaign_multicurrency_sums_data) ? $campaign_multicurrency_sums_data->{chips_cost} // 0 : 0;
            my $new_sum = ($campaign_multicurrency_sums_data) ? $campaign_multicurrency_sums_data->{sum} // 0 : 0;

            $converted_campaign_params->{campaigns}->{sum} = $new_sum;
            $converted_campaign_params->{campaigns}->{sum_spent} = $new_sum_spent;
            $converted_campaign_params->{campaigns}->{balance_tid} = 0;
            if ($campaign->{mediaType} eq 'wallet') {
                # общую сумму поступлений по кампаниям под кошельком вычисляем по курсу фишек к валюте
                # со временем, после завершения конвертации, из баланса прийдет более точная сумма, с возможными корректировками по скидкам, промо-кодам
                my $total_sum = get_one_field_sql(PPC(ClientID => $client_id), ['SELECT total_sum FROM wallet_campaigns', WHERE => {wallet_cid => $cid}]) // 0;
                $converted_campaign_params->{wallet_campaigns}->{total_sum} = $total_sum * $rate_nds;
                $converted_campaign_params->{wallet_campaigns}->{total_balance_tid} = 0;
            }
            $log->out('Converted params:', $converted_campaign_params);

            $log->out("Logging new campaign sum $new_sum");
            DBLog::log_balance($client_id, $campaign->{uid}, $cid, $new_sum, 0, 0, $campaign->{mediaType}, $currency);

            $log->out('Converting campaign');

            do_in_transaction {
                # лочим валюту в campaigns. в других местах можно также выбрать кампанию с FOR UPDATE и добиться последовательного выполнения транзакции на построчном локе.
                my $campaign_currency = get_one_field_sql(PPC(ClientID => $client_id), ['SELECT currency FROM campaigns', WHERE => {cid => $cid}, 'FOR UPDATE']);
                $log->die("Campaign $cid is already in $campaign_currency. That's VERY strange =)") if $campaign_currency && $campaign_currency ne 'YND_FIXED';

                Client::ConvertToRealMoney::convert_campaign_bids([$cid], 'YND_FIXED', $currency);
                Client::ConvertToRealMoney::convert_campaign_secondary_options([$cid], 'YND_FIXED', $currency);
                Client::ConvertToRealMoney::convert_campaign_payments_info([$cid], 'YND_FIXED', $currency);

                if ($converted_campaign_params->{camp_options} && %{$converted_campaign_params->{camp_options}}) {
                    do_update_table(PPC(ClientID => $client_id), 'camp_options', $converted_campaign_params->{camp_options}, where => {cid => $cid});
                }
                do_update_table(PPC(ClientID => $client_id), 'campaigns', {
                    currencyConverted => 'Yes',
                    # прогноз сконвертировали по курсу, но ставки могли вырасти (больше минимальная ставка), да и шаг торгов другой — надёжнее пересчитать
                    autobudgetForecastDate => undef,
                    statusBsSynced => 'No',
                    %{$converted_campaign_params->{campaigns}}
                }, where => {cid => $cid});

                my $additional_table = "campaigns_$campaign->{type}";
                if (exists $converted_campaign_params->{$additional_table}) {
                    # сейчас здесь только campaigns_performance
                    do_update_table(PPC(ClientID => $client_id), $additional_table,
                        $converted_campaign_params->{$additional_table}, where => {cid => $cid});
                }

                # записываем суммы до конвертации и после, чтобы показывать их на странице успеха
                my $money_before = $campaign->{total};
                my $money_after = $new_sum - $new_sum_spent;
                $log->out("Writing correspondence data for campaign $cid: money_before = $money_before, money_after = $money_after");
                do_insert_into_table(PPC(ClientID => $client_id), 'currency_convert_money_correspondence', {
                    ClientID => $client_id,
                    old_cid => $cid,
                    new_cid => $cid,
                    money_before => $money_before,
                    money_after => $money_after,
                    was_archived => ($campaign->{archived} && $campaign->{archived} eq 'Yes') ? 1 : 0,
                }, on_duplicate_key_update => 1, key => 'ClientID');

                 if ($converted_campaign_params->{wallet_campaigns} && %{$converted_campaign_params->{wallet_campaigns}}) {
                    do_update_table(PPC(ClientID => $client_id), 'wallet_campaigns', $converted_campaign_params->{wallet_campaigns}, where => {wallet_cid => $cid});
                }

                # multicurrency: monitor.agency_clients_month_stat
            };

            $log->out('Done converting campaign');
        }
        $log->out('Done converting campaigns');

        do_in_transaction {
            # все кампании сконвертировали. теперь записываем клиенту новую валюту
            $log->out('fixating new client currency');
            Client::ConvertToRealMoney::fixate_client_currency($client_id, $currency, $data->{country_region_id});

            $log->out('cleanup multicurrency data');
            Client::ConvertToRealMoney::cleanup_data_for_converted_client($client_id);

            $log->out('Fixating convert results');
            do_update_table(PPC(ClientID => $client_id), 'currency_convert_queue', {state => 'CONVERTING_DETAILED_STAT', in_state_since__dont_quote => 'NOW()'}, where => \%queue_cond);
        };

        return 1;
    };

    if ($success) {
        $log->out('finish');
    } else {
        $log->die("Error:", $@);
    }
}

=head2 convert_detailed_stat

=cut

sub convert_detailed_stat {
    my ($data) = @_;

    my $client_id = $data->{ClientID};
    my $currency = $data->{new_currency};

    my $client_msg_prefix = "[convert_detailed_stat] [ClientID $client_id]";
    my $prefix_guard = $log->msg_prefix_guard($client_msg_prefix);

    my $profile = Yandex::Trace::new_profile('ConvertToRealMoney:convert_detailed_stat');
    $log->out('start');

    my $success = eval {
        my $sql_lock = Client::get_per_client_convert_lock_guard($client_id);

        # проверяем наличие в очереди задачи в правильном состоянии
        my %queue_cond = (ClientID => $client_id, state => 'CONVERTING_DETAILED_STAT', convert_type => 'MODIFY');
        my ($is_task_correct) = get_one_line_array_sql(PPC(ClientID => $client_id), ['SELECT 1 FROM currency_convert_queue', WHERE => \%queue_cond]);
        die("Incorrect task state in queue") unless $is_task_correct;

        $log->out('skip rollbacking stat');
        do_update_table(PPC(ClientID => $client_id), 'currency_convert_queue', {state => 'FETCHING_BALANCE_DATA', in_state_since__dont_quote => 'NOW()'}, where => \%queue_cond);

        return 1;
    };

    if ($success) {
        $log->out('finish');
    } else {
        $log->die("Error:", $@);
    }
}

=head2 fetch_balance_data

    * выбрасывает данные о бюджете клиенте (через сутки-двое бюджет пересчитается и обновится)
    * выбрасываем данные о доступных суммах овердрафта (новые данные должны прийти через несколько минут после начала конвертации)

    Переводит заявку на переход из состояния FETCHING_BALANCE_DATA в NOTIFY или OVERDRAFT_WAITING (в зависимости от наличия овердрафта).

=cut

sub fetch_balance_data {
    my ($data) = @_;

    my $client_id = $data->{ClientID};
    my $currency = $data->{new_currency};

    my $client_msg_prefix = "[fetch_balance_data] [ClientID $client_id]";
    my $prefix_guard = $log->msg_prefix_guard($client_msg_prefix);

    my $profile = Yandex::Trace::new_profile('ConvertToRealMoney:fetch_balance_data');
    $log->out('start');

    my $success = eval {
        my $sql_lock = Client::get_per_client_convert_lock_guard($client_id);

        # проверяем наличие в очереди задачи в правильном состоянии
        my %queue_cond = (ClientID => $client_id, state => 'FETCHING_BALANCE_DATA');
        my ($is_task_correct) = get_one_line_array_sql(PPC(ClientID => $client_id), ['SELECT 1 FROM currency_convert_queue', WHERE => \%queue_cond]);
        die "Incorrect task state in queue" unless $is_task_correct;

        # выкидываем данные о бюджете. он пересчитается только через день-два и заберётся штатным способом.
        # на это время ставим заказу нулевую скидку. если по факту скидки нет, то клиент ничего не увидит.
        # если скидка есть, то всюду увидит только скидку из графика (см. код про отличающиеся валюты в get_mass_overdraft_info)
        my %overdraft_changes = (discount => 0, budget => 0, border_next => 0, discount_next => 0, border_prev => 0);

        my $skip_overdraft_waiting;
        if (get_one_field_sql(PPC(ClientID => $client_id), 'SELECT 1 FROM clients_options WHERE ClientID = ? AND (overdraft_lim > 0 OR debt > 0)', $client_id)) {
            $overdraft_changes{overdraft_lim} = 0;
            $overdraft_changes{debt} = 0;
            $overdraft_changes{balance_tid} = 0;
            $skip_overdraft_waiting = 0;
        } else {
            # если овердрафта не было до конвертации, то не ждём его и после
            $skip_overdraft_waiting = 1;
        }

        $log->out('Resetting data in overdraft:', \%overdraft_changes);
        do_update_table(PPC(ClientID => $client_id), 'clients_options', \%overdraft_changes, where => {ClientID => $client_id});

        my $new_state = ($skip_overdraft_waiting) ? 'NOTIFY' : 'OVERDRAFT_WAITING';
        $log->out("Fixating results (next state is $new_state)");
        do_update_table(PPC(ClientID => $client_id), 'currency_convert_queue', {state => $new_state, in_state_since__dont_quote => 'NOW()'}, where => \%queue_cond);

        return 1;
    };

    if ($success) {
        $log->out('finish');
    } else {
        $log->die("Error:", $@);
    }
}

=head2 wait_for_overdraft

    Дожидается получения от Баланса нотификации о новой сумме достпного овердрафта.

    Переводит заявку на конвертацию из состояния OVERDRAFT_WAITING в NOTIFY.

=cut

sub wait_for_overdraft {
    my ($data) = @_;

    my $client_id = $data->{ClientID};

    my $client_msg_prefix = "[wait_for_overdraft] [ClientID $client_id]";
    my $prefix_guard = $log->msg_prefix_guard($client_msg_prefix);

    my $profile = Yandex::Trace::new_profile('ConvertToRealMoney:wait_for_overdraft');
    $log->out('start');

    my $success = eval {
        my $sql_lock = Client::get_per_client_convert_lock_guard($client_id);

        # проверяем наличие в очереди задачи в правильном состоянии
        my %queue_cond = (ClientID => $client_id, state => 'OVERDRAFT_WAITING');
        my ($is_task_correct, $in_state_since) = get_one_line_array_sql(PPC(ClientID => $client_id), ['SELECT 1, in_state_since FROM currency_convert_queue', WHERE => \%queue_cond]);
        $log->die("Incorrect task state in queue") unless $is_task_correct;

        my $can_continue = 0;
        if ($in_state_since && check_mysql_date($in_state_since) && time() - mysql2unix($in_state_since) > $TIME_TO_WAIT_FOR_OVERDRAFT*60) {
            $log->out("No overdraft found, but $TIME_TO_WAIT_FOR_OVERDRAFT minutes have already passed. Will stop waiting and continue.");
            $can_continue = 1;
        } else {
            # приход овердрафта определяем по появлению ненулевого номера транзакции (на прошлом шаге его сбросили в ноль)
            my $balance_tid = get_one_field_sql(PPC(ClientID => $client_id), ['SELECT balance_tid FROM clients_options', where => {ClientID => SHARD_IDS}]);
            if ($balance_tid && $balance_tid > 0) {
                $log->out('Got overdraft notification, fixating results');
                $can_continue = 1;
            }
        }

        if ($can_continue) {
            $log->out('Processing to next step');
            do_update_table(PPC(ClientID => $client_id), 'currency_convert_queue', {state => 'NOTIFY', in_state_since__dont_quote => 'NOW()'}, where => \%queue_cond);
            return 1;
        } else {
            $log->out('No overdraft found, will continue waiting');
            return 0;
        }
    };

    if (defined $success) {
        if ($success) {
            $log->out('finish');
        } else {
            # блок eval не упал, но и успешно не закончился
            # значит, сейчас он успешно завершиться не может (например, кампании ещё не остановились)
            $log->out('task not completed, finishing');
        }
    } else {
        $log->die("Error:", $@);
    }
}

=head2 _get_migrate_date($start_convert_at, $convert_type)

    Возвращает значение MIGRATE_TO_CURRENCY, которое нужно отправить в Баланс.
    Возвращает текстовую строку в MySQL-совместимом формате (YYYY-MM-DD HH:MM:SS)
    Для конвертации копированием назначенное время конвертации возвращается как есть.
    Для конвертации без копирования возвращается либо ближайшая, либо следюущая полночь [если ближайшая уже наступила].

=cut

sub _get_migrate_date {
    my ($start_convert_at, $convert_type) = @_;

    die 'no convert type given' unless $convert_type;

    if ($convert_type eq 'COPY') {
        if (time() > mysql2unix($start_convert_at)) {
            return unix2mysql(time() + 60);
        } else {
            return $start_convert_at;
        }
    } elsif ($convert_type eq 'MODIFY') {
        if (time() > mysql2unix($start_convert_at) + 10*60) {
            # время конвертации [полночь] уже наступило, а мы ещё никому не сказала о конвертации клиента
            # переносим конвертацию на следующие сутки. лучше уж так, чем совсем запороть конвертацию.
            return tomorrow();
        } else {
            return $start_convert_at;
        }
    } else {
        die "unknown convert type $convert_type given";
    }
}

1;
