package Client::CustomOptions;

=head1 DESCRIPTION

Модуль для работы с редко востребованными пользовательскими опциями.

Опции хранятся в таблице clients_custom_options в виде
    ClientID | keyname | value

Джойниться с этой таблицей крайне не рекомендуется, т.к. на одного пользователя может приходиться
неограниченое количество опций.

value имеет тип int

option_name в данный момент никак не проверяется

В API есть своя таблица api_special_user_options с аналогичной структурой и свойствами
В отличие от api, где записи читаются всегда перед каждым запросом, эта таблица читается
только по мере надобности

=cut

use strict;
use warnings;
use utf8;
use base qw/Exporter/;

use Settings;
use Yandex::DBTools;

our @EXPORT = qw/
    get_user_custom_options
    set_user_custom_options
    delete_user_custom_options
    mass_get_user_custom_options
/;

=head2 mass_get_user_custom_options

Получить кастомные опции для нескольких клиентов
mass_get_user_custom_options(\@client_ids) = {
    $ClientID => {
        option1 => 1,
        ...
    },
    ...
}

=cut

sub mass_get_user_custom_options
{
    my $client_ids = shift;
    my $rows = get_all_sql(PPC(ClientID => $client_ids), 
        [ "select ClientID, keyname, `value` from clients_custom_options", 
            where => { ClientID => $client_ids } 
    ]);

    my %ret;
    for my $r (@$rows) {
        $ret{ $r->{ClientID} }{ $r->{keyname} } = $r->{value};
    }
    return \%ret;
}

=head2 get_user_custom_options

Вернуть все кастомные опции клиента в виде хеша

get_user_custom_options($client_id) = {
    option1 => 1,
    option2 => 0,
    ...
}

=cut

sub get_user_custom_options
{
    my $client_id = shift;
    return mass_get_user_custom_options([$client_id])->{$client_id} || {};
}

=head2 set_user_custom_options

Установить кастомные опции для клиента

set_user_custom_options($client_id, { opt1 => val1, opt2 => val2, ... })

=cut

sub set_user_custom_options
{
    my $client_id = shift;
    my $opts = shift;
    my @insert = map { [ $client_id, $_, $opts->{$_} ] } keys %$opts;
    do_mass_insert_sql(PPC(ClientID => $client_id), 
        "insert into clients_custom_options (ClientID, keyname, `value`) values %s
        on duplicate key update `value`=values(`value`)", 
        \@insert
    );
}

=head2 delete_user_custom_options

Удалить кастомные опции для клиента

delete_user_custom_options($client_id, 'option_name')
delete_user_custom_options($client_id, ['option_name1', 'option_name2'])

=cut

sub delete_user_custom_options
{
    my ($client_id, $options) = @_;
    do_delete_from_table(PPC(ClientID => $client_id), 'clients_custom_options', where => { ClientID => $client_id, keyname => $options });
}

1;
