package Client::MCC;

# $Id$

=head1 NAME

    Client::MCC

=head1 DESCRIPTION

    Модуль для операций с клиентским МСС-аккаунтом

=cut

use Direct::Modern;

use Settings;
use Yandex::DBTools;
use Yandex::DBShards;
use Rbac qw/:const/;

our @EXPORT = qw/
    add_client_to_mcc
    delete_client_from_mcc

    get_mcc_clients_chief_rep_info
    get_mcc_control_clients_chief_rep_info

/;

=head2 add_client_to_mcc

    Привязать клиента к управляющему аккаунту

=cut

sub add_client_to_mcc {
    my ($client_id, $control_client_id) = @_;

    my $relation_id = get_new_id('relation_id');
    my $reverse_relation_id = get_new_id('reverse_relation_id');

    do_insert_into_table(PPC(ClientID => $client_id), 'clients_relations', {
            relation_id => $relation_id,
            client_id_from => $control_client_id,
            client_id_to => $client_id,
            type => 'mcc'
        },
        ignore => 1
    );

    do_insert_into_table(PPC(ClientID => $control_client_id), 'reverse_clients_relations', {
            reverse_relation_id => $reverse_relation_id,
            client_id_from => $control_client_id,
            client_id_to => $client_id,
            type => 'mcc'
        },
        ignore => 1
    );

    return;
}

=head2 delete_client_from_mcc

    Отвязать клиента от управляющего аккаунта

=cut

sub delete_client_from_mcc {
    my ($client_id, $control_client_id) = @_;

    do_delete_from_table(PPC(ClientID => $client_id), 'clients_relations',
        where => {
            client_id_from => $control_client_id,
            client_id_to => $client_id,
            type => 'mcc'
        }
    );

    do_delete_from_table(PPC(ClientID => $control_client_id), 'reverse_clients_relations',
        where => {
            client_id_from => $control_client_id,
            client_id_to => $client_id,
            type => 'mcc'
        }
    );

    return;
}

=head2 get_mcc_clients_chief_rep_info

    Возвращает данные о управляемых клиентах MCC-аккаунта по списку ClientID

=cut

sub get_mcc_clients_chief_rep_info($) {
    my ($client_ids) = @_;
    return [] if !$client_ids || !@$client_ids;

    return get_all_sql(PPC(ClientID => $client_ids), [
        "SELECT ClientID as client_id, work_currency as currency, login, FIO as fio
        FROM users JOIN clients using(ClientID)",
        where => { ClientID => SHARD_IDS, rep_type => $REP_CHIEF }
    ]);
}

=head2 get_mcc_control_clients_chief_rep_info

    Возвращает данные об управляющих клиентах MCC-аккаунтов для заданного ClientID

=cut

sub get_mcc_control_clients_chief_rep_info($) {
    my ($client_id) = @_;
    return [] if !$client_id;

    my $control_client_ids = get_one_column_sql(PPC(ClientID => $client_id), [
        "SELECT client_id_from
        FROM clients_relations",
        where => { client_id_to => $client_id, type => 'mcc' }
    ]);

    return get_all_sql(PPC(ClientID => $control_client_ids), [
        "SELECT ClientID as client_id, login, FIO as fio
        FROM users JOIN clients using(ClientID)",
        where => { ClientID => SHARD_IDS, rep_type => $REP_CHIEF }
    ]);
}

1;
