package Client::NDSDiscountSchedule;

=encoding utf8

=head1 NAME

    Client::NDSDiscountSchedule

=head1 DESCRIPTION

    В модуль вынесена функциональность получения и обработки НДС из Баланса.
    Используется:
      - из скрипта balanceGetClientNDSDiscountSchedule.pl;
      - из интерфейса при выставлении счёта новым клиентом, для которого ещё нет графика НДС;
      - при конвертации для определения графика НДС по стране для клиента без выбранной в Балансе страны.

    Раньше еще были скидки (поэтому такое название модуля), но они больше не обновляются - DIRECT-70057

=cut

use Direct::Modern;

use base qw/Exporter/;

our @EXPORT_OK = qw/
    sync_nds_schedule_for_clients
    sync_all_nds_schedules
/;

use List::MoreUtils qw/part zip uniq/;
use List::Util qw/min max/;
use JSON;

use BalanceWrapper;
use Settings;
use Yandex::DBTools;
use Yandex::DBShards;
use RBACElementary;
use Primitives;
use ShardingTools;
use BS::ResyncQueue ();

use Yandex::ListUtils qw/xsort/;
use Yandex::Validate qw/is_valid_int is_valid_float/;
use Yandex::HashUtils qw/hash_diff hash_cut hash_merge/;
use Yandex::TimeCommon qw/unix2human mysql2unix yesterday tomorrow check_mysql_date today mysql_round_day/;
use Yandex::Balance qw/balance_get_client_nds balance_create_client/;
use Yandex::Retry qw/relaxed_guard/;
use geo_regions;


my $BALANCE_FETCH_CHUNKS_COUNT = 512;
my $UIDS_RESYNC_CHUNK_SIZE = 1_000;
my $OPERATOR_UID = 1;

my %operations = (
    NDS => {
        balance_fetch_cmd => \&balance_get_client_nds,
        preprocess_cmd => \&_preprocess_nds_schedule_data,
        row_serialization_cmd => \&_serialize_nds_row,
        row_deserialization_cmd => \&_deserialize_nds_row,
        table_name => 'client_nds',
        field_names => [qw/ClientID date_from date_to nds/],  # nds field is also hardcoded in _check_and_filter_nds_data
        conditional_table_name => 'agency_nds',
        conditional_filter_cmd => \&_filter_agency,
    },
);

# Список клиентов по которым мы расходимся с Балансом графиком НДС
# будем заполнять по мере выяснения причин, по которым графики расходятся
my @KNOWN_BAD_NDS_CLIENTIDS = ();

=head2

    2 функции ниже принимают следующие именованные параметры:
        log
        log_data_from_balance
        dont_die   - не кидать исключение, если от Баланса пришел измененный график задним числом

=cut

sub sync_nds_schedule_for_clients {
    return _sync_schedule_for_clients('NDS', @_, empty_data_is_error => 1);
}

sub sync_all_nds_schedules {
    return _sync_all_schedules('NDS', @_);
}

sub _sync_schedule_for_clients {
    my ($operation_name, $client_ids, %O) = @_;

    return unless $client_ids && @$client_ids;

    my $log = $O{log};

    my $old_msg_prefix;
    my $has_errors = 0;
    $old_msg_prefix = $log->msg_prefix if $log;
    my $clientids_for_fake_byn_nds;
    my $clientids_for_fake_quasi_currency_nds;
    # список клиентов, которым не обновляем график НДС
    # дополнительно заполняется ниже белорусскими клиентами в RUB, USD и EUR
    my $dont_touch_clientids_graph = { map { $_ => undef } @KNOWN_BAD_NDS_CLIENTIDS };
    if ($operation_name eq 'NDS') {
        # NB: ниже есть похожие запросы, выбирающий такие же данные, но из всех шардов для всех клиентов
        $clientids_for_fake_byn_nds = _get_clientids_for_fake_byn_nds($client_ids);

        # квазивалютными сейчас бывают только казахи
        $clientids_for_fake_quasi_currency_nds = _get_clientids_for_fake_quasi_currency_nds($client_ids);

        # Белорусы в рублях, долларах и евро
        my $belarus_rub_usd_clientids = get_hash_sql(PPC(ClientID => $client_ids), ['SELECT ClientID, NULL FROM clients',
            WHERE => {
                ClientID => SHARD_IDS,
                work_currency => ["RUB", "USD", "EUR"],
                country_region_id => $geo_regions::BY,
            } ]);
        hash_merge($dont_touch_clientids_graph, $belarus_rub_usd_clientids);
    }
    for my $client_id (@$client_ids) {
        $log->msg_prefix(($old_msg_prefix ||'') . "[$operation_name, ClientID $client_id]") if $log;
        # для получения данных по единственному клиенту делим на <много>, тогда остаток от деления будет равен ClientID
        # и мы получим записи по единственному клиенту
        my $success = eval {
            $has_errors = _request_and_process_clients_data(
                mod => $Settings::MAX_INT_VAL,
                rem => $client_id,
                operation_name => $operation_name,
                ($operation_name eq 'NDS' ? (
                    fake_byn_nds_clients => $clientids_for_fake_byn_nds,
                    fake_quasi_currency_nds_clients => $clientids_for_fake_quasi_currency_nds,
                    dont_touch_clientids_graph => $dont_touch_clientids_graph,
                    all_schedules => 0) : ()),
                %{hash_cut \%O, qw/log log_data_from_balance empty_data_is_error timeout dont_die/},
            ) || $has_errors;
            return 1;
        };
        if (!$success) {
            if (!$O{dont_die}) {
                _die_log($log, $@);
            } elsif ($log) {
                $log->warn({
                        error => "error processing client",
                        client_id => $client_id,
                        message => $@,
                    });
                $has_errors = 1;
            }
        }
        $log->msg_prefix($old_msg_prefix) if $log;
    }

    return $has_errors;
}

sub _sync_all_schedules {
    my ($operation_name, %O) = @_;

    my $log = $O{log};

    my $old_msg_prefix;
    $old_msg_prefix = $log->msg_prefix if $log;

    my $clientids_for_fake_byn_nds;
    my $clientids_for_fake_quasi_currency_nds;
    my $dont_touch_clientids_graph = { map { $_ => undef } @KNOWN_BAD_NDS_CLIENTIDS };
    if ($operation_name eq 'NDS') {
        $clientids_for_fake_byn_nds = get_one_column_sql(PPC(shard => "all"), ['SELECT ClientID FROM clients', 
            WHERE => { 
                _OR => {
                    work_currency => "BYN",
                    _AND => { country_region_id => $geo_regions::BY, _OR => {work_currency => "YND_FIXED", work_currency__is_null => 1}},
                }
            } ]);
        # квазивалютными сейчас бывают только казахи
        $clientids_for_fake_quasi_currency_nds = get_one_column_sql(PPC(shard => "all"),
            ['SELECT c.ClientID FROM clients c
              JOIN clients_options co on c.ClientID=co.ClientID',
              WHERE => {'c.country_region_id' => $geo_regions::KAZ, 'c.work_currency' => 'KZT', 'co.is_using_quasi_currency' => 1}]);
        # Белорусы в рублях, долларах и евро
        my $belarus_rub_usd_clientids = get_hash_sql(PPC(shard => "all"), ['SELECT ClientID, NULL FROM clients', 
            WHERE => {
                work_currency => ["RUB", "USD", "EUR"],
                country_region_id => $geo_regions::BY,
            } ]);
        hash_merge($dont_touch_clientids_graph, $belarus_rub_usd_clientids);
    } else {
        $clientids_for_fake_byn_nds = [];
        $clientids_for_fake_quasi_currency_nds = [];
    }

    my $has_errors = 0;
    for (my $chunk_index = 0; $chunk_index < $BALANCE_FETCH_CHUNKS_COUNT; $chunk_index++) {
        $log->msg_prefix("[$operation_name, $chunk_index/$BALANCE_FETCH_CHUNKS_COUNT]");
        my @clientids_for_fake_byn_nds_chunk = grep {  $_ % $BALANCE_FETCH_CHUNKS_COUNT == $chunk_index } @$clientids_for_fake_byn_nds;
        my @clientids_for_fake_quasi_currency_nds_chunk = grep {  $_ % $BALANCE_FETCH_CHUNKS_COUNT == $chunk_index } @$clientids_for_fake_quasi_currency_nds;
        my $guard = relaxed_guard times => 0.2;
        eval {
            $has_errors = _request_and_process_clients_data(
                mod => $BALANCE_FETCH_CHUNKS_COUNT,
                rem => $chunk_index,
                operation_name => $operation_name,
                ($operation_name eq 'NDS' ? (fake_byn_nds_clients => \@clientids_for_fake_byn_nds_chunk,
                                             fake_quasi_currency_nds_clients => \@clientids_for_fake_quasi_currency_nds_chunk,
                                             dont_touch_clientids_graph => $dont_touch_clientids_graph,
                                             all_schedules => 1)
                                          : ()),
                %{hash_cut \%O, qw/log log_data_from_balance timeout dont_die/},
            ) || $has_errors;
        };
        if ($@) {
            if (!$O{dont_die}) {
                _die_log($log, $@);
            } elsif ($log) {
                $log->warn({
                        error => "error processing chunk",
                        message => $@,
                    });
                $has_errors = 1;
            }
        }
    }

    $log->msg_prefix($old_msg_prefix) if $log;

    return $has_errors;
}

sub _request_and_process_clients_data {
    my (%O) = @_;

    my $log = $O{log};
    my $operation_name = $O{operation_name};
    my $operation_params = $operations{$operation_name};

    $log->out('Fetching chunk') if $log;
    my $data = $operation_params->{balance_fetch_cmd}->(mod => $O{mod}, rem => $O{rem}, timeout => $O{timeout});
    $log->out('Got chunk with ' . scalar(@$data) . ' records') if $log;
    if ($O{log_data_from_balance} && $log) {
        $log->out('Data from Balance:');
        $log->out($_) for @$data;
    }

    if (!@$data) {
        if ($operation_name eq 'NDS'
            && $O{mod} == $Settings::MAX_INT_VAL
            && $O{fake_byn_nds_clients} && ref $O{fake_byn_nds_clients} eq 'ARRAY'
            && $O{fake_quasi_currency_nds_clients} && ref $O{fake_quasi_currency_nds_clients} eq 'ARRAY'
            && grep { $O{rem} == $_ } (@{ $O{fake_byn_nds_clients} }, @{ $O{fake_quasi_currency_nds_clients} })
        ) {
            # специальная ветка кода, чтобы не провалиться в следующие ветви условия, в которых или die или return
            # нужно попасть дальше в preprocess_cmd, в которой фейковый график для BYN или квазивалюты подкладывается вместо данных из @$data
        } elsif ($O{empty_data_is_error}) {
            if ($O{mod} == $Settings::MAX_INT_VAL && !$O{no_manual_county_setting}) {
                # специальный костыль: Баланс график НДС определяет по стране. для старых уешных клиентов
                # страны нет, но при наличии оплат её можно однозначно определить. Баланс этого не делает,
                # поэтому делаем мы: получаем список доступных стран-валют и, если там только один вариант
                # страны, проставляем эту страну и получаем график снова.
                my $client_id = $O{rem};
                $log->out("Got empty NDS graph (before manual country setting) from Balance for mod = $O{mod}, rem => $O{rem}") if $log;
                my $agency_id = Primitives::get_client_first_agency($client_id);
                my $firm_country_currency_data = BalanceWrapper::get_firm_country_currency($client_id, AgencyID => $agency_id, currency_filter => 1, timeout => $O{timeout});
                if ($firm_country_currency_data && @$firm_country_currency_data) {
                    my @countries = uniq map { $_->{region_id} } @$firm_country_currency_data;
                    if (scalar(@countries) == 1) {
                        # единственная страна, её и прописываем у себя и в Балансе
                        my $country = $countries[0];
                        do_insert_into_table(PPC(ClientID => $client_id), 'clients', {ClientID => $client_id, country_region_id => $country}, on_duplicate_key_update => 1, key => ['ClientID']);
                        my ($error) = balance_create_client($OPERATOR_UID, {CLIENT_ID => $client_id, SERVICE_ID => $Settings::SERVICEID{direct}, REGION_ID => $country});
                        if (!$error) {
                            $log->out("Fetching NDS graph again (after manual country setting) for ClientID $client_id") if $log;
                            return _request_and_process_clients_data(%O, no_manual_county_setting => 1);
                        } else {
                            die("Error setting client country $country in Balance for ClientID $client_id:", $error);
                        }
                    } else {
                        $log->out("Too many countries: ", join(', ', @countries), " for manual country setting for ClientID $client_id") if $log;
                    }
                } else {
                    die("Error getting firm/country/currency data from Balance for manual settings for mod = $O{mod}, rem => $O{rem}");
                }
            } else {
                die("Got empty NDS graph (no manual setting) from Balance for mod = $O{mod}, rem => $O{rem}");
            }
        } else {
            return;
        }
    }

    my $fake_byn_nds_clients = $O{fake_byn_nds_clients};
    my $fake_quasi_currency_nds_clients = $O{fake_quasi_currency_nds_clients};

    if ($operation_name eq 'NDS' && $O{all_schedules}) {
        my %byn_nds_clients_to_check = map { $_ => undef } @$fake_byn_nds_clients;
        my %quasi_currency_nds_clients_to_check = map { $_ => undef } @$fake_quasi_currency_nds_clients;
        my %client_ids_to_check = (%byn_nds_clients_to_check, %quasi_currency_nds_clients_to_check);

        my @client_ids = map { $_->{CLIENT_ID} } @$data;
        # не проверяем уже проверенных
        @client_ids = grep { !(exists $client_ids_to_check{$_}) } @client_ids;

        my $clientids_for_fake_byn_nds_add = _get_clientids_for_fake_byn_nds(\@client_ids);
        my $clientids_for_fake_quasi_currency_nds_add = _get_clientids_for_fake_quasi_currency_nds(\@client_ids);

        push(@$fake_byn_nds_clients, @$clientids_for_fake_byn_nds_add);
        push(@$fake_quasi_currency_nds_clients, @$clientids_for_fake_quasi_currency_nds_add);
    }

    $log->out('Preprocessing fetched data') if $log;
    my $converted_data = $operation_params->{preprocess_cmd}->($data,
        (fake_byn_nds_clients => $fake_byn_nds_clients,
         fake_quasi_currency_nds_clients => $fake_quasi_currency_nds_clients));

    $log->out('Merging chunk into PPC.' . $operation_params->{table_name}) if $log;
    my ($affected_clientids, $has_errors) = _merge_schedule_to_table($converted_data,
        mod => $O{mod}, rem => $O{rem},
        %{hash_cut $operation_params, qw/row_serialization_cmd row_deserialization_cmd field_names table_name conditional_table_name conditional_filter_cmd/},
        log => $log,
        %{hash_cut \%O, qw/dont_die dont_touch_clientids_graph/},
    );

    if ($operation_params->{conditional_table_name}) {
        $log->out('Merging chunk into PPC.' . $operation_params->{conditional_table_name}) if $log;
        my $converted_data_filtered = $operation_params->{conditional_filter_cmd}->($converted_data);
        my (undef, $errors) = _merge_schedule_to_table($converted_data_filtered,
            mod => $O{mod}, rem => $O{rem},
            %{hash_cut $operation_params, qw/row_serialization_cmd row_deserialization_cmd field_names conditional_table_name conditional_filter_cmd/},
            table_name => $operation_params->{conditional_table_name},
            log => $log,
            all_shards => 1,
            %{hash_cut \%O, qw/dont_die dont_touch_clientids_graph/},
        );
        $has_errors ||= $errors;
    }

    _resend_changed_client_campaigns_to_BS($affected_clientids, %{hash_cut \%O, qw/log/});

    return $has_errors;
}

sub _validate_nds_schedule_row {
    my ($row) = @_;

    return "invalid ClientID" unless is_valid_int($row->{CLIENT_ID});
    return "invalid start date" unless check_mysql_date($row->{DT});
    return "invalid NDS percent value" unless is_valid_float($row->{NDS_PCT});

    return undef;
}

sub _preprocess_nds_schedule_data {
    my ($data, %O) = @_;

    my $log = $O{log};
    my $clientids_for_fake_byn_nds = $O{fake_byn_nds_clients} || [];
    my $clientids_for_fake_quasi_currency_nds = $O{fake_quasi_currency_nds_clients} || [];

    my %data_by_client;
    for my $row (@$data) {
        my $error = _validate_nds_schedule_row($row);
        if (!$error) {
            # переименовываем поля в наши названия, группируем по клиенту
            my $row_converted = {
                ClientID => $row->{CLIENT_ID},
                date_from => $row->{DT},
                nds => $row->{NDS_PCT},
            };
            push @{$data_by_client{$row_converted->{ClientID}}}, $row_converted;
        } else {
            my $error_message = to_json({
                    error => $error,
                    row_data => $row,
                });

            die($error_message);
        }
    }

    # специальный костыль для белорусов: вместо реальной валюты у них на самом деле "белорусские фишки", у которых нет графика НДС
    # но в Директе и БК эти "белоруские фишки" обрабатываются как реальная валюта и для них используется нулевой график НДС
    for my $client_id (@$clientids_for_fake_byn_nds) {
        my $real_data = $data_by_client{$client_id};
        # фейковый график НДС для белорусских рублей
        my @fake_data = ( 
            {date_from => '2003-01-01', nds => 0, ClientID => $client_id},
            {date_from => '2004-01-01', nds => 0, ClientID => $client_id},
        );
        $data_by_client{$client_id} = \@fake_data;
        if ($log) {
            $log->out({"using fake data" => \@fake_data, "dropped balance data" => $real_data});
        }
    }

    # фейковый график для клиентов с квазивалютой. Используется для новых клиентов в кахазском тенге (KZT)
    for my $client_id (@$clientids_for_fake_quasi_currency_nds) {
        my $real_data = $data_by_client{$client_id};
        # фейковый график НДС для квазивалюты
        my @fake_data = (
            {date_from => '1970-01-02', nds => 0, ClientID => $client_id},
        );
        $data_by_client{$client_id} = \@fake_data;
        if ($log) {
            $log->out({"using fake data" => \@fake_data, "dropped balance data" => $real_data});
        }
    }

    my $exists_clients = get_hash_sql(PPC(ClientID => [keys %data_by_client]),
        ['SELECT ClientID, 1 FROM users', where => {ClientID => SHARD_IDS},'GROUP BY ClientID']);

    # проставляем даты окончания действия НДСов
    my @converted_data;
    while (my ($client_id, $rows) = each %data_by_client) {
        my @modified_rows;
        if (! $exists_clients->{$client_id}) {
            if ($log) {
                $log->out({"skip nonexistent client" => $client_id, "dropped balance data" => $rows});
            }
            next;
        }
        # не храним скидки для клиентов, у которых их никогда не было
        my @rows_sorted = xsort {mysql2unix($_->{date_from})} @$rows;
        my $prev_row;
        for my $row (@rows_sorted) {
            if ($prev_row) {
                $prev_row->{date_to} = mysql_round_day(yesterday($row->{date_from}), delim => '-');
            }
            push @modified_rows, $row;
            $prev_row = $row;
        }
        # проставляем в последнем диапазоне на 2038-й год дату окончания как 2038й год
        $modified_rows[-1]->{date_to} = $Settings::END_OF_TIME;
        push @converted_data, @modified_rows;
    }

    return \@converted_data;
}

=head3 _merge_schedule_to_table

=cut

sub _merge_schedule_to_table {
    my ($data, %O) = @_;

    my $merge_opts = hash_cut \%O, qw(log mod rem row_serialization_cmd row_deserialization_cmd table_name field_names dont_die dont_touch_clientids_graph);
    my @affected_clientids;
    my $has_errors = 0;
    if ($O{all_shards}) {
        for my $shard (ppc_shards()) {
            my ($clids, $errors) = _merge_schedule_to_table_in_shard($shard, $data, %$merge_opts);
            $has_errors ||= $errors;
            push @affected_clientids, @$clids;
        }
    } else {
        foreach_shard ClientID => $data, sub {
            my ($shard, $data_chunk) = @_;
            my ($clids, $errors) = _merge_schedule_to_table_in_shard($shard, $data_chunk, %$merge_opts);
            $has_errors ||= $errors;
            push @affected_clientids, @$clids;
        };
    }
    return (\@affected_clientids, $has_errors);
}

{
    # "Bad" date ranges which are stored ad slices in our DB right now.
    # Stored in the following format:
    #   {
    #       "<date_from>:<date_to>": undef
    #   }
    # Related ticket: https://st.yandex-team.ru/DIRECT-47926
    my %NOT_GLUE_DATE_RANGES = (
        '2003-01-01:2003-12-31' => undef,
        '2004-01-01:2012-01-29' => undef,
        '2003-01-01:2003-12-31' => undef,
        '2004-01-01:2038-01-19' => undef,
        '2003-01-01:2012-01-29' => undef,
        '2012-01-30:2038-01-19' => undef,
    );

    sub _group_nds_data_by_client_id {
        # группирует массив хешей {ClientID , key , date_from, date_to} по
        # ClientID и сортирует по date_from

        my ($data, $log) = @_;
        my %data_hash;

        # group by client id
        for my $row (@$data) {
            push(@{ $data_hash{$row->{ClientID}} }, {%$row});
        }

        # sort by date_from and merge slices with same values
        my $today = mysql2unix(today());
        for my $client_id (keys %data_hash) {
            my $prev_row;
            my @client_data;
            for my $row (xsort {mysql2unix($_->{date_from})} @{$data_hash{$client_id}}) {
                my $df = mysql2unix($row->{date_from});
                if ($prev_row && mysql2unix(tomorrow($prev_row->{date_to})) == $df && $prev_row->{nds} == $row->{nds}) {
                    if (exists $NOT_GLUE_DATE_RANGES{"$prev_row->{date_from}:$prev_row->{date_to}"} || exists $NOT_GLUE_DATE_RANGES{"$row->{date_from}:$row->{date_to}"}) {
                        $log->out(
                            "rows for client $client_id " .
                            "[date_from: $prev_row->{date_from}, date_to: $prev_row->{date_to}] " .
                            "and " .
                            "[date_from: $row->{date_from}, date_to: $row->{date_to}] " .
                            "arn't glued because hardcoded as exception"
                        ) if $log;
                    } else {
                        # склеиваем одинаковые НДС, идущие подряд
                        if ($log && ($df < $today)) {
                            $log->out(
                                "rows in past for client $client_id " .
                                "[date_from: $prev_row->{date_from}, date_to: $prev_row->{date_to}] " .
                                "and " .
                                "[date_from: $row->{date_from}, date_to: $row->{date_to}] " .
                                "was glued because same nds value: $row->{nds}"
                            );
                        }
                        $prev_row->{date_to} = $row->{date_to};
                        next;
                    }
                }
                push @client_data, $row;
                $prev_row = $row;
            }

            $data_hash{$client_id} = \@client_data;
        }

        return \%data_hash;
    }
}

sub _check_and_filter_nds_data {
    # получает данные из баланса и из базы, и валидирует на отсутствие
    # изменений задним числом

    my ($balance_data, $db_data, $log, $dont_die, $table, $dont_touch_clientids_graph) = @_;

    my $conv_balance_data = _group_nds_data_by_client_id($balance_data);
    # we want to log only glued data from db
    my $conv_db_data = _group_nds_data_by_client_id($db_data, $log);

    my $errors_count = 0;
    my $skiped_count = 0;
    my $today = mysql2unix(today());

    for my $client_id (sort(uniq(keys %$conv_balance_data, keys %$conv_db_data))) {
        $log->out("processing client $client_id") if $log;

        my @db_rows = @{ $conv_db_data->{$client_id} // [] };
        my @balance_rows = @{ $conv_balance_data->{$client_id} // [] };

        # если в БД еще нет графика для данного клиента, всегда принимаем то, что пришло от Баланса
        unless (@db_rows) {
            next;
        }

        if (exists $dont_touch_clientids_graph->{$client_id}) {
            # если клиент есть в списке исключений - выкидвыаем данные из баланса и базы данных
            $log->out({"save original client data" => $conv_db_data->{$client_id}, "dropped balance data" => $conv_balance_data->{$client_id}}) if $log;
            delete $conv_balance_data->{$client_id};
            delete $conv_db_data->{$client_id};
            $skiped_count++;
        } else {
            # если данные в прошлом не совпадают - выкинуть их и из данных
            # баланса и из данных базы
            my $error;
            CHECK_CLIENT: {
                my $max_row_index = max(scalar(@db_rows), scalar(@balance_rows));
                for (my $row_index = 0; $row_index < $max_row_index; $row_index++) {
                    my $db_row = $db_rows[$row_index];
                    my $balance_row = $balance_rows[$row_index];

                    if (!defined($db_row)) {
                        if (mysql2unix($balance_row->{'date_from'}) < $today) {
                            $error = "row ".($row_index + 1)." exists in data from Balance, but doesn't exist in DB data";
                            last CHECK_CLIENT;
                        } else {
                            $log->out("no more data to check for client $client_id") if $log;
                            last CHECK_CLIENT;
                        }
                    } elsif (!defined($balance_row)) {
                        if (mysql2unix($db_row->{'date_from'}) < $today) {
                            $error = "row ".($row_index + 1)." exists in DB data, but doesn't exist in data from Balance";
                            last CHECK_CLIENT;
                        } else {
                            $log->out("no more data to check for client $client_id") if $log;
                            last CHECK_CLIENT;
                        }
                    }

                    my $db_df = mysql2unix($db_row->{'date_from'});
                    my $balance_df = mysql2unix($balance_row->{'date_from'});

                    if ($db_df >= $today && $balance_df >= $today) {
                        $log->out("data in past is checked and filtered for client id $client_id") if $log;
                        last CHECK_CLIENT;
                    }

                    if ($db_df != $balance_df) {
                        $error = "different date_from from balance: $balance_row->{date_from} and from db: $db_row->{date_from}";
                        last CHECK_CLIENT;
                    }

                    my $db_dt = min(mysql2unix($db_row->{'date_to'}), $today);
                    my $balance_dt = min(mysql2unix($balance_row->{'date_to'}), $today);

                    if ($db_dt != $balance_dt) {
                        $error = "different date_to from balance: $balance_row->{date_to} and from db: $db_row->{date_to}";
                        last CHECK_CLIENT;
                    }

                    if ($db_row->{'nds'} != $balance_row->{'nds'}) {
                        $error = "different nds value from balance: $balance_row->{nds} and from db: $db_row->{nds}";
                        last CHECK_CLIENT;
                    }
                }
            }

            if ($error) {
                $errors_count++;

                my $error_message = to_json({
                        error => $error,
                        balance_data => \@balance_rows,
                        db_data => \@db_rows,
                        table => $table,
                    }, {canonical => 1});

                if (!$dont_die) {
                    die $error_message;
                } elsif ($log) {
                    $log->warn($error_message);
                }

                delete $conv_balance_data->{$client_id};
                delete $conv_db_data->{$client_id};
            }
        }
    }

    $log->out("data filtered, $errors_count was dropped, $skiped_count was skiped") if $log;

    return (
        $conv_balance_data,
        $conv_db_data,
        $errors_count,
    );
}


=head3 _merge_schedule_to_table_in_shard

=cut

sub _merge_schedule_to_table_in_shard {
    my ($shard, $data_from_balance, %O) = @_;

    my $log = $O{log};

    my %affected_clientids;
    my $has_errors = 0;
    do_in_transaction {
        my $data_from_db = _fetch_data_from_db(shard => $shard, %{hash_cut \%O, qw/mod rem field_names table_name/});

        if ($O{table_name} eq 'client_nds' || $O{table_name} eq 'agency_nds') {
            ($data_from_balance, $data_from_db, $has_errors) = _check_and_filter_nds_data($data_from_balance, $data_from_db, $log, $O{dont_die}, $O{table_name}, $O{dont_touch_clientids_graph});
        }

        my $serialized_data_from_balance = _get_serialized_hash($data_from_balance, row_serialization_cmd => $O{row_serialization_cmd});
        my $serialized_data_from_db = _get_serialized_hash($data_from_db, row_serialization_cmd => $O{row_serialization_cmd});

        my $serialized_diffs = hash_diff $serialized_data_from_db, $serialized_data_from_balance;
        return [] unless %$serialized_diffs;

        # для удалённых записей hash_diff оставит undef в качестве значения
        my ($records_to_insert, $records_to_delete) = part {$serialized_diffs->{$_} ? 0 : 1} keys %$serialized_diffs;

        if ($records_to_delete && @$records_to_delete) {
            my $msg_prefix_guard;
            if ($log) {
                $msg_prefix_guard = $log->msg_prefix_guard("[delete_$O{table_name}]");
            }
            for my $serialized_record (@$records_to_delete) {
                # для графика НДС, вероятно, этот блок кода не случается
                my $record = $O{row_deserialization_cmd}->($serialized_record);
                $log->out($record) if $log;
                do_delete_from_table(PPC(shard => $shard), $O{table_name}, where => hash_cut($record, @{$O{field_names}})); 
                $affected_clientids{$record->{ClientID}} = 1;
            }
        }

        if ($records_to_insert && @$records_to_insert) {
            my $msg_prefix_guard;
            if ($log) {
                $msg_prefix_guard = $log->msg_prefix_guard("[insert_$O{table_name}]");
            }
            my @to_insert;
            for my $serialized_record (@$records_to_insert) {
                my $record = $O{row_deserialization_cmd}->($serialized_record);
                $log->out($record) if $log;
                push @to_insert, [map {$record->{$_}} @{$O{field_names}}];
                $affected_clientids{$record->{ClientID}} = 1;
            }
            my $field_names_sql = join ',', map {sql_quote_identifier($_)} @{$O{field_names}};
            do_mass_insert_sql(PPC(shard => $shard), "INSERT IGNORE INTO $O{table_name} ($field_names_sql) VALUES %s", \@to_insert);
        }
    };

    return (
        [keys %affected_clientids],
        $has_errors,
    );
}

sub _fetch_data_from_db {
    my %O = @_;

    my $field_names_sql = join ',', map {sql_quote_identifier($_)} @{$O{field_names}};
    if ($O{mod} == $Settings::MAX_INT_VAL) {
        # данные нужны по единственному клиенту, выбираем оптимально
        my $client_id = $O{rem};
        return get_all_sql(PPC(shard => $O{shard}), [qq/
            SELECT $field_names_sql
            FROM $O{table_name}
         /, WHERE => {ClientID => $client_id},
        ]);
    } else {
        # данные нужны по пачке разношардовых клиентов, выбираем ковровой бомбардировкой
        return get_all_sql(PPC(shard => $O{shard}), qq/
            SELECT $field_names_sql
            FROM $O{table_name}
            WHERE ClientID % ? = ?/, $O{mod}, $O{rem});
    }
}

sub _get_serialized_hash {
    my ($data, %O) = @_;

    my %serialized_hash;

    if (ref $data eq 'ARRAY') {
        for my $row (@$data) {
            my $serialized_row = $O{row_serialization_cmd}->($row);
            $serialized_hash{$serialized_row} = 1;
        }
    } elsif (ref $data eq 'HASH') {
        for my $client_id (keys %$data) {
            for my $row (@{ $data->{$client_id} }) {
                my $serialized_row = $O{row_serialization_cmd}->($row);
                $serialized_hash{$serialized_row} = 1;
            }
        }
    } else {
        confess 'unsupported data format';
    }

    return \%serialized_hash;
}

sub _serialize_nds_row {
    my ($row) = @_;

    return join ':',
        _serialize_int($row->{ClientID}, 1),
        _serialize_date($row->{date_from}),
        _serialize_date($row->{date_to}),
        _serialize_float($row->{nds});
}

sub _deserialize_nds_row {
    my ($serialized_row) = @_;

    my @fields = qw/ClientID date_from date_to nds/;
    my @values = split /:/, $serialized_row;
    my %row = zip @fields, @values;
    return \%row;
}

sub _serialize_int {
    my ($int, $min_value) = @_;

    die "invalid int value: $int" unless is_valid_int($int, $min_value);
    return int($int);
}

sub _serialize_date {
    my ($date) = @_;

    return unix2human(mysql2unix($date), '%Y%m%d');
}

sub _serialize_float {
    my ($float) = @_;

    die "invalid float value: $float" unless is_valid_float($float) && $float >= 0;
    return sprintf('%.2f', $float);
}

sub _resend_changed_client_campaigns_to_BS {
    my ($affected_clientids, %O) = @_;

    return unless $affected_clientids && @$affected_clientids;

    my $log = $O{log};

    # ставим на ленивую переотправку кампании тех клиентов, у которых есть изменения, чтобы отправить новый график в БК
    my $client_id2role = get_hash_sql(PPC(shard => 'all'), ["SELECT ClientID, role FROM clients", WHERE => { ClientID => $affected_clientids }]);
    my ($client_clientids, $agency_clientids) = part { $client_id2role->{$_} eq 'agency' ? 1 : 0 } @$affected_clientids;
    # обрабатываем субклиентов агентств
    for my $agency_clientid (@$agency_clientids) {
        $log->out("Fetching multicurrency campaigns for agency ClientID $agency_clientid with changed graph:") if $log;
        my $subclient_uids = get_one_column_sql(PPC(shard => 'all'), "SELECT chief_uid FROM clients WHERE agency_client_id = ?", $agency_clientid);
        for my $chunk (sharded_chunks(uid => $subclient_uids, $UIDS_RESYNC_CHUNK_SIZE)) {
            $log->out({uid => $chunk->{uid}}) if $log;
            my $cids = get_one_column_sql(PPC(shard => $chunk->{shard}), ['
                SELECT c.cid
                FROM campaigns c
                JOIN users u on u.uid = c.uid
             ', WHERE => {
                    'c.uid' => $chunk->{uid},
                    'u.not_resident' => 'No',
                    'c.AgencyID' => $agency_clientid,
                    _TEXT => 'IFNULL(c.currency, "YND_FIXED") <> "YND_FIXED"',
            }]);
            $log->out({cid => $cids}) if $log;
            BS::ResyncQueue::bs_resync_camps($cids, priority => BS::ResyncQueue::PRIORITY_NDSDISCOUNT_GRAPH_CHANGED);
        }
    }
    # обрабатываем самоходных клиентов и субклиентов-нерезидентов
    $log->out('Fetching multicurrency campaigns for client chief uids with changed graph:') if $log;
    my $affected_clientid2chief_uid = rbac_get_chief_reps_of_clients($client_clientids);
    my @affected_client_chief_uids = values %$affected_clientid2chief_uid;
    for my $chunk (sharded_chunks(uid => \@affected_client_chief_uids, $UIDS_RESYNC_CHUNK_SIZE)) {
        $log->out({uid => $chunk->{uid}}) if $log;
        my $cids = get_one_column_sql(PPC(shard => $chunk->{shard}), ['
            SELECT c.cid
            FROM campaigns c
            JOIN users u on u.uid = c.uid
         ', WHERE => {
                'c.uid' => $chunk->{uid},
                _OR => ['c.AgencyID' => 0, 'u.not_resident' => 'Yes'],
                _TEXT => 'IFNULL(c.currency, "YND_FIXED") <> "YND_FIXED"',
        }]);
        $log->out({cid => $cids}) if $log;
        BS::ResyncQueue::bs_resync_camps($cids, priority => BS::ResyncQueue::PRIORITY_NDSDISCOUNT_GRAPH_CHANGED);
    }
}

sub _die_log {
    my $log = shift;

    if ($log) {
        $log->die(@_);
    } else {
        die @_;
    }
}

=head2 _filter_agency

    Отбирает из переданного массива только ClientID агентств

    $data => $only_agencies => [
        {
            ClientID =>
            date_from =>
            date_to =>
            nds =>
        },
        ...
    ];
    $only_agencies = _filter_agency($data);

=cut

sub _filter_agency {
    my ($data) = @_;

    return [] unless $data && @$data;

    my @clientids = uniq map { $_->{ClientID} } @$data;
    my $agclid2uid = rbac_get_chief_reps_of_agencies(\@clientids);
    return [ grep { exists $agclid2uid->{ $_->{ClientID} } } @$data ];
}

=head3 _get_country_today_nds($country_id, $is_agency)

    общий код для get_country_today_nds_for_agency и get_country_today_nds_not_for_agency, отвечающий
    за получение данных из баланса и их фильтрацию

=cut

sub _get_country_today_nds {
    my ($country_id, $is_agency) = @_;

    die 'no country_id given' unless $country_id;
    die 'no is_agency given' unless defined $is_agency;

    my $nds_info = Yandex::Balance::balance_get_nds_info($Settings::SERVICEID{direct});
    my @our_country_data = grep {$_->{REGION_ID} == $country_id && $_->{AGENCY} == $is_agency} @$nds_info;
    if (scalar(@our_country_data) == 1) {
        return $our_country_data[0]->{NDS_PCT};
    } else {
        return undef;
    }
}

=head2 get_country_today_nds_for_agency

    Возвращает взятое из Баланса значение НДС в указанной стране, действующее сегодня для агентств и субклиентов
    Если Баланс НДСа в этой стране не знает, возвращает undef

    $nds_value = Client::NDSDiscountSchedule::get_country_today_nds_for_agency($geo_regions::KAZ);
    $nds_value => 0

=cut

sub get_country_today_nds_for_agency {
    return _get_country_today_nds(shift, 1);
}

=head2 get_country_today_nds_not_for_agency

    Возвращает взятое из Баланса значение НДС в указанной стране, действующее сегодня для самостоятельных и сервисируемых клиентов
    Если Баланс НДСа в этой стране не знает, возвращает undef

    $nds_value = Client::NDSDiscountSchedule::get_country_today_nds_not_for_agency($geo_regions::KAZ);
    $nds_value => 12

=cut

sub get_country_today_nds_not_for_agency {
    return _get_country_today_nds(shift, 0);
}

sub _get_clientids_for_fake_byn_nds {
    my ($client_ids) = @_;

    return get_one_column_sql(PPC(ClientID => $client_ids), ['SELECT ClientID FROM clients',
        WHERE => { 
            ClientID => SHARD_IDS,
            _OR => {
                work_currency => "BYN",
                _AND => { country_region_id => $geo_regions::BY, _OR => {work_currency => "YND_FIXED", work_currency__is_null => 1}}
            }
        }]);
}

sub _get_clientids_for_fake_quasi_currency_nds {
    my ($client_ids) = @_;

    return get_one_column_sql(PPC(ClientID => $client_ids), ['SELECT c.ClientID FROM clients c
        JOIN clients_options co on c.ClientID=co.ClientID',
        WHERE => {
            'c.ClientID' => SHARD_IDS,
            'c.country_region_id' => $geo_regions::KAZ,
            'c.work_currency' => 'KZT',
            'co.is_using_quasi_currency' => 1,
        }]);
}


1;
