package CommonMaps;

=head1 DESCRIPTION
    
    $Id:$
    
    Модуль для работы с Я.Картами в интерфейсе Директа

=cut 

use strict;
use warnings;

use utf8;

use Carp qw/croak/;

use Settings;

use Yandex::HashUtils;
use Yandex::ListUtils;
use Yandex::Retry;
use Yandex::I18n;

use Yandex::DBTools;
use Yandex::DBShards;
use HashingTools;
use Tools;
use TextTools qw/smartstrip/;
use Metro;

use List::MoreUtils qw/all/;

use Digest::MD5 qw(md5 md5_hex md5_base64);

use JavaIntapi::GeosearchSearchObject;

use base qw/Exporter/;
our @EXPORT_OK = qw/
    check_address_map
/;

# Координаты и границы для дефолтной точки (если гео-кодер ничего не нашел)
our $DEFAULT_AUTO_BOUNDS = '11.066975,43.267788,177.356039,74.690828';
our $DEFAULT_AUTO_POINT  = '37.611643,55.819692';

=head3 check_address_map(form, options)

    check_address_map(
        {country => 'Россия', city => 'Москва', street => 'Льва Толстого'}, 
        {ClientID => 42}
    );

    Пытается получить координаты для адреса на карте, обращаясь к геокодеру.
    Если задан ClientID, сначала ищем адрес в базе.
    Также определяет метро, если возможно.
    Если адрес найден, и передали ClientID, сохраняет адрес и точку в базе.

    Возвращает подготовленную структуру с описанием точки и куска карты.

    form - адресные поля (country, city,.. - поля, относящиеся к адресу в
           vcards)
           координаты точки и другие поля структуры, 
           возвращаемой API Яндекс.Карт

    options - ClientID  если он отсутствует, найденный адрес не будет
                        сохраняться в базу

              dont_save не сохранять адрес в базу

              ignore_manual_point  игнорировать отсутствие ручной точки,
                                   что бы это не значило) 

            no_metro_kludge — флаг для дезактивации костыля привязки станции метро (+не ходить за метро в геокодер)


=cut

sub check_address_map {
    my ($form, $options) = @_;
    $options ||= {};

    my $client_id = $options->{ClientID};
    my $host = $options->{host} ? Tools::get_direct_domain($options->{host}) 
                                : 'direct.yandex.ru';
    my $address = _filter_address( _get_address_form($form) );
    my $map; 
    if ($client_id) {
        $map = check_cached_map($address, $client_id);
    }
    $map ||= {};

    unless ($map->{auto_point}) {

        my $map_from_geocoder = {};

        # Если произойдет ошибка геокодера, то в хеше $map НЕ будет ключа auto_precision
        # и в функции _save_address мы НЕ будем сохранять точку (если нет ручной)
        eval {
            retry tries => 2, sub {

                my ($maps, $error) = JavaIntapi::GeosearchSearchObject->_search_address($address);
                if  ($error) {
                    warn $error->{server_status} . ' ' . $error->{server_error};
                    return;
                }
                $map_from_geocoder = _select_first_point($maps, $form->{city});

                if (!$options->{no_metro_kludge}
                    && defined $map_from_geocoder->{precision}
                    && $map_from_geocoder->{precision} ne 'other') 
                {
                    my $metro = search_metro((map { $map_from_geocoder->{point}->{$_} } qw/x y/), $form->{city}, $host);
                    $map_from_geocoder->{metro} = $metro;
                }
            };
        };
        warn $@ if $@;
        $map = hash_merge $map, $map_from_geocoder;
    }

    if (!$options->{no_metro_kludge}
        and $map->{point}
        and ref $map->{point} eq 'HASH'
        and keys %{$map->{point}}
        and not defined $map->{metro} 
        and $map->{precision} ne 'other'
    ) {
        eval {
            retry tries => 2, sub {
                my $metro = search_metro(@{$map->{point}}{qw/x y/}, $form->{city}, $host);
                $map->{metro} = $metro;
            };
        };
        warn $@ if $@;
    }

    # игнорируем отсутствие ручной точки (для ранних версий API и XLS)
    if ($options->{ignore_manual_point}) {
        $form->{manual_point} = $map->{manual_point};
        $form->{manual_bounds} = $map->{manual_bounds};
    }

    return $map if $options->{dont_save};


    if ($client_id) {
        my $result  = _save_address($client_id, $map, $address, $form);
        $map->{aid} = $result->{aid};
    }
    
    return $map;
}

=head3 _get_address_form

    Получает из формы данные для адреса и преобразует в строку.

=cut

sub _get_address_form {
    my $form = shift;

    return defined $form->{address}
        ? _join_address_form_fields($form, qw/country city address/)
        : _join_address_form_fields($form->{vcard} // $form, qw/country city street house build/);
}

sub _join_address_form_fields {
    my ($form, @fields) = @_;
    return join ',', (grep { defined $_ && length $_ } map { smartstrip($_) } map { $form->{$_} } @fields);
}



=head3 _save_address($ClientID, $map, $address_string, $form)

    Сохраняет результат запроса к геокодеру в базе

    Принимает следующие аргументы (обязательные):
        ClientID        - ID клиента-владельца визитки с таким адресом
                            важно нигде не потерять по пути ulogin для спец-ролей и не прислать ClientID оператора
        map             - ссылка на хеш с информацией о точке, которую следует сохранить
                            источником этого хеша преимущественно является гео-кодер и наш кеш точек
                            используются следующие ключи:
                                point => {x =>, y => }
                                bound => {x1 => , y1 => , x2 => , y2 => }
                                manual => {point => {...}, bound => {...}}
                                auto_precision
                                precision
                                kind
                                metro => {region_id => }
        address_string  - строка с адресом вида "россия,санкт-петербург,ул.калинина,10"
    (опционально)
        form            - ссылка на хеш с "формой" из интерфейса. используются следующие поля:
            auto_point      - автоматические координаты. используются "магическим образом" как признак того,
                                что из интерфейса действительно пришли РУЧНЫЕ координаты (отработала ajaxValidateStreet)
            manual_point    - новые координаты ручной точки - будут использованы ВМЕСТО map->{manual}->{point}
                                (это строка, имеет вид "37.611643,55.819692")
                                если определено auto_point и manual_point - undef или пустая строка - ручная точка будет УДАЛЕНА из кеша!
            manual_bounds   - новые ручные границы обзора - будут использованы ВМЕСТО map->{manual}->{bounds}
                                (это строка, имеет вид "11.066975,43.267788,177.356039,74.690828")

    Возвращает ссылку на хеш вида: 
    {
        aid     => address_id,  #   для vcards
        status  => new|updated|skipped|not_saved,
    }
    Статус показывает, в результате чего (какой) address_id мы возвращаем:
        new     - создали новую запись в таблице addresses
        updated - обновили какие-то данные в уже имеющейся записи addresses. вернули aid этой записи.
        skipped - данные совпадают, поэтому ничего в таблице не меняем. вернули aid имеющейся записи.
        not_saved - плохая точность авто-точки и нет ручной точки - ничего не сохраняли. (aid=0)

=cut

sub _save_address {
    my ($ClientID, $map, $address, $form) = @_;

    my $address_hash = url_hash_utf8($address);

    my $db_address = get_one_line_sql(PPC(ClientID => $ClientID), [q/
            SELECT aid, kind, `precision`, map_id, map_id_auto
                 , CONCAT_WS(',', maps.x, maps.y) AS manual_point
                 , CONCAT_WS(',', maps.x1, maps.y1, maps.x2, maps.y2) AS manual_bounds
                 , CONCAT_WS(',', maps_auto.x, maps_auto.y) AS auto_point
                 , CONCAT_WS(',', maps_auto.x1, maps_auto.y1, maps_auto.x2, maps_auto.y2) AS auto_bounds
                 , metro
              FROM addresses
                   LEFT JOIN maps ON addresses.map_id = maps.mid
                   LEFT JOIN maps maps_auto ON addresses.map_id_auto = maps_auto.mid
        /, where => {
            ahash    => $address_hash,
            address  => $address,
            ClientID => $ClientID,
        }
    ]) || {};

    # ручную точку задали явно
    if ($form && $form->{manual_point} && $form->{manual_bounds}) {
        my ($x, $y) = $form->{manual_point} =~ m/^(-?\d+(?:\.\d+)?) , (-?\d+(?:\.\d+)?)$/x;
        my ($x1, $y1, $x2, $y2) = $form->{manual_bounds} =~ m/^(-?\d+(?:\.\d+)?) , (-?\d+(?:\.\d+)?) , (-?\d+(?:\.\d+)?) , (-?\d+(?:\.\d+)?)$/x;
        if (all {defined $_} ($x, $y, $x1, $y1, $x2, $y2)) {
            $map->{manual}->{point} = {
                x => $x, 
                y => $y,
            };
            # TODO!!! исправить bound -> bounds (и в JavaIntapi::GeosearchSearchObject)
            $map->{manual}{bound} = {
                x1 => $x1, 
                y1 => $y1, 
                x2 => $x2, 
                y2 => $y2,
            };
        }
    }

    # ручную точку удаляют
    if ( $form &&
        ( 
            ( $form->{api_call} && !$form->{auto_point} && !$form->{manual_point} )
        ||
            ( $form->{auto_point} && !$form->{manual_point} && !$form->{manual_bounds} )
        )
    )
    {
        delete $map->{manual};
    }


    # если новая точка не достаточна точна или её нет вообще (ошибка геокодера)
    # и нет ручной точки, то точку не сохраняем
    if (! exists $map->{manual}
        && (! defined $map->{auto_precision}
            || $map->{auto_precision} !~ m/^(?:exact|number|near|street)$/)
    ) {
        return {
            aid     => 0,
            status  => 'not_saved',
        };
    }

    # если вручную не установлена точка, то берем копию той, что пришла из гео-кодера
    unless (exists $map->{manual}) {
        for my $k (qw/x y/) {
            $map->{manual}->{point}->{$k} = $map->{point}->{$k};
        }
        for my $k (qw/x1 y1 x2 y2/) {
            $map->{manual}->{bound}->{$k} = $map->{bound}->{$k};
        }
    }

    # сохраняем авто-точку и ручную точку в maps
    my $map_id_auto;
    if ($map->{auto_point} && (($db_address->{auto_point} || '') . ';' . ($db_address->{auto_bounds} || '') eq ($map->{auto_point} || '') . ';' . ($map->{auto_bounds} || ''))) {
        $map_id_auto = $db_address->{map_id_auto};
    } elsif ($map->{bound}->{x1}) { # не ясно, можно ли проверять по defined. Что там в кеше точек и геокодере
        my $point_data = hash_merge {},
                    hash_cut($map->{point}, qw/x y/),
                    hash_cut($map->{bound}, qw/x1 y1 x2 y2/);
        $map_id_auto = save_map_point(PPC(ClientID => $ClientID), $point_data);
    }

    my $map_id_manual;
    if ($form->{manual_point} && (($db_address->{manual_point} || '') . ';' . ($db_address->{manual_bounds} || '') eq ($form->{manual_point} || '') . ';' . ($form->{manual_bounds} || ''))) {
        $map_id_manual = $db_address->{map_id};
    } elsif (defined $map->{manual}->{bound}->{x1}) {
        my $point_data = hash_merge {},
                    hash_cut($map->{manual}->{point}, qw/x y/),
                    hash_cut($map->{manual}->{bound}, qw/x1 y1 x2 y2/);
        $map_id_manual = save_map_point(PPC(ClientID => $ClientID), $point_data);
    } else {
        $map_id_manual = $map_id_auto;
    }

    my $result = {};

    if ( !$db_address->{aid} ) {
        my $new_aid = get_new_id('aid');
        $result->{aid} = do_insert_into_table(PPC(ClientID => $ClientID), 'addresses', {
                aid         => $new_aid,
                ClientID    => $ClientID,
                address     => $address,
                map_id      => $map_id_manual,
                map_id_auto => $map_id_auto,
                ahash       => $address_hash,
                precision   => $map->{precision},
                kind        => $map->{kind},
                metro       => $map->{metro}->{region_id},
            },
            on_duplicate_key_update => 1,
            key => 'aid',
        ) || $new_aid;
        $result->{status} = 'new';

    } else {
        my $info1 = join ',', map {$db_address->{$_} || ''} 
                                qw/map_id map_id_auto kind precision metro/;
        my $info2 = join ',', map { $_ || '' } 
                                $map_id_manual, $map_id_auto,
                                (map { $map->{$_} } qw/kind precision/), 
                                $map->{metro}{region_id};

        if ($info1 ne $info2) {
            do_update_table(PPC(ClientID => $ClientID), 'addresses', {
                    precision           => $map->{precision},
                    kind                => $map->{kind},
                    map_id              => $map_id_manual,
                    map_id_auto         => $map_id_auto,
                    metro               => $map->{metro}->{region_id},
                    logtime__dont_quote => 'NOW()',
                }, where => {
                    aid      => $db_address->{aid},
                }
            );
            $result->{status} = 'updated';
        } else {
            $result->{status} = 'skipped';
        }

        $result->{aid} = $db_address->{aid};
    }

    return $result;
}

=head2 save_map_point($ppc, {x=>..., y=>..., x1=>..., y1=>..., x2=>..., y2=>...})

    сохранить точку в таблицу maps, вернуть id
    если такая точка уже существует - вернуть один из существующих id

=cut
sub save_map_point {
    my ($ppc, $point_data) = @_;
    if (@{xdiff([keys %$point_data], [qw/x y x1 y1 x2 y2/])}) {
        croak "Incorrect map point data: ".join(',', keys %$point_data);
    }
    my $rounded = hash_map {sprintf "%.06f", $_} $point_data;

    my $mid = get_one_field_sql($ppc, ["SELECT mid FROM maps", WHERE => $rounded]);
    if (!$mid) {
        $mid = get_new_id('maps_id');
        do_insert_into_table($ppc, 'maps', hash_merge({mid => $mid}, $rounded));
    }
    return $mid;
}

=head3 Возможные значения аттрибута kind 

    Тип объекта, определенного геокодером по указанному адресу:
    
    house	    отдельный дом	        Россия, Москва, улица Тверская, 7
    street	    улица	                Россия, Москва, улица Тверская
    metro	    станция метро	        Россия, Москва, Филевская линия, метро Арбатская
    district	район города	        Россия, Москва, Северо-Восточный административный округ
    locality	населённый пункт: город/поселок/деревня/село/...	Россия, Санкт-Петербург
    area	    район области	        Россия, Ленинградская область, Выборгский район
    province	область	                Россия, Нижегородская область
    country	    страна	Великобритания
    hydro	    река,озеро,ручей,водохранилище...	Россия, река Волга
    railway	    ж.д. станция	        Россия, Москва, Курский вокзал
    route	    линия метро / шоссе / ж.д. линия	Россия, Москва, Сокольническая линия
    vegetation	лес, парк...	        Россия, Санкт-Петербург, Михайловский сад
    cemetery	кладбище	            Россия, Москва, Ваганьковское кладбище
    bridge	    мост	                Россия, Санкт-Петербург, Аничков мост
    km	        километр шоссе	        Россия, Москва, МКАД, 42-й километр
    other	    разное	                Россия, Свердловская область, Екатеринбург, Шабур остров
=cut

=head3 _select_first_point(map, city)

    Возвращает первую запись результата поиска, подходящую по точности и типу найденного объекта

=cut
{
    my %PRECISION_SORT = (exact => 0, number => 1, near => 2, street => 3, other => 4);
sub _select_first_point
{
    my $smap = shift;
    my $city = shift;
    
    # выбираем первый вариант, с учетом допустимой точности
    # показываем карту, если найдены дом, улица, город, метро, определенный км шоссе, жд станция, и другие точные объекты.
    # не показываем - кладбище, мосты, реки, озера, области, страны, леса и парки.
    
    # затираем название области из названия города
    # его дописывает гео-селектор и в результате названия городов в Картах и у нас не совпадает
    $city ||= '';
    
    my ($adm_area) = $city =~ /\((.*?)\)/;
    $city =~ s/\s*\(.*?\)//i;
    
    my @objects = (sort {$PRECISION_SORT{ $a->{precision} } <=> $PRECISION_SORT{ $b->{precision} }}
                  grep {
                      my $smap = $_;
                      _valid_point($smap, city => $city, adm_area => $adm_area)
                   } @{$smap->{points} || []});
    @objects =   (grep {
                      my $smap = $_;
                      _valid_point($smap)
                   } @{$smap->{points} || []}) unless @objects;

    my $element = scalar @objects ? $objects[0] : undef;
    if ($element) {
        my $result = {};
        hash_merge $result, hash_cut($smap, grep { !/^points$/ } keys %{$smap||{}});
        hash_merge $result, $element;
        return $result;
    }

    # если ничего не найдено то зуммим на всю Россию и точку ставим в Москву
    return {
        manual_bounds => '',
        manual_point => '',
        auto_bounds => $DEFAULT_AUTO_BOUNDS,
        auto_point => $DEFAULT_AUTO_POINT,
        auto_precision => 'other',
    };
}}

=head2 _valid_point(smap, OPT)

    Выбираем точку отвечающую следующим требованиям:
        1. тип найденного объекта : house|street|locality|metro|km|railway|other
        2. в названии найденного города содержится название города, указанное пользователем (проверяем как city, так и sub-city)
        3. если присутствует уточнение для города по названию области - то ищем совпадение области (проверяем adm-area и sub-adm-area)

=cut

sub _valid_point
{
    my ($smap, %OPT) = @_;

    return 0 unless $smap->{kind} =~ /^(house|street|locality|metro|km|railway|other)$/i;

    return 0 unless (! $OPT{city} || _check_city($OPT{city}, @{$smap}{qw/city sub-city administrative-area/}));

    return 0 unless (! $OPT{adm_area}
              || ! ($smap->{'administrative-area'} || $smap->{'sub-administrative-area'})
              || $smap->{'administrative-area'} && $OPT{adm_area} =~ /\Q$smap->{'administrative-area'}\E/i 
              || $smap->{'sub-administrative-area'} && $OPT{adm_area} =~ /\Q$smap->{'sub-administrative-area'}\E/i
        );

    return 1;
}

sub _check_city
{
    my $icity = lc shift;
    # administrative-area иногда(?) не приходит. боремся с Use of uninitialized value $_ in lc at
    my @cities = map {lc $_} grep {defined $_} @_;
    
    # Может не совпадать в запросе и ответе геокодера. Пример: запрос - Орёл, ответ - Орел.
    $icity =~ tr/ё/е/;

    foreach my $city (@cities) {
        $city =~ tr/ё/е/;
        next if $city ne $icity;
        return 1;
    }

    return 0;
}

=head3 check_cached_map(address_string)

    Проверяет наличие координат точки(адреса) у нас в базе

=cut
sub check_cached_map {
    my ($address, $ClientID) = @_;
    
    my $data = get_one_line_sql(PPC(ClientID => $ClientID), [q/
            SELECT aid, ClientID, map_id, map_id_auto, address, metro, `precision`, kind, logtime
                 , CONCAT_WS(',', maps.x, maps.y) AS manual_point
                 , CONCAT_WS(',', maps.x1, maps.y1, maps.x2, maps.y2) AS manual_bounds
                 , CONCAT_WS(',', maps_auto.x, maps_auto.y) AS auto_point
                 , CONCAT_WS(',', maps_auto.x1, maps_auto.y1, maps_auto.x2, maps_auto.y2) AS auto_bounds
                 , maps_auto.x, maps_auto.y
                 , maps_auto.x1, maps_auto.y1, maps_auto.x2, maps_auto.y2
                 , maps.x AS m_x, maps.y AS m_y
                 , maps.x1 AS m_x1, maps.y1 AS m_y1, maps.x2 AS m_x2, maps.y2 AS m_y2
              FROM addresses adr
                   LEFT JOIN maps ON adr.map_id = maps.mid
                   LEFT JOIN maps maps_auto ON adr.map_id_auto = maps_auto.mid
        /, where => {
            ahash => url_hash_utf8($address),
            ClientID => $ClientID,
        },
    ]);
    
    return {} unless $data; # Здесь было and $data->{metro}.
    # но в таких случаях в городах без метро при пересохранении адреса терялась ручная точка

    return {
        aid => $data->{aid},
        text => $data->{address},
        metro => { region_id => $data->{metro} },
        found => ($data->{x} ? 1 : 0),
        precision => $data->{precision},
        auto_precision => $data->{precision},
        kind => $data->{kind},

        point => {
            x => $data->{x},
            y => $data->{y},
        },

        bound => {
            x1 => $data->{x1},
            y1 => $data->{y1},
            x2 => $data->{x2},
            y2 => $data->{y2},
        },

        # DIRECT-21854
        # Чтобы при последующем сохранении адреса, минуя интерфейс ($form->{manual_point})
        # не потерять ручную точку из "кеша".
        manual => {
            point => {
                x => $data->{m_x},
                y => $data->{m_y},
            },
            # TODO!!! исправить bound -> bounds (и в JavaIntapi::GeosearchSearchObject)
            bound => {
                x1 => $data->{m_x1},
                y1 => $data->{m_y1},
                x2 => $data->{m_x2},
                y2 => $data->{m_y2},
            },
        }, #///

        manual_point  => $data->{manual_point},
        manual_bounds => $data->{manual_bounds},
        auto_point    => $data->{auto_point},
        auto_bounds   => $data->{auto_bounds},
    };
}

=head3 _filter_address(string)

    Удаляет лишние пробелы и запятые

=cut

sub _filter_address
{
    my ($address) = @_;

    $address = lc $address;
    $address =~ s/\s+/ /gsi;
    $address =~ s/,(?:\s*?,)+/,/gsi;

    return $address;
}

=head2 get_map_key($host)

    Возвращает API ключ для карт

=cut

sub get_map_key {
    $Settings::MKEYS{Tools::get_direct_domain(shift)}
}

=head3 @addresses_fields_to_copy

    Список полей адреса, подлежащий копированию.

=cut
my @addresses_fields_to_copy = qw/
    map_id
    map_id_auto
    address
    metro
    ahash
    kind
    precision
/;

=head3 @maps_fields_to_copy

    Список полей привязки к картам, подлежащих копированию

=cut
my @maps_fields_to_copy = qw/
    x
    y
    x1
    y1
    x2
    y2
/;

=head2 copy_address($old_address_id, $old_chief_uid, $new_client_id)

    Копирует адрес от одного клиента - другому. Вынесено из Campaign::Copy::copy_camp
    Параметры:
        old_address_id  - id записи, которую нужно скопировать
        old_chief_uid   - uid владельца записи (нажно для определения шарда для выборки записи)
        new_client_id   - ClientID клиента, которому копируем адрес

    Возвращает id новой записи или undef (например, если старая запись не найдена)

=cut
sub copy_address($$$) {
    my ($old_address_id, $old_chief_uid, $new_client_id) = @_;
    my $new_address_id;

    my $old_address = get_one_line_sql(PPC(uid => $old_chief_uid),
        ['SELECT', sql_fields('aid', @addresses_fields_to_copy), 'FROM addresses', where => { aid => $old_address_id } ]
    );

    if ($old_address && $old_address->{aid}) {
        my $new_aid = get_new_id('aid');
        my $new_address = hash_merge(
            { aid => $new_aid, ClientID => $new_client_id },
            hash_cut($old_address, @addresses_fields_to_copy),
        );
        
        my @map_ids = grep {$_} map {$new_address->{$_}} qw/map_id map_id_auto/;
        if (@map_ids) {
            my $map_data = get_hashes_hash_sql(PPC(uid => $old_chief_uid), [
                    'SELECT', sql_fields('mid', @maps_fields_to_copy),
                    'FROM maps',
                    WHERE => {mid => \@map_ids}]);
            foreach my $mid (@map_ids) {
                my $new_mid = save_map_point(PPC(ClientID => $new_client_id), hash_cut($map_data->{$mid}, @maps_fields_to_copy));
                $map_data->{$mid}->{mid} = $new_mid;
            }
            foreach my $field (qw/map_id map_id_auto/){
                next unless $new_address->{$field};
                $new_address->{$field} = $map_data->{$new_address->{$field}}->{mid};
            }
        }

        $new_address_id = do_insert_into_table(PPC(ClientID => $new_client_id), 'addresses', $new_address,
            on_duplicate_key_update => 1, key => 'aid') || $new_aid;
    } else {
        $new_address_id = undef;
    }

    return $new_address_id;
}

1;
