package Currencies;

# $Id$

=head1 NAME
    
    Currencies

    TODO переименовать в Currency

=head1 DESCRIPTION

    Модуль для констант и функций, относящихся к денежным суммам и к валютам (единицам измерения денег).

=head1 COMMENTS

    Заметки к внедрению мультивалютности в Директе 

    Состояние "до мультивалютности": 
      + все суммы на текстовых кампаниях измеряются в у.е., 
      + в Легком интерфейсе все суммы пересчитываются в "как бы рубли", "как бы гривны" и т.п. по фиксированному коэффициенту
      + в Профи пишутся примечания "1 у.е. = 30 руб./130 тенге" с использованием тех же фиксированных коэффициентов. 
      + по тем же фиксированным коэффициентам пересчитываются итоговые суммы в прогнозах/медиапланах и т.п. 
    
    Желаемое идеальное состояние: 
      - на кампании хранится название валюты, в которой она считается, и все суммы на кампании хранятся именно в ней
      - для некоторых валют доступны "алиасы" с фиксированным коэффициентом. Пример: рубль -- алиас для у.е. 
        Суммы на кампании хранятся в основной валюте, а для вывода в интерфейсе домножаются на коэфф. 
      - примечаний "1 у.е. = ..." вообще не показывается (или для у.е./руб. можно оставить?)
      - для различных суммарных отчетов доступны текущие курсы валют
      - в интерфейсе итоговые суммы по кампаниям с разными валютами пересчитываются в превалирующую валюту клиентов или кампаний

    Предполагаемые порядок работ:
      + существующие "валюты" переименовываются в "как бы валюты" (pseudo_currency). 
        Соответственно меняются имена переменных и функций
      - шаблоны получают только id валюты/алиаса, а для вывода названий/сумм/курсов используются функции из Template::Plugin::Currencies
      - в Легком псевдовалюты превращаются (внутри) в алиасы, в какой-то момент отрывается привязка к домену (.ru/.ua/.kz/...) 
        ??? в какой-то момент надо будет отменить существующие Легкие гривны и тенге. Когда?
      - в Профи псевдовалюты исчезают
        * какое-то время может остаться функция conv_unit_explanation, которая выводит справку "1 у.е. = ..." (по домену/настройке/...)
        * в прогнозах/рекомендациях/медиапланах можно показывать предварительные расчеты в любой валюте. 
          Валюту определять по домену/настройке, для пересчета брать текущий курс
      - отдаем алиас "рубль" пользователям
      - (собственно Мультивалютность) появляются настоящие другие валюты
        * прием курсов для расчета прогнозов и т.п. 
        * выбор и отображение в интерфейсе, экспорт в БК, прием статистики и т.д. и т.п.
        * и все остальное, что потребуется

    По коду в комментариях с пометкой "multicurrency" -- дополнительные заметки к реализации

=cut

use Direct::Modern;

use Property;

use Settings;
use TextTools;

use Carp;
use List::Util qw/any max min none/;
use Yandex::HashUtils qw/hash_copy hash_merge/;
use String::Numeric qw/is_float/;

use Yandex::I18n;
use Yandex::DBTools;
use Yandex::ScalarUtils;
use GeoTools qw/get_geo_projection/;

use Hash::Util;
use JSON;

use base qw/Exporter/;
our @EXPORT = qw(
    get_currencies_list

    get_currency_constant

    sql_expr_for_currency_sorting

    is_valid_currency
    remove_nds
    remove_bonus
    
    add_bonus
    calc_bonus
    add_nds
    campaign_remove_nds_and_add_bonus
    
    currency_price_rounding
    round_price_to_currency_step    
);

our @EXPORT_OK = qw(
    inscribe_to_constants
    CPM_PRICE_CONSTANT_NAMES
    get_frontpage_min_price
    get_geo_list
    get_min_price
);

use constant CPM_PRICE_CONSTANT_NAMES => { min_const => 'MIN_CPM_PRICE', max_const => 'MAX_CPM_PRICE' };

use constant COUNTRY_CURRENCIES_RELOAD_INTERVAL => 24 * 60 * 60;

our $EPSILON = 1e-7;

# для стратегии автобюджета "недельный пакет кликов"
our $MIN_AUTOBUDGET_CLICKS_BUNDLE = 100;

=head2 MAX_DAILY_BUDGET_AMOUNT

    максимальное значение дневного бюджета (одинаковое для всех валют)

=cut

our $MAX_DAILY_BUDGET_AMOUNT = 1_000_000_000;
our $GBP_UID_ALLOWED_EVERYONE = '-1';

# флаг отладки. Если выставлен -- вместо die в случае использования некорректной валюты будет отладочная надпись 
our $CURRENCY_DEBUG ||= 0;

# проперти для показа новых валют новым клиентам из Турции
state $SHOW_NEW_CURRENCIES_FOR_CLIENTS_FROM_TURKEY = Property->new('SHOW_NEW_CURRENCIES_FOR_CLIENTS_FROM_TURKEY');
state $ALLOW_GBP_NONAGENCY_CLIENTS = Property->new('gbp_client_uid');
state $ALLOW_GBP_AGENCY_CLIENTS = Property->new('gbp_agency_client_id');

our %CURRENCY_FIELDS = (
    name => 'название',
    full_name => 'полное наименование',
    currency_sign => 'знак валюты', 

    MIN_PRICE => 'минимальная ставка',
    DEFAULT_PRICE => 'умолчальная ставка',
    AUCTION_STEP => 'шаг торгов',
    MAX_PRICE => 'максимальная ставка',
    MAX_SHOW_BID => 'максимальная отображаемая ставка из торгов', # DIRECT-74109
    MIN_AUTOBUDGET_AVG_PRICE => 'минимальная ставка для стратегии "Средняя цена клика"',
    MIN_AUTOBUDGET_BID => 'минимальная ставка для автобюджета',
    MIN_AUTOBUDGET_CLICKS_BUNDLE => 'минимальное количество кликов для стратегии "Пакет кликов"',
    MAX_AUTOBUDGET_CLICKS_BUNDLE => 'максимальное количество кликов для стратегии "Пакет кликов"',

    MIN_AUTOBUDGET_AVG_CPA => 'минимальная ставка для стратегии "Средняя цена конверсии"',
    MIN_CPM_PRICE => 'минимальная цена за тыс. показов',
    DEFAULT_CPM_PRICE => 'дефолтная цена за тыс. показов',
    MAX_CPM_PRICE => 'максимальная цена за тыс. показов',
    DEFAULT_AVG_CPV => 'дефолтная цена за просмотр',
    MIN_AVG_CPV => 'минимальная цена за просмотр',
    MIDDLE_AVG_CPV => 'значение промежуточного деления на шлкале прогноза цены за просмотр',
    MAX_AVG_CPV => 'максимальная цена за просмотр',
    MIN_CPM_FRONTPAGE_PRICE => 'минимальная цена за тыс. показов на главной',
    MIN_AUTOBUDGET_AVG_CPM => 'минимальная средняя цена за тыс показов',

    MIN_IMAGE_PRICE => 'минимальная цена за клик, при которой может быть показана картинка',
    
    MIN_CPC_CPA_PERFORMANCE => 'минимальная цена CPC/CPA для смарт баннеров',

    # Устарело и ни для чего не используется
    # значение константы MIN_PRICE_FOR_MFA сейчас используется только для YND_FIXED
    MIN_PRICE_FOR_MFA => 'ограничения ставок для МФА-сайтов',

    DIRECT_DEFAULT_PAY => 'умолчальная подсказка платежа',
    DEFAULT_AUTOBUDGET => 'умолчальная сумма недельного автобюджета',

    MIN_AUTOBUDGET => 'минимальная сумма недельного автобюджета',
    MAX_AUTOBUDGET => 'максимальная сумма недельного автобюджета',
    MIN_DAILY_BUDGET_FOR_PERIOD  => 'минимальная сумма бюджета на фиксированный период',
    MAX_DAILY_BUDGET_FOR_PERIOD  => 'максимальная сумма бюджета на фиксированный период',

    MIN_PAY => 'минимальный платёж',
    BIG_RATE => 'Подсветка больших значений ставок',
    RECOMMENDED_SUM_TO_PAY => 'рекомендуемая сумма оплаты (без использования грубого прогноза)',
    MIN_SUM_INTERPRETE_AS_PAYMENT => 'Минимальная сумма пополнения, начиная с которой мы интерпретируем пополнение как платеж',

    MIN_TRANSFER_MONEY => 'минимальная сумма для переноса с кампании на кампанию',
    MAX_TOPAY_SUGGEST => 'максимальная сумма счета, которую можем подсказать на странице оплаты',
    MAX_AUTOBUDGET_BID => 'максимальная допустимая ставка для автобюджета',
    MIN_DAY_BUDGET => 'минимальная сумма дневного бюджета',
    MIN_WALLET_DAY_BUDGET => 'минимальная сумма дневного бюджета общего счета',
    MAX_DAILY_BUDGET_AMOUNT => 'максимальная сумма дневного бюджета',

    MIN_AUTOPAY => 'минимальная сумма платежа в автопополнении',
    MAX_AUTOPAY_CARD => 'максимальная сумма платежа пластиковой картой',
    MAX_AUTOPAY_YAMONEY => 'максимальная сумма платежа Яндекс.Деньгами',
    MAX_AUTOPAY_REMAINING => 'максимальная сумма остатка на счете для автоплатежа',
    MAX_CLIENT_ARCHIVE => 'максимально допустимый оскаток для архивации клиента',

    BALANCE_CURRENCY_NAME => 'символьный код валюты в Балансе',  # или undef, если не требуется получение курса из Баланса

    ISO_NUM_CODE => 'числовой код валюты', # http://en.wikipedia.org/wiki/ISO_4217

    # магические константы, вынесенные из кода баблометра. описание может быть [частично] неверным.
    MONEYMETER_MAX_MIDDLE_SUM => 'Максимальное значение для средней точки баблометра',
    MONEYMETER_TYPICAL_MIDDLE_SUM_INTERVAL_BEGIN => 'Начало типичного интервала для средних позиций баблометра',
    MONEYMETER_TYPICAL_MIDDLE_SUM_INTERVAL_END => 'Конец типичного интервала для средних позиций баблометра',
    MONEYMETER_MIDDLE_PRICE_MIN => 'Минимальная ставка для средней цены баблометра',

    LIST_ORDER => 'Число, показывающее порядок валюты в списке валют',

    PRECISION_DIGIT_COUNT => 'Количество значащих знаков после запятой в суммах',

    YABS_RATIO => 'Делитель для сумм в БК - //home/yabs/dict/Currency.Ratio',

    # подсветки полей ограничений
    AUTOBUDGET_MAX_PRICE_WARNING => 'Подсветка максимальной ставки в автоматических стратегиях',
    AUTOBUDGET_SUM_WARNING => 'Подсветка недельного бюджета в автоматических стратегиях',
    AUTOBUDGET_AVG_PRICE_WARNING => 'Подсветка средней цены клика в стратегии "Средняя цена клика"',
    AUTOBUDGET_AVG_CPA_WARNING => 'Подсветка средней цены конверсии в стратегии "Средняя цена конверсии"',
    AUTOBUDGET_PAY_FOR_CONVERSION_AVG_CPA_WARNING => 'Подсветка средней цены конверсии в стратегии "Средняя цена конверсии" в режиме "Оплата за конверсии"',
    AUTOBUDGET_PAY_FOR_CONVERSION_AVG_CPA_WARNING_INCREASED => 'Подсветка средней цены конверсии в стратегии "Средняя цена конверсии" в режиме "Оплата за конверсии"',
    AUTOBUDGET_CLICKS_BUNDLE_WARNING => 'Подсветка количества кликов для стратегии "пакет кликов"',

    # суммы рекомендаций
    RECOMMENDATION_SUM_MIN => 'минимально рекомендуемая сумма оплаты',
    RECOMMENDATION_SUM_MID => 'средняя рекомендуемая сумма оплаты',
    RECOMMENDATION_SUM_MAX => 'рекомендуемая сумма оплаты',
    RECOMMENDATION_SUM_WARN => 'сумма, при которой нужно предупреждать пользователя о методах оплаты (только для рублей)',
    PAY_WITH_CASH_LIMIT => 'максимально допустимая сумма платежа наличными',

    BRAND_SURVEY_BUDGET_THRESHOLD => 'порог бюджета для Brand Lift-а',
    BRAND_SURVEY_BUDGET_THRESHOLD_DAILY => 'порог ежедневного бюджета для Band Lift-а',

    MONEY_OUT_LIMIT => 'порог отправки уведомлений об окончании средств',

    TOUCH_WEEK_BUDGET_SUM_FIRST => 'первая сумма недельного бюджета для тачевого интерфейса',
    TOUCH_WEEK_BUDGET_SUM_SECOND => 'вторая сумма недельного бюджета для тачевого интерфейса',
    TOUCH_WEEK_BUDGET_SUM_THIRD => 'третья сумма недельного бюджета для тачевого интерфейса',

    UC_DEFAULT_CONVERSION_VALUE => 'дефолтное значение ConversionValue для uc',
    UC_DEFAULT_WEEK_BUDGET => 'дефолтный недельный бюджет для uc',

    PAY_FOR_CONVERSION_MIN_RESERVED_SUM_DEFAULT_VALUE => 'минимальный остаток на счёте для CPA',
);
Hash::Util::lock_hash_recurse(%CURRENCY_FIELDS);

# при добавлении новых валют или изменении существующих - не забудь позвать protected/maintenance/make_custom_js_iget.pl
# а также сделать экспорт/импорт в java

# http://wiki.yandex-team.ru/anastasijabalakina/multivaljutnostinterfejjsa/konstantydljavaljut
our %_CURRENCY_DESCRIPTION = (
    # multicurrency: YND_FIXED -- фиксированная у.е. После перевода клиентов на настоящие валюты ее не останется совсем
    YND_FIXED =>{ 
        name => iget_noop('у.е.'),
        full_name => iget_noop('условные единицы'),
        currency_sign => 'y.e.',

        MIN_PRICE => 0.01,
        DEFAULT_PRICE => 0.1,
        AUCTION_STEP => 0.01,
        MIN_AUTOBUDGET_AVG_PRICE => 0.03,
        MAX_PRICE => 840,
        MAX_SHOW_BID => 168,

        MIN_IMAGE_PRICE => 0.01,

        MIN_CPC_CPA_PERFORMANCE => 0.1,

        MIN_PRICE_FOR_MFA => 0.05,

        DIRECT_DEFAULT_PAY => 50,
        DEFAULT_AUTOBUDGET => 50,

        MIN_PAY => 15,
        BIG_RATE => 10,
        RECOMMENDED_SUM_TO_PAY => 150,
        MIN_SUM_INTERPRETE_AS_PAYMENT => 10,

        MIN_AUTOBUDGET => 10,
        MAX_AUTOBUDGET => 10_000_000,
        MIN_DAILY_BUDGET_FOR_PERIOD => 10,

        MIN_TRANSFER_MONEY => 15,
        MAX_TOPAY_SUGGEST => 500,

        MIN_AUTOBUDGET_AVG_CPA => 0.03,
        MIN_CPM_PRICE => 0.17,
        MIN_CPM_FRONTPAGE_PRICE => 0.17,

        DEFAULT_AVG_CPV => undef,
        MIN_AVG_CPV => 0.01,
        MIDDLE_AVG_CPV => 0.12,
        MAX_AVG_CPV => 0.2,

        DEFAULT_CPM_PRICE => undef,
        MAX_CPM_PRICE => 100,

        MIN_DAY_BUDGET => 10,

        MIN_AUTOPAY => undef,
        MAX_AUTOPAY_CARD => undef,
        MAX_AUTOPAY_YAMONEY => undef,
        MAX_AUTOPAY_REMAINING => undef,
        MAX_CLIENT_ARCHIVE => 100,

        BALANCE_CURRENCY_NAME => undef,

        ISO_NUM_CODE => undef,

        MONEYMETER_MAX_MIDDLE_SUM => 500,
        MONEYMETER_TYPICAL_MIDDLE_SUM_INTERVAL_BEGIN => 30,
        MONEYMETER_TYPICAL_MIDDLE_SUM_INTERVAL_END => 60,
        MONEYMETER_MIDDLE_PRICE_MIN => 5,

        LIST_ORDER => 1,

        PRECISION_DIGIT_COUNT => 6,

        YABS_RATIO => 1,

        AUTOBUDGET_MAX_PRICE_WARNING => 10,
        AUTOBUDGET_SUM_WARNING => 34_000,
        AUTOBUDGET_AVG_PRICE_WARNING => 10,
        AUTOBUDGET_AVG_CPA_WARNING => 1_000,
        AUTOBUDGET_PAY_FOR_CONVERSION_AVG_CPA_WARNING => 160,
        AUTOBUDGET_PAY_FOR_CONVERSION_AVG_CPA_WARNING_INCREASED => 480,
        AUTOBUDGET_CLICKS_BUNDLE_WARNING => 10_000_000,

        RECOMMENDATION_SUM_MIN => 0,
        RECOMMENDATION_SUM_MID => 0,
        RECOMMENDATION_SUM_MAX => 0,
        RECOMMENDATION_SUM_WARN => 0,
        PAY_WITH_CASH_LIMIT => 0,

        BRAND_SURVEY_BUDGET_THRESHOLD => 0,
        BRAND_SURVEY_BUDGET_THRESHOLD_DAILY => 0,

        MONEY_OUT_LIMIT => 0.2,

        TOUCH_WEEK_BUDGET_SUM_FIRST => 40,
        TOUCH_WEEK_BUDGET_SUM_SECOND => 67,
        TOUCH_WEEK_BUDGET_SUM_THIRD => 100,

        UC_DEFAULT_CONVERSION_VALUE => undef,
        UC_DEFAULT_WEEK_BUDGET => undef,

        PAY_FOR_CONVERSION_MIN_RESERVED_SUM_DEFAULT_VALUE => undef,
    },
    # YND => {}, # multicurrency: если вдруг понадобится "плавающая" у.е. -- у нее будет код YND
    RUB =>{
        name => iget_noop('руб.'),
        full_name => iget_noop('российские рубли'),
        currency_sign => "\x{20BD}",

        MIN_PRICE => 0.3,
        DEFAULT_PRICE => 3,
        AUCTION_STEP => 0.1,
        MIN_AUTOBUDGET_AVG_PRICE => 0.9,
        MAX_PRICE => 25_000,
        MAX_SHOW_BID => 5_000,

        MIN_IMAGE_PRICE => 0.3,

        MIN_CPC_CPA_PERFORMANCE => 1,

        MIN_PRICE_FOR_MFA => 1.5,

        DIRECT_DEFAULT_PAY => 1500,
        DEFAULT_AUTOBUDGET => 1500,

        MIN_PAY => 300,
        BIG_RATE => 300,
        RECOMMENDED_SUM_TO_PAY => 4000,
        MIN_SUM_INTERPRETE_AS_PAYMENT => 300,

        MIN_AUTOBUDGET => 300,
        MAX_AUTOBUDGET => 300_000_000,
        MIN_DAILY_BUDGET_FOR_PERIOD => 300,

        MIN_TRANSFER_MONEY => 300,
        MAX_TOPAY_SUGGEST => 15000,

        MIN_AUTOBUDGET_AVG_CPA => 0.9,
        MIN_CPM_PRICE => 5,
        DEFAULT_CPM_PRICE => 100,
        MAX_CPM_PRICE => 3_000,
        MIN_CPM_FRONTPAGE_PRICE => 25,

        DEFAULT_AVG_CPV => 0.6,
        MIN_AVG_CPV => 0.1,
        MIDDLE_AVG_CPV => 1.2,
        MAX_AVG_CPV => 2,

        MIN_DAY_BUDGET => 300,
        MIN_WALLET_DAY_BUDGET => 1000,

        MIN_AUTOPAY => 300,
        MAX_AUTOPAY_CARD => 49999.99,
        MAX_AUTOPAY_YAMONEY => 15000,
        MAX_AUTOPAY_REMAINING => 999999.99,
        MAX_CLIENT_ARCHIVE => 3000,

        BALANCE_CURRENCY_NAME => undef,

        ISO_NUM_CODE => 643,

        MONEYMETER_MAX_MIDDLE_SUM => 13000,
        MONEYMETER_TYPICAL_MIDDLE_SUM_INTERVAL_BEGIN => 800,
        MONEYMETER_TYPICAL_MIDDLE_SUM_INTERVAL_END => 1_600,
        MONEYMETER_MIDDLE_PRICE_MIN => 150,

        LIST_ORDER => 2,

        PRECISION_DIGIT_COUNT => 4,

        YABS_RATIO => 100,

        AUTOBUDGET_MAX_PRICE_WARNING => 300,
        AUTOBUDGET_SUM_WARNING => 1_000_000,
        AUTOBUDGET_AVG_PRICE_WARNING => 300,
        AUTOBUDGET_AVG_CPA_WARNING => 30_000,
        AUTOBUDGET_PAY_FOR_CONVERSION_AVG_CPA_WARNING => 5000,
        AUTOBUDGET_PAY_FOR_CONVERSION_AVG_CPA_WARNING_INCREASED => 15000,
        AUTOBUDGET_CLICKS_BUNDLE_WARNING => 10_000_000,

        RECOMMENDATION_SUM_MIN => 5000,
        RECOMMENDATION_SUM_MID => 12500,
        RECOMMENDATION_SUM_MAX => 35000,
        RECOMMENDATION_SUM_WARN => 49999,
        PAY_WITH_CASH_LIMIT => 12500,

        BRAND_SURVEY_BUDGET_THRESHOLD => 1_000_000,
        BRAND_SURVEY_BUDGET_THRESHOLD_DAILY => 70_000,

        MONEY_OUT_LIMIT => 5,

        TOUCH_WEEK_BUDGET_SUM_FIRST => 1200,
        TOUCH_WEEK_BUDGET_SUM_SECOND => 2000,
        TOUCH_WEEK_BUDGET_SUM_THIRD => 3000,

        UC_DEFAULT_CONVERSION_VALUE => 100,
        UC_DEFAULT_WEEK_BUDGET => 2300,

        PAY_FOR_CONVERSION_MIN_RESERVED_SUM_DEFAULT_VALUE => 5000,
    },
    UAH =>{
        name => iget_noop('грн'),
        full_name => iget_noop('украинские гривны'),
        currency_sign => "\x{20B4}",

        MIN_PRICE => 0.08,
        DEFAULT_PRICE => 0.8,
        AUCTION_STEP => 0.01,
        MIN_AUTOBUDGET_AVG_PRICE => 0.3,
        MAX_PRICE => 9_000,
        MAX_SHOW_BID => 1_800,

        MIN_IMAGE_PRICE => 0.08,

        MIN_CPC_CPA_PERFORMANCE => 0.24,

        MIN_PRICE_FOR_MFA => 0.4,

        DIRECT_DEFAULT_PAY => 360,
        DEFAULT_AUTOBUDGET => 400,

        MIN_PAY => 250,
        BIG_RATE => 80,
        RECOMMENDED_SUM_TO_PAY => 1000,
        MIN_SUM_INTERPRETE_AS_PAYMENT => 80,

        MIN_AUTOBUDGET => 80,
        MAX_AUTOBUDGET => 74_000_000,
        MIN_DAILY_BUDGET_FOR_PERIOD => 135,

        MIN_TRANSFER_MONEY => 250,
        MAX_TOPAY_SUGGEST => 2500,

        MIN_AUTOBUDGET_AVG_CPA => 0.3,
        MIN_CPM_PRICE => 2.25,
        DEFAULT_CPM_PRICE => 45,
        MAX_CPM_PRICE => 1_360,
        MIN_CPM_FRONTPAGE_PRICE => 9.96,

        DEFAULT_AVG_CPV => 0.16,
        MIN_AVG_CPV => 0.03,
        MIDDLE_AVG_CPV => 0.36,
        MAX_AVG_CPV => 0.6,

        MIN_DAY_BUDGET => 80,

        MIN_AUTOPAY => undef,
        MAX_AUTOPAY_CARD => undef,
        MAX_AUTOPAY_YAMONEY => undef,
        MAX_AUTOPAY_REMAINING => undef,
        MAX_CLIENT_ARCHIVE => 800,

        BALANCE_CURRENCY_NAME => 'UAH',

        ISO_NUM_CODE => 980,

        MONEYMETER_MAX_MIDDLE_SUM => 3_400,
        MONEYMETER_TYPICAL_MIDDLE_SUM_INTERVAL_BEGIN => 200,
        MONEYMETER_TYPICAL_MIDDLE_SUM_INTERVAL_END => 400,
        MONEYMETER_MIDDLE_PRICE_MIN => 50,

        LIST_ORDER => 3,

        PRECISION_DIGIT_COUNT => 5,

        YABS_RATIO => 10,

        AUTOBUDGET_MAX_PRICE_WARNING => 80,
        AUTOBUDGET_SUM_WARNING => 250_000,
        AUTOBUDGET_AVG_PRICE_WARNING => 80,
        AUTOBUDGET_AVG_CPA_WARNING => 8_000,
        AUTOBUDGET_PAY_FOR_CONVERSION_AVG_CPA_WARNING => 1300,
        AUTOBUDGET_PAY_FOR_CONVERSION_AVG_CPA_WARNING_INCREASED => 3900,
        AUTOBUDGET_CLICKS_BUNDLE_WARNING => 10_000_000,

        RECOMMENDATION_SUM_MIN => 0,
        RECOMMENDATION_SUM_MID => 0,
        RECOMMENDATION_SUM_MAX => 0,
        RECOMMENDATION_SUM_WARN => 0,
        PAY_WITH_CASH_LIMIT => 0,

        BRAND_SURVEY_BUDGET_THRESHOLD => 0,
        BRAND_SURVEY_BUDGET_THRESHOLD_DAILY => 0,

        MONEY_OUT_LIMIT => 2,

        TOUCH_WEEK_BUDGET_SUM_FIRST => 466,
        TOUCH_WEEK_BUDGET_SUM_SECOND => 776,
        TOUCH_WEEK_BUDGET_SUM_THIRD => 1168,

        UC_DEFAULT_CONVERSION_VALUE => 35,
        UC_DEFAULT_WEEK_BUDGET => 850,

        PAY_FOR_CONVERSION_MIN_RESERVED_SUM_DEFAULT_VALUE => 0,
    },
    USD =>{
        name => iget_noop('долл.'),
        full_name => iget_noop('доллары США'),
        currency_sign => '$',

        MIN_PRICE => 0.01,
        DEFAULT_PRICE => 0.1,
        AUCTION_STEP => 0.01,
        MIN_AUTOBUDGET_AVG_PRICE => 0.03,
        MAX_PRICE => 700,
        MAX_SHOW_BID => 140,

        MIN_IMAGE_PRICE => 0.01,

        MIN_CPC_CPA_PERFORMANCE => 0.03,

        MIN_PRICE_FOR_MFA => 0.05,

        DIRECT_DEFAULT_PAY => 50,
        DEFAULT_AUTOBUDGET => 50,

        MIN_PAY => 15,
        BIG_RATE => 10,
        RECOMMENDED_SUM_TO_PAY => 150,
        MIN_SUM_INTERPRETE_AS_PAYMENT => 10,

        MIN_AUTOBUDGET => 10,
        MAX_AUTOBUDGET => 10_000_000,
        MIN_DAILY_BUDGET_FOR_PERIOD => 5,


        MIN_TRANSFER_MONEY => 15,
        MAX_TOPAY_SUGGEST => 500,

        MIN_AUTOBUDGET_AVG_CPA => 0.03,
        MIN_CPM_PRICE => 0.08,
        DEFAULT_CPM_PRICE => 1.5,
        MAX_CPM_PRICE => 52,
        MIN_CPM_FRONTPAGE_PRICE => 0.36,

        DEFAULT_AVG_CPV => 0.07,
        MIN_AVG_CPV => 0.01,
        MIDDLE_AVG_CPV => 0.12,
        MAX_AVG_CPV => 0.2,

        MIN_DAY_BUDGET => 10,

        MIN_AUTOPAY => undef,
        MAX_AUTOPAY_CARD => undef,
        MAX_AUTOPAY_YAMONEY => undef,
        MAX_AUTOPAY_REMAINING => undef,
        MAX_CLIENT_ARCHIVE => 100,

        BALANCE_CURRENCY_NAME => 'USD',

        ISO_NUM_CODE => 840,

        MONEYMETER_MAX_MIDDLE_SUM => 500,
        MONEYMETER_TYPICAL_MIDDLE_SUM_INTERVAL_BEGIN => 30,
        MONEYMETER_TYPICAL_MIDDLE_SUM_INTERVAL_END => 60,
        MONEYMETER_MIDDLE_PRICE_MIN => 5,

        LIST_ORDER => 4,

        PRECISION_DIGIT_COUNT => 6,

        YABS_RATIO => 1,

        AUTOBUDGET_MAX_PRICE_WARNING => 10,
        AUTOBUDGET_SUM_WARNING => 34_000,
        AUTOBUDGET_AVG_PRICE_WARNING => 10,
        AUTOBUDGET_AVG_CPA_WARNING => 1_000,
        AUTOBUDGET_PAY_FOR_CONVERSION_AVG_CPA_WARNING => 170,
        AUTOBUDGET_PAY_FOR_CONVERSION_AVG_CPA_WARNING_INCREASED => 510,
        AUTOBUDGET_CLICKS_BUNDLE_WARNING => 10_000_000,

        RECOMMENDATION_SUM_MIN => 80,
        RECOMMENDATION_SUM_MID => 200,
        RECOMMENDATION_SUM_MAX => 500,
        RECOMMENDATION_SUM_WARN => 0,
        PAY_WITH_CASH_LIMIT => 0,

        BRAND_SURVEY_BUDGET_THRESHOLD => 7000,
        BRAND_SURVEY_BUDGET_THRESHOLD_DAILY => 490,

        MONEY_OUT_LIMIT => 0.1,

        TOUCH_WEEK_BUDGET_SUM_FIRST => 20,
        TOUCH_WEEK_BUDGET_SUM_SECOND => 32,
        TOUCH_WEEK_BUDGET_SUM_THIRD => 48,

        UC_DEFAULT_CONVERSION_VALUE => 2,
        UC_DEFAULT_WEEK_BUDGET => 30,

        PAY_FOR_CONVERSION_MIN_RESERVED_SUM_DEFAULT_VALUE => 72,
    },
    EUR =>{
        name => iget_noop('евро'),
        full_name => iget_noop('евро'),
        currency_sign => "\x{20AC}",

        MIN_PRICE => 0.01,
        DEFAULT_PRICE => 0.1,
        AUCTION_STEP => 0.01,
        MIN_AUTOBUDGET_AVG_PRICE => 0.03,
        MAX_PRICE => 500,
        MAX_SHOW_BID => 100,

        MIN_IMAGE_PRICE => 0.01,

        MIN_CPC_CPA_PERFORMANCE => 0.03,

        MIN_PRICE_FOR_MFA => 0.05,

        DIRECT_DEFAULT_PAY => 50,
        DEFAULT_AUTOBUDGET => 50,

        MIN_PAY => 15,
        BIG_RATE => 10,
        RECOMMENDED_SUM_TO_PAY => 150,
        MIN_SUM_INTERPRETE_AS_PAYMENT => 10,

        MIN_AUTOBUDGET => 10,
        MAX_AUTOBUDGET => 10_000_000,
        MIN_DAILY_BUDGET_FOR_PERIOD => 4.2,

        MIN_TRANSFER_MONEY => 15,
        MAX_TOPAY_SUGGEST => 500,

        MIN_AUTOBUDGET_AVG_CPA => 0.03,
        MIN_CPM_PRICE => 0.07,
        DEFAULT_CPM_PRICE => 1.5,
        MAX_CPM_PRICE => 44,
        MIN_CPM_FRONTPAGE_PRICE => 0.31,

        DEFAULT_AVG_CPV => 0.07,
        MIN_AVG_CPV => 0.01,
        MIDDLE_AVG_CPV => 0.12,
        MAX_AVG_CPV => 0.2,

        MIN_DAY_BUDGET => 10,

        MIN_AUTOPAY => undef,
        MAX_AUTOPAY_CARD => undef,
        MAX_AUTOPAY_YAMONEY => undef,
        MAX_AUTOPAY_REMAINING => undef,
        MAX_CLIENT_ARCHIVE => 100,

        BALANCE_CURRENCY_NAME => 'EUR',

        ISO_NUM_CODE => 978,

        MONEYMETER_MAX_MIDDLE_SUM => 350,
        MONEYMETER_TYPICAL_MIDDLE_SUM_INTERVAL_BEGIN => 20,
        MONEYMETER_TYPICAL_MIDDLE_SUM_INTERVAL_END => 40,
        MONEYMETER_MIDDLE_PRICE_MIN => 5,

        LIST_ORDER => 5,

        PRECISION_DIGIT_COUNT => 6,

        YABS_RATIO => 1,

        AUTOBUDGET_MAX_PRICE_WARNING => 10,
        AUTOBUDGET_SUM_WARNING => 34_000,
        AUTOBUDGET_AVG_PRICE_WARNING => 10,
        AUTOBUDGET_AVG_CPA_WARNING => 1_000,
        AUTOBUDGET_PAY_FOR_CONVERSION_AVG_CPA_WARNING => 170,
        AUTOBUDGET_PAY_FOR_CONVERSION_AVG_CPA_WARNING_INCREASED => 510,
        AUTOBUDGET_CLICKS_BUNDLE_WARNING => 10_000_000,

        RECOMMENDATION_SUM_MIN => 80,
        RECOMMENDATION_SUM_MID => 200,
        RECOMMENDATION_SUM_MAX => 500,
        RECOMMENDATION_SUM_WARN => 0,
        PAY_WITH_CASH_LIMIT => 0,

        BRAND_SURVEY_BUDGET_THRESHOLD => 7000,
        BRAND_SURVEY_BUDGET_THRESHOLD_DAILY => 490,

        MONEY_OUT_LIMIT => 0.1,

        TOUCH_WEEK_BUDGET_SUM_FIRST => 18,
        TOUCH_WEEK_BUDGET_SUM_SECOND => 29,
        TOUCH_WEEK_BUDGET_SUM_THIRD => 44,

        UC_DEFAULT_CONVERSION_VALUE => 2,
        UC_DEFAULT_WEEK_BUDGET => 30,

        PAY_FOR_CONVERSION_MIN_RESERVED_SUM_DEFAULT_VALUE => 64,
    },
    KZT => {
        name => iget_noop('тенге'),
        full_name => iget_noop('казахстанские тенге'),
        currency_sign => "\x{20B8}",

        MIN_PRICE => 1,
        DEFAULT_PRICE => 10,
        AUCTION_STEP => 1,
        MIN_AUTOBUDGET_AVG_PRICE => 3,
        MAX_PRICE => 130_000,
        MAX_SHOW_BID => 26_000,

        MIN_IMAGE_PRICE => 1,

        MIN_CPC_CPA_PERFORMANCE => 3,

        MIN_PRICE_FOR_MFA => 5,

        DIRECT_DEFAULT_PAY => 6_500,
        DEFAULT_AUTOBUDGET => 6_500,

        MIN_PAY => 5_000,
        BIG_RATE => 1_300,
        RECOMMENDED_SUM_TO_PAY => 20_000,
        MIN_SUM_INTERPRETE_AS_PAYMENT => 1_300,

        MIN_AUTOBUDGET => 1_300,
        MAX_AUTOBUDGET => 1_300_000_000,
        MIN_DAILY_BUDGET_FOR_PERIOD => 1700,

        MIN_TRANSFER_MONEY => 5_000,
        MAX_TOPAY_SUGGEST => 65_000,

        MIN_AUTOBUDGET_AVG_CPA => 3,
        MIN_CPM_PRICE => 29,
        DEFAULT_CPM_PRICE => 600,
        MAX_CPM_PRICE => 18_000,
        MIN_CPM_FRONTPAGE_PRICE => 138.89,

        DEFAULT_AVG_CPV => 1.6,
        MIN_AVG_CPV => 0.3,
        MIDDLE_AVG_CPV => 3.6,
        MAX_AVG_CPV => 6,

        MIN_DAY_BUDGET => 1_300,

        MIN_AUTOPAY => undef,
        MAX_AUTOPAY_CARD => undef,
        MAX_AUTOPAY_YAMONEY => undef,
        MAX_AUTOPAY_REMAINING => undef,
        MAX_CLIENT_ARCHIVE => 13_000,

        BALANCE_CURRENCY_NAME => 'KZT',

        ISO_NUM_CODE => 398,

        MONEYMETER_MAX_MIDDLE_SUM => 65_000,
        MONEYMETER_TYPICAL_MIDDLE_SUM_INTERVAL_BEGIN => 4_000,
        MONEYMETER_TYPICAL_MIDDLE_SUM_INTERVAL_END => 8_000,
        MONEYMETER_MIDDLE_PRICE_MIN => 650,

        LIST_ORDER => 7,

        PRECISION_DIGIT_COUNT => 6,

        YABS_RATIO => 100,

        AUTOBUDGET_MAX_PRICE_WARNING => 1_300,
        AUTOBUDGET_SUM_WARNING => 4_300_000,
        AUTOBUDGET_AVG_PRICE_WARNING => 1_300,
        AUTOBUDGET_AVG_CPA_WARNING => 130_000,
        AUTOBUDGET_PAY_FOR_CONVERSION_AVG_CPA_WARNING => 30000,
        AUTOBUDGET_PAY_FOR_CONVERSION_AVG_CPA_WARNING_INCREASED => 90000,
        AUTOBUDGET_CLICKS_BUNDLE_WARNING => 10_000_000,

        RECOMMENDATION_SUM_MIN => 28000,
        RECOMMENDATION_SUM_MID => 70000,
        RECOMMENDATION_SUM_MAX => 200000,
        RECOMMENDATION_SUM_WARN => 0,
        PAY_WITH_CASH_LIMIT => 0,

        BRAND_SURVEY_BUDGET_THRESHOLD => 1_500_000,
        BRAND_SURVEY_BUDGET_THRESHOLD_DAILY => 107_000,

        MONEY_OUT_LIMIT => 30,

        TOUCH_WEEK_BUDGET_SUM_FIRST => 7158,
        TOUCH_WEEK_BUDGET_SUM_SECOND => 11930,
        TOUCH_WEEK_BUDGET_SUM_THIRD => 17962,

        UC_DEFAULT_CONVERSION_VALUE => 500,
        UC_DEFAULT_WEEK_BUDGET => 11500,

        PAY_FOR_CONVERSION_MIN_RESERVED_SUM_DEFAULT_VALUE => 30000,
    },
    CHF => {
        name => iget_noop('CHF'),
        full_name => iget_noop('швейцарские франки'),
        currency_sign => "\x{20A3}",

        MIN_PRICE => 0.01,
        DEFAULT_PRICE => 0.1,
        AUCTION_STEP => 0.01,
        MIN_AUTOBUDGET_AVG_PRICE => 0.03,
        MAX_PRICE => 600,
        MAX_SHOW_BID => 120,

        MIN_IMAGE_PRICE => 0.01,

        MIN_CPC_CPA_PERFORMANCE => 0.03,

        MIN_PRICE_FOR_MFA => 0.05,

        DIRECT_DEFAULT_PAY => 50,
        DEFAULT_AUTOBUDGET => 50,

        MIN_PAY => 15,
        BIG_RATE => 10,
        MIN_SUM_INTERPRETE_AS_PAYMENT => 10,

        MIN_AUTOBUDGET => 10,
        MAX_AUTOBUDGET => 10_000_000,
        MIN_DAILY_BUDGET_FOR_PERIOD => 5,
        RECOMMENDED_SUM_TO_PAY => 150,

        MIN_TRANSFER_MONEY => 15,
        MAX_TOPAY_SUGGEST => 500,

        MIN_AUTOBUDGET_AVG_CPA => 0.03,
        MIN_CPM_PRICE => 0.08,
        DEFAULT_CPM_PRICE => 1.5,
        MAX_CPM_PRICE => 50,
        MIN_CPM_FRONTPAGE_PRICE => 0.35,

        DEFAULT_AVG_CPV => 0.07,
        MIN_AVG_CPV => 0.01,
        MIDDLE_AVG_CPV => 0.12,
        MAX_AVG_CPV => 0.2,

        MIN_DAY_BUDGET => 10,

        MIN_AUTOPAY => undef,
        MAX_AUTOPAY_CARD => undef,
        MAX_AUTOPAY_YAMONEY => undef,
        MAX_AUTOPAY_REMAINING => undef,
        MAX_CLIENT_ARCHIVE => 100,

        BALANCE_CURRENCY_NAME => 'CHF',

        ISO_NUM_CODE => 756,

        MONEYMETER_MAX_MIDDLE_SUM => 400,
        MONEYMETER_TYPICAL_MIDDLE_SUM_INTERVAL_BEGIN => 20,
        MONEYMETER_TYPICAL_MIDDLE_SUM_INTERVAL_END => 40,
        MONEYMETER_MIDDLE_PRICE_MIN => 5,

        LIST_ORDER => 8,

        PRECISION_DIGIT_COUNT => 6,

        YABS_RATIO => 1,

        AUTOBUDGET_MAX_PRICE_WARNING => 10,
        AUTOBUDGET_SUM_WARNING => 34_000,
        AUTOBUDGET_AVG_PRICE_WARNING => 10,
        AUTOBUDGET_AVG_CPA_WARNING => 1_000,
        AUTOBUDGET_PAY_FOR_CONVERSION_AVG_CPA_WARNING => 170,
        AUTOBUDGET_PAY_FOR_CONVERSION_AVG_CPA_WARNING_INCREASED => 510,
        AUTOBUDGET_CLICKS_BUNDLE_WARNING => 10_000_000,

        RECOMMENDATION_SUM_MIN => 80,
        RECOMMENDATION_SUM_MID => 200,
        RECOMMENDATION_SUM_MAX => 500,
        RECOMMENDATION_SUM_WARN => 0,
        PAY_WITH_CASH_LIMIT => 0,

        BRAND_SURVEY_BUDGET_THRESHOLD => 0,
        BRAND_SURVEY_BUDGET_THRESHOLD_DAILY => 0,

        MONEY_OUT_LIMIT => 0.1,

        TOUCH_WEEK_BUDGET_SUM_FIRST => 20,
        TOUCH_WEEK_BUDGET_SUM_SECOND => 31,
        TOUCH_WEEK_BUDGET_SUM_THIRD => 47,

        UC_DEFAULT_CONVERSION_VALUE => 1,
        UC_DEFAULT_WEEK_BUDGET => 30,

        PAY_FOR_CONVERSION_MIN_RESERVED_SUM_DEFAULT_VALUE => 0,
    },
    TRY => {
        name => iget_noop('TL'),
        full_name => iget_noop('турецкие лиры'),
        currency_sign => "\x{20BA}",

        MIN_PRICE => 0.01,
        DEFAULT_PRICE => 0.1,
        AUCTION_STEP => 0.01,
        MIN_AUTOBUDGET_AVG_PRICE => 0.03,
        MAX_PRICE => 1_500,
        MAX_SHOW_BID => 300,

        MIN_IMAGE_PRICE => 0.01,

        MIN_CPC_CPA_PERFORMANCE => 0.03,

        MIN_PRICE_FOR_MFA => 0.05,

        DIRECT_DEFAULT_PAY => 100,
        DEFAULT_AUTOBUDGET => 85,

        MIN_PAY => 50,
        BIG_RATE => 17,
        RECOMMENDED_SUM_TO_PAY => 200,
        MIN_SUM_INTERPRETE_AS_PAYMENT => 17,

        MIN_AUTOBUDGET => 17,
        MAX_AUTOBUDGET => 17_000_000,
        MIN_DAILY_BUDGET_FOR_PERIOD => 18,

        MIN_TRANSFER_MONEY => 50,
        MAX_TOPAY_SUGGEST => 1000,

        MIN_AUTOBUDGET_AVG_CPA => 0.03,
        MIN_CPM_PRICE => 0.3,
        DEFAULT_CPM_PRICE => 6,
        MAX_CPM_PRICE => 178,
        MIN_CPM_FRONTPAGE_PRICE => 1.89,

        DEFAULT_AVG_CPV => 0.07,
        MIN_AVG_CPV => 0.01,
        MIDDLE_AVG_CPV => 0.12,
        MAX_AVG_CPV => 0.2,

        MIN_DAY_BUDGET => 17,

        MIN_AUTOPAY => undef,
        MAX_AUTOPAY_CARD => undef,
        MAX_AUTOPAY_YAMONEY => undef,
        MAX_AUTOPAY_REMAINING => undef,
        MAX_CLIENT_ARCHIVE => 170,

        BALANCE_CURRENCY_NAME => 'TRY',

        ISO_NUM_CODE => 949,

        MONEYMETER_MAX_MIDDLE_SUM => 850,
        MONEYMETER_TYPICAL_MIDDLE_SUM_INTERVAL_BEGIN => 50,
        MONEYMETER_TYPICAL_MIDDLE_SUM_INTERVAL_END => 100,
        MONEYMETER_MIDDLE_PRICE_MIN => 10,

        LIST_ORDER => 9,

        PRECISION_DIGIT_COUNT => 6,

        YABS_RATIO => 1,

        AUTOBUDGET_MAX_PRICE_WARNING => 17,
        AUTOBUDGET_SUM_WARNING => 57_000,
        AUTOBUDGET_AVG_PRICE_WARNING => 17,
        AUTOBUDGET_AVG_CPA_WARNING => 1_700,
        AUTOBUDGET_PAY_FOR_CONVERSION_AVG_CPA_WARNING => 450,
        AUTOBUDGET_PAY_FOR_CONVERSION_AVG_CPA_WARNING_INCREASED => 1350,

        AUTOBUDGET_CLICKS_BUNDLE_WARNING => 10_000_000,

        RECOMMENDATION_SUM_MIN => 500,
        RECOMMENDATION_SUM_MID => 1200,
        RECOMMENDATION_SUM_MAX => 3200,
        RECOMMENDATION_SUM_WARN => 0,
        PAY_WITH_CASH_LIMIT => 0,

        BRAND_SURVEY_BUDGET_THRESHOLD => 0,
        BRAND_SURVEY_BUDGET_THRESHOLD_DAILY => 0,

        MONEY_OUT_LIMIT => 1,

        TOUCH_WEEK_BUDGET_SUM_FIRST => 114,
        TOUCH_WEEK_BUDGET_SUM_SECOND => 189,
        TOUCH_WEEK_BUDGET_SUM_THIRD => 285,

        UC_DEFAULT_CONVERSION_VALUE => 10,
        UC_DEFAULT_WEEK_BUDGET => 250,

        PAY_FOR_CONVERSION_MIN_RESERVED_SUM_DEFAULT_VALUE => 495,
    },
    BYN => {
        name => iget_noop('BYN'),
        full_name => iget_noop('белорусские рубли'),
        currency_sign => 'Br',

        MIN_PRICE => 0.01,
        DEFAULT_PRICE => 0.1,
        AUCTION_STEP => 0.01,
        MIN_AUTOBUDGET_AVG_PRICE => 0.03,
        MAX_PRICE => 780,
        MAX_SHOW_BID => 158,

        MIN_IMAGE_PRICE => 0.01,

        MIN_CPC_CPA_PERFORMANCE => 0.03,

        MIN_PRICE_FOR_MFA => 0.05,

        DIRECT_DEFAULT_PAY => 47,
        DEFAULT_AUTOBUDGET => 47,

        MIN_PAY => 30,
        BIG_RATE => 9,
        RECOMMENDED_SUM_TO_PAY => 120,
        MIN_SUM_INTERPRETE_AS_PAYMENT => 9,

        MIN_AUTOBUDGET => 9,
        MAX_AUTOBUDGET => 9_400_000,
        MIN_DAILY_BUDGET_FOR_PERIOD => 9,

        MIN_TRANSFER_MONEY => 30,
        MAX_TOPAY_SUGGEST => 470,

        MIN_AUTOBUDGET_AVG_CPA => 0.03,
        MIN_CPM_PRICE => 0.15,
        DEFAULT_CPM_PRICE => 3.5,
        MAX_CPM_PRICE => 100,
        MIN_CPM_FRONTPAGE_PRICE => 0.78,

        DEFAULT_AVG_CPV => 0.07,
        MIN_AVG_CPV => 0.01,
        MIDDLE_AVG_CPV => 0.12,
        MAX_AVG_CPV => 0.2,

        MIN_DAY_BUDGET => 9,

        # автопополнения нет
        MIN_AUTOPAY => undef,
        MAX_AUTOPAY_CARD => undef,
        MAX_AUTOPAY_YAMONEY => undef,
        MAX_AUTOPAY_REMAINING => undef,
        MAX_CLIENT_ARCHIVE => 100,

        BALANCE_CURRENCY_NAME => 'BYN',

        ISO_NUM_CODE => 933,

        MONEYMETER_MAX_MIDDLE_SUM => undef,
        MONEYMETER_TYPICAL_MIDDLE_SUM_INTERVAL_BEGIN => undef,
        MONEYMETER_TYPICAL_MIDDLE_SUM_INTERVAL_END => undef,
        MONEYMETER_MIDDLE_PRICE_MIN => undef,

        LIST_ORDER => 6,

        PRECISION_DIGIT_COUNT => 4,

        YABS_RATIO => 100,

        AUTOBUDGET_MAX_PRICE_WARNING => 9,
        AUTOBUDGET_SUM_WARNING => 31_000,
        AUTOBUDGET_AVG_PRICE_WARNING => 9,
        AUTOBUDGET_AVG_CPA_WARNING => 940,
        AUTOBUDGET_PAY_FOR_CONVERSION_AVG_CPA_WARNING => 160,
        AUTOBUDGET_PAY_FOR_CONVERSION_AVG_CPA_WARNING_INCREASED => 480,
        AUTOBUDGET_CLICKS_BUNDLE_WARNING => 10_000_000,

        RECOMMENDATION_SUM_MIN => 150,
        RECOMMENDATION_SUM_MID => 400,
        RECOMMENDATION_SUM_MAX => 1000,
        RECOMMENDATION_SUM_WARN => 0,
        PAY_WITH_CASH_LIMIT => 0,

        BRAND_SURVEY_BUDGET_THRESHOLD => 8_500,
        BRAND_SURVEY_BUDGET_THRESHOLD_DAILY => 584,

        MONEY_OUT_LIMIT => 1,

        TOUCH_WEEK_BUDGET_SUM_FIRST => 42,
        TOUCH_WEEK_BUDGET_SUM_SECOND => 69,
        TOUCH_WEEK_BUDGET_SUM_THIRD => 103,

        UC_DEFAULT_CONVERSION_VALUE => 5,
        UC_DEFAULT_WEEK_BUDGET => 75,

        PAY_FOR_CONVERSION_MIN_RESERVED_SUM_DEFAULT_VALUE => 170,
    },
    GBP =>{
        name => iget_noop('GBP'),
        full_name => iget_noop('фунты стерлингов'),
        currency_sign => "\x{00A3}",

        MIN_PRICE => 0.01,
        DEFAULT_PRICE => 0.1,
        AUCTION_STEP => 0.01,
        MIN_AUTOBUDGET_AVG_PRICE => 0.03,
        MAX_PRICE => 500,
        MAX_SHOW_BID => 100,

        MIN_IMAGE_PRICE => 0.01,

        MIN_CPC_CPA_PERFORMANCE => 0.03,

        MIN_PRICE_FOR_MFA => 0.05,

        DIRECT_DEFAULT_PAY => 50,
        DEFAULT_AUTOBUDGET => 50,

        MIN_PAY => 15,
        BIG_RATE => 10,
        RECOMMENDED_SUM_TO_PAY => 150,
        MIN_SUM_INTERPRETE_AS_PAYMENT => 10,

        MIN_AUTOBUDGET => 10,
        MAX_AUTOBUDGET => 10_000_000,
        MIN_DAILY_BUDGET_FOR_PERIOD => 4.2,

        MIN_TRANSFER_MONEY => 15,
        MAX_TOPAY_SUGGEST => 500,

        MIN_AUTOBUDGET_AVG_CPA => 0.03,
        MIN_CPM_PRICE => 0.07,
        DEFAULT_CPM_PRICE => 1.5,
        MAX_CPM_PRICE => 44,
        MIN_CPM_FRONTPAGE_PRICE => 0.31,

        DEFAULT_AVG_CPV => 0.07,
        MIN_AVG_CPV => 0.01,
        MIDDLE_AVG_CPV => 0.12,
        MAX_AVG_CPV => 0.2,

        MIN_DAY_BUDGET => 10,

        MIN_AUTOPAY => undef,
        MAX_AUTOPAY_CARD => undef,
        MAX_AUTOPAY_YAMONEY => undef,
        MAX_AUTOPAY_REMAINING => undef,
        MAX_CLIENT_ARCHIVE => 100,

        BALANCE_CURRENCY_NAME => 'GBP',

        ISO_NUM_CODE => 826,

        MONEYMETER_MAX_MIDDLE_SUM => 350,
        MONEYMETER_TYPICAL_MIDDLE_SUM_INTERVAL_BEGIN => 20,
        MONEYMETER_TYPICAL_MIDDLE_SUM_INTERVAL_END => 40,
        MONEYMETER_MIDDLE_PRICE_MIN => 5,

        LIST_ORDER => 5,

        PRECISION_DIGIT_COUNT => 6,

        YABS_RATIO => 1,

        AUTOBUDGET_MAX_PRICE_WARNING => 10,
        AUTOBUDGET_SUM_WARNING => 34_000,
        AUTOBUDGET_AVG_PRICE_WARNING => 10,
        AUTOBUDGET_AVG_CPA_WARNING => 1_000,
        AUTOBUDGET_PAY_FOR_CONVERSION_AVG_CPA_WARNING => 170,
        AUTOBUDGET_PAY_FOR_CONVERSION_AVG_CPA_WARNING_INCREASED => 510,
        AUTOBUDGET_CLICKS_BUNDLE_WARNING => 10_000_000,

        RECOMMENDATION_SUM_MIN => 80,
        RECOMMENDATION_SUM_MID => 200,
        RECOMMENDATION_SUM_MAX => 500,
        RECOMMENDATION_SUM_WARN => 0,
        PAY_WITH_CASH_LIMIT => 0,

        BRAND_SURVEY_BUDGET_THRESHOLD => 5000,
        BRAND_SURVEY_BUDGET_THRESHOLD_DAILY => 350,

        MONEY_OUT_LIMIT => 0.1,

        TOUCH_WEEK_BUDGET_SUM_FIRST => 18,
        TOUCH_WEEK_BUDGET_SUM_SECOND => 29,
        TOUCH_WEEK_BUDGET_SUM_THIRD => 44,

        UC_DEFAULT_CONVERSION_VALUE => 2,
        UC_DEFAULT_WEEK_BUDGET => 30,

        PAY_FOR_CONVERSION_MIN_RESERVED_SUM_DEFAULT_VALUE => 64,
    },
    # DIRECT-107095 - заранее определяем пороги отправки сообщений об окончании средств для узбекских сум и вьетнамских донгов
    # UZS => {
    #     MONEY_OUT_LIMIT => 750,
    # },
    # VND => {
    #     MONEY_OUT_LIMIT => 1900,
    # },
);
our %_CURRENCY_DESCRIPTION_I18N_KEYS = map {$_ => 1} qw/name full_name/;
for my $cur (keys %_CURRENCY_DESCRIPTION) {
    my $cur_data = $_CURRENCY_DESCRIPTION{$cur};
    $cur_data->{'MIN_AUTOBUDGET_BID'} = $cur_data->{'MIN_PRICE'};
    $cur_data->{'MAX_AUTOBUDGET_BID'} = $cur_data->{'MAX_PRICE'};
    $cur_data->{'MAX_AUTOBUDGET_CLICKS_BUNDLE'} = int($cur_data->{'MAX_AUTOBUDGET'} / $cur_data->{'MIN_AUTOBUDGET_BID'});
    $cur_data->{'MIN_AUTOBUDGET_CLICKS_BUNDLE'} = $MIN_AUTOBUDGET_CLICKS_BUNDLE;
    $cur_data->{'MAX_DAILY_BUDGET_AMOUNT'} = $MAX_DAILY_BUDGET_AMOUNT;
    $cur_data->{'MAX_DAILY_BUDGET_FOR_PERIOD'} = 100_000_000;
    $cur_data->{'MIN_WALLET_DAY_BUDGET'} //= $cur_data->{'MIN_PAY'};
    $cur_data->{'MIN_AUTOBUDGET_AVG_CPM'} = $cur_data->{'MIN_CPM_PRICE'};
}
Hash::Util::lock_hash_recurse(%_CURRENCY_DESCRIPTION);

=head2 @CURRENCIES_FOR_OTHER_COUNTRIES

    Список валют, доступных для выбора клиентам, указавшим страну, для
    которой мы не получили из баланса списка доступных валют (см. ppcFetchCountryCurrencies.pl). 

=cut

our @CURRENCIES_FOR_OTHER_COUNTRIES = qw/RUB USD EUR/;

=head2 FIRM_NAME

    Названия фирм-обслуживания по их ID по метабазе
    http://wiki.yandex-team.ru/balance/xmlrpc из описания Balance.GetFirmCountryCurrency

=cut
our %FIRM_NAME = (
    1   => 'ООО Яндекс',
    2   => 'ООО Яндекс.Украина',
    3   => 'КазНет Медиа',
    4   => 'Yandex Inc',
    7   => 'Yandex Europe AG',
    8   => 'Yandex Turkey',
);

=head2 REGION_ID_BY

    region_id Белоруссии. отдельно захардкожен, чтобы не тащить везде тяжелый geo_regions.pm

=cut

use constant REGION_ID_BY => 149;

use constant REGION_ID_TURKEY => 983;

{
# кешируем, чтобы не забирать каждый раз из БД
# инвалидируется вместе со смертью чайлдов или через COUNTRY_CURRENCIES_RELOAD_INTERVAL
    state ($currencies_by_country_for_agency, $currencies_by_country_not_for_agency);

    sub _maybe_reload_country_currency_data {
        state $last_fetched_time;
        state $last_state_of_show_new_currencies_for_clients_from_turkey;

        my $current_state_of_show_new_currencies_for_clients_from_turkey = ($SHOW_NEW_CURRENCIES_FOR_CLIENTS_FROM_TURKEY->get(60) // 0);

        if (time() - ($last_fetched_time // 0) > COUNTRY_CURRENCIES_RELOAD_INTERVAL
            || ($last_state_of_show_new_currencies_for_clients_from_turkey // 0) ne $current_state_of_show_new_currencies_for_clients_from_turkey) {
            my (%agency, %not_agency);
            $_->{other} = \@CURRENCIES_FOR_OTHER_COUNTRIES for (\%agency, \%not_agency);

            my $country_currencies = get_all_sql(PPCDICT, ['SELECT is_agency, region_id, currency',
                                                           'FROM country_currencies',
                                                            WHERE => {
                                                                is_agency__int => [0, 1],
                                                                # чтобы добавление новых валют в базу не ломало обновлённый код
                                                                currency => [get_currencies_list()],
                                                            },
                                                           'GROUP BY is_agency, region_id, currency',
                                                 ]);
            for my $row (@$country_currencies) {
                my $ref;
                if ($row->{is_agency} == 0) {
                    $ref = \%not_agency;
                } elsif ($row->{is_agency} == 1) {
                    $ref = \%agency;
                } else {
                    # не должны сюда попадать
                    die 'impossible';
                }
                push @{ $ref->{ $row->{region_id} } }, $row->{currency};
            }

            $currencies_by_country_for_agency = \%agency;
            $currencies_by_country_not_for_agency = \%not_agency;
            $last_fetched_time = time();
            $last_state_of_show_new_currencies_for_clients_from_turkey = $current_state_of_show_new_currencies_for_clients_from_turkey;
        }
    }

=head2 get_currencies_by_country_hash_for_agency

    Возвращает возможные валюты для стран для агентств и их субклиентов.
    Страны возвращаются в виде идентификаторов регионов из геобазы.

    $country2currencies = get_currencies_by_country_hash_for_agency(
        $region_id1 => ['RUB', 'USD', 'KZT', ...],
        $region_id2 => [...],
        [...]
        other => ['EUR', 'TRY', ...],
    );

=cut

    sub get_currencies_by_country_hash_for_agency {
        my ($uid, $agency_client_id, $O) = @_;
        _maybe_reload_country_currency_data();
        my %filtered_hash_map;
        my $check_row_params = {uid => $uid, AgencyID => $agency_client_id};
        hash_merge($check_row_params, $O // {});
        foreach my $region_id (keys %$currencies_by_country_for_agency) {
            foreach my $currency (@{ $currencies_by_country_for_agency->{$region_id} }) {
                 my $row = {region_id => $region_id, currency => $currency};
                 if (check_contry_currency_restrictions($row, $check_row_params)) {
                     push @{ $filtered_hash_map{$region_id} }, $currency;
                 }
            }
        }
        return \%filtered_hash_map;
    }

=head2 get_currencies_by_country_hash_not_for_agency

    Возвращает возможные валюты для стран для самостоятельных и сервисируемых клиентов.
    Страны возвращаются в виде идентификаторов регионов из геобазы.

    $country2currencies = get_currencies_by_country_hash_not_for_agency(
        $region_id1 => ['RUB', 'USD', 'KZT', ...],
        $region_id2 => [...],
        [...]
        other => ['EUR', 'TRY', ...],
    );

=cut

    sub get_currencies_by_country_hash_not_for_agency {
        my ($uid, $O) = @_;
        _maybe_reload_country_currency_data();
        my %filtered_hash_map;
        my $check_row_params = {uid => $uid};
        hash_merge($check_row_params, $O // {});
        foreach my $region_id (keys %$currencies_by_country_not_for_agency) {
            foreach my $currency (@{ $currencies_by_country_not_for_agency->{$region_id} }) {
                 my $row = {region_id => $region_id, currency => $currency};
                 if (check_contry_currency_restrictions($row, $check_row_params)) {
                     push @{ $filtered_hash_map{$region_id} }, $currency;
                 }
            }
        }
        return \%filtered_hash_map;
    }
}

# --------------------------------------------------------------------

=head2 get_currencies_list()

    Возвращает список всех поддерживаемых валют

=cut

sub get_currencies_list {
    my $cd = \%_CURRENCY_DESCRIPTION;
    return sort { (exists $cd->{$a}->{LIST_ORDER} ? $cd->{$a}->{LIST_ORDER} : 99) <=> (exists $cd->{$b}->{LIST_ORDER} ? $cd->{$b}->{LIST_ORDER} : 99) || 
                  $a cmp $b  } keys %_CURRENCY_DESCRIPTION;
}

=head2 get_currency_constant

=cut 

sub get_currency_constant($$);
sub get_currency_constant($$)
{
    my ($currency, $name) = @_;

    my $const;
    if (exists $_CURRENCY_DESCRIPTION{$currency} && exists $_CURRENCY_DESCRIPTION{$currency}->{$name}) {
        $const = $_CURRENCY_DESCRIPTION{$currency}->{$name};
    } else {
        if (!$CURRENCY_DEBUG) {
            die Carp::longmess("can't get const $name for currency $currency");
        } else {
            $const = '*UNKNOWN CURRENCY*';
        }
    }

    if (exists $_CURRENCY_DESCRIPTION_I18N_KEYS{$name}) {
        # для констант с текстом вызываем iget(), чтобы получить значение для текущего языка
        return iget($const);
    } else {
        return $const;
    }
}

=head2 get_max_autopay_card

    Переопределяем MAX_AUTOPAY_CARD для рублей по проперте
    Включаем в DIRECT-149354
    После включения выпиливаем в DIRECT-149770

=cut

sub get_max_autopay_card($) {
    my ($currency) = @_;
    if (Property->new('NEW_CARD_PAY_LIMIT')->get(60)
        && $currency eq 'RUB') {
        return 249999.99;
    }
    return get_currency_constant($currency, "MAX_AUTOPAY_CARD");
}


=head2 sql_expr_for_currency_sorting

    возвращает sql-выражение, пригодное для сортировки выборки по полю currency

    Параметры позиционные

    $field_name -- имя поля с валютой, по умолчанию 'currency'. 
    Может быть полезно, если полей несколько, и надо выбрать одну из таблиц: c.currency

=cut 

sub sql_expr_for_currency_sorting
{
    my $field_name = shift || 'currency';
    return qq!(CASE IFNULL($field_name, "YND_FIXED") WHEN "RUB" THEN 0 WHEN "UAH" THEN 2 ELSE IFNULL($field_name, "YND_FIXED") END)!;

}

=head2 is_valid_currency

    По трёхбуквенному коду валюты проверяет, заведена ли у нас такая валюта.

    $valid = is_valid_currency('USD'); # $valid => 1
    $invalid = is_valid_currency('test'); # $invalid => 0

=cut

sub is_valid_currency {
    return defined $_[0] && exists $_CURRENCY_DESCRIPTION{$_[0]} ? 1 : 0;
};

=head2 get_dominant_currency

    Выбирает из повторяющегося списка валют ту, что встречается чаще всего.
    Если встречаются одинаково часто, то победителем будет валюта с минимальным [в лексикографическом смысле] буквенным кодом.

    @currency_list = qw/RUB RUB USD RUB YND_FIXED RUB USD EUR UAH YND_FIXED/;
    $winner = get_dominant_currency(\@currency_list);
    $winner => 'RUB';

=cut

sub get_dominant_currency {
    my ($currency_list) = @_;

    return undef unless $currency_list && @$currency_list;

    my %count;
    $count{$_}++ for @$currency_list;
    my ($winner, $winner_count) = each %count;
    while (my ($maybe, $maybe_count) = each %count) {
        if ($maybe_count > $winner_count || $maybe_count == $winner_count && $maybe le $winner) {
            $winner = $maybe;
            $winner_count = $maybe_count;
        }
    }

    return $winner;
}

=head2 remove_nds

=cut

sub remove_nds {
    my ($sum, $nds) = @_;

    unless (defined $nds) {
        warn "Not specified NDS";
        return $sum;
    }

    return $sum / (1 + $nds/100);
}

=head2 add_nds

=cut

sub add_nds {
    my ($sum, $nds) = @_;

    unless (defined $nds) {
        warn "Not specified NDS";
        return $sum;
    }
    return $sum * (1 + $nds/100);
}

=head2 add_bonus

=cut

sub add_bonus {
    my ($sum, $discount) = @_;

    return ($sum > 0) ? $sum / (1 - $discount/100) : $sum;
}

sub calc_bonus {
    my ($sum, $discount) = @_;
    return ($sum > 0) ? add_bonus($sum, $discount) - $sum : 0;
}
=head2 remove_bonus

=cut

sub remove_bonus {
    my ($sum, $discount) = @_;

    return $sum * (1 - $discount/100);
}

=head2 campaign_remove_nds_and_add_bonus

    Изменяет переданный хеш кампании, убирая из sum и sum_spent НДС и добавляя скидочный бонус.

    campaign_remove_nds_and_add_bonus($camp, client_nds => $client_nds, client_discount => $client_discount);

        client_nds - НДС клиента в процентах
                     если указан, sum/sum_spent в мультивалютных кампаниях будут без НДС
        client_discount - текущая скидка клиента в процентах
                          если указан, total и sum будут с учётом скидочного бонуса (сумма которого вернётся в поле bonus)

=cut

sub campaign_remove_nds_and_add_bonus {
    my ($camp, %O) = @_;
    if ($camp->{currency} && ($camp->{currency} ne 'YND_FIXED')) {
        if ($O{client_nds}) {
            for my $field (qw/sum sum_spent total wallet_sum wallet_sum_spent wallet_total sum_balance/) {
                next unless defined $camp->{$field};
                $camp->{"${field}_include_nds"} = $O{sums_with_include_nds} ? $camp->{$field} : 0;
                $camp->{$field} = Currencies::remove_nds($camp->{$field}, $O{client_nds});
            }
        }
        if (defined $O{client_discount}) {
            my $total_without_bonus = $camp->{total} // ($camp->{sum} - $camp->{sum_spent});
            my $wallet_total_without_bonus = $camp->{wallet_total};
            for my $field (qw/sum sum_spent total wallet_sum wallet_sum_spent wallet_total sum_balance/) {
                next unless defined $camp->{$field};
                $camp->{$field} = Currencies::add_bonus($camp->{$field}, $O{client_discount});
                $camp->{"${field}_include_bonus"} = 1;
            }
            my $total_with_bonus = $camp->{total} // ($camp->{sum} - $camp->{sum_spent});
            $camp->{bonus} = $total_with_bonus - $total_without_bonus;
            # Скидочный бонус для ОС считаем только для sums_uni
            $camp->{wallet_bonus} = $wallet_total_without_bonus - $camp->{wallet_total} if $O{_is_sums_uni};
            $camp->{discount} = $O{client_discount} if $O{_is_sums_uni};
        }

        # Поддержка sums_uni
        if ($camp->{sums_uni}) {
            campaign_remove_nds_and_add_bonus($camp->{sums_uni}, %O, _is_sums_uni => 1);
        }
    }
}

=head2 currency_price_rounding(sum, currency, %O)

    Приводит цену к допустимому диапазону [MIN_PRICE;MAX_PRICE] и округляет её до ближайшего шага торгов

    Параметры:
        up => 1 — округлять до шага торгов вверх
        down => 1 — округлять до шага торгов вниз
        min_const => имя константы минимальной цены
        max_const => имя константы максимальной цены

    Пример:
        $sum = currency_price_rounding(0.02345, "USD"); # 0.02
        $price_cut = currency_price_rounding($price, $currency, up => 1);
        $price_cut = currency_price_rounding($price, $currency, down => 1);

=cut

sub currency_price_rounding 
{
    my ($price, $currency, %O) = @_;

    die "no currency given" unless $currency;
    return undef unless defined $price;

    $price = max($price, get_currency_constant($currency, $O{min_const} || 'MIN_PRICE'));
    $price = min($price, get_currency_constant($currency, $O{max_const} || 'MAX_PRICE'));

    if ($O{down}) {
        $price = round_price_to_currency_step($price, $currency, down => 1);
    } elsif ($O{up}) {
        $price = round_price_to_currency_step($price, $currency, up => 1);
    }

    return $price;
}

=head2 round_price_to_currency_step

    Округляет ставку в большую сторону до шага торгов в указанной валюте.
    Принимает именованные параметры:
        up -- округлять в большую сторону
        down -- округлять в меньшую сторону

    Эту функцию можно заменить на Math::Round::nearest_(ceil|floor) ценой дополнительной зависимости.

    $price_rounded_up = round_price_to_currency_step($price, $currency, up => 1);
    $price_rounded_down = round_price_to_currency_step($price, $currency, down => 1);

=cut

sub round_price_to_currency_step {
    my ($price, $currency, %O) = @_;

    die "invalid price: " . str($price) unless is_float($price) && $price >= 0;

    my $step = get_currency_constant($currency, 'AUCTION_STEP');
    # умножение на 1e6 нужно, чтобы избежать ошибок, связанных с неточным представлением чисел
    # пример: perl -e 'print (int(1.2/0.1))*0.1'
    my $price_rounded_down = int(round2s(($price * 1e6) / ($step * 1e6))) * $step;
    my $resulting_price;
    if ($O{down}) {
        $resulting_price = $price_rounded_down;
    } elsif ($O{up}) {
        if (($price - $price_rounded_down > $EPSILON) || $price_rounded_down == 0) {
            $resulting_price = $price_rounded_down + $step;
        } else {
            $resulting_price = $price_rounded_down;
        }
    } else {
        die "up or down should be given";
    }

    # ставка никогда не должна превышать максимально разрешенную
    $resulting_price = min($resulting_price, get_currency_constant($currency, $O{max_const} || 'MAX_PRICE'));

    return round2s($resulting_price);
}

=head2 get_currencies_description

Возвращает хеш с описанием валют

=cut

sub get_currencies_description {
    # dclone в этом месте сохраняет признак залоченности хеша, поэтому используем клонирование через json
    state $json_currencies_description = encode_json(\%_CURRENCY_DESCRIPTION);
    return decode_json($json_currencies_description);
}

=head2 get_consts_for_js



=cut

sub get_consts_for_js {
    my %currencies_for_js = %{ get_currencies_description() };
    for my $currency ( keys %currencies_for_js ){
        for my $field (qw/MIN_AUTOBUDGET_BID MAX_AUTOBUDGET_BID MAX_AUTOBUDGET_CLICKS_BUNDLE
                        MIN_AUTOBUDGET_CLICKS_BUNDLE MIN_DAY_BUDGET MIN_PAY MIN_CPC_CPA_PERFORMANCE/){
            $currencies_for_js{$currency}->{$field} = get_currency_constant($currency, $field);
        }
    }
    return \%currencies_for_js;
}

{
    # хранит ссылку на хеш соответствия is_agency => count(*)
    state $ccc_data;

    sub _maybe_reload_country_currency_cnt {
        state $last_fetched_time;
        if (time() - ($last_fetched_time // 0) > COUNTRY_CURRENCIES_RELOAD_INTERVAL) {
            $ccc_data = get_hash_sql(PPCDICT, ['SELECT is_agency, COUNT(*)',
                                               'FROM country_currencies',
                                               WHERE => {
                                                    is_agency__int => [0, 1],
                                                    # чтобы добавление новых валют в базу не ломало обновлённый код
                                                    currency => [get_currencies_list()],
                                               },
                                               'GROUP BY is_agency',
                                       ]);
            $last_fetched_time = time();
        }
    }

=head2 get_any_country_currency_cnt_for_agency

    Возвращает максимальное возможное количество сочетаний стран-валют для агентств и субклиентов.
    По доступности такого количества вариант определяем клиентов, которым доступно всё-всё,
    т.е. они не платили и могут использовать совсем всё.

    $any_country_currency_cnt = Currencies::get_any_country_currency_cnt_for_agency();
    $any_country_currency_cnt => 672

=cut

    sub get_any_country_currency_cnt_for_agency {
        _maybe_reload_country_currency_cnt();
        return $ccc_data->{1};
    }
}

=head2 round_to_currency_digit_count

    Округляет сумму в указанной валюты до количества знаков в этой валюте (см. PRECISION_DIGIT_COUNT).

    $sum_rounded = Currencies::round_to_currency_digit_count($sum, $currency);

=cut

sub round_to_currency_digit_count {
    my ($sum, $currency) = @_;

    return undef unless defined $sum;

    my $digit_count = get_currency_constant($currency, 'PRECISION_DIGIT_COUNT');
    return sprintf("%.${digit_count}f", $sum);
}

=head2 inscribe_to_constants

    Вписывает значение суммы в диапазон, заданный валютными константами.

    $sum_cut = inscribe_to_constants($sum, $currency, min_constant => 'MIN_PRICE', max_constant => 'MAX_PRICE');
    get_currency_constant($currency, 'MIN_PRICE') <= $sum_cut <= get_currency_constant($currency, 'MAX_PRICE')

=cut

sub inscribe_to_constants {
    my ($sum, $currency, %O) = @_;

    if ($O{min_constant}) {
        my $min_const = get_currency_constant($currency, $O{min_constant});
        $sum = max($sum, $min_const);
    }
    if ($O{max_constant}) {
        my $max_const = get_currency_constant($currency, $O{max_constant});
        $sum = min($sum, $max_const);
    }

    return $sum;
}

=head2 check_contry_currency_restrictions({region_id => 149, currency => "BYN"})

    Проверяет ограничения на допустимость страны-валюты

    Параметры:
        $row - hashref с полями region_id и currency
        $O - параметры фильтрации

    Результат:
        0   - пара страна/валюта не разрешена
        1   - разрешена

=cut

sub check_contry_currency_restrictions {
    my ($cc_row, $O) = @_;
    if ($cc_row->{region_id} == REGION_ID_BY() && $cc_row->{currency} ne "BYN") {
        return 0;
    }
    if (!$SHOW_NEW_CURRENCIES_FOR_CLIENTS_FROM_TURKEY->get(60)
                && $cc_row->{region_id} == REGION_ID_TURKEY()
                && $cc_row->{currency} ne "TRY") {
        return 0;
    }
    if (!$O->{force_allow_all} && $O->{uid}) {
        return country_currency_allowed_in_direct_for_user($cc_row->{currency}, $O->{uid});
    } 
    if (!$O->{force_allow_all} && $O->{AgencyID}) {
        return country_currency_allowed_in_direct_for_agency_subclient($cc_row->{currency}, $O->{AgencyID}); 
    }
    return $O->{force_allow_all} ? 1 : 0;
}

=head2 country_currency_allowed_in_direct_for_user

    Проверяет по uid, доступна ли связка страны и валюты клиенту

=cut

sub country_currency_allowed_in_direct_for_user {
    my ($currency, $uid) = @_;
    return $currency ne "GBP" || gbp_allowed_for_uid($uid);
}

=head2 country_currency_allowed_in_direct_for_agency_subclient

    Проверяет по uid, доступна ли связка страны и валюты клиенту

=cut

sub country_currency_allowed_in_direct_for_agency_subclient {
    my ($currency, $agency_client_id) = @_;
    return $currency ne "GBP" || gbp_subclients_allowed_for_agency_client_id($agency_client_id);
}

=head2 gbp_allowed_for_uid

    Проверяет по uid, разрешено ли пользоваться GBP

=cut

sub gbp_allowed_for_uid {
    my $uid = shift;
    my $uids_allowed_prop_value = $ALLOW_GBP_NONAGENCY_CLIENTS->get(60);
    return 1 if $uids_allowed_prop_value eq $GBP_UID_ALLOWED_EVERYONE;
    return $uid && any { $_ eq $uid } split(',', $uids_allowed_prop_value);
}

=head2 gbp_subclients_allowed_for_agency_client_id

    Проверяет по id клиента агентства, разрешено ли создавать субклиента в GBP

=cut

sub gbp_subclients_allowed_for_agency_client_id {
    my $client_id = shift;
    my $client_ids_allowed_prop_value = $ALLOW_GBP_AGENCY_CLIENTS->get(60);
    return 1 if $client_ids_allowed_prop_value eq $GBP_UID_ALLOWED_EVERYONE;
    return $client_id && any { $_ eq $client_id } split(',', $client_ids_allowed_prop_value);
}

my %_CURRENCY_MIN_FRONTPAGE_PRICES;

=head2 make_frontpage_min_price_hash

=cut 

sub make_frontpage_min_price_hash {
    my $data = get_all_sql(PPCDICT(), ["select * from cpm_yndx_frontpage_min_bids"]);
    my %result;
    for my $row (@$data) {
        $result{ $row->{currency} }{ $row->{region_id} }{ $row->{frontpage_type} } = $row->{min_bid};
    }
    return \%result;
}

=head2 get_frontpage_min_price

=cut 

sub get_frontpage_min_price($$$) {
    my ($currency, $geo, $pages) = @_;
    if (!%_CURRENCY_MIN_FRONTPAGE_PRICES){
        %_CURRENCY_MIN_FRONTPAGE_PRICES = %{ make_frontpage_min_price_hash() }
    }
    my $const;
    if ( exists $_CURRENCY_MIN_FRONTPAGE_PRICES{$currency} ) {
        my $min_prices_by_page;
        if ( $_CURRENCY_MIN_FRONTPAGE_PRICES{$currency}{$geo} ) {
            $min_prices_by_page = $_CURRENCY_MIN_FRONTPAGE_PRICES{$currency}{$geo};
        } else {
            $min_prices_by_page = $_CURRENCY_MIN_FRONTPAGE_PRICES{$currency}{0};
        }
        my $min_prices = hash_copy({}, $min_prices_by_page, @$pages);

        $const = min(values %$min_prices);

    } else {
        if (!$CURRENCY_DEBUG) {
            die Carp::longmess("can't get min price for currency $currency for cpm_yndx_frontpage");
        } else {
           $const = '*UNKNOWN CURRENCY*';
        }
    }
    return $const;
}
 
=head2 get_geo_list 
 
=cut 
 
sub get_geo_list($) {
    my ($currency) = @_;
    if (!%_CURRENCY_MIN_FRONTPAGE_PRICES){
        %_CURRENCY_MIN_FRONTPAGE_PRICES = %{ make_frontpage_min_price_hash() }
    }
    my $const;
    if (exists $_CURRENCY_MIN_FRONTPAGE_PRICES{$currency}) {
        return [ keys %{$_CURRENCY_MIN_FRONTPAGE_PRICES{$currency}} ];
    } else {
        if (!$CURRENCY_DEBUG) {
            die Carp::longmess("can't get geo for currency $currency for cpm_yndx_frontpage");
        } else {
            $const = '*UNKNOWN CURRENCY*';
        }
    }
}

=head2 get_min_price 
 
=cut 

sub get_min_price {
    my ($currency, $camp_geo, $frontpage_types, $client_id) = @_;

    my $geo_list = get_geo_list($currency);
    my @camp_geo_list = grep {$_ > 0} split ',' => $camp_geo;

    my @min_prices;
    
    for my $camp_geo (@camp_geo_list) {
        my $geo = get_geo_projection($camp_geo, {geo_list => $geo_list, ClientID => $client_id});
        push @min_prices, get_frontpage_min_price($currency, $geo, $frontpage_types);
    }
    my $min_price = min(@min_prices);

    return $min_price;
} 


1;
