package Currency::Format;

# $Id$

=head1 NAME
    
    Currency::Format -- функции для вывода (форматирования) строчек с валютами

=head1 DESCRIPTION


=cut

use Direct::Modern;

use POSIX qw/floor/;

use TTTools qw();
use Currencies;
use Currency::Rate;
use Currency::Pseudo;

use Yandex::I18n;
use Yandex::ListUtils qw(nsort);
use Yandex::TimeCommon;

use base qw/Exporter/;
our @EXPORT = qw(
    format_sum_of_money
    format_sum_of_money_pseudo
    format_const
    format_const_millis
    format_currency
    conv_unit_explanation
);

use utf8; 


=head2 format_sum_of_money

    Возвращает строчку "денежная сумма". 
    Параметры позиционные: 
        $sum      -- сумма
        $currency -- код валюты
        $opt      -- ссылка на хеш с необязательными именованными параметрами:
                         delim => разделитель суммы и валюту
                         nds => писать ли фразу "включая НДС" (1), "без учёта НДС" (0) или нет (undef)
                                для фишек про НДС не пишем вне зависимости от этого флага
                         discount => значение скидки в процентах
                                     для клиентов с без скидки (со скидкой 0%) добавляется уточнение об учёте скидки

    $sum = format_sum_of_money('KZT', 123.4567);
    $sum => '123.46 тенге';

    $sum = format_sum_of_money('KZT', 123.4567, {delim => '___', nds => 1});
    $sum => '123.46___тенге (включая НДС)'

    $sum = format_sum_of_money('KZT', 123.4567, {delim => '___', nds => 1, discount => 10});
    $sum => '123.46___тенге (включая НДС и без учёта скидки)'

=cut

sub format_sum_of_money
{
    my ($currency, $sum, $opt) = @_;

    $opt ||= {};
    my $delim = defined $opt->{delim} ? $opt->{delim} : ' ';
    my $str = sprintf "%s$delim%s", TTTools::format_price($sum), format_currency($currency);
    if (defined $opt->{nds} && $currency ne 'YND_FIXED') {
        if ($opt->{discount} && $opt->{discount} > 0) {
            if ($opt->{nds} && $opt->{nds} > 0) {
                $str = iget("%s%sвключая НДС и без учёта скидки", $str, $delim);
            } else {
                $str = iget("%s%sбез учёта НДС и скидки", $str, $delim);
            }
        } else {
            if ($opt->{nds} && $opt->{nds} > 0) {
                $str = iget("%s%sвключая НДС", $str, $delim);
            } else {
                $str = iget("%s%sбез учёта НДС", $str, $delim);
            }
        }
    }

    return $str;
}

=head2 format_sum_of_money_pseudo

    то же, что format_sum_of_money, но для псевдовалют

    Параметры позиционные: 
        $pseudo_currency_id -- псевдовалюта
        $sum      -- сумма

    После упразднения фиксированных у.е. 
    TODO псевдовалюты можно упразднить, функцию тоже

=cut

sub format_sum_of_money_pseudo
{
    my ($pseudo_currency_id, $sum, $delim) = @_;

    my $pseudo_currency = get_pseudo_currency(id => $pseudo_currency_id);

    $delim ||= ' ';
    my $str = sprintf("%s$delim%s", 
        TTTools::format_price($sum*$pseudo_currency->{rate}), 
        iget($pseudo_currency->{name})
    );

    return $str;
}


=head2 format_const_millis($currency, $cont_name)

    Возвращает значение и валюту для одной из валютных констант в виде суммы умноженной на 1e6 (для API).

    Параметры позиционные:
        $currency -- валюта (RUB|EUR|KZT|...); обязательный
        $const_name -- название константы (MIN_PRICE|MAX_PRICE|MIN_PRICE_FOR_MFA|...); полный список см. в %Currencies::CURRENCY_FIELDS; обязательный

=cut

sub format_const_millis {
    my ($currency, $const_name) = @_;
    return convert_to_millis(get_currency_constant( $currency, $const_name ));
}


=head2 format_const

    Возвращает значение и валюту для одной из валютных констант

    Параметры позиционные:
        $currency -- валюта (RUB|EUR|KZT|...); обязательный
        $const_name -- название константы (MIN_PRICE|MAX_PRICE|MIN_PRICE_FOR_MFA|...); полный список см. в %Currencies::CURRENCY_FIELDS; обязательный
        $pseudo_currency_id -- псевдовалюта; необязательный
        $opt -- ссылка на хеш с дополнительными параметрами:
            nds => писать ли фразу "включая НДС" (1), "без учёта НДС" (0) или нет (undef)

    $const_sum_str = format_const('KZT', 'MIN_PRICE');
    $const_sum_str => '1.5 тенге';


    TODO: упразднить $pseudo_currency_id вместе с псевдовалютами

=cut

sub format_const
{
    my ($currency, $const_name, $pseudo_currency_id, $opt) = @_;

    $opt ||= {};
    die "".(caller(0))[3].": illegal combination: currency $currency and pseudo_currency $pseudo_currency_id" 
        if $currency ne 'YND_FIXED' && $pseudo_currency_id;

    my $sum = get_currency_constant( $currency, $const_name ); 

    my $str = $pseudo_currency_id ?
        format_sum_of_money_pseudo($pseudo_currency_id, $sum) : 
        format_sum_of_money($currency, $sum, {nds => $opt->{nds}});

    return $str;
}

=head2 format_currency

    Возвращает строчку "название валюты"
    Принимает один обязательный позиционный параметр:
    $currency -- код валюты
    И необязательные именованные параметры:
    full => 1|0 -- если установлен в истину, будет возвращено полное наимнование валюты (по умолчанию возвращается краткое)

    SET currency = format_currency('RUB'); # currency => 'руб.'
    SET currency = format_currency('RUB', {full => 1}); # currency => 'российские рубли'

=cut

sub format_currency
{
    my ($currency, $O) = @_;

    return get_currency_constant($currency, ($O && ref($O) eq 'HASH' && $O->{full} ? 'full_name' : 'name'));
}


=head2 conv_unit_explanation

    Возвращает строчку "1 у.е. = ..."
    Параметры позиционные 
    $work_currency -- код рабочей валюты (а надо ли?)
    $pay_currency -- код валюты платежа
    $opt -- опциональные параметры:
        delim — разделитель для строки 1 у.е. = ..., по умолчанию неразрывный пробел
        nds — писать ли фразу "включая НДС" (1), "без учёта НДС" (0) или нет (undef)
        discount — значение скидки в процентах, добавляется уточнение об учёте скидки
        with_history — 1/0, показывать ли исторические значения курсов

    conv_unit_explanation('YND_FIXED', 'RUB');    # '1 у.е. = XXX руб.' 
    conv_unit_explanation('YND_FIXED', 'RUB', {nds => 1});    # '1 у.е. = 30 руб. (включая НДС)'

=cut
sub conv_unit_explanation
{
    my ($work_currency, $pay_currency, $opt) = @_;

    die unless $work_currency eq 'YND_FIXED';
    $opt ||= {};
    $opt->{delim} ||= "\x{00a0}";

    my $name = get_currency_constant($work_currency, 'name');
    my $one_ynd_fixed_equals_template = "1$opt->{delim}$name$opt->{delim}=$opt->{delim}%s";

    my ($rate_str, @history_rates);
    if ( exists $Currency::Pseudo::PSEUDO_CURRENCIES->{$pay_currency} ){
        # псевдовалюта
        $rate_str = format_sum_of_money_pseudo($pay_currency, 1, $opt->{delim});
    } else {
        # валюта настоящая
        my $convert_with_nds = (defined $opt->{nds}) ? $opt->{nds} : 1;
        my $rate = convert_currency(1, $work_currency, $pay_currency, with_nds => $convert_with_nds);
        $rate_str = format_sum_of_money($pay_currency, $rate, {nds => $opt->{nds}, discount => $opt->{discount}, delim => $opt->{delim}});

        # собираем строку истории курса в виде массива строк "до YYYY.MM.DD г. <курс>"
        if ($opt->{with_history}) {
            my $rates = $Currency::Rate::_CURRENCY_RATE{'YND_FIXED'}{$pay_currency};
            if (ref($rates) eq 'HASH' && scalar keys(%$rates) > 1) {
                # идём от свежих курсов, подставляя во фразу дату предыдущего курса
                # самый актуальный курс пропускаем, т.к. он не история, а написан рядом как основной из $rate
                my $date_till;
                for my $first_date (reverse nsort keys %$rates) {
                    if (defined $date_till) {
                        my $date_till_human = mysql2human_date($date_till);
                        my $nds_key = ($opt->{nds}) ? 'with_nds' : 'without_nds';
                        my $history_rate = $rates->{$first_date}->{$nds_key};
                        my $history_rate_str = format_sum_of_money($pay_currency, $history_rate, {delim => $opt->{delim}});
                        my $history_str = sprintf($one_ynd_fixed_equals_template, $history_rate_str);
                        unshift @history_rates, iget('до%s%s%sг.%s%s', $opt->{delim}, $date_till_human, $opt->{delim}, $opt->{delim}, $history_str);
                    }
                    $date_till = $first_date;
                }
            }
        }
    }

    my $str = sprintf($one_ynd_fixed_equals_template, $rate_str);
    if (@history_rates) {
        my $history_str = join ', ', @history_rates;
        $str .= iget("%s(для оплат %s)", $opt->{delim}, $history_str);
    }

    return $str;
}

=head2 convert_to_millis {

    Выводить суммы в виде целых чисел, сумма умножается на миллион

=cut

sub convert_to_millis {
    my $num = shift;
    my $epsilon = $Currencies::EPSILON;
    if (!defined $num) {
        return;
    } else {
        return floor(($num * 1000_000) + $epsilon);
    }
}

1;
