package Currency::Pseudo;

=head1 NAME

    Currency::Pseudo

=head1 DESCRIPTION

    Модуль для констант и функций, относящихся к псевдовалютам

=cut

use Direct::Modern;

use Currencies;
use Currency::Rate;

use Hash::Util;

use Yandex::URL qw(get_num_level_domain);

use base qw/Exporter/;
our @EXPORT = qw(
    get_pseudo_currency
    get_real_currency_from_pseudo_currency_id
);

# описания "псевдовалют", соответствия валют доменам (для Легкого)
# псевдовалюта -- фиксированный множитель для у.е. + название. Настоящей валютой не является, т.к. жестко привязана к у.е.
# multicurrency -- ??? что будем делать с нынешними Легкими кампаниями? Переведем всех на рубли без выбора?
our $PSEUDO_CURRENCIES = {
    ynd_fixed => 'YND_FIXED',
    rub => 'RUB',
    grivna => 'UAH',
    tenge => 'KZT',
    tr_lira => 'TRY',
};
Hash::Util::lock_hash_recurse(%$PSEUDO_CURRENCIES);

# multicurrency -- ??? нужна ли будет эта логика с валютами по доменам?
# Возможно, умолчальная валюта при создании новой кампании
my %pseudo_currency_by_domain =(
    ru  => 'rub',
    com => 'rub',
    ua  => 'grivna',
    kz  => 'tenge',
    tr  => 'tr_lira'
);
Hash::Util::lock_hash_recurse(%pseudo_currency_by_domain);

=head2 get_pseudo_currency( %options )

    Отдает хеш с описанием валюты
    {
        id   => 'rub',
        name => 'руб.',
        rate => 30,
        rate_without_nds => 25.42,
    }

    Параметры: домен или id валюты, но не оба одновременно

=cut

sub get_pseudo_currency
{
    my (%options) = @_;

    my $id;

    if ($options{id}){
        $id = $options{id};
    } else {
        my $domain = get_num_level_domain($options{hostname}, 1);
        $id = (exists $pseudo_currency_by_domain{$domain} && $pseudo_currency_by_domain{$domain})
           || 'rub';
    }

    die "unknown pseudo_currency id '$id'" unless exists $PSEUDO_CURRENCIES->{$id};

    my $real_currency = $PSEUDO_CURRENCIES->{$id};
    return {
        id => $id,
        name => get_currency_constant($real_currency, 'name'),
        rate => convert_currency(1, 'YND_FIXED', $real_currency, with_nds => 1),
        rate_without_nds => convert_currency(1, 'YND_FIXED', $real_currency, with_nds => 0),
    };
}

=head2 get_template_pseudo_currency_data

    Возвращает информацию о псевдовалютах для использования в шаблонах лёгкого интерфейса

    hash_merge $vars, Currencies::get_template_pseudo_currency_data($r->hostname);

=cut

sub get_template_pseudo_currency_data {
    my ($hostname) = @_;

    my $pseudo_currency = get_pseudo_currency( hostname => $hostname );
    my %vars = (
        pseudo_currency => $pseudo_currency,
    );
    if ($pseudo_currency->{id} eq 'ynd_fixed'){
        my $domain = get_num_level_domain($hostname, 1);
        $vars{explain_currency} = $pseudo_currency_by_domain{$domain};
    }
    $vars{pay_currency} = $vars{explain_currency} || $pseudo_currency->{id};

    return \%vars;
}

=head2 get_real_currency_from_pseudo_currency_id

    Возвращает идентификатор валюты на основе идентификатора псевдовалюты

=cut

sub get_real_currency_from_pseudo_currency_id {
    my ($pseudo_currency_id) = @_;

    return $PSEUDO_CURRENCIES->{$pseudo_currency_id};
}

1;
