package Currency::Rate;

# $Id$

=head1 NAME
    
    Currency::Rate

=head1 DESCRIPTION

    Модуль для констант и функций, относящихся к пересчету денежных сумм между разными валютами.

=cut

use strict;
use warnings;
use feature 'state';

use Hash::Util qw/lock_hash_recurse/;
use Storable qw/dclone/;

use Settings;

use Yandex::Validate qw/is_valid_date is_valid_float/;
use Yandex::DBTools;
use Yandex::TimeCommon qw(today mysql2unix);
use Yandex::ListUtils qw(nsort);

use Yandex::HashUtils qw/hash_merge hash_cut/;
use List::MoreUtils qw/all/;
use Currencies;
use Data::Dumper;

use base qw/Exporter/;
our @EXPORT = qw(
    mass_get_currency_rate_in_roubles
    get_currency_rate_in_roubles
    convert_currency
    get_conv_unit_rate
);

use utf8; 


# Фиксированные курсы
our %_CURRENCY_RATE = (
    YND_FIXED => {
        RUB => {
                   # старое значение НДС совместимо с историческими данными, для новых данных конвертация не должна использоваться
                   # ошибка в текущих сценариях использования признана некритичной (DIRECT-87957)
                   0 => { without_nds => remove_nds(30, 100*$Settings::NDS_RU_DEPRECATED), with_nds => 30 },
        },
        UAH => {
                   0 => { without_nds => 5, with_nds => add_nds(5, 100*$Settings::NDS_UA) },
            20121201 => { without_nds => 6.75, with_nds => add_nds(6.75, 100*$Settings::NDS_UA) },
            20140901 => { without_nds => 10, with_nds => 12 },
        },
        USD => {
                   0 => { without_nds => 0.85, with_nds => 0.85 },
            20141201 => { without_nds => 0.56, with_nds => 0.56 },
            20150401 => { without_nds => 0.41, with_nds => 0.41 },
        },
        EUR => {
                   0 => { without_nds => 0.65, with_nds => 0.65 },
            20141201 => { without_nds => 0.45, with_nds => 0.45 },
            20150401 => { without_nds => 0.39, with_nds => 0.39 },
        },
        KZT => {
                   0 => { without_nds => 130, with_nds => 145.6 },
            20141201 => { without_nds => 100, with_nds => 112 },
            20150401 => { without_nds => 75, with_nds => 84 },
            20150701 => { without_nds => 100, with_nds => 112 },
            20151001 => { without_nds => 105, with_nds => 117.6 },
        },
        CHF => {
                   0 => { without_nds => 0.77, with_nds => 0.77 },
            20141201 => { without_nds => 0.53, with_nds => 0.57 },
            20150401 => { without_nds => 0.41, with_nds => 0.41 },
        },
        TRY => {
                   0 => { without_nds => 1.7, with_nds => 2.0 },
            20141201 => { without_nds => 1.2, with_nds => 1.42 },
            20150401 => { without_nds => 1.07, with_nds => 1.26 },
        },
        BYN => {
                   0 => { without_nds => 0.8, with_nds => 0.8 },
        },
        GBP => {
                   0 => { without_nds => 0.65, with_nds => 0.65 },
            20141201 => { without_nds => 0.45, with_nds => 0.45 },
            20150401 => { without_nds => 0.39, with_nds => 0.39 },
        },
    },
);

lock_hash_recurse(%_CURRENCY_RATE);

=head2 mass_get_currency_rate_in_roubles

    Возвращает курсы указанных валюты по отношению к рублю на указанную дату, если дата не указана — на текущую дату.
    Принимает позиционные параметры:
        currency_codes — ссылка на список директовских кодов валют (['UAH', 'USD', ...]); обязательный параметр
        date           — дата, на которую требуются курсы в формате YYYY-MM-DD; необязательный параметр
    Возвращает значения курсов в виде ссылки на хеш. Ключи — коды валют, значения — курсы.
    Если курс на указанную дату не найден, ключ отсутствует в хеше. В случаях каких-либо ошибок умирает.

    $rate = mass_get_currency_rate_in_roubles(['USD','EUR','YNDX_FIXED']);
    $rate = mass_get_currency_rate_in_roubles(['USD','EUR','YNDX_FIXED'], '2011-06-25');
    $rate => {USD => 32.12345, EUR => 45.6789, YNDX_FIXED => 30};

=cut

sub mass_get_currency_rate_in_roubles {
    my($currency_codes, $date) = @_;

    state $rates_daily_cache = {};

    die 'currency_codes не указан или указан неверно: ' . Dumper($currency_codes) unless $currency_codes && ref($currency_codes) eq 'ARRAY' && @$currency_codes && all { $_ && !ref($_) } @$currency_codes;
    die 'date не указан или указан неверно: ' . Dumper($date) unless !defined $date || ($date && is_valid_date($date));

    my (%currency_rate_in_roubles, %balance_currency2our_currenies, @currency_codes_to_fetch, @balance_currency_codes_to_fetch);
    for my $currency_code (@$currency_codes) {
        if ($currency_code eq 'RUB') {
            $currency_rate_in_roubles{$currency_code} = 1;
        } else {
            my $balance_currency = $Currencies::_CURRENCY_DESCRIPTION{$currency_code}->{BALANCE_CURRENCY_NAME};
            die "для валюты $currency_code нет кода валюты в Балансе" unless $balance_currency;
            push @{$balance_currency2our_currenies{$balance_currency}}, $currency_code;
            push @balance_currency_codes_to_fetch, $balance_currency;
            push @currency_codes_to_fetch, $currency_code;
        }
    }

    if (%balance_currency2our_currenies) {
        $date = today() unless defined $date;

        my $rates = ($rates_daily_cache->{$date} //= {});
        return hash_merge(\%currency_rate_in_roubles, hash_cut($rates, @currency_codes_to_fetch))
            if all { $rates->{$_} } @currency_codes_to_fetch;

        my $currency_rates_data = get_all_sql(PPCDICT, ['SELECT currency, rate FROM currency_rates', WHERE => {currency => \@balance_currency_codes_to_fetch, date => $date}]);
        if ($currency_rates_data && @$currency_rates_data) {
            for my $currency_rate(@$currency_rates_data) {
                my $our_currencies = $balance_currency2our_currenies{$currency_rate->{currency}};
                for my $our_currency(@$our_currencies) {
                    $currency_rate_in_roubles{$our_currency} = $currency_rate->{rate};
                    $rates->{$our_currency} = $currency_rate->{rate};
                }
            }
        }
    }

    return \%currency_rate_in_roubles;
}

=head2 get_currency_rate_in_roubles

    Возвращает курс указанной валюты по отношению к рублю на указанную дату, если дата не указана — на текущую дату.
    Принимает позиционные параметры:
        currency_code — директовский код валюты ('UAH', 'USD'  и т.п.); обязательный параметр
        date          — дата, на которую требуется курс в формате YYYY-MM-DD; необязательный параметр
    Возвращает значение курса. Если курс на указанную дату не найден, умирает. В случаях каких-либо ошибок тоже умирает.

    $rate = get_currency_rate_in_roubles('YNDX_FIXED');
    $rate = get_currency_rate_in_roubles('UAH', '2011-06-25');
    $rate => 4.5678;

=cut

sub get_currency_rate_in_roubles {
    my($currency_code, $date) = @_;

    my $rates = mass_get_currency_rate_in_roubles([$currency_code], $date);
    die "не найден курс валюты $currency_code на $date" unless $rates->{$currency_code};
    return $rates->{$currency_code};
}

=head2 convert_currency

    Переводит сумму из одной валюты в другую по курсам на указанную дату. Если дата не указана, используется курс на сегодня.
    Принимает позиционные параметры:
        sum             — денежная сумма в исходной валюте
        source_currency — 3-х символьный директовский код исходной валюты ('USD', 'EUR', 'UAH', ...) 
        target_currency — 3-х символьный директовский код валюты, в которую необходим перевод ('USD', 'EUR', 'UAH', ...)
    Принимает именованные параметры:
        date            — дата, на которую требуется курс в формате YYYY-MM-DD; необязательный параметр
        with_nds        — указывает, что исходная и результирующая сумма должны включать в себя НДС; нужно для перевода из денег в у.е.
    Возвращает сумму в целевой валюте. Если курса на заданную дату нет, умирает. В случаях каких-либо ошибок тоже умирает.
    Возвращаемая сумма НЕ ВКЛЮЧАЕТ НДС, если не указан параметр with_nds.

    $sum_usd = convert_currency(100, 'RUB', 'USD');
    $sum_usd = convert_currency(100, 'RUB', 'USD', date => '2011-06-25');
    $sum_usd => 3.106;

    $sum_conv_units = convert_currency(100, 'RUB', 'YND_FIXED', with_nds => 1);

=cut

sub convert_currency {
    my ($sum, $source_currency, $target_currency, %O) = @_;

    # TODO: могут быть суммы отрицательными!
    die 'sum не указан или указан неверно: ' . Dumper($sum) unless defined $sum && !ref($sum);
    die 'source_currency не указан или указан неверно: ' . Dumper($source_currency) unless $source_currency && !ref($source_currency);
    die 'second_currency не указан или указан неверно: ' . Dumper($target_currency) unless $target_currency && !ref($target_currency);

    return $sum if $source_currency eq $target_currency;
    return 0 if $sum == 0;

    my $date = $O{date};
    die 'date не указан или указан неверно: ' . Dumper($date) unless !defined $date || ($date && is_valid_date($date));
    $date = today() unless defined $date;

    if ($source_currency eq 'YND_FIXED' || $target_currency eq 'YND_FIXED') {
        my $other_currency = ($source_currency eq 'YND_FIXED') ? $target_currency : $source_currency;
        my $rates = $_CURRENCY_RATE{'YND_FIXED'}->{$other_currency};
        my $rate;
        if (ref($rates) eq 'HASH') {
            my $date_ts = mysql2unix($date);
            for my $first_date (nsort keys %$rates) {
                last if $date_ts < mysql2unix($first_date);
                my $nds_key = ($O{with_nds}) ? 'with_nds' : 'without_nds';
                $rate = $rates->{$first_date}->{$nds_key};
            }
        } elsif (ref($rates) eq 'HASH' && scalar(keys %$rates) == 1) {
            $rate = (values %$rates)->[0];
        } else {
            my $rate_rub = convert_currency(1, $source_currency, 'RUB', date => $date);
            $rate = convert_currency($rate_rub, 'RUB', $target_currency, date => $date);
        }

        if ($source_currency eq 'YND_FIXED') {
            return $sum * $rate;
        } else {
            return $sum / $rate;
        }
    } else {
        # перевод через рубли
        my $rates = mass_get_currency_rate_in_roubles([$source_currency, $target_currency], $date);
        die "не найден курс валюты $source_currency на $date" unless $rates->{$source_currency} && $rates->{$source_currency} > 0;
        die "не найден курс валюты $target_currency на $date" unless $rates->{$target_currency} && $rates->{$target_currency} > 0;
        return $sum*$rates->{$source_currency}/$rates->{$target_currency};
    }
}

=head2 get_conv_unit_rate

    Возвращает курс у.е. к рублю
    Использоваться может только на старых немультивалютных страницах или в Баяне.
    multicurrency: Со временем хочется оторвать и переехать либо на псевдовалюты, либо на честную мультивалютность.

=cut

sub get_conv_unit_rate {
    return convert_currency(1, 'YND_FIXED', 'RUB', with_nds => 1);
}

# --------------------------------------------------------------------

=head2 get_sql_currencies_rates_to_ynd_fixed

Для перевода валюты из YND_FIXED, отдаем case с текущими курсами для YND_FIXED
Принимает имя поля с валютой.

Отдаем:
    CASE `currency` WHEN 'KZT' THEN '84' WHEN 'EUR' THEN '0.39' WHEN 'YND_FIXED' THEN '1' WHEN 'CHF' THEN '0.41' WHEN 'UAH' THEN '12'... ELSE '1' END

Пример:
    my $currencies_case_sql = Currency::Rate::get_sql_currencies_rates_to_ynd_fixed("currency");
    get_all_sql(DB, "select cid from table where sum / $currencies_case_sql > ?", $max_sum_ynd_fixed);

=cut

sub get_sql_currencies_rates_to_ynd_fixed($) {
    my $field = shift;

    my $current_currencies = {
        map {$_ => convert_currency(1, "YND_FIXED", $_, with_nds => 1)}
        get_currencies_list()
    };
    my $currencies_case_sql = sql_case($field, $current_currencies, default => 1);

    return $currencies_case_sql;
}

=head2 get_fixed_rates_for_js

    Возвращает словарь с историей курсов фишки к валютам для использования в js.
    Ожидается, что будет использоваться в функции u.currencies.convUnitExplanation
        (на замену перловой conv_unit_explanation), которая делает строчку "1 у.е. = ..."

=cut
sub get_fixed_rates_for_js {
    return dclone(\%_CURRENCY_RATE);
}

1;
