#######################################################################
#
#  Direct.Yandex.ru
#
#  DBStat
#
#  $Id$
#
#######################################################################

=encoding utf8

=head1 NAME

Stat

=head1 DESCRIPTION

DBStat

=cut

package DBStat;

use Direct::Modern;

use POSIX qw/strftime/;
use Data::Dumper;
use JSON;
use Date::Calc qw(Localtime Add_Delta_YMD Week_of_Year Monday_of_Week);
use Carp qw/longmess/;

use GeoTools;
use ShardingTools;
use Tools;
use BannerTemplates;

use Models::Banner;

use Settings;

use Yandex::DateTime;
use Yandex::TimeCommon;
use Yandex::HashUtils;
use Yandex::ListUtils qw/xuniq chunks xminus nsort xflatten xsort/;
use Yandex::ScalarUtils qw/str/;
use Yandex::Log;
use Yandex::Trace;
use PrimitivesIds;
use YaCatalogApi;
use Yandex::SendMail;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::Overshard;
use Yandex::Validate qw/is_valid_int is_valid_float/;
use Yandex::I18n;
use Yandex::Overshard;

use Campaign::Types;
use MTools;
use Direct::ResponseHelper qw/respond_data/;
use Tag;
use Holidays qw/is_holiday/;
use Primitives;

use List::Util qw(max min maxstr minstr reduce sum);
use List::MoreUtils qw/uniq zip any none/;
use Stat::Tools qw/field_list_sum field_list_sum_if_exists field_list_avg suffix_list/;
use Stat::Const qw/:base :enums/;
use Stat::OrderStatDay;

our $DEBUG = 0;



our $SPEC_PREFIX = "spec_";

=head2 new

    Конструктор класса
    Возможные параметры:
      OrderID - база данных будет выбрана автоматически
      dbh - будет использован указанный dbh

=cut
sub new {
    my $class = shift;
    my %params = @_;
    my $self = \%params;
    return bless $self, $class;
}

=head2 set_order_id

    привязывает к объекту новый OrderID

=cut
sub set_order_id {
    my ($self, $OrderID) = @_;
    $self->{OrderID} = $OrderID;
}


# Получить объект Yandex::Log
sub log {
    my $self = shift;
    return $_[0]->{log} ||= Yandex::Log->new(
                                date_suf      => "%Y%m%d", 
                                log_file_name => 'dbstat',
                                msg_prefix    => "PID:$$,OrderID:".($self->{OrderID}||0),
                                auto_rotate   => 1,
                                umask         => 0,
                                );
}

###########################################################
# Разные виды статистики (верхний уровень)
###########################################################

=head2 get_common_stat

    Возвращает общую статистику

=cut

sub get_common_stat {
    my ( $self, $oid, $start_date, $end_date, %O ) = @_;
    _debug("get_stat_phrase(".Dumper([$oid, $start_date, $end_date, %O]).")");
    my $profile = Yandex::Trace::new_profile('dbstat:get_stat_phrase');

    my $group_hash = { banner => 1 };
    if ($O{with_phrases}) {
        hash_merge $group_hash, { map { $_ => 1 } @Stat::Const::ANY_PHRASES_GROUP_BY };
    }
    my $stat_options = hash_cut \%O, qw/
        with_nds with_discount single_currency currency extra_countable_fields operator_ClientID external_countable_fields_override
    /;

    my ($stat_stream, $stat_stream_ts);

        # использовать логику выбора источника статистики заложенную в Stat::CustomizedArray
        die "Method get_stat_phrase with (use_customized_array => 1) option is supported only in Stat::CustomizedArray* classes" unless ref($self) =~ /CustomizedArray/;
        my $client_id = $O{translocal_params} ? $O{translocal_params}{ClientID} : $self->{translocal_params}{ClientID};
        $self->set_report_parameters( oid        => $oid,
                                      start_date => $start_date,
                                      end_date   => $end_date,
                                      group_by   => [ keys %$group_hash ],
                                      filter    => {attribution_model => $O{attribution_model} || get_common_stat_default_attribution()},
                                      options    => {%$stat_options,
                                                     countable_fields_by_targettype => 1,
                                                     no_spec_and_all_prefix => 1,
                                                    },
                                      translocal_params => $O{translocal_params} // $self->{translocal_params},
                                      ClientID_for_stat_experiments => $client_id,
                                      stat_type => 'common',
                                    );
        my $stat_customized_array = $self->generate_with_metrika;
        $stat_stream = $stat_customized_array->{data_array};
        $stat_stream_ts = $stat_customized_array->{stat_stream_ts};

    # Формируем хэш с данными
    my $data = {};
    my %phrase_ids;
    my %ret_ids;
    my %dyn_ids; # dyn_cond_id
    my %perf_ids; # bids_performance.perf_filter_id

    my $tag_BannerIDs;
    if (@{$O{tag_ids} // []}) {
        my $pids = Tag::get_pids_by_tags($O{tag_ids});
        my $banners = Models::Banner::get_pure_creatives([map {{pid => $_}} @$pids], {}, {get_all_images => 1});
        $tag_BannerIDs = {
            map {$_ => 1} 
            (@$banners
            ? (map {
                    my @ids = ($_->{BannerID});
                    push @ids, split /,/, $_->{all_images_BannerID};
                    @ids} @$banners)
            : ())
        };
    }

    my $banners_info = {};
        $banners_info = Stat::Tools::mass_get_banner_info(BannerID => [uniq grep {$_} map { $_->{BannerID} } @$stat_stream],
                                                          translocal_params => $self->{translocal_params},
                                                          sharding_params => {OrderID => $oid});
    while( my $row = shift @$stat_stream ) {
        if (!is_valid_int($row->{BannerID}, 1)) {
            my $msg = "Skipping bad BannerID ($row->{BannerID}) for OrderID $oid and dates between $start_date and $end_date: " . to_json($row);
            send_alert(Carp::longmess($msg), 'get_stat_phrase error');
            next;
        }

        next if $tag_BannerIDs && !$tag_BannerIDs->{$row->{BannerID}};

        # Работаем с баннером
        if ( !defined $data->{$row->{BannerID}} ) {
            # Получаем информацию о баннере
            $data->{$row->{BannerID}} = $banners_info->{$row->{BannerID}} //
                                        Stat::Tools::get_banner_info( BannerID => $row->{BannerID}, 
                                                                      translocal_params => $self->{translocal_params},
                                                                      sharding_params => {OrderID => $oid} );
        }

        if ($O{with_phrases}) {
            if ($row->{ContextType} == $CONTEXT_TYPE_PHRASE) {
                $phrase_ids{$row->{PhraseID}} = undef;
            } elsif ($row->{ContextType} == $CONTEXT_TYPE_RET) {
                $ret_ids{$row->{PhraseID}} = undef;
            } elsif ($row->{ContextType} == $CONTEXT_TYPE_DYNAMIC) {
                $dyn_ids{$row->{PhraseID}} = undef
            } elsif ($row->{ContextType} == $CONTEXT_TYPE_PERFORMANCE) {
                $perf_ids{$row->{PhraseID}} = undef;
            }
        }

        my $banner = $data->{$row->{BannerID}};
        # Считаем суммарную статистику по баннеру
        $self->aggregation_stat($banner,$row);
        if ($O{with_phrases}) {
            $self->calc_sec_stat( $row );
            push @{ $banner->{phrases} }, $row;
        }
    }

    my $phrases_text = {};
    my $retargetings_text = {};
    my $performance_text = {};

    my $dynamic_data = Stat::Tools::get_dynamic_data([keys %dyn_ids],
                                                     OrderID => $oid,
                                                     with_condition => (defined $O{dynamic_data_ref} ? 1 : 0),
                                                     );
    if (defined $O{dynamic_data_ref}) {
        ${ $O{dynamic_data_ref} } = $dynamic_data;
    }

    # Теперь делаем из дат - массив
    my @banners;
    if ( any { defined $_->{group_name} } values %$data ) {
        @banners = xsort { $_->{group_name} // '', $_->{pid} // 0, $_->{bid} // 0 } values %$data;
    } else {
        @banners = xsort { $_->{bid} // 0 } values %$data;
    }
    my $res = { banners=> \@banners };
    for my $banner ( @banners ) {
        # Теперь считаем ctr и ср. цену клика
        $self->calc_sec_stat( $banner);
        next unless $O{with_phrases};
        # Раскидываем фразы по массивам
        for my $phr ( @{$banner->{phrases}} ) {
            my $PhraseID = $phr->{PhraseID};
            if ($phr->{ContextType} == $CONTEXT_TYPE_PHRASE) {
                $phr->{text} = $phr->{phrase} || $phrases_text->{$phr->{PhraseID}} || '';
            } elsif ($phr->{ContextType} == $CONTEXT_TYPE_RET) {
                $phr->{text} = $phr->{phrase} || $retargetings_text->{$phr->{PhraseID}} || '';
            } elsif ($phr->{ContextType} == $CONTEXT_TYPE_DYNAMIC) {
                $phr->{text} = $phr->{phrase} || $dynamic_data->{$phr->{PhraseID}}->{name} || '';
            } elsif ($phr->{ContextType} == $CONTEXT_TYPE_PERFORMANCE) {
                $phr->{text} = $phr->{phrase} || $performance_text->{$phr->{PhraseID}} || '';
            } elsif ($phr->{ContextType} == $CONTEXT_TYPE_BROADMATCH) {
                $phr->{text} = Stat::Tools::special_phrase_title($BROADMATCH_PHRASE_ID);
            } else {
                $phr->{text} = '';
            }


            my $postfix = '';
            if ( $phr->{text} =~ /^\@(\d+)/ ) {
                $phr->{text} = get_category_name( $1 );
                $postfix = '_categories';
            } elsif ($phr->{category_id}) {
                $phr->{text} = $phr->{phrase};
                $postfix = '_categories';
            }
            if (exists $SPECIAL_PHRASES{$PhraseID}
                && none { $phr->{ContextType} == $_ } @NOT_TEXT_CONTEXT_TYPE
            ) {
                $phr->{special_flag} = 1;
                push @{ $banner->{"active$postfix"} }, $phr;
            } elsif (any { $phr->{ContextType} == $_ } @NOT_TEXT_CONTEXT_TYPE) {
                push @{ $banner->{"active$postfix"} }, $phr;
            } elsif ( defined $banner->{phrases_hash}->{$PhraseID} && $banner->{phrases_hash}->{$PhraseID}->{rank} ) {
                delete $banner->{phrases_hash}->{$PhraseID};
                push @{ $banner->{"active$postfix"} }, $phr;
            } elsif ( defined $banner->{phrases_hash}->{$PhraseID} ) {
                delete $banner->{phrases_hash}->{$PhraseID};
                push @{ $banner->{"bad_ctr$postfix"} }, $phr;
            } else {
                push @{ $banner->{"past$postfix"} }, $phr;
            }
        }
        # Добиваем фразы с нулевой статистикой
        while( my ( $PhraseID, $phr ) = each %{$banner->{phrases_hash}} ) {
            $phr->{text} = $phr->{phrase};
            for my $f ( field_list_sum(), field_list_avg() ) {
                for my $suf ( suffix_list() ) {
                    $phr->{"$f$suf"} = undef;
                }
            }

            if ( $phr->{rank} ) {
                push @{ $banner->{"active"} }, $phr;
            } else {
                push @{ $banner->{"bad_ctr"} }, $phr;
            }
        }
    }
    # Общая статистика по кампании
    my $days_num = Stat::OrderStatDay::get_order_days_num( $oid, $start_date, $end_date ) || 1;
    $self->calc_order_stat( $res, $days_num );

    # Если заданы параметры пейджинга по баннерам - оставляем только нужный чанк
    $res->{stat_banners_num} = scalar(@banners);
    if ($O{banners_on_page} && $O{page}) {
        @banners = splice @banners, $O{banners_on_page}*($O{page}-1), $O{banners_on_page};
    }


    hash_merge $res, {stat_stream_ts => $stat_stream_ts} if defined $stat_stream_ts;

    return $res;
}


sub _mass_get_region_id_by_oid {
    my ($OrderIDs) = @_;

    my %oid2region_id;
    foreach_shard OrderID => $OrderIDs, chunk_size => 20_000, sub {
        my ($shard, $OrderIDs_chunk) = @_;
        my $oid2timezone_id = get_hash_sql(PPC(shard => $shard),
                ["SELECT OrderID, timezone_id FROM campaigns", WHERE => {OrderID => $OrderIDs_chunk}]);
        for my $OrderID (keys %$oid2timezone_id) {
            $oid2region_id{$OrderID} = TimeTarget::cached_tz_by_id($oid2timezone_id->{$OrderID})->{country_id};
        }
    };
    return \%oid2region_id;
}


sub _get_region_id_by_oid {
    my ($OrderID) = @_;

    return _mass_get_region_id_by_oid([$OrderID])->{$OrderID};
}


=head2 get_stat_phrase_date

    Получить данные для вкладки "Фразы по дням"

=cut

sub get_stat_phrase_date {
    my ( $self, $oid, $start_date, $end_date, $dateagg, $filter, $limits, %O ) = @_;
    _debug("get_stat_phrase_date(".Dumper([$oid, $start_date, $end_date, $dateagg, $filter, $limits]).")");
    my $profile = Yandex::Trace::new_profile('dbstat:get_stat_phrase_date');

    my $group_hash = { banner => 1, date => $dateagg, map { $_ => 1 } @Stat::Const::ANY_PHRASES_GROUP_BY };
    my $filter_hash = { single_goal_id => $filter->{single_goal_id},
                        $filter->{attribution_model} ? (attribution_model => $filter->{attribution_model}) : () };

    my $stat_options = hash_cut \%O, qw/
        with_nds with_discount single_currency currency with_reach_stat extra_countable_fields four_digits_precision
        operator_ClientID external_countable_fields_override
    /;

    my ($stat_stream, $stat_ts, $reach_stat_totals);
        # использовать логику выбора источника статистики заложенную в Stat::CustomizedArray
        die "Method get_stat_phrase_date with (use_customized_array => 1) option is supported only in Stat::CustomizedArray* classes" unless ref($self) =~ /CustomizedArray/;
        my $client_id = $O{translocal_params} ? $O{translocal_params}{ClientID} : $self->{translocal_params}{ClientID};
        $self->set_report_parameters( oid        => $oid,
                                      start_date => $start_date,
                                      end_date   => $end_date,
                                      group_by   => [ keys %$group_hash ],
                                      filter     => $filter_hash,
                                      date_aggregation_by => $dateagg,
                                      ClientID_for_stat_experiments => $client_id,
                                      options    => {%$stat_options,
                                                     countable_fields_by_targettype => $stat_options->{with_reach_stat} ? 0 : 1,
                                                     no_spec_and_all_prefix => 1,
                                                    },
                                      translocal_params => $O{translocal_params} // $self->{translocal_params},
                                      stat_type => 'phrase_date',
                                    );
        my $stat_customized_array = $self->generate_with_metrika;
        $stat_stream = $stat_customized_array->{data_array};
        $stat_ts = $stat_customized_array->{stat_stream_ts};
        if ($stat_options->{with_reach_stat}) {
            $reach_stat_totals = hash_cut($stat_customized_array, qw/uniq_viewers avg_view_freq tavg_cpm/);
        }

    # Формируем хэш с данными
    my $data = {};
    my (%phrase_ids, %ret_ids, %dyn_ids, %perf_ids);

    my $tag_BannerIDs;
    if (@{$filter->{tag_ids} // []}) {
        my $pids = Tag::get_pids_by_tags($filter->{tag_ids});
        my $banners = Models::Banner::get_pure_creatives([map {{pid => $_}} @$pids], {}, {get_all_images => 1});
        $tag_BannerIDs = {
            map {$_ => 1} 
            (@$banners
            ? (map {
                    my @ids = ($_->{BannerID});
                    push @ids, split /,/, $_->{all_images_BannerID};
                    @ids} @$banners)
            : ())
        };
    }

    my @region_ids = values %{ _mass_get_region_id_by_oid(ref($oid) eq 'ARRAY' ? $oid : [$oid]) };
    my $region_id = $region_ids[0];
    unless (@region_ids == grep { $_ == $region_id } @region_ids) {
        die sprintf("Multiple order IDs (%s) with different regions were provided for get_stat_phrase_date", join(',', @$oid));
    }

    my $banners_info = {};
        $banners_info = Stat::Tools::mass_get_banner_info(BannerID => [uniq grep {$_} map { $_->{BannerID} } @$stat_stream],
                                                          translocal_params => $self->{translocal_params},
                                                          sharding_params => {OrderID => $oid});

    while( my $row = shift @$stat_stream ) {
        if (!is_valid_int($row->{BannerID}, 1)) {
            my $msg = "Skipping bad BannerID (".($row->{BannerID} // 'undef').") for OrderID $oid and dates between $start_date and $end_date: " . to_json($row);
            send_alert(Carp::longmess($msg), 'get_stat_phrase_date error');
            next;
        }

        next if $tag_BannerIDs && !$tag_BannerIDs->{$row->{BannerID}};
        
        # Работаем с баннером
        if ( !defined $data->{$row->{BannerID}} ) {
            # Получаем информацию о баннере
            
                $data->{$row->{BannerID}} = $banners_info->{$row->{BannerID}} //
                                            Stat::Tools::get_banner_info( BannerID => $row->{BannerID}, 
                                                                          translocal_params => $self->{translocal_params},
                                                                          sharding_params => {OrderID => $oid} );
        }
        my $banner = $data->{$row->{BannerID}};
        # Работа с датой
        ( $row->{sorting} = $row->{stat_date} ) =~ s/\D//g;
        $row->{Date} = Stat::Tools::format_date( $row->{stat_date}, $dateagg );
        if ( !$dateagg || $dateagg eq 'day' ) {
            $row->{holiday} = is_holiday($row->{stat_date}, $region_id);
        }

        if ($row->{ContextType} == $CONTEXT_TYPE_PHRASE) {
            $phrase_ids{$row->{PhraseID}} = undef;
        } elsif ($row->{ContextType} == $CONTEXT_TYPE_RET) {
            $ret_ids{$row->{PhraseID}} = undef;
        } elsif ($row->{ContextType} == $CONTEXT_TYPE_DYNAMIC) {
            $dyn_ids{$row->{PhraseID}} = undef;
        } elsif ($row->{ContextType} == $CONTEXT_TYPE_PERFORMANCE) {
            $perf_ids{$row->{PhraseID}} = undef;
        }

        $self->calc_sec_stat( $row );
        ## no critic (Freenode::DollarAB)
        for my $b ($banner) {
            next unless $b;
            # Считаем суммарную статистику по баннеру
            $self->aggregation_stat($b,$row, spec_phrases_separate => 1);
            push @{$b->{all_stat}}, hash_merge {}, $row;
        }
    }

    my $phrases_text = {};
    my $retargetings_text = {};
    my $performance_text = {};

    my $dynamic_data = Stat::Tools::get_dynamic_data([keys %dyn_ids],
                                                     OrderID => $oid,
                                                     with_condition => (defined $O{dynamic_data_ref} ? 1 : 0),
                                                     );
    if (defined $O{dynamic_data_ref}) {
        ${ $O{dynamic_data_ref} } = $dynamic_data;
    }

    # Теперь делаем из дат - массив
    my @banners;
    if ( any { defined $_->{group_name} } values %$data ) {
        @banners = xsort { $_->{group_name} // '', $_->{pid} // 0, $_->{bid} // 0 } values %$data;
    } else {
        @banners = xsort { $_->{bid} // 0 } values %$data;
    }

    my $res = {
        banners         =>  \@banners,
    };
    
    my $uniq_periods = {};
    for my $banner_list (map {$res->{$_} } qw/banners/) {
        for my $banner ( @$banner_list ) {
            # Теперь считаем ctr и ср. цену клика
            $self->calc_sec_stat( $banner , spec_phrases_separate=>1 );

            # Раскидываем фразы по массивам
            for my $phr (sort {$a->{ContextType} <=> $b->{ContextType} || $a->{PhraseID} <=> $b->{PhraseID}} @{$banner->{all_stat}}) {
                $uniq_periods->{$phr->{Date}} = 1;

                if ($phr->{ContextType} == $CONTEXT_TYPE_PHRASE) {
                    $phr->{phrase} ||= $phrases_text->{$phr->{PhraseID}} || '';
                } elsif ($phr->{ContextType} == $CONTEXT_TYPE_RET) {
                    $phr->{phrase} ||= $retargetings_text->{$phr->{PhraseID}} || '';
                } elsif ($phr->{ContextType} == $CONTEXT_TYPE_DYNAMIC) {
                    $phr->{phrase} ||= $dynamic_data->{$phr->{PhraseID}}->{name} || '';
                } elsif ($phr->{ContextType} == $CONTEXT_TYPE_PERFORMANCE) {
                    $phr->{phrase} ||= $performance_text->{$phr->{PhraseID}} || '';
                } elsif ($phr->{ContextType} == $CONTEXT_TYPE_BROADMATCH) {
                    $phr->{phrase} = Stat::Tools::special_phrase_title($BROADMATCH_PHRASE_ID);
                } elsif ($phr->{ContextType} == $CONTEXT_TYPE_RELEVANCE_MATCH) {
                    $phr->{phrase} = iget($RELEVANCE_MATCH_PHRASE_PLACEHOLDER);
                } else {
                    $phr->{phrase} = '';
                }

                if (exists $SPECIAL_PHRASES{$phr->{PhraseID}}
                    && none { $phr->{ContextType} == $_ } @NOT_TEXT_CONTEXT_TYPE
                ) {
                    $phr->{special_flag} = 1;
                    push @{$banner->{SpecialPhrases}}, $phr;
                } elsif ( $phr->{phrase} =~ /^\@(\d+)/ ) {
                    $phr->{phrase} = get_category_name( $1 );
                    push @{$banner->{CategoryDates}}, $phr;
                } else {
                    if ( @{$banner->{Dates}||=[]} && $banner->{Dates}[-1]{PhraseID} == $phr->{PhraseID} and  $banner->{Dates}[-1]{Date} eq $phr->{Date} ) {
                        # Агрегируем аддитивные величины и перевычисляем прочие
                        $self->aggregation_stat($banner->{Dates}[-1],$phr);
                        $self->calc_sec_stat($banner->{Dates}[-1]);
                    } else {
                        push @{$banner->{Dates}}, $phr;
                    }
                }
            }
        }
    }
    # Общая статистика по кампании
    my $days_num = Stat::OrderStatDay::get_order_days_num( $oid, $start_date, $end_date ) || 1;
    my $periods_num = scalar keys %$uniq_periods;
    $self->calc_order_stat($res, $days_num, $periods_num);

    # Если заданы параметры пейджинга по баннерам - оставляем только нужный чанк
    $res->{stat_banners_num} = scalar(@banners);
    if ($O{banners_on_page} && $O{page}) {
        @banners = splice @banners, $O{banners_on_page}*($O{page}-1), $O{banners_on_page};
    }

    hash_merge $res, {stat_stream_ts => $stat_ts};
    hash_merge $res, $reach_stat_totals if $stat_options->{with_reach_stat};

    return $res;
}

=head2 get_stat_customized

    Получить статистику с самыми обобщёнными группировками и фильтрами
    Вкладка "Статистика по дням", с опцией "Детальная статистика по объявлениям"

    Метод вызывать ТОЛЬКО на экземпляре Stat::CustomizedArray или его производных
    
    %options:
        ...
        no_analytic     => 0|1 # требуются ли данные метрики (влияет на то, можем ли мы применять стримовую статистику за давние периоды)
        onpage          => 10  # количество записей на странице (работает только в комплекте с page)
        page            => 1   # номер страницы (работает только в комплекте с onpage)
        with_nds        => 0|1 # включать ли в суммы НДС
        with_discount   => 0|1 # применять ли скидку
        single_currency => 0|1 # привести суммы к единоу валюте (currency // RUB)
        currency        => 'RUB' # в какую валюту сконвертровать суммы (имеет смысл вместе с single_currency)
        use_page_id     => 0|1 # срез по площадкам детализировать до PageID
        no_spec_and_all_prefix  => 0|1 # не формировать поля с префиксами all_ и spec_
        countable_fields_by_targettype => 0|1|[f1,f2] # по-умолчанию =1, означает необходимость детализации отдельных или всех полей по типам площадок, поиск/контекст
        secondary_group_by => [qw/date banner/] # возвращать дополнительно данные статистики, сгруппировнные по вторичному списку полей для группировки
                                                # !!! вторичная группировка должна быть подмножеством первичной
        four_digits_precision => 0|1 – округлять денежные поля до 4 знаков
        external_countable_fields_override => [qw/shows clicks sum aseslen asesnum asesnumlim agoalnum/] — запрашивать только эти вычислимые поля из БК

=cut

sub get_stat_customized {
    my ( $self, $oids, $start_date, $end_date, $group_by, $dateagg, $filter, %options) = @_;

    _debug("get_stat_customized(".Dumper([$oids, $start_date, $end_date, $group_by, $dateagg, $filter, \%options]).")");
    my $profile = Yandex::Trace::new_profile('dbstat:get_stat_customized');
    $filter ||= {};

    my $group_hash = Stat::Tools::get_group_by_as_hash($group_by, $dateagg);

    my $limits = hash_cut \%options, qw/order_by/;
    if ($options{onpage} && $options{page}) {
        $limits->{limit} = $options{onpage};
        $limits->{offset} = ($options{page}-1)*$options{onpage}; 
    }

    my $stat_options = hash_cut \%options, qw/
        with_nds with_discount single_currency currency use_page_id no_spec_and_all_prefix external_countable_fields_override
        countable_fields_by_targettype with_reach_stat four_digits_precision operator_ClientID
    /;
    $stat_options->{countable_fields_by_targettype} //= 1;
    
    die "Method get_stat_customized is supported only in Stat::CustomizedArray* classes" unless (ref $self) =~ /CustomizedArray/;
    $self->set_report_parameters(
        oid => $oids
        , start_date => $start_date
        , end_date => $end_date
        , group_by => $group_by
        , date_aggregation_by => $group_hash->{date}
        , filter => $filter
        , limits => $limits
        , options => $stat_options
        , translocal_params => $self->{translocal_params}
        , ClientID_for_stat_experiments => $self->{translocal_params}{ClientID}
        , stat_type => delete $options{stat_type}
    );
    my $fetched_stat = $self->generate_with_metrika;

    my %data_versions = (main => {group_by => $group_by, data => {}});
    if ($options{secondary_group_by}) {
        if (@{xminus $options{secondary_group_by}, $group_by}) {
            die sprintf("Secondary group_by is not a subset of main group_by: secondary(%s), main(%s)", 
                            join(',', @{ $options{secondary_group_by} }), join(',', @$group_by));
        }
        $data_versions{secondary} = {group_by => $options{secondary_group_by}, data => {}}
    }

    # поддержка нескольких OrderID добавлена для подлежащих кампаний
    # они, как и мастер-кампания, не должны быть медийными, а их регионы должны совпадать с регионом мастер-кампании
    # поддержка более общего случая потребовала бы слишком сильных изменений в коде
    my $is_media;
    my $region_id;
    if (ref($oids) eq 'ARRAY') {
        my @is_media_camps = map { is_media_camp(OrderID => $_) } @$oids;
        $is_media = $is_media_camps[0];
        unless (@is_media_camps == grep { $_ == $is_media } @is_media_camps) { # unless all equal
            die sprintf("Multiple order IDs (%s) with different is_media_camp state were provided for get_stat_customized",
                join(',', @$oids));
        }
        my @region_ids = values %{ _mass_get_region_id_by_oid($oids) };
        $region_id = $region_ids[0];
        unless (@region_ids == grep { $_ == $region_id } @region_ids) {
            die sprintf("Multiple order IDs (%s) with different regions were provided for get_stat_customized",
                join(',', @$oids));
        }
    } else {
        $is_media = is_media_camp(OrderID => $oids);
        $region_id = _get_region_id_by_oid($oids);
    }

    for my $data_version_opts (values %data_versions) {
        #############################################################
        # фетчим данные
        # фильтруем то, что можем,
        # формируем хэш
        my $data = $data_version_opts->{data};

        $data->{period} = [ $start_date||'1970-01-01', $end_date||sprintf("%04d-%02d-%02d", Date::Calc::Today) ];

        my $geo_name_field = get_geo_name_field();
        my $translocal_georeg = GeoTools::get_translocal_georeg($self->{translocal_params});

        my (%media_groups);
        my $media_group;

        my $banners_info = {};
        unless ($is_media) {
            $banners_info = Stat::Tools::mass_get_banner_info(BannerID => [uniq grep {$_} map { $_->{BannerID} } @{$fetched_stat->{data_array}}],
                                                              translocal_params => $self->{translocal_params},
                                                              sharding_params => {OrderID => $oids});
        }

        FETCH_LOOP:
        for my $row (@{$fetched_stat->{data_array}}) {
            # По группировкам доходим до нужного уровня
            my $cur_data = $data;
            for my $group ( @{$data_version_opts->{group_by}} ) {
                $cur_data->{next_data} = $group;
                if ( $group eq 'banner' ) {
                    if (!is_valid_int($row->{BannerID}, 1)) {
                        my $msg = "Skipping bad BannerID (".($row->{BannerID} // 'undef').") for OrderID $oids and dates between $start_date and $end_date: " . to_json($row);
                        send_alert(Carp::longmess($msg), 'get_stat_customized error');
                        next;
                    }
                    # Работаем с баннером
                    my $BannerID = $row->{BannerID};
                    if ( !defined $cur_data->{data}->{$BannerID} ) {
                        # Получаем информацию о баннере
                        if ( $is_media ) {   # Работаем с медийными объявлениями
                            ## no critic (Freenode::DollarAB)
                            my $b = $cur_data->{data}{$BannerID} = hash_merge 
                                get_media_banner({
                                    BannerID    =>  $BannerID,
                                    OrderID     => $oids,
                                    }),
                                { map { $_ => undef } qw/shows clicks ctr/ };
                            unless ($b->{mgid}) {
                                warn "no banner with BannerID $BannerID\n"
                            }
                            $b->{bid} = $b->{mbid};
                            $media_group = $media_groups{$b->{mgid}} ||= hash_merge
                                get_media_group({
                                    mgid        =>  $b->{mgid},
                                    no_banners  =>  1,
                                    translocal_params => $self->{translocal_params},
                                    }),
                                { media_banners => [] },
    			                { map { $_ => undef } qw/shows clicks ctr/ };
                            push @{$media_group->{media_banners}}, $b;
                        } else {    # Текстовая реклама
                            $cur_data->{data}{$BannerID} = $banners_info->{$BannerID};
                        }
                    }
                    $cur_data = $cur_data->{data}->{$BannerID};
                    $media_group = $media_groups{$cur_data->{mgid}} if $cur_data->{mgid};
                } elsif ( $group eq 'page' ) {
                    # Работаем с сайтом

                    my $data_group = $options{use_page_id} ? 'page_id' : 'page_group';
                    $row->{ $data_group } = '' unless defined $row->{ $data_group };   
                    my $row_group_value = $row->{ $data_group };
                    # при получении названия площадки из МОЛ, `page_id` больше не является уникальным ключом,
                    # т.к. название площадки может вычисляться по странице, в которую встроен рекламный блок (TopAncestor)
                    if ($options{group_by_page_name_detailed}) {
                        $row_group_value .= '.'.$row->{page_name};
                    }
                    unless (defined $cur_data->{data} && defined $cur_data->{data}->{ $row_group_value }) {
                        # Фильтруем
                        if ( $filter->{page} && index(lc($row->{page_name}), $filter->{page}) == -1 ) {
                            next FETCH_LOOP;
                        }
                        $cur_data->{data}->{$row_group_value} = {
                                                    page_group   => $row->{page_group},
                                                    name         => $row->{page_name},
                                                    sorting      => str($row->{page_sorting}).str($row->{page_name}),
                                                    page_sorting => $row->{page_sorting},
                                                    TargetType   => $row->{TargetType},
                                                    page_id      => $row->{page_id},
                                                    page_domain  => $row->{page_domain},
                                                    };
                    }
                    $cur_data = $cur_data->{data}->{ $row_group_value };

                } elsif ( $group eq 'phrase' ) {
                    # группировка по фразе
                    my $PhraseID = $row->{PhraseID};
                    if ( !defined $cur_data->{data}->{$PhraseID} ) {
                        $cur_data->{data}->{$PhraseID} = {
                            PhraseID => $PhraseID,
                        };
                    }
                    $cur_data = $cur_data->{data}->{$PhraseID};
                } elsif ( $group eq 'geo' ) {
                    # группировка по региону
                    my $region = $row->{region}||0;
                    if ( !defined $cur_data || !defined $cur_data->{data} || !defined $cur_data->{data}->{$region} ) {
                        my $reg_defined = $region && exists $translocal_georeg->{$region};
                        my $name = $reg_defined ? $translocal_georeg->{$region}->{$geo_name_field} : iget('не определен');

                        # Фильтруем
                        if ( $filter->{geo} && index(lc($name), $filter->{geo}) == -1 ) {
                            next FETCH_LOOP;
                        }
                        $cur_data->{data}->{$region} = { region => $region, name => $name, sorting => ($reg_defined ? '0' : '1').$name};
                    }
                    # подготовка к вычитаниям
                    if ( exists $translocal_georeg->{$region} ) {
                        for my $parent ( grep {$_} @{ $translocal_georeg->{$region}->{parents} } ) {
                            $cur_data->{regions_deduct}->{$parent}->{$region} = 1;
                        }
                    }
                    $cur_data = $cur_data->{data}->{$region};
                } elsif ( $group eq 'date' ) {
                    # группировка по date
                    my $date = $row->{stat_date};
                    if ( !defined $cur_data->{data}->{$date} ) {
                        (my $sorting = $date) =~ s/\D+//g;
                        $cur_data->{data}->{$date} = {
                                        stat_date => $row->{stat_date},
                                        period => [$self->period_by_date( $date, $dateagg, @{$data->{period}})],
                                        date => Stat::Tools::format_date( $date, $dateagg ),
                                        sorting => $sorting,
                                        holiday => ( !$dateagg || $dateagg eq 'day' ? is_holiday($row->{stat_date}, $region_id) : 0 ),
                                    };
                    }
                    $cur_data = $cur_data->{data}->{$date};
                } else {
                    die "Unknown grouping: '$group'";
                }
            }
            $self->aggregation_stat($cur_data, $row);
        }

        $self->{media_groups} = \%media_groups;
        my $dumn_copy;
        $dumn_copy = sub {
            my ($self, $d1, $d2) = @_;
            unless ( $d1->{next_data} ) {
                $self->aggregation_stat($d2, $d1);
            } else {
                $d2->{next_data} = $d1->{next_data};
                $d2->{data} ||= {};
                if ( $d1->{next_data} ne 'banner') {
                    while ( my ($key, $val) = each %{$d1->{data}} ) {
                        if (ref $val eq 'HASH') {
                            $d2->{data}->{$key} ||= $self->setnulval_field( hash_merge 
                                (hash_grep {!ref $_ or ref $_ eq 'ARRAY'} $val),
                                {data=>{}});
                            $dumn_copy->($self, $val, $d2->{data}{$key}||={});
                        } elsif (ref $val eq 'ARRAY') {
                            $d2->{data}->{$key} = [map {$_} @$val];
                        } else {
                            $d2->{data}->{$key} = $val;
                        }
                    }
                } else {
                    while ( my ($key, $val) = each %{$d1->{data}} ) {
                        $d2->{data}{$val->{mgid}} ||= $self->setnulval_field( hash_merge 
                            $self->{media_groups}{$val->{mgid}},
                            hash_cut( $d1, qw/next_data/),
                            {data=>{}}); 
                        $dumn_copy->($self, $val, $d2->{data}{$val->{mgid}}||={});
                    }
                }
            }
        };

        my $groupped_data = {};
        $dumn_copy->($self, $data, $groupped_data) if $is_media;

        # Получаем текст фраз
        my $phrases_text =  {};
        my $retargetings_text = {};

        #############################################################
        for my $the_data ($data, $groupped_data) {
            next unless $the_data;
            # дофильтровываем данные
            # превращаем хеши в массивы
            # аггрегируем
            my $flat = exists $options{flat} && $options{flat};
            $self->data_f_c_a($the_data, $filter, $phrases_text, $retargetings_text, undef, $flat, %{hash_cut \%options, qw/no_spec_and_all_prefix/});

            #переименуем поля
            Stat::Tools::rename_field($the_data);

            # перетрем местные totals пришедшими от Stat::CusomizedArray
            hash_merge $the_data, $fetched_stat;
            delete $the_data->{data_array};
        }

        $data->{media_groups} = $groupped_data->{data} if $is_media;

        $data->{period} = [ map { sprintf "%04d%02d%02d", split /\D+/, $_} @{$data->{period}} ];

        for my $g (@{$data->{media_groups}}) {
            $g->{banners} = $g->{media_banners};
            $_->{statusBanner} = calc_media_banner_status($_, $g) for @{$g->{banners}};
        }
    
    }

    my @result = ($data_versions{main}->{data}, 
                  $data_versions{secondary} ? $data_versions{secondary}->{data} : ());
    return wantarray ? @result : $result[0];
}

###############################################
# Вспомогательные ф-ции по подсчёту статистики
###############################################
#shows clicks sum ctr av_sum adepth aconv agoalcost
#

sub setnulval_field {
    my $self = shift;
    my $data = shift;
    my %sum_field_if_exists = map { $_ => 1 } field_list_sum_if_exists();
    for my $suf ( suffix_list() ) {
        for my $f ( field_list_sum(), field_list_sum_if_exists() ) {        
            my $key = "$f$suf";    
            next if $sum_field_if_exists{$f} && !exists $data->{$key};
            $data->{$key} = 0;
        }
    }   
    return $data;
}

sub _agr_field {
    my ($self, $data, $row, %options) = @_;
    
    my $prefix = $options{prefix};
    $prefix = $prefix ? 't' : '';
    if ($options{spec_phrases_separate} && exists $SPECIAL_PHRASES{$row->{PhraseID}}) {
        $prefix = $SPEC_PREFIX.$prefix;
    }
    my %sum_field_if_exists = map { $_ => 1 } field_list_sum_if_exists();
    for my $suf ( suffix_list() ) {
        for my $f ( field_list_sum(), field_list_sum_if_exists() ) {
            my $key = "$f$suf";
            unless ($sum_field_if_exists{$f} && !exists $row->{$key}) {
                $data->{"$prefix$key"} += $row->{$key} || 0;
            }
            unless ($sum_field_if_exists{$f} && !exists $row->{$SPEC_PREFIX.$key}) {
                $data->{"$prefix$key"} += ($row->{$SPEC_PREFIX.$key} || 0) if $options{prefix};
            }
        }
    }
}

sub aggregation_stat {
    my $self   = shift;
    my $data   = shift;
    my $row    = shift;
    my %options = @_;
    $self->_agr_field($data,$row, %options);
}

sub aggregation_campaign_stat {
    my $self   = shift;
    my $data   = shift;
    my $row    = shift;
    $self->_agr_field($data,$row, prefix => 1);
}

# Посчитать общую статистику заказа по баннерам
# $days_num -- кол-во дней статистики
# $periods_num -- кол-во периодов группировки по которым есть статистика (для вычисления среднего значения)
sub calc_order_stat {
    my ($self, $order, $days_num, $periods_num) = @_;
    for my $banner ( @{ $order->{banners} } ) {
        # Общая статистика по кампании
        $self->aggregation_campaign_stat($order,$banner);
    }
    $days_num ||= 1;
    # Общая статистика по кампании
    for my $suf ( suffix_list() ) {
        Stat::Tools::calc_average_campaign_stat($order, $suf, $days_num, $periods_num, undef, four_digits_precision => $self->options->{four_digits_precision});
    }
}

# Подсчёт вторичных параметров (типа ctr)
sub calc_sec_stat {
    my ( $self, $h, %options) = @_;
    for my $suf ( suffix_list() ) {
        Stat::Tools::calc_average_stat($h, $suf, %options, four_digits_precision => $self->options->{four_digits_precision});
    }
}

# дофильтровываем данные
# превращаем хеши в массивы
# аггрегируем
sub data_f_c_a {
    my ($self, $data, $filter, $phrases, $retargetings_text, $banner, $flat, %options) = @_;

    if ( defined $data->{data} ) {
        #проставляем полям нулевое значение
        $self->setnulval_field($data);

        my $geo_name_field = get_geo_name_field();
        my $translocal_georeg = GeoTools::get_translocal_georeg($self->{translocal_params});

        # Фильтруем данные
        while( my ( $key, $hash ) = each %{$data->{data}} ) {
            if ( $data->{next_data} eq 'banner' ) {
                $banner = $hash;
            }
            # Рекурсивно спускаемся вниз
            $self->data_f_c_a($hash, $filter, $phrases, $retargetings_text, $banner, $flat);
            if ( !$hash->{shows} && !$hash->{clicks} ) {
                delete $data->{data}{$key};
                next;
            }
            # делаем специфические, зависящие от уровня операции
            if ( $data->{next_data} eq 'phrase' ) {
                # для фраз определяем текст и статус
                my $PhraseID = $hash->{PhraseID};
                $hash->{text} = $hash->{phrase} || $phrases->{$PhraseID} || '';
                if ( $filter->{phrase} && index(lc($hash->{text}), $filter->{phrase}) == -1 ) {
                    delete $data->{data}->{$key};
                    next;
                }
                my ( $postfix, $sorting_postfix ) = ( '', '0' );
                if ( $hash->{text} =~ /^\@(\d+)/ ) {
                    $hash->{text} = get_category_name( $1 );
                    $postfix = '_categories';
                    $sorting_postfix = '1';
                }
                $hash->{sorting} = "$sorting_postfix$hash->{text}";
                if ( $banner ) {
                    if (exists $SPECIAL_PHRASES{$PhraseID}
                        && none { $hash->{ContextType} == $_ } @NOT_TEXT_CONTEXT_TYPE
                    ) {
                        $hash->{special_flag} = 1;
                        $hash->{phrase_status} = "active$postfix";
                    } elsif (any { $hash->{ContextType} == $_ } @NOT_TEXT_CONTEXT_TYPE) {
                        $hash->{phrase_status} = "active$postfix";
                    } elsif ( defined $banner->{phrases_hash}->{$PhraseID} && $banner->{phrases_hash}->{$PhraseID}->{rank} ) {
                        delete $banner->{phrases_hash}->{$PhraseID};
                        $hash->{phrase_status} = "active$postfix";
                    } elsif ( defined $banner->{phrases_hash}->{$PhraseID} ) {
                        delete $banner->{phrases_hash}->{$PhraseID};
                        $hash->{phrase_status} = "bad_ctr$postfix";
                    } else {
                        $hash->{phrase_status} = "past$postfix";
                    }
                }
            } elsif ( $data->{next_data} eq 'geo' ) {
                # Вычитаем регионы
                $hash->{region_name} = $hash->{full_name} = $hash->{name};
                if ( defined $data->{regions_deduct}->{$hash->{region} // ''} ) {
                    my @childs =
                        sort {
                            $translocal_georeg->{$a}->{level} <=> $translocal_georeg->{$b}->{level}
                            or $translocal_georeg->{$a}->{$geo_name_field} cmp $translocal_georeg->{$b}->{$geo_name_field}
                            }
                        keys %{$data->{regions_deduct}->{$hash->{region}}};
                    if ( @childs ) {
                        $hash->{full_name} = $hash->{name} = iget('%s (более точно регион не определён)', $hash->{name});
                    }
                }
            } elsif ( $data->{next_data} eq 'banner' ) {
                delete $hash->{phrases_hash};
            }

            # аггрегируем
            $self->aggregation_stat($data,$hash);
        }
        if ( $data->{next_data} eq 'geo' ) {
            delete $data->{regions_deduct};
        } elsif ( $data->{next_data} eq 'banner' ) {
            delete $data->{phrases_hash};
        }
        # Превращаем хэш в сортированный массив
        if ( $data->{next_data} eq 'banner' ) {
            $data->{data} = [ sort {($a->{bid}||0) <=> ($b->{bid}||0)} values %{$data->{data}} ];
            $data->{banners} = $data->{data};
        } else {
            $data->{data} = [ sort {($a->{sorting}||'') cmp ($b->{sorting}||'')} values %{$data->{data}} ];
            if ( $data->{next_data} eq 'geo' ) {
                if ($flat) {
                    $data->{regions_flat} = $data->{regions} = $data->{data};
                } else {
                    my $tree = $self->create_regions_tree( $data->{data} );
                    # DIRECT-60662: не отображаем родителя, если он - вся планета
                    if ($tree->{region} == 0) {
                        $data->{regions} = $tree->{regions};
                    } else {
                        $data->{regions} = [$tree];
                    }
                    $data->{regions_flat} = $data->{data};
                }
            } else {
                $data->{$data->{next_data}.'s'} = $data->{data};
            }
        }
    }
    # Считаем общие показатели
    $self->calc_sec_stat($data, ($options{no_spec_and_all_prefix} ? (spec_and_all_prefix => 0) : ()) );
}

#########################################
# Мелкие вспомогательные ф-ции
#########################################

# Округлить до недели, месяца, квартала, года
sub round_date {
    my ( $self, $date, $dateagg ) = @_;

    if ( $dateagg && $dateagg eq 'week' ) {
        return str_round_week($date);
    } elsif ( $dateagg && $dateagg eq 'quarter' ) {
        return str_round_quarter($date);
    } elsif ( $dateagg && $dateagg eq 'month' ) {
        return str_round_month($date);
    } elsif ( $dateagg && $dateagg eq 'year' ) {
        return str_round_year($date);
    }
    else {
        return str_round_day($date);
    }
}

sub period_by_date {
    my ($self, $date, $dateagg, $min_date, $max_date) = @_;

    ($min_date, $max_date, $date) = map {defined $_ ? [split(/\D+/, $_)] : undef} ($min_date, $max_date, $date);

    my $start_date = my $end_date = $date;
    if ( !$dateagg ) {
    } elsif ( $dateagg eq 'week' ) {
        $start_date = [Monday_of_Week(Week_of_Year(@$date))];
        $end_date   = [Add_Delta_YMD(@$start_date, 0, 0, 6)];
    } elsif ( $dateagg eq 'month' ) {
        $start_date = [$date->[0], $date->[1], 1];
        $end_date   = [Add_Delta_YMD(@$start_date, 0, 1,-1)];
    } elsif ( $dateagg eq 'quarter') {
        $start_date = [$date->[0], (int(($date->[1] - 1)/3)*3 + 1), 01];
        $end_date   = [Add_Delta_YMD(@$start_date, 0, 3, -1)];
    } elsif ( $dateagg eq 'year' ) {
        $start_date = [$date->[0], 01, 01];
        $end_date   = [Add_Delta_YMD(@$start_date, 1, 0,-1)];
    }

    ($min_date, $max_date, $start_date, $end_date) = map {defined $_ ? sprintf("%04d%02d%02d", @$_) : undef } $min_date, $max_date, $start_date, $end_date;

    $start_date = maxstr grep {$_} $start_date, $min_date;
    $end_date   = minstr grep {$_} $end_date,   $max_date;

    return ($start_date, $end_date);
}

sub hash_clone {
    my $hash = shift;
    return { map { $_ => $hash->{$_} } grep { !ref $hash->{$_} } keys %$hash };
}

# По массиву создать дерево регионов
# вернуть массив из верхних потомков
sub create_regions_tree {
    my ( $self, $arr, $cur_reg ) = @_;
    $cur_reg ||= 0;
    my $translocal_region = GeoTools::get_translocal_region($cur_reg, $self->{translocal_params});
    my @result;
    my %childs = map {$_ => 1} @{ $translocal_region->{childs} };
    # Рекурсивно добавляем детей
    for my $child ( keys %childs ) {
        my $res = $self->create_regions_tree( $arr, $child );
        if ( $res ) {
            push @result, $res;
            delete $childs{$child};
        }
    }
    # Находим детей и себя в массиве
    my $tree;
    for my $child ( @$arr ) {
        if ( defined $childs{ $child->{region} // '' } ) {
            push @result, hash_clone($child);
        }
        $tree = hash_clone($child) if defined $child->{region} && $child->{region} == $cur_reg;
    }

    my $geo_name_field = get_geo_name_field();

    # Проверяем, сколько детей есть в массиве
    if ( defined $tree ) {
        if ( @result ) {
            my $oth = hash_clone($tree);
            $oth->{region_name} = iget('прочие');
            $oth->{sorting} = '9' . iget('прочие');
            push @result, $oth;
            $tree->{regions} = \@result;
        }
    } else {
        if ( @result == 0 ) {
            return undef;
        } elsif ( @result == 1 ) {
            $tree = shift @result;
        } else {
            $tree = {
                region => $cur_reg,
                region_name => $translocal_region->{$geo_name_field},
                sorting => '0' . $translocal_region->{$geo_name_field},
                regions => \@result,
                };
        }
    }
    # суммируем
    if ( @result ) {
        # обнуляем собственные цифры
        my %sum_field_if_exists = map { $_ => 1 } field_list_sum_if_exists();
        for my $suf ( suffix_list() ) {
            for my $f ( field_list_sum(), field_list_sum_if_exists() ) {
                next if $sum_field_if_exists{$f} && !exists $tree->{"$f$suf"};
                $tree->{"$f$suf"} = 0;
            }
        }
        # суммируем детей
        foreach my $row (@result) {
            $self->aggregation_stat($tree,$row);
        }
        $self->calc_sec_stat($tree);
    }
    return $tree;
}

###############################
# Работа с PPC
###############################


########################################

sub get_orders_stat_date {
    my ( $self, $orders, $start_date, $end_date, $dateagg, %option ) = @_;
    my $data;

    my $stat_options = hash_cut \%option, qw/
        with_nds with_discount single_currency currency with_avg_position four_digits_precision operator_ClientID external_countable_fields_override
    /;
    my $stat_stream_ts;

        # использовать логику выбора источника статистики заложенную в Stat::CustomizedArray
        die "Method get_orders_stat_date with (use_customized_array => 1) option is supported only in Stat::CustomizedArray* classes" unless (ref $self) =~ /CustomizedArray/;
        my $client_id = $option{translocal_params} ? $option{translocal_params}{ClientID} : $self->{translocal_params}{ClientID};
        $self->set_report_parameters(oid        => [grep { /^\d+$/ } map { split /\s*,\s*/, ($_->{OrderID} || '') } @$orders],
                                     start_date => $start_date,
                                     end_date   => $end_date,
                                     filter    => {attribution_model => $option{attribution_model} || get_common_stat_default_attribution()},
                                     group_by   => ['campaign','date'],
                                     date_aggregation_by => 'day', # заменить на $dateagg когда сможем корректно для каждой строки (в том числе тоталы по всем кампаниям) 
                                                                   # вычислять кол-во дней для среднего расхода за день
                                     ClientID_for_stat_experiments => $client_id,
                                     options    => {%$stat_options,
                                                    countable_fields_by_targettype => $option{by_target} ? 1 : 0,
                                                    no_spec_and_all_prefix => $option{no_spec_and_all_prefix},
                                                    without_round => 1,
                                                   },
                                     translocal_params => $option{translocal_params} // $self->{translocal_params},
                                     stat_type => 'common',
                                    );
        my $stat_customized_array = $self->generate_with_metrika;
        $stat_stream_ts = $stat_customized_array->{stat_stream_ts};
        $data = convert_stat_stream_to_orders_stat_date($stat_customized_array->{data_array}, 'is_customized_array_names');

    my $stat = $self->get_orders_stat_date_format( $orders, $start_date, $end_date, $dateagg, $data, %option );

    # считаем количество показов для каждого OrderID
    # $ordinfo = {
    #   $OrderID => количество показов за выбранный интервал дат,
    #   ...
    # }
    my $ordinfo = { 
        map { my $i = $_; 
            $i => sum(map { $_->{Shows} || 0 } values %{$data->{$i}}) || 0 } 
        keys %$data };
    $stat->{ordinfo} = $ordinfo;
    
    hash_merge $stat, {stat_stream_ts => $stat_stream_ts} if defined $stat_stream_ts;

    return $stat;
}

sub convert_stat_stream_to_orders_stat_date {
    my ($stat_stream, $is_customized_array_names) = @_;

    my %fields_map = (shows => 'Shows',
                      clicks => 'Clicks',
                      sum => 'Cost',
                      bonus => 'Bonus',
                      goals_num => 'agoalnum',
                      goals_income => 'agoalincome',
                      sessions_len => 'aseslen',
                      sessions_num => 'asesnum',
                      sessions_num_limited => 'asesnumlim',
                      OrderID => 'order_id' );

    my $stat_data = {analytic => {}};
    foreach my $row (@$stat_stream) {
        $stat_data->{$row->{OrderID}} = {} unless $stat_data->{$row->{OrderID}};
        my $date_stat = {};
        foreach my $fld (qw/shows clicks sum bonus/) {
            foreach my $suf ( suffix_list() ) {
                next unless exists $row->{$fld.$suf};
                $date_stat->{($fields_map{$fld} || $fld).$suf} = $row->{$fld.$suf};
            }
        }
        $stat_data->{$row->{OrderID}}->{join '', split /-/, $row->{stat_date}} = $date_stat;

        # метрика
        my $date_analytics = {};
        my $have_analytics = 0;
        foreach my $fld (map { $is_customized_array_names ? $fields_map{$_} || $_ : $_ } qw/goals_num goals_income sessions_len sessions_num sessions_num_limited/) {
            foreach my $suf ( suffix_list() ) {
                next unless exists $row->{$fld.$suf};
                $date_analytics->{($fields_map{$fld} || $fld).$suf} = $row->{$fld.$suf};
                $have_analytics = 1 if $row->{$fld.$suf} && $row->{$fld.$suf} > 0;
            }
        }
        foreach my $fld (qw/OrderID stat_date/) {
            next unless exists $row->{$fld};
            $date_analytics->{($fields_map{$fld} || $fld)} = $row->{$fld};
        }
        $stat_data->{analytic}->{$row->{OrderID}.':'.$row->{stat_date}} = $date_analytics;
    }
    $stat_data->{analytic} = undef unless keys %{$stat_data->{analytic}};
    return $stat_data;

}

sub get_orders_stat_date_format {
    my ( $self, $orders, $start_date, $end_date, $dateagg, $data, %option ) = @_;

    my $analytic = $data->{analytic};
    my @suffix = suffix_list();
    @suffix = ($suffix[0]) unless $option{by_target};

    my $sprintf_fmt = $option{four_digits_precision} ? "%.4f" : "%.2f";

    my (%round_date_cache, %format_date_cache);
    my @order_ids;
    for my $order (@$orders) {
        push @order_ids, grep {$_} split ',', $order->{OrderID};
    }
    my $oid2region_id = _mass_get_region_id_by_oid(\@order_ids);
    # Переформатируем их
    my $res = { orders => $orders };
    for my $order (@$orders) {

        #   очищаем аддитивные статистические данные
        map { delete $order->{$_} } map {
            ## no critic (Freenode::DollarAB)
            my $a = $_;
            map { $a . $_ } @suffix
        } qw/shows clicks sum bonus asesnum asesnumlim aseslen agoalnum gsum agoalincome/;

        # Чуть дальше хэш заменяется на список значений, инициализируем чтобы сдалать код реентрабельным
        $order->{dates} = {};

        for my $OrderID ( split ',', $order->{OrderID} ) {
            my $region_id = $oid2region_id->{$OrderID};
            while ( my ( $exact_date, $vals ) = each %{ $data->{$OrderID} } ) {
                $exact_date =~ s/^(\d{4})(\d{2})(\d{2}).*/$1-$2-$3/;
                my $date = $round_date_cache{$dateagg // ''}{$exact_date}
                            ||= $self->round_date($exact_date, $dateagg);
                my $row     = $order->{dates}->{$date} ||= {};
                my $res_row = $res->{dates}->{$date}   ||= {};

                # Считаем уникальные дни/периоды на всех уровнях
                for my $level_item ($row, $order, $res, $res_row) {
                    $level_item->{uniq_dates}->{$exact_date} = 1;
                    $level_item->{uniq_periods}->{$date} = 1;
                }

                # Добавляем доп. поля
                ( $row->{sorting} = $date ) =~ s/\D//g;
                if ( !$dateagg || $dateagg eq 'day' ) {
                    $row->{holiday} = is_holiday($date, $region_id);
                }

                for my $f (qw/sorting holiday/) {
                    $res_row->{$f} = $row->{$f};
                }

                # Добавляем цифры
                my $metrica = "$OrderID:$exact_date";
                for my $r ( $row, $order, $res_row, $res ) {
                    $r->{ stat_date } = $date;
                    foreach my $suf (@suffix) {                        
                        no warnings 'uninitialized';
                        $r->{"shows$suf"}  += $vals->{"Shows$suf"}  if exists $vals->{"Shows$suf"};
                        $r->{"clicks$suf"} += $vals->{"Clicks$suf"} if exists $vals->{"Clicks$suf"};
                        $r->{"sum$suf"}    += $vals->{"Cost$suf"}   if exists $vals->{"Cost$suf"};
                        $r->{"bonus$suf"}  += $vals->{"Bonus$suf"}  if exists $vals->{"Bonus$suf"};
                        next unless $analytic->{$metrica};
                        
                        $r->{"$_$suf"} += ($analytic->{$metrica}->{"$_$suf"} || 0) foreach qw/asesnum aseslen agoalnum agoalincome/;
                        for (qw/asesnumlim/) {
                            next unless exists $analytic->{$metrica}->{"$_$suf"};
                            $r->{"$_$suf"} += ($analytic->{$metrica}->{"$_$suf"} || 0) ;
                        }
                    }
                }
            }    # / while
        }

        $order->{ dates } = [map { ## no critic (ProhibitComplexMappings)
            $self->calc_sec_stat($_, spec_and_all_prefix => $option{no_spec_and_all_prefix} ? 0 : undef);
            my $days_num = ( scalar keys %{$_->{uniq_dates}} )||1;
            $_->{min_date} = minstr keys %{$_->{uniq_dates}};
            $_->{max_date} = maxstr keys %{$_->{uniq_dates}};
            delete $_->{uniq_dates};
            my $periods_num = ( scalar keys %{$_->{uniq_periods}} )||1;
            delete $_->{uniq_periods};
            foreach my $suf (@suffix) {
                $_->{"av_day$suf"} = sprintf $sprintf_fmt, ($_->{"sum$suf"} || 0) / $days_num;
                $_->{"av_day_shows$suf"} = sprintf '%.02f', ($_->{"shows$suf"} || 0) / $days_num;
                $_->{"av_grouping$suf"} = sprintf $sprintf_fmt, ($_->{"sum$suf"} || 0) / $periods_num;
            }
            $_
        } values %{$order->{ dates }}];

        # Общая статистика по кампании
        my $days_num = ( scalar keys %{ $order->{uniq_dates} } ) || 1;
        $order->{min_date} = minstr keys %{ $order->{uniq_dates} };
        $order->{max_date} = maxstr keys %{ $order->{uniq_dates} };
        delete $order->{uniq_dates};
        my $periods_num = ( scalar keys %{$order->{uniq_periods}} )||1;
        delete $order->{uniq_periods};
        $self->calc_sec_stat($order, spec_and_all_prefix => $option{no_spec_and_all_prefix} ? 0 : undef);
        foreach my $suf (@suffix) {
            $order->{"av_day$suf"} = sprintf($sprintf_fmt, ($order->{"sum$suf"} || 0)/$days_num);
            $order->{"av_day_shows$suf"} = sprintf("%.02f", ($order->{"shows$suf"} || 0)/$days_num);
            $order->{"av_grouping$suf"} = sprintf($sprintf_fmt, ($order->{"sum$suf"} || 0)/$periods_num);
        }
    }


    my $days_num = ( scalar keys %{$res->{uniq_dates}} ) || 1;
    $res->{min_date} = minstr keys %{$res->{uniq_dates}};
    $res->{max_date} = maxstr keys %{$res->{uniq_dates}};
    delete $res->{uniq_dates};
    my $periods_num = ( scalar keys %{$res->{uniq_periods}} )||1;
    delete $res->{uniq_periods};
    $self->calc_sec_stat( $res, spec_and_all_prefix => $option{no_spec_and_all_prefix} ? 0 : undef );
    foreach my $suf (@suffix) {
        $res->{"av_day$suf"} = sprintf $sprintf_fmt, ($res->{"sum$suf"} || 0) / $days_num;
        $res->{"av_day_shows$suf"} = sprintf '%.02f', ($res->{"shows$suf"} || 0) / $days_num;
        $res->{"av_grouping$suf"} = sprintf $sprintf_fmt, ($res->{"sum$suf"} || 0) / $periods_num;
    }

    my $dates = [];
    foreach my $client ( values %{$res->{dates}} ) {

        $client->{date} = $format_date_cache{$client->{stat_date}} 
            ||= Stat::Tools::format_date($client->{stat_date}, $dateagg)
        ;

        $self->calc_sec_stat($client, spec_and_all_prefix => $option{no_spec_and_all_prefix} ? 0 : undef);
        my $days_num = ( scalar keys %{$client->{uniq_dates}} ) || 1;
        $client->{min_date} = minstr keys %{$client->{uniq_dates}};
        $client->{max_date} = maxstr keys %{$client->{uniq_dates}};
        delete $client->{uniq_dates};
        my $periods_num = ( scalar keys %{$client->{uniq_periods}} )||1;
        delete $client->{uniq_periods};
        foreach my $suf (@suffix) {
            $client->{"av_day$suf"} = sprintf $sprintf_fmt, ($client->{"sum$suf"} || 0) / $days_num;
            $client->{"av_day_shows$suf"} = sprintf '%.02f', ($client->{"shows$suf"} || 0) / $days_num;
            $client->{"av_grouping$suf"} = sprintf $sprintf_fmt, ($client->{"sum$suf"} || 0) / $periods_num;
        }

        push @$dates, $client;
    }
    $res->{dates} = $dates;

    foreach my $client ( @$orders ) {
        foreach my $order ( @{$client->{dates} || []} ) {
            $order->{date} = $format_date_cache{$order->{stat_date}} 
                ||= Stat::Tools::format_date($order->{stat_date}, $dateagg)
            ;
        }
    }

    return $res;
}

=head2 plot_stat

    Отдает картинку с графиком статистики (используется в Баяне)

=cut

sub plot_stat{
    my ($vars,$r) = @_;

    require Cairo;
    require CairoGraph;

    my $surface = Cairo::ImageSurface->create ('argb32', 634, 490);
    my $cr = Cairo::Context->create ($surface);

    my $month_full_name = [
            iget_noop("январь"),
            iget_noop("февраль"),
            iget_noop("март"),
            iget_noop("апрель"),
            iget_noop("май"),
            iget_noop("июнь"),
            iget_noop("июль"),
            iget_noop("август"),
            iget_noop("сентябрь"),
            iget_noop("октябрь"),
            iget_noop("ноябрь"),
            iget_noop("декабрь"),
        ];

    my $season_name = [
            iget_noop("зима"),
            iget_noop("весна"),
            iget_noop("лето"),
            iget_noop("осень"),
        ];

    my $number_postfix = [
            undef, 
            iget_noop("тыс"),
            iget_noop("млн"),
            iget_noop("млрд"),
            iget_noop("трлн"),
        ];

    unless ( CairoGraph::cr_draw_multi_plot($cr, CairoGraph::hash_merge($vars,
        {
            x=>30, y=>444, width=>603, height=>416,
            markup => ($vars->{group}),
            month_full_name =>  [map { iget $_ } @$month_full_name],
            season_name     =>  [map { iget $_ } @$season_name],
            number_postfix  =>  [map { $_ ? iget($_) : '' } @$number_postfix ]
        }
        ))
        ) {
        print STDERR "  CAIRO_GRAPH:>>:EORROS:>>:\n",join("\n", CairoGraph::cr_err_list()),"\n";
    }

    #$surface->write_to_png ('tmp.png');
    my $buff;
    $surface->write_to_png_stream(sub {
        my ($closure, $data) = @_;
        $buff .= $data;
    });

    return respond_data($r, $buff, 'image/png');
}

=head2 associate_phrases_from_stat(cid, bids)

    Make associate phrases from stat with real phrases
    Returned hash with 
        { bids.PhraseID => {id => bids.id, price => bids.price, autobudgetPriority => bids.aP} }

=cut

sub associate_phrases_from_stat($$) {
    my ($cid, $banners) = @_;    

    return if ref $banners ne 'ARRAY';

    my $res = get_all_sql(PPC(cid => $cid), ['SELECT STRAIGHT_JOIN
                                    bi.cid, b.bid, bi.id, bi.PhraseID, bi.price, bi.price_context,
                                    bi.autobudgetPriority, bi.phrase,
                                    IFNULL(auct.rank, 0) as rank, b.pid as adgroup_id
                                  FROM 
                                    banners b
                                    JOIN bids bi on bi.pid=b.pid
                                    LEFT JOIN bs_auction_stat auct on auct.pid = bi.pid and auct.PhraseID = bi.PhraseID'
                                  , where => { 'b.cid' => $cid, 'b.bid' => [map {$_->{bid}} @$banners]}]);
    
    return {map {$_->{bid}."_".$_->{PhraseID} => $_} @$res};
}

# --------------------------------------------------------------------

=head2 associate_retargetings_from_stat

    достаем данные по условиям ретаргетинга, которые есть в статистике
    $vars->{real_retargetings} = DBStat::associate_retargetings_from_stat($data->{cid}, $campstat->{banners});
    в ответе хеш с ключами "${bid}_${ret_id}" и значение - хеш с условием из bids_retargeting
    Если условие есть в статистике, но нет в bids_retargeting, значения ret_id, price_context и
    autobudgetPriority будут равны undef.

=cut

sub associate_retargetings_from_stat($$) {
    my ($cid, $banners) = @_;

    return if ref $banners ne 'ARRAY';

    my $result = {};

    my @bids;
    my %PhraseIDs;
    for my $banner (@$banners) {
        push @bids, $banner->{bid};

        if ($banner->{phrases}) {
            foreach my $phrase (@{$banner->{phrases}}) {
                unless ($phrase->{ContextType} == $CONTEXT_TYPE_RET) {
                    next;
                }
                $PhraseIDs{$phrase->{PhraseID}} = 1;
                my $key = $banner->{bid}.'_'.$phrase->{PhraseID};
                $result->{$key} = {ret_cond_id => $phrase->{PhraseID}, bid => $banner->{bid}};
            }
        } elsif ($banner->{all_stat}) {
            foreach my $row (@{$banner->{all_stat}}) {
                unless ($row->{ContextType} == $CONTEXT_TYPE_RET) {
                    next;
                }
                $PhraseIDs{$row->{PhraseID}} = 1;
                my $key = $banner->{bid}.'_'.$row->{PhraseID};
                $result->{$key} = {ret_cond_id => $row->{PhraseID}, bid => $banner->{bid}};
            }
        }
    }

    my $retargetings_details = get_all_sql(PPC(cid => $cid), [
        "select br.ret_id, br.ret_cond_id, b.bid, br.price_context, br.autobudgetPriority, br.pid as adgroup_id
         from bids_retargeting br
            join phrases p ON p.pid = br.pid
            join banners b ON b.pid = p.pid
        ", where => {
            'p.cid' => $cid,
            'b.bid' => \@bids,
            'br.ret_cond_id' => [keys %PhraseIDs],
        }
    ]);

    for (@$retargetings_details) {
        my $key = $_->{bid}.'_'.$_->{ret_cond_id};
        if ($result->{$key}) {
            hash_merge $result->{$key}, $_;
        }
    }

    return $result;
}

=head2 associate_relevance_match_from_stat

    Достаем данные по беcфразному таргетингу, которые есть в статистике
    $vars->{real_relevance_match} = DBStat::associate_relevance_match_from_stat($data->{cid}, $campstat->{banners});
    в ответе хеш с ключами "${bid}_${RELEVANCE_MATCH_PHRASE_ID}" и значение - хеш с условием из bids_base
    Если условие есть в статистике, но нет в bids_base, значения price, price_context и
    autobudgetPriority будут равны undef.

=cut

sub associate_relevance_match_from_stat($$) {
    my ($cid, $banners) = @_;

    return if ref $banners ne 'ARRAY';

    my $result = {};

    my @bids;
    for my $banner (@$banners) {
        #Для беcфразного таргетинга PhraseID всегда равно константе $RELEVANCE_MATCH_PHRASE_ID
        next unless any {$_->{ContextType} == $CONTEXT_TYPE_RELEVANCE_MATCH} (@{$banner->{phrases}}, @{$banner->{all_stat}});

        my $key = $banner->{bid}.'_'.$RELEVANCE_MATCH_PHRASE_ID;
        $result->{$key} = {bid => $banner->{bid}};
        push @bids, $banner->{bid};
    }

    my $details = @bids ? get_all_sql(PPC(cid => $cid), [
        "select brm.bid_id, b.bid, brm.price, brm.price_context, brm.autobudgetPriority, brm.pid as adgroup_id
         from bids_base brm
            join banners b ON b.pid = brm.pid
        ", where => {
            'brm.cid' => $cid,
            'b.bid' => \@bids,
            'brm.bid_type' => 'relevance_match',
        }
    ]) : [];

    for (@$details) {
        my $key = $_->{bid}.'_'.$RELEVANCE_MATCH_PHRASE_ID;
        if ($result->{$key}) {
            hash_merge $result->{$key}, $_;
        }
    }

    return $result;
}

sub _debug {
    return if !$DEBUG;
    my $text = join ", ", @_;
    for my $line (split /\n/, $text) {
        print STDERR strftime("%Y-%m-%d:%H:%M:%S", localtime)."\t[$$]\tDBStat\t$line\n";
    }
}

=head2 had_nds_by_camp

Был ли когда-нибдь на данных кампаниях НДС

    had_nds_by_camp({ 'cid' => \@cids });

Возвращает 1 если был и 0, если не был

=cut

sub had_nds_by_camp
{
    my $where = shift;
    my @shard = choose_shard_param($where, [qw/cid OrderID/], set_shard_ids => 1);

    my $max_nds = overshard group => 1, max => [qw/nds/],
                                     get_all_sql(PPC(@shard), [
                "SELECT MAX(IFNULL(cna.nds, cn.nds)) nds
                   from campaigns c join users u on u.uid = c.uid
                   LEFT JOIN clients_options clo ON clo.ClientID = u.ClientID
                   left join client_nds cn ON IF(c.AgencyID > 0 AND IFNULL(clo.non_resident, 0) = 0, c.AgencyID, u.ClientID) = cn.ClientID AND cn.date_from < NOW()
                   left join agency_nds cna ON (c.AgencyID > 0 AND IFNULL(clo.non_resident, 0) = 0) AND cna.ClientID = c.AgencyID AND cna.date_from < NOW()",
                   where => $where,
            ]);
    $max_nds = $max_nds->[0] // {};

    return (($max_nds->{nds} // 0) > 0 ? 1 : 0);
}

1;
