package DayBudgetAlerts;

=head1 NAME
    
    DayBudgetAlerts

=head1 DESCRIPTION

    Уведомления об остановке кампаний по дневному бюджету.
    Для отправки письма у нас должна быть дата приостановки и по статистике должна быть истрачена сумма, превышающая дневной бюджет.
    Уведомления для остановленных по временному таргетингу кампаний не отправляются.

=cut

use Direct::Modern;

use Settings;
use TimeTarget;
use Notification;
use Campaign::Types qw/get_camp_kind_types/;
use WalletUtils;
use Client::ClientFeatures;

use List::MoreUtils qw/uniq/;
use List::Util qw/sum0/;

use Yandex::DBShards qw/SHARD_IDS/;
use Yandex::DBTools;
use Yandex::TimeCommon qw/today/;
use Yandex::Log;
use Yandex::HashUtils qw/hash_cut/;

use base qw/Exporter/;
our @EXPORT = qw/
    send_campaigns_paused_by_day_budget_alerts
    send_wallets_paused_by_day_budget_alerts
/;

our $LOG;

# типы кампаний у которых может быть включен дневной бюджет
our $DAY_BUDGET_ALLOWED_CAMP_TYPES = get_camp_kind_types('web_edit_base');

# условие по которому мы определяем, что кампания или общий счет были остановлены по дневному бюджету, а уведомление еще не отсылалась
my @SQL_WHERE_READY_FOR_BUDGET_NOTIFICATION = (
    'co.day_budget_stop_time__ge' => get_day_budget_start_time(),
    'co.day_budget_notification_status' => 'Ready',
    'c.day_budget__gt' => 0,
);
# условие по которому мы определяем, что для кампании имеет смысл отсылать уведомление про остановку по дневному бюджету
my @SQL_WHERE_CAMP_MAKES_SENSE_TO_SEND_NOTIFICATION = (
    'c.OrderID__gt' => 0,
    'c.statusEmpty' => 'No',
    'c.archived' => 'No',
    'c.statusShow' => 'Yes',
    'c.statusModerate' => 'Yes',
    'c.type' => $DAY_BUDGET_ALLOWED_CAMP_TYPES,
);

=head2 send_campaigns_paused_by_day_budget_alerts

    выбирает кампании, по которым нужно отослать уведомление об остановке по дневному бюджету, и отправляет уведомления

=cut

sub send_campaigns_paused_by_day_budget_alerts {
    my ($shard, $only_cids) = @_;
    $only_cids //= [];

    _log()->out('Fetching campaigns to send notifications');
    my $cids;
    if (!@$only_cids) {
        $cids = get_one_column_sql(PPC(shard => $shard), [
           'SELECT STRAIGHT_JOIN co.cid
            FROM camp_options co JOIN campaigns c ON c.cid = co.cid',
            WHERE => \@SQL_WHERE_READY_FOR_BUDGET_NOTIFICATION
        ]);
    } else {
        $cids = $only_cids;
    }

    my $campaigns = get_all_sql(PPC(shard => $shard), [
       'SELECT co.cid, c.name, c.OrderID, c.day_budget, co.day_budget_stop_time, c.uid, u.ClientID, u.login, c.timeTarget, c.timezone_id
            , c.wallet_cid, c.sum, c.sum_spent
        FROM campaigns c
            JOIN camp_options co ON co.cid = c.cid
            JOIN users u ON u.uid = c.uid
            LEFT JOIN campaigns wc ON wc.cid = c.wallet_cid',
        WHERE => [
            'c.cid' => $cids,
            @SQL_WHERE_READY_FOR_BUDGET_NOTIFICATION,
            @SQL_WHERE_CAMP_MAKES_SENSE_TO_SEND_NOTIFICATION,
            'co.email_notifications__scheck' => {paused_by_day_budget => 1},
        ]
    ]);

    $campaigns = filter_camps_active_by_feature_disabled($campaigns);
    $campaigns = filter_camps_active_by_timetarget($campaigns);
    $campaigns = filter_camps_active_by_has_money($campaigns);

    if (@$campaigns) {
        _log()->out('Found ' . scalar(@$campaigns) . ' campaigns to send notifications');

        my $cids = [ map {$_->{cid}} @$campaigns ];
        my @warned_cids;
        for my $campaign (@$campaigns) {
                _log()->out("Sending notification for cid = $campaign->{cid}");
                my $vars = {campaign => hash_cut($campaign, qw/cid name day_budget_stop_time ClientID/), chief_rep => hash_cut($campaign, qw/uid login/)};
                my $success = eval {
                    add_notification(undef, 'paused_by_day_budget', $vars);
                    return 1;
                };
                if ($success) {
                    push @warned_cids, $campaign->{cid};
                } else {
                    _log()->out("Error sending notification for cid $campaign->{cid}, skipping: $@");
                }
        }
        _log()->out('Updating notification status for campaigns: '.join(',', @warned_cids));
        if (@warned_cids) {
            do_update_table(PPC(shard => $shard), 'camp_options', {day_budget_notification_status => 'Sent'}, where => {cid => \@warned_cids});
        }
    }
}

=head2 send_wallets_paused_by_day_budget_alerts

    выбирает кошельки, по которым нужно отослать уведомление об остановке по дневному бюджету, и отправляет уведомления

=cut

sub send_wallets_paused_by_day_budget_alerts {
    my ($shard, $only_cids) = @_;
    $only_cids //= [];

    _log()->out('Fetching wallets to send notifications');
    my $cids;
    if (!@$only_cids) {
        $cids = get_one_column_sql(PPC(shard => $shard), [
           'SELECT STRAIGHT_JOIN co.cid
            FROM wallet_campaigns wcc JOIN camp_options co ON co.cid = wcc.wallet_cid JOIN campaigns c ON c.cid = wcc.wallet_cid',
            WHERE => \@SQL_WHERE_READY_FOR_BUDGET_NOTIFICATION
        ]);
    } else {
        $cids = $only_cids;
    }

    my $wallets = get_all_sql(PPC(shard => $shard), [
       'SELECT co.cid, c.day_budget, co.day_budget_stop_time, c.uid, u.ClientID, u.login, co.sms_time, c.type,
                 FIND_IN_SET("paused_by_day_budget", co.email_notifications) AS send_email
        FROM wallet_campaigns wcc
            JOIN camp_options co ON co.cid = wcc.wallet_cid
            JOIN campaigns c ON c.cid = wcc.wallet_cid
            JOIN users u ON u.uid = c.uid',
        WHERE => [
            'wcc.wallet_cid' => $cids,
            'c.type' => 'wallet',
            @SQL_WHERE_READY_FOR_BUDGET_NOTIFICATION,
            _OR => [
                'co.email_notifications__scheck' => {paused_by_day_budget => 1},
                'co.sms_flags__scheck' => {paused_by_day_budget_sms => 1},
            ],
        ]
    ]);

    if (@$wallets) {
    	$wallets = filter_camps_active_by_feature_disabled($wallets);
        _log()->out('Found ' . scalar(@$wallets) . ' wallets to possibly send notifications');
        my @warned_cids;

        # проверяем что под кошельками есть хотя бы одна кампания удовлетворяющая условиям отправки уведомления
        my %wallet_has_suitable_camps;
        my $camps_under_wallets = get_all_sql(PPC(shard => $shard), [
           'SELECT c.cid, c.OrderID, c.timeTarget, c.timezone_id, c.wallet_cid, c.sum, c.sum_spent
            FROM campaigns c
                JOIN camp_options co ON co.cid = c.cid
                JOIN users u ON u.uid = c.uid
                LEFT JOIN campaigns wc ON wc.cid = c.wallet_cid',
            WHERE => [
                'c.uid' => [ map { $_->{uid} } @$wallets],
                'c.wallet_cid' => [ map { $_->{cid} } @$wallets],
                @SQL_WHERE_CAMP_MAKES_SENSE_TO_SEND_NOTIFICATION,
            ]
        ]);
        $camps_under_wallets = filter_camps_active_by_timetarget($camps_under_wallets);
        $camps_under_wallets = filter_camps_active_by_has_money($camps_under_wallets);
        $wallet_has_suitable_camps{$_->{wallet_cid}} = 1 for @$camps_under_wallets;

        for my $wallet (@$wallets) {
            next unless $wallet_has_suitable_camps{$wallet->{cid}};

            _log()->out("Sending notification for wallet_cid = $wallet->{cid}");
            my $vars = {
                cid => $wallet->{cid}, 
                campaign => hash_cut($wallet, qw/cid day_budget_stop_time ClientID type/), 
                chief_rep => hash_cut($wallet, qw/uid login/), 
                sms_time => $wallet->{sms_time}
            };
            my $success = eval {
                add_notification(undef, 'paused_by_day_budget', $vars, hash_cut($wallet, qw/send_email/));
                return 1;
            };
            if ($success) {
                push @warned_cids, $wallet->{cid};
            } else {
                _log()->out("Error sending notification for wallet_cid $wallet->{cid}, skipping: $@");
            }
        }

        _log()->out('Updating notification status for wallets: '.join(',', @warned_cids));
        if (@warned_cids) {
            do_update_table(PPC(shard => $shard), 'camp_options', {day_budget_notification_status => 'Sent'}, where => {cid => \@warned_cids});
        }
    }
}

=head2 filter_camps_active_by_feature_disabled
	
    возвращает кампании, принадлжещие пользователям с отлкюченной фичей paused_by_day_budget_warnings
    
=cut

sub filter_camps_active_by_feature_disabled() {
    my $campaigns = shift;
    
    my @client_ids = uniq map {$_->{ClientID}} @$campaigns;
    my $feature_enabled = Client::ClientFeatures::is_paused_by_day_budget_warnings_enabled_for_client_ids(\@client_ids);
    
    return [ grep {
    	!($feature_enabled->{$_->{ClientID}} // 0)
    } @$campaigns ]; 
}


=head2 filter_camps_active_by_timetarget

    возвращает только те кампании, которые не остановлены на текущий момент по временному таргетингу

=cut

sub filter_camps_active_by_timetarget {
    my $campaigns = shift;

    return [ grep {
        (
            TimeTarget::timetarget_status_raw(
                $_->{'timeTarget'}, $_->{'timezone_id'}
            )->{'status_id'} // ''
        ) eq 'ACTIVE'
    } @$campaigns ];
}

=head2 filter_camps_active_by_has_money

    возвращает только те кампании которые могут крутиться по наличию средств (собственных или общего счета)

=cut

sub filter_camps_active_by_has_money {
    my $campaigns = shift;

    my @wallet_cids = uniq grep { $_ > 0 } map { $_->{wallet_cid} } @$campaigns;
    my $wallet_sums = get_hash_sql(PPC(cid => \@wallet_cids), ['SELECT cid, sum FROM campaigns', WHERE => {type=>'wallet', cid => SHARD_IDS}]);
    my $wallet_debts = WalletUtils::get_sum_debts_for_wallets(\@wallet_cids);

    return [ grep {
        has_money($_, $wallet_sums, $wallet_debts)
    } @$campaigns ];
}

=head2 has_money

    проверить есть ли у кампании деньги доступные к трате

    Параметры:
        $camp - hashref с данными о кампании, должны быть wallet_cid, sum, sum_spent
        $wallet_sums - словарь wallet_cid => sum (сумма самого кошелька по таблице campaigns)
        $wallet_debts - словарь wallet_cid => wallet_debt (перетраты кампаний под ОС, значение - отрицательное) 

=cut 

sub has_money {
    my ($camp, $wallet_sums, $wallet_debts) = @_;

    my $wallet_cid = $camp->{wallet_cid};
    my $wallet_sum = $wallet_sums->{$wallet_cid} // 0;
    my $wallet_debt = $wallet_debts->{$wallet_cid} // 0;

    my $rest = $wallet_sum + $wallet_debt;
    my $sum_total = $camp->{sum} - $camp->{sum_spent};
    if ($sum_total > 0) {
        $rest += $sum_total;
    }

    return $rest > $Currencies::EPSILON
}

=head2 get_day_budget_start_time

    Возвращает время, начиная с которого сейчас отсчитывается дневной бюджет, т.е. начало дня.
    Если остановка кампании/общего счета произошла до этого времени, она уже не актуальна, т.к. кампанию уже должны были перезапустить

=cut

sub get_day_budget_start_time {
    return today();
}

=head2 _log

    возвращает объект Yandex::Log

=cut

sub _log {
    $LOG //= new Yandex::Log(no_log => 1);
    return $LOG;
}

1;
