package DeviceTargeting;

# $Id$

=head1 NAME
    
    DeviceTargeting

=head1 DESCRIPTION

    Модуль про "Таргетинг на устройства"

=cut

use strict;
use warnings;

use utf8;

use User qw/get_user_options/;

use base qw/Exporter/;
our @EXPORT = qw/
    is_valid_device_targeting
    can_view_user_device_targeting_option
    can_view_camp_device_targeting
    can_save_camp_device_targeting
    is_only_phone_device_targeting
    get_target_devices
/;

use constant OTHER_DEVICE   => 0b10000; # Персональные компьютeры, ноутбуки и всё другое, кроме iOS и Android
use constant PHONE_DEVICE   => 0b01000; # Сматрфоны
use constant TABLET_DEVICE  => 0b00010; # Планшеты
use constant IOS_DEVICE     => 0b00100; # iPad и iPhone
use constant ANDROID_DEVICE => 0b00001; # телефоны и планшеты под Android'ом

# Хеш с доступными таргетингами
our %DEVICES = (
    other_devices   => OTHER_DEVICE,
    ipad            => IOS_DEVICE | TABLET_DEVICE,
    iphone          => IOS_DEVICE | PHONE_DEVICE,
    android_phone   => ANDROID_DEVICE | PHONE_DEVICE,
    android_tablet  => ANDROID_DEVICE | TABLET_DEVICE,
);

=head2 settings in user_options

    В YAML user_options могут быть следющие опции, имеющие отношение к таргетингу на устройства
    allow_device_targeting - deprecated (DIRECT-23075)
        - относится к логинам менеджеров
        - означает право менеджера видеть/ставить на кампаниях таргетинг на устройства
    can_have_device_targeting
        - относится к логинам клиентов (в том числе: суб-, суперсуб-)
        - означает право менеджера или саппорта видеть/ставить таргетинг на устройства на кампании ЭТОГО клиента

=cut

=head2 is_valid_device_targeting
    
    Валидирует строку с таргетингом. Возвращает 0 при ошибках, 1 если все ок.

=cut
sub is_valid_device_targeting {
    my $targeting = shift;

    my $devs;
    map {$devs->{$_}++} split (/\,/, $targeting, -1);

    while (my ($dev, $cnt) = each %$devs) {
        # например две запятые подряд
        return 0 if $dev eq '';
        # неизвестное устройство
        return 0 unless exists $DEVICES{$dev};
        # дубликаты
        return 0 if $cnt > 1;
    }

    # Всё ок или пустой таргетинг (который по смыслу аналогичен полному)
    return 1;
}

=head2 can_view_camp_device_targeting

    Проверяет, можно ли пользователю видеть "таргетинг на устройства" в параметрах кампании
        Принимает ссылку на экземпляр Direct::Context
        Возвращает 0/1
    Используется в:
        DoCmd::cmd_showCampSettings (show_camp_settings.html)
        DoCmd::cmd_editCamp (edit_camp.html)
        DoCmd::cmd_saveNewCamp (edit_camp.html)
        DoCmd::cmd_saveCamp (edit_camp.html)

=cut
sub can_view_camp_device_targeting {
    my $c = shift;

    if ($c->{login_rights}->{super_control}
        || ($c->{login_rights}->{superreader_control} || $c->{login_rights}->{limited_support_control})
        || (($c->{login_rights}->{manager_control} || $c->{login_rights}->{support_control})
            && get_user_options($c->{uid})->{can_have_device_targeting})
    ){
        return 1;
    } else {
        return 0;
    }
}

=head2 can_save_camp_device_targeting
    
    Проверяет, может ли этот пользователь задать на кампании "таргетинг на устройства"
        Принимает ссылку на экземпляр Direct::Context
        Возвращает 0/1
    Используется в Campaign::save_camp

=cut
sub can_save_camp_device_targeting {
    my $c = shift;

    if ($c->{login_rights}->{super_control}
        || (($c->{login_rights}->{manager_control} || $c->{login_rights}->{support_control})
            && get_user_options($c->{uid})->{can_have_device_targeting})
    ) {
        return 1;
    } else {
        return 0;
    }
}

=head2 can_view_user_device_targeting_option

    Проверяет, можно ли в настройках логина увидеть галочку "таргетинг на устройства"
        Принимает ссылку на DirectContext
        Возвращает 0/1
    Используется в DoCmdStaff::cmd_modifyUser (b-modify-user__general-settings.tt2)

=cut
sub can_view_user_device_targeting_option {
    my $c = shift;

    # Суперы могут не только просматривать, но и сохранять
        # Проверка прав при сохранении - в Staff::save_edited_user
    # У остальных галочка задисейблена в интерфейсе и её значение игнорируется при сохранении

    if ($c->{login_rights}->{super_control}
        || $c->{login_rights}->{superreader_control}
        || $c->{login_rights}->{support_control}
    ) {
        return 1;
    } else {
        return 0;
    }
}


=head2 is_only_phone_device_targeting
    
    Проверяет что таргетинг установлен только на смартфоны

=cut

sub is_only_phone_device_targeting {
    my $targeting = shift;
    if ($targeting) {
        my %devices = map {$_ => 1} split (/\s*\,\s*/, $targeting);

        for my $device (keys %devices) {
            return 0 unless $DEVICES{$device} & PHONE_DEVICE;
        }
    } else {
        return 0;
    }
    return 1;
}


=head2 get_target_devices

    Возвращает список из двух элементов:
        - список с типом устройств на которое таргируемся (0 - десктоп, 1 - мобильные, 2 - планшеты)
            максимум 2 элемента в списке, таргетинг на все три типа == пустой список
        - признак таргетирования на мобильную ОС (2 - Android, 3 - iOS)
            0 - в случае отсутствия таргетирования на ОС

=cut

sub get_target_devices {
    my $targeting = shift;

    return ([], 0, ) unless $targeting;

    my ($is_desktop, $is_phone, $is_tablet, $is_ios, $is_android);
    my %devices = map {$_ => 1} split (/\s*\,\s*/, $targeting);

    for my $device (keys %devices) {
        $is_desktop = 1 if $DEVICES{$device} == OTHER_DEVICE;
        $is_phone = 1 if $DEVICES{$device} == ($DEVICES{$device} | PHONE_DEVICE);
        $is_tablet = 1 if $DEVICES{$device} == ($DEVICES{$device} | TABLET_DEVICE);
        $is_ios = 1 if $DEVICES{$device} == ($DEVICES{$device} | IOS_DEVICE);
        $is_android = 1 if $DEVICES{$device} == ($DEVICES{$device} | ANDROID_DEVICE);
    }

    # таргетинг на все типы устройств равнозначен пустому таргетингу
    return ([], 0, ) if $is_desktop && $is_phone && $is_tablet;

    my $target_detailed_device_type = 0;
    if ($is_android && !$is_ios && !$is_desktop) {
        $target_detailed_device_type = 2; # только Android
    }
    if ($is_ios && !$is_android && !$is_desktop) {
        $target_detailed_device_type = 3; # только iOS
    }
    
    my @target_device_group;
    push @target_device_group, 0 if $is_desktop;
    push @target_device_group, 1 if $is_phone;
    push @target_device_group, 2 if $is_tablet;

    return (\@target_device_group, $target_detailed_device_type, );
}

1;
