package Direct::AdGroups2;

use Mouse;
with qw/Direct::Role::Copyable/;

use Direct::Modern;

use Settings;

use Direct::Model::AdGroup;
use Direct::Model::AdGroupText;
use Direct::Model::AdGroupDynamic;
use Direct::Model::AdGroupMcbanner;
use Direct::Model::AdGroupCpmBanner;
use Direct::Model::AdGroupCpmVideo;
use Direct::Model::AdGroupMobileContent;
use Direct::Model::MobileContent;
use Direct::Model::AdGroupPerformance;
use Direct::Model::Tag;
use Direct::Model::AdGroupBsTags;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::I18n;
use Yandex::TimeCommon qw/unix2mysql/;
use List::MoreUtils qw/any all pairwise part/;

use GeoTools qw//;
use PrimitivesIds qw/get_clientid/;
use Tools qw/log_cmd make_copy_sql_strings/;
use LogTools qw//;
use HierarchicalMultipliers qw/mass_get_hierarchical_multipliers copy_hierarchical_multipliers/;
use MailNotification qw/mass_mail_notification/;
use ModerateChecks qw//;
use MinusWordsTools;
use TextTools qw/get_num_array_by_str/;
use JSON qw/to_json/;

use JavaIntapi::GenerateObjectIds;

has 'items' => (is => 'ro', isa => 'ArrayRef[Direct::Model::AdGroup]');
has 'total' => (is => 'ro', isa => 'Int');
has 'data'  => (is => 'ro', isa => 'HashRef', init_arg => undef, lazy => 1, default => sub { +{}; });

around BUILDARGS => sub { my ($orig, $class) = (shift, shift); $class->$orig(@_ == 1 ? (items => $_[0]) : @_) };

=head2 manager_class
=head2 manager
=cut

sub manager_class { 'Direct::Model::AdGroup::Manager' }
sub manager { $_[0]->manager_class->new(items => $_[0]->items) }

=head2 WEB_FIELD_NAMES
=cut

sub WEB_FIELD_NAMES {(
    href_params => {field => sprintf('"%s"', iget('Параметры в ссылке'))},
    main_domain => { feed_or_domain_required  => iget('Необходимо задать основной домен группы объявлений или фид') },
)}

=head2 get($adgroup_ids, %options)

По заданным идентификаторам ($adgroup_ids) возвращает instance с выбранными группами.
Поддерживаемые %options см. в get_by.

=cut

sub get {
    my ($class, $adgroup_ids, %options) = @_;
    return $class->get_by(adgroup_id => $adgroup_ids, %options);
}

=head2 get_by($key, $vals, %options)

По заданному критерию возвращает instance с выбранными группами.

Параметры:
    $key  -> по какому ключу выбирать группы: adgroup_id/campaign_id
    $vals -> (Int|ArrayRef[Int]); список идентификаторов
    %options:
        adgroup_type     -> (Str|ArrayRef[Str]); группы каких типов выбирать: base/dynamic/mobile_content/etc
        format           -> Str; для динамических групп ограничение по формату условий нацеливания dynamic_feed/dynamic_domain
        limit/offset     -> параметры для постраничной выборки
        total_count      -> при использовании limit/offset также вычислять общее количество элементов
        extended         -> создавать группы с ролью Extended (и посчитанными полями banners_count/has_show_conditions/etc)
        with_tags        -> выбирать теги
        with_adgroup_bs_tags -> выбирать теги для отправки в БК
        with_multipliers -> выбирать коэффициенты на группу (по умолчанию - 0)

=cut

sub get_by {
    my ($class, $key, $vals, %options) = @_;
# TODO DIRECT-67003 поддержать mcbanner
    croak "only `adgroup_id`/`campaign_id` keys are supported" unless $key =~ /^(?:adgroup|campaign)_id$/;

    $vals = [$vals // ()] if ref($vals) ne 'ARRAY';
    return $class->new(items => []) if !@$vals;

    # Поддерживаемые типы групп в этой функции
    state $supported_adgroup_types = {
        base => 1, dynamic => 1, mobile_content => 1, performance => 1, mcbanner => 1, cpm_banner => 1, cpm_video => 1,
    };

    state $format_rules = {
        dynamic_domain => {'gd.main_domain_id__is_not_null' => 1},
        dynamic_feed=> {'gd.feed_id__is_not_null' => 1},
    };

    my %format;
    if (my $fmt = delete $options{format}) {
        croak "unknown format `$fmt` specified" if !exists $format_rules->{$fmt};
        %format = %{ $format_rules->{$fmt} };
    }

    if (defined $options{adgroup_type}) {
        $options{adgroup_type} = [$options{adgroup_type}] unless ref($options{adgroup_type}) eq 'ARRAY';
        croak "found unsupported adgroup_type in options" unless all { $supported_adgroup_types->{$_} } @{$options{adgroup_type}};
    }
    $options{adgroup_type} //= [keys %$supported_adgroup_types];

    # Хеш с типам групп, которые следует выбирать
    my $with_adgroup_type = {map { $_ => 1 } @{$options{adgroup_type}}};

    my (@select_columns, @from_tables);

    push @select_columns,
        Direct::Model::AdGroup->get_db_columns(phrases => 'g', prefix => '', exclude_fields => [qw/is_bs_rarely_loaded/]),
        'mw.mw_text AS minus_words_str',
        Direct::Model::AdGroup->get_db_columns(group_params => 'gp', prefix => ''),
        'u.ClientID AS client_id',
        "IF(c.archived = 'Yes' OR g.adgroup_type IN ('dynamic', 'performance'), 0, g.is_bs_rarely_loaded) AS is_bs_rarely_loaded";


    push @from_tables,
        'phrases g',
        'LEFT JOIN minus_words mw ON (mw.mw_id = g.mw_id)',
        'LEFT JOIN group_params gp ON (gp.pid = g.pid)',
        'JOIN campaigns c ON (c.cid = g.cid)',
        'JOIN users u ON (u.uid = c.uid)';

    if ($with_adgroup_type->{dynamic}) {
        push @select_columns,
            Direct::Model::AdGroupDynamic->get_db_columns(adgroups_dynamic => 'gd', prefix => ''),
            Direct::Model::AdGroupDynamic->get_db_columns(domains => 'd', prefix => '');

        push @from_tables,
            'LEFT JOIN adgroups_dynamic gd ON (gd.pid = g.pid)',
            'LEFT JOIN domains d ON (d.domain_id = gd.main_domain_id)';
    }

    if ($with_adgroup_type->{mobile_content}) {
        push @select_columns,
            Direct::Model::AdGroupMobileContent->get_db_columns(adgroups_mobile_content => 'gmc', prefix => ''),
            Direct::Model::MobileContent->get_db_columns(mobile_content => 'mc', prefix => 'mc_');

        push @from_tables,
            'LEFT JOIN adgroups_mobile_content gmc ON (gmc.pid = g.pid)',
            'LEFT JOIN mobile_content mc ON (mc.mobile_content_id = gmc.mobile_content_id)';
    }

    if ($with_adgroup_type->{performance}) {
        push @select_columns,
            Direct::Model::AdGroupPerformance->get_db_columns(adgroups_performance => 'gperf', prefix => '');

        push @from_tables, 'LEFT JOIN adgroups_performance gperf ON (gperf.pid = g.pid)',
    }

    if ($with_adgroup_type->{cpm_banner}) {
        push @select_columns,
            Direct::Model::AdGroupCpmBanner->get_db_columns(adgroups_cpm_banner => 'acpmb', prefix => ''),

        push @from_tables,
            'LEFT JOIN adgroups_cpm_banner acpmb ON (acpmb.pid = g.pid)',
    }


    my $calc_found_rows = $options{limit} && $options{total_count} ? 'SQL_CALC_FOUND_ROWS' : '';
    my %shard_selector = (adgroup_id => 'pid', campaign_id => 'cid');

    my $adgroup_rows = get_all_sql(PPC($shard_selector{$key} => $vals), [
        sprintf("SELECT $calc_found_rows %s FROM %s", join(', ', @select_columns), join(' ', @from_tables)),
        where => {
            'g.'.$shard_selector{$key} => SHARD_IDS,
            'g.adgroup_type' => $options{adgroup_type},
            %format,
        },
        $options{limit} ? (
            'ORDER BY g.pid',
            limit => $options{limit}, $options{offset} ? (offset => $options{offset}) : (),
        ) : (),
    ]);

    my $found_rows = $calc_found_rows ? select_found_rows(PPC($shard_selector{$key} => $vals)) : undef;
    my $self = $class->new(items => [], $calc_found_rows ? (total => $found_rows) : ());

    return $self unless @$adgroup_rows;

    # Если требуется дополнительная информация по группам - посчитаем и добавим к основной выборке
    if ($options{extended}) {
        my $banners_count = get_hashes_hash_sql(PPC(pid => [map { $_->{pid} } @$adgroup_rows]), [q{
            SELECT
                pid, COUNT(bid) banners_count, COUNT(IF(statusArch = 'Yes', 1, NULL)) archived_banners_count
            FROM banners
        }, where => {pid => SHARD_IDS}, 'GROUP BY pid']);

        my $has_show_conditions = get_hash_sql(PPC(pid => [map { $_->{pid} } @$adgroup_rows]), [q{
            SELECT g.pid, 1 FROM phrases g JOIN campaigns c ON (c.cid = g.cid)},
            WHERE => {'g.pid' => SHARD_IDS}, q{
            AND (
                g.adgroup_type IN ('base', 'mobile_content', 'mcbanner', 'cpm_banner') AND (
                    c.archived = 'No' AND EXISTS (SELECT 1 FROM bids bi WHERE bi.pid = g.pid) OR
                    c.archived = 'Yes' AND EXISTS (SELECT 1 FROM bids_arc bi WHERE bi.pid = g.pid AND bi.cid = g.cid)
                ) OR
                g.adgroup_type IN ('base', 'mobile_content', 'cpm_banner', 'cpm_video') AND (
                    EXISTS (SELECT 1 FROM bids_retargeting b WHERE b.pid = g.pid) OR
                    EXISTS (SELECT 1 FROM bids_base bb WHERE bb.pid = g.pid AND NOT FIND_IN_SET('deleted', opts) AND bb.bid_type <> 'keyword')
                ) OR
                g.adgroup_type = 'dynamic' AND EXISTS (SELECT 1 FROM bids_dynamic bd WHERE bd.pid = g.pid)
                OR
                g.adgroup_type = 'performance' AND EXISTS (SELECT 1 FROM bids_performance bp WHERE bp.pid = g.pid AND bp.is_deleted = 0)
            )
        }]);

        my $keywords_count = get_hashes_hash_sql(PPC(pid => [map { $_->{pid} } grep { my $t = $_->{adgroup_type}; any { $t eq $_ } qw/base mobile_content mcbanner cpm_banner/ } @$adgroup_rows]), [q{
            SELECT 
                pid, COUNT(id) keywords_count
            FROM bids
        }, where => {pid => SHARD_IDS}, 'GROUP BY pid']);

        my $relevance_matches_count = get_hashes_hash_sql(PPC(pid => [map { $_->{pid} } grep { $_->{adgroup_type} eq 'base' } @$adgroup_rows]), [q{
            SELECT
                pid, COUNT(bid_id) relevance_matches_count, COUNT(IF(FIND_IN_SET('suspended', opts) OR FIND_IN_SET('deleted', opts), NULL, 1)) active_relevance_matches_count
            FROM bids_base
        }, where => {pid => SHARD_IDS, bid_type => 'relevance_match'}, 'GROUP BY pid']);

        my $retargetings_count = get_hashes_hash_sql(PPC(pid => [map { $_->{pid} } grep { my $t = $_->{adgroup_type}; any { $t eq $_ } qw/base mobile_content cpm_banner cpm_video/ } @$adgroup_rows]), [q{
            SELECT
                br.pid,
                COUNT(IF(FIND_IN_SET('interest', rc.properties), NULL, 1)) retargetings_count, COUNT(IF(FIND_IN_SET('interest', rc.properties), NULL, IF(br.is_suspended > 0, NULL, 1))) active_retargetings_count,
                COUNT(IF(FIND_IN_SET('interest', rc.properties), 1, NULL)) target_interests_count, COUNT(IF(FIND_IN_SET('interest', rc.properties), IF(br.is_suspended > 0, NULL, 1), NULL)) active_target_interests_count
            FROM bids_retargeting br JOIN retargeting_conditions rc ON (br.ret_cond_id = rc.ret_cond_id)
        }, WHERE => {pid => SHARD_IDS}, 'GROUP BY pid']);

        my $dyn_conds_count = get_hashes_hash_sql(PPC(pid => [map { $_->{pid} } grep { $_->{adgroup_type} eq 'dynamic' } @$adgroup_rows]), [q{
            SELECT
                pid, COUNT(dyn_id) dyn_conds_count
            FROM bids_dynamic
        }, WHERE => {pid => SHARD_IDS}, 'GROUP BY pid']);

        my $perf_filters_count = get_hashes_hash_sql(PPC(pid => [map { $_->{pid} } grep { $_->{adgroup_type} eq 'performance' } @$adgroup_rows]), [q{
            SELECT
                pid, COUNT(perf_filter_id) perf_filters_count
            FROM bids_performance
        }, WHERE => {pid => SHARD_IDS, is_deleted => 0}, 'GROUP BY pid']);

        for my $row (@$adgroup_rows) {
            my $pid = $row->{pid};
            $row->{banners_count} = $banners_count->{$pid}->{banners_count} // 0;
            $row->{archived_banners_count} = $banners_count->{$pid}->{archived_banners_count} // 0;
            $row->{has_show_conditions} = $has_show_conditions->{$pid};
            if (any { $row->{adgroup_type} eq $_ } qw/base mobile_content/) {
                $row->{keywords_count} = $keywords_count->{$pid}->{keywords_count} // 0;
                $row->{retargetings_count} = $retargetings_count->{$pid}->{retargetings_count} // 0;
                $row->{target_interests_count} = $retargetings_count->{$pid}->{target_interests_count} // 0;
                $row->{active_target_interests_count} = $retargetings_count->{$pid}->{active_target_interests_count} // 0;
                if ($row->{adgroup_type} eq 'base') {
                    $row->{relevance_matches_count} = $relevance_matches_count->{$pid}->{relevance_matches_count} // 0;
                    $row->{active_relevance_matches_count} = $relevance_matches_count->{$pid}->{active_relevance_matches_count} // 0;
                }
            } elsif ($row->{adgroup_type} eq 'cpm_banner' || $row->{adgroup_type} eq 'cpm_video') {
                $row->{keywords_count} = $keywords_count->{$pid}->{keywords_count} // 0;
                $row->{retargetings_count} = $retargetings_count->{$pid}->{retargetings_count} // 0;
            } elsif ($row->{adgroup_type} eq 'mcbanner') {
                $row->{keywords_count} = $keywords_count->{$pid}->{keywords_count} // 0;
            } elsif ($row->{adgroup_type} eq 'dynamic') {
                $row->{dyn_conds_count} = $dyn_conds_count->{$pid}->{dyn_conds_count} // 0;
            } elsif ($row->{adgroup_type} eq 'performance') {
                $row->{perf_filters_count} = $perf_filters_count->{$pid}->{perf_filters_count} // 0;
            }
        }
    }

    my $banners_minus_geo = get_hash_sql(PPC(pid => [map { $_->{pid} } @$adgroup_rows]), [
        "select pid, group_concat(bmg.minus_geo) as minus_geo
        from banners b join banners_minus_geo bmg on bmg.bid = b.bid",
        where => {
            'b.pid' => SHARD_IDS,
        },
        'group by b.pid'
    ]);

    # Получим коэффициенты
    if (@$adgroup_rows && $options{with_multipliers}) {
        my $hierarchical_multiplier_items = HierarchicalMultipliers::mass_get_hierarchical_multipliers([map { +{cid => $_->{cid}, pid => $_->{pid}} } @$adgroup_rows]);
        for (my $i = 0; $i <= $#$adgroup_rows; $i++) {
            $adgroup_rows->[$i]->{hierarchical_multipliers} = $hierarchical_multiplier_items->[$i];
        }
    }

    my $_cache;
    for my $row (@$adgroup_rows) {
        # DIRECT-54448: подменяем устаревшие регионы
        $row->{geo} = GeoTools::substitute_temporary_geo($row->{geo});

        $row->{effective_geo} = $row->{disabled_geo} = [];

        my ($effective_geo, $disabled_geo) = ('', []);

        my @minus_geo = @{get_num_array_by_str($banners_minus_geo->{ $row->{pid} })};
        if (@minus_geo) {
            ($effective_geo, $disabled_geo) = GeoTools::exclude_region($row->{geo}, \@minus_geo, { ClientID => $row->{client_id} });
        }

        $row->{minus_geo} = \@minus_geo;
        $row->{effective_geo} = [ split /,/, $effective_geo ];
        $row->{disabled_geo} = $disabled_geo;

        $row->{minus_words} = MinusWordsTools::minus_words_str2array(delete $row->{minus_words_str}) // [];
        if ($row->{adgroup_type} eq 'base') {
            push @{$self->items}, Direct::Model::AdGroupText->from_db_hash($row, \$_cache);
        } elsif ($row->{adgroup_type} eq 'mcbanner') {
            push @{$self->items}, Direct::Model::AdGroupMcbanner->from_db_hash($row, \$_cache);
        } elsif ($row->{adgroup_type} eq 'dynamic') {
            push @{$self->items}, Direct::Model::AdGroupDynamic->from_db_hash($row, \$_cache);
        } elsif ($row->{adgroup_type} eq 'mobile_content') {
            my $adgroup = Direct::Model::AdGroupMobileContent->from_db_hash($row, \$_cache);
            $adgroup->mobile_content(
                Direct::Model::MobileContent->from_db_hash($row, \$_cache, prefix => 'mc_')
            );
            push @{$self->items}, $adgroup;
        } elsif ($row->{adgroup_type} eq 'performance') {
            push @{$self->items}, Direct::Model::AdGroupPerformance->from_db_hash($row, \$_cache);
        } elsif ($row->{adgroup_type} eq 'cpm_banner') {
            push @{$self->items}, Direct::Model::AdGroupCpmBanner->from_db_hash($row, \$_cache);
        } elsif ($row->{adgroup_type} eq 'cpm_video') {
            push @{$self->items}, Direct::Model::AdGroupCpmVideo->from_db_hash($row, \$_cache);
        } elsif ($row->{adgroup_type} eq 'cpm_geoproduct') {
            push @{$self->items}, Direct::Model::AdGroupCpmGeoproduct->from_db_hash($row, \$_cache);
        } else {
            croak "fetched unsupported adgroup type: ".$row->{adgroup_type};
        }

        $row->{cpm_banners_type} = $row->{adgroup_type};
    }

    if ($options{with_tags}) {
        my $tag_rows = get_all_sql(PPC(pid => [map { $_->id } @{$self->items}]), [
            q{SELECT tg.pid, tcl.tag_id, tcl.cid, tcl.tag_name, tcl.createtime FROM tag_campaign_list tcl JOIN tag_group tg ON (tg.tag_id = tcl.tag_id)},
            where => {'tg.pid' => SHARD_IDS},
        ]);
        my (%tags_by_gid, $_cache);
        push @{$tags_by_gid{ $_->{pid} }}, Direct::Model::Tag->from_db_hash($_, \$_cache) for @$tag_rows;

        $_->tags($tags_by_gid{ $_->id } || []) for @{$self->items};
    }
    
    if ($options{with_adgroup_bs_tags}) {
        my $adgroup_bs_tags_row = get_all_sql(PPC(pid => [map { $_->id } @{$self->items}]), [
            q{SELECT abt.pid, abt.page_group_tags_json, abt.target_tags_json FROM adgroup_bs_tags abt},
            where => {'abt.pid' => SHARD_IDS},
        ]);
        my (%bs_tags_by_gid, $_cache);
        $bs_tags_by_gid{$_->{pid}} = Direct::Model::AdGroupBsTags->from_db_hash($_, \$_cache) for @$adgroup_bs_tags_row;

        $_->adgroup_bs_tags($bs_tags_by_gid{ $_->id }) for @{$self->items};
    }

    return $self;
}

=head2 items_by($key)

Возвращает структуру с группами, вида:
    $key //eq 'id' => {$adgroup1->id => $adgroup1, $adgroup2->id => $adgroup2, ...};
    $key eq 'cid'  => {$campaign_id1 => [$adgroup1, $adgroup2], $campaign_id2 => [$adgroup3, ...], ...};

=cut

sub items_by {
    my ($self, $key) = @_;

    $key //= 'id';
    croak "by `id`/`cid`/`adgroup_type` only supported" unless $key =~ /^(?:id|cid|campaign_id|adgroup_type)$/;

    my %result;
    if ($key eq 'id') {
        $result{$_->id} = $_ for @{$self->items};
    } elsif ($key eq 'adgroup_type') {
        push @{$result{ $_->adgroup_type }}, $_ for @{$self->items};
    } else {
        push @{$result{ $_->campaign_id }}, $_ for @{$self->items};
    }

    return \%result;
}

=head2 set_client_id_by_uid($uid)

Установка client_id (если не задан) по $uid

=cut

sub set_client_id_by_uid {
    my ($self, $uid) = @_;

    if (any { !$_->has_client_id || !$_->client_id } @{$self->items}) {
        my $client_id = get_clientid(uid => $uid);
        $_->client_id($client_id) for @{$self->items};
    }

    return $self;
}

=head2 prepare_create($uid)

Подготовка списка групп объявлений к созданию для пользователя $uid.

=cut

sub prepare_create {
    my ($self, $uid) = @_;

    my $ids = JavaIntapi::GenerateObjectIds->new(
        object_type => 'adgroup',
        count => scalar(@{$self->items}),
        client_id => get_clientid(uid => $uid))->call();

    for my $adgroup (@{$self->items}) {
        $adgroup->id(shift @$ids);
        $adgroup->bs_priority_id(0) if $adgroup->has_bs_priority_id;

        $adgroup->status_moderate('New');
        $adgroup->status_post_moderate('No');

        # не копируем, т.к. штатная синхронизация значений - односторонняя: YT -> ppc
        $adgroup->is_bs_rarely_loaded(0) if $adgroup->has_is_bs_rarely_loaded;
    }

    return $self;
}

=head2 prepare_update(%options)

Подготовка списка групп объявлений к обновлению.

Параметры:
    %options:
        translocal_opt -> параметры транслокальности (по умолчанию: {ClientID => $adgroup->client_id})

=cut

sub prepare_update {
    my ($self, %options) = @_;

    for my $adgroup (@{$self->items}) {
        # Учитываем, что статус модерации группы может поменяться на 'New' с любого другого
        if ($adgroup->status_moderate eq 'New') {
            $adgroup->status_post_moderate('No');
        }

        # На изменение has_phraseid_href не навешиваем никаких действий,
        # т.к. это поле носит информационный характер и не может переходить из true в false

        if (
            $adgroup->is_minus_words_changed || $adgroup->is_geo_changed || $adgroup->is_href_params_changed ||
            $adgroup->is_hierarchical_multipliers_changed
        ) {
            # Если изменились минус-слова, гео-таргетинг, параметры href или коэффициенты - то переотправим в БК
            $adgroup->status_bs_synced('No');

            # При изменении коэффициентов/параметров href обновим также время последнего изменения группы
            $adgroup->do_update_last_change(1) if $adgroup->is_hierarchical_multipliers_changed || $adgroup->is_href_params_changed;
        }

        # Если изменился гео-таргетинг: нужно обновить geoflag на баннерах и проверить необходимость перемодерации
        if ($adgroup->is_geo_changed) {
            my $translocal_opt = $options{translocal_opt} // {ClientID => $adgroup->client_id};
            GeoTools::refine_geoid($adgroup->geo, \my $new_geoflag, $translocal_opt);
            GeoTools::refine_geoid($adgroup->old->geo, \my $old_geoflag, $translocal_opt);
            if (defined $new_geoflag && $new_geoflag != ($old_geoflag // -1)) {
                $adgroup->do_update_banners_geoflag($new_geoflag);
            }

            if ($adgroup->status_moderate ne 'New') {
                # DIRECT-100514 не трогаем статусы модерации смартов
                if (ModerateChecks::check_moderate_region($adgroup->geo, $adgroup->old->geo) && $adgroup->adgroup_type ne 'performance') {
                    $adgroup->status_moderate('Ready');
                    $adgroup->status_post_moderate('No') if $adgroup->status_post_moderate ne 'Rejected';
                    $adgroup->do_update_status_post_moderate_unless_rejected(1);
                    $adgroup->do_clear_banners_moderation_flags(1);
                }

                # для смартов flags всегда NULL, и мы не попадаем в условие
                if (
                    my @remoderate_banner_ids = map { $_->id } grep {
                        $_->status_moderate ne 'New' &&
                        ModerateChecks::check_moderate_banner_for_regions({$_->flags}, $adgroup->geo, $adgroup->old->geo, $translocal_opt)
                    } @{$adgroup->banners}
                ) {
                    $adgroup->do_remoderate_banner_ids(\@remoderate_banner_ids);
                    $adgroup->do_clear_banners_moderation_flags(1);
                }
            }
        }

        # Пересчет прогноза: при измении минус-слов или гео-таргетинга
        if ($adgroup->is_geo_changed || $adgroup->is_minus_words_changed) {
            $adgroup->status_shows_forecast('New');
            $adgroup->do_schedule_forecast(1);
        }
    }

    return $self;
}

=head2 copy_extra

Копирование групп.

Параметры:
    см. Direct::Role::Copyable
    %options:
        with_multipliers - выполнить "правильное" копирование коэффициентов (например, вместе с ретаргетингом)
        with_mod_reasons - копировать информацию о принятии/отколонении условий показа на модерации

Результат:
    $src_pid2dst_pid - хеш {source_pid => destination_pid}

=cut

sub copy_extra {
    my ($self, $from_client_id, $src_adgroups, $to_client_id, $dst_adgroups, $src_pid2dst_pid, %options) = @_;

    if ($options{with_multipliers}) {
        copy_hierarchical_multipliers($from_client_id, $to_client_id, [
            pairwise { [
                {cid => $a->campaign_id, pid => $a->id},
                {cid => $b->campaign_id, pid => $b->id}
            ] } @{$src_adgroups->items}, @{$dst_adgroups->items}
        ]);
    }

    if ($options{with_mod_reasons}) {
        my @mod_reasons_fields_to_copy = qw/type statusModerate statusPostModerate reason/;
        my %src_pid2dst_cid = pairwise {$a->id => $b->campaign_id} @{$src_adgroups->items}, @{$dst_adgroups->items};
        my %mod_reasons_override = (
            id => $src_pid2dst_pid,
            timeCreated => unix2mysql(time()),
            statusSending => 'Yes',
            ClientID => $to_client_id,
            cid => \%src_pid2dst_cid,
        );
        my ($mod_reasons_fields_str, $mod_reasons_values_str) = make_copy_sql_strings(\@mod_reasons_fields_to_copy, \%mod_reasons_override, by => 'id');
        do_insert_select_sql(PPC(ClientID => $from_client_id),
            "INSERT INTO mod_reasons ($mod_reasons_fields_str) VALUES %s",
            ["SELECT $mod_reasons_values_str FROM mod_reasons", WHERE => {id => [keys %$src_pid2dst_pid], type => 'phrases'}],
            dbw => PPC(ClientID => $to_client_id),
        );
    }

    return $src_pid2dst_pid;
}


=head2 prepare_logging($action, %params)
=head2 do_logging

Методы для логирования событий (действий).

Параметры:
    $action -> выполненное действие: create/update
    %params:
        uid -> uid пользователя, над которым выполняется операция (по умолчанию берется из %LogTools::context)

=cut

sub prepare_logging {
    my ($self, $action, %params) = @_;

    my %log_context = %LogTools::context;
    my $data_for_log_cmd = {};

    for my $adgroup (@{$self->items}) {
        if ($action eq 'update') {
            push @{$self->data->{notifications}}, {
                object     => 'adgroup',
                event_type => 'adgr_geo',
                group_name => $adgroup->adgroup_name,
                object_id  => $adgroup->id,
                old_text   => $adgroup->old->geo,
                new_text   => $adgroup->geo,
                uid        => $params{uid} || $log_context{uid},
            } if $adgroup->old->geo ne $adgroup->geo;
        }
        if ($action eq 'create' || ($action eq 'update' && $adgroup->old->geo ne $adgroup->geo)) {
            push @{ $data_for_log_cmd->{ $adgroup->campaign_id }->{ $adgroup->geo } }, $adgroup->id;
        }
    }

    foreach my $cid (keys %$data_for_log_cmd) {
        push @{$self->data->{log_cmd}}, {
            %log_context,
            cmd => '_set_common_geo',
            cid => $cid,
            uid => $params{uid} || $log_context{uid},
            new_geo_data => to_json($data_for_log_cmd->{$cid}),
        };
    }
    return;
}

sub do_logging {
    my ($self) = @_;

    log_cmd($_) for @{$self->data->{log_cmd} // []};
    mass_mail_notification($self->data->{notifications}) if $self->data->{notifications};

    return;
}

1;
