package Direct::AdGroups2::MobileContent;

use Mouse;
use Direct::Modern;

extends 'Direct::AdGroups2';

use Settings;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::CheckMobileRedirect qw/parse_store_url/;
use Yandex::IDN qw//;
use Yandex::MirrorsTools::Hostings qw/strip_www/;

use Direct::Model::AdGroupMobileContent;
use Direct::Model::AdGroupMobileContent::Manager;

has '+items' => (isa => 'ArrayRef[Direct::Model::AdGroupMobileContent]');

=head2 manager_class
=cut

sub manager_class { 'Direct::Model::AdGroupMobileContent::Manager' }

=head2 get($adgroup_ids, %options)

=cut

sub get {
    my ($class, $adgroup_ids, %options) = @_;
    $class->SUPER::get($adgroup_ids, %options, adgroup_type => 'mobile_content');
}

=head2 get_by($key, $vals, %options)

=cut

sub get_by {
    my ($class, $key, $vals, %options) = @_;
    $class->SUPER::get_by($key, $vals, %options, adgroup_type => 'mobile_content');
}

=head2 get_mobile_content_by($key, $vals)

=cut

sub get_mobile_content_by {
    my ($class, $key, $vals) = @_;

    croak "only `adgroup_id` key is supported" unless $key eq 'adgroup_id';
    
    return {} unless defined $vals;
    $vals = [$vals] unless ref($vals) eq 'ARRAY';
    return {} unless @$vals;

    my $select_columns_str = Direct::Model::MobileContent->get_db_columns(
        mobile_content => 'mc',
        prefix => '',
        fields => [qw/
            id client_id store_content_id store_country os_type content_type bundle_id is_available
            create_time modify_time status_bs_synced name prices_json rating rating_votes icon_hash icon_status_moderate
            min_os_version app_size_bytes available_actions publisher_domain_id genre age_label
        /],
    );

    my $rows = get_all_sql(PPC(pid => $vals), [
        "SELECT g.pid, $select_columns_str, d.domain AS publisher_domain 
        FROM adgroups_mobile_content g
        JOIN mobile_content mc ON (mc.mobile_content_id = g.mobile_content_id)
        LEFT JOIN domains d ON mc.publisher_domain_id = d.domain_id",
        where => {
            'g.pid' => SHARD_IDS,
        },
    ]);
    for (@$rows) {
        if ($_->{publisher_domain}) {
            $_->{publisher_domain} = strip_www(Yandex::IDN::idn_to_ascii($_->{publisher_domain}));
        }
    }

    my $_cache;
    my %result = map { ($_->{pid} => Direct::Model::MobileContent->from_db_hash($_, \$_cache)) } @$rows;

    return \%result;
}

=head2 get_client_mobile_contents($client_id, $mobile_content_id)

    Поиск заданных мобильных приложений клиента
    
    Параметры:
        $client_id - 
        $mobile_content_id - 
        
    Результат:
        Хеш с данными о приложениях
        {mobile_content_id => Direct::Model::MobileContent, ...}

=cut

sub get_client_mobile_contents {
    my ($class, $client_id, $mobile_content_id) = @_;
    
    my $select_columns_str = Direct::Model::MobileContent->get_db_columns(
        mobile_content => 'mc',
        prefix => '',
        fields => [qw/
            id client_id store_content_id store_country os_type content_type bundle_id is_available
            create_time modify_time status_bs_synced name prices_json rating rating_votes icon_hash icon_status_moderate
            min_os_version app_size_bytes available_actions publisher_domain_id genre age_label
        /],
    );
    
    my $rows = get_all_sql(PPC(ClientID => $client_id), [
        "SELECT $select_columns_str, d.domain AS publisher_domain
        FROM mobile_content mc LEFT JOIN domains d ON mc.publisher_domain_id = d.domain_id",
        where => {
            'mc.mobile_content_id' => $mobile_content_id            
        },
    ]);
    for (@$rows) {
        if ($_->{publisher_domain}) {
            $_->{publisher_domain} = strip_www(Yandex::IDN::idn_to_ascii($_->{publisher_domain}));
        }
    }

    my $_cache;
    return {map {
        $_->{mobile_content_id} => Direct::Model::MobileContent->from_db_hash($_, \$_cache)
    } @$rows};
}

=head2 prepare_create($uid)

=cut

sub prepare_create {
    my ($self, $uid) = @_;

    # Создадим Direct::Model::MobileContent для каждого объекта группы
    my $_cache;
    for my $adgroup (@{$self->items}) {
        my $mobile_content_data = parse_store_url($adgroup->store_content_href);
        $mobile_content_data->{ClientID} = $adgroup->client_id;
        $adgroup->mobile_content(
            Direct::Model::MobileContent->from_db_hash($mobile_content_data, \$_cache)
        );
    }

    $self->SUPER::prepare_create($uid);

    return $self;
}

=head2 create

=cut

sub create {
    my ($self, $uid) = @_;

    $self->set_client_id_by_uid($uid);
    $self->prepare_create($uid);
    $self->prepare_logging('create', uid => $uid);
    Direct::Model::AdGroupMobileContent::Manager->new(items => $self->items)->create();
    $self->do_logging();

    return;
}

=head2 prepare_update(%options)

=cut

sub prepare_update {
    my ($self, %options) = @_;

    $self->SUPER::prepare_update(%options);

    for my $adgroup (@{$self->items}) {
        if ($adgroup->is_min_os_version_changed || $adgroup->is_device_type_targeting_changed || $adgroup->is_network_targeting_changed) {
            # При изменении device/network таргетинга сбросим статус синхронизации с БК и обновим last_change
            $adgroup->status_bs_synced('No');
            $adgroup->do_update_last_change(1);
        }
    }

    return $self;
}

=head2 update($uid, %options)

=cut

sub update {
    my ($self, $uid, %options) = @_;

    $self->set_client_id_by_uid($uid);
    $self->prepare_update(%options);
    $self->prepare_logging('update', uid => $uid);
    Direct::Model::AdGroupMobileContent::Manager->new(items => $self->items)->update();
    $self->do_logging();

    return;
}

=head2 prepare_logging($action, %params)

Методы для логирования событий (действий).

Параметры:
    $action -> выполненное действие: create/update
    %params:
        uid -> uid пользователя, над которым выполняется операция (по умолчанию берется из %LogTools::context)

=cut

sub prepare_logging {
    my ($self, $action, %params) = @_;

    $self->SUPER::prepare_logging($action, map { $_ => $params{$_} } qw/uid/);

    my %log_context = %LogTools::context;
    for my $adgroup ($action eq 'update' ? @{$self->items} : ()) {
            my $old_devices =  join ',', sort @{$adgroup->old->device_type_targeting};
            my $new_devices =  join ',', sort @{$adgroup->device_type_targeting};
            push @{$self->data->{notifications}}, {
                object     => 'adgroup',
                event_type => 'adgr_mobile_content_device_type_targeting',
                group_name => $adgroup->adgroup_name,
                object_id  => $adgroup->id,
                old_text   => $old_devices,
                new_text   => $new_devices,
                uid        => $params{uid} || $log_context{uid},
            } if $old_devices ne $new_devices;
            
            my $old_network =  join ',', sort @{$adgroup->old->network_targeting};
            my $new_network =  join ',', sort @{$adgroup->network_targeting};
            push @{$self->data->{notifications}}, {
                object     => 'adgroup',
                event_type => 'adgr_mobile_content_network_targeting',
                group_name => $adgroup->adgroup_name,
                object_id  => $adgroup->id,
                old_text   => $old_network,
                new_text   => $new_network,
                uid        => $params{uid} || $log_context{uid},
            } if $old_network ne $new_network;
    }
}

1;
