package Direct::AdditionsItemCallouts;

use Direct::Modern;

use List::MoreUtils qw/part uniq/;
use Mouse;

use ModerateDiagnosis qw/ mass_get_banner_additions_diags /;
use Settings;

use Direct::Model::AdditionsItemCallout;
use Direct::Model::AdditionsItemCallout::Manager;

use Yandex::DBTools;

has 'items' => (is => 'ro', isa => 'ArrayRef[Direct::Model::AdditionsItemCallout]');

=head2 get_by

По заданному критерию возвращает список уточнений баннеров (моделей Direct::Model::AdditionsItemCallout).

    my $callouts = Direct::AdditionsItemCallouts::get_by(%opts);

где C<%opts>:

    id - искать по точному совпадению с указанным иденитификатором(ами)
    client_id - вернуть все уточнения по клиенту, включая не привязанные к баннерам
    campaign_id - вернуть все уточнения по кампаниям
    adgroup_id - вернуть все уточнения по группе(ам)
    banner_id - вернуть все уточнения по баннер(у|ам)
    callout_text - вернуть все уточнения по тексту уточнения
    status_moderate => 'Yes|No' - выбирать дополнения с заданным статусом модерации
    modified_since => '2016-03-10 00:00:00' - выбирать дополнения по дате последнего изменения начиная с указанной
    shard => если доступен шард, например при выгрузке уточнений для транспорта модерации
    with_mod_reasons => 0|1 - вернуть причины отклонения на модерации для дополнений
    with_deleted => 0|1 - вернуть все дополнения, включая удаленные
    only_deleted => 0|1 - вернуть только удаленные дополнения
    get_banner_id => 0|1 - возвращать banner_id в Direct::Model::AdditionsItemCallout, все дополнения должы быть привязаны к баннерам
    fill_in_linked => 0|1 - заполнять поле is_linked, признак, что условие привязано хотя бы к одному баннеру
    except_client_ids => [] - не возвращать дополнения принадлежащие указанным ClientID

    limit, offset - параметры для пагинации
    sort - [{column_name => 'asc'|'desc'}, {column_name2  => 'asc'}, ...] - сортировка по списку полей

Example:
    perl -MDirect::AdditionsItemCallouts -MDDP -e 'p(Direct::AdditionsItemCallouts->get_by(get_banner_id => 1, banner_id => 1824065, sort => [{client_id => "asc"}, {id => "asc"}])->items)'

=cut

# опции доступные для get_by()
our @additions_item_callouts_valid_opt = qw/
    adgroup_id
    banner_id
    callout_text
    campaign_id
    client_id
    fill_in_linked
    get_banner_id
    id
    limit
    modified_since
    offset
    only_deleted
    shard
    sort
    status_moderate
    with_deleted
    with_mod_reasons
    except_client_ids
/;

my %_get_by_valid_opt = (map {$_ => 1} @additions_item_callouts_valid_opt);

sub get_by {
    my ($class, %opt) = @_;

    croak "get_by() opt is not valid" if grep {! $_get_by_valid_opt{$_}} keys %opt;

    my $shard;
    my %where;

    # по умолчанию возвращаем не удаленные (is_deleted == 0)
    if ($opt{only_deleted}) {
        $where{is_deleted} = 1;
    } elsif (! $opt{with_deleted}) {
        $where{is_deleted} = 0;
    }

    for my $search_by_field (grep {$opt{$_}} qw/id client_id status_moderate/) {
        my ($db_field) = Direct::Model::AdditionsItemCallout->get_db_columns_list('additions_item_callouts', [$search_by_field]);
        $where{$db_field} = $opt{$search_by_field};
    }
    $where{bid} = $opt{banner_id} if $opt{banner_id};
    $where{"b.cid"} = $opt{campaign_id} if $opt{campaign_id};
    $where{"b.pid"} = $opt{adgroup_id} if $opt{adgroup_id};

    if ($opt{callout_text}) {
        $where{callout_text} = $opt{callout_text};
        $where{hash} = Direct::Model::AdditionsItem::get_hash($opt{callout_text});
    }

    if ($opt{modified_since}) {
        $where{last_change__ge} = $opt{modified_since};
    }

    if ($opt{except_client_ids}) {
        my ($db_field) = Direct::Model::AdditionsItemCallout->get_db_columns_list('additions_item_callouts', ['client_id']);
        $where{$db_field.'__not_in'} = $opt{except_client_ids};
    }

    if ($opt{shard}) {
        $shard = PPC(shard => $opt{shard});
    } elsif ($opt{client_id}) {
        $shard = PPC(ClientID => $opt{client_id});
    } elsif ($opt{campaign_id}) {
        $shard = PPC(cid => $opt{campaign_id});
    } elsif ($opt{adgroup_id}) {
        $shard = PPC(pid => $opt{adgroup_id});
    } elsif ($opt{banner_id}) {
        $shard = PPC(bid => $opt{banner_id});
    } else {
        croak "Direct::AdditionsItemCallouts->get_by() client_id/campaign_id/banner_id/shard required";
    }

    my @fields = map {"additions_item_callouts.$_"} Direct::Model::AdditionsItemCallout->get_db_columns_list('additions_item_callouts');
    push @fields, 'banners_additions.bid as banner_id' if $opt{get_banner_id};
    my $fields = join(", ", @fields);

    my $join_sql = '';
    if ($opt{campaign_id} || $opt{adgroup_id}) {
        $join_sql = 'join banners_additions using(additions_item_id)
                     join banners b using(bid)
                    ';
    } elsif ($opt{get_banner_id} || $opt{banner_id}) {
        $join_sql = 'join banners_additions using(additions_item_id)';
    }

    my $limit_sql = '';
    if ($opt{limit}) {
        $limit_sql = " limit " . int($opt{limit}) . " ";
        $limit_sql .= " offset " . int($opt{offset}) . " " if $opt{offset};
    }

    my @order_clauses;
    if ($opt{sort}) {
        for my $sort_part (@{$opt{sort}}) {
            my ($field, $direction) = (%$sort_part); # (key, value)
            my ($db_field) = Direct::Model::AdditionsItemCallout->get_db_columns_list('additions_item_callouts', [$field]);
            push @order_clauses, sql_quote_identifier($db_field) . " " . ($direction eq 'asc' ? 'asc' : 'desc');
        }
    }

    # при запросе по баннерам сортируем в правильной последовательности
    if ($opt{get_banner_id} || $opt{banner_id}) {
        push @order_clauses, 'banners_additions.bid', 'banners_additions.sequence_num asc';
    }

    my $sort_sql = @order_clauses ? "order by " . join(",", @order_clauses) : '';


    my $rows = get_all_sql($shard, [
        "select $fields
         from additions_item_callouts
         $join_sql
        ",
        where => \%where,
        $sort_sql,
        $limit_sql,
    ]);

    my $items = Direct::Model::AdditionsItemCallout->from_db_hash_multi($rows);
   
    if ($opt{with_mod_reasons}) {
        my %item_by_id = map { $_->id => $_ } @$items;

        my @ids = keys %item_by_id;

        my $reasons = mass_get_banner_additions_diags($shard, 'callout', \@ids);
        for my $id (@ids) {
            $item_by_id{ $id }->moderation_reasons( $reasons->{ $id } // [] );
        }
    }

    if ($opt{fill_in_linked} && @$items) {
        my $additions_item_ids = [uniq map {$_->id} @$items];
        my $link_data = get_hash_sql($shard, [
            "select additions_item_id, 1
             from banners_additions
             join banners using(bid)
             join campaigns c using(cid)
            ", where => {
                additions_item_id => $additions_item_ids
                , "c.statusEmpty" => "No"
            },
            "group by additions_item_id"
        ]);

        for my $item (@$items) {
            $item->is_linked($link_data->{$item->id} ? 1 : 0);
        }
    }

    my $result = $class->new(items => $items);

    return $result;
}

=head2 items_by

    Вернуть сгруппированный по ключу хеш
    при группировке по banner_id:
        - в исходном объекте должен быть banner_id, например полученный через get_by(... get_banner_id => 1 ...)
        - не привязанные баннеры не возвращаются

Example:
    perl -MDirect::AdditionsItemCallouts -MDDP -e 'p(Direct::AdditionsItemCallouts->get_by(get_banner_id => 1, banner_id => 1824065, sort => [{client_id => "asc"}, {id => "asc"}])->items_by("banner_id"))'

=cut

sub items_by {
    my ($self, $key) = @_;

    $key //= 'banner_id';
    croak "by `banner_id, client_id` only supported" unless $key =~ /^(?:banner_id|client_id)$/;

    my %result;
    if ($key eq 'banner_id') {
        for my $item (@{$self->items}) {
            next unless $item->banner_id;
            push @{$result{$item->banner_id}}, $item;
        }
    } elsif ($key eq 'client_id') {
        for my $item (@{$self->items}) {
            push @{$result{$item->client_id}}, $item;
        }
    }

    return \%result;
}

=head2 save

Сохранение списка уточнений

    Direct::AdditionsItemCallouts->new(items => [Direct::Model::AdditionsItemCallout->new(client_id => 123, callout_text => "text")])->save();

=cut

sub save {
    my ($self) = @_;

    my ($updates, $creates) = part {($_->has_id && $_->id) ? 0 : 1} @{$self->items};
    Direct::Model::AdditionsItemCallout::Manager->new(items => $creates)->create() if $creates;
    Direct::Model::AdditionsItemCallout::Manager->new(items => $updates)->update() if $updates;
}

1;
