
package Direct::Banners;

use Mouse;
with qw/Direct::Role::Copyable/;

use Direct::Modern;

use Settings;

use Direct::Model::Banner;
use Direct::Model::BannerText;
use Direct::Model::BannerDynamic;
use Direct::Model::BannerMobileContent;
use Direct::Model::BannerMcbanner;
use Direct::Model::BannerPerformance;
use Direct::Model::BannerCpmBanner;
use Direct::Model::BannerCpcVideo;
use Direct::Model::BannerContentPromotion;
use Direct::Model::BannerCpmOutdoor;
use Direct::Model::BannerCpmIndoor;
use Direct::Model::BannerCpmAudio;
use Direct::Model::BannerImageAd;
use Direct::Model::ImageFormat;
use Direct::Model::CanvasCreative;
use Direct::Model::Creative;
use Direct::Model::Creative::Factory;
use Direct::Model::VCard;
use Direct::Model::BannerImage;
use Direct::Model::Sitelink;
use Direct::Model::SitelinksSet;
use Direct::Model::VideoAddition;

use Direct::Model::Banner::Manager;
use Direct::BannersPixels;

use BS::CheckUrlAvailability;

use Yandex::DBShards;
use Yandex::DBTools;
use Yandex::ScalarUtils qw/str/;
use Yandex::I18n;
use Yandex::TimeCommon qw/unix2mysql/;
use Yandex::IDN qw/idn_to_unicode/;
use List::MoreUtils qw/any all uniq pairwise/;

use ModerateChecks qw//;
use RedirectCheckQueue qw//;
use GeoTools qw//;
use PrimitivesIds qw/get_clientid/;
use Tools qw/log_cmd make_copy_sql_strings/;
use LogTools qw//;
use TextTools qw/html2string/;

use Sitelinks qw//;
use VCards qw//;
use MailNotification qw/mass_mail_notification/;

use Exporter qw/import/;
our @EXPORT_OK = qw/
    get_banners
/;

has 'items' => (is => 'ro', isa => 'ArrayRef[Direct::Model::Banner]');
has 'total' => (is => 'ro', isa => 'Int');
has 'data'  => (is => 'ro', isa => 'HashRef', init_arg => undef, lazy => 1, default => sub { +{}; });

around BUILDARGS => sub { my ($orig, $class) = (shift, shift); $class->$orig(@_ == 1 ? (items => $_[0]) : @_) };

=head2 manager_class
=head2 manager
=cut

sub manager_class { 'Direct::Model::Banner::Manager' }
sub manager { $_[0]->manager_class->new(items => $_[0]->items) }

=head2 WEB_FIELD_NAMES
=cut

sub WEB_FIELD_NAMES {(
    title  => {field => sprintf('"%s"', iget('Заголовок 1'))},
    title_extension  => {field => sprintf('"%s"', iget('Заголовок 2'))},
    body   => {field => sprintf('"%s"', iget('Текст объявления'))},
    href   => {field => sprintf('"%s"', iget('Основная ссылка объявления'))},
    domain => {field => sprintf('"%s"', iget('Отображаемый домен объявления'))},
    hash   => {field => sprintf('"%s"', 'hash')},
    image  => {field => iget('"Изображение"')},
)}

=head2 WEB_MOBILE_CONTENT_FIELD_NAMES

Имена полей, специфичных для баннеров РМП

=cut

sub WEB_MOBILE_CONTENT_FIELD_NAMES {(
    title  => {field => sprintf('"%s"', iget('Заголовок'))},
    body   => {field => sprintf('"%s"', iget('Текст объявления'))},
)}

=head2 BANNER_TYPE_TO_BL_CLASS

Ссылка на хеш соответствий 
    <тип баннера> => <наименование класса бизнес-логики>

=cut

sub BANNER_TYPE_TO_BL_CLASS {
    return {        
        text => 'Direct::Banners::Text',
        dynamic => 'Direct::Banners::Dynamic',
        mobile_content => 'Direct::Banners::MobileContent',
        performance => 'Direct::Banners::Performance',
        image_ad => 'Direct::Banners::ImageAd',
        mcbanner => 'Direct::Banners::Mcbanner',
        cpm_banner => 'Direct::Banners::CpmBanner',
        cpc_video => 'Direct::Banners::CpcVideo',
    }
}


=head2 get_business_logic_class

Получаем класс бизнес-логики по типу баннера
Класс уже загружен

=cut

sub get_business_logic_class {
    my ($type) = @_;

    my $class = BANNER_TYPE_TO_BL_CLASS->{$type};
    croak "No class fot type <$type>"  if !$class;

    require (($class =~ s!::!/!gr ) . ".pm");
    return $class;
}


=head2 get_banners($banner_ids, %options)

Deprecated функции, не рекомендуются к использованию, т.к. могут не учитывать особенности некоторых типов баннеров.
Лучше заменить их на вызовы соответствующих функций для каждого типа баннеров

=cut

sub get_banners { my ($banner_ids, %options) = @_; __PACKAGE__->get($banner_ids, %options)->items }

=head2 get($banner_ids, %options)

По заданным идентификаторам ($banner_ids) возвращает instance с выбранными баннерами.
Поддерживаемые %options см. в get_by.

=cut

sub get {
    my ($class, $banner_ids, %options) = @_;
    return $class->get_by(banner_id => $banner_ids, %options);
}

=head2 get_by($key, $vals, %options)

По заданному критерию возвращает instance с выбранными баннерами.

Параметры:
    $key -> по какому ключу выбирать баннеры: banner_id/adgroup_id/campaign_id
    $vals -> (Int|ArrayRef[Int]); список идентификаторов
    %options:
        banner_type    -> (Str|ArrayRef[Str]); баннеры каких типов выбирать: text/dynamic/mobile_content/etc
        limit/offset   -> параметры для постраничной выборки
        total_count    -> при использовании limit/offset также вычислять общее количество элементов
        with_roles     -> (Str|ArrayRef[Str]); список ролей, которые нужно применить к баннерам
        with_vcard     -> выбрать визитку
        with_image     -> выбрать картинку
        with_sitelinks -> выбрать сайтлинки
        with_additions_callouts -> выбрать дополнения баннеров: уточнения
        with_video_resources -> выбрать видео ресурсы
        only_with_banner_additions -> ArrayRef[id1, id2, ...]; выбрать баннеры только с уточнениями (по списку additions_item_id)
        skip_canvas_image_ad -> для баннеров image_ad отбирает только баннеры с картинокой
        filter         -> (HashRef); дополнительный фильтр

    Дополнительный фильтр:
        Применимы любые поля из таблиц баннеров. Если префикс таблицы отсутствует, будет подставлен умолчательный `b.`

=cut

sub get_by {
    my ($class, $key, $vals, %options) = @_;

    croak "only `banner_id`/`adgroup_id`/`campaign_id` keys are supported" unless $key =~ /^(?:banner|adgroup|campaign)_id$/;

    $vals = [$vals // ()] if ref($vals) ne 'ARRAY';
    return $class->new(items => []) if !@$vals;

    state $supported_banner_types = {
        text => 1, dynamic => 1, mobile_content => 1, performance => 1, image_ad => 1, mcbanner => 1, cpm_banner => 1,
        cpc_video => 1, performance_main => 1
    };

    if (defined $options{banner_type}) {
        $options{banner_type} = [$options{banner_type}] unless ref($options{banner_type}) eq 'ARRAY';
        croak "found unsupported banner_type in options" unless all { $supported_banner_types->{$_} } @{$options{banner_type}};
    }
    $options{banner_type} //= [keys %$supported_banner_types];

    # Хеш с типам баннеров, которые следует выбирать
    my $with_banner_type = {map { $_ => 1 } @{$options{banner_type}}};

    my (@select_columns, @from_tables);

    push @select_columns,
        Direct::Model::Banner->get_db_columns(banners => 'b', prefix => 'b_'),
        'IFNULL(fd.filter_domain, b.domain) AS b_filter_domain',
        'u.ClientID AS b_client_id';

    push @from_tables,
        'banners b',
        'LEFT JOIN filter_domain fd ON (fd.domain = b.domain)',
        'JOIN campaigns c ON (c.cid = b.cid)',
        'JOIN users u ON (u.uid = c.uid)';
    
    push @select_columns, Direct::Model::Banner->get_db_columns(banners_minus_geo => 'bmg', prefix => 'b_');
    push @from_tables, 'LEFT JOIN banners_minus_geo bmg on (b.bid = bmg.bid and bmg.type = "current")';

    if ($options{with_permalinks}) {
        push @select_columns, Direct::Model::BannerText->get_db_columns(banner_permalinks => 'bpml', prefix => 'b_');
        push @from_tables, 'LEFT JOIN banner_permalinks bpml on (b.bid = bpml.bid and bpml.permalink_assign_type = "manual")';
    }
 
    # дисклеймер и эксперимент
    push @select_columns, 'aid.disclaimer_text AS b_disclaimer';
    # эксперимент
    push @select_columns, 'aie.experiment_json AS b_experiment';
    push @from_tables,
        'LEFT JOIN banners_additions bad ON (bad.bid=b.bid AND bad.additions_type IN ("disclaimer", "experiment"))',
        'LEFT JOIN additions_item_disclaimers aid ON (aid.additions_item_id = bad.additions_item_id)',
        'LEFT JOIN additions_item_experiments aie ON (aie.additions_item_id = bad.additions_item_id)';

    if ($with_banner_type->{text}) {
        push @select_columns, Direct::Model::Banner->get_db_columns(banner_display_hrefs => 'bdh', prefix => 'b_');
        push @from_tables, 'LEFT JOIN banner_display_hrefs bdh ON (b.bid = bdh.bid)';
    }

    if ($with_banner_type->{mobile_content}) {
        push @select_columns, Direct::Model::BannerMobileContent->get_db_columns(banners_mobile_content => 'bmc', prefix => 'b_');
        push @from_tables, 'LEFT JOIN banners_mobile_content bmc ON (b.bid = bmc.bid)';
    }

    if ($with_banner_type->{performance} || $options{with_video_resources} || ($with_banner_type->{image_ad} && !$options{skip_canvas_image_ad})
        || $with_banner_type->{cpm_banner} || $with_banner_type->{cpc_video}) {
        push @select_columns,
            Direct::Model::BannerCreative->get_db_columns(banners_performance => 'bp', prefix => 'bp_'),
            Direct::Model::BannerPerformance->get_db_columns(banners_performance => 'bp', prefix => 'b_'),
            Direct::Model::Creative->get_db_columns(perf_creatives => 'perfc', prefix => 'perfc_');
        push @from_tables,
            'LEFT JOIN banners_performance bp ON (b.bid = bp.bid)',
            'LEFT JOIN perf_creatives perfc ON perfc.creative_id = bp.creative_id';
    }

    if ($with_banner_type->{image_ad} || $with_banner_type->{mcbanner}) {
        push @select_columns, Direct::Model::Image->get_db_columns(images => 'im', prefix => 'im_');
        push @select_columns, Direct::Model::ImageFormat->get_db_columns(banner_images_formats => 'imf', prefix => 'imf_');
        
        my @join_banner_types;  # не джойним лишние типы
        if ($with_banner_type->{image_ad}) {
            push @join_banner_types, 'image_ad';
        }
        if ($with_banner_type->{mcbanner}) {
            push @join_banner_types, 'mcbanner';
        }
        push @from_tables, q/LEFT JOIN images im on im.bid = if(/ . sql_condition({q/b.banner_type/ => \@join_banner_types}) . q/, b.bid, NULL)/,
                 q/LEFT JOIN banner_images_formats imf on imf.image_hash = im.image_hash/;
    }

    if ($options{with_vcard}) {
        push @select_columns,
            Direct::Model::VCard->get_db_columns(vcards => 'vc', prefix => 'vc_'),
            Direct::Model::VCard->get_db_columns(org_details => 'orgd', prefix => 'vc_'),
            Direct::Model::VCard->get_db_columns(addresses => 'addr', prefix => 'vc_'),
            "IF(map.mid IS NOT NULL, CONCAT_WS(',', map.x, map.y), NULL) AS vc_manual_point",
            "IF(map.mid IS NOT NULL, CONCAT_WS(',', map.x1, map.y1, map.x2, map.y2), NULL) AS vc_manual_bounds",
            "IF(map_auto.mid IS NOT NULL, CONCAT_WS(',', map_auto.x, map_auto.y), NULL) AS vc_auto_point",
            "IF(map_auto.mid IS NOT NULL, CONCAT_WS(',', map_auto.x1, map_auto.y1, map_auto.x2, map_auto.y2), NULL) AS vc_auto_bounds";

        push @from_tables,
            'LEFT JOIN vcards vc ON (vc.vcard_id = b.vcard_id)',
            'LEFT JOIN org_details orgd ON (orgd.org_details_id = vc.org_details_id)',
            'LEFT JOIN addresses addr ON (addr.aid = vc.address_id)',
            'LEFT JOIN maps map ON (map.mid = addr.map_id)',
            'LEFT JOIN maps map_auto ON (map_auto.mid = addr.map_id_auto)';
    }
    
    if ($options{with_turbolanding}) {
        #турболендинг
        push @select_columns, Direct::Model::TurboLanding::Banner->get_db_columns( banner_turbolandings => 'btl', prefix => 'tl_');
        push @from_tables, 'LEFT JOIN banner_turbolandings btl on (b.bid = btl.bid and b.cid = btl.cid)';
        push @select_columns, Direct::Model::TurboLanding::Banner->get_db_columns( turbolandings => 'tl', prefix => 'tl_');
        push @from_tables, 'LEFT JOIN turbolandings tl on (btl.tl_id = tl.tl_id)';
        push @select_columns, Direct::Model::Banner->get_db_columns(banner_turbolanding_params => 'btlp', prefix => 'b_');
        push @from_tables, 'LEFT JOIN banner_turbolanding_params btlp on (b.bid = btlp.bid)';
    }

    # Проверяем остановку объявления по мониторингу (от БК)
    my $calc_status_metrika_stop_value_sql = "IF(co.statusMetricaControl='Yes' AND bs_dead_domains.domain_id IS NOT NULL, 'Yes', 'No')";
    push @select_columns, "$calc_status_metrika_stop_value_sql b_statusMetricaStop";
    push @from_tables, BS::CheckUrlAvailability::off_by_monitoring_domain_id_joins_sql('b'),
                       "LEFT JOIN camp_options co ON c.cid=co.cid";

    if ($options{with_image}) {
        push @select_columns,
            Direct::Model::Banner->get_db_columns(banner_images => 'bim', prefix => 'b_'),
            Direct::Model::BannerImage->get_db_columns(banner_images => 'bim', prefix => 'bim_'),
            Direct::Model::ImageFormat->get_db_columns(banner_images_formats => 'bimf', prefix => 'bimf_');

        push @from_tables, 'LEFT JOIN banner_images bim ON (bim.bid = b.bid AND bim.statusShow = "Yes")';
        push @from_tables, 'LEFT JOIN banner_images_formats bimf ON (bimf.image_hash = bim.image_hash)';
    }

    if ($options{with_sitelinks}) {
        push @select_columns, Direct::Model::SitelinksSet->get_db_columns(sitelinks_sets => 'sls', prefix => 'sls_');
        push @from_tables, 'LEFT JOIN sitelinks_sets sls ON (sls.sitelinks_set_id = b.sitelinks_set_id)';
    }

    my @group_by;
    my %custom_where;

    if ($options{only_with_banner_additions}) {
        push @from_tables, 'JOIN banners_additions ON (banners_additions.bid = b.bid)';
        $custom_where{"banners_additions.additions_item_id"} = $options{only_with_banner_additions};
        push @group_by, "b.bid";
    }

    my $calc_found_rows = $options{limit} && $options{total_count} ? 'SQL_CALC_FOUND_ROWS' : '';
    my %shard_selector = (banner_id => 'bid', adgroup_id => 'pid', campaign_id => 'cid');

    my @plain_conditions;
    for my $field (keys %{$options{filter} // {}}) {
        if ($field eq 'status_metrika_stop') {
            $custom_where{_TEXT} = $calc_status_metrika_stop_value_sql." = ".sql_quote($options{filter}->{$field});
        } else {
            $custom_where{$field =~ /^\w+\./ ? $field : 'b.'.$field} = $options{filter}->{$field};
        }
    }
    if ($with_banner_type->{image_ad} && $options{skip_canvas_image_ad}) {
        push @plain_conditions, "IF(b.banner_type = 'image_ad', im.image_id IS NOT NULL, 1 = 1)";
    }

    my $banner_rows = get_all_sql(PPC($shard_selector{$key} => $vals), [
        sprintf("SELECT $calc_found_rows %s FROM %s", join(', ', @select_columns), join(' ', @from_tables)),
        where => {
            'b.'.$shard_selector{$key} => SHARD_IDS,
            'b.banner_type' => $options{banner_type},
            %custom_where,
        },
        (@plain_conditions ? (" AND " . join " AND ", @plain_conditions) : ()),
        (@group_by ? ("GROUP BY " . join(", ", @group_by)) : ()),
        $options{limit} ? (
            'ORDER BY b.bid',
            limit => $options{limit}, $options{offset} ? (offset => $options{offset}) : (),
        ) : (),
    ]);

    my $found_rows = $calc_found_rows ? select_found_rows(PPC($shard_selector{$key} => $vals)) : undef;
    my $self = $class->new(items => [], $calc_found_rows ? (total => $found_rows) : ());

    return $self unless @$banner_rows;

    my $_cache;
    # отдельный кеш для ImageFormat ГО-картинок, иначе пытаемся собирать по неправильным полям
    my $_alt_cache;
    for my $row (@$banner_rows) {
        my $banner;
        my $banner_creative;
        if ($row->{b_banner_type} =~ /text|mobile_content/ && $row->{bp_bid}) {
            $banner_creative = Direct::Model::BannerCreative->from_db_hash($row, \$_cache, prefix => 'bp_');
            my $creative = Direct::Model::Creative::Factory->create($row, \$_cache, prefix => 'perfc_');
            $banner_creative->creative($creative);
        }

        if ($row->{b_banner_type} eq 'text') {
            $banner = Direct::Model::BannerText->from_db_hash($row, \$_cache, prefix => 'b_', with => $options{with_roles});
            if ($banner_creative) {
                $banner->creative($banner_creative);
            }
        } elsif ($row->{b_banner_type} eq 'dynamic') {
            $banner = Direct::Model::BannerDynamic->from_db_hash($row, \$_cache, prefix => 'b_', with => $options{with_roles});
        } elsif ($row->{b_banner_type} eq 'mobile_content') {
            $banner = Direct::Model::BannerMobileContent->from_db_hash($row, \$_cache, prefix => 'b_', with => $options{with_roles});
            if ($banner_creative) {
                $banner->creative($banner_creative);
            }
        } elsif ($row->{b_banner_type} eq 'performance') {
            $banner = Direct::Model::BannerPerformance->from_db_hash($row, \$_cache, prefix => 'b_', with => $options{with_roles});
            $banner->creative(Direct::Model::Creative->from_db_hash($row, \$_cache, prefix => 'perfc_'));
        } elsif ($row->{b_banner_type} eq 'performance_main') {
            # Костыли для того, чтобы моделька создалась (чтобы не пришлось добавлять полноценный новый тип баннера в перл)
            $row->{b_banner_creative_id} = 0;
            $row->{b_creative_id} = 0;
            $row->{b_bc_statusModerate} = 'Yes';
            $banner = Direct::Model::BannerPerformance->from_db_hash($row, \$_cache, prefix => 'b_', with => $options{with_roles});
        } elsif ($row->{b_banner_type} eq 'image_ad' || $row->{b_banner_type} eq 'mcbanner'
            || $row->{b_banner_type} eq 'cpm_banner' || $row->{b_banner_type} eq 'cpc_video') {
            if ($row->{b_banner_type} eq 'image_ad') {
                $banner = Direct::Model::BannerImageAd->from_db_hash($row, \$_cache, prefix => 'b_', with => $options{with_roles});
            } elsif ($row->{b_banner_type} eq 'mcbanner') {
                $banner = Direct::Model::BannerMcbanner->from_db_hash($row, \$_cache, prefix => 'b_', with => $options{with_roles});
            } elsif ($row->{b_banner_type} eq 'cpm_banner') {
                $banner = Direct::Model::BannerCpmBanner->from_db_hash($row, \$_cache, prefix => 'b_', with => $options{with_roles});
            } elsif ($row->{b_banner_type} eq 'cpc_video') {
                $banner = Direct::Model::BannerCpcVideo->from_db_hash($row, \$_cache, prefix => 'b_', with => $options{with_roles});
            }
            if ($row->{im_image_id}) {
                # imagead/picture
                my $image_ad = Direct::Model::Image->from_db_hash($row, \$_cache, prefix => 'im_');
                my $image_format = Direct::Model::ImageFormat->from_db_hash($row, \$_cache, prefix => 'imf_');
                $image_ad->hash($image_format->hash);
                $image_ad->format($image_format);
                $banner->image_ad($image_ad);
            } else {
                # canvas-creative
                my $banner_creative = Direct::Model::BannerCreative->from_db_hash($row, \$_cache, prefix => 'bp_');
                my $canvas_creative = Direct::Model::Creative::Factory->create($row, \$_cache, prefix => 'perfc_');
                $banner_creative->creative_id($canvas_creative->id);
                $banner_creative->creative($canvas_creative);
                $banner->creative($banner_creative);
            }
        } else {
            croak "fetched unsupported banner type: ".$row->{b_banner_type};
        }

        if ($options{with_vcard} && $row->{vc_vcard_id}) {
            $banner->vcard(Direct::Model::VCard->from_db_hash($row, \$_cache, prefix => 'vc_'));
        }
        
        if ($options{with_turbolanding} && $row->{tl_tl_id}) {
            $banner->turbolanding(Direct::Model::TurboLanding::Banner->from_db_hash($row, \$_cache, prefix => 'tl_'));
        }
        
        if ($options{with_image} && $row->{bim_image_id} && $row->{b_banner_type} ne 'image_ad') {
            my $image = Direct::Model::BannerImage->from_db_hash($row, \$_cache, prefix => 'bim_');
            # workaround для DIRECT-62806, сделать бесуловным выполнение кода (удалить if-обертку)
            # после применения миграции из DIRECT-62807
            if (defined $row->{bimf_image_hash}) {
                # используем альтернативный кеш!
                $image->format(Direct::Model::ImageFormat->from_db_hash($row, \$_alt_cache, prefix => 'bimf_'));
            }
            $banner->image($image);
        }
        if ($options{with_sitelinks} && $row->{sls_sitelinks_set_id}) {
            $banner->sitelinks_set(Direct::Model::SitelinksSet->from_db_hash($row, \$_cache, prefix => 'sls_'));
        }

        push @{$self->items}, $banner;
    }

    if ($options{with_sitelinks}) {
        my @banners_with_sitelinks_set = grep { $_->has_sitelinks_set } @{$self->items};

        my $sitelinks_by_set_id = Sitelinks::get_sitelinks_by_set_id_multi([map { $_->sitelinks_set_id } @banners_with_sitelinks_set]);
        foreach my $banner (@banners_with_sitelinks_set) {
            my @links;
            foreach my $sl_row (@{$sitelinks_by_set_id->{$banner->sitelinks_set_id} // []}){
                my $sl = Direct::Model::Sitelink->from_db_hash($sl_row, \$_cache);
                if ($sl_row->{turbolanding}) {
                    #Если есть лендинг - добавляем связку
                    $sl->turbolanding( Direct::Model::TurboLanding::Sitelink->from_db_hash($sl_row->{turbolanding}, \$_cache) );
                }
                push @links, $sl;
            }
            $banner->sitelinks_set->links(\@links);
        }
    }

    if ($options{with_additions_callouts}) {
        my $callouts = Direct::BannersAdditions->get_by(
            banner_id => [map {$_->id} @{$self->items}],
            additions_type => 'callout',
            get_banner_id => 1,
        )->items_callouts_by('banner_id');

        for my $banner (@{$self->items}) {
            $banner->additions_callouts($callouts->{$banner->id} // []);
        }
    }

    if ($options{with_pixels}) {
        my $pixels = Direct::BannersPixels->get_by(banner_id => [map {$_->id} @{$self->items}])->items_by();
        foreach my $banner (@{$self->items}) {
            $banner->pixels($pixels->{$banner->id} // []);
        }
    }

    return $self;
}

=head2 items_by($key)

Возвращает структуру с баннерами, вида:
    $key //eq 'id' => {$banner1->id => $banner1, $banner2->id => $banner2, ...};
    $key eq 'gid'  => {$adgroup_id1 => [$banner1, $banner2, ...], $adgroup_id2 => [$banner3, ...], ...};

=cut

sub items_by {
    my ($self, $key) = @_;

    $key //= 'id';
    croak "by `id`/`adgroup_id(gid)`/`banner_type` only supported" unless $key =~ /^(?:id|gid|adgroup_id|banner_type)$/;

    my %result;
    if ($key eq 'id') {
        $result{$_->id} = $_ for @{$self->items};
    } elsif ($key eq 'banner_type') {
        push @{$result{ $_->banner_type }}, $_ for @{$self->items};
    } else {
        push @{$result{ $_->adgroup_id }}, $_ for @{$self->items};
    }

    return \%result;
}

=head2 get_lang_banners($key => $vals)

    Выбрать баннеры только для определения языка

=cut

sub get_lang_banners {
    my ($class, $key, $vals) = @_;

    croak "only `banner_id`/`adgroup_id`/`campaign_id` keys are supported" unless $key =~ /^(?:banner|adgroup|campaign)_id$/;

    my %shard_selector = (banner_id => 'bid', adgroup_id => 'pid', campaign_id => 'cid');
    my $rows = get_all_sql(PPC($shard_selector{$key} => $vals),
                ["SELECT b.cid, b.title, b.title_extension, b.body, b.language, b.banner_type, pc.creative_id, pc.moderate_info, c.ClientID AS client_id
                 FROM banners b
                 LEFT JOIN campaigns c ON b.cid = c.cid
                 LEFT JOIN banners_performance bp ON bp.bid = IF(b.banner_type in ('image_ad', 'cpm_banner', 'cpc_video'), b.bid, NULL)
                 LEFT JOIN perf_creatives pc ON pc.creative_id = bp.creative_id",
                 WHERE => {
                     'b.'.$shard_selector{$key} => SHARD_IDS
                 }]);

    my ($_cache, @items);
    for my $row (@$rows) {
        my $banner;
        if ($row->{banner_type} eq 'text') {
            $banner = Direct::Model::BannerText->from_db_hash($row, \$_cache);
        } elsif ($row->{banner_type} eq 'dynamic') {
            $banner = Direct::Model::BannerDynamic->from_db_hash($row, \$_cache);
        } elsif ($row->{banner_type} eq 'mobile_content') {
            $banner = Direct::Model::BannerMobileContent->from_db_hash($row, \$_cache);
        } elsif ($row->{banner_type} eq 'performance') {
            $banner = Direct::Model::BannerPerformance->from_db_hash($row, \$_cache);
        } elsif ($row->{banner_type} eq 'cpc_video') {
            $banner = Direct::Model::BannerCpcVideo->from_db_hash($row, \$_cache);
        } elsif ($row->{banner_type} eq 'image_ad') {
            $banner = Direct::Model::BannerImageAd->from_db_hash($row, \$_cache);
            if ($row->{creative_id}) {
                my $creative = Direct::Model::BannerCreative->new();
                $creative->creative(Direct::Model::CanvasCreative->new(_moderate_info => $row->{moderate_info}));
                $banner->creative($creative);
            }
        } elsif ($row->{banner_type} eq 'mcbanner') {
            $banner = Direct::Model::BannerMcbanner->from_db_hash($row, \$_cache);
        } elsif ($row->{banner_type} eq 'cpm_banner') {
            $banner = Direct::Model::BannerCpmBanner->from_db_hash($row, \$_cache);
            my $creative = Direct::Model::BannerCreative->new();
            $creative->creative(Direct::Model::CanvasCreative->new(_moderate_info => $row->{moderate_info}));
            $banner->creative($creative);
        } elsif ($row->{banner_type} eq 'cpc_video') {
            $banner = Direct::Model::BannerCpcVideo->from_db_hash($row, \$_cache);
            my $creative = Direct::Model::BannerCreative->new();
            $creative->creative(Direct::Model::VideoAddition->new(_moderate_info => $row->{moderate_info}));
            $banner->creative($creative);
        } elsif ($row->{banner_type} eq 'cpm_outdoor') {
            $banner = Direct::Model::BannerCpmOutdoor->from_db_hash($row, \$_cache);
            my $creative = Direct::Model::BannerCreative->new();
            $creative->creative(Direct::Model::VideoAddition->new(_moderate_info => $row->{moderate_info}));
            $banner->creative($creative);
        } elsif ($row->{banner_type} eq 'cpm_indoor') {
            $banner = Direct::Model::BannerCpmIndoor->from_db_hash($row, \$_cache);
            my $creative = Direct::Model::BannerCreative->new();
            $creative->creative(Direct::Model::VideoAddition->new(_moderate_info => $row->{moderate_info}));
            $banner->creative($creative);
        } elsif ($row->{banner_type} eq 'content_promotion') {
            $banner = Direct::Model::BannerContentPromotion->from_db_hash($row, \$_cache);
        } elsif ($row->{banner_type} eq 'cpm_audio') {
            $banner = Direct::Model::BannerCpmAudio->from_db_hash($row, \$_cache);
            my $creative = Direct::Model::BannerCreative->new();
            $creative->creative(Direct::Model::VideoAddition->new(_moderate_info => $row->{moderate_info})); # NEED TO FIX
            $banner->creative($creative);
        } else {
            croak "fetched unsupported banner type: ".$row->{banner_type};
        }
        push @items, $banner;
    }
    return $class->new(items => \@items);
}

=head2 set_client_id_by_uid($uid)

Установка client_id (если не задан) по $uid

=cut

sub set_client_id_by_uid {
    my ($self, $uid) = @_;

    if (any { !$_->has_client_id || !$_->client_id } @{$self->items}) {
        my $client_id = get_clientid(uid => $uid);
        $_->client_id($client_id) for @{$self->items};
    }

    return $self;
}

=head2 prepare_create($uid, %options)

Подготовка списка баннеров к созданию для пользователя $uid.

Параметры:
    $uid -> пользователь, для которого создаются баннеры
    %options:
        translocal_opt -> параметры транслокальности (по умолчанию: {ClientID => $banner->client_id})

=cut

sub prepare_create {
    my ($self, $uid, %options) = @_;

    my $ids = get_new_id_multi(bid => scalar(@{$self->items}), uid => $uid);

    for my $banner (@{$self->items}) {
        $banner->id(shift @$ids);
        $banner->bs_banner_id(0);
        if ($banner->is_image_supported && $banner->has_image) {
            $banner->image->bs_banner_id(0);
            $banner->image->bs_priority_id(0);
        }

        if (defined $banner->href) {
            # Если есть phraseid на баннере, то проставим соответствующий параметр у группы.
            $banner->do_set_adgroup_has_phraseid_href(1) if $banner->has_phraseid_param;

            my ($domain_target, $domain_check_redirect) = RedirectCheckQueue::domain_need_check_redirect({href => $banner->href});
            $banner->do_check_redirect(1) if $domain_check_redirect;

            # Отображаемый домен можно задать отличный от href
            $banner->domain($domain_target) if !$banner->has_domain || !$banner->domain;
            $banner->domain(Yandex::IDN::idn_to_unicode($banner->domain));

            $banner->do_update_filter_domain(1);
            $banner->do_update_aggregator_domain(1);

            $banner->do_bs_sync_adgroup(1) if $banner->has_coef_goal_context_id_param;
        } else {
            $banner->domain(undef);
        }

        $banner->status_moderate('New') unless $banner->has_status_moderate;

        # В группе без условий показа баннер можен быть только черновиком
        croak "Cannot create non-draft banner in AdGroup without show conditions" if !$banner->adgroup->has_show_conditions && (
                $banner->status_moderate ne 'New' && !($banner->adgroup->status_moderate eq 'New' && $banner->status_moderate eq 'Ready'));

        if ($banner->status_moderate eq 'Ready' && $banner->adgroup->status_moderate eq 'New') {
            $banner->do_moderate_adgroup(1);
            $banner->do_moderate_campaign_if_no_active_banners(1);
        }
        $banner->status_post_moderate('No');

        $banner->status_bs_synced('No');

        my $geoflag;
        my $translocal_opt = $options{translocal_opt} // {ClientID => $banner->client_id};
        GeoTools::refine_geoid($banner->adgroup->geo, \$geoflag, $translocal_opt);
        $banner->geoflag($geoflag) if defined $geoflag;

        # При копировании баннера копируются и флаги (предупреждения), которые отправляются в модерацию через MagicQueue
        $banner->do_push_flags_to_moderation(1) if $banner->_has_flags && defined $banner->_flags;
    }

    return $self;
}

=head2 prepare_update

Подготовка списка баннеров к обновлению. Баннеры должны быть созданы с ролью `Update`.

=cut

sub prepare_update {
    my ($self) = @_;

    for my $banner (@{$self->items}) {
        my $domain_check_redirect;
        if ($banner->is_href_supported) {
            if (defined $banner->href) {
                (my $domain_target, $domain_check_redirect) = RedirectCheckQueue::domain_need_check_redirect(
                    {href => $banner->href},
                    {href => $banner->old->href, domain => $banner->old->domain}
                );
                # Отображаемый домен можно задать отличный от href
                $banner->domain($domain_target) if !$banner->has_domain || !$banner->domain;
                $banner->domain(Yandex::IDN::idn_to_unicode($banner->domain));
            } else {
                $banner->domain(undef);
            }
            my ($old_domain, $new_domain) = ($banner->old->domain, $banner->domain);
            if ((defined($old_domain) && defined($new_domain) && $old_domain eq $new_domain) || 
               (!defined($old_domain) && !defined($new_domain))) {
                $banner->is_domain_changed(0);
                $banner->is_reverse_domain_changed(0);
            }
        }
        next if !$banner->is_changed;

        croak "Cannot change banner type" if $banner->is_banner_type_changed;

        if ($banner->is_href_supported) {
            if (defined $banner->href) {
                # Если появился phraseid (т.е. до изменений его небыло) на баннере, то проставим соответствующий параметр у группы.
                $banner->do_set_adgroup_has_phraseid_href(1) if $banner->has_phraseid_param && !$banner->old->has_phraseid_param;

                $banner->do_check_redirect(1) if $domain_check_redirect;
                $banner->do_update_filter_domain(1) if $banner->is_domain_changed;

                $banner->do_bs_sync_adgroup(1) if $banner->has_coef_goal_context_id_param && !$banner->old->has_coef_goal_context_id_param;
            }
            $banner->do_update_aggregator_domain(1) if $banner->is_href_changed;
        }

        my $banner_has_changed = $banner->is_title_changed || $banner->is_title_extension_changed || $banner->is_body_changed || $banner->is_href_changed || $banner->is_domain_changed || $banner->is_flags_changed;
        
        my $need_moderate_banner = ModerateChecks::check_moderate_banner(
            {title => $banner->title, title_extension => $banner->title_extension, body => $banner->body, href => $banner->href, domain => $banner->domain, phone => $banner->has_vcard_id},
            {title => $banner->old->title, title_extension => $banner->old->title_extension, body => $banner->old->body, href => $banner->old->href, domain => $banner->old->domain, statusModerate => $banner->old->status_moderate},
        );
        $need_moderate_banner ||= $banner->is_status_moderate_changed && $banner->status_moderate eq 'Ready';
        my $banner_is_draft = $banner->status_moderate eq 'New';

        if (!$banner_is_draft && ($need_moderate_banner || $banner_has_changed)) {
            # Баннер может быть незначительно изменен, в таком случае модерация не требуется
            if ($need_moderate_banner) {
                $banner->status_moderate('Ready');

                $banner->status_post_moderate('No') if $banner->status_post_moderate ne 'Rejected';
                $banner->do_update_status_post_moderate_unless_rejected(1);

                if ($banner->adgroup->status_moderate eq 'New' && $banner->adgroup->has_show_conditions) {
                    $banner->do_moderate_adgroup(1);
                    $banner->do_moderate_campaign_if_no_active_banners(1);
                }
            }

            $banner->status_bs_synced('No');
            $banner->do_clear_moderation_flags(1);
            $banner->do_push_flags_to_moderation(1) if $banner->is_flags_changed;
        } elsif ($banner_is_draft) {
            # Если баннер черновик, то status_post_moderate должен быть No
            $banner->status_post_moderate('No');
        }
        $banner->do_bs_sync_adgroup(1)
            if $banner->has_image_hash && $banner->image_hash && (!$banner->old->has_image_hash || !defined $banner->old->image_hash);
    }

    $self->set_moderate_turbolandings();
    $self->set_bs_synced_when_turbolanding_params_changed();

    return $self;
}

=head2 prepare_suspend

Подготовка списка баннеров для остановки показов.

=cut

sub prepare_suspend {
    my ($self) = @_;

    for my $banner (@{$self->items}) {
        $banner->status_show('No');
        $banner->status_bs_synced('No');
        $banner->do_update_last_change(1);

        # надо переотправить группу, если у баннера есть минус-гео, который ограничивает гео-таргетинг группы
        $banner->do_bs_sync_adgroup(1)
            if @{GeoTools::get_disabled_geo($banner->adgroup->geo, $banner->minus_geo, $banner->client_id)};

        $banner->do_resume_adgroup_autobudget_show(1);
        $banner->do_schedule_forecast(1);
    }

    return $self;
}

=head2 suspend($uid)

Аналогично prepare_suspend, но с применением изменений в БД. Также выполняется отправка нотификаций.

=cut

sub suspend {
    my ($self, $uid) = @_;

    $self->prepare_suspend();
    $self->prepare_logging('suspend', uid => $uid);
    $self->manager->update(where => {statusShow => 'Yes'});
    $self->do_logging();

    return;
}

=head2 prepare_resume

Подготовка списка баннеров для возобновления показов.

=cut

sub prepare_resume {
    my ($self) = @_;

    for my $banner (@{$self->items}) {
        $banner->status_show('Yes');
        $banner->status_bs_synced('No');
        $banner->do_update_last_change(1);

        # Надо переотправить группу, если у баннера есть минус-гео, который ограничивает гео-таргетинг группы
        $banner->do_bs_sync_adgroup(1)
            if @{GeoTools::get_disabled_geo($banner->adgroup->geo, $banner->minus_geo, $banner->client_id)};

        $banner->do_resume_adgroup_autobudget_show(1);
        $banner->do_schedule_forecast(1);
    }

    return $self;
}

=head2 resume($uid)

Аналогично prepare_resume, но с применением изменений в БД. Также выполняется отправка нотификаций.

=cut

sub resume {
    my ($self, $uid) = @_;

    $self->prepare_resume();
    $self->prepare_logging('resume', uid => $uid);
    $self->manager->update(where => {
        statusShow => 'No',
        statusArch => 'No',
    });
    $self->do_logging();

    return;
}

=head2 prepare_archive

Подготовка списка баннеров для архивации.

=cut

sub prepare_archive {
    my ($self) = @_;

    for my $banner (@{$self->items}) {
        $banner->status_archived('Yes');
        $banner->status_show('No');
        $banner->status_bs_synced('No');
        $banner->do_update_last_change(1);
    }

    return $self;
}

=head2 archive($uid)

Аналогично prepare_archive, но с применением изменений в БД. Также выполняется отправка нотификаций.

=cut

sub archive {
    my ($self, $uid) = @_;

    $self->prepare_archive();
    $self->prepare_logging('archive', uid => $uid);
    $self->manager->update(where => {_OR => {statusShow => 'No', BannerID => 0}});
    $self->do_logging();

    return;
}

=head2 prepare_unarchive

Подготовка списка баннеров для разархивации.

=cut

sub prepare_unarchive {
    my ($self) = @_;

    for my $banner (@{$self->items}) {
        $banner->status_archived('No');
        $banner->status_bs_synced('No');
        $banner->do_update_last_change(1);

        # Отправляем на модерацию, если в группе есть условия показа
        if ($banner->adgroup->has_show_conditions) {
            $banner->status_moderate('Ready');
            $banner->status_post_moderate('No');

            $banner->vcard_status_moderate('Ready') if $banner->vcard_id;
            $banner->sitelinks_status_moderate('Ready') if $banner->sitelinks_set_id;
            $banner->display_href_status_moderate('Ready') if $banner->is_display_href_supported && $banner->display_href;
            $banner->image->status_moderate('Ready') if $banner->has_image;

            $banner->do_moderate_adgroup(1) if $banner->adgroup->is_archived;
            $banner->do_moderate_campaign_if_no_active_banners(1);
        }
    }

    return $self;
}

=head2 unarchive($uid)

Аналогично prepare_unarchive, но с применением изменений в БД. Также выполняется отправка нотификаций.

=cut

sub unarchive {
    my ($self, $uid) = @_;

    $self->prepare_unarchive();
    $self->prepare_logging('unarchive', uid => $uid);
    $self->manager->update(where => {statusArch => 'Yes'});
    $self->do_logging();

    return;
}

=head2 prepare_moderate(%options)

Подготовка списка баннеров-черновиков к отправке на модерацию.
Для принудительной отправки баннеров не-черновиков -- лучше воспользоваться методом prepare_update,
с выставленым status_moderate в Ready.
Это связано с тем, что в этом методе нет логики удаления предыдущих непромодерированных объектов из очереди модерации.

Параметры:
    %options:
        post_moderate/pre_moderate/moderate_accept ->
            выставление соответствующего флага добавляет баннер на модерацию в нужную очередь

=cut

sub prepare_moderate {
    my ($self, %options) = @_;

    for my $banner (@{$self->items}) {
        # Группы без условий показа пропускаем
        next if !$banner->adgroup->has_show_conditions;

        $banner->do_post_moderate(1) if $options{post_moderate} && $banner->campaign->user->status_post_moderate eq 'Yes';
        $banner->do_pre_moderate(1) if $options{pre_moderate};
        $banner->do_auto_moderate(1) if $options{moderate_accept};

        $banner->status_moderate('Ready');
        $banner->status_post_moderate('No') if $banner->status_post_moderate ne 'Rejected';
        $banner->do_update_status_post_moderate_unless_rejected(1);

        $banner->vcard_status_moderate('Ready') if $banner->vcard_id;
        $banner->sitelinks_status_moderate('Ready') if $banner->sitelinks_set_id;
        $banner->display_href_status_moderate('Ready') if $banner->is_display_href_supported && $banner->display_href;
        $banner->image->status_moderate('Ready') if $banner->has_image;
        $banner->turbolanding->status_moderate('Ready') if $banner->has_turbolanding;

        $banner->do_moderate_adgroup(1) if $banner->adgroup->status_moderate eq 'New';
        $banner->do_moderate_campaign_if_no_active_banners(1);

        $banner->do_update_last_change(0) unless $banner->do_update_last_change;
    }

    return $self;
}

=head2 moderate(%optons)

Аналогично prepare_moderate, но с применением изменений в БД.

=cut

sub moderate {
    my ($self, %options) = @_;

    $self->prepare_moderate(%options);
    $self->manager->update();

    return;
}

=head2 prepare_logging($action, %params)
=head2 do_logging

Методы для логирования событий (действий).

Параметры:
    $action -> выполненное действие: create/update/suspend/resume/etc
    %params:
        uid -> uid пользователя, над которым выполняется операция (по умолчанию берется из %LogTools::context)

=cut

sub _get_banner_text {
    my $banner = shift;
    return html2string(join "\n",
        $banner->title,
        $banner->title_extension // (),
        (map { str $banner->$_ } qw/body href domain/),
        $banner->has_vcard
            ? VCards::get_contacts_string($banner->vcard->to_db_hash, "\n", {dont_include_ids => 1})
            : (),
        $banner->has_sitelinks_set
            ? Sitelinks::get_diff_string([map { $_->to_db_hash } @{$banner->sitelinks_set->links}])
            : ()
    );
}

sub prepare_logging {
    my ($self, $action, %params) = @_;

    my %log_context = %LogTools::context;

    for my $banner (@{$self->items}) {
        if ($action eq 'create') {
            # Создание новых баннеров логируем через псевдо-команду
            push @{$self->data->{log_cmd}}, {
                %log_context,
                ($params{uid} ? (uid => $params{uid}) : ()),
                cmd => '_new_banner',
                cid => $banner->campaign_id,
                pid => $banner->adgroup_id,
                bid => $banner->id,
                banner_type => $banner->banner_type,
                is_mobile => $banner->is_mobile,
            };

            push @{$self->data->{notifications}}, {
                object     => 'banner',
                event_type => 'b_create',
                object_id  => $banner->id,
                old_text   => '',
                new_text   => _get_banner_text($banner),
                uid        => $params{uid} || $log_context{uid},
            };

            push @{$self->data->{new_bids}}, $banner->id;
        }

        elsif ($action eq 'update') {
            my $old_text = _get_banner_text($banner->old);
            my $new_text = _get_banner_text($banner);

            push @{$self->data->{notifications}}, {
                object     => 'banner',
                event_type => 'b_text',
                object_id  => $banner->id,
                old_text   => $old_text,
                new_text   => $new_text,
                uid        => $params{uid} || $log_context{uid},
            } if $banner->is_changed && $old_text ne $new_text;
        }

        elsif ($action eq 'suspend' || $action eq 'resume') {
            push @{$self->data->{notifications}}, {
                object     => 'banner',
                event_type => 'b_status',
                object_id  => $banner->id,
                old_text   => $action eq 'suspend' ? 'start' : 'stop',
                new_text   => $action eq 'suspend' ? 'stop'  : 'start',
                uid        => $params{uid} || $log_context{uid},
            };
        }
    }

    return;
}

sub do_logging {
    my ($self) = @_;

    log_cmd($_) for @{$self->data->{log_cmd} // []};
    mass_mail_notification($self->data->{notifications}) if $self->data->{notifications};

    return;
}

=head2 copy_extra

Копирование баннера (без сайтлинков, визитки, картинки)

Параметры:
    см. Direct::Role::Copyable
    %options:
        with_mod_reasons - копировать информацию о принятии/отколонении баннеров на модерации

Результат:
    $old_bid2new_bid - отображение нового bid в исходный bid
    $new_bid2new_pid - отображение нового bid в группу (pid) в которую он записан

=cut

sub copy_extra {
    my ($self, $from_client_id, $source, $to_client_id, $destination, $src_bid2dst_bid, %options) = @_;

    if ($options{with_mod_reasons}) {
        my @mod_reasons_fields_to_copy = qw/type statusModerate statusPostModerate reason/;

        my %src_bid2dst_cid = pairwise {no warnings 'once'; ($a->id => $b->campaign_id)} @{$source->items}, @{$destination->items};
        my %mod_reasons_override = (
            id => $src_bid2dst_bid,
            timeCreated => unix2mysql(time()),
            statusSending => 'Yes',
            ClientID => $to_client_id,
            cid => \%src_bid2dst_cid,
        );
        my ($mod_reasons_fields_str, $mod_reasons_values_str) = make_copy_sql_strings(\@mod_reasons_fields_to_copy, \%mod_reasons_override, by => 'id');
        do_insert_select_sql(PPC(ClientID => $from_client_id),
            "INSERT INTO mod_reasons ($mod_reasons_fields_str) VALUES %s",
            ["SELECT $mod_reasons_values_str FROM mod_reasons", WHERE => {id => [keys %$src_bid2dst_bid], type => 'banner'}],
            dbw => PPC(ClientID => $to_client_id),
        );
        
        my ($bmg_fields_str, $bmg_values_str) = make_copy_sql_strings(
            [qw/bid minus_geo/],
            {
                bid => $src_bid2dst_bid,
            },
            by => 'bid'
        );
        do_insert_select_sql(PPC(ClientID => $from_client_id), "INSERT INTO banners_minus_geo ($bmg_fields_str) VALUES %s",
            ["SELECT $bmg_values_str FROM banners_minus_geo", where => { bid => [ keys %$src_bid2dst_bid ], type => 'current' }],
            dbw => PPC(ClientID => $to_client_id),
        );

    }

    my %new_bid2new_pid = map { $_->id => $_->adgroup_id } @{$destination->items};
    return $src_bid2dst_bid, \%new_bid2new_pid;
}

=head2 delete_minus_geo

    Direct::Banners::delete_minus_geo(bid => \@bids, %opt);

Удалить записи о минус-гео для баннеров
$key - столбец для выбора баннеров и определения шарда
$values - массив значений для $key
%opt:
    check_status => 1 - удалять только те записи, где баннер находится на модерации (Ready|Sending|Sent)
    status => $status - statusModerate, который ранее был установлен для баннеров

=cut

sub delete_minus_geo
{
    my ($key, $values, %opt) = @_;
    my $shard_key = ($key =~ s!.+?\.!!r);
    if ($key !~ /\./) {
        $key = 'b.'.$key;
    }
    my %where = (
        $key => SHARD_IDS,
    );
    if (exists $opt{status} && $opt{status} && $opt{status} !~ /Ready|Sending|Sent/) {
        # удаляем записи только если баннеры отправляются на модерацию
        return;
    }
    if ($opt{check_status}) {
        $where{'b.statusModerate__in'} = [qw/Ready Sending Sent/],
    }
    $where{'bmg.type'} = 'current';
    do_sql(PPC($shard_key => $values), [qq#
        DELETE bmg FROM banners_minus_geo bmg
        JOIN banners b on b.bid = bmg.bid#,
        where => \%where,
    ]);
}

=head2 mark_deleted_moderate_banner_pages

    Direct::Banners::mark_deleted_moderate_banner_pages(bid => \@bids, %opt);

Пометить удаленными записи о модерации пейджей для баннеров
$key - столбец для выбора баннеров и определения шарда
$values - массив значений для $key
%opt:
    check_status => 1 - удалять только те записи, где баннер находится на модерации (Ready|Sending|Sent)
    status => $status - statusModerate, который ранее был установлен для баннеров

=cut

sub mark_deleted_moderate_banner_pages
{
    my ($key, $values, %opt) = @_;
    my $shard_key = ($key =~ s!.+?\.!!r);
    if ($key !~ /\./) {
        $key = 'b.'.$key;
    }
    my %where = (
        $key => SHARD_IDS,
    );
    if (exists $opt{status} && $opt{status} && $opt{status} !~ /Ready|Sending|Sent/) {
        # удаляем записи только если баннеры отправляются на модерацию
        return;
    }
    if ($opt{check_status}) {
        $where{'b.statusModerate__in'} = [qw/Ready Sending Sent/],
    }
    do_sql(PPC($shard_key => $values), [qq#
       update moderate_banner_pages mbp
        JOIN banners b on b.bid = mbp.bid
        set mbp.is_removed = 1#,
        where => \%where,
    ]);
}

=head2 set_remove_old_video_additions

Выставить флаги на удаление видео-дополнения, если оно было заменено дополнением другого типа

=cut

sub set_remove_old_video_additions
{
    my $self = shift;
    for my $banner (@{$self->items}) {
        if (my $old_creative = $banner->old_creative) {
            if (!$banner->has_creative || !$banner->creative->has_creative) {
                # креатив был удален - нужно отправить в модерацию команду на удаление видео-дополнения
                $banner->do_delete_video_addition_from_moderation(1);
            }
            next unless $banner->has_creative && $banner->creative->has_creative;
            if ($old_creative->resource_type eq 'media' && $banner->creative->creative->resource_type eq 'creative') {
                $banner->do_remove_media_video(1);
            }
            elsif ($old_creative->resource_type eq 'creative' && $banner->creative->creative->resource_type eq 'media') {
                $banner->do_remove_creative_video(1);
            }
        }
    }
}

=head2 set_moderate_creatives

Выставить баннерам статус do_moderate_creative, если креатив нужно отправить на модерацию

=cut

sub set_moderate_creatives
{
    my $self = shift;
    
    for my $banner (@{$self->items}) {
        if ($banner->status_moderate ne 'New' && $banner->has_creative 
            && ( $banner->is_text_changed_significantly || $banner->is_creative_changed)
        ) {
            # Видео-дополнение должно отправляться всегда на Модерацию:
            # если изменился текст баннера, который требует модерации
            # видео добавлено/изменено у баннера
            $banner->do_moderate_creative(1);
        }
    }
}

=head2 set_moderate_turbolandings

Выставить баннерам статус do_moderate_turbolanding, если турболандинг нужно отправить на модерацию

=cut

sub set_moderate_turbolandings
{
    my $self = shift;
    
    for my $banner (@{$self->items}) {
        next unless $banner->is_turbolanding_supported && $banner->has_turbolanding;
        if (
            ( $banner->status_moderate eq 'Ready' &&
                (!$banner->turbolanding->has_status_moderate || (any {$banner->turbolanding->status_moderate eq $_ } qw/New Error/))
            )
            ||
            ($banner->is_text_changed_significantly() || $banner->is_href_changed)
        ) {
            $banner->do_moderate_turbolanding(1);
        }
    }
}

=head2 set_bs_synced_when_turbolanding_params_changed

Сбрасываем баннерам status_bs_synced, если они промодерированы,
а доп. параметры турболендингов изменились.

=cut

sub set_bs_synced_when_turbolanding_params_changed
{
    my $self = shift;

    for my $banner (@{$self->items}) {
        next unless $banner->is_turbolanding_supported();
        next if $banner->status_moderate() ne 'Yes';
        next if !$banner->is_turbolanding_href_params_changed();

        $banner->status_bs_synced('No') if
                                $banner->has_turbolanding
                                || ($banner->has_sitelinks_set && (any {$_->has_turbolanding} @{$banner->sitelinks_set->links}));
    }
    
    return;
}

1;
