package Direct::Banners::CpcVideo;

use Mouse;
use Direct::Modern;

extends 'Direct::Banners';

use Direct::Model::BannerCpcVideo::Manager;

has '+items' => (isa => 'ArrayRef[Direct::Model::BannerCpcVideo]');

=head2 manager_class
=cut

sub manager_class { 'Direct::Model::BannerCpcVideo::Manager' }

=head2 get($banner_ids, %options)

Аналогичен Direct::Banners->get с banner_type => 'cpc_video'.

=cut

sub get {
    my ($class, $banner_ids, %options) = @_;
    $class->SUPER::get($banner_ids, %options, banner_type => 'cpc_video');
}

=head2 get_by($key, $vals, %options)

Аналогичен Direct::Banners->get_by с banner_type => 'cpc_video'.

=cut

sub get_by {
    my ($class, $key, $vals, %options) = @_;
    $class->SUPER::get_by($key, $vals, %options, banner_type => 'cpc_video');
}

=head2 _copy_moderate_info

Для того что бы работала классификация, вычисление языка, отправка в
БК и прочее при создании ВО/изменении креатива в ВО переносим title и
body с креатива в баннер и сохраняем в БД

=cut

sub _copy_moderate_info
{
    my ($banner, $creative) = @_;
    my $moderate_info = $creative->moderate_info;
    my %text_data = map { ( $_->{type} => $_->{text} ) } @{$moderate_info->{texts}//[]};
    $banner->title($text_data{title});
    $banner->body($text_data{body});
    $banner->filter_domain($text_data{domain});
}

=head2 prepare_create($uid, %options)

Подготовка списка баннеров к созданию для пользователя $uid.
%options унаследованы от родительского метода.

=cut

sub prepare_create {
    my ($self, $uid, %options) = @_;

    $self->SUPER::prepare_create($uid, %options);

    for my $banner (@{$self->items}) {
        my $creative = $banner->creative;
        $creative->status_moderate($banner->status_moderate);
        $creative->campaign_id($banner->campaign_id);
        $creative->adgroup_id($banner->adgroup_id);
        $creative->banner_id($banner->id);

        _copy_moderate_info($banner, $creative->creative);
    }
    return $self;
}

=head2 create($uid, %options)

Создание списка баннеров в БД для пользователя $uid.
%options аналогичны prepare_create.

=cut

sub create {
    my ($self, $uid, %options) = @_;

    $self->set_client_id_by_uid($uid);
    $self->prepare_create($uid, translocal_opt => $options{translocal_opt});
    $self->prepare_logging('create', uid => $uid);
    $self->manager->create();
    $self->do_logging();

    return;
}

=head2 prepare_update($uid, %options)


=cut

sub prepare_update {
    my ($self, $uid, %options) = @_;

    $self->SUPER::prepare_update();
    $self->_send_to_moderate_child();

    for my $banner (@{$self->items}) {
        if ($banner->creative->is_creative_id_changed) {
            _copy_moderate_info($banner, $banner->creative->creative);
            
            if ($banner->status_moderate ne 'New') {
                $banner->status_moderate('Ready');
                $banner->creative->status_moderate('Ready');
                $banner->status_bs_synced('No');
                $banner->do_moderate_adgroup(1);

            } else {
                $banner->creative->status_moderate('New');
            }
        }
    }

    return $self;
}

=head2 update($uid, %options)

Обновление списка баннеров в БД для пользователя $uid.
%options аналогичны prepare_update.

=cut

sub update {
    my ($self, $uid, %options) = @_;

    $self->set_client_id_by_uid($uid);
    $self->prepare_update($uid, %options);
    $self->prepare_logging('update', uid => $uid);
    $self->manager->update();
    $self->do_logging();

    return;
}

=head2 _send_to_moderate_child

В случае отправки баннера на модерацию делает тоже самое и
для дочернего (креатив) объекта

=cut

sub _send_to_moderate_child {
    my ($self) = @_;
    for my $banner (@{$self->items}) {
        if ($banner->status_moderate eq 'Ready' && $banner->creative->status_moderate eq 'New') {
            $banner->creative->status_moderate('Ready');
        }
    }
}

=head2 prepare_moderate

Подготовка списка баннеров к отправке на модерацию.

=cut

sub prepare_moderate {
    my ($self, %options) = @_;

    $self->SUPER::prepare_moderate(%options);
    $self->_send_to_moderate_child();

    return $self;
}

=head2 prepare_delete

Подготовка списка баннеров к удалению.

=cut

sub prepare_delete { }

=head2 delete($uid)

Удаление списка баннеров из БД для пользователя $uid.

=cut

sub delete {
    my ($self, $uid) = @_;

    $self->set_client_id_by_uid($uid);
    $self->prepare_delete();
    $self->prepare_logging('delete', uid => $uid);
    $self->manager->delete();
    $self->do_logging();

    return;
}

1;
