package Direct::Banners::ImageAd;

use Mouse;
use Direct::Modern;

extends 'Direct::Banners';

use Direct::Model::BannerImageAd::Manager;

has '+items' => (isa => 'ArrayRef[Direct::Model::BannerImageAd]');

=head2 manager_class
=cut

sub manager_class { 'Direct::Model::BannerImageAd::Manager' }

=head2 get($banner_ids, %options)

Аналогичен Direct::Banners->get с banner_type => 'image_ad'.

=cut

sub get {
    my ($class, $banner_ids, %options) = @_;
    $class->SUPER::get($banner_ids, %options, banner_type => 'image_ad');
}

=head2 get_by($key, $vals, %options)

Аналогичен Direct::Banners->get_by с banner_type => 'image_ad'.

=cut

sub get_by {
    my ($class, $key, $vals, %options) = @_;
    $class->SUPER::get_by($key, $vals, %options, banner_type => 'image_ad');
}

=head2 prepare_create($uid, %options)

Подготовка списка баннеров к созданию для пользователя $uid.
%options унаследованы от родительского метода.

=cut

sub prepare_create {
    my ($self, $uid, %options) = @_;

    $self->SUPER::prepare_create($uid, %options);

    for my $banner (@{$self->items}) {
        $banner->child_status_moderate($banner->status_moderate);
        $banner->child_campaign_id($banner->campaign_id);
        $banner->child_adgroup_id($banner->adgroup_id);
        $banner->child_banner_id($banner->id);
    }
    return $self;
}

=head2 create($uid, %options)

Создание списка баннеров в БД для пользователя $uid.
%options аналогичны prepare_create.

=cut

sub create {
    my ($self, $uid, %options) = @_;

    $self->set_client_id_by_uid($uid);
    $self->prepare_create($uid, translocal_opt => $options{translocal_opt});
    $self->prepare_logging('create', uid => $uid);
    $self->manager->create();
    $self->do_logging();

    return;
}

=head2 prepare_update($uid, %options)


=cut

sub prepare_update {
    my ($self, $uid, %options) = @_;

    $self->SUPER::prepare_update();
    $self->_send_to_moderate_child();

    for my $banner (@{$self->items}) {
        if ($banner->is_child_changed()) {
            if ($banner->status_moderate ne 'New') {
                $banner->status_moderate('Ready');
                $banner->child_status_moderate('Ready');
                $banner->status_bs_synced('No');
                $banner->do_moderate_adgroup(1);
            } else {
                $banner->child_status_moderate('New');
            }
        }
    }

    return $self;
}

=head2 update($uid, %options)

Обновление списка баннеров в БД для пользователя $uid.
%options аналогичны prepare_update.

=cut

sub update {
    my ($self, $uid, %options) = @_;

    $self->set_client_id_by_uid($uid);
    $self->prepare_update($uid, %options);
    $self->prepare_logging('update', uid => $uid);
    $self->manager->update();
    $self->do_logging();

    return;
}

=head2 _send_to_moderate_child

В случае отправки баннера на модерацию делает тоже самое и
для дочернего (картинка или креатив) объекта

=cut

sub _send_to_moderate_child {
    my ($self) = @_;
    for my $banner (@{$self->items}) {
        if ($banner->status_moderate eq 'Ready' && $banner->child_status_moderate eq 'New') {
            $banner->child_status_moderate('Ready');
        }
    }
}

=head2 prepare_moderate

Подготовка списка баннеров к отправке на модерацию.

=cut

sub prepare_moderate {
    my ($self, %options) = @_;

    $self->SUPER::prepare_moderate(%options);
    $self->_send_to_moderate_child();

    return $self;
}

=head2 prepare_delete

Подготовка списка баннеров к удалению.

=cut

sub prepare_delete { }

=head2 delete($uid)

Удаление списка баннеров из БД для пользователя $uid.

=cut

sub delete {
    my ($self, $uid) = @_;

    $self->set_client_id_by_uid($uid);
    $self->prepare_delete();
    $self->prepare_logging('delete', uid => $uid);
    $self->manager->delete();
    $self->do_logging();

    return;
}

1;

