package Direct::Banners::Performance;

use Mouse;
use Direct::Modern;

use List::MoreUtils qw/any/;

extends 'Direct::Banners';

use Direct::Model::BannerPerformance::Manager;

has '+items' => (isa => 'ArrayRef[Direct::Model::BannerPerformance]');

=head2 manager_class
=cut

sub manager_class { 'Direct::Model::BannerPerformance::Manager' }

=head2 get($banner_ids, %options)

Аналогичен Direct::Banners->get с banner_type => 'performance'.

=cut

sub get {
    my ($class, $banner_ids, %options) = @_;
    $class->SUPER::get($banner_ids, %options, banner_type => ['performance', 'performance_main']);
}

=head2 get_by($key, $vals, %options)

Аналогичен Direct::Banners->get_by с banner_type => 'performance'.

=cut

sub get_by {
    my ($class, $key, $vals, %options) = @_;
    $class->SUPER::get_by($key, $vals, %options, banner_type => ['performance', 'performance_main']);
}

=head2 prepare_create($uid, %options)

Подготовка списка перфоманс баннеров к созданию для пользователя $uid.
%options унаследованы от родительского метода.

=cut

sub prepare_create {
    my ($self, $uid, %options) = @_;

    $self->SUPER::prepare_create($uid, %options);

    my $translocal_opt = $options{translocal_opt};
    for my $banner (@{$self->items}) {
        $translocal_opt //= {ClientID => $banner->client_id};

        # Зададим sum_geo для новых креативов
        $banner->do_set_geo_for_new_creative({translocal_opt => $translocal_opt}) if any { $banner->creative->status_moderate eq $_ } qw/New Error/;

        # Статус модерации на баннере должен быть `New` или `Yes`
        if ($banner->status_moderate eq 'New') {
            $banner->creative_status_moderate('New');
        }
        else {
            $banner->status_moderate('Yes');
            $banner->creative_status_moderate('Yes');
            $banner->status_post_moderate('Yes');
            $banner->status_bs_synced('No');

            $banner->do_moderate_creative(1) if any { $banner->creative->status_moderate eq $_ } qw/New Error/;

            if ($banner->adgroup->status_moderate eq 'New') {
                $banner->do_moderate_adgroup(1);
                $banner->do_moderate_campaign_if_no_active_banners(1);
            }

            # Добавление нового баннера - переотправка всего условия показа
            $banner->do_bs_sync_adgroup(1);
            
            if ($banner->do_moderate_adgroup) {
                # Если группа отправляется на модерацию,
                #   то поставим также статус генерации фраз в BannerLand в `Processing`
                $banner->do_set_adgroup_bl_status('Processing');
            }
        }
    }

    return $self;
}

=head2 create($uid, %options)

Создание списка перфоманс баннеров в БД для пользователя $uid.
%options аналогичны prepare_create.

=cut

sub create {
    my ($self, $uid, %options) = @_;

    $self->set_client_id_by_uid($uid);
    $self->prepare_create($uid, translocal_opt => $options{translocal_opt});
    $self->prepare_logging('create', uid => $uid);
    $self->manager->create();
    $self->do_logging();

    return;
}

=head2 prepare_update($uid, %options)

Подготовка списка перфоманс баннеров к обновлению.
Параметры:
    $uid -> (не используется)
    %options:
        translocal_opt -> параметры транслокальности (по умолчанию: {ClientID => $banner->client_id})

=cut

sub prepare_update {
    my ($self, $uid, %options) = @_;

    $self->SUPER::prepare_update();

    my $translocal_opt = $options{translocal_opt};
    for my $banner (@{$self->items}) {
        $translocal_opt //= {ClientID => $banner->client_id};

        if ($banner->is_creative_id_changed) {
            # При изменении креатива нужно также посчитать sum_geo
            $banner->do_set_geo_for_new_creative({translocal_opt => $translocal_opt}) if any { $banner->creative->status_moderate eq $_ } qw/New Error/;
            # Отправим новый креатив на модерацию, если он еще не модерировался, а баннер не черновик
            $banner->do_moderate_creative(1) if 
                ( any { $banner->creative->status_moderate eq $_ } qw/New Error/ ) &&
                $banner->status_moderate ne 'New';
            $banner->status_bs_synced('No');
            $banner->do_update_last_change(1);
        }

        # Статус модерации на баннере должен быть `New` или `Yes`
        if (
            ($banner->status_moderate eq 'Yes' && $banner->is_status_moderate_changed) ||
            ($banner->status_moderate ne 'Yes' && $banner->status_moderate ne 'New')
        ) {
            $banner->status_moderate('Yes');
            $banner->creative_status_moderate('Yes');
            $banner->status_post_moderate('Yes');
            $banner->status_bs_synced('No');

            $banner->do_moderate_creative(1) if any { $banner->creative->status_moderate eq $_ } qw/New Error/;

            if ($banner->adgroup->status_moderate eq 'New') {
                $banner->do_moderate_adgroup(1);
                $banner->do_moderate_campaign_if_no_active_banners(1);
            }
        }
        elsif ($banner->status_moderate eq 'New') {
            $banner->creative_status_moderate('New');
        }

        # Баннер изменился, но ни разу не отправлялся в БК - препосылка условия показа
        $banner->do_bs_sync_adgroup(1) if $banner->is_changed && !$banner->bs_banner_id;

        if ($banner->do_moderate_adgroup) {
            # Если группа отправляется на модерацию,
            #   то поставим также статус генерации фраз в BannerLand в `Processing`
            $banner->do_set_adgroup_bl_status('Processing');
        }
    }

    return $self;
}

=head2 update($uid, %options)

Обновление списка перфоманс баннеров в БД для пользователя $uid.
%options аналогичны prepare_update.

=cut

sub update {
    my ($self, $uid, %options) = @_;

    $self->set_client_id_by_uid($uid);
    $self->prepare_update($uid, %options);
    $self->prepare_logging('update', uid => $uid);
    $self->manager->update();
    $self->do_logging();

    return;
}

=head2 prepare_moderate

Подготовка списка перфоманс баннеров к отправке на модерацию.

=cut

sub prepare_moderate {
    my ($self) = @_;

    for my $banner (@{$self->items}) {
        # Группы без условий показа пропускаем
        next if !$banner->adgroup->has_show_conditions;

        if ($banner->{banner_type} eq 'performance') {
            $banner->status_moderate('Yes');
            $banner->status_post_moderate('Yes');
            $banner->status_bs_synced('No');

            $banner->do_moderate_creative(1) if any {$banner->creative->status_moderate eq $_} qw/New Error/;
        } elsif ($banner->{banner_type} eq 'performance_main') {
            $banner->status_moderate('Ready');
            $banner->status_bs_synced('No');
        }

        if ($banner->adgroup->status_moderate eq 'New') {
            $banner->do_moderate_adgroup(1);
            $banner->do_moderate_campaign_if_no_active_banners(1);
            $banner->do_set_adgroup_bl_status('Processing');
        }

        $banner->do_update_last_change(0) unless $banner->do_update_last_change;
    }

    return $self;
}

=head2 prepare_delete

Подготовка списка перфоманс баннеров к удалению.

=cut

sub prepare_delete { 
    my ($self) = @_;

    for my $banner (@{$self->items}) {
        $banner->do_set_geo_for_new_creative({translocal_opt => {ClientID => $banner->client_id}}) if any { $banner->creative->status_moderate eq $_ } qw/New Error/;
    }

    return $self;
}

=head2 delete($uid)

Удаление списка перфоманс баннеров из БД для пользователя $uid.

=cut

sub delete {
    my ($self, $uid) = @_;

    $self->set_client_id_by_uid($uid);
    $self->prepare_delete();
    $self->prepare_logging('delete', uid => $uid);
    $self->manager->delete();
    $self->do_logging();

    return;
}

1;
