package Direct::BannersAdditions;

use Direct::Modern;
use Mouse;

use Settings;
use Direct::AdditionsItemCallouts;
use Direct::Model::AdditionsItemCallout;
use Direct::AdditionsItemExperiments;
use Direct::Model::AdditionsItemExperiment;
use PrimitivesIds qw/get_key2clientid/;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::HashUtils qw/hash_cut/;


has 'items_callouts' => (is => 'ro', isa => 'ArrayRef[Direct::Model::AdditionsItemCallout]');


use base qw/Exporter/;
our @EXPORT_OK = qw/
    link_unlink_additions_to_banners
/;

=head2 get_by

По заданному критерию возвращает список уточнений баннеров (моделей Direct::Model::AdditionsItemCallout).

    my $callouts = Direct::BannersAdditions::get_by(%opts);

где C<%opts>:

    id - искать по точному совпадению с указанным иденитификатором(ами)
    client_id - вернуть все уточнения по клиенту, включая не привязанные к баннерам
    banner_id - вернуть все уточнения по баннер(у|ам)
    campaign_id - вернуть все уточнения по кампаниям
    callout_text - вернуть все уточнения по тексту уточнения
    additions_type => 'callout|...' - тип дополнений
    status_moderate => 'Yes|No' - выбирать дополнения с заданным статусом модерации
    modified_since => '2016-03-10 00:00:00' - выбирать дополнения по дате последнего изменения начиная с указанной
    shard => если доступен шард, например при выгрузке уточнений для транспорта модерации
    with_mod_reasons => 0|1 - вернуть причины отклонения на модерации для дополнений
    with_deleted => 0|1 - вернуть все дополнения, включая удаленные
    only_deleted => 0|1 - вернуть только удаленные дополнения
    get_banner_id => 0|1 - возвращать banner_id в Direct::Model::AdditionsItemCallout, все дополнения должы быть привязаны к баннерам
    fill_in_linked => 0|1 - заполнять поле is_linked, признак, что условие привязано хотя бы к одному баннеру

    limit, offset - параметры для пагинации
    sort - [{column_name => 'asc'|'desc'}, {column_name2  => 'asc'}, ...] - сортировка по списку полей

Example:
    perl -MDirect::BannersAdditions -MDDP -e 'p(Direct::BannersAdditions->get_by(additions_type => "callout", get_banner_id => 1, banner_id => 1824065, sort => [{client_id => "asc"}, {id => "asc"}])->items_callouts)'

=cut

sub get_by {
    my ($class, %opt) = @_;

    croak "additions_type is not valid" if ! $opt{additions_type} || $opt{additions_type} !~ /^(callout)$/;

    my $callouts;

    if ($opt{additions_type} eq 'callout') {
        my $callout_opts = hash_cut({%opt}, @Direct::AdditionsItemCallouts::additions_item_callouts_valid_opt);
        $callouts = Direct::AdditionsItemCallouts->get_by( %$callout_opts );
    }

    my $result = $class->new($callouts ? (items_callouts => $callouts->items) : ());

    return $result;
}

=head2 items_callouts_by

    Вернуть сгруппированный по ключу хеш
    при группировке по banner_id:
        - в исходном объекте должен быть banner_id, например полученный через get_by(... get_banner_id => 1 ...)
        - не привязанные баннеры не возвращаются

Example:
    perl -MDirect::BannersAdditions -MDDP -e 'p(Direct::BannersAdditions->get_by(additions_type => "callout", get_banner_id => 1, banner_id => 1824065, sort => [{client_id => "asc"}, {id => "asc"}])->items_callouts_by("banner_id"))'

=cut

sub items_callouts_by {
    my ($self, $key) = @_;

    $key //= 'banner_id';
    croak "by `banner_id|client_id` only supported" unless $key =~ /^(?:banner_id|client_id)$/;

    my %result;
    if ($key eq 'banner_id') {
        for my $item (@{$self->items_callouts}) {
            next unless $item->banner_id;
            push @{$result{$item->banner_id}}, $item;
        }
    } elsif ($key eq 'client_id') {
        for my $item (@{$self->items_callouts}) {
            push @{$result{$item->client_id}}, $item;
        }
    }

    return \%result;
}

=head2 save

Cохранение списка дополнений баннеров

Если в каждом уточнении установленно поле banner_id, то выполняем привязку к баннерам, предполагаем что переданы полные данные (все additions_item_id) по баннерам,
иначе отсутствующие уточнения будут отвязаны. Если привязка не нужна, заполнять banner_id не нужно.

=cut

sub save {
    my ($self) = @_;

    if ($self->items_callouts) {
        my $additions_item_callouts = Direct::AdditionsItemCallouts->new(items => $self->items_callouts);
        $additions_item_callouts->save();

        my $link_data = {};
        for my $callout (@{$self->items_callouts}) {
            if ($callout->has_banner_id) {
                push @{$link_data->{$callout->banner_id}}, $callout->id;
            }
        }
        link_to_banners($link_data, 'callout') if %$link_data;
    }
}

=head2 link_to_banners

Привязка/отвязка дополнений к баннерам
На входе хеш со списком из баннеров и наборов additions_item_id к ним.
    $banners_links: {
        bid1 => [additions_item_id1, additions_item_id2, ...],
        bid2 => [additions_item_id1, additions_item_id3, ...],
        ...
    }

и тип дополнений

    perl -MDirect::BannersAdditions -MDDP -e 'Direct::BannersAdditions::link_to_banners({1824065 => [1, 2, 3]}, "callout")'

=cut

sub link_to_banners($$) {
    my ($banners_links_raw, $additions_type) = @_;

    # хеш со списками -> хеш с хешами
    my $new_banners_links = {};
    for my $new_bid (keys %$banners_links_raw) {
        my $additions = $banners_links_raw->{$new_bid};
        $new_banners_links->{$new_bid} = {map {$additions->[$_] => $_+1} (0 .. $#$additions)};
    }

    foreach_shard bid => [keys %$banners_links_raw], chunk_size => 500, sub {
        my ($shard, $bids) = @_;

        my $changed_bids = {};
        do_in_transaction {
            my $old_data_raw = get_all_sql(PPC(shard => $shard), [
                "select bid, additions_item_id, additions_type, sequence_num
                 from banners_additions
                ", where => {
                    bid => $bids,
                    additions_type => $additions_type,
                }, 'FOR UPDATE'
            ]);
            # группируем по bid и additions_item_id
            my $old_banners_links = {};
            for my $row (@$old_data_raw) {
                $old_banners_links->{ $row->{bid} }->{ $row->{additions_item_id} } = $row->{sequence_num};
            }

            my (@add_additions, @delete_additions);
            # проходим по новым связям, если их нет в старых или в другой позиции - то вставляем
            for my $new_bid (@$bids) {
                for my $new_additions_item_id (keys %{$new_banners_links->{$new_bid}}) {
                    my $old_num = $old_banners_links->{$new_bid}->{$new_additions_item_id};
                    my $new_num = $new_banners_links->{$new_bid}->{$new_additions_item_id};
                    next if $old_num && $old_num == $new_num;

                    push @add_additions, [$new_bid, $new_additions_item_id, $additions_type, $new_num];
                    $changed_bids->{$new_bid} = 1;
                }
            }

            # проходим по старым связям, если их нет в новых - то удаляем
            for my $old_bid (keys %$old_banners_links) {
                for my $old_additions_item_id (keys %{ $old_banners_links->{$old_bid} }) {
                    if (! exists $new_banners_links->{$old_bid}->{$old_additions_item_id}) {
                        push @delete_additions, _AND => {bid => $old_bid, additions_item_id => $old_additions_item_id};
                        $changed_bids->{$old_bid} = 1;
                    }
                }
            }

            do_mass_insert_sql(PPC(shard => $shard),
                'INSERT INTO banners_additions (bid, additions_item_id, additions_type, sequence_num) VALUES %s
                ON DUPLICATE KEY UPDATE sequence_num=VALUES(sequence_num)',
                \@add_additions,
            );
            if (@delete_additions) {
                my $sql_where = sql_condition(PPC(shard => $shard), {_OR => \@delete_additions});
                do_sql(PPC(shard => $shard), "delete from banners_additions where $sql_where");
            }
        }; # /do_in_transaction
        # обновляем LastChange и отсылаем в БК баннеры на которых изменились привязки
        # TODO обновлять в Direct::Banners::Text/Dynamic
        do_sql(PPC(shard => $shard), ["update banners set LastChange = NOW(), statusBsSynced = 'No'", where => {bid => [keys %$changed_bids]}]);
    };
}

=head2 unlink_from_banners

Отвязка конкретных дополнений от баннеров

На входе хеш со списком из баннеров и наборов additions_item_id к ним, которые нужно отвязать.
    $banners_links: {
        bid1 => [additions_item_id1, additions_item_id2, ...],
        bid2 => [additions_item_id1, additions_item_id3, ...],
        ...
    }

=cut

sub unlink_from_banners($) {
    my ($banners_links) = @_;

    for my $chunk (sharded_chunks bid => [keys %$banners_links], chunk_size => 5000) {
        my ($bids, $shard) = ($chunk->{bid}, $chunk->{shard});

        my $delete_data = [];

        for my $bid (@$bids) {
            for my $additions_item_id (@{ $banners_links->{$bid} }) {
                push @$delete_data, _AND => {bid => $bid, additions_item_id => $additions_item_id};
            }
        }

        my $sql_where = sql_condition(PPC(shard => $shard), {_OR => $delete_data});
        do_sql(PPC(shard => $shard), "delete from banners_additions where $sql_where");

        # обновляем LastChange и отсылаем в БК баннеры на которых изменились привязки
        do_sql(PPC(shard => $shard), ["update banners set LastChange = NOW(), statusBsSynced = 'No'", where => {bid => $bids}]);
    }
}

=head2 add_callouts_to_banners

В список баннеров добавить хеш с коллаутами, нужно для интерфейса.
В каждый баннер добавляется ключ callouts с уточнениями привязанными к баннеру.

    Direct::BannersAdditions::add_callouts_to_banners($banners);

=cut

sub add_callouts_to_banners($) {
    my ($banners) = @_;

    # заполняем уточнения
    my $callout_bids = [map {$_->{bid}} grep {defined $_->{adgroup_type} && ($_->{adgroup_type} eq 'base' || $_->{adgroup_type} eq 'dynamic')} @$banners];
    my $callouts = @$callout_bids
        ? Direct::BannersAdditions->get_by(additions_type => "callout", get_banner_id => 1, banner_id => $callout_bids)->items_callouts_by('banner_id')
        : {};
    for my $banner (@$banners) {
        $banner->{callouts} = [map {$_->to_template_hash} @{exists $callouts->{$banner->{bid}} ? $callouts->{$banner->{bid}} : []}];
    }
}


=head2 save_banners_experiments

Хелпер для сохранения экспериментов

    my %bid2experiment_json = (
        $bid1 => "{'param1':123,'param2':456}",
        ...
    );
    Direct::BannersAdditions::save_banners_experiments(\%bid2experiment_json);

Для удаления экспериментa надо передать пустое значение

=cut

sub save_banners_experiments {
    my ($bid2experiment_json, %params) = @_;

    my $bid2client_id = get_key2clientid(bid => [keys %$bid2experiment_json]);

    my %bid2experiment;
    my %link_data;
    for my $banner_id (keys %$bid2experiment_json) {
        my $experiment_json = $bid2experiment_json->{$banner_id};
        if (!$experiment_json) {
            $link_data{$banner_id} = [];
            next;
        }

        my $client_id = $bid2client_id->{$banner_id};
        if (!$client_id) {
            croak "Unknown client for bid=$banner_id";
        }

        $bid2experiment{$banner_id} = Direct::Model::AdditionsItemExperiment->new(
            client_id => $client_id,
            experiment_json => $experiment_json,
        );
    }

    if (%bid2experiment) {
        Direct::AdditionsItemExperiments->new(items => [values %bid2experiment])->save();
    }

    for my $banner_id (keys %bid2experiment) {
        $link_data{$banner_id} = [$bid2experiment{$banner_id}->id];
    }

    link_to_banners(\%link_data, 'experiment')  if %link_data;
    return;
}


=head2 link_unlink_additions_to_banners

    Проставление нужных флагов на баннере при сохранении дополнений

    link_unlink_additions_to_banners($items)
    На входе список ArrayRef[Direct::Model::Banner (Text/Dynamic)] у которых заполненно поле additions_callouts

=cut

sub link_unlink_additions_to_banners($) {
    my ($banners) = @_;

    for my $banner (@$banners) {
        my $old_callouts_texts =
            $banner->has_old && $banner->old->has_additions_callouts
            ? join("/", map {$_->callout_text} @{$banner->old->additions_callouts})
            : "";

        my $new_callouts_texts =
            $banner->has_additions_callouts
            ? join("/", map {$_->callout_text} @{$banner->additions_callouts})
            : "";

        next if $old_callouts_texts eq $new_callouts_texts;

        $banner->status_bs_synced('No');
        $banner->do_update_last_change(1);
        $banner->do_additions_change(1);

        if ($banner->has_additions_callouts && @{$banner->additions_callouts}) {
            for my $additions_callout (@{$banner->additions_callouts}) {
                # для привязки к баннерам нужен banner_id
                $additions_callout->banner_id($banner->id);
            }
        }
    }

    return;
}

=head2 delete

Удаление дополнений с отвязкой от баннеров

    Direct::BannersAdditions->new(items_callouts => [...])->delete();

=cut

sub delete {
    my ($self) = @_;

    if ($self->items_callouts) {
        my $callouts_by_client = $self->items_callouts_by('client_id');
        for my $client_id (keys %$callouts_by_client) {
            my $additions_items = $callouts_by_client->{$client_id};
            my $additions_item_ids = [map {$_->id} @$additions_items];

            do_in_transaction {
                my $banners_additions = Direct::BannersAdditions->get_by(
                    additions_type => 'callout',
                    client_id => $client_id,
                    id => $additions_item_ids,
                    get_banner_id => 1,
                )->items_callouts_by('banner_id');

                my $unlink_banner_data = {};
                for my $bid (keys %$banners_additions) {
                    push @{$unlink_banner_data->{$bid}}, map {$_->id} @{ $banners_additions->{$bid} };
                }

                Direct::BannersAdditions::unlink_from_banners($unlink_banner_data);

                Direct::Model::AdditionsItemCallout::Manager->new(items => $additions_items)->delete();
            } # /do_in_transaction
        }
    }
}

1;
