package Direct::BannersPermalinks;

use Direct::Modern;

use Yandex::I18n;
use Yandex::DBTools;
use Yandex::DBShards;

use Direct::Organizations;
use JavaIntapi::GetOrganizationsInfo;
use JavaIntapi::CheckOrganizationsAccess;

use Settings;
use List::MoreUtils qw/uniq/;


=head2 add_banner_permalinks_to_banners

    В список баннеров добавить ID пермалинка организации из Справочника если она указана.
    Нужно для интерфейса.
    Добавляется только указанный вручную опубликованный пермалинк.

=cut
sub add_banner_permalinks_to_banners {
    my ($banners) = @_;
    
    my $bids = [ map {$_->{bid}} grep {($_->{real_banner_type} // '') eq 'text'} @$banners ];
    if (!@$bids) {
        return;
    }

    my $permalinks_by_banner_id = get_hashes_hash_sql(PPC(bid => $bids), [
        "SELECT bid, permalink, prefer_vcard_over_permalink
         FROM banner_permalinks",
        where => {
            'bid__in'               => SHARD_IDS,
            'permalink_assign_type' => 'manual',
        }
    ]);

    for my $banner (@{$banners}) {
        if ($permalinks_by_banner_id->{$banner->{bid}}) {
            $banner->{permalink} = $permalinks_by_banner_id->{$banner->{bid}}->{permalink};
            $banner->{prefer_vcard_over_permalink} = $permalinks_by_banner_id->{$banner->{bid}}->{prefer_vcard_over_permalink};
        }
    }
}

=head2 get_permalinks_for_cids

    Возвращает список всех пермалинков для всех баннеров, имеющихся в этой кампании.

=cut
sub get_permalinks_for_cids {
    my ($cids) = @_;

    if (!@$cids) {
        return [];
    }
    my @permalinks;

    foreach my $chunk (sharded_chunks(cid => $cids, 100)) {
        my $shard = $chunk->{shard};
        my $cids_chunk = $chunk->{cid};

        my $permalinks_chunk = get_one_column_sql(PPC(shard => $shard), [
            "SELECT DISTINCT bpml.permalink
            FROM campaigns c
            JOIN banners b ON b.cid = c.cid
            STRAIGHT_JOIN banner_permalinks bpml ON b.bid = bpml.bid
            STRAIGHT_JOIN organizations org ON org.permalink_id = bpml.permalink AND org.ClientID = c.ClientID",
            where => {
                "c.cid__in"                  => $cids_chunk,
                "c.type__in"                 => ['text'],
                "bpml.permalink_assign_type" => 'manual',
                "org.status_publish"         => 'published',
            }
        ]);
        push @permalinks, @$permalinks_chunk;
    }

    return [uniq @permalinks];
}

=head3 _get_language_for_tycoon

    Получение языка для запроса в справочник из языка интерфейса

=cut
sub _get_language_for_tycoon {
    my ($lang) = @_;

    return {
        tr => 'TR',
        com => 'EN',
        en => 'EN',
        ua => 'UK',
        uk => 'UK',
        by => 'BE',
        be => 'BE',
        kz => 'KK',
        kk => 'KK',
    }->{ $lang } || 'RU';
}

=head2 get_organizations_info

    Вызывает JavaIntapi, который получает из API справочника данные об организациях по пермалинкам,
    после чего преобразует ответ к формату, нужному для старого интерфейса

=cut
sub get_organizations_info {
    my ($permalinks, $lang) = @_;

    if (!@$permalinks) {
        return {};
    }

    my $organizations_raw = JavaIntapi::GetOrganizationsInfo
        ->new(permalinkIds => $permalinks, language => _get_language_for_tycoon($lang))
        ->call();
    my @available_organizations = grep { $_->{statusPublish} ne 'UNPUBLISHED' } @$organizations_raw;
    return {map { $_->{permalinkId} => Direct::Organizations::to_hash($_) } @available_organizations};
}

=head2 check_organizations_access

    Вызывает JavaIntapi, который получает из API справочника информацию о доступности организаций пользователям

=cut
sub check_organizations_access {
    my ($uids, $permalinks) = @_;

    if (!@$uids || !@$permalinks) {
        return {};
    }

    return JavaIntapi::CheckOrganizationsAccess
        ->new(uids => $uids, permalinkIds => $permalinks)
        ->call();
}

1;
