package Direct::Bids;

use Mouse;
with qw/Direct::Role::Copyable/;

use Direct::Modern;

use Settings;

use Direct::Model::Bid;
use Direct::Model::Bid::Manager;
use Direct::Model::BidRelevanceMatch;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::I18n;
use Yandex::HashUtils qw/hash_cut/;
use List::MoreUtils qw/uniq any all/;

use LogTools qw//;
use ShardingTools qw/choose_shard_param/;
use MailNotification qw/mass_mail_notification/;
use Primitives qw//;

has 'items' => (is => 'ro', isa => 'ArrayRef[Direct::Model::Bid]');
has 'total' => (is => 'ro', isa => 'Int');
has 'data' => (is => 'ro', isa => 'HashRef', init_arg => undef, lazy => 1, default => sub { + { }; });

around BUILDARGS => sub {
        my ($orig, $class) = (shift, shift);
        $class->$orig(@_ == 1 ? (items => $_[0]) : @_)
    };

=head2 manager_class
=head2 manager
=cut

sub manager_class { 'Direct::Model::Bid::Manager' }
sub manager { $_[0]->manager_class->new(items => $_[0]->items) }

=head2 WEB_FIELD_NAMES

Название полей в web интерфейсе для расшифровки результатов валидации.

=cut

sub WEB_FIELD_NAMES {(
    price               => { field => sprintf('"%s"', iget('Цена на поиске')) },
    price_context       => { field => sprintf('"%s"', iget('Цена на сети')) },
    autobudget_priority => { field => sprintf('"%s"', iget('Приоритет автобюджета')) },
)}


my %KEYS_MAP = (cid => 'campaign_id', gid => 'adgroup_id', pid => 'adgroup_id', id => 'bid_id');

=head2 get_by($key, $vals, %options)

По заданному критерию возвращает instance с выбранными условиями показа.

Параметры:
    $key      -> по какому ключу выбирать: campaign_id/adgroup_id/bid_id
    $vals     -> scalar|arrayref; значение ключа
    %options:
        bid_type        -> (Str|ArrayRef[Str]); ставки каких типов выбирать: keyword/relevance_match (по-умолчанию - все)
        with_deleted    -> в том числе выбрать удаленные условия показа (в opts выставлен флаг deleted)
        'shard_key'     -> (если $key равен 'bid_id'); идентификатор для выбора номера шарда,
                            должен быть одним из @{$ShardingTools::DEFAULT_MAPPING_SHARD_KEYS}
        limit/offset    -> параметры для постраничной выборки
        filter          -> HashRef; дополнительный фильтр

=cut

sub get_by {
    my ($class, $key, $vals, %options) = @_;

    $key = $KEYS_MAP{$key} if exists $KEYS_MAP{$key};
    croak "only `campaign_id/adgroup_id/bid_id` keys are supported" unless $key =~ /^(?:campaign|adgroup|bid)_id$/;

    return $class->new(items => [ ]) if !defined $vals || (ref($vals) eq 'ARRAY' && !@$vals);

    state $supported_bid_types = { map { $_ => 1 } @Direct::Model::Bid::BID_TYPES };

    my @shard;
    my %where = %{$options{filter} // { }};
    if ($key eq 'adgroup_id') {
        @shard = (pid => $vals);
        $where{'bid.pid'} = SHARD_IDS;
    } elsif ($key eq 'campaign_id') {
        @shard = (cid => $vals);
        $where{'bid.cid'} = SHARD_IDS;
    } else {
        # bid_id
        @shard = $options{shard} ? (shard => $options{shard}) : choose_shard_param(\%options);
        $where{'bid.bid_id'} = $vals;
    }

    if (defined $options{bid_type}) {
        $options{bid_type} = [ $options{bid_type} ] unless ref($options{bid_type}) eq 'ARRAY';
        croak "found unsupported bid_type in options" unless all { $supported_bid_types->{$_} } @{$options{bid_type}};
        $where{'bid.bid_type'} = $options{bid_type};
    }
    unless ($options{with_deleted}) {
        $where{_TEXT} = 'NOT FIND_IN_SET("deleted", bid.opts)';
    }

    my (@select_columns, @from_tables);

    push @select_columns,
        Direct::Model::Bid->get_db_columns(bids_base => 'bid', prefix => ''),
        Direct::Model::BidRelevanceMatch->get_db_columns(bids_href_params => 'bid_hp', prefix => '', fields => [ qw/href_param1 href_param2/ ]);

    push @from_tables,
        'bids_base bid',
        'LEFT JOIN bids_href_params bid_hp ON (bid.cid = bid_hp.cid AND bid.bid_id=bid_hp.id)';

    my $calc_found_rows = $options{limit} && $options{total_count} ? 'SQL_CALC_FOUND_ROWS' : '';
    my $bid_rows = get_all_sql(
        PPC(@shard),
        [
            sprintf("SELECT $calc_found_rows %s FROM %s", join(', ', @select_columns), join(' ', @from_tables)),
            WHERE => \%where,
            'ORDER BY bid.bid_id',
            ($options{limit} ? (LIMIT => $options{limit}, $options{offset} ? (OFFSET => $options{offset}) : ()) : ()),
        ]
    );

    my $found_rows = $calc_found_rows ? select_found_rows(PPC(@shard)) : undef;
    my $self = $class->new(items => [ ], $calc_found_rows ? (total => $found_rows) : ());

    return $self unless @$bid_rows;

    my $_cache;
    for my $row (@$bid_rows) {
        if ($row->{'bid_type'} eq 'keyword') {
            croak "fetched unsupported bid type: ".$row->{'bid_type'};
        } elsif ($row->{'bid_type'} eq 'relevance_match' || $row->{'bid_type'} eq 'relevance_match_search') {
            push @{$self->items}, Direct::Model::BidRelevanceMatch->from_db_hash($row, \$_cache);
        } else {
            croak "fetched unsupported bid type: ".$row->{'bid_type'};
        }
    }

    return $self;
}

=head2 items_by($key)

Возвращает структуру со ставками, вида:
    $key //eq 'id' => {$bid->id => $bid1, $bid2->id => $bid2, ...};
    $key eq 'gid'  => {$adgroup_id1 => [$bid1, $bid2, ...], adgroup_id2 => [$bid3, ...], ...};

=cut

sub items_by {
    my ($self, $key) = @_;

    $key //= 'id';
    $key = $KEYS_MAP{$key} if exists $KEYS_MAP{$key};
    croak "only `adgroup_id/bid_id` keys are supported" unless $key =~ /^(?:adgroup|bid)_id$/;

    my %result;
    if ($key eq 'bid_id') {
        $result{$_->id} = $_ for @{$self->items};
    } else {
        push @{$result{ $_->adgroup_id } //= [ ]}, $_ for @{$self->items};
    }

    return \%result;
}



=head2 prepare_create

Подготовка списка ставок к созданию.

=cut

sub prepare_create {
    my ($self) = @_;

    for my $bid (@{$self->items}) {
        $bid->last_change('now');
        $bid->status_bs_synced('No');

        # Если ставка добавляется в группу не-черновик
        if ($bid->adgroup->status_moderate ne 'New') {
            # Пересинхронизируем группу
            $bid->do_bs_sync_adgroup(1);
        }

        # Если группа не промодерирована и не черновик - отправляем на модерацию
        if ($bid->has_adgroup && !any { $bid->adgroup->status_moderate eq $_ } qw/New Yes/) {
            $bid->do_moderate_adgroup(1);
        }

        $bid->do_update_adgroup_last_change(1);
    }

    return $self;
}

=head2 create

Создание списка ставок в БД.

=cut

sub create {
    my ($self, $uid) = @_;

    $self->prepare_create();
    $self->manager->create();
    $self->prepare_logging('create', uid => $uid);
    $self->do_logging();

    return;
}

=head2 prepare_update

Подготовка списка ставок к обновлению.

=cut

sub prepare_update {
    my ($self) = @_;

    for my $bid (@{$self->items}) {
        # Last change
        if (any { $bid->$_ } map { "is_${_}_changed" } qw/price price_context autobudget_priority is_suspended is_deleted href_param1 href_param2/) {
            $bid->last_change('now');
        }
        else {
            $bid->last_change('keep');
        }

        # Статус синхронизации группы, время последнего обновления группы
        if (any { $bid->$_ } map { "is_${_}_changed" } qw/is_suspended is_deleted href_param1 href_param2/) {
            $bid->do_bs_sync_adgroup(1);
            $bid->do_update_adgroup_last_change(1);
        }

        # При изменении цен/приоритета_автобюджета условия отправляются отдельным транспортом
        if (any { $bid->$_ } map { "is_${_}_changed" } qw/price price_context autobudget_priority/) {
            $bid->status_bs_synced('No');
        }

        # Если автотаргетинг восстановили и группа не промодерирована и не черновик - отправляем на модерацию
        if ($bid->has_adgroup && $bid->is_is_suspended_changed && !any { $bid->adgroup->status_moderate eq $_ } qw/New Yes/) {
            $bid->do_moderate_adgroup(1);
        }
    }

    return;
}

=head2 update(%options)

Обновление списка ставок в БД.

Параметры:
    %options:
        log_price__type -> тип логирования для цен

=cut

sub update {
    my ($self, %options) = @_;

    $self->prepare_update();
    $self->prepare_logging('update', %{hash_cut \%options, qw/uid log_price__type/});
    $self->manager->update();
    $self->do_logging();

    return;
}

=head2 prepare_delete

Подготовка списка ставок к удалению.

=cut

sub prepare_delete {
    my ($self) = @_;

    for my $bid (@{$self->items}) {
        $bid->price(0.00);
        $bid->price_context(0.00);
        $bid->autobudget_priority(undef);
        $bid->is_deleted(1);
        $bid->do_update_adgroup_last_change(1);
        $bid->do_bs_sync_adgroup(1) if $bid->adgroup->status_moderate ne 'New';
    }

    return $self;
}

=head2 delete

Удаление списка ставок (помечаем как удаленные в БД).

=cut

sub delete {
    my ($self, $uid) = @_;

    $self->prepare_delete();
    $self->prepare_logging('delete', uid => $uid);
    $self->manager->update();
    $self->do_logging();

    return;
}

=head2 prepare_logging($action, %params)
=head2 do_logging

Методы для логирования событий (действий).

Параметры:
    $action -> выполненное действие: create/update/delete
    %params:
        log_price__type -> тип логирования для цен
        uid             -> uid пользователя, над которым выполняется операция (по умолчанию берется из %LogTools::context)

=cut

sub prepare_logging {
    my ($self, $action, %params) = @_;

    my %log_context = %LogTools::context;
    my $uid = $params{uid} || $log_context{uid};

    croak "undefined uid" if !defined $uid;

    my @adgroup_ids = uniq(map { $_->adgroup_id } @{$self->items});
    my $adgroup_id2campaign = get_hashes_hash_sql(PPC(pid => \@adgroup_ids), [ q{
        SELECT g.pid, g.cid, IFNULL(c.currency, 'YND_FIXED') AS currency FROM phrases g JOIN campaigns c ON (g.cid = c.cid)
    }, where => {
                'g.pid' => SHARD_IDS,
            } ]);

    my $main_bid_by_gid = Primitives::get_main_banner_ids_by_pids(\@adgroup_ids);

    for my $bid (@{$self->items}) {
        my $main_banner_id = $main_bid_by_gid->{$bid->adgroup_id};
        my %logprice_tmpl = (
            %{$adgroup_id2campaign->{$bid->adgroup_id}}, # cid, pid, currency
            id        => $bid->id,
            price     => $bid->has_price ? $bid->price : 0,
            price_ctx => $bid->has_price_context ? $bid->price_context : 0,
        );

        if ($action eq 'create') {
            push @{$self->data->{log_price}}, { %logprice_tmpl, type => $params{log_price__type} // 'insert1' };
        } elsif ($action eq 'update') {
            if ($bid->is_price_changed || $bid->is_price_context_changed) {
                push @{$self->data->{log_price}}, { %logprice_tmpl, type => $params{log_price__type} // 'update1' };
            }
            push @{$self->data->{notifications}}, {
                    object     => 'phrase',
                    event_type => 'ph_price',
                    object_id  => $main_banner_id,
                    old_text   => $bid->old->price,
                    new_text   => $bid->price,
                    uid        => $uid,
                } if $main_banner_id && $bid->is_price_changed;

        } elsif ($action eq 'delete') {
            push @{$self->data->{log_price}}, { %logprice_tmpl, type => $params{log_price__type} // 'delete1' };
        } else {
            croak "Unknown action: $action";
        }
    }

    return;
}

sub do_logging {
    my ($self) = @_;

    LogTools::log_price($self->data->{log_price}) if $self->data->{log_price};
    mass_mail_notification($self->data->{notifications}) if $self->data->{notifications};

    return;
}

=head2 copy_extra
=cut

sub copy_extra {}


1;
