package Direct::BillingAggregates;


=head1 NAME

Direct::BillingAggregates

=head1 DESCRIPTION

Работа с биллинговыми агрегатами.

Биллинговые агрегаты - это "кампании", предназначенные для агрегации откруток
рекламных материалов определенного типа продукта.
Технически возможно записывать открутки на сами кампании, содержащие рекламные материалы,
и, частично, это происходит и сейчас, но при этом нельзя откручивать рекламные материалы
разных типов продукта внутри одной кампании.
Решением является создание нескольких биллинговых агрегатов, и отправка в БК инструкции,
что открутки по каким-то условиям записываются на разные ID агрегатов (заказов в Биллинге)

Биллинговые агрегаты создаются во время создания кампании, если эта кампания находится
под общим счетом. Кампании-агрегаты заносятся под этот же общий счет, при этом под одним
общим счетом не должно быть несколько биллинговых агрегатов с одним и тем же типом продукта.
При создании кампании создаются только агрегаты недостающих типов, которые нужны для откруток
в данном типе кампании.
У кампаний есть тип продукта по умолчанию (он совпадает с ProductID самой кампании),
и, возможно, дополнительные продукты (см. %SPECIAL_PRODUCT_TYPES_BY_CAMP_TYPE).

=cut

use Direct::Modern;

use List::MoreUtils qw/uniq/;
use Mouse;
use Readonly;

use Campaign::Types;
use Client;
use Direct::Model::BillingAggregate;
use Direct::Model::BillingAggregate::Manager;
use LogTools qw/log_messages/;
use Primitives;

use Settings;

use Yandex::I18n;
use Yandex::DBShards;
use Yandex::DBTools;
use Yandex::ListUtils qw/xminus/;

has 'items' => (is => 'ro', isa => 'ArrayRef[Direct::Model::BillingAggregate]');

=head2 %SPECIAL_PRODUCT_TYPES_BY_CAMP_TYPE

    Дополнительные типы продуктов, которые могут показываться в кампаниях определенного типа.
    Например, при создании кампании типа cpm_banner, нужно создать биллинговый агрегат с типом
    cpm_banner, и, дополнительно, с типом cpm_video, т.к. в этой кампании могут появиться
    рекламные материалы этого типа.

    При добавлении новых типов продукта в уже существующие типы кампаний, нужно запустить
    <TODO: некоторый скрипт>, который досоздаст биллинговые агрегаты всем клиентам, у которых
    уже есть кампании этого типа.

=cut
Readonly my %SPECIAL_PRODUCT_TYPES_BY_CAMP_TYPE => (
    'cpm_banner' => [qw/cpm_video cpm_outdoor cpm_indoor cpm_audio/],
);


=head2 %AGGREGATE_NAME_BY_PRODUCT_TYPE

    Названия заказов-агрегатов в Биллинге.

=cut
Readonly our %AGGREGATE_NAME_BY_PRODUCT_TYPE => (
    'text' => iget('Общий счет: покликовый Директ'),
    'cpm_deals' => iget('Общий счет: частные сделки'),
    'cpm_banner' => iget('Общий счет: медийный баннер'),
    'cpm_video' => iget('Общий счет: медийное видео'),
    'cpm_audio' => iget('Общий счет: медийное аудио'),
    'cpm_yndx_frontpage' => iget('Общий счет: баннер на главной'),
    'cpm_outdoor' => iget('Общий счет: outdoor видео'),
    'cpm_indoor' => iget('Общий счет: indoor видео'),
    'cpm_price' => iget('Общий счет: прайсовые продажи'),
);

=head2 %UIDS_TO_DISABLE_BILLING_AGGREGATES

    Словарь uid, которым запрещено создавать биллинговые агрегаты.
    Нужен для автотеста на неотправку биллинговых агрегатов в БК

=cut
Readonly my %UIDS_TO_DISABLE_BILLING_AGGREGATES => (
    # at-transport-archived-ba
    783297797 => 1,
);

=head2 manager($self, $operator_uid)

    Создание объекта модель-менеджера для всех агрегатов в коллекции.
    Менеджер будет создавать заказы от лица $operator_uid.

=cut
sub manager {
    my ($self, $operator_uid) = @_;
    return Direct::Model::BillingAggregate::Manager->new(
        items => $self->items,
        operator_uid => $operator_uid,
    );
}


=head2 is_autocreate_disabled($class, $client_id, $user_id, $currency, %O)

    Указывает, что клиенту нельзя автоматически создавать биллинговые агрегаты.

    Необязательные именованные параметры:

        preloaded_feature_by_clients - хеш вида {$client_id_1 => 1/0, $client_id_2 => 1/0, ...},
                Если он указан, функция не будет проверять наличие фичи автосоздания сама,
                а возьмет это знание из хеша.

=cut
sub is_autocreate_disabled {
    my ($class, $client_id, $user_id, $currency, %O) = @_;
    my $preloaded_feature_by_clients = delete $O{preloaded_feature_by_clients};
    if (keys %O) { die "unknown params ".join(', ', keys %O); };

    if ($currency eq 'YND_FIXED') {
        return 1;
    }

    my $feature_by_clients = $preloaded_feature_by_clients;
    if (!$feature_by_clients) {
        $feature_by_clients = {
            $client_id => Client::ClientFeatures::need_create_billing_aggregates($client_id),
        };
    }

    if (!$feature_by_clients->{$client_id}) {
        return 1;
    }

    if ($UIDS_TO_DISABLE_BILLING_AGGREGATES{$user_id}) {
        return 1;
    }

    return 0;
}


=head2 get_missing_product_types($class, $client_id, $wallet_id, \@camp_types, %O)

    Проверяет, есть ли у клиента $client_id под ОС $wallet_id
    все биллинговые агрегаты с типами продука, которые нужны
    для кампаний типов @camp_types.
    Возвращает ссылку на массив с недостающими типами продуктов.

    $missing_types = $class->get_missing_product_types($client_id, $wallet_id, [qw/cpm_banner text/]);
    $missing_types ~~ [qw/text cpm_banner cpm_video/];

    Необязательные именованные параметры:

        preloaded_existing_aggs - хешик с моделями биллинговых агрегатов, который можно получить таким вызовом:
                Direct::BillingAggregates->get_by(...)->items_by_wallet_and_product();
                В этот хеш должен входить ОС клиента $client_id, иначе функция может ошибочно посчитать,
                что клиенту нужно досоздать биллинговый агрегат.
                Но это будет не очень страшно, т.к. функция создания перепроверит этот факт.

=cut
sub get_missing_product_types {
    my ($class, $client_id, $wallet_id, $camp_types, %O) = @_;
    my $preloaded_existing_aggs = delete $O{preloaded_existing_aggs};
    if (keys %O) { die "unknown params ".join(', ', keys %O); };

    my @need_product_types;
    for my $camp_type (@$camp_types) {
        push @need_product_types, Campaign::Types::default_product_type_by_camp_type($camp_type) // ();
        push @need_product_types, @{get_special_product_types_by_camp_type($camp_type)};
    }

    my $existing_agg = $preloaded_existing_aggs;
    if (!$existing_agg) {
        $existing_agg = $class->get_by(client_id => $client_id)->items_by_wallet_and_product();
    }
    my @missing_product_types = grep {
        !$existing_agg->{$wallet_id}{$_} && _can_auto_create_product_type($_)
    } @need_product_types;

    return \@missing_product_types;
}

sub _can_auto_create_product_type {
    my ($product_type) = @_;

    state $auto_products_hash = {
        map { $_ => 1 } @$Settings::BILLING_AGGREGATES_AUTO_CREATE_TYPES,
    };

    return $auto_products_hash->{$product_type};
}

=head2 make_new_aggregates_for_client($class, $client_id, \@types, $wallet)

    Для указанного клиента $client_id, создает модели биллинговых агрегатов
    с типами продукта $types, привязанные к общему счету $wallet.
    $wallet - объект Direct::Model::Wallet общего счета.

    Возвращает коллекцию Direct::BillingAggregates с этими моделями.

=cut
sub make_new_aggregates_for_client {
    my ($class, $client_id, $types, $wallet) = @_;

    my $currency = $wallet->currency;
    my $quasi_currency_flag = 0;
    if ($currency eq 'KZT') {
        # Для клиентов в тенге испольузем квазивалютный продукт, если стоит флаг
        my $client_data = Client::get_client_data($client_id, [qw/is_using_quasi_currency/]);
        $quasi_currency_flag = 1 if $client_data->{is_using_quasi_currency};
    }
    my $product_info_by_type = Primitives::product_info_by_product_types($types, $currency, $quasi_currency_flag);

    my @to_create = map { _make_billing_aggregate($_, $wallet, $product_info_by_type) } @$types;
    return $class->new(items => \@to_create);
}

sub _make_billing_aggregate {
    my ($type, $wallet, $product_info_by_type) = @_;

    my $aggregate_name = $AGGREGATE_NAME_BY_PRODUCT_TYPE{$type};
    if (!defined($aggregate_name)) {
        croak "can't find billing aggregate name for product type $type";
    }
    my $obj = Direct::Model::BillingAggregate->new(
        (map { $_ => $wallet->$_() } qw/
            user_id
            client_id
            currency
            client_fio
            email
            agency_user_id
            agency_id
            manager_user_id
        /),
        status_empty => 'No',
        status_archived => 'No',
        status_moderate => 'Yes',
        status_post_moderate => 'Accepted',
        campaign_name => $AGGREGATE_NAME_BY_PRODUCT_TYPE{$type},
        product_id => $product_info_by_type->{$type}->{ProductID},
        wallet_id => $wallet->id,
    );

    return $obj;
}

=head2 get_by($class, $key, $vals)

    Выбрать агрегаты из базы по ключу $key.
    Поддерживаются ключи id, client_id и wallet_id

=cut
sub get_by {
    my ($class, $key, $vals) = @_;
    croak "by `id`, `client_id`, `wallet_id` only supported" unless $key =~ /^(?:id|client_id|wallet_id)$/;

    my $self = $class->new(items => []);

    return $self unless defined $vals;
    $vals = [$vals] unless ref($vals) eq 'ARRAY';
    return $self unless @$vals;

    my (@select_columns, @from_tables);
    push @select_columns, Direct::Model::BillingAggregate->get_db_columns(campaigns     => 'c',  prefix => '');
    push @select_columns, Direct::Model::BillingAggregate->get_db_columns(camp_options  => 'co', prefix => '');
    push @from_tables, 'campaigns c';
    push @from_tables, 'JOIN camp_options co ON co.cid = c.cid';

    my %shard_selector = (campaign_id => 'cid', client_id => 'ClientID', wallet_id => 'cid');
    my %camp_key = (campaign_id => 'cid', client_id => 'ClientID', wallet_id => 'wallet_cid');
    my $campaign_rows = get_all_sql( PPC( $shard_selector{$key} => $vals ), [
        sprintf('SELECT %s FROM %s', join(', ' => @select_columns), join(' ' => @from_tables)),
        WHERE => [
            'c.'.$camp_key{$key} => SHARD_IDS,
            'c.type' => 'billing_aggregate',
            'c.statusEmpty__ne' => 'Yes',
        ],
    ]);

    my $_cache;
    for my $row (@$campaign_rows) {
        push @{$self->items}, Direct::Model::BillingAggregate->from_db_hash($row, \$_cache);
    }

    return $self;
}

=head2 items_by_wallet_and_product($self)

    Группирует агрегаты в коллекции по wallet_id и типу продукта.
    Возвращает хеш такого вида:
    {
        1231345 => {
            'text' => Direct::Model::BillingAggregate->new(...),
            'cpm_banner' => Direct::Model::BillingAggregate->new(...),
            'cpm_video' => Direct::Model::BillingAggregate->new(...),
        },
        ...
    }

    Если под одним ОС обнаруживается два биллинговых агрегата для одного и того
    же типа продукта, про это пишется сообщение в лог, а агрегат с бОльшим ID
    игнорируется.

=cut
sub items_by_wallet_and_product {
    my ($self) = @_;

    my $product_info = product_info(hash_by_id => 1);
    my %result;
    for my $item (sort { $a->id <=> $b->id } @{$self->items}) {
        my $product_type = $product_info->{$item->product_id}->{type};
        if (!$product_type) {
            croak "unknown ProductID ".$item->product_id;
        }
        if ($result{$item->wallet_id}{$product_type}) {
            log_messages(
                "[billing_aggregates]",
                {
                    message => "duplicate billing aggregate product types detected",
                    client_id => $item->client_id,
                    wallet_id => $item->wallet_id,
                    original_id => $result{$item->wallet_id}{$product_type}->id,
                    duplicate_id => $item->id,
                    product_type => $product_type,
                }
            );
            next;
        }
        $result{$item->wallet_id}{$product_type} = $item;
    }
    return \%result;
}

=head2 create($self, $operator_uid)

    Сохранение агрегатов из текущей коллекции в БД.
    Описание параметров см. в документации к методу manager

=cut
sub create {
    my ($self, $operator_uid) = @_;

    $self->manager($operator_uid)->create();
}

=head2 get_special_product_types_by_camp_type($camp_type)

    Возвращает arrayref с дополнительными типами продуктов, которые
    могут показываться в кампаниях типа $camp_type (помимо типа продукта самой кампании)

=cut
sub get_special_product_types_by_camp_type {
    my ($camp_type) = @_;

    return $SPECIAL_PRODUCT_TYPES_BY_CAMP_TYPE{$camp_type} // [];
}


=head2 get_relevant_camp_types_by_product_types(\@product_types)

    По списку типов продуктов возвращает массив типов кампаний, которые могут
    использовать биллинговые агрегаты с такими типами продуктов.

=cut
sub get_relevant_camp_types_by_product_types {
    my ($product_types) = @_;

    state $product_type2camp_types;
    if (!$product_type2camp_types) {
        $product_type2camp_types = {};
        my $all_camp_types = xminus(
            Campaign::Types::get_camp_kind_types('all'),
            Campaign::Types::get_camp_kind_types('without_billing_aggregates')
        );

        for my $camp_type (@$all_camp_types) {
            my @prod_types = (
                Campaign::Types::default_product_type_by_camp_type($camp_type),
                @{get_special_product_types_by_camp_type($camp_type)},
            );
            push @{$product_type2camp_types->{$_}}, $camp_type for @prod_types;
        }
    }

    return [uniq map { @{$product_type2camp_types->{$_}} } @$product_types];
}

1;
