package Direct::BsData;

use Direct::Modern;

use Settings;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::HashUtils qw/hash_cut/;
use Yandex::URL qw/get_host strip_protocol/;
use Yandex::IDN qw//;
use Yandex::CheckMobileRedirect qw/parse_store_url/;
use List::UtilsBy qw/partition_by/;
use List::MoreUtils qw/any all uniq/;

use BSAuction qw//;
use BS::TrafaretAuction qw//;
use Stat::OrderStatDay;
use TextTools qw/round2s/;
use TimeTarget qw//;

use Models::AdGroup qw//;


sub _get_or {
    my ($model, $attr, $default) = @_;
    my $has_attr = "has_${attr}";
    return $model->id || $model->$has_attr ? $model->$attr : $default;
}

sub _get_domain {
    my ($banner) = @_;
    return undef if !$banner->has_href || !defined $banner->href;
    return Yandex::IDN::idn_to_unicode(get_host(strip_protocol($banner->href)));
}

=head2 enrich_smart_data_items_with_bs_data($smart, %options)

=cut
sub enrich_smart_data_items_with_bs_data {
    my ($class, $smart, %options) = @_;

    my @bs_req;

    my $campaign_data = _get_campaigns_data($smart->campaign_id)->{$smart->campaign_id};

    for my $data_item (@{$smart->_data_to_apply}) {
        # Группа и основной баннер
        my $adgroup = $data_item->{adgroup};
        my @banners = (scalar(@{$data_item->{banners}}) ? @{$data_item->{banners}} : @{$smart->_ex_banners_by_gid->{$adgroup->id}//[]});
        next unless @banners;

        my $adgroup_data = _get_adgroup_from_model($adgroup, \@banners);
        my $req = {
            %$campaign_data,
            %$adgroup_data,
        };

        # Ключевые фразы
        if (exists $data_item->{keywords}) {
            $req->{phrases} = [map { +{
                %{$_->to_db_hash},
                phr => $_->text,
                phraseIdHistory => ($_->has_bs_history ? $_->bs_history : undef) // ($_->has_ctr_source ? $_->ctr_source : undef),
                _obj => $_,
            } } @{$data_item->{keywords}}];
        }

        # # Ретаргетинг
        # if (exists $data_item->{retargetings}) {
        #     $req->{retargetings} = [map { +{
        #         ret_cond_id => $_->{ret_cond_id},
        #         _obj => $_,
        #     } } @{$data_item->{retargetings}}];
        # }

        # # Условия нацеливания
        # if (exists $data_item->{dynamic_conditions}) {
        #     $req->{dynamic_conditions} = [map { +{
        #         dyn_cond_id => $_->dyn_cond_id,
        #         _obj => $_,
        #     } } @{$data_item->{dynamic_conditions}}];
        # }

        # # Перфоманс фильтры
        # if (exists $data_item->{performance_filters}) {
        #     $req->{performance_filters} = [map { +{
        #         perf_filter_id => $_->id,
        #         _obj => $_,
        #     } } @{$data_item->{performance_filters}}];
        # }

        push @bs_req, $req;
    }

    _perform_bs_req(\@bs_req, %{hash_cut \%options, qw/bs_auction bs_stat/});

    return;
}

=head2 enrich_with_bs_data($show_conditions)
=cut

sub enrich_with_bs_data {
    my ($class, $show_conditions) = @_;

    return if !@$show_conditions;

    my %show_conditions_by_gid = partition_by { $_->adgroup_id } @$show_conditions;
    my $adgroups_data = _get_adgroups_data([keys %show_conditions_by_gid]);
    my $campaigns_data = _get_campaigns_data([uniq(map { $_->{cid} } values %$adgroups_data)]);

    my @bs_req;
    while (my ($adgroup_id, $adgroup_show_conds) = each %show_conditions_by_gid) {
        my $adgroup_data = $adgroups_data->{$adgroup_id};
        my $campaign_data = $campaigns_data->{ $adgroup_data->{cid} };

        # adgroup_data уже содержит информацию по главному объявлению группы
        my $req = {%$campaign_data, %$adgroup_data};

        for my $show_cond (@$adgroup_show_conds) {
            # Ключевые фразы
            if ($show_cond->isa('Direct::Model::Keyword')) {
                push @{$req->{phrases}}, +{
                        %{$show_cond->to_db_hash},
                        phr => $show_cond->text,
                        phraseIdHistory => $show_cond->bs_history, # to make sure that the field is exists
                        _obj => $show_cond,
                    };
            }
            # Условия нацеливания
            elsif ($show_cond->isa('Direct::Model::DynamicCondition')) {
                push @{$req->{dynamic_conditions}}, +{
                        dyn_cond_id => $show_cond->dyn_cond_id,
                        _obj => $show_cond,
                    };
            }
            # Перфоманс фильтры
            elsif ($show_cond->isa('Direct::Model::PerformanceFilter')) {
                push @{$req->{performance_filters}}, +{
                        perf_filter_id => $show_cond->id,
                        _obj => $show_cond,
                    };
            } else {
                croak "Unknown show condition passed: ".ref($show_cond);
            }
        }

        push @bs_req, $req;
    }

    _perform_bs_req(\@bs_req);

    return;
}

sub _get_adgroup_from_model {
    my ($adgroup, $banners) = @_;
    my $main_banner = Models::AdGroup::get_main_banner({ banners =>
        [ map { + { %{$_->to_db_hash}, _obj => $_ } } @{$banners} ] })->{_obj};

    my $adgroup_hash = {
        # adgroup data
        pid                 => $adgroup->id,
        adgroup_type        => $adgroup->adgroup_type,
        geo                 => $adgroup->geo,
        statusShowsForecast => _get_or($adgroup => 'status_shows_forecast', 'New'),
	is_bs_rarely_loaded => $adgroup->is_bs_rarely_loaded,

        # main_banner data
        bid                 => $main_banner->id,
        BannerID            => _get_or($main_banner => 'bs_banner_id', 0),
        image_BannerID      => $main_banner->has_image ? _get_or($main_banner->image => 'bs_banner_id') : undef,
        title               => $main_banner->title,
        title_extension     => $main_banner->title_extension,
        body                => $main_banner->body,
        filter_domain       => _get_or($main_banner => 'filter_domain') // _get_domain($main_banner),
        phone               => $main_banner->has_vcard ? $main_banner->vcard->phone : undef,
    };
    if ($adgroup->adgroup_type eq 'mobile_content') {
        $adgroup_hash->{device_type_targeting} = $adgroup->device_type_targeting;
        $adgroup_hash->{mobile_content} = parse_store_url($adgroup->store_content_href);
    }
    return $adgroup_hash;
}


sub _get_campaigns_data {
    my ($cids) = @_;

    my $campaigns_data = get_hashes_hash_sql(PPC(cid => $cids), [q{
        SELECT
            c.cid, c.OrderID,
            IFNULL(c.currency, 'YND_FIXED') AS currency,
            co.fairAuction = 'Yes' AS fairAuction,
            c.autobudget,
            if(json_type(c.strategy_data->>'$.bid')='NULL', NULL, c.strategy_data->>'$.bid') as autobudget_bid,
            c.ContextPriceCoef,
            c.platform,
            IF(c.strategy_name='no_premium', c.strategy_data->>'$.place', NULL) as strategy_no_premium,
            c.day_budget,
            c.timeTarget, c.timezone_id, co.strategy, co.device_targeting,
            FIND_IN_SET('no_extended_geotargeting', c.opts) as no_extended_geotargeting,
            c.sum + IFNULL(wc.sum, 0) AS sum,
            c.sum_spent + IFNULL(wc.sum_spent, 0) AS sum_spent
        FROM
            campaigns c
            JOIN camp_options co ON (co.cid = c.cid)
            LEFT JOIN campaigns wc ON (wc.cid = c.wallet_cid AND wc.uid = c.uid)
    }, WHERE => {
        'c.cid' => SHARD_IDS,
    }]);
    for my $camp (values %$campaigns_data) {
        $camp->{timetarget_coef} = TimeTarget::timetarget_current_coef($camp->{timeTarget}, $camp->{timezone_id});
        $camp->{camp_rest} = round2s($camp->{sum} - $camp->{sum_spent});
        $camp->{spent_today} = Stat::OrderStatDay::get_order_spent_today($camp->{OrderID}) if $camp->{OrderID} && $camp->{day_budget};
    }

    return $campaigns_data;
}

sub _get_adgroups_data {
    my ($gids) = @_;

    my $main_bid_by_gid = Primitives::get_main_banner_ids_by_pids($gids);

    my $adgroups_data = get_hashes_hash_sql(PPC(bid => [values %$main_bid_by_gid]), [q{
        SELECT
            g.pid, g.cid, g.adgroup_type, g.geo, g.statusShowsForecast,
            g.is_bs_rarely_loaded, 
            b.bid, b.BannerID, MAX(bi.BannerID) image_BannerID,
            b.title, b.title_extension, b.body, IFNULL(fd.filter_domain, b.domain) filter_domain, vc.phone
        FROM
            banners b
            JOIN phrases g ON (g.pid = b.pid)
            LEFT JOIN banner_images bi ON (bi.bid = b.bid)
            LEFT JOIN filter_domain fd ON (fd.domain = b.domain)
            LEFT JOIN vcards vc ON (vc.vcard_id = b.vcard_id)
    }, WHERE => {
        'b.bid' => SHARD_IDS,
    }]);

    return $adgroups_data;
}

sub _perform_bs_req {
    my ($bs_req, %options) = @_;

    # Обновим прогнозы показов (showsForecast) для фраз
    Models::AdGroup::update_phrases_shows_forecast($bs_req);

    # Запросим данные у БК
    my (@bs_auction_req, @bs_stat_req);
    for my $req (@$bs_req) {
        if (any { $req->{adgroup_type} eq $_ } qw/base mobile_content/) {
            if (
                (($req->{strategy} // '') ne 'different_places' || $req->{platform} ne 'context') && @{$req->{phrases} // []}
            ) {
                # Не ходим в полные торги если поиск отключен или нет фраз
                push @bs_auction_req, $req;
            } else {
                # Ходим за статистикой за 28 дней
                push @bs_stat_req, $req;
            }
        } else {
            # Для остальных типов групп ходим только за статистикой
            push @bs_stat_req, $req;
        }
    }

    BS::TrafaretAuction::trafaret_auction(\@bs_auction_req) if ($options{bs_auction} // 1) && @bs_auction_req;
    BSAuction::bs_get_phrases_statuses(\@bs_stat_req, {update_all => 1}) if ($options{bs_stat} // 1) && @bs_stat_req;

    for my $data (@$bs_req) {
        for my $phrase_data (@{$data->{phrases}}) {
            my $keyword = $phrase_data->{_obj};

            $phrase_data->{ctr} //= $phrase_data->{Ctr} if exists $phrase_data->{Ctr};

            $keyword->bs_data(
                hash_cut $phrase_data, qw/
                    cbroker broker larr ectr pectr premium guarantee weighted
                    clicks shows ctr ctx_clicks ctx_shows ctx_ctr
                /
            );
        }
        # TODO: Retargetings, DynamicConditions, PerformanceFilters
    }

    return;
}

1;
