package Direct::Campaigns;

use Direct::Modern;

use List::MoreUtils qw/any/;
use Mouse;

use DateTime::Format::MySQL;

use BS::Export::Queues qw//;

use Direct::Model::Campaign;
use Direct::Model::Campaign::Manager;

use Direct::Model::CampaignGeo;
use Direct::Model::CampaignText;
use Direct::Model::CampaignMedia;
use Direct::Model::CampaignMobileContent;
use Direct::Model::CampaignDynamic;
use Direct::Model::CampaignPerformance;
use Direct::Model::CampaignMcbanner;
use Direct::Model::CampaignCpmBanner;
use Direct::Model::CampaignCpmDeals;
use Direct::Model::CampaignCpmYndxFrontpage;
use Direct::Model::CampaignInternalDistrib;
use Direct::Model::CampaignInternalFree;
use Direct::Model::CampaignContentPromotion;
use Direct::Model::CampaignCpmPrice;

use LogTools qw//;

use MailNotification qw/mass_mail_notification/;

use RBACDirect qw/rbac_delete_campaign/;

use Settings;

use Tools qw/log_cmd/;
use GeoTools qw//;
use JSON;

use Yandex::DBShards;
use Yandex::DBTools;
use Yandex::DateTime qw/now/;
use Yandex::HashUtils qw/hash_cut/;
use Yandex::SendMail qw/send_alert/;

our $SEND_ALERT_MAILS = 1;

our @MODEL_CLASSES = qw/
    Direct::Model::CampaignGeo
    Direct::Model::CampaignText
    Direct::Model::CampaignMedia
    Direct::Model::CampaignMobileContent
    Direct::Model::CampaignDynamic
    Direct::Model::CampaignPerformance
    Direct::Model::CampaignMcbanner
    Direct::Model::CampaignCpmBanner
    Direct::Model::CampaignCpmDeals
    Direct::Model::CampaignCpmYndxFrontpage
    Direct::Model::CampaignInternalDistrib
    Direct::Model::CampaignInternalFree
    Direct::Model::CampaignContentPromotion
    Direct::Model::CampaignCpmPrice
/;

our %MODEL_CLASS_BY_TYPE = map { $_->supported_type => $_ } @MODEL_CLASSES;

=head2 items

=cut

has 'data'  => (is => 'ro', isa => 'HashRef', init_arg => undef, lazy => 1, default => sub { +{}; });
has 'items' => (is => 'ro', isa => 'ArrayRef[Direct::Model::Campaign]');

=head2 manager_class

=cut

sub manager_class { 'Direct::Model::Campaign::Manager' }

=head2 supported_type

    Тип кампании, работу с которым реализует класс бизнес-логики
    Необходимо переопределить в подклассе

=cut

sub supported_type {
    croak "Not implemented in base class";
}

=head2 get($campaign_ids, %options)

    Возвращает массив (ArrayRef) кампаний по заданным идентификаторам ($campaign_ids).
    Поддерживаемые %options описаны в get_by.

=cut

sub get {
    my ($class, $campaign_ids, %options) = @_;
    $campaign_ids = [$campaign_ids]  if !ref $campaign_ids;
    return $class->get_by(campaign_id => $campaign_ids, %options);
}

=head2 get_by($key, $vals, %options)

    По заданному критерию возвращает список кампаний

    Параметры:
        $key -> по какому ключу выбирать кампании: campaign_id/...
        $vals -> (Int|ArrayRef[Int]); список идентификаторов
        %options:
            campaign_type -> (Str|ArrayRef[Str]); кампании каких типов выбирать: text/geo/...
            limit/offset  -> параметры для постраничной выборки
            with_roles    -> (Str|ArrayRef[Str]); список ролей, которые нужно применить к кампаниям
            fields        -> список колонок для выбора из БД, структура вида { table1 => [qw/ field1 field2 ... /], ... }

=cut

sub get_by {
    my ($class, $key, $vals, %options) = @_;

    croak 'only `campaign_id` keys are supported' unless $key =~ /^(?:campaign)_id$/;

    my $self = $class->new(items => []);

    return $self unless defined $vals;
    $vals = [$vals] unless ref($vals) eq 'ARRAY';
    return $self unless @$vals;

    if (defined $options{campaign_type}) {
        $options{campaign_type} = [$options{campaign_type}] unless ref($options{campaign_type}) eq 'ARRAY';
    }

    my (@select_columns, @from_tables);

    my ($campaigns_fields, $camp_options_fields, $cdmo_fields, $cyfp_fields);
    if ( defined $options{fields} ) {
        ($campaigns_fields, $camp_options_fields, $cdmo_fields, $cyfp_fields)
            = @{ $options{fields} }{qw/ campaigns camp_options campaigns_performance campaigns_cpm_yndx_frontpage/};
    }

    push @select_columns,
        Direct::Model::Campaign->get_db_columns(campaigns    => 'c',  prefix => '', fields => $campaigns_fields),
        Direct::Model::Campaign->get_db_columns(camp_options => 'co', prefix => '', fields => $camp_options_fields),
        Direct::Model::CampaignPerformance->get_db_columns(campaigns_performance => 'cdmo', prefix => '', fields => $cdmo_fields),
        Direct::Model::CampaignCpmYndxFrontpage->get_db_columns(campaigns_cpm_yndx_frontpage => 'cyfp', prefix => '', fields => $cyfp_fields),
        'GROUP_CONCAT(mc.metrika_counter) as _metrika_counters';

    push @from_tables,
        'campaigns c',
        'JOIN camp_options co ON (c.cid = co.cid)',
        'LEFT JOIN campaigns_performance cdmo ON (c.type="performance" AND c.cid = cdmo.cid)',
        'LEFT JOIN campaigns_cpm_yndx_frontpage cyfp ON (c.type="cpm_yndx_frontpage" AND c.cid = cyfp.cid)',
        'LEFT JOIN metrika_counters mc ON (c.cid = mc.cid)';

    my %shard_selector = (campaign_id => 'cid');
    my $campaign_rows = get_all_sql( PPC( $shard_selector{$key} => $vals ), [
        sprintf('SELECT %s FROM %s', join(', ' => @select_columns), join(' ' => @from_tables)),
        where => {
            'c.'.$shard_selector{$key} => SHARD_IDS,
            $options{campaign_type} ? ('c.type' => $options{campaign_type}) : (),
        },
        $options{limit} ? (
            'GROUP BY c.cid',
            'ORDER BY c.cid',
            limit => $options{limit}, $options{offset} ? (offset => $options{offset}) : (),
        ) : ('GROUP BY c.cid'),
    ]);

    if ( defined $options{with_roles} ) {

        my $roles = $options{with_roles};

        $roles = [ $roles ] unless ref( $roles ) eq 'ARRAY';

        if ( any { $_ eq 'BsQueue' } @$roles ) {

            my @cids = map { $_->{cid} } @$campaign_rows;

            my $in_bs_export = get_hashes_hash_sql(PPC(cid => \@cids), [
                'SELECT cid FROM bs_export_candidates', WHERE => { cid => SHARD_IDS },
                'UNION SELECT cid FROM bs_export_queue', WHERE => {
                        cid => SHARD_IDS,
                        _OR => [
                            par_id__ne => $BS::Export::Queues::SPECIAL_PAR_TYPES{nosend_for_drop_sandbox_client},
                            par_id__is_null => 1,
                        ],
                    },
            ]);

            foreach ( @$campaign_rows ) {
                $_->{is_in_bs_queue} = exists $in_bs_export->{ $_->{cid} } ? 1 : 0;
            }
        }
    }

    my $_cache;
    for my $row (@$campaign_rows) {
        # DIRECT-54448: подменяем устаревшие регионы
        $row->{geo} = GeoTools::substitute_temporary_geo($row->{geo});

        # set default value for currency
        if (exists $row->{currency} and not defined $row->{currency}) {
            $row->{currency} = 'YND_FIXED';
        }

        my $model_class = $self->get_model_class_by_type($row->{type});
        my $campaign = $model_class->from_db_hash($row, \$_cache, with => $options{with_roles});

        push @{$self->items}, $campaign;
    }

    return $self;
}


=head2 get_model_class_by_type

Возвращает класс для типа

Опции:

    skip_unknown

=cut

sub get_model_class_by_type {
    my $class = shift;
    my ($type, %O) = @_;

    my $model_class = $MODEL_CLASS_BY_TYPE{$type};
    return 'Direct::Model::Campaign'  if !$model_class && $O{skip_unknown};
    croak "Unsupported campaign type: $type"  if !$model_class;

    return $model_class;
}


=head2 items_by($key)

    Возвращает структуру с кампаниями вида:
        $key //eq 'id' => {$campaign1->id => $campaign1, $campaign2->id => $campaign2, ...};

=cut

sub items_by {
    my ($self, $key) = @_;

    $key //= 'id';
    croak "by `id` only supported" unless $key =~ /^(?:id)$/;

    my %result;
    if ($key eq 'id') {
        $result{$_->id} = $_ for @{$self->items};
    }

    return \%result;
}

=head2 prepare_suspend

    Подготовка списка кампаний для остановки показов

=cut

sub prepare_suspend {
    my ($self) = @_;

    for my $campaign (@{$self->items}) {
        if ($campaign->status_show ne 'No') {
            $campaign->status_show('No');
            $campaign->status_bs_synced('No');
            $campaign->stop_time(DateTime::Format::MySQL->format_datetime(now()));
            $campaign->set_db_column_value('camp_options', 'stopTime', 'NOW()', dont_quote => 1);
            $campaign->do_update_last_change(1);
        }
    }

    return $self;
}

=head2 suspend($uid)

    Аналогично prepare_suspend, но с применением изменений в БД. Также выполняется отправка нотификаций.

=cut

sub suspend {
    my ($self, $uid) = @_;

    $self->prepare_suspend();
    $self->prepare_logging('suspend', uid => $uid);
    $self->manager_class->new(items => $self->items)->update(where => {statusShow => 'Yes'});
    $self->do_logging();

    return;
}

=head2 prepare_resume

    Подготовка списка кампаний для запуска показов

=cut

sub prepare_resume {
    my ($self) = @_;

    for my $campaign (@{$self->items}) {
        $campaign->status_show('Yes');
        $campaign->status_bs_synced('No');
        $campaign->do_update_last_change(1);
    }

    return $self;
}

=head2 resume($uid)

    Аналогично prepare_resume, но с применением изменений в БД. Также выполняется отправка нотификаций.

=cut

sub resume {
    my ($self, $uid) = @_;

    $self->prepare_resume();
    $self->prepare_logging('resume', uid => $uid);
    $self->manager_class->new(items => $self->items)->update(where => {statusShow => 'No'});
    $self->do_logging();

    return;
}

=head2 prepare_delete

    Подготовка списка кампаний для удаления

=cut

sub prepare_delete {
    my ($self) = @_;

    for my $campaign (@{$self->items}) {
        $campaign->status_empty('Yes');
        $campaign->do_enqueue_for_delete(1);
    }

    return $self;
}

=head2 delete($uid)

    Аналогично prepare_delete, но с применением изменений в БД. Также выполняется отправка нотификаций.

=cut

sub delete {
    my ($self, undef, $uid) = @_;

    $self->prepare_delete();

    my @deleted_in_rbac;
    my $camps = $self->items;
    foreach my $camp ( @$camps ) {
        my $errcode = eval { rbac_delete_campaign( undef, $camp->id, $uid ) };
        if ($@) {
            send_alert('RBAC die on delete campaign (cid '. $camp->id .', uid '. $uid ."): $@", 'API5 error') if $SEND_ALERT_MAILS;
            next;
        } elsif ($errcode) {
            send_alert('RBAC return error on delete campaign (cid: '. $camp->id .', uid '. $uid ."), error code: $errcode", 'API5 error') if $SEND_ALERT_MAILS;
            next;
        }

        push @deleted_in_rbac, $camp;
    }

    $self->manager_class->new(items => \@deleted_in_rbac)->update();

    return [ map { $_->id } @deleted_in_rbac ];
}


=head2 prepare_set_strategy

Установка стратегии в кампаниях

Опции:
    is_search_stop
    is_net_stop
    is_different_places
    has_edit_avg_cpm_without_restart_enabled

=cut

sub prepare_set_strategy {
    my ($self, $strategy, %opt) = @_;

    for my $campaign (@{$self->items}) {
        # для перфомансов надо подтянуть цели
        if (
            $campaign->campaign_type eq 'performance'
            && $strategy->can('goal_id') && $strategy->goal_id && (
                !$campaign->_has_strategy
                || !$campaign->get_strategy->can('goal_id')
                || $strategy->goal_id != $campaign->get_strategy->goal_id
            )
        ) {
            $campaign->do_save_metrika_goals(1);
        }

        $campaign->set_strategy($strategy, %{hash_cut(\%opt, qw/has_edit_avg_cpm_without_restart_enabled
            has_conversion_strategy_learning_status_enabled is_attribution_model_changed/)});
        $campaign->do_update_last_change(1);

        if (exists $opt{is_search_stop} || exists $opt{is_net_stop}) {
            $campaign->platform($campaign->detect_platform(%opt, old => $campaign->platform));
        }

        $campaign->is_different_places($opt{is_different_places})  if defined $opt{is_different_places};
    }

    return $self;
}


=head2 set_strategy

То же, что и prepare_set_strategy, но с логированием

Опции те же

=cut

sub set_strategy {
    my ($self, $uid, $strategy, %opt) = @_;

    $self->prepare_set_strategy($strategy, %opt);
    $self->prepare_logging('set_strategy', uid => $uid, strategy => $strategy);

    return $self;
}


=head2 save

Сохранить изменения в кампаниях

=cut

sub save {
    my $self = shift;
    $self->manager_class->new(items => $self->items)->update();
    $self->do_logging();
    return $self;
}


=head2 prepare_logging($action, %params)
=head2 do_logging

    Методы для логирования событий (действий).

    Параметры:
        $action -> выполненное действие: suspend/resume/...
        %params:
            uid -> uid пользователя, над которым выполняется операция (по умолчанию берется из %LogTools::context)

=cut

sub prepare_logging {
    my ($self, $action, %params) = @_;

    my %log_context = %LogTools::context;

    for my $campaign (@{$self->items}) {
        if ($action eq 'suspend' || $action eq 'resume') {
            push @{$self->data->{notifications}}, {
                object     => 'camp',
                event_type => 'c_status',
                object_id  => $campaign->id,
                old_text   => $action eq 'suspend' ? 'start' : 'stop',
                new_text   => $action eq 'suspend' ? 'stop'  : 'start',
                uid        => $params{uid} || $log_context{uid},
            };
        }
        elsif ($action eq 'set_strategy') {
            push @{$self->data->{log_cmd}}, {
                %log_context,
                ($params{uid} ? (uid => $params{uid}) : ()),
                cmd => '_camp_set_strategy',
                cid => $campaign->id,
                strategy => to_json($campaign->get_flat_strategy_app_hash($params{strategy})),
            };
        }
    }

    return;
}

sub do_logging {
    my ($self) = @_;

    log_cmd($_) for @{$self->data->{log_cmd} // []};
    mass_mail_notification($self->data->{notifications}) if $self->data->{notifications};

    return;
}

1;
