package Direct::CanvasCreatives;

use Mouse;

use JSON;
use Settings;
use Yandex::HTTP;
use Yandex::TVM2;

use Direct::Modern;
use Direct::Creatives qw//;

has 'items' => (is => 'ro', isa => 'ArrayRef[Direct::Model::CanvasCreative]');

around BUILDARGS => sub { my ($orig, $class) = (shift, shift); $class->$orig(@_ == 1 ? (items => $_[0]) : @_) };

=head2 get_by($key, $vals, $uid)

По заданному критерию возвращает instance с выбранными креативами.

Параметры:
    $key -> по какому ключу выбирать: creative_id
    $vals -> (Int|ArrayRef[Int]); список идентификаторов
    uid - владелец креативов

=cut

sub get_by {
    my ($class, $key, $vals, $uid) = @_;

    croak "only `creative_id` keys are supported" unless $key =~ /^creative_id$/;

    $vals = [$vals // ()] if ref($vals) ne 'ARRAY';
    return $class->new([]) if !@$vals;

    my $creatives = Direct::Creatives::_get_creatives($uid, {
        'pc.creative_id' => $vals,
        'pc.creative_type' => [qw/canvas/]
    })->{creatives};
    return $class->new($creatives);
}

=head2 items_by($key)

Возвращает структуру с креативами, вида:
    $key //eq 'creative_id' => {$creative1->id => $creative1, $creative2->id => $creative2, ...};

=cut

sub items_by {
    my ($self, $key) = @_;

    $key //= 'creative_id';
    croak "by `creative_id` only supported" unless $key =~ /^(?:creative_id)$/;

    my %result;
    if ($key eq 'creative_id') {
        $result{$_->id} = $_ for @{$self->items};
    }

    return \%result;
}

=head2 push_creatives($operator_uid, $ClientID, $creatives)

Создает переданные креативы через ручку intapi-java

=cut

sub push_creatives {
    my ($class, $operator_uid, $ClientID, $creatives) = @_;
    
    state ($log, $json);
    $log //= Yandex::Log->new(
        no_log => 1,
        use_syslog => 1,
        syslog_prefix => 'INTAPI-JAVA',
        log_file_name => "CanvasCreatives.log",
    );
    $json //= JSON->new->allow_unknown(1)->allow_blessed(1)->convert_blessed(1);

    my $url_params = {operator_uid => $operator_uid, client_id => $ClientID};
    my $url = Yandex::HTTP::make_url(
        $Settings::DIRECT_INTAPI_URL . 'DisplayCanvas/upload_creatives',
        $url_params,
    );

    $log->out(['REQUEST:', 'POST', $url, $url_params]);
    my $ticket = eval{Yandex::TVM2::get_ticket($Settings::TVM2_APP_ID{intapi})} or $log->die("Cannot get ticket for $Settings::TVM2_APP_ID{intapi}: $@");
    my $response = Yandex::HTTP::http_parallel_request(POST => { 1 => {
            url => $url,
            body => encode_json($creatives),
        }},
        timeout => 15, log => $log,
        headers => { 'Content-Type' => 'application/json', 'Accept' => '*/*', 'X-Ya-Service-Ticket' => $ticket},
    )->{1};

    my ($result, $error);
    if ($response->{is_success}) {
        utf8::decode($response->{content});
        $result = $json->decode($response->{content})->{uploadResults} if $response->{content};
    }
    else {
        $error = $response->{content} || ('Status: ' . $response->{headers}->{Status} . ' (' . $response->{headers}->{Reason} . ')');
    }

    $log->out(['RESPONSE:', $result // $error, ( $error ? ( $response->{headers}->{Reason}) : () ) ]);

    return $result, $error;
}

=head2 extract_creative_id

    Direct::CanvasCreatives->extract_creative_id("https://canvas.yandex.ru/preview/27364")

    Выделяет из урла на просмотр креатива его ID, или undef если найти ничего не удалось

=cut

sub extract_creative_id {
    my ($class, $url) = @_;
    if ($url =~ m{/html5/[a-z0-9]*/(\d+)/preview}xms) {
        my $id5 = $1;
        return $id5 + 0;
    }
    my ($id_1, $id_2) = $url =~ m{/creatives/ (?: (\d+)/preview | \d+/(\d+) )}xms;
    my $id = $id_1 || $id_2;
    return $id ? $id + 0 : undef;
}

1;
