package Direct::Clients;

use List::MoreUtils qw/uniq/;
use Mouse;
use Readonly;

use Direct::Modern;

use Rbac qw/
    $ROLE_CLIENT
    $ROLE_AGENCY
    $ROLE_MANAGER
/;
use Settings;

use Direct::Model::AgencyClient;
use Direct::Model::AgencyClientRelation;
use Direct::Model::Client;
use Direct::Model::ClientClient;
use Direct::Model::ManagerClient;
use Direct::Users;
use Primitives qw/get_agency_clients_relations_data/;
use PrimitivesIds qw/get_key2clientid/;
use RBAC2::Extended;
use RBACDirect qw/
    rbac_get_managers_of_clients_by_clientids
    rbac_get_all_managers_of_agencies_clientids
/;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::I18n;

Readonly our $CLIENT_MODEL  => 'Direct::Model::ClientClient';
Readonly our $AGENCY_MODEL  => 'Direct::Model::AgencyClient';
Readonly our $MANAGER_MODEL => 'Direct::Model::ManagerClient';

Readonly our %ROLE_TO_SUBCLASS => (
    $ROLE_CLIENT    => $CLIENT_MODEL,
    $ROLE_AGENCY    => $AGENCY_MODEL,
    $ROLE_MANAGER   => $MANAGER_MODEL,
);

Readonly our %SUPPORTED_ROLES => (
    (map { $_ => 1 } keys %ROLE_TO_SUBCLASS ),
);

has 'items' => (is => 'ro', isa => 'ArrayRef[Direct::Model::Client]');

around BUILDARGS => sub { my ($orig, $class) = (shift, shift); $class->$orig(@_ == 1 ? (items => $_[0]) : @_) };

=head2 manager_class

=cut

sub manager_class { croak 'Not implemented' } # 'Direct::Model::User::Manager'

=head2 get_by($key, $vals)

    По заданному критерию возвращает instance с выбранными пользователями.

    Параметры:
        $key  -> по какому ключу выбирать: id
        $vals -> scalar|arrayref; значение ключа

    Именованые параметры:
        with_roles => 1/0 -> определять роль пользователя и в зависимости от нее создает экземпляры классов
                             Direct::Model::User
                             Direct::Model::UserClient

=cut

sub get_by {
    my ($class, $key, $vals, %O) = @_;

    croak "only `id` key is supported" unless $key =~ /^id$/;

    return $class->new(items => []) if !defined $vals || (ref($vals) eq 'ARRAY' && !@$vals);

    my (@select_columns, @from_tables);

    my $ids;
    if ($key eq 'id') {
        $ids = $vals;
    }

    push @select_columns, Direct::Model::Client->get_db_columns(clients => 'cl', prefix => '');

    push @from_tables, 'clients cl';

    my $rows = get_all_sql(PPC('ClientID' => $ids), [
        sprintf('SELECT %s FROM %s', join(', ', @select_columns), join(' ', @from_tables)),
        where => {
            'cl.ClientID' => SHARD_IDS,
        },
    ]);

    my $self = $class->new(items => []);

    return $self unless @$rows;

    my $_cache;
    for my $row (@$rows) {
        my $model_class = 'Direct::Model::Client';

        my $id = $row->{ClientID};
        my $role = $row->{role};
        if ($SUPPORTED_ROLES{$role}) {
            $model_class = $ROLE_TO_SUBCLASS{$role};
        }

        push @{$self->items}, $model_class->from_db_hash($row, \$_cache);
    }

    return $self;
}

=head2 enrich_with_agencies

    Для всех клиентских записей в $self->items(), заполнить связи с агентствами $client->agency_relations()
    А для всех связей также проставить клиента связанного агентства $agency_relation->agency()
    Если у клиента нет агентств, в $client->agency_relations() будет проставлен пустой массив.

=cut
sub enrich_with_agencies {
    my ($self, %O) = @_;

    my $opt_include_unbound = delete $O{include_unbound};
    if (%O) {
        die 'unsupported options: '.join(', ', keys %O);
    }

    my @clients = grep { $_->role() eq $ROLE_CLIENT } @{$self->items()};

    my @client_ids = map { $_->id() } @clients;
    my $client_relation_data = get_agency_clients_relations_data(\@client_ids);

    my $_cache;
    my @agency_client_relations = map { Direct::Model::AgencyClientRelation->from_db_hash($_, \$_cache) }
            grep { $opt_include_unbound || $_->{bind} eq 'Yes' } @$client_relation_data;

    my @agency_ids = uniq map { $_->agency_client_id() } @agency_client_relations;
    my $agencies = Direct::Clients->get_by(id => \@agency_ids);

    if (scalar(@agency_ids) != scalar(@{$agencies->items()})) {
        croak "some agency relations have no agencies";
    }

    my $agencies_by_id = $agencies->items_by('id');

    my %client_id_to_relations;
    for my $relation (@agency_client_relations) {
        my $agency = $agencies_by_id->{$relation->agency_client_id()};
        if ($agency->role() ne $ROLE_AGENCY) {
            next;
        }
        $relation->agency($agencies_by_id->{$relation->agency_client_id()});
        push @{$client_id_to_relations{$relation->client_client_id()}}, $relation;
    }

    for my $client (@clients) {
        $client->agency_relations($client_id_to_relations{$client->id()} // []);
    }
}

=head2 enrich_with_managers

    Для всех клиентских и агентских записей из $self->items() проставить их менеджеров в $client->managers()
    Если у клиента или агентства нет менеджеров, в $client->managers() будет проставлен пустой массив.

=cut
sub enrich_with_managers {
    my ($self) = @_;

    my @clients = grep { $_->role() eq $ROLE_CLIENT } @{$self->items()};
    my @agencies = grep { $_->role() eq $ROLE_AGENCY } @{$self->items()};

    my $rbac = RBAC2::Extended->get_singleton(0);
    my @client_ids = map { $_->id() } @clients;
    my $client_id_to_manager_uids = @client_ids ? rbac_get_managers_of_clients_by_clientids($rbac, \@client_ids) : {};

    my @agency_ids = map { $_->id() } @agencies;
    my $agency_id_to_manager_uids = @agency_ids ? rbac_get_all_managers_of_agencies_clientids($rbac, \@agency_ids) : {};

    my @all_manager_uids = uniq map { @$_ } values %$client_id_to_manager_uids, values %$agency_id_to_manager_uids;
    my $manager_uid_to_id = get_key2clientid(uid => \@all_manager_uids);
    my $managers_by_id = Direct::Clients->get_by(id => [values %$manager_uid_to_id])->items_by('id');

    for my $item (@clients, @agencies) {
        my $manager_uids = $item->role() eq $ROLE_CLIENT ?
                $client_id_to_manager_uids->{$item->id()} :
                $agency_id_to_manager_uids->{$item->id()};
        $manager_uids //= [];
        my @managers =
            grep { $_->role() eq $ROLE_MANAGER }
                map { $managers_by_id->{$manager_uid_to_id->{$_}} } @$manager_uids;
        $item->managers(\@managers);
    }
}

=head2 enrich_with_chief_reps

    Для всех записей из $self->items() проставить их шефов в $client->chief_rep()

=cut
sub enrich_with_chief_reps {
    my ($self) = @_;

    my @chief_uids = grep { defined } map { $_->chief_uid() } @{$self->items()};
    my $chiefs_by_id = Direct::Users->get_by(id => \@chief_uids)->items_by('id');

    for my $item (@{$self->items()}) {
        next unless $item->chief_uid();
        $item->chief_rep($chiefs_by_id->{$item->chief_uid()});
    }
}

=head2 items_by($key)

    Возвращает хеш с элементами $self->items() разложенными по полу $key этих элементов в качестве ключа.

=cut
sub items_by {
    my ($self, $key) = @_;

    $key //= 'id';
    croak "by `id` only supported" unless $key =~ /^id$/;

    my %result;
    $result{$_->id} = $_ for @{$self->items};

    return \%result;
}


1;
